"""
t3Block.py

Provides a generic class representing a block in a t3 image file. Generally
this won't be used directly, but as the superclass for classes representing
specific block types.
"""

class Block:
	"""
	A data block in a t3 file. See the T3 VM Image File Format specification
	for more details. 
	
	Initialised with three arguments:
	
	datastream  - either a file, a StringIO object or a class providing read()
	and tell() methods.
	
	image - the t3Image instance that is creating this block.
	
	attribs - a dictionary containing the information in the header of the
	block.
	"""
	
	name = "Generic Block"
	debug = False
	
	def __init__(self, datastream, image, attribs):
		"""
		Initialises the block. Takes a datastream and an int as arguments. The
		datastream can be an open file object or any type of stream that provides
		read, tell and close methods. The int argument is the offset from the beginning
		of the filestream. The image argument will be an instance of T3Image or
		another similar class instance. Finally, the attribs argument is a dictionary
		that this class uses to add in header information.
		"""
		self.datastream = datastream
		self.datastreamidx = datastream.tell()
		self.image = image
		self.extend_dict(attribs)
		self.s_longBlocktype = self.__class__.name
		self.data = {}

	def extend_dict(self, attribs):
		for k in attribs.keys():
			self.__dict__[k] = attribs[k]

	def read_block(self):
		"""
		Read the data contained within the block and store useful information
		in the self.data dictionary. This does nothing here and subclasses
		are expected to override it.
		"""
		return
		
	def report(self):
		"""
			Returns a string giving information about the block.
		"""
		return self.report_header() + "\n" + self.report_data()
		
	def report_header(self):
		"""
			Returns a string giving information about the block's header.
		"""
		sl = []
		sl.append("block type: " + self.s_longBlocktype + "\n")
		sl.append("block size: " + str(self.i_size) + "\n")
		sl.append("flags: ")
		sl.append("".join([str(int(flag)) for flag in self.l_flags]) + "\n")
		sl.append("mandatory: " + str(self.b_mandatoryFlag) + "\n")
		return "".join(sl)
		
	def report_data(self):
		"""
			Returns a string giving information about the block's data. This
			should be overridden by subclasses.
		"""
		return ""