//
//  XTDirectoryHelper.m
//  TadsTerp
//
//  Created by Rune Berg on 31/10/14.
//  Copyright (c) 2014 Rune Berg. All rights reserved.
//

#import "XTFileNameHelper.h"
#import "XTFileUtils.h"
#import "XTPrefs.h"
#import "XTLogger.h"


@interface XTFileNameHelper ()

@property XTPrefs *prefs;
@property NSURL *lastUsedGameDirectoryRegardless;

@end


@implementation XTFileNameHelper

static XTFileNameHelper *singletonInstance;

static XTLogger* logger;

+ (void)initialize
{
	logger = [XTLogger loggerForClass:[XTFileNameHelper class]];
	singletonInstance = [XTFileNameHelper new];
	singletonInstance.prefs = [XTPrefs prefs];
}

+ (id)helper
{
	return singletonInstance;
}

- (NSURL*)findDefaultDirectoryForFileType:(XTadsFileNameDialogFileType)fileType
{
	XT_DEF_SELNAME;
	
	NSURL *res = nil;
	
	switch (fileType) {
		case XTadsFileNameDialogFileTypeGame:
			res = [self findDefaultGamesDirectory];
			break;
		case XTadsFileNameDialogFileTypeSavedGame:
			res = [self findDefaultSavesDirectory];
			break;
		case XTadsFileNameDialogFileTypeTranscript:
			res = [self findDefaultTranscriptsDirectory];
			break;
		case XTadsFileNameDialogFileTypeCommandScript:
			res = [self findDefaultCommandScriptsDirectory];
			break;
		case XTadsFileNameDialogFileTypeGeneral:
			// no default dir for this
			break;
		default:
			XT_WARN_1(@"unknown fileType %d", fileType);
			break;
	}
	
	return res;
}

- (void)noteUsedDirectory:(NSURL *)fileUrl
			  forFileType:(XTadsFileNameDialogFileType)fileType
{
	XT_DEF_SELNAME;
	
	switch (fileType) {
		case XTadsFileNameDialogFileTypeGame:
			[self noteUsedGamesDirectory:fileUrl];
			break;
		case XTadsFileNameDialogFileTypeSavedGame:
			[self noteUsedSavesDirectory:fileUrl];
			break;
		case XTadsFileNameDialogFileTypeTranscript:
			[self noteUsedTranscriptsDirectory:fileUrl];
			break;
		case XTadsFileNameDialogFileTypeCommandScript:
			[self noteUsedCommandScriptsDirectory:fileUrl];
			break;
		case XTadsFileNameDialogFileTypeGeneral:
			// nothing to note for this
			break;
		default:
			XT_WARN_1(@"unknown fileType %d", fileType);
			break;
	}
}

- (NSURL *)findDefaultGamesDirectory
{
	XT_DEF_SELNAME;
	
	NSURL *res = nil;
	
	switch (self.prefs.gamesDirectoryMode) {
		case XTPREFS_DIR_MODE_SPECIFIC:
			res = self.prefs.gamesDirectoryWhenSpecific;
			break;
		case XTPREFS_DIR_MODE_NONE:
			res = nil;
			break;
		case XTPREFS_DIR_MODE_LAST_SELECTED:
			res = self.prefs.gamesDirectoryLastUsed;
			break;
		default:
			XT_WARN_1(@"unexpected dir mode %lu", self.prefs.gamesDirectoryMode);
			break;
	}
	
	XT_TRACE_1(@"-> \"%@\"", res);
	
	return res;
}

- (void)noteUsedGamesDirectory:(NSURL *)gameFileUrl
{
	XT_DEF_SELNAME;
	XT_TRACE_1(@"%@", gameFileUrl);
	
	self.lastUsedGameDirectoryRegardless = [XTFileUtils existingDirectoryUrlFor:gameFileUrl];
	self.prefs.gamesDirectoryLastUsed = self.lastUsedGameDirectoryRegardless;
}

- (NSURL *)findDefaultSavesDirectory
{
	XT_DEF_SELNAME;
	
	NSURL *res = nil;
	
	switch (self.prefs.savesDirectoryMode) {
		case XTPREFS_DIR_MODE_SPECIFIC:
			res = self.prefs.savesDirectoryWhenSpecific;
			break;
		case XTPREFS_DIR_MODE_NONE:
			res = nil;
			break;
		case XTPREFS_DIR_MODE_LAST_SELECTED:
			res = self.prefs.savesDirectoryLastUsed;
			break;
		case XTPREFS_DIR_MODE_CURRENT_GAMEFILE:
			res = self.lastUsedGameDirectoryRegardless;
			break;
		default:
			XT_WARN_1(@"unexpected dir mode %lu", self.prefs.savesDirectoryMode);
			break;
	}
	
	XT_TRACE_1(@"-> \"%@\"", res);
	
	return res;
}

- (void)noteUsedSavesDirectory:(NSURL *)saveFileUrl
{
	self.prefs.savesDirectoryLastUsed = [XTFileUtils existingDirectoryUrlFor:saveFileUrl];
}

- (NSURL *)findDefaultTranscriptsDirectory
{
	XT_DEF_SELNAME;
	
	NSURL *res = nil;
	
	switch (self.prefs.transcriptsDirectoryMode) {
		case XTPREFS_DIR_MODE_SPECIFIC:
			res = self.prefs.transcriptsDirectoryWhenSpecific;
			break;
		case XTPREFS_DIR_MODE_NONE:
			res = nil;
			break;
		case XTPREFS_DIR_MODE_LAST_SELECTED:
			res = self.prefs.transcriptsDirectoryLastUsed;
			break;
		case XTPREFS_DIR_MODE_CURRENT_GAMEFILE:
			res = self.lastUsedGameDirectoryRegardless;
			break;
		default:
			XT_WARN_1(@"unexpected dir mode %lu", self.prefs.transcriptsDirectoryMode);
			break;
	}

	XT_TRACE_1(@"-> \"%@\"", res);
	
	return res;
}

- (void)noteUsedTranscriptsDirectory:(NSURL *)transcriptFileUrl
{
	self.prefs.transcriptsDirectoryLastUsed = [XTFileUtils existingDirectoryUrlFor:transcriptFileUrl];
}

- (NSURL *)findDefaultCommandScriptsDirectory
{
	XT_DEF_SELNAME;
	
	NSURL *res = nil;
	
	switch (self.prefs.commandScriptsDirectoryMode) {
		case XTPREFS_DIR_MODE_SPECIFIC:
			res = self.prefs.commandScriptsDirectoryWhenSpecific;
			break;
		case XTPREFS_DIR_MODE_NONE:
			res = nil;
			break;
		case XTPREFS_DIR_MODE_LAST_SELECTED:
			res = self.prefs.commandScriptsDirectoryLastUsed;
			break;
		case XTPREFS_DIR_MODE_CURRENT_GAMEFILE:
			res = self.lastUsedGameDirectoryRegardless;
			break;
		default:
			XT_WARN_1(@"unexpected dir mode %lu", self.prefs.commandScriptsDirectoryMode);
			break;
	}
	
	XT_TRACE_1(@"-> \"%@\"", res);
	
	return res;
}

- (void)noteUsedCommandScriptsDirectory:(NSURL *)commandScriptFileUrl
{
	self.prefs.commandScriptsDirectoryLastUsed = [XTFileUtils existingDirectoryUrlFor:commandScriptFileUrl];
}

- (NSString*)findDefaultFileBasenameForGameFileUrl:(NSURL *)gameFileUrl
										  fileType:(XTadsFileNameDialogFileType)fileType
{
	XT_DEF_SELNAME;
	
	NSString *res = nil;
	
	switch (fileType) {
		case XTadsFileNameDialogFileTypeSavedGame:
			res = [self makeDefaultFileBasenameForGameFileUrl:gameFileUrl fileNameMode:self.prefs.savesFileNameMode];
			break;
		case XTadsFileNameDialogFileTypeTranscript:
			res = [self makeDefaultFileBasenameForGameFileUrl:gameFileUrl fileNameMode:self.prefs.transcriptsFileNameMode];
			break;
		case XTadsFileNameDialogFileTypeCommandScript:
			res = [self makeDefaultFileBasenameForGameFileUrl:gameFileUrl fileNameMode:self.prefs.commandScriptsFileNameMode];
			break;
		case XTadsFileNameDialogFileTypeGeneral:
			// no default basename for this
			break;
		default:
			XT_WARN_1(@"unknown fileType %d", fileType);
			break;
	}
	
	return res;
}

- (NSString *)makeDefaultFileBasenameForGameFileUrl:(NSURL *)gameFileUrl
									   fileNameMode:(XTPrefsFileNameMode)fileNameMode
{
	XT_DEF_SELNAME;
	
	NSString *res = nil;
	
	switch (fileNameMode) {
		case XTPREFS_FILENAME_MODE_GAMENAME_DATETIME: {
			NSURL *gameFileUrlMinusExtension = [gameFileUrl URLByDeletingPathExtension];
			NSString *gameFileBasename = [gameFileUrlMinusExtension lastPathComponent];
			NSDateFormatter *dateFormatter = [NSDateFormatter new];
			[dateFormatter setDateFormat:@"yyyyMMdd_HHmmss"];
			NSDate *now = [NSDate date];
			NSString *nowString = [dateFormatter stringFromDate:now];
			res = [NSString stringWithFormat:@"%@-%@", gameFileBasename, nowString];
			break;
		}
		case XTPREFS_FILENAME_MODE_GAMENAME: {
			NSURL *gameFileUrlMinusExtension = [gameFileUrl URLByDeletingPathExtension];
			NSString *gameFileBasename = [gameFileUrlMinusExtension lastPathComponent];
			res = [NSString stringWithFormat:@"%@", gameFileBasename];
			break;
		}
		case XTPREFS_FILENAME_MODE_UNTITLED:
			res = @"untitled";
			break;
		default:
			XT_WARN_1(@"unknown fileNameMode %d", fileNameMode);
			break;
	}
	
	return res;
}


@end
