/*
    DosGlk  --  A Glk implementation for MS-DOS
    Copyright (C) 1998  Matt Kimball

    Permission is hereby granted, free of charge, to any person
    obtaining a copy of this software and associated documentation
    files (the "Software"), to deal in the Software without
    restriction, including without limitation the rights to use, copy,
    modify, merge, publish, distribute, sublicense, and/or sell copies
    of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following condition:
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT.  IN NO EVENT SHALL MATT KIMBALL BE LIABLE FOR ANY
    CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
    CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
    WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdio.h>
#include <stdlib.h>

#include "glk.h"   
#include "list.h"
#include "win.h"
#include "stream.h"
#include "file.h"

static list *strlist = NULL;
#define getstr(id) (lglk_list_index(strlist, (unsigned)(id) - 1))
static strid_t current = 0;

stream *lglk_get_stream(strid_t id) {
	return getstr(id);
}

strid_t glk_window_get_stream(winid_t id) {
	win *w;
	stream *s;
	strid_t ret;
	
	w = lglk_get_window(id);
	if(w == NULL) 
		return 0;
		
	if(w->stream)
		return w->stream;
		
	if(strlist == NULL) 
		strlist = lglk_list_create(sizeof(stream));
	if(strlist == NULL)
		return 0;
		
	ret = (strid_t)lglk_list_add(strlist);
	if(ret == LGLK_NULL_INDEX)
		return 0;
	ret++;
	s = getstr(ret);
	
	w->stream = ret;
	s->type = stream_Window;
	s->val.win = id;
	
	return ret;
}

void glk_window_set_echo_stream(winid_t id, strid_t str) {
	win *w;
	stream *s;
	
	w = lglk_get_window(id);
	if(w == NULL)
		return;
	s = getstr(str);
	if(s == NULL)
		return;
		
	w->echo = str;
}

strid_t glk_window_get_echo_stream(winid_t id) {
	win *w;
	
	w = lglk_get_window(id);
	if(w == NULL)
		return 0;
		
	return w->echo;
}

void glk_set_window(winid_t win) {
	glk_stream_set_current(glk_window_get_stream(win));
}

char *lglk_mode_string(glui32 fmode, int textmode) {
	switch(fmode) {    
		case filemode_Write:
			if(textmode)
				return "w";
			else
				return "wb";
			
		case filemode_ReadWrite:
			if(textmode)
				return "rw";
			else
				return "rwb";
			
		case filemode_WriteAppend:
			if(textmode)
				return "a";
			else
				return "ab";
	
		default:
			if(textmode)
				return "r";
			else
				return "rb";
	}
}

strid_t glk_stream_open_file(frefid_t id, glui32 fmode,
    						 glui32 rock) {
	fileref *f;
	strid_t ret;
	stream *s;
	
	f = lglk_get_file(id);
	if(f == NULL)
		return 0;
		
	if(strlist == NULL) 
		strlist = lglk_list_create(sizeof(stream));
	if(strlist == NULL)
		return 0;	

	ret = (strid_t)lglk_list_add(strlist);
	if(ret == LGLK_NULL_INDEX)
		return 0;
	ret++;
	s = getstr(ret);
	
	s->type = stream_File;
	s->rock = rock;
	s->val.f = fopen(f->name, lglk_mode_string(fmode, f->mode));
	
	if(s->val.f == NULL) {
		lglk_list_remove(strlist, (unsigned)ret - 1);
		return 0;
	}
	
	return ret;
}
    						
strid_t glk_stream_open_memory(void *buf, glui32 buflen, glui32 fmode,
    						   glui32 rock) {
	strid_t ret;    						   
	stream *s;
    						   
	if(strlist == NULL)
		strlist = lglk_list_create(sizeof(stream));
	if(strlist == NULL)
		return 0;
		
	ret = (strid_t)lglk_list_add(strlist);
	if(ret == LGLK_NULL_INDEX)
		return 0;
	ret++;
	
	s = getstr(ret);
	s->type = stream_Memory;
	s->rock = rock;
	s->val.mem = buf;
	s->len = buflen;
	
	if(fmode == filemode_WriteAppend)
		s->pos = s->len;
		
	return ret;
}
    						   
void glk_stream_close(strid_t id, stream_result_t *result) {
	stream *s;
	winid_t at;
	
	s = getstr(id);
	if(s == NULL)
		return;
		
	if(s->type == stream_File) {
		fclose(s->val.f);
	}
	
	for(at = 0; at = glk_window_iterate(at, NULL); ) {
		win *w;
		
		w = lglk_get_window(at);
		if(w->echo == id)
			w->echo = 0;
	}
	
	if(result != NULL) {
		result->readcount = s->readcount;
		result->writecount = s->writecount;
	}
		
	lglk_list_remove(strlist, (unsigned)id - 1);
}

strid_t glk_stream_iterate(strid_t id, glui32 *rockptr) {
	stream *s;

	while((unsigned)id <= lglk_list_size(strlist)) {
		s = getstr(++id);
		if(s) {
			if(rockptr)
				*rockptr = s->rock;
			return id;
		}
	}
	return 0;
}

glui32 glk_stream_get_rock(strid_t str) {
	stream *s;
	
	s = getstr(str);
	if(s == NULL)
		return 0;
		
	return s->rock;
}

void glk_stream_set_position(strid_t str, glsi32 pos, glui32 seekmode) {
	stream *s;
	
	s = getstr(str);
	if(s == NULL)
		return;
		
	if(s->type == stream_File) {     
		if(seekmode == seekmode_Start) 
			fseek(s->val.f, pos, SEEK_SET);
		else if(seekmode == seekmode_Current)
			fseek(s->val.f, pos, SEEK_CUR);
		else if(seekmode == seekmode_End)
			fseek(s->val.f, pos, SEEK_END);
	} else if(s->type == stream_Memory) {
		s->pos = pos;
	}	
}

glui32 glk_stream_get_position(strid_t str) {
	stream *s;
	
	s = getstr(str);
	if(s == NULL)
		return 0;
		
	if(s->type == stream_File) {     
		return ftell(s->val.f);
	} else if(s->type == stream_Memory) {
		return s->pos;
	}	
}

void glk_stream_set_current(strid_t str) {
	current = str;
}

strid_t glk_stream_get_current(void) {
	return current;
}
