/*
    DosGlk  --  A Glk implementation for MS-DOS
    Copyright (C) 1998  Matt Kimball

    Permission is hereby granted, free of charge, to any person
    obtaining a copy of this software and associated documentation
    files (the "Software"), to deal in the Software without
    restriction, including without limitation the rights to use, copy,
    modify, merge, publish, distribute, sublicense, and/or sell copies
    of the Software, and to permit persons to whom the Software is
    furnished to do so, subject to the following condition:
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT.  IN NO EVENT SHALL MATT KIMBALL BE LIABLE FOR ANY
    CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
    CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
    WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <memory.h>
#include <stdlib.h>

#include "list.h"

#define list_batch_size 0x08
#define list_batch_mask	0x07

struct _list {
	/*  The size of each item in the list  */
	unsigned item_size;
	                   
	/*  The number of batches allocating  */	                   
	unsigned batch_count;                   
	
	/*  The amount of space for batches in batch and batch_mask members  */
	unsigned batch_space;

	/*  Pointers to batches of items  */
	unsigned char **batch;	            
	
	/*  Masks which mark the used slots in existing batches  */
	unsigned char *batch_mask;	
};
                                                
/*  Create a list for space for several batches and one batch already there  */                                                
list *lglk_list_create(unsigned list_item_size) {
	list *l;
	unsigned at;
	
	l = (list *)malloc(sizeof(list));
	if(l == NULL)
		return NULL;
	
	l->item_size = list_item_size;
	l->batch_count = 1;
	l->batch_space = list_batch_size;
	
	l->batch = (unsigned char **)malloc(sizeof(unsigned char *) * l->batch_space);
	if(l->batch == NULL) {
		free(l);
		return NULL;
	}
	
	l->batch_mask = (unsigned char *)malloc(sizeof(unsigned char) * l->batch_space);
	if(l->batch_mask == NULL) {
		free(l);
		free(l->batch);
		return NULL;
	}
	
	for(at = 0; at < l->batch_space; at++) {
		l->batch[at] = NULL;
		l->batch_mask[at] = 0;
	}
	
	l->batch[0] = (unsigned char *)malloc(l->item_size * list_batch_size);
	
	return l;
}

/*  Delete all batches and all batch space  */
void lglk_list_delete(list *l) {
	unsigned at;
	
	if(l == NULL)
		return;
	
	for(at = 0; at < l->batch_count; at++) {
		free(l->batch[at]);
	}
	free(l->batch);
	free(l->batch_mask);
	
	memset(l, 0xFF, sizeof(list));
	free(l);
}
    
/*  Find an unused slot and return an index to it.  Grow lists as needed.  
	Return LGLK_NULL_INDEX if we run out of memory.  */
unsigned lglk_list_add(list *l) {
	unsigned at;
	
	if(l == NULL)
		return LGLK_NULL_INDEX;
	
	/*  Find an unused slot in existing batches if possible.  */
	for(at = 0; at < l->batch_count * list_batch_size; at++) {
		if(!(l->batch_mask[at / list_batch_size] & (1 << (at & list_batch_mask)))) {
			l->batch_mask[at / list_batch_size] |= 1 << (at & list_batch_mask);
			          
			memset(l->batch[at / list_batch_size] + l->item_size * (at & list_batch_mask),
				   0x00, l->item_size);
				   
			return at;
		}
	}
	
	/*  Creaste space for pointers and flags to new new batches.  */
	if(l->batch_count == l->batch_space) {
		unsigned char **new_batch, *new_batch_mask;
		unsigned new_batch_count = l->batch_space + list_batch_size;
		
		new_batch = (unsigned char **)realloc(l->batch, sizeof(unsigned char *) * new_batch_count);
		if(new_batch == NULL)
			return LGLK_NULL_INDEX;
		l->batch = new_batch;			
			
		new_batch_mask = (unsigned char *)realloc(l->batch_mask, new_batch_count);
		if(new_batch_mask == NULL)
			return LGLK_NULL_INDEX;
		l->batch_mask = new_batch_mask;  
		
		for(at = l->batch_count; at < new_batch_count; at++) {
			l->batch[at] = NULL;
			l->batch_mask[at] = 0;
		}
	
		l->batch_space = new_batch_count;
	}
	                               
	/*  Add another batch on to the end of the list  */	                               
	l->batch[l->batch_count] = (unsigned char *)malloc(l->item_size * list_batch_size);
	if(l->batch[l->batch_count] == NULL)
		return LGLK_NULL_INDEX;
		                        
	/*  Mark this space as used  */		                        
	l->batch_mask[l->batch_count] = 1;
	
	memset(l->batch[l->batch_count], 0x00, l->item_size);
	return l->batch_count++ * list_batch_size;
}
    
/*  Return a pointer to the given item  */
void *lglk_list_index(list *l, unsigned ix) {
	if(l == NULL)
		return NULL;

	/*  If the index is out of range, return NULL  */
	if(ix < 0 || ix > l->batch_count * list_batch_size)
		return NULL;
		                                             
	/*  If the index isn't used, return NULL  */		                                             
	if(!(l->batch_mask[ix / list_batch_size] & (1 << (ix & list_batch_mask)))) {
		return NULL;
	}
	
	return l->batch[ix / list_batch_size] + l->item_size * (ix & list_batch_mask);
}

/*  Mark a particular slot as unused  */
void lglk_list_remove(list *l, unsigned ix) {
	if(l == NULL)
		return;
		
	if(ix < 0 || ix > l->batch_count * list_batch_size)
		return;
		
	l->batch_mask[ix / list_batch_size] &= ~(1 << (ix & list_batch_mask));
}

unsigned lglk_list_size(list *l) {
	unsigned ret, at;
	
	if(l == NULL)
		return 0;
	                
	ret = 0;
	for(at = 0; at < l->batch_count * list_batch_size; at++) {
		if(l->batch_mask[at / list_batch_size] & (1 << (at & list_batch_mask))) 
			ret = at + 1;
	}
	
	return ret;
}

void lglk_list_pack(list *l) {
	unsigned len, at, moving, dest;
	unsigned char *from, *to;
	
	len = lglk_list_size(l);
	moving = 0;
	for(at = 0; at < len; at++) {
		from = lglk_list_index(l, at);
		
		if(from != NULL) {
			if(moving) {
				dest = lglk_list_add(l);
				to = lglk_list_index(l, dest);
			
				memcpy(to, from, l->item_size);
				lglk_list_remove(l, at);
			}
		} else {       
			moving = 1;
		}
	}
}

