/************************************************************************
*
* Program: GINA System LISP
* Module ginaobj.c
* Version 0.1, October 1994.
*
* Copyright 1994, 1995, Jeff Standish.  (jestandi@cs.indiana.edu)
* All rights reserved.
* Permission is hereby granted for unrestricted non-commercial use,
* provided full attribution of source code origin is included.
*
* Version 0.1
*
* This version contains the basic object manipulation routines.
*
************************************************************************/

#include <stdio.h>
#include "ginas.h"


	/* externally defined global variables */
extern NODEZ  *functions, *global;
extern OBJECT *objectroot, *classlist;
extern DAEMON *daemonlist;
extern ACTION *actionlist;



/************************************************************************
*
* find_object() - Given an object's identifier and a pointer into an
*	object, check that object and all of its remaining siblings
*	and children to see if any of them have the given identifier.
*	Returns either a pointer to the object if found, or NULL if no
*	such object can be found
*
************************************************************************/

#ifdef _ANSI_
OBJECT *find_object(IDNODE *idptr, OBJECT *obptr)
#else
OBJECT *find_object(idptr, obptr)
IDNODE *idptr;
OBJECT *obptr;
#endif
{
    OBJECT *op;

	/* search all siblings, and recurse on all children */
    while (obptr != NULL) {
	if (obptr->idptr == idptr)
	    return (obptr);

	    /* search children, if get back a pointer, then have found the
		desired object (unless more than one object has the same
		name/identifier */
	op = find_object(idptr, obptr->child);
	if (op != NULL)
	    return (op);

	obptr = obptr->sibling;
    }

	/* unable to locate object, so return NULL */
    return (NULL);
}



/************************************************************************
*
* find_class() - Search through the list of classes for the one with the
*	given identifier.
*
************************************************************************/

#ifdef _ANSI_
OBJECT *find_class(IDNODE *idptr)
#else
OBJECT *find_class(idptr)
IDNODE *idptr;
#endif
{
    OBJECT *clptr;

	/* loop through the list until the class is found */
    clptr = classlist;
    while (clptr != NULL) {
	if (clptr->idptr == idptr)
	    return (clptr);

	clptr = clptr->sibling;
    }

	/* no such class exists, return NULL */
    return (NULL);
}



/************************************************************************
*
* is_subclass() - recursively search backwards from the given object/class
*	to see if it inherits from the class with the given classid.
*
************************************************************************/

#ifdef _ANSI_
int is_subclass(NODEZ *ptr, IDNODE *classid)
#else
int is_subclass(ptr, classid)
NODEZ  *ptr;
IDNODE *classid;
#endif
{
    OBJECT *clptr;

	/* loop through all classes from which directly inherits */
    while (ptr != NULL) {
	clptr = ptr->value.ptr.car->value.objptr;

	    /* recursively check all classes to see if they inherit
		from the given class */
	if ((clptr->idptr == classid) || (is_subclass(clptr->classes,classid)))
	    return (1);

	ptr = ptr->value.ptr.cdr;
    }

	/* failure, so return false */
    return 0;
}



/************************************************************************
*
* make_class_list() - Given a list of class names, recursively convert
*	them into a list of class pointers.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *make_class_list(NODEZ *nptr)
#else
NODEZ *make_class_list(nptr)
NODEZ *nptr;
#endif
{
    OBJECT *obptr;
    NODEZ  *ptr, *newptr;

	/* if given an empty list, return nil */
    if ((nptr == NULL) || (nptr->type != TYPElisthead))
	return (NULL);

	/* get the class's name */
    ptr = nptr->value.ptr.car;
    if ((ptr == NULL) || (ptr->type != TYPEidname)) {
	printf("Error: Invalid class name in class list: ");
	print_tree(ptr, stdout);
	printf("\n\tIgnoring remaining classes in list.\n");
	return (NULL);
    }

	/* get a pointer to that class */
    obptr = find_class(ptr->value.idptr);
    if (obptr == NULL) {
	printf("Error: undefined class \"%s\" in class list\n",
		ptr->value.idptr->name);
	return (NULL);
    }

	/* convert the class pointer to a lisp node
	   note: can't use obj_to_node() since it works on objects,
		 not classes */
    newptr = new_node();
    newptr->type = TYPEclass;
    newptr->value.objptr = obptr;

	/* cons them up into a list */
    ptr = constructor(newptr, make_class_list(nptr->value.ptr.cdr));

    return (ptr);
}



/************************************************************************
*
* find_property() - Given an object, perform a breadth-first search of
*	it and all of the classes from which it inherits until the
*	given property is found.  Note that this uses a dumb search
*	method which does not keep track of what classes it has already
*	searched, so it may search the same class more than once if
*	different classes inherit from the same class.  But it is fast
*	and simple, and easier to program than forward-looking searches
*	that avoid repeated searching of the same class multiple times.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *find_property(OBJECT *obptr, IDNODE *propid)
#else
NODEZ *find_property(obptr, propid)
OBJECT *obptr;
IDNODE *propid;
#endif
{
    int    queuehead, queuetail;
    NODEZ  *ptr;
    OBJECT *classqueue[QUEUESIZE], *thisobj;

	/* initialize the queue */
    queuehead = 2;
    queuetail = 0;
    classqueue[1] = obptr;

	/* loop through the circular queue until it is empty, or until
	   the class is found */
    while (((queuetail + 1) % QUEUESIZE) != queuehead) {

	    /* pop this class from the queue */
	queuetail = (queuetail + 1) % QUEUESIZE;
	thisobj = classqueue[queuetail];

	    /* if this class holds the desired property, return it */
	if ((ptr = find_prop_rec(thisobj, propid)) != NULL)
	    return (ptr);

	    /* otherwise, push all of the inherited classes onto the queue
		and continue searching */
	ptr = thisobj->classes;
	while (ptr != NULL) {
	    classqueue[queuehead] = ptr->value.ptr.car->value.objptr;
	    queuehead = (queuehead + 1) % QUEUESIZE;
	    if (queuehead == queuetail) {
		printf("Error: class queue overloaded, too many classes\n");
		exit(1);
	    }
	    ptr = ptr->value.ptr.cdr;
	}
    }

	/* class not found, return null */
    return (NULL);
}



/************************************************************************
*
* find_prop_rec() - Given an object, search its list of properties to
*	see if it has the specified properties.  This is the sub-function
*	for find_property().
*
************************************************************************/

#ifdef _ANSI_
NODEZ *find_prop_rec(OBJECT *obptr, IDNODE *propid)
#else
NODEZ *find_prop_rec(obptr, propid)
OBJECT *obptr;
IDNODE *propid;
#endif
{
    NODEZ  *listptr, *ptr, *nameptr;

	/* search through the list of properties */
    listptr = obptr->properties;
    while (listptr != NULL) {

	    /* break if hit a node that should not be in the list */
	if (listptr->type != TYPElisthead)
	    break;

	    /* check this property to see if it is the desired one */
	ptr = listptr->value.ptr.car;
	if ((ptr != NULL) && (ptr->type == TYPElisthead)) {
	    nameptr = ptr->value.ptr.car;
	    if ((nameptr != NULL) && (nameptr->type == TYPEidname)
			&& (nameptr->value.idptr == propid))
		return (ptr);
	}

	    /* otherwise search next properties */
	listptr = listptr->value.ptr.cdr;
    }

	/* property not found, return null */
    return (NULL);
}



/************************************************************************
*
* find_method() - Perform a breadth-first search on the methods an
*	object inherits in search of the method specified by idptr.
*	This uses a stupid search that does not bother to avoid
*	re-searching the same class multiple times.  But it is a
*	fast, simple, easy-to-program method that works.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *find_method(OBJECT *obptr, IDNODE *idptr)
#else
NODEZ *find_method(obptr, idptr)
OBJECT *obptr;
IDNODE *idptr;
#endif
{
    int    queuehead, queuetail;
    NODEZ  *ptr;
    OBJECT *classqueue[QUEUESIZE], *thisobj;

	/* initialize the circular queue */
    queuehead = 2;
    queuetail = 0;
    classqueue[1] = obptr;

	/* search through all classes inherited */
    while (((queuetail + 1) % QUEUESIZE) != queuehead) {

	    /* pop the current class from the queue */
	queuetail = (queuetail + 1) % QUEUESIZE;
	thisobj = classqueue[queuetail];

	    /* if this is the desired method, return the pointer to it */
	if ((ptr = find_meth_rec(thisobj, idptr)) != NULL)
	    return (ptr);

	    /* push all inherited classes into the queue */
	ptr = thisobj->classes;
	while (ptr != NULL) {
	    classqueue[queuehead] = ptr->value.ptr.car->value.objptr;
	    queuehead = (queuehead + 1) % QUEUESIZE;
	    if (queuehead == queuetail) {
		printf("Error: class queue overloaded, too many classes\n");
		exit(1);
	    }
	    ptr = ptr->value.ptr.cdr;
	}
    }

	/* method not found, return null */
    return (NULL);
}



/************************************************************************
*
* find_meth_rec() - Given a class, search it to see if has the method
*	specified by idptr.  This is the subfunction for find_method().
*
************************************************************************/

#ifdef _ANSI_
NODEZ *find_meth_rec(OBJECT *obptr, IDNODE *idptr)
#else
NODEZ *find_meth_rec(obptr, idptr)
OBJECT *obptr;
IDNODE *idptr;
#endif
{
    NODEZ  *listptr, *ptr, *nameptr;

	/* loop through all methods defined for this class */
    listptr = obptr->methods;
    while (listptr != NULL) {
	if (listptr->type != TYPElisthead)
	    break;

	    /* if this is the correct method, return the pointer to it */
	ptr = listptr->value.ptr.car;
	if ((ptr != NULL) && (ptr->type == TYPElisthead)) {
	    nameptr = ptr->value.ptr.car;
	    if ((nameptr != NULL) && (nameptr->type == TYPEidname)
			&& (nameptr->value.idptr == idptr))
		return (ptr);
	}

	    /* otherwise keep on searching */
	listptr = listptr->value.ptr.cdr;
    }

	/* method not found, so return null */
    return (NULL);
}



/************************************************************************
*
* assemble_object() - Given an internal representation of an object or
*	class, convert it into a LISP-type list.  This function works
*	for both objects and classes.  	If isobject is set, then add in
*	the name of the parent object which is not relavant for classes.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *assemble_object(OBJECT *obptr, int isobject)
#else
NODEZ *assemble_object(obptr, isobject)
OBJECT *obptr;
int    isobject;
#endif
{
    NODEZ *newptr, *ptr;

	/* cons up a list of inherited classes, properties, and methods */
    ptr = constructor(obptr->methods, NULL);
    ptr = constructor(obptr->properties, ptr);
    ptr = constructor(assemble_classlist(obptr->classes), ptr);

	/* if it is an object and not a class, include the parent/location */
    if (isobject)
	ptr = constructor(obj_to_idname(obptr->parent), ptr);

	/* add in the name of the object/class */
    newptr = new_node();
    newptr->type = TYPEidname;
    newptr->value.idptr = obptr->idptr;
    ptr = constructor(newptr, ptr);

	/* return the resulting list */
    return (ptr);
}



/************************************************************************
*
* assemble_classlist() - Recursively take a list of class pointers and
*	convert them into a list of class names.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *assemble_classlist(NODEZ *list)
#else
NODEZ *assemble_classlist(list)
NODEZ *list;
#endif
{
    if (list == NULL)
	return (NULL);

    return (constructor(obj_to_idname(list->value.ptr.car->value.objptr),
		assemble_classlist(list->value.ptr.cdr)));
}



/************************************************************************
*
* assemble_action() - Convert the internal representation of an action
*	definition into a lisp representation.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *assemble_action(ACTION *actptr)
#else
NODEZ *assemble_action(actptr)
ACTION *actptr;
#endif
{
    NODEZ *ptr, *newptr;

    newptr = new_node();
    newptr->type = TYPEidname;
    newptr->value.idptr = actptr->idptr;

    ptr = constructor(actptr->after, NULL);
    ptr = constructor(actptr->before, ptr);
    ptr = constructor(actptr->words, ptr);
    ptr = constructor(newptr, ptr);

    return (ptr);
}



/************************************************************************
*
* destroy_object() - Given a pointer to an internal object structure,
*	remove that object from the object tree.
*
************************************************************************/

#ifdef _ANSI_
void destroy_object(OBJECT *obptr)
#else
destroy_object(obptr)
OBJECT *obptr;
#endif
{
    OBJECT *parptr, *sibptr;

	/* ignore null pointers */
    if (obptr == NULL)
	return;

	/* if object is first child of parent, remove from list */
    parptr = obptr->parent;
    if (parptr->child == obptr)
	parptr->child = obptr->sibling;

	/* otherwise, skim through all children of parent until reach
	   previous sibling of object, then remove object from list */
    else {
	sibptr = parptr->child;
	while ((sibptr != NULL) && (sibptr->sibling != obptr))
	    sibptr = sibptr->sibling;

	if (sibptr != NULL)
	    sibptr->sibling = obptr->sibling;
	else {
	    printf("Error: destroy_object(): object not in sibling list\n");
	    exit(1);
	}
    }

	/* now remove this object from object tree */
    obptr->sibling = NULL;
    free_object(obptr);
}



/************************************************************************
*
* dump_state() - Dump the state of the system to a file in lisp format
*	so that it can be loaded, or so that the resulting file can be
*	used as the startup file (the intention is to create a game map).
*
************************************************************************/

#ifdef _ANSI_
void dump_state(char *filename)
#else
dump_state(filename)
char *filename;
#endif
{
    NODEZ  *ptr, *pt2, *pt3;
    FILE   *fopen(), *outfile;
    ACTION *actptr;
    OBJECT *clptr;
    DAEMON *daeptr;

    if ((outfile = fopen(filename, "w")) == NULL) {
	printf("Error: unable to open file \"%s\" for output.\n", filename);
	exit(1);
    }

    pt3 = new_node();

    pt3->type = TYPEfuncname;
    pt3->value.number = CLASSnum;
    clptr = classlist;
    while (clptr != NULL) {
	dump_tree(constructor(pt3, assemble_object(clptr, 0)), outfile);
	clptr = clptr->sibling;
    }

    pt3->type = TYPEfuncname;
    pt3->value.number = OBJECTnum;
    dump_state_rec(pt3, objectroot->child, outfile);

    pt3->type = TYPEfuncname;
    pt3->value.number = ACTIONnum;
    actptr = actionlist;
    while (actptr != NULL) {
	dump_tree(constructor(pt3, assemble_action(actptr)), outfile);
	actptr = actptr->next;
    }

    pt3->type = TYPEfuncname;
    pt3->value.number = SETnum;
    ptr = global;
    while (ptr != NULL) {
	pt2 = ptr->value.ptr.car;
	if (pt2->value.ptr.cdr != NULL)
	    dump_tree(constructor(pt3, constructor(pt2->value.ptr.car,
		constructor(quoter(pt2->value.ptr.cdr), NULL))), outfile);
	else
	    dump_tree(constructor(pt3, constructor(pt2->value.ptr.car,
		constructor(NULL, NULL))), outfile);
	ptr = ptr->value.ptr.cdr;
    }

    pt3->type = TYPEidname;
    pt3->value.idptr = find_idstring("function");
    ptr = functions;
    while (ptr != NULL) {
	pt2 = ptr->value.ptr.car;
	dump_tree(constructor(pt3, pt2), outfile);
	ptr = ptr->value.ptr.cdr;
    }

    daeptr = daemonlist;
    while (daeptr != NULL) {
	dump_tree(save_spawner(daeptr), outfile);
	daeptr = daeptr->next;
    }

    fclose(outfile);
}


#ifdef _ANSI_
void dump_state_rec(NODEZ *ptr, OBJECT *objptr, FILE *outfile)
#else
dump_state_rec(ptr, objptr, outfile)
NODEZ  *ptr;
OBJECT *objptr;
FILE   *outfile;
#endif
{
    while (objptr != NULL) {
	dump_tree(constructor(ptr, assemble_object(objptr, 1)), outfile);
	dump_state_rec(ptr, objptr->child, outfile);
	objptr = objptr->sibling;
    }
}



/************************************************************************
*
* save_objects() - Given an object and the name of a file, convert the
*	object and its children into a lisp-description which is dumped
*	into the file.
*
************************************************************************/

#ifdef _ANSI_
void save_objects(OBJECT *obptr, char *filename)
#else
save_objects(obptr, filename)
OBJECT *obptr;
char   *filename;
#endif
{
    FILE  *fopen(), *outfile;
    NODEZ *nodeptr;

	/* open the file */
    if ((outfile = fopen(filename, "w")) == NULL) {
	printf("Error: save_objects(): unable to open file \"%s\"\n",filename);
	exit(1);
    }

	/* create a node with the identifier for the object function */
    nodeptr = new_node();
    nodeptr->type = TYPEfuncname;
    nodeptr->value.number = OBJECTnum;

	/* go an recursively save the objects */
    save_objects_rec(obptr, nodeptr, outfile);

	/* all done, close the file */
    fclose(outfile);
}



/************************************************************************
*
* save_objects_rec() - This is the recursive function which does all of
*	the work of writing out a file that contains all of the necessary
*	function calls required to recreate the given objects, with their
*	related daemons.
*
************************************************************************/

#ifdef _ANSI_
void save_objects_rec(OBJECT *obptr, NODEZ *nodeptr, FILE *outfile)
#else
save_objects_rec(obptr, nodeptr, outfile)
OBJECT *obptr;
NODEZ  *nodeptr;
FILE   *outfile;
#endif
{
    DAEMON *daeptr;

	/* abort if given a null pointer */
    if (obptr == NULL)
	return;

	/* convert the internal representation of the object into a
	   function call that will create an identical object, which
	   is then printed to the output file */
    print_tree(constructor(nodeptr, assemble_object(obptr, 1)), outfile);
    putc('\n', outfile);

	/* loop through all daemons, looking for any that were spawned
	   on this object, if find any then need to create function
	   calls to create identical daemons */
    daeptr = daemonlist;
    while (daeptr != NULL) {
	if (daeptr->objectptr == obptr) {
	    print_tree(save_spawner(daeptr), outfile);
	}

	daeptr = daeptr->next;
    }

	/* now recursively save all children of this object */
    obptr = obptr->child;
    while (obptr != NULL) {
	save_objects_rec(obptr, nodeptr, outfile);
	obptr = obptr->sibling;
    }
}



/************************************************************************
*
* save_spawner() - Create a LISP list that represents a function call
*	to spawn off a daemon that is identical to the one given.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *save_spawner(DAEMON *daeptr)
#else
NODEZ *save_spawner(daeptr)
DAEMON *daeptr;
#endif
{
    NODEZ *ptr, *ptr2;

    ptr = new_node();
    ptr->type = TYPEfuncname;
    ptr->value.number = SPAWNnum;

    ptr2 = new_node();
    ptr2->type = TYPEidname;
    ptr2->value.idptr = daeptr->idptr;

    return (constructor(ptr,
	    constructor(make_findobj(daeptr->objectptr),
	    constructor(ptr2,
	    save_spawner_rec(daeptr->variables, NULL)))));
}



/************************************************************************
*
* save_spawner_rec() - This is a recursive function for save_spawner()
*	which recursively assembles the parameter list for the daemon,
*	which is then returned.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *save_spawner_rec(NODEZ *arglist, NODEZ *result)
#else
NODEZ *save_spawner_rec(arglist, result)
NODEZ *arglist, *result;
#endif
{
    NODEZ *ptr, *newptr;

    if (arglist == NULL)
	return (result);

    ptr = arglist->value.ptr.car->value.ptr.cdr;
    if (ptr == NULL)
	newptr = NULL;
    else if ((ptr->type == TYPEnumber) || (ptr->type == TYPEstring))
	newptr = ptr;
    else if (ptr->type == TYPEobject)
	newptr = make_findobj(ptr->value.objptr);
    else if (ptr->type == TYPEclass) {
	printf("Error: unexpected class pointer in daemon variable list\n");
	exit(1);
    }
    else 
	newptr = quoter(ptr);

    return (save_spawner_rec(arglist->value.ptr.cdr,
			     constructor(newptr, result)));
}

