/************************************************************************
*
* Program: GINA System LISP
* Module ginamem.c
* Version 0.1, January 1994.
* Version 0.2, October 1994.
* Version 0.3, February, 1995.
*
* Copyright 1994, 1995, Jeff Standish.  (jestandi@cs.indiana.edu)
* All rights reserved.
* Permission is hereby granted for unrestricted non-commercial use,
* provided full attribution of source code origin is included.
*
* Version 0.1
* This version served as the interface with the special routines for
* completely accessing all memory in the IBM PC.
*
* All identifiers are stored in a binary tree, sorted both by the
* identifier string, as well as the unique idnum associated with each
* identifier.  This prevents the lisp interpreter from having to deal
* with string comparisons, and also hopefully saves a little memory
* by preventing multiple storage of strings.
*
* Version 0.2
* Revised into standard C, removing dependencies upon special functions
* for accessing memory via 8088.
*
* The memory allocation scheme for lisp nodes was changed.  Now blocks
* of nodes are allocated, each of which contains BLOCKSIZE nodes.  This
* method was selected because it allows a more efficient method of
* linking together nodes for garbage collection.  This way, instead of
* each individual lisp node have a next pointer for garbage collection,
* large numbers of lisp nodes can be stuck in a single block, which
* then only has one next pointer for garbage collection purposes.
*
* Version 0.3
* The noginas flag was changed so that it will only prevent entering
* the interactive interpreter if it is defined as a non-nil value,
* rather than simply blocking if its is defined (whether nil or non-nil).
*
* Implemented the *warning-messages* and *collection-messages* for
* controlling the displaying of warning messages and garbage collection
* messages from within GINAS, rather than having to rely upon command
* line flags.
*
* IDNODE was changed from being a double-indexed tree to just being
* indexed by the strings.  The need to identify each idnode with a unique
* number was replaced by simply identifying the idnode with the pointer
* to it.  It was also changed so that strings could be stored as idnodes
* to allow unused strings to be freed up, as well as to prevent duplication
* of strings (such as words used in parsing).
*
* Version 0.4:
* The *no-ginas* and +player+ bindings were removed from the internal
* declarations in conjunction with the reconfiguring of the built-in
* parser.
*
************************************************************************/

#include <ctype.h>
#include <stdio.h>
#include "ginas.h"


	/* data for string output */
char outputbuffer[256];
int  outputptr, outlength, docap;


	/* locally defined global variables */
ACTION	*actionlist, *lastaction;
NODEZ	*global, *locals, *functions;
IDNODE	*idtree;
BLOCK	*firstblock, *freeblock;
OBJECT	*objectroot, *freeobjects, *classlist, *lastclass;
OBJECT	*obspecroot;

	/* special root pointer for loading objects */
DAEMON	*daemonlist, *freedaemons;

	/* some values used to keep track of the number of nodes allocated
	   for garbage collection purposes */
int	totalnodes, totalused, nodetally;

IDNODE	*nounid, *adjid;
IDNODE	*trueid, *selfid;
IDNODE	*garbmesgid, *warningid, *debugid, *initialid, *objnameid;

	/* pointers to some common global bindings */
NODEZ	*truebind, *selfbind;
NODEZ   *actorbind, *instrbind, *subjbind, *locatbind;
NODEZ   *actorptrbind, *subjptrbind, *instrptrbind, *locatptrbind;
NODEZ	*actorquantbind, *subjquantbind, *instrquantbind, *locatquantbind;
NODEZ	*garbmesgbind, *warningbind, *debugbind;

PARSY	*freeparsy;

	/* input file */
FILE *lispfile;
int  linenumber;
int  debuglevel;


NODEZ *parserbindlist[MAXBIND];



/************************************************************************
*
* initialize_memory() - Initialize all of the pre-defined data structures
*	and list values.
*
************************************************************************/

#ifdef _ANSI_
void initialize_memory(void)
#else
initialize_memory()
#endif
{
    IDNODE *foo;
    NODEZ  *ptr;

    linenumber = 1;
    debuglevel = 0;

    freeparsy = NULL;
    daemonlist = NULL;
    freedaemons = NULL;
    actionlist = NULL;
    lastaction = NULL;
    idtree = NULL;
    freeblock = NULL;
    firstblock = NULL;
    totalnodes = 0;
    nodetally = 0;
    new_block();

    global = NULL;
    objectroot = new_object();
    objectroot->idptr = find_idstring("megaverse***");
    obspecroot = objectroot;
    freeobjects = NULL;
    classlist = NULL;
    lastclass = NULL;

    initialid = find_idstring("Initialize");
    objnameid = find_idstring("PrintObjectName");
    nounid = find_idstring("nouns");
    adjid = find_idstring("adjectives");
    trueid = find_idstring("t");
    selfid = find_idstring("self");
    garbmesgid = find_idstring("*collection-messages*");
    warningid = find_idstring("*warning-messages*");
    debugid = find_idstring("*debug*");

	/* create the true node */
    ptr = new_node();
    ptr->type = TYPEidname;
    ptr->value.idptr = trueid;

	/* enter some global bindings */
    assign_global(warningid, ptr);
    assign_global(debugid, ptr);
    assign_global(garbmesgid, NULL);
    assign_global(selfid, NULL);
    assign_global(trueid, ptr);

	/* pointers to bindings for these global values */
    warningbind = find_binding(warningid);
    debugbind = find_binding(debugid);
    garbmesgbind = find_binding(garbmesgid);
    selfbind = find_binding(selfid);
    truebind = find_binding(trueid);

	/* set bindings and pointers for parsing objects */
    foo = find_idstring("*actor*");
    assign_global(foo, NULL);
    actorbind = find_binding(foo);
    parserbindlist[0] = actorbind;

    foo = find_idstring("*actorptr*");
    assign_global(foo, NULL);
    actorptrbind = find_binding(foo);
    parserbindlist[1] = actorptrbind;

    foo = find_idstring("*actorquant*");
    assign_global(foo, NULL);
    actorquantbind = find_binding(foo);
    parserbindlist[2] = actorquantbind;

    foo = find_idstring("*subject*");
    assign_global(foo, NULL);
    subjbind = find_binding(foo);
    parserbindlist[3] = subjbind;

    foo = find_idstring("*subjectptr*");
    assign_global(foo, NULL);
    subjptrbind = find_binding(foo);
    parserbindlist[4] = subjptrbind;

    foo = find_idstring("*subjectquant*");
    assign_global(foo, NULL);
    subjquantbind = find_binding(foo);
    parserbindlist[5] = subjquantbind;

    foo = find_idstring("*instrument*");
    assign_global(foo, NULL);
    instrbind = find_binding(foo);
    parserbindlist[6] = instrbind;

    foo = find_idstring("*instrumentptr*");
    assign_global(foo, NULL);
    instrptrbind = find_binding(foo);
    parserbindlist[7] = instrptrbind;

    foo = find_idstring("*instrumentquant*");
    assign_global(foo, NULL);
    instrquantbind = find_binding(foo);
    parserbindlist[8] = instrquantbind;

    foo = find_idstring("*location*");
    assign_global(foo, NULL);
    locatbind = find_binding(foo);
    parserbindlist[9] = locatbind;

    foo = find_idstring("*locationptr*");
    assign_global(foo, NULL);
    locatptrbind = find_binding(foo);
    parserbindlist[10] = locatptrbind;

    foo = find_idstring("*locationquant*");
    assign_global(foo, NULL);
    locatquantbind = find_binding(foo);
    parserbindlist[11] = locatquantbind;


    locals = new_node();
    locals->type = TYPElisthead;
    functions = NULL;

    lispfile = NULL;

    outputbuffer[0] = '\0';
    outputptr = 0;
    outlength = 0;
    docap = 0;
}



/************************************************************************
*
* new_node() - Allocates a new lisp/cons node.
*
************************************************************************/

#ifdef _ANSI_
NODEZ *new_node(void)
#else
NODEZ *new_node()
#endif
{
    NODEZ *ptr;
    int   i;

	/* increment the count of nodes allocated since the last time
	   garbage collection was performed */
    ++nodetally;

	/* if there is no pointer to a block of free cons nodes, reset to
	   the first block of nodes */
    if (freeblock == NULL)
	freeblock = firstblock;

	/* loop through the blocks of nodes, searching for an unused
	   cons node to reallocate */
    while (freeblock != NULL) {

	    /* search through all of the nodes in the block, starting
		with the next unchecked block */
	for (i = freeblock->nextcheck; i < BLOCKSIZE; ++i) {

		/* if an unused block is found, reallocate it and return
		   the pointer to it */
	    if (freeblock->nodelist[i].inuse == 0) {
		freeblock->nextcheck = i + 1;
		ptr = &(freeblock->nodelist[i]);
		ptr->inuse = 1;
		ptr->value.ptr.car = NULL;
		ptr->value.ptr.cdr = NULL;
		return (ptr);
	    }
	}

	    /* otherwise reset to the next block to check to be a large
		value to indicate no remaining free nodes in this block */
	freeblock->nextcheck = BLOCKSIZE;

	    /* if no more blocks remain, allocate a new one */
	if (freeblock->nextblock == NULL)
	    new_block();

	    /* move to the next block of nodes */
	freeblock = freeblock->nextblock;
    }

	/* this statement should never be reached, but to be safe... */
    return (NULL);
}



/************************************************************************
*
* new_action() - Allocate a new action structure, inserting it into
*	the list of actions.
*
************************************************************************/

#ifdef _ANSI_
ACTION *new_action(void)
#else
ACTION *new_action()
#endif
{
    ACTION *ptr;

    if ((ptr = (ACTION*)malloc(sizeof(ACTION))) == NULL) {
	printf("Error: new_action(): failure to allocate enough memory\n");
	exit(1);
    }

	/* insert the actions into the list in the order in which they are
	   defined */
    if (actionlist == NULL)
	actionlist = ptr;
    else
	lastaction->next = ptr;
    lastaction = ptr;

	/* initialize the values */
    ptr->before = NULL;
    ptr->after = NULL;
    ptr->next = NULL;

    return (ptr);
}



/************************************************************************
*
* find_action() - Given an idnode pointer, search through the list of
*	actions for the correct one and return a pointer to it, or NULL
*	if no such action can be found.
*
************************************************************************/

#ifdef _ANSI_
ACTION *find_action(IDNODE *idptr)
#else
ACTION *find_action(idptr)
IDNODE *idptr;
#endif
{
    ACTION *ptr;

    ptr = actionlist;
    while (ptr != NULL) {
	if (ptr->idptr == idptr)
	    return (ptr);
	ptr = ptr->next;
    }

    return (NULL);
}



/************************************************************************
*
* new_block() - Allocate a new block of lisp nodes, adding
*
************************************************************************/

#ifdef _ANSI_
void new_block(void)
#else
new_block()
#endif
{
    BLOCK *ptr;
    int   i;

    if ((ptr = (BLOCK*)malloc(sizeof(BLOCK))) == NULL) {
	printf("Error: new_block(): failure to allocate enough memory\n");
	exit(1);
    }

	/* increase the count of total allocated nodes */
    totalnodes += BLOCKSIZE;

	/* indicate all of the nodes as being free */
    for (i = 0; i < BLOCKSIZE; ++i)
	ptr->nodelist[i].inuse = 0;

	/* note that the next free node to check is the first one */
    ptr->nextcheck = 0;

	/* insert the new block after the last block of free nodes
	   as indicated by freeblock, or insert it arbitrarily if
	   freeblock is undefined */
    if (freeblock == NULL) {
	freeblock = ptr;
	firstblock = ptr;
	ptr->nextblock = NULL;
    }
    else {
	ptr->nextblock = freeblock->nextblock;
	freeblock->nextblock = ptr;
    }
}



/************************************************************************
*
* new_daemon() - Allocate a new structure for storing a spawned daemon.
*	Add this daemon structure to the end of the list of spawned
*	daemons, so that daemons get executed in the same order in
*	which they are spawned.  (This assures that if one daemon
*	spawns another, they will both be executed in the same cycle.)
*
************************************************************************/

#ifdef _ANSI_
DAEMON *new_daemon(void)
#else
DAEMON *new_daemon()
#endif
{
    DAEMON *ptr, *daeptr;

	/* if there are any free daemon nodes, reuse them,
	   otherwise allocate a new one */
    if (freedaemons != NULL) {
	daeptr = freedaemons;
	freedaemons = freedaemons->next;
    }
    else if ((daeptr = (DAEMON*)malloc(sizeof(DAEMON))) == NULL) {
	printf("Error: new_daemon(): failure to allocate enough memory\n");
	exit(1);
    }

	/* set up a list of daemons, making the new daemon the last one
	   in the list, so that daemons are executed in the order they
	   are spawned; thus if the list of daemons is empty, make this
	   one the first one, else scan through and find the last one
	   and insert this one after it */
    if (daemonlist == NULL)
	daemonlist = daeptr;
    else {
	ptr = daemonlist;
	while (ptr->next != NULL)
	    ptr = ptr->next;

	ptr->next = daeptr;
    }

	/* initialize the daemon's pointers to NULL */
    daeptr->next = NULL;
    daeptr->variables = NULL;
    daeptr->objectptr = NULL;

    return (daeptr);
}



/************************************************************************
*
* free_daemon() - Free up the given daemon structure, adding it to the
*	list of free daemon structures, so that it can be reused later
*	without the overhead of a free() and malloc().
*
************************************************************************/

#ifdef _ANSI_
void free_daemon(DAEMON *ptr)
#else
free_daemon(ptr)
DAEMON *ptr;
#endif
{
    ptr->next = freedaemons;
    ptr->variables = NULL;
    ptr->objectptr = NULL;
    freedaemons = ptr;
}




/************************************************************************
*
* new_object() - Allocate a new object structure, initializing its
*	values to nil.
*
************************************************************************/

#ifdef _ANSI_
OBJECT *new_object(void)
#else
OBJECT *new_object()
#endif
{
    OBJECT *ptr;

	/* try to reuse any freed objects */
    if (freeobjects != NULL) {
	ptr = freeobjects;
	freeobjects = freeobjects->sibling;
    }
    else if ((ptr = (OBJECT*)malloc(sizeof(OBJECT))) == NULL) {
	printf("Error: new_object(): memory allocation failure\n");
	exit(1);
    }

    ptr->idptr = NULL;
    ptr->parent = NULL;
    ptr->sibling = NULL;
    ptr->child = NULL;
    ptr->properties = NULL;
    ptr->classes = NULL;
    ptr->methods = NULL;

    return (ptr);
}



/************************************************************************
*
* free_object() - Recursively free up an object and all of its siblings
*	and children.  Be certain to remove the object from the object
*	tree before calling this function to avoid trashing a significant
*	part of the object tree by accident.
*
************************************************************************/

#ifdef _ANSI_
void free_object(OBJECT *obptr)
#else
free_object(obptr)
OBJECT *obptr;
#endif
{
    DAEMON *daeptr, *prevdae, *nextdae;
    OBJECT *ptr;

	/* loop through and free this object and all of its siblings */
    while (obptr != NULL) {

	    /* kill any daemons active on this object */
	daeptr = daemonlist;
	prevdae = NULL;
	while (daeptr != NULL) {
	    nextdae = daeptr->next;
	    if (daeptr->objectptr == obptr) {
		if (prevdae == NULL)
		    daemonlist = nextdae;
		else
		    prevdae->next = nextdae;

		free_daemon(daeptr);
		daeptr = nextdae;
	    }
	    else {
		prevdae = daeptr;
		daeptr = nextdae;
	    }
	}

	    /* recurse on any children of this object */
	free_object(obptr->child);

	    /* add this object to the list of free objects,
		and proceed onwards to its sibling */
	ptr = obptr->sibling;
	obptr->sibling = freeobjects;
	freeobjects = obptr;
	obptr = ptr;
    }
}



/************************************************************************
*
* find_idstring() - Given a string, search through the table of
*	identifiers, returning the pointer to the idnode containing
*	the given string.  If no such string can be found, create a
*	new one, and return its pointer.
*
************************************************************************/

#ifdef _ANSI_
IDNODE *find_idstring(char *name)
#else
IDNODE *find_idstring(name)
char *name;
#endif
{
    IDNODE *ptr, *ptr2;
    int    flag, id, rslt;

	/* if the tree is empty, create a new root */
    if (idtree == NULL) {
	ptr = new_idnode();
	ptr->name = new_string(name);
	idtree = ptr;
    }
    else {
	    /* search through the binary tree for the name, if it is found
		return the pointer to its idnode */
	ptr = idtree;
	for (;;) {
	    rslt = strcmp(name, ptr->name);
	    if (rslt < 0) {

		    /* traverse left */
		if (ptr->left != NULL)
		    ptr = ptr->left;

		   /* insert the string to the left */
		else {
		    ptr->left = new_idnode();
		    ptr = ptr->left;
		    ptr->name = new_string(name);
		    break;
		}
	    }
	    else if (rslt > 0) {

		    /* traverse right */
		if (ptr->right != NULL)
		    ptr = ptr->right;

		    /* insert the string to the right */
		else {
		    ptr->right = new_idnode();
		    ptr = ptr->right;
		    ptr->name = new_string(name);
		    break;
		}
	    }

		/* otherwise, have hit the string, so stop at this point */
	    else
		break;
	}
    }

    return (ptr);
}



/************************************************************************
*
* new_idnode() - Allocate a new identifier node.
*
************************************************************************/

#ifdef _ANSI_
IDNODE *new_idnode(void)
#else
IDNODE *new_idnode()
#endif
{
    IDNODE *ptr;

    if ((ptr = (IDNODE*)malloc(sizeof(IDNODE))) == NULL) {
	printf("Error: new_idnode(): failure to allocate enough memory\n");
	exit(1);
    }

    ptr->name = NULL;
    ptr->left = NULL;
    ptr->right = NULL;
    ptr->inuse = 1;

    return(ptr);
}



/************************************************************************
*
* dump_idnodes() - Recursively print out all of the identifiers stored
*	in the identifier tree.
*
************************************************************************/

#ifdef _ANSI_
void dump_idnodes(void)
#else
dump_idnodes()
#endif
{
   dump_id_recurse(idtree, 1);
}

#ifdef _ANSI_
void dump_id_recurse(IDNODE *ptr, int depth)
#else
dump_id_recurse(ptr, depth)
IDNODE *ptr;
int    depth;
#endif
{
    int i;

    if (ptr != NULL) {
	dump_id_recurse(ptr->left, depth+1);

	for (i = 0; i < depth; ++i)
	    putchar(' ');

	printf("%s\n", ptr->name);
	dump_id_recurse(ptr->right, depth+1);
    }
}



/************************************************************************
*
* new_parsy() - Allocate a new node for storing intermediate results of
*	parser in disambiguation of which objects are being referenced.
*
************************************************************************/

#ifdef _ANSI_
PARSY *new_parsy(void )
#else
PARSY *new_parsy()
#endif
{
    int   i;
    PARSY *ptr;

	/* first try to reallocate any freed nodes */
    if (freeparsy != NULL) {
	ptr = freeparsy;
	freeparsy = freeparsy->next;
    }
    else if ((ptr = (PARSY*)malloc(sizeof(PARSY))) == NULL) {
	printf("Error: new_parsy(): failure to allocate enough memory\n");
	exit(1);
    }

	/* set array of pointers to null */
    ptr->next = NULL;
    for (i = 0; i < MAXBINDOBJ; ++i)
	ptr->obarray[i] = NULL;

    return (ptr);
}



/************************************************************************
*
* free_parsy() - Free up a list of parser disabiguation nodes.
*
************************************************************************/

#ifdef _ANSI_
void free_parsy(PARSY *ptr)
#else
free_parsy(ptr)
PARSY *ptr;
#endif
{
    PARSY *tmp;

    while (ptr != NULL) {
	tmp = ptr;
	ptr = ptr->next;
	tmp->next = freeparsy;
	freeparsy = tmp;
    }
}



/************************************************************************
*
* new_string() - Dynamically allocate a new string, and return the
*	pointer to it.
*
************************************************************************/

#ifdef _ANSI_
char *new_string(char *strg)
#else
char *new_string(strg)
char *strg;
#endif
{
    char *ptr;

    if ((ptr = (char*)malloc(strlen(strg)+1)) == NULL) {
	printf("Error: new_string(): failure to allocate enough memory\n");
	exit(1);
    }

    strcpy(ptr, strg);

    return(ptr);
}



/************************************************************************
*
* collect_garbage() - Go through all of the blocks of lisp nodes, marking
*	which are in use and which are free.
*
************************************************************************/

#ifdef _ANSI_
void collect_garbage(int infoflag)
#else
collect_garbage(infoflag)
int infoflag;
#endif
{
    BLOCK *ptr;
    int   nodenum;

	/* reset count of nodes allocated since last collection */
    nodetally = 0;

    if (infoflag)
	printf("Collecting Garbage\n");

	/* loop through all blocks of nodes and mark them as being free */
    ptr = firstblock;
    while (ptr != NULL) {
	ptr->nextcheck = 0;
	for (nodenum = 0; nodenum < BLOCKSIZE; ++nodenum)
	    ptr->nodelist[nodenum].inuse = 0;
	ptr = ptr->nextblock;
    }

	/* go through all currently used nodes and mark them as being
	   in use, so will know which ones are free */
    totalused = 0;
    mark_list(global);
    mark_list(locals);
    mark_list(functions);
    mark_object(objectroot);
    mark_object(classlist);
    mark_action(actionlist);
    mark_daemon(daemonlist);
    freeblock = firstblock;

	/* print out a message about memory usage */
    if (infoflag) {
	printf("Total nodes: %d, Used: %d, Free: %d, Usage: %f\n",
		totalnodes, totalused, totalnodes - totalused,
		(((float)totalused) / ((float)totalnodes)));
    }
}



/************************************************************************
*
* mark_list() - Recursively mark a list of lisp nodes as being in use.
*
************************************************************************/

#ifdef _ANSI_
void mark_list(NODEZ *ptr)
#else
mark_list(ptr)
NODEZ *ptr;
#endif
{
    while (ptr != NULL) {
	if (ptr->inuse == 0) {
	    ptr->inuse = 1;
	    ++totalused;
	}

	if (ptr->type != TYPElisthead)
	    break;

	mark_list(ptr->value.ptr.car);
	ptr = ptr->value.ptr.cdr;
    }
}



/************************************************************************
*
* mark_object() - Mark all lisp nodes used by objects as being in use.
*
************************************************************************/

#ifdef _ANSI_
void mark_object(OBJECT *ptr)
#else
mark_object(ptr)
OBJECT *ptr;
#endif
{
    while (ptr != NULL) {
	mark_list(ptr->classes);
	mark_list(ptr->properties);
	mark_list(ptr->methods);
	mark_object(ptr->child);

	ptr = ptr->sibling;
    }
}



/************************************************************************
*
* mark_action() - Mark all lisp nodes used by action structures.
*
************************************************************************/

#ifdef _ANSI_
void mark_action(ACTION *ptr)
#else
mark_action(ptr)
ACTION *ptr;
#endif
{
    while (ptr != NULL) {
	mark_list(ptr->words);
	mark_list(ptr->before);
	mark_list(ptr->after);

	ptr = ptr->next;
    }
}



/************************************************************************
*
* mark_daemon() - Mark all lisp nodes used by daemons as being in use.
*
************************************************************************/

#ifdef _ANSI_
void mark_daemon(DAEMON *ptr)
#else
mark_daemon(ptr)
DAEMON *ptr;
#endif
{
    while (ptr != NULL) {
	mark_list(ptr->variables);

	ptr = ptr->next;
    }
}



/************************************************************************
*
* new_line() - Clean up the output string buffer, flushing any
*	words not yet printed.
*
************************************************************************/

#ifdef _ANSI_
void new_line(void)
#else
new_line()
#endif
{
    if (outputptr)
	flush_outbuf();

    if (outlength) {
        outlength = 0;
	putchar('\n');
    }
}



/************************************************************************
*
* flush_outbuf() - If the output buffer contains anything in it, display
*	the contents to the screen.  This is called to clean up or at the
*	start of a new word.
*
************************************************************************/

#ifdef _ANSI_
void flush_outbuf(void)
#else
flush_outbuf()
#endif
{
    if (outputptr) {
	if ((outlength + outputptr + 1) > 76) {
	    putchar('\n');
	    outlength = 0;
	}
	else if (outlength) {
	    putchar(' ');
	    ++outlength;
	}

	if (docap && islower(*outputbuffer))
	    *outputbuffer = toupper(*outputbuffer);
	docap = 0;

	printf("%s", outputbuffer);
	outlength += outputptr;
	outputbuffer[0] = '\0';
	outputptr = 0;
    }
}



/************************************************************************
*
* print_string() - Print out a string through the string buffer, so that
*	words will be properly broken across lines.
*
************************************************************************/

#ifdef _ANSI_
void print_string(char *strg)
#else
print_string(strg)
char *strg;
#endif
{
    while (*strg != '\0') {
	if (*strg == '\\') {
	    ++strg;
	    if (*strg == '\0')				/* end of string */
		outputbuffer[outputptr++] = '\0';
	    else if (*strg == 't') {			/* tab */
		flush_outbuf();
		putchar('\t');
		outlength += 8 - (outlength % 8);
		++outlength;
	    }
	    else if (*strg == 'n')			/* newline */
		new_line();
	    else if (*strg == 'b') {			/* blank line */
		new_line();
		putchar('\n');
	    }
	    else if (*strg == 'p') {			/* paragraph */
		new_line();
		printf("     ");
		outlength += 5;
	    }
	    else if (*strg == 'c')			/* cap. next word*/
		docap = 1;
	    else if (*strg == ' ') {			/* space */
		flush_outbuf();
		putchar(' ');
		++outlength;
	    }
	    else					/* else leave as is */
		outputbuffer[outputptr++] = *strg;

	    outputbuffer[outputptr] = '\0';
	}
	else if (*strg == ' ')
	    flush_outbuf();
	else {
	    outputbuffer[outputptr++] = *strg;
	    outputbuffer[outputptr] = '\0';
	}

	++strg;
    }
}


