/*
 * mxOpen.c --
 *	Implementation of the MxOpenCmd procedure that creates
 *	a top-level mx editor window.
 *
 * Copyright (c) 1992 Xerox Corporation.
 * Use and copying of this software and preparation of derivative works based
 * upon this software are permitted. Any distribution of this software or
 * derivative works must comply with all applicable United States export
 * control laws. This software is made available AS IS, and Xerox Corporation
 * makes no warranty about the software, its performance or its conformity to
 * any specification.
 */

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "tk.h"
#include "tkConfig.h"
#include "tkInt.h"
#include "mxWidget.h"

#ifdef TCL_DEBUG
#include "Dbg.h"
#endif

/*
 * This script is ordinarily used to initialize the editor application.
 * This can be overridden with -script option
 */
static char *script = "mxedit.tk";
/*
 * This is the directory where the mxedit scripts are kept.
 * It is defined in the Makefile, plus you can override it on the
 * command line with the -libDir option, or by defining the
 * MXEDIT_LIB environment variable.  The command line arg has
 * precedence over the environment variable.
 */
static char *libDir = MX_LIBRARY;

/*
 * Hook to install the mxedit widget
 */
extern int MxWidgetInstall _ANSI_ARGS_((Tk_Window w, Tcl_Interp *interp));

extern int MxGlobalEval _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]));

extern char *MxEnvironmentSetup _ANSI_ARGS_((Tk_Window tkwin, Tcl_Interp *interp, char *libDir, int argc, char *argv[]));

/*
 * Globals used by main()
 */
extern int mxNumWindows;
extern int mxDontFork;

/*
 * Variables behind command line arguments.
 * Left over arguments are considered names of files to edit.
 */

static int debug = 0;
static int tcldebug = 0;
static int nowin = 0;
static int synchronize = 0;
static char *name = NULL;
static char *display = NULL;
static char *globalCmd = NULL;
static char *initCmd = NULL;
static char *geometry = NULL;
static char *saveGeometry = NULL;
static char *fontName = NULL;
static char *saveFontName = NULL;

Tk_ArgvInfo argTable[] = {
    {"-display", TK_ARGV_STRING, (char *) NULL, (char *) &display,
	"Display to use"},
    {"-script", TK_ARGV_STRING, (char *) NULL, (char *) &script,
	"Initialization file for application"},
    {"-libDir", TK_ARGV_STRING, (char *) NULL, (char *) &libDir,
	"Library directory for application"},
    {"-font", TK_ARGV_STRING, (char *) NULL, (char *) &fontName,
	"Font to use"},
    {"-geometry", TK_ARGV_STRING, (char *) NULL, (char *) &geometry,
	"Geometry for edit window"},
    {"-name", TK_ARGV_STRING, (char *) NULL, (char *) &name,
	"Name to use for application"},
    {"-debug", TK_ARGV_CONSTANT, (char *) 1, (char *) &debug,
	"Set things up for gdb-type debugging (-sync -noFork)"},
    {"-initCmd", TK_ARGV_STRING, (char *) NULL, (char *) &initCmd,
	"Initial command for per-window interpreter"},
    {"-globalCmd", TK_ARGV_STRING, (char *) NULL, (char *) &globalCmd,
	"Initial command for global interpreter"},
    {"-nowin", TK_ARGV_CONSTANT, (char *) NULL, (char *) &nowin,
	"Do not display the global window"},
    {"-tcldebug", TK_ARGV_CONSTANT, (char *) 1, (char *) &tcldebug,
	"Set things up for tcl-debug (also -sync -noFork)"},
    {"-sync", TK_ARGV_CONSTANT, (char *) 1, (char *) &synchronize,
	"Use synchronous mode for display server"},
    {"-noFork", TK_ARGV_CONSTANT, (char *) 1, (char *) &mxDontFork,
	"Do not detach into the background"},
    {"-D", TK_ARGV_CONSTANT, (char *) 1, (char *) &mxDontFork,
	"Same as -noFork"},
    {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL,
	(char *) NULL}
};

/*
 * StructureProc --
 *
 * This procedure is called when the main window changes.
 */
	/* ARGSUSED */
static void
StructureProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    extern int mxNumWindows;

    if (eventPtr->type == DestroyNotify) {
	mxNumWindows--;
    }
}

/*
 * MxErrorProc --
 *	Toe-hold for debugging XProtocol botches.
 */
int
MxErrorProc(w, errEventPtr)
    Tk_Window w;		/* ClientData is main window */
    XErrorEvent *errEventPtr;
{
    fprintf(stderr, "mxedit X error: error=%d request=%d minor=%d\n",
	    errEventPtr->error_code, errEventPtr->request_code,
	    errEventPtr->minor_code);
    return 0;		/* Claim to have handled the error */
}


/*
 * MxOpenCmd --
 *	Create a new top-level window for a view on a file.  This
 *	creates a new interpreter context for the view so that the
 *	set of widgets composed around the editing window can most
 *	easily communicate.  This sources the initialization scripts
 *	and calls a single initialization proc to set things up.
 */
	/* ARGSUSED */
int
MxOpenCmd(clientData, oldInterp, argc, argv)
    ClientData clientData;		/* process global interpreter */
    Tcl_Interp *oldInterp;		/* Could be NULL */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    Tk_Window tkwin;
    Tcl_Interp *interp;		/* New interpreter for this window */
    Tk_3DBorder border;
    char *args, *p;
    struct stat statBuf;
    char buf[200];

    /*
     * We'll create a new interpreter context for this new window.
     */
    interp = Tcl_CreateInterp();
    /*
     * Create the mxopen command,
     * which maps back to this procedure.
     */
    Tcl_CreateCommand(interp, "mxOpen", MxOpenCmd, (ClientData) clientData,
	    (void (*)()) NULL);
    /*
     * Create mxGlobalEval that executes stuff in the global interpreter.
     */
    Tcl_CreateCommand(interp, "mxGlobalEval", MxGlobalEval,
	    (ClientData) clientData, (void (*)()) NULL);
    /*
     * Deal with arguments.
     */
#ifdef TCL_DEBUG
    /*
     * Save a copy of argc, argv
     */
    Dbg_ArgcArgv(argc, argv, 1);
#endif
    name = NULL;
    fontName = NULL;
    geometry = NULL;
    if (Tk_ParseArgv(interp, (Tk_Window) NULL, &argc, argv, argTable, 0)
	    != TCL_OK) {
	fprintf(stderr, "%s\n", interp->result);
	exit(1);
    }
    if (debug || tcldebug) {
	synchronize = 1;
	mxDontFork = 1;
    }
    /*
     * Inherit font and geometry from existing window, if it exists.
     */
    if (fontName == NULL) {
	if (oldInterp != NULL) {
	    fontName = Tcl_GetVar(oldInterp, "mxFont", TCL_GLOBAL_ONLY);
	}
	if (fontName == NULL) {
	    fontName = "fixed";
	}
    }
    Tcl_SetVar(interp, "mxFont", fontName, TCL_GLOBAL_ONLY);

    if (geometry == NULL) {
	if (oldInterp != NULL) {
	    geometry = Tcl_GetVar(oldInterp, "mxGeometry", TCL_GLOBAL_ONLY);
	}
	if (geometry == NULL) {
	    geometry = "80x25";
	}
    }
    Tcl_SetVar(interp, "mxGeometry", geometry, TCL_GLOBAL_ONLY);

    /*
     * All left-over arguments are going to be interpreted
     * by mxInit as filenames.  We'll pick off the first one
     * to use in the interp/application name
     */
    if (name == NULL) {
	if (argv[1] == NULL) {
	    name = malloc(strlen("mxedit ") + strlen("tutorial") + 1);
	    sprintf(name, "mxedit %s", "tutorial");
	} else if (strlen(argv[1]) == 0) {
	    static scratchCounter = 0;
	    scratchCounter++;
	    name = malloc(strlen("mxedit ") + strlen("scratch") + 10);
	    sprintf(name, "mxedit %s-%d", "scratch", scratchCounter);	    
	} else {
	    name = malloc(strlen("mxedit ") + strlen(argv[1]) + 1);
	    sprintf(name, "mxedit %s", argv[1]);
	}
    }
    /*
     * Create a new, top-level X window that we'll fill out with widgets later.
     */
    tkwin = Tk_CreateMainWindow(interp, display, name);
    if (tkwin == NULL) {
	fprintf(stderr, "%s\n", interp->result);
	exit(1);
    }
    free(name);

    args = MxEnvironmentSetup(tkwin, interp, libDir, argc, argv);

    Tk_CreateEventHandler(tkwin, StructureNotifyMask, StructureProc,
	    (ClientData) NULL);
    Tk_CreateErrorHandler(Tk_Display(tkwin), -1, -1, -1, MxErrorProc,
	    (ClientData)tkwin);
    if (synchronize) {
	XSynchronize(Tk_Display(tkwin), True);
    }

    Tk_GeometryRequest(tkwin, 200, 200);
    border = Tk_Get3DBorder(interp, tkwin, None, "white");
    if (border == NULL) {
	Tk_SetWindowBackground(tkwin, WhitePixelOfScreen(Tk_Screen(tkwin)));
    } else {
	Tk_SetBackgroundFromBorder(tkwin, border);
    }
    XSetForeground(Tk_Display(tkwin), DefaultGCOfScreen(Tk_Screen(tkwin)),
	    BlackPixelOfScreen(Tk_Screen(tkwin)));
    /*
     * Register the mxedit TCL command that creates mx widgets.
     */
    mxNumWindows++;
    MxWidgetInstall(tkwin, interp);
#ifdef TCL_DEBUG
    /* 
     * Enter the debugger if desired.
     */
    if (tcldebug) {
	Dbg_On(interp, 0);
    }
#endif
    /*
     * Call out to the script-level to initialize the window
     * with all of its widgets.
     */

    sprintf(buf, "source $mxLibrary/%s", script);
    if (Tcl_Eval(interp, buf, 0, (char **) NULL) != TCL_OK) {
	    fprintf(stderr, "Init script \"%s\" failed: %s\n",
		    script, interp->result);
	    fprintf(stderr, "**** TCL trace ****\n");
	    fprintf(stderr, "%s\n", Tcl_GetVar(interp, "errorInfo",
		TCL_GLOBAL_ONLY));
	    return TCL_ERROR;
    }
    sprintf(buf, "mxInit %s %s %s", fontName, geometry, args);
    if (Tcl_Eval(interp, buf, 0, (char **) NULL) != TCL_OK) {
	if ((strncmp(interp->result, "Active:", 7) == 0) ||
	    (strncmp(interp->result, "command aborted", 15) == 0)) {
	    /*
	     * Hit other editing session - just bail.
	     */
	} else {
	    fprintf(stderr, "mxInit %s failed: %s\n", args, interp->result);
	    fprintf(stderr, "**** TCL trace ****\n");
	    fprintf(stderr, "%s\n", Tcl_GetVar(interp, "errorInfo",
		TCL_GLOBAL_ONLY));
	}
	if (oldInterp != NULL) {
	    Tcl_SetResult(oldInterp, interp->result, TCL_VOLATILE);
	}
	Tk_DestroyWindow(tkwin);
	return TCL_ERROR;
    }
    /*
     * Run the user-supplied, per-window command.
     */
    if (initCmd != NULL) {
	if (Tcl_Eval(interp, initCmd, 0, (char **)NULL) != TCL_OK) {
	    fprintf(stderr, "initCmd \"%s\" failed: %s\n",
		    initCmd, interp->result);
	    fprintf(stderr, "**** TCL trace ****\n");
	    fprintf(stderr, "%s\n", Tcl_GetVar(interp, "errorInfo",
		TCL_GLOBAL_ONLY));
	}
    }
    ckfree(args);
    /*
     * Force update of the image, then map it onto the screen.
     */
    strcpy(buf, "update");
    (void) Tcl_Eval(interp, buf, 0, (char **) NULL);
    Tk_MapWindow(tkwin);

    /*
     * Update state about editing sessions.
     */
    MxManagerRecordSession(oldInterp, interp, display);
    MxManagerRecordOpen(interp, display, Tk_Name(tkwin));

    if (oldInterp != NULL) {
	Tcl_SetResult(oldInterp, Tk_Name(tkwin), TCL_STATIC);
    }
    return TCL_OK;
}


