(* bug429.sml *)

signature WR = sig

type Wr

val to_stdout: unit -> Wr
val to_file: string -> Wr
val to_nowhere: unit -> Wr
val to_wrs: Wr list -> Wr
val to_fn: (string->unit) -> (unit->unit) -> Wr
val to_string: unit -> Wr
val extract_str: Wr -> string

val close: Wr -> unit

val write_wr: Wr -> string -> unit

end
signature PP = sig

structure Wr: WR

type Pp

val pp_from_wr: Wr.Wr -> Pp
val wr_from_pp: Pp -> Wr.Wr;

val pwrite : Pp -> string -> unit
val setb: Pp -> unit
val endb: Pp -> unit
val break: Pp -> bool -> int -> unit
val expbreak: Pp -> bool -> string -> unit
val set_margin: Pp -> int -> unit

val DEBUG: bool ref

end
signature WRMGT = sig

(* Maintains a notion of a current (prettyprinting) writer 
   and its associated prettyprinter *)

structure Wr: WR;
structure Pp: PP;
sharing Pp.Wr = Wr;

val set_current_wr: Wr.Wr -> unit;
val get_current_wr: unit -> Wr.Wr;
val stdpp: unit -> Pp.Pp;

val write: string -> unit;

end
signature STRINGUTILS = sig

end
signature REGISTRY = sig

type registeredtype

val register: string -> (registeredtype->unit) -> unit
val registerflag: string -> (registeredtype ref) -> unit

val set_flag: string -> registeredtype -> unit
val set_all: registeredtype -> unit

end
signature LISTUTILS = sig

val memq: ('a -> 'a -> bool) -> 'a list -> 'a -> bool

val mapappend: ('a -> 'b list) -> ('a list) -> ('b list)
val mapunit: ('a -> 'b) -> ('a list) -> unit
val mapunit_tuple: ('a -> unit) -> (unit -> unit) -> ('a list) -> unit

val mapfold: ('a -> 'b) -> ('b -> 'b -> 'b) -> 'b -> ('a list) -> 'b
val forall: ('a -> bool) -> ('a list) -> bool
val forsome: ('a -> bool) -> ('a list) -> bool

val filter: ('a -> bool) -> ('a list) -> ('a list)

end
signature ID = sig

type T 

val intern: string -> T
val tostr: T -> string

val hashcode: T -> int 
val new: unit -> T
val new_from: T -> T

val == : T -> T -> bool

end

(* May eventually want to support these too:

   val lexlt : T -> T -> bool
*)

signature DEBUGUTILS = sig

val wrap: (bool ref) -> string -> (unit -> 'a) -> (unit -> unit) -> ('a -> unit) -> 'a

end
signature GLOBALS = sig

structure Wr: WR
structure Pp: PP
structure WrMgt: WRMGT
structure Id: ID

sharing Pp.Wr = Wr
sharing WrMgt.Pp = Pp

include WRMGT

include LISTUTILS
include STRINGUTILS
include DEBUGUTILS
include REGISTRY

sharing type registeredtype = bool

exception CantHappen

end
signature TYPPVT = sig

structure Globals: GLOBALS
open Globals

datatype pretyp = 
	    PRETVAR of Id.T
	  | PREARROW of pretyp * pretyp
	  | PREALL of Id.T * pretyp * pretyp
	  | PREMEET of pretyp list

datatype T = 
	    TVAR of unit * int
	  | ARROW of unit * T * T
	  | ALL of {name:Id.T} * T * T
	  | MEET of unit * (T list)
	  
datatype tenvelt = BND of Id.T * T
		 | ABB of Id.T * T
		 | VBND of Id.T * T

datatype tenv = TENV of tenvelt list

val empty_tenv: tenv
val push_bound: tenv -> Id.T -> T -> tenv
val push_abbrev: tenv -> Id.T -> T -> tenv
val push_binding: tenv -> Id.T -> T -> tenv
val pop: tenv -> tenv

val index: tenv -> Id.T -> int
val lookup_name: tenv -> int -> Id.T
val lookup_and_relocate_bound: tenv -> int -> T
val lookup_and_relocate_binding: tenv -> int -> T
val lookup_and_relocate: tenv -> int -> tenvelt
val lookup: tenv -> int -> tenvelt
val relocate: int -> T -> T

(* Substitute the first arg for instances of var #0 in the second arg *)
val tsubst_top: T -> T -> T

exception UnknownId of string
exception WrongKindOfId of tenv * int * string
val debruijnify: tenv -> pretyp -> T

val prt: Pp.Pp -> tenv -> T -> unit
val prt_tenv: Pp.Pp -> tenv -> unit

val NS: T

end

signature LEQ = sig

structure Typ: TYPPVT
structure Globals: GLOBALS
sharing Globals = Typ.Globals

val leq: Typ.tenv -> Typ.T -> Typ.T -> bool

end
(* ML-Yacc Parser Generator (c) 1989 Andrew W. Appel, David R. Tarditi *)

(* LR_TABLE: signature for an LR Table.

   The list of actions and gotos passed to mkLrTable must be ordered by state
   number. The values for state 0 are the first in the list, the values for
    state 1 are next, etc.
*)

signature LR_TABLE =
    sig
	datatype state = STATE of int
	datatype term = T of int
	datatype nonterm = NT of int
	datatype action = SHIFT of state
			| REDUCE of int
			| ACCEPT
			| ERROR
	type table
	
	val numStates : table -> int
	val describeActions : table -> state ->
				((term * action) list) * action
	val describeGoto : table -> state -> (nonterm * state) list
	val action : table -> state * term -> action
	val goto : table -> state * nonterm -> state
	val initialState : table -> state
	exception Goto of state * nonterm

	val mkLrTable : {actions : (((term * action) list) * action) list,
			 gotos : (nonterm * state) list list,
			 numStates : int,
			 initialState : state} -> table
    end

(* ML-Yacc Parser Generator (c) 1989 Andrew W. Appel, David R. Tarditi *)

(* import "lr_table.sig"; *)

(* TOKEN: signature revealing the internal structure of a token. This signature
   TOKEN distinct from the signature {parser name}_TOKENS produced by ML-Yacc.
   The {parser name}_TOKENS structures contain some types and functions to
    construct tokens from values and positions.

   The representation of token was very carefully chosen here to allow the
   polymorphic parser to work without knowing the types of semantic values
   or line numbers.

   This has had an impact on the TOKENS structure produced by SML-Yacc, which
   is a structure parameter to lexer functors.  We would like to have some
   type 'a token which functions to construct tokens would create.  A
   constructor function for a integer token might be

	  INT: int * 'a * 'a -> 'a token.
 
   This is not possible because we need to have tokens with the representation
   given below for the polymorphic parser.

   Thus our constructur functions for tokens have the form:

	  INT: int * 'a * 'a -> (svalue,'a) token

   This in turn has had an impact on the signature that lexers for SML-Yacc
   must match and the types that a user must declare in the user declarations
   section of lexers.
*)

signature TOKEN =
    sig
	structure LrTable : LR_TABLE
        datatype ('a,'b) token = TOKEN of LrTable.term * ('a * 'b * 'b)
	val sameToken : ('a,'b) token * ('a,'b) token -> bool
    end

(* ML-Yacc Parser Generator (c) 1989 Andrew W. Appel, David R. Tarditi *)

(*
import "lr_table.sig";
import "token.sig";
*)

(* PARSER_DATA: the signature of ParserData structures in {parser name}LrValsFun
   produced by  SML-Yacc.  All such structures match this signature.  

   The {parser name}LrValsFun produces a structure which contains all the values
   except for the lexer needed to call the polymorphic parser mentioned
   before.

*)

signature PARSER_DATA =
   sig
        (* the type of line numbers *)

	type pos

	(* the type of semantic values *)

	type svalue

         (* the type of the user-supplied argument to the parser *)
 	type arg
 
	(* the intended type of the result of the parser.  This value is
	   produced by applying extract from the structure Actions to the
	   final semantic value resultiing from a parse.
	 *)

	type result

	structure LrTable : LR_TABLE
	structure Token : TOKEN
	sharing Token.LrTable = LrTable

	(* structure Actions contains the functions which mantain the
	   semantic values stack in the parser.  Void is used to provide
	   a default value for the semantic stack.
	 *)

	structure Actions : 
	  sig
	      val actions : int * pos *
		   (LrTable.state * (svalue * pos * pos)) list * arg->
		         LrTable.nonterm * (svalue * pos * pos) *
			 ((LrTable.state *(svalue * pos * pos)) list)
	      val void : svalue
	      val extract : svalue -> result
	  end

	(* structure EC contains information used to improve error
	   recovery in an error-correcting parser *)

	structure EC :
	   sig
     val is_keyword : LrTable.term -> bool
	     val noShift : LrTable.term -> bool
	     val preferred_subst : LrTable.term -> LrTable.term list
	     val preferred_insert : LrTable.term -> bool
	     val errtermvalue : LrTable.term -> svalue
	     val showTerminal : LrTable.term -> string
	     val terms: LrTable.term list
	   end

	(* table is the LR table for the parser *)

	val table : LrTable.table
    end

signature FMEET_TOKENS =
sig
type ('a,'b) token
type svalue
val T_PACK: ('a * 'a) ->(svalue,'a) token
val T_END: ('a * 'a) ->(svalue,'a) token
val T_OPEN: ('a * 'a) ->(svalue,'a) token
val T_SOME: ('a * 'a) ->(svalue,'a) token
val T_INSTALL: ('a * 'a) ->(svalue,'a) token
val T_OBSERVE: ('a * 'a) ->(svalue,'a) token
val T_FOR: ('a * 'a) ->(svalue,'a) token
val T_OF: ('a * 'a) ->(svalue,'a) token
val T_CASE: ('a * 'a) ->(svalue,'a) token
val T_NS: ('a * 'a) ->(svalue,'a) token
val T_IN: ('a * 'a) ->(svalue,'a) token
val T_ALL: ('a * 'a) ->(svalue,'a) token
val T_WITH: ('a * 'a) ->(svalue,'a) token
val T_CHECK: ('a * 'a) ->(svalue,'a) token
val T_DEBUG: ('a * 'a) ->(svalue,'a) token
val T_RESET: ('a * 'a) ->(svalue,'a) token
val T_SET: ('a * 'a) ->(svalue,'a) token
val T_TYPE: ('a * 'a) ->(svalue,'a) token
val T_USE: ('a * 'a) ->(svalue,'a) token
val T_STR_CONST: ((string) * 'a * 'a) ->(svalue,'a) token
val T_INT_CONST: ((string) * 'a * 'a) ->(svalue,'a) token
val T_ID: ((string) * 'a * 'a) ->(svalue,'a) token
val T_BIGLAMBDA: ('a * 'a) ->(svalue,'a) token
val T_LAMBDA: ('a * 'a) ->(svalue,'a) token
val T_INTER: ('a * 'a) ->(svalue,'a) token
val T_RCURLY: ('a * 'a) ->(svalue,'a) token
val T_LCURLY: ('a * 'a) ->(svalue,'a) token
val T_RANGLE: ('a * 'a) ->(svalue,'a) token
val T_LANGLE: ('a * 'a) ->(svalue,'a) token
val T_RBRACK: ('a * 'a) ->(svalue,'a) token
val T_LBRACK: ('a * 'a) ->(svalue,'a) token
val T_RPAREN: ('a * 'a) ->(svalue,'a) token
val T_LPAREN: ('a * 'a) ->(svalue,'a) token
val T_DARROW: ('a * 'a) ->(svalue,'a) token
val T_ARROW: ('a * 'a) ->(svalue,'a) token
val T_AT: ('a * 'a) ->(svalue,'a) token
val T_DOLLAR: ('a * 'a) ->(svalue,'a) token
val T_DOUBLEEQ: ('a * 'a) ->(svalue,'a) token
val T_EQ: ('a * 'a) ->(svalue,'a) token
val T_APOST: ('a * 'a) ->(svalue,'a) token
val T_COMMA: ('a * 'a) ->(svalue,'a) token
val T_LEQ: ('a * 'a) ->(svalue,'a) token
val T_SEMICOLON: ('a * 'a) ->(svalue,'a) token
val T_COLON: ('a * 'a) ->(svalue,'a) token
val T_DOT: ('a * 'a) ->(svalue,'a) token
val T_EOF: ('a * 'a) ->(svalue,'a) token
end
signature FMEET_LRVALS =
sig
structure Tokens : FMEET_TOKENS
structure ParserData:PARSER_DATA
sharing type ParserData.Token.token = Tokens.token
sharing type ParserData.svalue = Tokens.svalue
end
(* Externally visible aspects of the lexer and parser *)

signature INTERFACE =
sig

type pos
val line : pos ref
val init_line : unit -> unit
val next_line : unit -> unit
val error : string * pos * pos -> unit

end  (* signature INTERFACE *)

signature TYP = sig

structure Globals: GLOBALS
open Globals

datatype pretyp = 
	    PRETVAR of Id.T
	  | PREARROW of pretyp * pretyp
	  | PREALL of Id.T * pretyp * pretyp
	  | PREMEET of pretyp list

type T

type tenv
val empty_tenv: tenv
val push_bound: tenv -> Id.T -> T -> tenv
val push_abbrev: tenv -> Id.T -> T -> tenv
val push_binding: tenv -> Id.T -> T -> tenv
val pop: tenv -> tenv

exception UnknownId of string
exception WrongKindOfId of tenv * int * string
val debruijnify: tenv -> pretyp -> T

val prt: Pp.Pp -> tenv -> T -> unit
val prt_tenv: Pp.Pp -> tenv -> unit

val NS: T

end

signature TRM = sig

structure Globals: GLOBALS
structure Typ: TYP
sharing Typ.Globals = Globals
open Globals

datatype pretrm = 
	    PREVAR of Id.T
	  | PREABS of Id.T * Typ.pretyp * pretrm
	  | PREAPP of pretrm * pretrm
	  | PRETABS of Id.T * Typ.pretyp * pretrm
	  | PRETAPP of pretrm * Typ.pretyp
	  | PREFOR of Id.T * (Typ.pretyp list) * pretrm

type T

exception UnknownId of string
val debruijnify: Typ.tenv -> pretrm -> T

val prt: Pp.Pp -> Typ.tenv -> T -> unit

end

signature PARSERES = sig

structure Typ : TYP
structure Trm : TRM
structure Globals: GLOBALS
sharing Typ.Globals = Globals
sharing Trm.Typ = Typ

datatype T =
    Leq of Typ.pretyp * Typ.pretyp
  | Type_Assumption of Globals.Id.T * Typ.pretyp
  | Type_Abbrev of Globals.Id.T * Typ.pretyp
  | Term_Def of Globals.Id.T * Trm.pretrm
  | Term_Assumption of Globals.Id.T * Typ.pretyp
  | Use of string
  | Set of string * string
  | Nothing

end 
signature PARSE =
sig

structure ParseRes : PARSERES

val file_parse: string -> ParseRes.T;
val stream_parse: instream -> ParseRes.T;
val top_parse: unit -> ParseRes.T;

end  (* signature PARSE *)

(* ML-Yacc Parser Generator (c) 1989 Andrew W. Appel, David R. Tarditi *)

(* STREAM: signature for a lazy stream.*)

signature STREAM =
 sig type 'xa stream
     val streamify : (unit -> '_a) -> '_a stream
     val cons : '_a * '_a stream -> '_a stream
     val get : '_a stream -> '_a * '_a stream
 end

(* ML-Yacc Parser Generator (c) 1989 Andrew W. Appel, David R. Tarditi *)

(*
import "token.sig";
import "stream.sig";
*)

(* signature PARSER is the signature that most user parsers created by 
   SML-Yacc will match.
*)

signature PARSER =
    sig
        structure Token : TOKEN
	structure Stream : STREAM
	exception ParseError

	(* type pos is the type of line numbers *)

	type pos

	(* type result is the type of the result from the parser *)

	type result

         (* the type of the user-supplied argument to the parser *)
 	type arg
	
	(* type svalue is the type of semantic values for the semantic value
	   stack
	 *)

	type svalue

	(* val makeLexer is used to create a stream of tokens for the parser *)

	val makeLexer : (int -> string) ->
			 (svalue,pos) Token.token Stream.stream

	(* val parse takes a stream of tokens and a function to prt
	   errors and returns a value of type result and a stream containing
	   the unused tokens
	 *)

	val parse : int * ((svalue,pos) Token.token Stream.stream) *
		    (string * pos * pos -> unit) * arg ->
				result * (svalue,pos) Token.token Stream.stream

	val sameToken : (svalue,pos) Token.token * (svalue,pos) Token.token ->
				bool
     end

functor Parse (structure Globals : GLOBALS
	       structure ParseRes : PARSERES
	       structure Interface : INTERFACE
	       structure Parser : PARSER
	          sharing type Parser.pos = Interface.pos
		  sharing type Parser.result = ParseRes.T
		  sharing type Parser.arg = unit
	       structure Tokens : FMEET_TOKENS
	          sharing type Tokens.token = Parser.Token.token
		  sharing type Tokens.svalue = Parser.svalue
               ) : PARSE =
struct

structure ParseRes = ParseRes
open Globals

val parse = fn (lookahead,reader : int -> string) =>
    let val _ = Interface.init_line()
	val empty = !Interface.line
	val dummyEOF = Tokens.T_EOF(empty,empty)
	fun invoke lexer = 
	   Parser.parse(lookahead,lexer,Interface.error,())
        fun loop lexer =
	  let val (result,lexer) = invoke lexer
	      val (nextToken,lexer) = Parser.Stream.get lexer
	  in if Parser.sameToken(nextToken,dummyEOF) then result
	     else loop lexer
	  end
     in loop (Parser.makeLexer reader)
     end

fun string_reader s =
 let val next = ref s
 in fn _ => !next before next := ""
 end
    
val string_parse = fn s => parse (0, string_reader s)

val file_parse = fn name =>
  let val dev = open_in name
   in (parse (15,(fn i => input(dev,i)))) before close_in dev
   end

fun prefix line len = substring(line,0,min(len,size line))    

fun echo_line line =
    if (line = "\n") orelse (line="")
       then write line
    else if prefix line 3 = "%% "
       then write (substring(line,3,size(line)-3))
    else if prefix line 2 = "%%"
       then write (substring(line,2,size(line)-2))
    else write ("> " ^ line)

fun convert_tabs s =
    implode (map (fn "\t" => "        " | s => s) (explode s));

fun stream_parse dev =
   parse (15,(fn i => 
   		 let val line = convert_tabs(input_line(dev))
   		     val _ = echo_line line
   		 in line
   		 end))

val top_parse = fn () => parse (0,
		let val not_first_flag = ref(false)
		in fn i => (( if (!not_first_flag)
			     then (write "> "; flush_out std_out)
			     else not_first_flag := true );
			    input_line std_in)
		end)

end  (* functor Parse *)

signature SYNTH = sig

structure Globals: GLOBALS
structure Trm: TRM
structure Typ: TYP
structure Leq: LEQ
sharing Trm.Typ = Typ
    and Leq.Typ = Typ
    and Typ.Globals = Globals
open Globals

val synth: Typ.tenv -> Trm.T -> Typ.T

end
(* Copyright 1989 by AT&T Bell Laboratories *)
(* util/strghash.sml *)

(* Functorized by BCP, 1991 *)

functor StrgHash() =
struct

  val prime = 8388593 (* largest prime less than 2^23 *)
  val base = 128

(* the simple version --
    fun hashString(str: string) : int =
        let fun loop (0,n) = n
	      | loop (i,n) = 
	          let val i = i-1
		      val n' = (base * n + ordof(str,i)) 
		   in loop (i, (n' - prime * (n' quot prime)))
		  end
	 in loop (size str,0)
	end
*)

  fun hashString(str: string) : int =
      let val l = size str
       in case l
	    of 0 => 0
	     | 1 => ord str
	     | 2 => ordof(str,0) + base * ordof(str,1)
	     | 3 => ordof(str,0) + base * (ordof(str,1) + base * ordof(str,2))
	     | _ =>
		let fun loop (0,n) = n
		      | loop (i,n) = 
			  let val i = i-1
			      val n' = (base * n + ordof(str,i)) 
			   in loop (i, (n' - prime * (n' quot prime)))
			  end
		 in loop (l,0)
		end
      end

end (* structure StrgHash *)
functor StringUtils() : STRINGUTILS = struct

end
(* ML-Yacc Parser Generator (c) 1989 Andrew W. Appel, David R. Tarditi *)

(* LEXER: a signature that most lexers produced for use with SML-Yacc's
   outut will match.  The user is responsible for declaring type token,
   type pos, and type svalue in the UserDeclarations section of a lexer.

   Note that type token is abstract in the lexer.  This allows SML-Yacc to
   create a TOKENS signature for use with lexers produced by ML-Lex that
   treats the type token abstractly.  Lexers that are functors parametrized by
   a Tokens structure matching a TOKENS signature cannot examine the structure
   of tokens.
*)

signature LEXER =
   sig
       structure UserDeclarations :
	   sig
	        type ('a,'b) token
		type pos
		type svalue
	   end
	val makeLexer : (int -> string) -> unit -> 
         (UserDeclarations.svalue,UserDeclarations.pos) UserDeclarations.token
   end

functor FMEETLexFun(structure Tokens: FMEET_TOKENS structure Interface: INTERFACE) : LEXER=
   struct
    structure UserDeclarations =
      struct
structure Tokens = Tokens
structure Interface = Interface
open Interface

type pos = Interface.pos
type svalue = Tokens.svalue
type ('a,'b) token = ('a,'b) Tokens.token
type lexresult= (svalue,pos) token

val eof = fn () => Tokens.T_EOF(!line,!line)

val str_begin = ref(!line);
val str_const = ref([]:string list);

end (* end of user routines *)
exception LexError (* raised if illegal leaf action tried *)
structure Internal =
	struct

datatype yyfinstate = N of int
type statedata = {fin : yyfinstate list, trans: string}
(* transition & final state table *)
val tab = let
val s0 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000"
val s1 =
"\007\007\007\007\007\007\007\007\007\097\099\007\007\007\007\007\
\\007\007\007\007\007\007\007\007\007\007\007\007\007\007\007\007\
\\097\007\007\007\096\095\007\094\093\092\007\007\091\089\088\086\
\\084\084\084\084\084\084\084\084\084\084\083\082\080\077\076\007\
\\075\072\010\010\010\010\010\010\010\010\010\010\010\010\070\010\
\\010\010\010\066\010\010\010\010\010\010\010\065\063\062\007\007\
\\061\010\010\053\048\045\042\010\010\040\010\010\010\010\010\035\
\\031\010\026\023\019\016\010\012\010\010\010\009\007\008\007\007\
\\007"
val s3 =
"\100\100\100\100\100\100\100\100\100\100\101\100\100\100\100\100\
\\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\
\\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\
\\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\
\\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\
\\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\
\\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\
\\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\100\
\\100"
val s5 =
"\102\102\102\102\102\102\102\102\102\102\104\102\102\102\102\102\
\\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\
\\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\
\\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\
\\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\
\\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\
\\103\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\
\\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\102\
\\102"
val s10 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s12 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\013\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s13 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\014\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s14 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\015\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s16 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\017\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s17 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\018\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s19 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\020\011\000\000\000\000\000\
\\000"
val s20 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\021\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s21 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\022\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s23 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\024\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s24 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\025\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s26 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\027\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s27 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\028\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s28 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\029\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s29 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\030\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s31 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\032\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s32 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\033\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s33 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\034\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s35 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\039\011\011\011\011\011\011\011\011\011\
\\036\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s36 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\037\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s37 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\038\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s40 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\041\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s42 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\043\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s43 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\044\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s45 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\046\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s46 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\047\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s48 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\049\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s49 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\050\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s50 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\051\011\011\011\011\011\000\000\000\000\000\
\\000"
val s51 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\052\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s53 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\058\011\011\011\011\011\011\054\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s54 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\055\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s55 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\056\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s56 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\057\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s58 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\059\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s59 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\060\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s63 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\064\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000"
val s66 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\067\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s67 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\068\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s68 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\069\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s70 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\071\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s72 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\073\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s73 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\011\000\000\000\000\000\000\000\000\
\\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\000\
\\000\011\011\011\011\011\011\011\011\011\011\011\011\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\011\
\\000\011\011\011\011\011\011\011\011\011\011\011\074\011\011\011\
\\011\011\011\011\011\011\011\011\011\011\011\000\000\000\000\000\
\\000"
val s77 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\079\078\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000"
val s80 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\081\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000"
val s84 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\085\085\085\085\085\085\085\085\085\085\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000"
val s86 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\087\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000"
val s89 =
"\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\090\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000"
val s97 =
"\000\000\000\000\000\000\000\000\000\098\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\098\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
\\000"
in arrayoflist
[{fin = [], trans = s0},
{fin = [(N 32)], trans = s1},
{fin = [(N 32)], trans = s1},
{fin = [], trans = s3},
{fin = [], trans = s3},
{fin = [], trans = s5},
{fin = [], trans = s5},
{fin = [(N 144)], trans = s0},
{fin = [(N 80),(N 144)], trans = s0},
{fin = [(N 78),(N 144)], trans = s0},
{fin = [(N 137),(N 144)], trans = s10},
{fin = [(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s12},
{fin = [(N 137)], trans = s13},
{fin = [(N 137)], trans = s14},
{fin = [(N 91),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s16},
{fin = [(N 137)], trans = s17},
{fin = [(N 3),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s19},
{fin = [(N 137)], trans = s20},
{fin = [(N 137)], trans = s21},
{fin = [(N 8),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s23},
{fin = [(N 137)], trans = s24},
{fin = [(N 12),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s26},
{fin = [(N 137)], trans = s27},
{fin = [(N 137)], trans = s28},
{fin = [(N 137)], trans = s29},
{fin = [(N 18),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s31},
{fin = [(N 137)], trans = s32},
{fin = [(N 137)], trans = s33},
{fin = [(N 122),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s35},
{fin = [(N 137)], trans = s36},
{fin = [(N 137)], trans = s37},
{fin = [(N 117),(N 137)], trans = s10},
{fin = [(N 99),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s40},
{fin = [(N 129),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s42},
{fin = [(N 137)], trans = s43},
{fin = [(N 103),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s45},
{fin = [(N 137)], trans = s46},
{fin = [(N 126),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s48},
{fin = [(N 137)], trans = s49},
{fin = [(N 137)], trans = s50},
{fin = [(N 137)], trans = s51},
{fin = [(N 24),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s53},
{fin = [(N 137)], trans = s54},
{fin = [(N 137)], trans = s55},
{fin = [(N 137)], trans = s56},
{fin = [(N 30),(N 137)], trans = s10},
{fin = [(N 137)], trans = s58},
{fin = [(N 137)], trans = s59},
{fin = [(N 96),(N 137)], trans = s10},
{fin = [(N 142),(N 144)], trans = s0},
{fin = [(N 72),(N 144)], trans = s0},
{fin = [(N 134),(N 144)], trans = s63},
{fin = [(N 132)], trans = s0},
{fin = [(N 70),(N 144)], trans = s0},
{fin = [(N 137),(N 144)], trans = s66},
{fin = [(N 137)], trans = s67},
{fin = [(N 137)], trans = s68},
{fin = [(N 112),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s70},
{fin = [(N 86),(N 137)], trans = s10},
{fin = [(N 137),(N 144)], trans = s72},
{fin = [(N 137)], trans = s73},
{fin = [(N 107),(N 137)], trans = s10},
{fin = [(N 42),(N 144)], trans = s0},
{fin = [(N 76),(N 144)], trans = s0},
{fin = [(N 52),(N 144)], trans = s77},
{fin = [(N 61)], trans = s0},
{fin = [(N 55)], trans = s0},
{fin = [(N 74),(N 144)], trans = s80},
{fin = [(N 58)], trans = s0},
{fin = [(N 44),(N 144)], trans = s0},
{fin = [(N 38),(N 144)], trans = s0},
{fin = [(N 140),(N 144)], trans = s84},
{fin = [(N 140)], trans = s84},
{fin = [(N 144)], trans = s86},
{fin = [(N 64)], trans = s0},
{fin = [(N 46),(N 144)], trans = s0},
{fin = [(N 144)], trans = s89},
{fin = [(N 83)], trans = s0},
{fin = [(N 48),(N 144)], trans = s0},
{fin = [(N 68),(N 144)], trans = s0},
{fin = [(N 66),(N 144)], trans = s0},
{fin = [(N 50),(N 144)], trans = s0},
{fin = [(N 36),(N 144)], trans = s0},
{fin = [(N 40),(N 144)], trans = s0},
{fin = [(N 32),(N 144)], trans = s97},
{fin = [(N 32)], trans = s97},
{fin = [(N 34)], trans = s0},
{fin = [(N 148)], trans = s0},
{fin = [(N 146)], trans = s0},
{fin = [(N 154)], trans = s0},
{fin = [(N 152),(N 154)], trans = s0},
{fin = [(N 150)], trans = s0}]
end
structure StartStates =
	struct
	datatype yystartstate = STARTSTATE of int

(* start state definitions *)

val COMMENT = STARTSTATE 3;
val INITIAL = STARTSTATE 1;
val STRING = STARTSTATE 5;

end
type result = UserDeclarations.lexresult
	exception LexerError (* raised if illegal leaf action tried *)
end

fun makeLexer yyinput = 
let 
	val yyb = ref "\n" 		(* buffer *)
	val yybl = ref 1		(*buffer length *)
	val yybufpos = ref 1		(* location of next character to use *)
	val yygone = ref 1		(* position in file of beginning of buffer *)
	val yydone = ref false		(* eof found yet? *)
	val yybegin = ref 1		(*Current 'start state' for lexer *)

	val YYBEGIN = fn (Internal.StartStates.STARTSTATE x) =>
		 yybegin := x

fun lex () : Internal.result =
let fun continue() = lex() in
  let fun scan (s,AcceptingLeaves : Internal.yyfinstate list list,l,i0) =
	let fun action (i,nil) = raise LexError
	| action (i,nil::l) = action (i-1,l)
	| action (i,(node::acts)::l) =
		case node of
		    Internal.N yyk => 
			(let val yytext = substring(!yyb,i0,i-i0)
			     val yypos = i0+ !yygone
			open UserDeclarations Internal.StartStates
 in (yybufpos := i; case yyk of 

			(* Application actions *)

  103 => (Tokens.T_FOR(!line,!line))
| 107 => (Tokens.T_ALL(!line,!line))
| 112 => (Tokens.T_SOME(!line,!line))
| 117 => (Tokens.T_OPEN(!line,!line))
| 12 => (Tokens.T_SET(!line,!line))
| 122 => (Tokens.T_PACK(!line,!line))
| 126 => (Tokens.T_END (!line,!line))
| 129 => (Tokens.T_IN(!line,!line))
| 132 => (Tokens.T_BIGLAMBDA(!line,!line))
| 134 => (Tokens.T_LAMBDA(!line,!line))
| 137 => (Tokens.T_ID (yytext,!line,!line))
| 140 => (Tokens.T_INT_CONST (yytext,!line,!line))
| 142 => (str_begin:=(!line); str_const:=[]; YYBEGIN STRING; lex())
| 144 => (error ("ignoring illegal character" ^ yytext,
			   !line,!line); lex())
| 146 => (next_line(); YYBEGIN INITIAL; lex())
| 148 => (lex())
| 150 => (next_line(); lex())
| 152 => (YYBEGIN INITIAL;
		    Tokens.T_STR_CONST(implode(rev(!str_const)),
		    		       !str_begin,!line))
| 154 => (str_const:=(yytext::(!str_const)); lex())
| 18 => (Tokens.T_RESET(!line,!line))
| 24 => (Tokens.T_DEBUG(!line,!line))
| 3 => (Tokens.T_USE(!line,!line))
| 30 => (Tokens.T_CHECK(!line,!line))
| 32 => (lex())
| 34 => (next_line(); lex())
| 36 => (YYBEGIN COMMENT; lex())
| 38 => (Tokens.T_COLON(!line,!line))
| 40 => (Tokens.T_DOLLAR(!line,!line))
| 42 => (Tokens.T_AT(!line,!line))
| 44 => (Tokens.T_EOF(!line,!line))
| 46 => (Tokens.T_DOT(!line,!line))
| 48 => (Tokens.T_COMMA(!line,!line))
| 50 => (Tokens.T_APOST(!line,!line))
| 52 => (Tokens.T_EQ(!line,!line))
| 55 => (Tokens.T_DOUBLEEQ(!line,!line))
| 58 => (Tokens.T_LEQ(!line,!line))
| 61 => (Tokens.T_DARROW(!line,!line))
| 64 => (Tokens.T_INTER(!line,!line))
| 66 => (Tokens.T_LPAREN(!line,!line))
| 68 => (Tokens.T_RPAREN(!line,!line))
| 70 => (Tokens.T_LBRACK(!line,!line))
| 72 => (Tokens.T_RBRACK(!line,!line))
| 74 => (Tokens.T_LANGLE(!line,!line))
| 76 => (Tokens.T_RANGLE(!line,!line))
| 78 => (Tokens.T_LCURLY(!line,!line))
| 8 => (Tokens.T_TYPE(!line,!line))
| 80 => (Tokens.T_RCURLY(!line,!line))
| 83 => (Tokens.T_ARROW(!line,!line))
| 86 => (Tokens.T_NS(!line,!line))
| 91 => (Tokens.T_WITH(!line,!line))
| 96 => (Tokens.T_CASE(!line,!line))
| 99 => (Tokens.T_OF(!line,!line))
| _ => raise Internal.LexerError

		) end )

	val {fin,trans} = Internal.tab sub s
	val NewAcceptingLeaves = fin::AcceptingLeaves
	in if l = !yybl then
	     if trans = #trans(Internal.tab sub 0)
	       then action(l,NewAcceptingLeaves) else
	    let val newchars= if !yydone then "" else yyinput 1024
	    in if (size newchars)=0
		  then (yydone := true;
		        if (l=i0) then UserDeclarations.eof ()
		                  else action(l,NewAcceptingLeaves))
		  else (if i0=l then yyb := newchars
		     else yyb := substring(!yyb,i0,l-i0)^newchars;
		     yygone := !yygone+i0;
		     yybl := size (!yyb);
		     scan (s,AcceptingLeaves,l-i0,0))
	    end
	  else let val NewChar = ordof(!yyb,l)
		val NewState = if NewChar<128 then ordof(trans,NewChar) else ordof(trans,128)
		in if NewState=0 then action(l,NewAcceptingLeaves)
		else scan(NewState,NewAcceptingLeaves,l+1,i0)
	end
	end
(*
	val start= if substring(!yyb,!yybufpos-1,1)="\n"
then !yybegin+1 else !yybegin
*)
	in scan(!yybegin (* start *),nil,!yybufpos,!yybufpos)
    end
end
  in lex
  end
end

functor Registry(
	    type registeredtype
	    ): REGISTRY = struct

type registeredtype = registeredtype

val registry = ref(nil: (string * (registeredtype->unit)) list)

fun register name callback = 
  registry := (name,callback)::(!registry)

fun registerflag name flagref =
  registry := (name,(fn b => flagref := b))::(!registry)

exception NotRegistered of string

fun set_flag name v = 
  let fun f [] = raise NotRegistered(name)
        | f ((n,callback)::tl) = if name=n 
        		           then (callback v)
        		           else f tl
  in f (!registry)
  end

fun set_all v =
  let fun f [] = ()
        | f ((n,callback)::tl) = (callback v; f tl)
  in f (!registry)
  end

end
functor Typ(
	    structure Globals: GLOBALS
	    ) : TYPPVT
	    = struct

structure Globals = Globals
open Globals
open Pp

datatype pretyp = 
	    PRETVAR of Id.T
	  | PREARROW of pretyp * pretyp
	  | PREALL of Id.T * pretyp * pretyp
	  | PREMEET of pretyp list

datatype T = 
	    TVAR of unit * int
	  | ARROW of unit * T * T
	  | ALL of {name:Id.T} * T * T
	  | MEET of unit * (T list)

type idindex = int

val NS = MEET ((),[])

exception UnknownId of string

datatype tenvelt = BND of Id.T * T
		 | ABB of Id.T * T
		 | VBND of Id.T * T

datatype tenv = TENV of tenvelt list

fun push_bound (TENV(te)) i t = TENV(BND(i,t)::te)

fun push_abbrev (TENV(te)) i t = TENV(ABB(i,t)::te)

fun push_binding (TENV(te)) i t = TENV(VBND(i,t)::te)

val empty_tenv = TENV(nil)

fun index (TENV(bvs)) i =
  let fun ind [] n =
            raise UnknownId(Id.tostr i)
        | ind (BND(i',_)::rest) n =
            if Id.== i i'
               then n
               else ind rest (n+1)
        | ind (VBND(i',_)::rest) n =
            if Id.== i i'
               then n
               else ind rest (n+1)
        | ind (ABB(i',_)::rest) n =
            if Id.== i i'
               then n
               else ind rest (n+1)
  in ind bvs 0
  end

exception TypeVariableOutOfRange of int

fun old_lookup_name (TENV(te)) i = 
  (case (nth (te,i)) of
    BND(name,_) => name
  | VBND(name,_) => name
  | ABB(name,_) => name)
  handle Nth => Id.intern(("<BAD INDEX: " ^ (makestring i) ^ ">"))

fun lookup_name (TENV(te)) i = 
  let fun l [] _ _ = Id.intern(("<BAD INDEX: " ^ (makestring i) ^ ">"))
        | l (hd::tl) rest 0 = 
	    let val name = case hd of BND(n,_) => n | VBND(n,_) => n | ABB(n,_) => n
	    in if memq Id.== rest name 
	         then Id.intern ((Id.tostr name) ^ "^" ^ (makestring i))
	         else name
	    end
        | l (hd::tl) rest j = 
	    let val name = case hd of BND(n,_) => n | VBND(n,_) => n | ABB(n,_) => n
	    in l tl (name::rest) (j-1)
	    end
  in l te [] i
  end

exception WrongKindOfId of tenv * int * string

fun lookup (TENV(te)) i =
  nth (te,i)
  handle Nth => raise TypeVariableOutOfRange(i)

exception TriedToPopEmptyTEnv
fun pop (TENV(hd::tl)) = TENV(tl)
  | pop _ = raise TriedToPopEmptyTEnv

fun inner_relocate offset cutoff t =
  let fun r c (TVAR((),i)) = if i>=c 
  				then TVAR((),i + offset)
  				else TVAR((),i)
        | r c (ARROW((),t1,t2)) = ARROW((), r c t1, r c t2)
        | r c (ALL({name=i},t1,t2)) = ALL({name=i}, r c t1, r (c+1) t2)
        | r c (MEET((),ts)) = MEET((), map (fn t => r c t) ts)
  in r cutoff t
  end

fun relocate offset t = inner_relocate offset 0 t

fun lookup_and_relocate (te) i =
  case lookup te i of
    BND(n,b) => BND(n, relocate (i+1) b)
  | VBND(n,b) => VBND(n, relocate (i+1) b)
  | ABB(n,b) => ABB(n, relocate (i+1) b)

fun lookup_and_relocate_bound te i = 
  case lookup_and_relocate te i of
    BND(_,b) => b
  | VBND(n,_) => raise WrongKindOfId(te,i,"tvar")
  | ABB(n,_) => raise WrongKindOfId(te,i,"tvar")

fun lookup_and_relocate_binding te i = 
  case lookup_and_relocate te i of
    BND(n,b) => raise WrongKindOfId(te,i,"var")
  | VBND(n,b) => b
  | ABB(n,b) => raise WrongKindOfId(te,i,"var")

fun lookup_abbrev te i = 
  case lookup_and_relocate te i of
    BND(n,_) => raise WrongKindOfId(te,i,"tabbrev")
  | VBND(n,b) => raise WrongKindOfId(te,i,"tabbrev")
  | ABB(n,b) => b

fun debruijnify te (PRETVAR i) =
      TVAR((), index te i)
  | debruijnify te (PREARROW (pt1,pt2)) =
      ARROW((), debruijnify te pt1, debruijnify te pt2)
  | debruijnify te (PREALL (i,pt1,pt2)) =
      ALL({name=i}, debruijnify te pt1, debruijnify (push_bound te i NS) pt2)
  | debruijnify te (PREMEET pts) =
      MEET((), map (fn pt => debruijnify te pt) pts)

fun tsubst_top targ tbody =
  let fun s i (t as TVAR(x,i')) = if i = i' 
  				    then relocate i targ
  				  else if i < i'
  				    then TVAR(x,i'-1)
  				  else t
        | s i (ARROW(x,t1,t2)) = ARROW(x, s i t1, s i t2)
        | s i (ALL(x,t1,t2)) = ALL(x, s i t1, s (i+1) t2)
        | s i (MEET(x,ts)) = MEET(x, map (fn t => s i t) ts)
  in s 0 tbody 
  end

fun prt pp te t =
  let fun p te (TVAR(_,i)) =
              Pp.pwrite pp (Id.tostr (lookup_name te i))
        | p te (ARROW(_,t1,t2)) =
              (Pp.pwrite pp "(";
               p te t1;
               Pp.pwrite pp "->";
               p te t2;
               Pp.pwrite pp ")")
        | p te (ALL({name=i},t1,t2)) =
              (Pp.pwrite pp "(All ";
               Pp.pwrite pp (Id.tostr i);
               Pp.pwrite pp "<=";
               p te t1;
               Pp.pwrite pp ". ";
               p (push_bound te i t1) t2;
               Pp.pwrite pp ")")
        | p te (MEET(_,[])) =
               Pp.pwrite pp "NS"
        | p te (MEET(_,ts)) =
              (Pp.pwrite pp "/\\[";
               plist te ts;
               Pp.pwrite pp "]")
      and plist te [] = 
            ()
        | plist te [t] = 
            p te t
        | plist te (hd::tl) = 
            (p te hd; pwrite pp ","; plist te tl)
  in p te t
  end

val short_tenvs = ref(true);
val _ = registerflag "shorttenvs" short_tenvs;

fun prt_tenv pp (TENV(te')) =
  let fun p [] = ()
        | p [(BND(i,t))] = 
            (Pp.pwrite pp (Id.tostr i);
             Pp.pwrite pp "<=";
             prt pp (TENV([])) t)
        | p ((BND(i,t))::tl) = 
            (if (!short_tenvs)
               then pwrite pp "... "
               else p tl; 
             Pp.pwrite pp ", ";
             Pp.break pp false 0;
             Pp.pwrite pp (Id.tostr i);
             Pp.pwrite pp "<=";
             prt pp (TENV(tl)) t)
        | p [(VBND(i,t))] = 
            (Pp.pwrite pp (Id.tostr i);
             Pp.pwrite pp ":";
             prt pp (TENV([])) t)
        | p ((VBND(i,t))::tl) = 
            (if (!short_tenvs)
               then pwrite pp "... "
               else p tl; 
             Pp.pwrite pp ", ";
             Pp.break pp false 0;
             Pp.pwrite pp (Id.tostr i);
             Pp.pwrite pp ":";
             prt pp (TENV(tl)) t)
        | p [(ABB(i,t))] = 
            (Pp.pwrite pp (Id.tostr i);
             Pp.pwrite pp "=";
             prt pp (TENV([])) t)
        | p ((ABB(i,t))::tl) = 
            (if (!short_tenvs)
               then pwrite pp "... "
               else p tl; 
             Pp.pwrite pp ", ";
             Pp.break pp false 0;
             Pp.pwrite pp (Id.tostr i);
             Pp.pwrite pp "=";
             prt pp (TENV(tl)) t)
  in Pp.pwrite pp "{";
     Pp.setb pp;
     p te';
     Pp.endb pp;
     Pp.pwrite pp "}"
  end

end
functor Leq(
	  structure Typ: TYPPVT
	  structure Globals: GLOBALS
	  sharing Typ.Globals = Globals
	  ) : LEQ = struct

structure Typ = Typ
structure Globals = Globals
open Globals
open Typ

datatype lhsqueue = 
	    ARROW_LHS of Typ.T
	  | ALL_LHS   of Id.T * Typ.T

datatype rhs_flag = EXPAND | FIX

val DEBUG = ref(false)
val _ = (registerflag "leq" DEBUG;
	 registerflag "Leq" DEBUG)

fun describe_rest pp te [] t flag = 
      (Pp.pwrite pp "] -> ";
       Typ.prt pp te t;
       case flag of
         EXPAND => Pp.pwrite pp " (EXPAND)?  "
       | FIX    => Pp.pwrite pp " (FIX)?  ")
  | describe_rest pp te [ARROW_LHS(t1)] t2 flag = 
      (Typ.prt pp te t1;
       describe_rest pp te [] t2 flag)
  | describe_rest pp te ((ARROW_LHS(t1))::X2) t2 flag = 
      (Typ.prt pp te t1;
       Pp.pwrite pp ",";
       describe_rest pp te X2 t2 flag)
  | describe_rest pp te [ALL_LHS(v,t1)] t2 flag = 
      (Pp.pwrite pp (Id.tostr v);
       Pp.pwrite pp "<=";
       Typ.prt pp te t1;
       describe_rest pp (push_bound te v t1) [] t2 flag)
  | describe_rest pp te ((ALL_LHS(v,t1))::X2) t2 flag = 
      (Pp.pwrite pp (Id.tostr v);
       Pp.pwrite pp "<=";
       Typ.prt pp te t1;
       Pp.pwrite pp ",";
       describe_rest pp (push_bound te v t1) X2 t2 flag)

fun describe_problem pp te s X t flag =
  (Pp.setb pp;
   Typ.prt pp te s;
   Pp.break pp true ~3;
   Pp.pwrite pp " <= ";
   Pp.pwrite pp "[";
   describe_rest pp te X t flag;
   Pp.endb pp)

fun bindings_in [] = 0
  | bindings_in (ARROW_LHS(_)::tl) = bindings_in tl
  | bindings_in (ALL_LHS(_)::tl) = 1 + (bindings_in tl)

fun leqq' te s X (MEET(_,ts)) EXPAND =
      forall (fn t => leqq te s X t EXPAND) ts
  | leqq' te s X (ARROW(_,t1,t2)) EXPAND =
      leqq te s (X@[ARROW_LHS(t1)]) t2 EXPAND
  | leqq' te s X (ALL({name=i},t1,t2)) EXPAND =
      leqq te s (X@[ALL_LHS(i,t1)]) t2 EXPAND
  | leqq' te s X (t as TVAR(_,vt)) EXPAND =
      let val bx = bindings_in X
      in if vt < bx
         then leqq te s X t FIX
         else case Typ.lookup te (vt - bx) of 
               BND(_,_)  => leqq te s X t FIX
             | VBND(n,_)  => raise Typ.WrongKindOfId(te, vt - bx,"tvar or tabbrev")
             | ABB(_,ab) => leqq te s X (Typ.relocate (vt + bx) ab) EXPAND
      end
  | leqq' te (MEET(_,ss)) X (t as (TVAR(_,vt))) FIX =
      forsome (fn s => leqq te s X t FIX) ss
  | leqq' te (ARROW(_,s1,s2)) (ARROW_LHS(t1)::X) (t as (TVAR(_,vt))) FIX =
      (leqq te t1 [] s1 EXPAND)
      andalso
      (leqq te s2 X t FIX)
  | leqq' te (ALL(_,s1,s2)) (ALL_LHS(i,t1)::X) (t as (TVAR(_,vt))) FIX =
      (leqq (push_bound te i t1) s2 X t FIX)
      andalso
      (leqq te t1 [] s1 EXPAND)
  | leqq' te (TVAR(_,vs)) X (t as (TVAR(_,vt))) FIX =
      (vs = vt andalso (null X))
      orelse
      (case lookup_and_relocate te vs of
         BND(_,bnd) => (leqq te bnd X t FIX)
       | VBND(n,ab)  => raise Typ.WrongKindOfId(te,vs,"tvar or tabbrev")
       | ABB(_,ab)  => (leqq te ab X t FIX))
  | leqq' te s X t flag =
      false

and leqq te s X t flag =
  wrap DEBUG "leqq"
    (fn () => 
      leqq' te s X t flag)
    (fn () => describe_problem (stdpp()) te s X t flag)
    (fn b => write (if b then "Yes" else "No"))
   
(* and leqq te s X t = leqq' te s X t *)

fun leq te s t = leqq te s [] t EXPAND

end
(* Gene Rollins
   School of Computer Science
   Carnegie-Mellon University
   Pittsburgh, PA 15213
   rollins@cs.cmu.edu *)

functor HashFun () = struct

val version = 1.0

type ('a,'b) table = ('a*'a->bool) * (('a*int*'b) list array) * int

fun create (sample'key :'1a) (equality :'1a * '1a -> bool)
           table'size (sample'value :'1b) :('1a,'1b) table =
  let val mt = tl [(sample'key, 0, sample'value)]
  in (equality, array (table'size, mt), table'size)
  end

val defaultSize = 97 (* a prime; or try primes 37, 997 *)

fun defaultEqual ((x :string), (y :string)) :bool = (x = y)

fun createDefault (sample'value :'1b) :(string,'1b) table =
  let val mt = tl [("", 0, sample'value)]
  in (defaultEqual, array (defaultSize, mt), defaultSize)
  end

fun enter ((equal, table, table'size) :('a,'b) table) key hash value = 
  let val place = hash mod table'size
      val bucket = table sub place
      fun put'in [] = [(key,hash,value)]
        | put'in ((k,h,v)::tail) =
	    if (h = hash) andalso equal (k, key)
	      then (key,hash,value)::tail
	      else (k,h,v)::(put'in tail)
  in
    update (table, place, put'in bucket)
  end

fun remove ((equal, table, table'size) :('a,'b) table) key hash =
  let val place = hash mod table'size
      val bucket = table sub place
      fun take'out [] = []
        | take'out ((k,h,v)::tail) =
	    if (h = hash) andalso equal (k, key)
	      then tail
	      else (k,h,v)::(take'out tail)
  in
    update (table, place, take'out bucket)
  end

fun lookup ((equal, table, table'size) :('a,'b) table) key hash =
  let val place = hash mod table'size
      val bucket = table sub place
      fun get'out [] = NONE
        | get'out ((k,h,v)::tail) =
	    if (h = hash) andalso equal (k, key)
	      then SOME v
	      else get'out tail
  in
    get'out bucket
  end

fun print ((_, table, table'size) :('a,'b) table)
          (print'key :'a -> unit) (print'value :'b -> unit) =
  let fun pr'bucket [] = ()
        | pr'bucket ((key,hash,value)::rest) =
            (print'key key; String.print ": ";
             Integer.print hash; String.print ": ";
	     print'value value; String.print "\n"; pr'bucket rest)
      fun pr i =
        if i >= table'size then ()
	  else
	    case (table sub i) of
	       [] => (pr (i+1))
             | (b as (h::t)) =>
	         (String.print "["; Integer.print i; String.print "]\n";
	          pr'bucket b; pr (i+1))
  in pr 0 end

fun scan ((_, table, table'size) :('a,'b) table) operation =
  let fun map'bucket [] = ()
        | map'bucket ((key,hash,value)::rest) =
            (operation key hash value; map'bucket rest)
      fun iter i =
        if i >= table'size then ()
	  else (map'bucket (table sub i); iter (i+1))
  in iter 0 end

fun fold ((_, table, table'size) :('a, 'b) table)
         (operation :'a -> int -> 'b -> 'g -> 'g) (init :'g) :'g =
  let fun fold'bucket [] acc = acc
        | fold'bucket ((key,hash,value)::rest) acc =
             fold'bucket rest (operation key hash value acc)
      fun iter i acc =
        if i >= table'size then acc
	  else iter (i+1) (fold'bucket (table sub i) acc)
  in iter 0 init end

fun scanUpdate ((_, table, table'size) :('a,'b) table) operation =
  let fun map'bucket [] = []
        | map'bucket ((key,hash,value)::rest) =
            ((key,hash,operation key hash value)::(map'bucket rest))
      fun iter i =
        if i >= table'size then ()
	  else (update (table, i, map'bucket (table sub i)); iter (i+1))
  in iter 0 end

fun eliminate ((_, table, table'size) :('a,'b) table) predicate =
  let fun map'bucket [] = []
        | map'bucket ((key,hash,value)::rest) =
            if predicate key hash value then map'bucket rest
              else (key,hash,value)::(map'bucket rest)
      fun iter i =
        if i >= table'size then ()
	  else (update (table, i, map'bucket (table sub i)); iter (i+1))
  in iter 0 end

fun bucketLengths ((_, table, table'size) :('a,'b) table) (maxlen :int)
    :int array =
  let val count :int array = array (maxlen+1, 0)
      fun inc'sub x = 
        let val y = min (x, maxlen) in
          update (count, y, (count sub y) + 1)
        end
      fun iter i =
        if i >= table'size then ()
	  else (inc'sub (length (table sub i)); iter (i+1))
  in
    iter 0;
    count
  end

end
(* Cribbed from...
   ML-Yacc Parser Generator (c) 1989 Andrew W. Appel, David R. Tarditi *)

signature ORDSET =
   sig
      type set
      type elem
      exception Select_arb
      val app : (elem -> 'b) -> set -> unit
	  and card: set -> int
          and closure: set * (elem -> set) -> set
          and difference: set * set -> set
          and elem_eq: (elem * elem -> bool)
	  and elem_gt : (elem * elem -> bool)
          and empty: set
	  and exists: (elem * set) -> bool
	  and find : (elem * set)  ->  elem option
	  and fold: ((elem * 'b) -> 'b) -> set -> 'b -> 'b
          and insert: (elem * set) -> set
          and is_empty: set -> bool
          and make_list: set -> elem list
          and make_set: (elem list -> set)
          and partition: (elem -> bool) -> (set -> set * set)
          and remove: (elem * set) -> set
	  and revfold: ((elem * 'b) -> 'b) -> set -> 'b -> 'b
          and select_arb: set -> elem
	  and set_eq: (set * set) -> bool
	  and set_gt: (set * set) -> bool
          and singleton: (elem -> set)
          and union: set * set -> set
   end

signature TABLE =
   sig
	type 'a table
	type key
	val size : 'a table -> int
	val empty: 'a table
	val exists: (key * 'a table) -> bool
	val find : (key * 'a table)  ->  'a option
	val insert: ((key * 'a) * 'a table) -> 'a table
	val make_table : (key * 'a ) list -> 'a table
	val make_list : 'a table -> (key * 'a) list
	val fold : ((key * 'a) * 'b -> 'b) -> 'a table -> 'b -> 'b
   end

signature HASH =
  sig
    type table
    type elem

    val size : table -> int
    val add : elem * table -> table
    val find : elem * table -> int option
    val exists : elem * table -> bool
    val empty : table
  end;

(* Cribbed from...
   ML-Yacc Parser Generator (c) 1989 Andrew W. Appel, David R. Tarditi *)

(*III
import "tarditi.sig";
III*)

(* Implementation of ordered sets using ordered lists and red-black trees.  The
   code for red-black trees was originally written by Norris Boyd, which was
   modified for use here.
*)   

(* ordered sets implemented using ordered lists.

   Upper bound running times for functions implemented here:

   app  = O(n)
   card = O(n)
   closure = O(n^2)
   difference = O(n+m), where n,m = the size of the two sets used here.
   empty = O(1)
   exists = O(n)
   find = O(n)
   fold = O(n)
   insert = O(n)
   is_empty = O(1)
   make_list = O(1)
   make_set = O(n^2)
   partition = O(n)
   remove = O(n)
   revfold = O(n)
   select_arb = O(1)
   set_eq = O(n), where n = the cardinality of the smaller set
   set_gt = O(n), ditto
   singleton = O(1)
   union = O(n+m)
*)

functor ListOrdSet(B : sig type elem
		  	val gt : elem * elem -> bool
			val eq : elem * elem -> bool
		    end ) : ORDSET =

struct
 type elem = B.elem
 val elem_gt = B.gt
 val elem_eq = B.eq 

 type set = elem list
 exception Select_arb
 val empty = nil

 val insert = fn (key,s) =>
	let fun f (l as (h::t)) =
		 if elem_gt(key,h) then h::(f t)
		 else if elem_eq(key,h) then key::t
		 else key::l
 	      | f nil = [key]
	in f s
	end
		
 val select_arb = fn nil => raise Select_arb
 		   | a::b => a

 val exists = fn (key,s) =>
	let fun f (h::t) = if elem_gt(key,h) then f t
			   else elem_eq(h,key) 
 	      | f nil = false
	in f s
	end

 val find = fn (key,s) =>
	let fun f (h::t) = if elem_gt(key,h) then f t
			   else if elem_eq(h,key) then SOME h
			   else NONE
 	      | f nil = NONE
	in f s
	end
   
 val revfold = List.revfold
 val fold = List.fold
 val app = List.app

fun set_eq(h::t,h'::t') = 
	(case elem_eq(h,h')
	  of true => set_eq(t,t')
	   | a => a)
  | set_eq(nil,nil) = true
  | set_eq _ = false

fun set_gt(h::t,h'::t') =
	(case elem_gt(h,h')
	  of false => (case (elem_eq(h,h'))
			of true => set_gt(t,t')
			 | a => a)
	   |  a => a)
  | set_gt(_::_,nil) = true
  | set_gt _ = false
		
fun union(a as (h::t),b as (h'::t')) =
	  if elem_gt(h',h) then h::union(t,b)
	  else if elem_eq(h,h') then h::union(t,t')
	  else h'::union(a,t')
  | union(nil,s) = s
  | union(s,nil) = s

val make_list = fn s => s

val is_empty = fn nil => true | _ => false

val make_set = fn l => List.fold insert l nil

val partition = fn f => fn s =>
    fold (fn (e,(yes,no)) =>
	    if (f e) then (e::yes,no) else (e::no,yes)) s (nil,nil)

val remove = fn (e,s) =>
    let fun f (l as (h::t)) = if elem_gt(h,e) then l
			      else if elem_eq(h,e) then t
			      else h::(f t)
	  | f nil = nil
    in f s
    end

 (* difference: X-Y *)

 fun difference (nil,_) = nil
   | difference (r,nil) = r
   | difference (a as (h::t),b as (h'::t')) =
	  if elem_gt (h',h) then h::difference(t,b)
	  else if elem_eq(h',h) then difference(t,t')
	  else difference(a,t')

 fun singleton X = [X]

 fun card(S) = fold (fn (a,count) => count+1) S 0

      local
	    fun closure'(from, f, result) =
	      if is_empty from then result
	      else
		let val (more,result) =
			fold (fn (a,(more',result')) =>
				let val more = f a
				    val new = difference(more,result)
				in (union(more',new),union(result',new))
				end) from
				 (empty,result)
		in closure'(more,f,result)
		end
      in
         fun closure(start, f) = closure'(start, f, start)
      end
end

(* ordered set implemented using red-black trees:

   Upper bound running time of the functions below:

   app: O(n)
   card: O(n)
   closure: O(n^2 ln n)
   difference: O(n ln n)
   empty: O(1)
   exists: O(ln n)
   find: O(ln n)
   fold: O(n)
   insert: O(ln n)
   is_empty: O(1)
   make_list: O(n)
   make_set: O(n ln n)
   partition: O(n ln n)
   remove: O(n ln n)
   revfold: O(n)
   select_arb: O(1)
   set_eq: O(n)
   set_gt: O(n)
   singleton: O(1)
   union: O(n ln n)
*)

functor RbOrdSet (B : sig type elem
			 val eq : (elem*elem) -> bool
		 	 val gt : (elem*elem) -> bool
		     end
		) : ORDSET =
struct

 type elem = B.elem
 val elem_gt = B.gt
 val elem_eq = B.eq 

 datatype Color = RED | BLACK

 abstype set = EMPTY | TREE of (B.elem * Color * set * set)
 with exception Select_arb
      val empty = EMPTY

 fun insert(key,t) =
  let fun f EMPTY = TREE(key,RED,EMPTY,EMPTY)
        | f (TREE(k,BLACK,l,r)) =
	    if elem_gt (key,k)
	    then case f r
		 of r as TREE(rk,RED, rl as TREE(rlk,RED,rll,rlr),rr) =>
			(case l
			 of TREE(lk,RED,ll,lr) =>
				TREE(k,RED,TREE(lk,BLACK,ll,lr),
					   TREE(rk,BLACK,rl,rr))
			  | _ => TREE(rlk,BLACK,TREE(k,RED,l,rll),
						TREE(rk,RED,rlr,rr)))
		  | r as TREE(rk,RED,rl, rr as TREE(rrk,RED,rrl,rrr)) =>
			(case l
			 of TREE(lk,RED,ll,lr) =>
				TREE(k,RED,TREE(lk,BLACK,ll,lr),
					   TREE(rk,BLACK,rl,rr))
			  | _ => TREE(rk,BLACK,TREE(k,RED,l,rl),rr))
	          | r => TREE(k,BLACK,l,r)
	    else if elem_gt(k,key)
	    then case f l
	         of l as TREE(lk,RED,ll, lr as TREE(lrk,RED,lrl,lrr)) =>
			(case r
			 of TREE(rk,RED,rl,rr) =>
				TREE(k,RED,TREE(lk,BLACK,ll,lr),
					   TREE(rk,BLACK,rl,rr))
			  | _ => TREE(lrk,BLACK,TREE(lk,RED,ll,lrl),
						TREE(k,RED,lrr,r)))
		  | l as TREE(lk,RED, ll as TREE(llk,RED,lll,llr), lr) =>
			(case r
			 of TREE(rk,RED,rl,rr) =>
				TREE(k,RED,TREE(lk,BLACK,ll,lr),
					   TREE(rk,BLACK,rl,rr))
			  | _ => TREE(lk,BLACK,ll,TREE(k,RED,lr,r)))
	          | l => TREE(k,BLACK,l,r)
	    else TREE(key,BLACK,l,r)
        | f (TREE(k,RED,l,r)) =
	    if elem_gt(key,k) then TREE(k,RED,l, f r)
	    else if elem_gt(k,key) then TREE(k,RED, f l, r)
	    else TREE(key,RED,l,r)
   in case f t
      of TREE(k,RED, l as TREE(_,RED,_,_), r) => TREE(k,BLACK,l,r)
       | TREE(k,RED, l, r as TREE(_,RED,_,_)) => TREE(k,BLACK,l,r)
       | t => t
  end

 fun select_arb (TREE(k,_,l,r)) = k
   | select_arb EMPTY = raise Select_arb
   
 fun exists(key,t) =
  let fun look EMPTY = false
	| look (TREE(k,_,l,r)) =
		if elem_gt(k,key) then look l
		else if elem_gt(key,k) then look r
		else true
   in look t
   end

 fun find(key,t) =
  let fun look EMPTY = NONE
	| look (TREE(k,_,l,r)) =
		if elem_gt(k,key) then look l
		else if elem_gt(key,k) then look r
		else SOME k
   in look t
  end

  fun revfold f t start =
     let fun scan (EMPTY,value) = value
	   | scan (TREE(k,_,l,r),value) = scan(r,f(k,scan(l,value)))
     in scan(t,start)
     end

   fun fold f t start =
	let fun scan(EMPTY,value) = value
	      | scan(TREE(k,_,l,r),value) = scan(l,f(k,scan(r,value)))
	in scan(t,start)
	end

   fun app f t =
      let fun scan EMPTY = ()
            | scan(TREE(k,_,l,r)) = (scan l; f k; scan r)
      in scan t
      end

(* equal_tree : test if two trees are equal.  Two trees are equal if
   the set of leaves are equal *)

   fun set_eq (tree1 as (TREE _),tree2 as (TREE _)) =
     let datatype pos = L | R | M
	 exception Done
	 fun getvalue(stack as ((a,position)::b)) =
	    (case a
	     of (TREE(k,_,l,r)) =>
		(case position
		 of L => getvalue ((l,L)::(a,M)::b)
		  | M => (k,case r of  EMPTY => b | _ => (a,R)::b)
		  | R => getvalue ((r,L)::b)
		 )
	      | EMPTY => getvalue b
	     )
	    | getvalue(nil) = raise Done
	  fun f (nil,nil) = true
	    | f (s1 as (_ :: _),s2 as (_ :: _ )) =
			  let val (v1,news1) = getvalue s1
			      and (v2,news2) = getvalue s2
			  in (elem_eq(v1,v2)) andalso f(news1,news2)
			  end
	    | f _ = false
      in f ((tree1,L)::nil,(tree2,L)::nil) handle Done => false
      end
    | set_eq (EMPTY,EMPTY) = true
    | set_eq _ = false

   (* gt_tree : Test if tree1 is greater than tree 2 *)

   fun set_gt (tree1,tree2) =
     let datatype pos = L | R | M
	 exception Done
	 fun getvalue(stack as ((a,position)::b)) =
	    (case a
	     of (TREE(k,_,l,r)) =>
		(case position
		 of L => getvalue ((l,L)::(a,M)::b)
		  | M => (k,case r of EMPTY => b | _ => (a,R)::b)
		  | R => getvalue ((r,L)::b)
		 )
	      | EMPTY => getvalue b
	     )
	    | getvalue(nil) = raise Done
	  fun f (nil,nil) = false
	    | f (s1 as (_ :: _),s2 as (_ :: _ )) =
			  let val (v1,news1) = getvalue s1
			      and (v2,news2) = getvalue s2
			  in (elem_gt(v1,v2)) orelse (elem_eq(v1,v2) andalso f(news1,news2))
			  end
	    | f (_,nil) = true
	    | f (nil,_) = false
      in f ((tree1,L)::nil,(tree2,L)::nil) handle Done => false
      end

      fun is_empty S = (let val _ = select_arb S in false end
                         handle Select_arb => true)

      fun make_list S = fold (op ::) S nil

      fun make_set l = List.fold insert l empty

      fun partition F S = fold (fn (a,(Yes,No)) =>
				if F(a) then (insert(a,Yes),No)
				else (Yes,insert(a,No)))
			     S (empty,empty)

      fun remove(X, XSet) =
             let val (YSet, _) =
                        partition (fn a => not (elem_eq (X, a))) XSet
             in  YSet
             end

      fun difference(Xs, Ys) =
	   fold (fn (p as (a,Xs')) =>
		      if exists(a,Ys) then Xs' else insert p)
	   Xs empty

      fun singleton X = insert(X,empty)

      fun card(S) = fold (fn (_,count) => count+1) S 0

      fun union(Xs,Ys)= fold insert Ys Xs

      local
	    fun closure'(from, f, result) =
	      if is_empty from then result
	      else
		let val (more,result) =
			fold (fn (a,(more',result')) =>
				let val more = f a
				    val new = difference(more,result)
				in (union(more',new),union(result',new))
				end) from
				 (empty,result)
		in closure'(more,f,result)
		end
      in
         fun closure(start, f) = closure'(start, f, start)
      end
   end
end

(*
signature TABLE =
   sig
	type 'a table
	type key
	val size : 'a table -> int
	val empty: 'a table
	val exists: (key * 'a table) -> bool
	val find : (key * 'a table)  ->  'a option
	val insert: ((key * 'a) * 'a table) -> 'a table
	val make_table : (key * 'a ) list -> 'a table
	val make_list : 'a table -> (key * 'a) list
	val fold : ((key * 'a) * 'b -> 'b) -> 'a table -> 'b -> 'b
   end
*)

functor Table (B : sig type key
		      val gt : (key * key) -> bool
		     end
		) : TABLE =
struct

 datatype Color = RED | BLACK
 type key = B.key

 abstype 'a table = EMPTY
		  | TREE of ((B.key * 'a ) * Color * 'a table * 'a table)
 with

 val empty = EMPTY

 fun insert(elem as (key,data),t) =
  let val key_gt = fn (a,_) => B.gt(key,a)
      val key_lt = fn (a,_) => B.gt(a,key)
	fun f EMPTY = TREE(elem,RED,EMPTY,EMPTY)
        | f (TREE(k,BLACK,l,r)) =
	    if key_gt k
	    then case f r
		 of r as TREE(rk,RED, rl as TREE(rlk,RED,rll,rlr),rr) =>
			(case l
			 of TREE(lk,RED,ll,lr) =>
				TREE(k,RED,TREE(lk,BLACK,ll,lr),
					   TREE(rk,BLACK,rl,rr))
			  | _ => TREE(rlk,BLACK,TREE(k,RED,l,rll),
						TREE(rk,RED,rlr,rr)))
		  | r as TREE(rk,RED,rl, rr as TREE(rrk,RED,rrl,rrr)) =>
			(case l
			 of TREE(lk,RED,ll,lr) =>
				TREE(k,RED,TREE(lk,BLACK,ll,lr),
					   TREE(rk,BLACK,rl,rr))
			  | _ => TREE(rk,BLACK,TREE(k,RED,l,rl),rr))
	          | r => TREE(k,BLACK,l,r)
	    else if key_lt k
	    then case f l
	         of l as TREE(lk,RED,ll, lr as TREE(lrk,RED,lrl,lrr)) =>
			(case r
			 of TREE(rk,RED,rl,rr) =>
				TREE(k,RED,TREE(lk,BLACK,ll,lr),
					   TREE(rk,BLACK,rl,rr))
			  | _ => TREE(lrk,BLACK,TREE(lk,RED,ll,lrl),
						TREE(k,RED,lrr,r)))
		  | l as TREE(lk,RED, ll as TREE(llk,RED,lll,llr), lr) =>
			(case r
			 of TREE(rk,RED,rl,rr) =>
				TREE(k,RED,TREE(lk,BLACK,ll,lr),
					   TREE(rk,BLACK,rl,rr))
			  | _ => TREE(lk,BLACK,ll,TREE(k,RED,lr,r)))
	          | l => TREE(k,BLACK,l,r)
	    else TREE(elem,BLACK,l,r)
        | f (TREE(k,RED,l,r)) =
	    if key_gt k then TREE(k,RED,l, f r)
	    else if key_lt k then TREE(k,RED, f l, r)
	    else TREE(elem,RED,l,r)
   in case f t
      of TREE(k,RED, l as TREE(_,RED,_,_), r) => TREE(k,BLACK,l,r)
       | TREE(k,RED, l, r as TREE(_,RED,_,_)) => TREE(k,BLACK,l,r)
       | t => t
  end

 fun exists(key,t) =
  let fun look EMPTY = false
	| look (TREE((k,_),_,l,r)) =
		if B.gt(k,key) then look l
		else if B.gt(key,k) then look r
		else true
   in look t
   end

 fun find(key,t) =
  let fun look EMPTY = NONE
	| look (TREE((k,data),_,l,r)) =
		if B.gt(k,key) then look l
		else if B.gt(key,k) then look r
		else SOME data
   in look t
  end

  fun fold f t start =
	let fun scan(EMPTY,value) = value
	      | scan(TREE(k,_,l,r),value) = scan(l,f(k,scan(r,value)))
	in scan(t,start)
	end

  fun make_table l = List.fold insert l empty

  fun size S = fold (fn (_,count) => count+1) S 0

  fun make_list table = fold (op ::) table nil

  end
end;

(* assumes that a functor Table with signature TABLE from table.sml is
   in the environment *)

(*
signature HASH =
  sig
    type table
    type elem

    val size : table -> int
    val add : elem * table -> table
    val find : elem * table -> int option
    val exists : elem * table -> bool
    val empty : table
  end
*)

(* hash: creates a hash table of size n which assigns each distinct member
   a unique integer between 0 and n-1 *)

functor Hash(B : sig type elem
		     val gt : elem * elem -> bool
		 end) : HASH =
struct
    type elem=B.elem
    structure HashTable = Table(type key=B.elem
				val gt = B.gt)

    type table = {count : int, table : int HashTable.table}

    val empty = {count=0,table=HashTable.empty}
    val size = fn {count,table} => count
    val add = fn (e,{count,table}) =>
		{count=count+1,table=HashTable.insert((e,count),table)}
    val find = fn (e,{table,count}) => HashTable.find(e,table)
    val exists = fn (e,{table,count}) => HashTable.exists(e,table)
end;
(*III
import "interface.sig";
III*)

functor Interface () : INTERFACE =
struct

type pos = int
val line = ref 0
fun init_line () = (line := 0)
fun next_line () = (line := !line + 1)
fun error (errmsg,line:pos,_) =
  output (std_out, ("Line " ^ (makestring line) ^ ": " ^ errmsg ^ "\n"))

end  (* functor INTERFACE *)
functor Globals(
	    structure Wr: WR
	    structure Pp: PP
	    structure WrMgt: WRMGT
	    structure ListUtils: LISTUTILS
	    structure StringUtils: STRINGUTILS
	    structure DebugUtils: DEBUGUTILS
	    structure Id: ID
	    structure Registry: REGISTRY
	    sharing Pp.Wr = Wr
	        and WrMgt.Pp = Pp
	        and type Registry.registeredtype = bool
	    ) : GLOBALS 
	    = struct

structure Wr = Wr;
open Wr;

structure Pp = Pp;
open Pp;

structure WrMgt = WrMgt;
open WrMgt;

structure Id = Id;

structure Registry = Registry

open ListUtils
open StringUtils
open DebugUtils
open Registry

exception CantHappen

end

signature TRMPVT = sig

structure Globals: GLOBALS
structure Typ: TYPPVT
sharing Typ.Globals = Globals
open Globals

datatype pretrm = 
	    PREVAR of Id.T
	  | PREABS of Id.T * Typ.pretyp * pretrm
	  | PREAPP of pretrm * pretrm
	  | PRETABS of Id.T * Typ.pretyp * pretrm
	  | PRETAPP of pretrm * Typ.pretyp
	  | PREFOR of Id.T * (Typ.pretyp list) * pretrm

datatype T = 
	    VAR of unit * int
	  | ABS of {name:Id.T} * Typ.T * T
	  | APP of unit * T * T
	  | TABS of {name:Id.T} * Typ.T * T
	  | TAPP of unit * T * Typ.T
	  | FOR of {name:Id.T} * (Typ.T list) * T

exception UnknownId of string
val debruijnify: Typ.tenv -> pretrm -> T

val prt: Pp.Pp -> Typ.tenv -> T -> unit

end

functor DebugUtils(
	    structure WrMgt: WRMGT
	    ) : DEBUGUTILS = struct

open WrMgt
open Pp;

val level = ref(0);

(* $$$ belongs in globals: *)
fun unwind_protect f cleanup =
  (f())
  handle e => (cleanup(); raise e)

fun do_wrap pp name f pbefore pafter =
  (pwrite pp "[";
   setb pp;  
   pwrite pp (makestring (!level));
   pwrite pp "] ";
   pwrite pp name;
   pwrite pp "? ";
   pbefore();
   pwrite pp "\n";
   level := (!level) + 1;
   let val result = unwind_protect 
   			f
   		        (fn () => level := (!level) - 1)
   in
      level := (!level) - 1;
      break pp true ~3;
      pwrite pp "   [";
      pwrite pp (makestring (!level));
      pwrite pp "] ";
      pwrite pp name;
      pwrite pp ": ";
      pafter(result);
      pwrite pp "\n";
      endb pp;
      result
   end
  )

fun wrap DEBUG name f pbefore pafter =
  if (not (!DEBUG))
    then f()
    else do_wrap (stdpp()) name f pbefore pafter;

end
functor Trm(
	    structure Globals: GLOBALS
	    structure Typ: TYPPVT
	    sharing Typ.Globals = Globals
	    ) : TRMPVT
	    = struct

structure Globals = Globals
structure Typ = Typ
open Globals
open Typ
open Pp

datatype pretrm = 
	    PREVAR of Id.T
	  | PREABS of Id.T * pretyp * pretrm
	  | PREAPP of pretrm * pretrm
	  | PRETABS of Id.T * pretyp * pretrm
	  | PRETAPP of pretrm * pretyp
	  | PREFOR of Id.T * (pretyp list) * pretrm

datatype T = 
	    VAR of unit * int
	  | ABS of {name:Id.T} * Typ.T * T
	  | APP of unit * T * T
	  | TABS of {name:Id.T} * Typ.T * T
	  | TAPP of unit * T * Typ.T
	  | FOR of {name:Id.T} * (Typ.T list) * T

fun debruijnify te (PREVAR i) =
      VAR((), index te i)
  | debruijnify te (PREABS(i,ptyp,ptrm)) =
      ABS({name=i}, Typ.debruijnify te ptyp, 
      		    debruijnify (push_binding te i NS) ptrm)
  | debruijnify te (PREAPP(ptrm1,ptrm2)) =
      APP((), debruijnify te ptrm1, debruijnify te ptrm2)
  | debruijnify te (PRETABS(i,ptyp,ptrm)) =
      TABS({name=i}, Typ.debruijnify te ptyp, 
      		     debruijnify (push_bound te i NS) ptrm)
  | debruijnify te (PRETAPP(ptrm,ptyp)) =
      TAPP((), debruijnify te ptrm, Typ.debruijnify te ptyp)
  | debruijnify te (PREFOR(i,ptyps,ptrm)) =
      FOR({name=i}, map (fn pt => Typ.debruijnify te pt) ptyps, 
      		     debruijnify (push_bound te i NS) ptrm)

fun prt pp te trm =
  let fun p te (VAR(_,i)) =
              Pp.pwrite pp (Id.tostr (lookup_name te i))
        | p te (ABS({name=i},t,body)) =
              (Pp.pwrite pp "(\\";
               Pp.pwrite pp (Id.tostr i);
               Pp.pwrite pp ":";
               Typ.prt pp te t;
               Pp.pwrite pp ". ";
               p (push_binding te i t) body;
               Pp.pwrite pp ")")
        | p te (APP(_,trm1,trm2)) =
              (Pp.pwrite pp "(";
               p te trm1;
               Pp.pwrite pp " ";
               p te trm2;
               Pp.pwrite pp ")")
        | p te (TABS({name=i},t,body)) =
              (Pp.pwrite pp "(\\\\";
               Pp.pwrite pp (Id.tostr i);
               Pp.pwrite pp "<=";
               Typ.prt pp te t;
               Pp.pwrite pp ". ";
               p (push_bound te i t) body;
               Pp.pwrite pp ")")
        | p te (TAPP(_,trm1,t)) =
              (Pp.pwrite pp "(";
               p te trm1;
               Pp.pwrite pp " [";
               Typ.prt pp te t;
               Pp.pwrite pp "])")
        | p te (FOR({name=i},ts,body)) =
              (Pp.pwrite pp "(for ";
               Pp.pwrite pp (Id.tostr i);
               Pp.pwrite pp " in ";
               mapunit_tuple (fn t => Typ.prt pp te t) (fn () => Pp.pwrite pp ",") ts;
               Pp.pwrite pp ". ";
               p (push_abbrev te i NS) body;
               Pp.pwrite pp ")")
  in p te trm
  end

end
functor ListUtils() : LISTUTILS = struct

fun mapunit f l = 
  let fun mu [] = ()
        | mu (hd::tl) = (f hd; mu tl)
  in mu l
  end

fun mapunit_tuple f betw ts =
  let fun mut [] = ()
        | mut [e] = f e
        | mut (e::tl) = (f e; betw(); mut tl)
  in mut ts
  end

fun mapfold fm ff z =
  let fun m []       = z
        | m (hd::tl) = ff (fm hd) (m tl)
  in m
  end

fun memq eq l e =
  let fun m [] = false
        | m (hd::tl) = (eq e hd) orelse (m tl)
  in m l
  end

fun mapappend f l =
  let fun ma [] = []
        | ma (hd::tl) = (f hd) @ (ma tl)
  in ma l
  end

fun filter b l =
  let fun f [] = []
        | f (hd::tl) = if (b hd) then hd::(f tl) else f tl
  in f l
  end

fun forall f = mapfold f (fn x => fn y => x andalso y) true

fun forsome f = mapfold f (fn x => fn y => x orelse y) false

end
functor Synth(
	    structure Globals: GLOBALS
	    structure Typ: TYPPVT
	    structure Trm: TRMPVT
	    structure Leq: LEQ
	    sharing Typ.Globals = Globals
	        and Trm.Typ = Typ
	        and Leq.Typ = Typ
	    ) : SYNTH = struct

structure Globals = Globals
structure Typ = Typ
structure Trm = Trm
structure Leq = Leq
open Globals
open Typ
open Trm

val DEBUG = ref(false)
val _ = (registerflag "synth" DEBUG;
	 registerflag "Synth" DEBUG)

fun arrowbasis te t = 
  let fun ab (TVAR(_,v)) = 
   	    (case lookup_and_relocate te v of
		BND(_,b) => ab b
	      | VBND(n,b) => raise Typ.WrongKindOfId(te,v,"tvar or abbrev")
	      | ABB(_,b) => ab b)
        | ab (t as ARROW(_)) =
            [t]
        | ab (ALL(_)) =
            []
        | ab (MEET(_,ts)) =
            mapappend ab ts
  in ab t
  end

fun allbasis te t = 
  let fun ab (TVAR(_,v)) = 
   	    (case lookup_and_relocate te v of
		BND(_,b) => ab b
	      | VBND(n,b) => raise Typ.WrongKindOfId(te,v,"tvar or abbrev")
	      | ABB(_,b) => ab b)
        | ab (t as ARROW(_)) =
            []
        | ab (t as ALL(_)) =
            [t]
        | ab (MEET(_,ts)) =
            mapappend ab ts
  in ab t
  end

fun synth' te (VAR(_,v)) = Typ.lookup_and_relocate_binding te v
  | synth' te (ABS({name=i},t,body)) = 
      let val t_body = synth (push_binding te i t) body
          val t' = relocate ~1 t_body
      in ARROW((),t,t')
      end
  | synth' te (APP(_,trm1,trm2)) = 
      let val t1 = synth te trm1
          and t2 = synth te trm2
          val basis = arrowbasis te t1
          fun collect_apps [] =
                []
            | collect_apps ((ARROW(_,tb1,tb2))::tl) = 
                if Leq.leq te t2 tb1
                   then tb2::(collect_apps tl)
                   else (collect_apps tl)
            | collect_apps _ = raise CantHappen
          val ts = collect_apps basis
      in MEET((),ts)
      end
  | synth' te (TABS({name=i},t,body)) = 
      let val t' = synth (push_bound te i t) body
      in ALL({name=i},t,t')
      end
  | synth' te (TAPP(_,body,t)) =
      let val t_body = synth te body
          val basis = allbasis te t_body
          fun collect_apps [] =
                []
            | collect_apps ((ALL(_,t1,t2))::tl) = 
                if Leq.leq te t t1
                   then (tsubst_top t t2)::(collect_apps tl)
                   else (collect_apps tl)
            | collect_apps _ = raise CantHappen
          val ts = collect_apps basis
      in MEET((),ts)
      end
  | synth' te (FOR({name=i},ts,body)) =
      let fun f t = 
              let val tb = synth (push_abbrev te i t) body
                  val tb' = tsubst_top t tb
              in tb'
              end
      in MEET((), map f ts)
      end

and synth te e =
  wrap (DEBUG) "synth"
       (fn () => synth' te e)
       (fn () => 
       	  (Trm.prt (stdpp()) te e;
       	   Pp.pwrite (stdpp()) "\n";
       	   Typ.prt_tenv (stdpp()) te))
       (fn t => 
          (Typ.prt (stdpp()) te t))

end

functor FMEETLrValsFun ( structure Token : TOKEN
			        structure Globals : GLOBALS
			        structure ParseRes : PARSERES
			      ) : FMEET_LRVALS = 
struct
structure ParserData=
struct
structure Header = 
struct
structure ParseRes = ParseRes
open ParseRes
open Trm
open Typ
open Globals

end
structure LrTable = Token.LrTable
structure Token = Token
local open LrTable in 
val table=let val actionT =
"\
\\001\000\022\000\014\000\021\000\023\000\020\000\
\\024\000\019\000\025\000\018\000\026\000\017\000\027\000\016\000\
\\028\000\015\000\029\000\014\000\030\000\013\000\031\000\012\000\
\\032\000\011\000\033\000\010\000\040\000\009\000\000\000\001\000\
\\000\000\141\000\
\\014\000\025\000\016\000\024\000\025\000\018\000\000\000\140\000\
\\000\000\116\000\
\\000\000\147\000\
\\005\000\028\000\008\000\027\000\009\000\026\000\000\000\137\000\
\\000\000\118\000\
\\025\000\018\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\025\000\038\000\000\000\001\000\
\\025\000\039\000\000\000\001\000\
\\025\000\040\000\000\000\001\000\
\\025\000\018\000\000\000\001\000\
\\025\000\042\000\000\000\001\000\
\\000\000\139\000\
\\000\000\138\000\
\\000\000\136\000\
\\025\000\018\000\000\000\001\000\
\\025\000\018\000\000\000\001\000\
\\014\000\021\000\023\000\020\000\024\000\019\000\
\\025\000\018\000\026\000\017\000\027\000\016\000\040\000\009\000\000\000\001\000\
\\000\000\117\000\
\\000\000\149\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\021\000\023\000\020\000\024\000\019\000\
\\025\000\018\000\026\000\017\000\027\000\016\000\040\000\009\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\021\000\023\000\020\000\024\000\019\000\
\\025\000\018\000\026\000\017\000\027\000\016\000\040\000\009\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\036\000\053\000\000\000\001\000\
\\006\000\054\000\000\000\134\000\
\\005\000\057\000\012\000\056\000\022\000\055\000\000\000\001\000\
\\000\000\122\000\
\\025\000\018\000\000\000\001\000\
\\000\000\126\000\
\\025\000\018\000\000\000\001\000\
\\016\000\060\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\000\000\120\000\
\\000\000\121\000\
\\000\000\119\000\
\\005\000\063\000\009\000\062\000\000\000\001\000\
\\000\000\110\000\
\\036\000\064\000\000\000\001\000\
\\002\000\066\000\005\000\065\000\006\000\054\000\000\000\134\000\
\\003\000\067\000\000\000\001\000\
\\015\000\068\000\000\000\001\000\
\\000\000\137\000\
\\012\000\056\000\017\000\069\000\022\000\055\000\000\000\001\000\
\\015\000\070\000\000\000\001\000\
\\012\000\056\000\022\000\055\000\000\000\114\000\
\\000\000\115\000\
\\012\000\056\000\022\000\055\000\000\000\112\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\025\000\018\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\002\000\078\000\005\000\077\000\000\000\001\000\
\\002\000\080\000\005\000\079\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\012\000\056\000\015\000\082\000\022\000\055\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\021\000\023\000\020\000\024\000\019\000\
\\025\000\018\000\026\000\017\000\027\000\016\000\040\000\009\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\000\000\148\000\
\\000\000\151\000\
\\000\000\150\000\
\\002\000\089\000\000\000\001\000\
\\006\000\090\000\012\000\056\000\022\000\055\000\000\000\132\000\
\\000\000\135\000\
\\012\000\056\000\022\000\055\000\000\000\124\000\
\\012\000\056\000\000\000\123\000\
\\012\000\056\000\022\000\055\000\000\000\109\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\017\000\095\000\000\000\001\000\
\\000\000\127\000\
\\012\000\056\000\022\000\055\000\000\000\113\000\
\\012\000\056\000\022\000\055\000\000\000\111\000\
\\002\000\096\000\000\000\001\000\
\\002\000\097\000\012\000\056\000\022\000\055\000\000\000\001\000\
\\000\000\143\000\
\\002\000\098\000\000\000\001\000\
\\014\000\021\000\023\000\020\000\024\000\019\000\
\\025\000\018\000\026\000\017\000\027\000\016\000\040\000\009\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\002\000\101\000\012\000\056\000\022\000\055\000\000\000\001\000\
\\012\000\056\000\022\000\055\000\000\000\130\000\
\\002\000\102\000\012\000\056\000\022\000\055\000\000\000\001\000\
\\012\000\056\000\022\000\055\000\000\000\128\000\
\\000\000\125\000\
\\014\000\021\000\023\000\020\000\024\000\019\000\
\\025\000\018\000\026\000\017\000\027\000\016\000\040\000\009\000\000\000\001\000\
\\014\000\021\000\023\000\020\000\024\000\019\000\
\\025\000\018\000\026\000\017\000\027\000\016\000\040\000\009\000\000\000\001\000\
\\014\000\021\000\023\000\020\000\024\000\019\000\
\\025\000\018\000\026\000\017\000\027\000\016\000\040\000\009\000\000\000\001\000\
\\000\000\146\000\
\\000\000\133\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\014\000\037\000\022\000\036\000\025\000\018\000\
\\035\000\035\000\037\000\034\000\043\000\033\000\000\000\001\000\
\\000\000\145\000\
\\000\000\144\000\
\\000\000\142\000\
\\012\000\056\000\022\000\055\000\000\000\131\000\
\\012\000\056\000\022\000\055\000\000\000\129\000\
\\001\000\000\000\004\000\000\000\000\000\001\000\
\"
val gotoT =
"\
\\001\000\106\000\002\000\006\000\003\000\005\000\
\\005\000\004\000\008\000\003\000\009\000\002\000\010\000\001\000\000\000\000\000\
\\000\000\000\000\
\\003\000\021\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\028\000\004\000\027\000\000\000\000\000\
\\003\000\030\000\006\000\029\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\039\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\042\000\004\000\041\000\000\000\000\000\
\\003\000\043\000\000\000\000\000\
\\003\000\045\000\005\000\004\000\008\000\044\000\
\\009\000\002\000\010\000\001\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\030\000\006\000\046\000\000\000\000\000\
\\003\000\045\000\005\000\004\000\008\000\047\000\
\\009\000\002\000\010\000\001\000\000\000\000\000\
\\003\000\030\000\006\000\048\000\000\000\000\000\
\\003\000\045\000\005\000\004\000\008\000\049\000\
\\009\000\002\000\010\000\001\000\000\000\000\000\
\\003\000\030\000\006\000\050\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\056\000\000\000\000\000\
\\000\000\000\000\
\\003\000\057\000\000\000\000\000\
\\000\000\000\000\
\\003\000\030\000\006\000\059\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\030\000\006\000\070\000\007\000\069\000\000\000\000\000\
\\003\000\028\000\004\000\071\000\000\000\000\000\
\\003\000\030\000\006\000\072\000\000\000\000\000\
\\003\000\030\000\006\000\073\000\000\000\000\000\
\\003\000\030\000\006\000\074\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\030\000\006\000\070\000\007\000\079\000\000\000\000\000\
\\000\000\000\000\
\\003\000\030\000\006\000\081\000\000\000\000\000\
\\003\000\030\000\006\000\082\000\000\000\000\000\
\\003\000\030\000\006\000\070\000\007\000\083\000\000\000\000\000\
\\003\000\030\000\006\000\084\000\000\000\000\000\
\\003\000\045\000\005\000\004\000\008\000\085\000\
\\009\000\002\000\010\000\001\000\000\000\000\000\
\\003\000\030\000\006\000\070\000\007\000\086\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\030\000\006\000\089\000\000\000\000\000\
\\003\000\030\000\006\000\090\000\000\000\000\000\
\\003\000\030\000\006\000\091\000\000\000\000\000\
\\003\000\030\000\006\000\092\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\045\000\005\000\004\000\008\000\097\000\
\\009\000\002\000\010\000\001\000\000\000\000\000\
\\003\000\030\000\006\000\070\000\007\000\098\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\045\000\005\000\004\000\008\000\101\000\
\\009\000\002\000\010\000\001\000\000\000\000\000\
\\003\000\045\000\005\000\004\000\008\000\102\000\
\\009\000\002\000\010\000\001\000\000\000\000\000\
\\003\000\045\000\005\000\004\000\008\000\103\000\
\\009\000\002\000\010\000\001\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\003\000\030\000\006\000\104\000\000\000\000\000\
\\003\000\030\000\006\000\105\000\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\\000\000\000\000\
\"
val numstates = 107
val string_to_int = fn(s,index) => (ordof(s,index) + 
			ordof(s,index+1)*256,index+2)
	val convert_string_to_row = fn (conv_key,conv_entry) =>
	     fn(s,index) =>
		let fun f (r,index) =
			let val (num,index) = string_to_int(s,index)
			    val (i,index) = string_to_int(s,index)
			in if num=0 then ((rev r,conv_entry i),index)
			   else f((conv_key (num-1),conv_entry i)::r,index)
			end
		in f(nil,index)
		end
	 val convert_string_to_row_list = fn conv_funcs => fn s =>
		    let val convert_row =convert_string_to_row conv_funcs
		 	fun f(r,index) =
			  if index < String.length s then
			    let val (newlist,index) = convert_row (s,index)
			    in f(newlist::r,index)
			    end
			  else rev r
		    in f(nil,0)
		    end
	 val entry_to_action = fn j =>
		       if j=0 then ACCEPT
		       else if j=1 then ERROR
		       else if j >= (numstates+2) then REDUCE (j-numstates-2)
		       else SHIFT (STATE (j-2))
	 val make_goto_table = convert_string_to_row_list(NT,STATE)
	 val make_action_table=convert_string_to_row_list(T,entry_to_action)
	 val gotoT = map (fn (a,b) => a) (make_goto_table gotoT)
	 val actionT = make_action_table actionT
     in LrTable.mkLrTable {actions=actionT,gotos=gotoT,
	  numStates=numstates,initialState=STATE 0}
     end
end
local open Header in
type pos = int
type arg = unit
structure MlyValue = 
struct
datatype svalue = VOID | ntVOID of unit | T_STR_CONST of  (string)
 | T_INT_CONST of  (string) | T_ID of  (string)
 | bnd of  (ParseRes.Trm.pretrm) | appl of  (ParseRes.Trm.pretrm)
 | term of  (ParseRes.Trm.pretrm)
 | tplist of  (ParseRes.Typ.pretyp list)
 | tp of  (ParseRes.Typ.pretyp) | const of  (Id.T)
 | idlist of  (Id.T list) | id of  (Id.T) | setcmd of  (ParseRes.T)
 | start of  (ParseRes.T)
end
type svalue = MlyValue.svalue
type result = ParseRes.T
end
structure EC=
struct
open LrTable
val is_keyword =
fn _ => false
val preferred_insert =
fn (T 1) => true | (T 38) => true | _ => false
val preferred_subst =
fn  _ => nil
val noShift = 
fn (T 3) => true | (T 0) => true | _ => false
val showTerminal =
fn (T 0) => "T_EOF"
  | (T 1) => "T_DOT"
  | (T 2) => "T_COLON"
  | (T 3) => "T_SEMICOLON"
  | (T 4) => "T_LEQ"
  | (T 5) => "T_COMMA"
  | (T 6) => "T_APOST"
  | (T 7) => "T_EQ"
  | (T 8) => "T_DOUBLEEQ"
  | (T 9) => "T_DOLLAR"
  | (T 10) => "T_AT"
  | (T 11) => "T_ARROW"
  | (T 12) => "T_DARROW"
  | (T 13) => "T_LPAREN"
  | (T 14) => "T_RPAREN"
  | (T 15) => "T_LBRACK"
  | (T 16) => "T_RBRACK"
  | (T 17) => "T_LANGLE"
  | (T 18) => "T_RANGLE"
  | (T 19) => "T_LCURLY"
  | (T 20) => "T_RCURLY"
  | (T 21) => "T_INTER"
  | (T 22) => "T_LAMBDA"
  | (T 23) => "T_BIGLAMBDA"
  | (T 24) => "T_ID"
  | (T 25) => "T_INT_CONST"
  | (T 26) => "T_STR_CONST"
  | (T 27) => "T_USE"
  | (T 28) => "T_TYPE"
  | (T 29) => "T_SET"
  | (T 30) => "T_RESET"
  | (T 31) => "T_DEBUG"
  | (T 32) => "T_CHECK"
  | (T 33) => "T_WITH"
  | (T 34) => "T_ALL"
  | (T 35) => "T_IN"
  | (T 36) => "T_NS"
  | (T 37) => "T_CASE"
  | (T 38) => "T_OF"
  | (T 39) => "T_FOR"
  | (T 40) => "T_OBSERVE"
  | (T 41) => "T_INSTALL"
  | (T 42) => "T_SOME"
  | (T 43) => "T_OPEN"
  | (T 44) => "T_END"
  | (T 45) => "T_PACK"
  | _ => "bogus-term"
val errtermvalue=
let open Header in
fn _ => MlyValue.VOID
end
val terms = (T 0) :: (T 1) :: (T 2) :: (T 3) :: (T 4) :: (T 5) :: (T 6
) :: (T 7) :: (T 8) :: (T 9) :: (T 10) :: (T 11) :: (T 12) :: (T 13)
 :: (T 14) :: (T 15) :: (T 16) :: (T 17) :: (T 18) :: (T 19) :: (T 20)
 :: (T 21) :: (T 22) :: (T 23) :: (T 27) :: (T 28) :: (T 29) :: (T 30)
 :: (T 31) :: (T 32) :: (T 33) :: (T 34) :: (T 35) :: (T 36) :: (T 37)
 :: (T 38) :: (T 39) :: (T 40) :: (T 41) :: (T 42) :: (T 43) :: (T 44)
 :: (T 45) :: nil
end
structure Actions =
struct 
exception mlyAction of int
val actions = 
let open Header
in
fn (i392,defaultPos,stack,
    (()):arg) =>
case (i392,stack)
of (0,(_,(MlyValue.tp (tp2),tp2left,tp2right)) :: (_,(_,T_LEQleft as 
T_LEQ1left,T_LEQright as T_LEQ1right)) :: (_,(MlyValue.tp (tp1 as tp),
tpleft as tp1left,tpright as tp1right)) :: (_,(_,T_CHECKleft as 
T_CHECK1left,T_CHECKright as T_CHECK1right)) :: rest671) =>
let val result = 
MlyValue.start (((Leq(tp1,tp2))))
in (LrTable.NT 0,(result,T_CHECK1left,tp2right),rest671)
end
| (1,(_,(MlyValue.T_ID (T_ID1 as T_ID),T_IDleft as T_ID1left,
T_IDright as T_ID1right)) :: (_,(_,T_USEleft as T_USE1left,
T_USEright as T_USE1right)) :: rest671) =>
let val result = 
MlyValue.start (((Use(T_ID))))
in (LrTable.NT 0,(result,T_USE1left,T_ID1right),rest671)
end
| (2,(_,(MlyValue.tp (tp1 as tp),tpleft as tp1left,tpright as tp1right
)) :: (_,(_,T_LEQleft as T_LEQ1left,T_LEQright as T_LEQ1right)) :: 
(_,(MlyValue.id (id1 as id),idleft as id1left,idright as id1right
)) :: (_,(_,T_TYPEleft as T_TYPE1left,T_TYPEright as T_TYPE1right
)) :: rest671) =>
let val result = 
MlyValue.start (((Type_Assumption(id,tp))))
in (LrTable.NT 0,(result,T_TYPE1left,tp1right),rest671)
end
| (3,(_,(MlyValue.tp (tp1 as tp),tpleft as tp1left,tpright as tp1right
)) :: (_,(_,T_LEQleft as T_LEQ1left,T_LEQright as T_LEQ1right)) :: 
(_,(MlyValue.id (id1 as id),idleft as id1left,idright as id1right
)) :: rest671) =>
let val result = 
MlyValue.start (((Type_Assumption(id,tp))))
in (LrTable.NT 0,(result,id1left,tp1right),rest671)
end
| (4,(_,(MlyValue.tp (tp1 as tp),tpleft as tp1left,tpright as tp1right
)) :: (_,(_,T_DOUBLEEQleft as T_DOUBLEEQ1left,T_DOUBLEEQright as 
T_DOUBLEEQ1right)) :: (_,(MlyValue.id (id1 as id),idleft as id1left,
idright as id1right)) :: (_,(_,T_TYPEleft as T_TYPE1left,
T_TYPEright as T_TYPE1right)) :: rest671) =>
let val result = 
MlyValue.start (((Type_Abbrev(id,tp))))
in (LrTable.NT 0,(result,T_TYPE1left,tp1right),rest671)
end
| (5,(_,(MlyValue.tp (tp1 as tp),tpleft as tp1left,tpright as tp1right
)) :: (_,(_,T_DOUBLEEQleft as T_DOUBLEEQ1left,T_DOUBLEEQright as 
T_DOUBLEEQ1right)) :: (_,(MlyValue.id (id1 as id),idleft as id1left,
idright as id1right)) :: rest671) =>
let val result = 
MlyValue.start (((Type_Abbrev(id,tp))))
in (LrTable.NT 0,(result,id1left,tp1right),rest671)
end
| (6,(_,(MlyValue.term (term1 as term),termleft as term1left,
termright as term1right)) :: (_,(_,T_EQleft as T_EQ1left,T_EQright as 
T_EQ1right)) :: (_,(MlyValue.id (id1 as id),idleft as id1left,
idright as id1right)) :: rest671) =>
let val result = 
MlyValue.start (((Term_Def(id,term))))
in (LrTable.NT 0,(result,id1left,term1right),rest671)
end
| (7,(_,(MlyValue.term (term1 as term),termleft as term1left,
termright as term1right)) :: rest671) =>
let val result = 
MlyValue.start (((Term_Def(Id.intern "it",term))))
in (LrTable.NT 0,(result,term1left,term1right),rest671)
end
| (8,(_,(_,T_EOFleft as T_EOF1left,T_EOFright as T_EOF1right)) :: rest671) =>
let val result = 
MlyValue.start (((Nothing)))
in (LrTable.NT 0,(result,T_EOF1left,T_EOF1right),rest671)
end
| (9,(_,(MlyValue.setcmd (setcmd1 as setcmd),setcmdleft as setcmd1left
,setcmdright as setcmd1right)) :: rest671) =>
let val result = 
MlyValue.start (((setcmd)))
in (LrTable.NT 0,(result,setcmd1left,setcmd1right),rest671)
end
| (10,(_,(MlyValue.T_ID (T_ID1 as T_ID),T_IDleft as T_ID1left,
T_IDright as T_ID1right)) :: (_,(_,T_SETleft as T_SET1left,
T_SETright as T_SET1right)) :: rest671) =>
let val result = 
MlyValue.setcmd (((Set(T_ID,"true"))))
in (LrTable.NT 1,(result,T_SET1left,T_ID1right),rest671)
end
| (11,(_,(MlyValue.T_ID (T_ID1 as T_ID),T_IDleft as T_ID1left,
T_IDright as T_ID1right)) :: (_,(_,T_DEBUGleft as T_DEBUG1left,
T_DEBUGright as T_DEBUG1right)) :: rest671) =>
let val result = 
MlyValue.setcmd (((Set(T_ID,"true"))))
in (LrTable.NT 1,(result,T_DEBUG1left,T_ID1right),rest671)
end
| (12,(_,(MlyValue.T_ID (T_ID1 as T_ID),T_IDleft as T_ID1left,
T_IDright as T_ID1right)) :: (_,(_,T_RESETleft as T_RESET1left,
T_RESETright as T_RESET1right)) :: rest671) =>
let val result = 
MlyValue.setcmd (((Set(T_ID,"false"))))
in (LrTable.NT 1,(result,T_RESET1left,T_ID1right),rest671)
end
| (13,(_,(MlyValue.id (id1 as id),idleft as id1left,idright as 
id1right)) :: rest671) =>
let val result = 
MlyValue.tp (((PRETVAR(id))))
in (LrTable.NT 5,(result,id1left,id1right),rest671)
end
| (14,(_,(MlyValue.tp (tp2),tp2left,tp2right)) :: (_,(_,
T_ARROWleft as T_ARROW1left,T_ARROWright as T_ARROW1right)) :: (_,(
MlyValue.tp (tp1 as tp),tpleft as tp1left,tpright as tp1right)) :: rest671) =>
let val result = 
MlyValue.tp (((PREARROW(tp1,tp2))))
in (LrTable.NT 5,(result,tp1left,tp2right),rest671)
end
| (15,(_,(MlyValue.tp (tp2),tp2left,tp2right)) :: (_,(_,
T_INTERleft as T_INTER1left,T_INTERright as T_INTER1right)) :: (_,(
MlyValue.tp (tp1 as tp),tpleft as tp1left,tpright as tp1right)) :: rest671) =>
let val result = 
MlyValue.tp (((PREMEET([tp1,tp2]))))
in (LrTable.NT 5,(result,tp1left,tp2right),rest671)
end
| (16,(_,(_,T_RBRACKleft as T_RBRACK1left,T_RBRACKright as 
T_RBRACK1right)) :: (_,(MlyValue.tplist (tplist1 as tplist),
tplistleft as tplist1left,tplistright as tplist1right)) :: (_,(_,
T_LBRACKleft as T_LBRACK1left,T_LBRACKright as T_LBRACK1right)) :: 
(_,(_,T_INTERleft as T_INTER1left,T_INTERright as T_INTER1right)) :: rest671) =>
let val result = 
MlyValue.tp (((PREMEET(tplist))))
in (LrTable.NT 5,(result,T_INTER1left,T_RBRACK1right),rest671)
end
| (17,(_,(_,T_NSleft as T_NS1left,T_NSright as T_NS1right)) :: rest671) =>
let val result = 
MlyValue.tp (((PREMEET([]))))
in (LrTable.NT 5,(result,T_NS1left,T_NS1right),rest671)
end
| (18,(_,(_,T_RPARENleft as T_RPAREN1left,T_RPARENright as 
T_RPAREN1right)) :: (_,(MlyValue.tp (tp1 as tp),tpleft as tp1left,
tpright as tp1right)) :: (_,(_,T_LPARENleft as T_LPAREN1left,
T_LPARENright as T_LPAREN1right)) :: rest671) =>
let val result = 
MlyValue.tp (((tp)))
in (LrTable.NT 5,(result,T_LPAREN1left,T_RPAREN1right),rest671)
end
| (19,(_,(MlyValue.tp (tp1 as tp),tpleft as tp1left,tpright as 
tp1right)) :: (_,(_,T_DOTleft as T_DOT1left,T_DOTright as T_DOT1right
)) :: (_,(MlyValue.id (id1 as id),idleft as id1left,idright as 
id1right)) :: (_,(_,T_ALLleft as T_ALL1left,T_ALLright as T_ALL1right
)) :: rest671) =>
let val result = 
MlyValue.tp (((PREALL(id, PREMEET[], tp))))
in (LrTable.NT 5,(result,T_ALL1left,tp1right),rest671)
end
| (20,(_,(MlyValue.tp (tp2),tp2left,tp2right)) :: (_,(_,T_DOTleft as 
T_DOT1left,T_DOTright as T_DOT1right)) :: (_,(MlyValue.tp (tp1 as tp),
tpleft as tp1left,tpright as tp1right)) :: (_,(_,T_LEQleft as 
T_LEQ1left,T_LEQright as T_LEQ1right)) :: (_,(MlyValue.id (id1 as id),
idleft as id1left,idright as id1right)) :: (_,(_,T_ALLleft as 
T_ALL1left,T_ALLright as T_ALL1right)) :: rest671) =>
let val result = 
MlyValue.tp (((PREALL(id, tp1, tp2))))
in (LrTable.NT 5,(result,T_ALL1left,tp2right),rest671)
end
| (21,(_,(MlyValue.tp (tp1 as tp),tpleft as tp1left,tpright as 
tp1right)) :: (_,(_,T_DOTleft as T_DOT1left,T_DOTright as T_DOT1right
)) :: (_,(MlyValue.id (id1 as id),idleft as id1left,idright as 
id1right)) :: (_,(_,T_SOMEleft as T_SOME1left,T_SOMEright as 
T_SOME1right)) :: rest671) =>
let val result = 
MlyValue.tp (((
let val b = Id.new()
		     val bv = PRETVAR(b)
		     val idv = PRETVAR(id)
		 in PREALL(b,PREMEET[], 
		 	   PREARROW(PREALL(id, PREMEET[], 
		 	   		PREARROW(tp, bv)),
			         bv))
		 end
)))
in (LrTable.NT 5,(result,T_SOME1left,tp1right),rest671)
end
| (22,(_,(MlyValue.tp (tp2),tp2left,tp2right)) :: (_,(_,T_DOTleft as 
T_DOT1left,T_DOTright as T_DOT1right)) :: (_,(MlyValue.tp (tp1 as tp),
tpleft as tp1left,tpright as tp1right)) :: (_,(_,T_LEQleft as 
T_LEQ1left,T_LEQright as T_LEQ1right)) :: (_,(MlyValue.id (id1 as id),
idleft as id1left,idright as id1right)) :: (_,(_,T_SOMEleft as 
T_SOME1left,T_SOMEright as T_SOME1right)) :: rest671) =>
let val result = 
MlyValue.tp (((
let val b = Id.new()
		     val bv = PRETVAR(b)
		     val idv = PRETVAR(id)
		 in PREALL(b,PREMEET[], 
		 	   PREARROW(PREALL(id, tp1, 
		 	   		PREARROW(tp2, bv)),
			         bv))
		 end
)))
in (LrTable.NT 5,(result,T_SOME1left,tp2right),rest671)
end
| (23,(_,(MlyValue.tp (tp1 as tp),tpleft as tp1left,tpright as 
tp1right)) :: rest671) =>
let val result = 
MlyValue.tplist ((([tp])))
in (LrTable.NT 6,(result,tp1left,tp1right),rest671)
end
| (24,(_,(MlyValue.tplist (tplist1 as tplist),tplistleft as 
tplist1left,tplistright as tplist1right)) :: (_,(_,T_COMMAleft as 
T_COMMA1left,T_COMMAright as T_COMMA1right)) :: (_,(MlyValue.tp (tp1
 as tp),tpleft as tp1left,tpright as tp1right)) :: rest671) =>
let val result = 
MlyValue.tplist (((tp::tplist)))
in (LrTable.NT 6,(result,tp1left,tplist1right),rest671)
end
| (25,(_,(MlyValue.id (id1 as id),idleft as id1left,idright as 
id1right)) :: rest671) =>
let val result = 
MlyValue.idlist ((([id])))
in (LrTable.NT 3,(result,id1left,id1right),rest671)
end
| (26,(_,(MlyValue.idlist (idlist1 as idlist),idlistleft as 
idlist1left,idlistright as idlist1right)) :: (_,(_,T_COMMAleft as 
T_COMMA1left,T_COMMAright as T_COMMA1right)) :: (_,(MlyValue.id (id1
 as id),idleft as id1left,idright as id1right)) :: rest671) =>
let val result = 
MlyValue.idlist (((id::idlist)))
in (LrTable.NT 3,(result,id1left,idlist1right),rest671)
end
| (27,(_,(MlyValue.T_ID (T_ID1 as T_ID),T_IDleft as T_ID1left,
T_IDright as T_ID1right)) :: rest671) =>
let val result = 
MlyValue.id (((Id.intern T_ID)))
in (LrTable.NT 2,(result,T_ID1left,T_ID1right),rest671)
end
| (28,(_,(MlyValue.id (id1 as id),idleft as id1left,idright as 
id1right)) :: rest671) =>
let val result = 
MlyValue.const (((id)))
in (LrTable.NT 4,(result,id1left,id1right),rest671)
end
| (29,(_,(MlyValue.T_INT_CONST (T_INT_CONST1 as T_INT_CONST),
T_INT_CONSTleft as T_INT_CONST1left,T_INT_CONSTright as 
T_INT_CONST1right)) :: rest671) =>
let val result = 
MlyValue.const (((Id.intern T_INT_CONST)))
in (LrTable.NT 4,(result,T_INT_CONST1left,T_INT_CONST1right),rest671)
end
| (30,(_,(MlyValue.T_STR_CONST (T_STR_CONST1 as T_STR_CONST),
T_STR_CONSTleft as T_STR_CONST1left,T_STR_CONSTright as 
T_STR_CONST1right)) :: rest671) =>
let val result = 
MlyValue.const (((Id.intern T_STR_CONST)))
in (LrTable.NT 4,(result,T_STR_CONST1left,T_STR_CONST1right),rest671)
end
| (31,(_,(MlyValue.appl (appl1 as appl),applleft as appl1left,
applright as appl1right)) :: rest671) =>
let val result = 
MlyValue.term (((appl)))
in (LrTable.NT 7,(result,appl1left,appl1right),rest671)
end
| (32,(_,(MlyValue.bnd (bnd1 as bnd),bndleft as bnd1left,bndright as 
bnd1right)) :: rest671) =>
let val result = 
MlyValue.term (((bnd)))
in (LrTable.NT 7,(result,bnd1left,bnd1right),rest671)
end
| (33,(_,(MlyValue.term (term1 as term),termleft as term1left,
termright as term1right)) :: (_,(_,T_DOTleft as T_DOT1left,
T_DOTright as T_DOT1right)) :: (_,(MlyValue.tplist (tplist1 as tplist)
,tplistleft as tplist1left,tplistright as tplist1right)) :: (_,(_,
T_COLONleft as T_COLON1left,T_COLONright as T_COLON1right)) :: (_,(
MlyValue.id (id1 as id),idleft as id1left,idright as id1right)) :: 
(_,(_,T_LAMBDAleft as T_LAMBDA1left,T_LAMBDAright as T_LAMBDA1right
)) :: rest671) =>
let val result = 
MlyValue.bnd (((
case tplist of 
					    [t] => PREABS(id,t,term)
					  | ts  => 
					      let val a = Id.new_from id
					      in PREFOR(a,ts,
					      	   PREABS(id,PRETVAR(a),term))
					      end
)))
in (LrTable.NT 9,(result,T_LAMBDA1left,term1right),rest671)
end
| (34,(_,(MlyValue.term (term1 as term),termleft as term1left,
termright as term1right)) :: (_,(_,T_DOTleft as T_DOT1left,
T_DOTright as T_DOT1right)) :: (_,(MlyValue.id (id1 as id),idleft as 
id1left,idright as id1right)) :: (_,(_,T_BIGLAMBDAleft as 
T_BIGLAMBDA1left,T_BIGLAMBDAright as T_BIGLAMBDA1right)) :: rest671) =>
let val result = 
MlyValue.bnd (((PRETABS(id,PREMEET[],term))))
in (LrTable.NT 9,(result,T_BIGLAMBDA1left,term1right),rest671)
end
| (35,(_,(MlyValue.term (term1 as term),termleft as term1left,
termright as term1right)) :: (_,(_,T_DOTleft as T_DOT1left,
T_DOTright as T_DOT1right)) :: (_,(MlyValue.tp (tp1 as tp),tpleft as 
tp1left,tpright as tp1right)) :: (_,(_,T_LEQleft as T_LEQ1left,
T_LEQright as T_LEQ1right)) :: (_,(MlyValue.id (id1 as id),idleft as 
id1left,idright as id1right)) :: (_,(_,T_BIGLAMBDAleft as 
T_BIGLAMBDA1left,T_BIGLAMBDAright as T_BIGLAMBDA1right)) :: rest671) =>
let val result = 
MlyValue.bnd (((PRETABS(id,tp,term))))
in (LrTable.NT 9,(result,T_BIGLAMBDA1left,term1right),rest671)
end
| (36,(_,(MlyValue.term (term1 as term),termleft as term1left,
termright as term1right)) :: (_,(_,T_DOTleft as T_DOT1left,
T_DOTright as T_DOT1right)) :: (_,(MlyValue.tplist (tplist1 as tplist)
,tplistleft as tplist1left,tplistright as tplist1right)) :: (_,(_,
T_INleft as T_IN1left,T_INright as T_IN1right)) :: (_,(MlyValue.idlist
 (idlist1 as idlist),idlistleft as idlist1left,idlistright as 
idlist1right)) :: (_,(_,T_BIGLAMBDAleft as T_BIGLAMBDA1left,
T_BIGLAMBDAright as T_BIGLAMBDA1right)) :: rest671) =>
let val result = 
MlyValue.bnd (((
let fun f [] = term
					       | f (v::vs) =
						    PREFOR(v, tplist, f vs)
					 in f idlist end
)))
in (LrTable.NT 9,(result,T_BIGLAMBDA1left,term1right),rest671)
end
| (37,(_,(MlyValue.term (term1 as term),termleft as term1left,
termright as term1right)) :: (_,(_,T_DOTleft as T_DOT1left,
T_DOTright as T_DOT1right)) :: (_,(MlyValue.tplist (tplist1 as tplist)
,tplistleft as tplist1left,tplistright as tplist1right)) :: (_,(_,
T_INleft as T_IN1left,T_INright as T_IN1right)) :: (_,(MlyValue.idlist
 (idlist1 as idlist),idlistleft as idlist1left,idlistright as 
idlist1right)) :: (_,(_,T_FORleft as T_FOR1left,T_FORright as 
T_FOR1right)) :: rest671) =>
let val result = 
MlyValue.bnd (((
let fun f [] = term
					       | f (v::vs) =
						    PREFOR(v, tplist, f vs)
					 in f idlist end
)))
in (LrTable.NT 9,(result,T_FOR1left,term1right),rest671)
end
| (38,(_,(MlyValue.const (const1 as const),constleft as const1left,
constright as const1right)) :: rest671) =>
let val result = 
MlyValue.appl (((PREVAR(const))))
in (LrTable.NT 8,(result,const1left,const1right),rest671)
end
| (39,(_,(_,T_RPARENleft as T_RPAREN1left,T_RPARENright as 
T_RPAREN1right)) :: (_,(MlyValue.term (term1 as term),termleft as 
term1left,termright as term1right)) :: (_,(_,T_LPARENleft as 
T_LPAREN1left,T_LPARENright as T_LPAREN1right)) :: rest671) =>
let val result = 
MlyValue.appl (((term)))
in (LrTable.NT 8,(result,T_LPAREN1left,T_RPAREN1right),rest671)
end
| (40,(_,(MlyValue.id (id1 as id),idleft as id1left,idright as 
id1right)) :: (_,(MlyValue.appl (appl1 as appl),applleft as appl1left,
applright as appl1right)) :: rest671) =>
let val result = 
MlyValue.appl (((PREAPP(appl,PREVAR(id)))))
in (LrTable.NT 8,(result,appl1left,id1right),rest671)
end
| (41,(_,(_,T_RPARENleft as T_RPAREN1left,T_RPARENright as 
T_RPAREN1right)) :: (_,(MlyValue.term (term1 as term),termleft as 
term1left,termright as term1right)) :: (_,(_,T_LPARENleft as 
T_LPAREN1left,T_LPARENright as T_LPAREN1right)) :: (_,(MlyValue.appl (
appl1 as appl),applleft as appl1left,applright as appl1right)) :: rest671) =>
let val result = 
MlyValue.appl (((PREAPP(appl,term))))
in (LrTable.NT 8,(result,appl1left,T_RPAREN1right),rest671)
end
| (42,(_,(_,T_RBRACKleft as T_RBRACK1left,T_RBRACKright as 
T_RBRACK1right)) :: (_,(MlyValue.tp (tp1 as tp),tpleft as tp1left,
tpright as tp1right)) :: (_,(_,T_LBRACKleft as T_LBRACK1left,
T_LBRACKright as T_LBRACK1right)) :: (_,(MlyValue.appl (appl1 as appl)
,applleft as appl1left,applright as appl1right)) :: rest671) =>
let val result = 
MlyValue.appl (((PRETAPP(appl,tp))))
in (LrTable.NT 8,(result,appl1left,T_RBRACK1right),rest671)
end
| _ => raise (mlyAction i392)
end
val void = MlyValue.VOID
val extract = fn a => (fn MlyValue.start x => x
| _ => let exception ParseInternal
	in raise ParseInternal end) a 
end
end
structure Tokens : FMEET_TOKENS =
struct
type svalue = ParserData.svalue
type ('a,'b) token = ('a,'b) Token.token
fun T_EOF (p1,p2) = Token.TOKEN (ParserData.LrTable.T 0,(
ParserData.MlyValue.VOID,p1,p2))
fun T_DOT (p1,p2) = Token.TOKEN (ParserData.LrTable.T 1,(
ParserData.MlyValue.VOID,p1,p2))
fun T_COLON (p1,p2) = Token.TOKEN (ParserData.LrTable.T 2,(
ParserData.MlyValue.VOID,p1,p2))
fun T_SEMICOLON (p1,p2) = Token.TOKEN (ParserData.LrTable.T 3,(
ParserData.MlyValue.VOID,p1,p2))
fun T_LEQ (p1,p2) = Token.TOKEN (ParserData.LrTable.T 4,(
ParserData.MlyValue.VOID,p1,p2))
fun T_COMMA (p1,p2) = Token.TOKEN (ParserData.LrTable.T 5,(
ParserData.MlyValue.VOID,p1,p2))
fun T_APOST (p1,p2) = Token.TOKEN (ParserData.LrTable.T 6,(
ParserData.MlyValue.VOID,p1,p2))
fun T_EQ (p1,p2) = Token.TOKEN (ParserData.LrTable.T 7,(
ParserData.MlyValue.VOID,p1,p2))
fun T_DOUBLEEQ (p1,p2) = Token.TOKEN (ParserData.LrTable.T 8,(
ParserData.MlyValue.VOID,p1,p2))
fun T_DOLLAR (p1,p2) = Token.TOKEN (ParserData.LrTable.T 9,(
ParserData.MlyValue.VOID,p1,p2))
fun T_AT (p1,p2) = Token.TOKEN (ParserData.LrTable.T 10,(
ParserData.MlyValue.VOID,p1,p2))
fun T_ARROW (p1,p2) = Token.TOKEN (ParserData.LrTable.T 11,(
ParserData.MlyValue.VOID,p1,p2))
fun T_DARROW (p1,p2) = Token.TOKEN (ParserData.LrTable.T 12,(
ParserData.MlyValue.VOID,p1,p2))
fun T_LPAREN (p1,p2) = Token.TOKEN (ParserData.LrTable.T 13,(
ParserData.MlyValue.VOID,p1,p2))
fun T_RPAREN (p1,p2) = Token.TOKEN (ParserData.LrTable.T 14,(
ParserData.MlyValue.VOID,p1,p2))
fun T_LBRACK (p1,p2) = Token.TOKEN (ParserData.LrTable.T 15,(
ParserData.MlyValue.VOID,p1,p2))
fun T_RBRACK (p1,p2) = Token.TOKEN (ParserData.LrTable.T 16,(
ParserData.MlyValue.VOID,p1,p2))
fun T_LANGLE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 17,(
ParserData.MlyValue.VOID,p1,p2))
fun T_RANGLE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 18,(
ParserData.MlyValue.VOID,p1,p2))
fun T_LCURLY (p1,p2) = Token.TOKEN (ParserData.LrTable.T 19,(
ParserData.MlyValue.VOID,p1,p2))
fun T_RCURLY (p1,p2) = Token.TOKEN (ParserData.LrTable.T 20,(
ParserData.MlyValue.VOID,p1,p2))
fun T_INTER (p1,p2) = Token.TOKEN (ParserData.LrTable.T 21,(
ParserData.MlyValue.VOID,p1,p2))
fun T_LAMBDA (p1,p2) = Token.TOKEN (ParserData.LrTable.T 22,(
ParserData.MlyValue.VOID,p1,p2))
fun T_BIGLAMBDA (p1,p2) = Token.TOKEN (ParserData.LrTable.T 23,(
ParserData.MlyValue.VOID,p1,p2))
fun T_ID (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 24,(
ParserData.MlyValue.T_ID i,p1,p2))
fun T_INT_CONST (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 25,(
ParserData.MlyValue.T_INT_CONST i,p1,p2))
fun T_STR_CONST (i,p1,p2) = Token.TOKEN (ParserData.LrTable.T 26,(
ParserData.MlyValue.T_STR_CONST i,p1,p2))
fun T_USE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 27,(
ParserData.MlyValue.VOID,p1,p2))
fun T_TYPE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 28,(
ParserData.MlyValue.VOID,p1,p2))
fun T_SET (p1,p2) = Token.TOKEN (ParserData.LrTable.T 29,(
ParserData.MlyValue.VOID,p1,p2))
fun T_RESET (p1,p2) = Token.TOKEN (ParserData.LrTable.T 30,(
ParserData.MlyValue.VOID,p1,p2))
fun T_DEBUG (p1,p2) = Token.TOKEN (ParserData.LrTable.T 31,(
ParserData.MlyValue.VOID,p1,p2))
fun T_CHECK (p1,p2) = Token.TOKEN (ParserData.LrTable.T 32,(
ParserData.MlyValue.VOID,p1,p2))
fun T_WITH (p1,p2) = Token.TOKEN (ParserData.LrTable.T 33,(
ParserData.MlyValue.VOID,p1,p2))
fun T_ALL (p1,p2) = Token.TOKEN (ParserData.LrTable.T 34,(
ParserData.MlyValue.VOID,p1,p2))
fun T_IN (p1,p2) = Token.TOKEN (ParserData.LrTable.T 35,(
ParserData.MlyValue.VOID,p1,p2))
fun T_NS (p1,p2) = Token.TOKEN (ParserData.LrTable.T 36,(
ParserData.MlyValue.VOID,p1,p2))
fun T_CASE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 37,(
ParserData.MlyValue.VOID,p1,p2))
fun T_OF (p1,p2) = Token.TOKEN (ParserData.LrTable.T 38,(
ParserData.MlyValue.VOID,p1,p2))
fun T_FOR (p1,p2) = Token.TOKEN (ParserData.LrTable.T 39,(
ParserData.MlyValue.VOID,p1,p2))
fun T_OBSERVE (p1,p2) = Token.TOKEN (ParserData.LrTable.T 40,(
ParserData.MlyValue.VOID,p1,p2))
fun T_INSTALL (p1,p2) = Token.TOKEN (ParserData.LrTable.T 41,(
ParserData.MlyValue.VOID,p1,p2))
fun T_SOME (p1,p2) = Token.TOKEN (ParserData.LrTable.T 42,(
ParserData.MlyValue.VOID,p1,p2))
fun T_OPEN (p1,p2) = Token.TOKEN (ParserData.LrTable.T 43,(
ParserData.MlyValue.VOID,p1,p2))
fun T_END (p1,p2) = Token.TOKEN (ParserData.LrTable.T 44,(
ParserData.MlyValue.VOID,p1,p2))
fun T_PACK (p1,p2) = Token.TOKEN (ParserData.LrTable.T 45,(
ParserData.MlyValue.VOID,p1,p2))
end
end
signature FILEUTILS = sig

val open_fmeet_file: string -> (instream * string)

end
functor Main(
	 structure Globals: GLOBALS
	 structure Typ: TYP
	 structure Trm: TRM
	 structure FileUtils: FILEUTILS
	 structure Parse: PARSE
	 structure Leq: LEQ
	 structure Synth: SYNTH
	 sharing Typ.Globals = Globals
	     and Parse.ParseRes.Typ = Typ
	     and Parse.ParseRes.Trm = Trm
	     and Leq.Typ = Typ
	     and Trm.Typ = Typ
	     and Synth.Trm = Trm
	     and Synth.Leq = Leq
	 val buildtime : string
	) = struct

open Globals;
open Parse.ParseRes;

val global_tenv = ref(Typ.empty_tenv)

exception NotABoolean

fun string_to_bool "true" = true
  | string_to_bool "True" = true
  | string_to_bool "TRUE" = true
  | string_to_bool "t" = true
  | string_to_bool "T" = true
  | string_to_bool "yes" = true
  | string_to_bool "Yes" = true
  | string_to_bool "YES" = true
  | string_to_bool "false" = false
  | string_to_bool "False" = false
  | string_to_bool "FALSE" = false
  | string_to_bool "f" = false
  | string_to_bool "F" = false
  | string_to_bool "no" = false
  | string_to_bool "No" = false
  | string_to_bool "NO" = false
  | string_to_bool _ = raise NotABoolean

fun rep_loop done parser error =
  while (not (done())) do
    (case parser() of
         Use(f) => 
           rep_loop_on_file f
       | Type_Assumption(i,pt) =>
       	   let val t = Typ.debruijnify (!global_tenv) pt
           in 
              write (Id.tostr i);
              write " <= ";
              Typ.prt (stdpp()) (!global_tenv) t;
              write "\n";
       	      global_tenv := Typ.push_bound (!global_tenv) i t
           end
       | Type_Abbrev(i,pt) =>
       	   let val t = Typ.debruijnify (!global_tenv) pt
       	       val _ = global_tenv := Typ.push_abbrev (!global_tenv) i t
           in 
              write (Id.tostr i);
              write " == ";
              Typ.prt (stdpp()) (!global_tenv) t;
              write "\n"
           end
       | Term_Def(i,ptrm) =>
       	   let val trm = Trm.debruijnify (!global_tenv) ptrm
       	       val typ = Synth.synth (!global_tenv) trm
           in 
              write (Id.tostr i);
              write " = ";
              Pp.setb (stdpp());
              Trm.prt (stdpp()) (!global_tenv) trm;
              Pp.break (stdpp()) true ~3;
              write " : ";
              Typ.prt (stdpp()) (!global_tenv) typ;
              Pp.break (stdpp()) true ~3;
              (* write " in ";
                 Typ.prt_tenv (stdpp()) (Typ.pop (!global_tenv)); *)
              Pp.endb (stdpp());
              write "\n";
              global_tenv := Typ.push_binding (!global_tenv) i typ
           end
       | Leq(pt1,pt2) =>
           let val t1 = Typ.debruijnify (!global_tenv) pt1
               val t2 = Typ.debruijnify (!global_tenv) pt2
           in 
              if Leq.leq (!global_tenv) t1 t2 
                then write "Yes\n"
                else write "No\n"
              end
       | Nothing => 
           ()
       | Set(name,v) => 
           (set_flag name (string_to_bool v))
       | _ => 
         write "Unimplemented ParseResult!\n"
    )
    handle 
      Typ.WrongKindOfId(te,i,which) => 
      	  (write ("Wrong kind of identifier: "^ (makestring i) ^" ("
      	  	  ^ which ^ " expected)\nin "); 
      	   Typ.prt_tenv (stdpp()) te;
       	   error())
    | unknown => 
          (write ("Exception: "^(System.exn_name unknown)^"\n"); 
       	   error())

and rep_loop_on_file fname =
	 let val (dev,real_name) = FileUtils.open_fmeet_file fname
	     val quit = ref false
	     fun parser() = Parse.stream_parse dev
	     fun done() = (!quit) orelse (end_of_stream dev)
	     fun error() = (quit := true);
	 in 
            write ("Reading from \"" ^ real_name ^ "\"...\n\n");
	    (rep_loop done parser error;
             write ("\nClosing " ^ real_name ^ "\n");
             close_in dev)
    	    handle Io(s) => write ("IO error on " ^ fname ^ ": " ^ s ^ "\n")
	 end

fun top() = 
  let fun top_done () = (print "> "; flush_out std_out; end_of_stream(std_in))
      fun top_error () = ()
  in
    write ("Welcome to FMEET (" ^ buildtime ^ ")...\n\n");
    rep_loop top_done Parse.top_parse top_error;
    write "\n"
  end

val read_from_file = ref "";

fun parse_switches ("-i"::s::rest) 
 	= (read_from_file := s;
 	   parse_switches rest)
  | parse_switches (s::rest) 
 	= (read_from_file := s;
 	   parse_switches rest)
  | parse_switches ([]) 
  	= ()

fun rep_command_line(argv,env) = 
  (parse_switches (tl argv);
   if (!read_from_file) = ""
      then top()
      else rep_loop_on_file (!read_from_file)
   )

fun process_file s = rep_command_line (["",s^".fm"],nil);

end


functor WrMgt(
	    structure Wr: WR
	    structure Pp: PP
	    sharing Pp.Wr = Wr
	    ) : WRMGT 
	    = struct

structure Wr = Wr;
structure Pp = Pp;

val current_underlying_wr = ref(Wr.to_stdout());
val current_pp = ref(Pp.pp_from_wr (!current_underlying_wr));
val current_wr = ref(Pp.wr_from_pp (!current_pp));

fun stdpp() = !current_pp;

fun get_current_wr() = !current_wr;

fun set_current_wr wr =
  (current_underlying_wr := wr;
   current_pp := Pp.pp_from_wr (!current_underlying_wr);
   current_wr := Pp.wr_from_pp (!current_pp))

fun write s = Pp.pwrite (!current_pp) s;

end
functor Id (structure SymTab: HASH
	    structure InvSymTab: TABLE
	    sharing type SymTab.elem = string
	        and type InvSymTab.key = int
	   ) : ID =
struct

val symtab = ref SymTab.empty;
val invsymtab = ref (InvSymTab.empty: string InvSymTab.table);

val DEBUG = ref false;

type T = int

exception CantHappen

fun intern (s:string) = 
      let val _ = if not (SymTab.exists(s,!symtab))
      			 then symtab := SymTab.add(s, (!symtab))
      			 else ()
          val i_opt = SymTab.find (s, (!symtab))
      in
         case i_opt of
           NONE => raise CantHappen
         | SOME(i) => 
             (invsymtab := InvSymTab.insert((i,s), (!invsymtab));
              i)
      end

fun hashcode i = i

exception UnknownId

fun tostr (i:T) : string = 
	let val s_opt = InvSymTab.find (i, (!invsymtab))
	in
	    case s_opt of
	       NONE => raise UnknownId
	     | SOME(s) => s
	end

val newvarcount = ref 0;

fun reset_new_counter() = (newvarcount := 0)

fun new_from i =
    let val _ = newvarcount := !newvarcount + 1
        val name = (tostr i) ^ "_" ^ (makestring (!newvarcount))
    in 
       if SymTab.exists(name,!symtab)
          then new_from i
          else intern name
    end

val id_x = intern "x"

fun new() = new_from id_x

fun == (i:T) (i':T) = (i = i')

fun lexlt (i:T) (i':T) = ((tostr i) < (tostr i'))

end
functor FileUtils(): FILEUTILS = struct

fun open_fmeet_file fname =
	(open_in fname,fname)
	handle Io(s) => 
	(open_in (fname ^ ".fm"), fname ^ ".fm")
	handle Io(s) => 
	(open_in ("examples/" ^ fname), "examples/" ^ fname)
	handle Io(s) => 
	(open_in ("examples/" ^ fname ^ ".fm"), "examples/" ^ fname ^ ".fm")
	handle Io(s) => raise Io(fname ^ " not found")

end
functor ParseRes 
	    (structure Typ: TYP
	     structure Trm: TRM
	     structure Globals: GLOBALS
	     sharing Typ.Globals = Globals
	         and Trm.Typ = Typ
	    ) : PARSERES 
	    = struct

structure Typ = Typ
structure Trm = Trm
structure Globals = Globals

datatype T =
    Leq of Typ.pretyp * Typ.pretyp
  | Type_Assumption of Globals.Id.T * Typ.pretyp
  | Type_Abbrev of Globals.Id.T * Typ.pretyp
  | Term_Def of Globals.Id.T * Trm.pretrm
  | Term_Assumption of Globals.Id.T * Typ.pretyp
  | Use of string
  | Set of string * string
  | Nothing

end 
(* ----------------------------------------------------------------------- *)
(*									   *)
(* Low-level prettyprinting stream package.  Based on notes by Greg Nelson *)
(*									   *)
(* ----------------------------------------------------------------------- *)

functor Pp (structure Wr: WR) : PP =
struct

structure Wr = Wr

(* ----------------------------------------------------------------------- *)
(*				 Utilities				   *)
(* ----------------------------------------------------------------------- *)

val DEBUG = ref false;

fun debug ss = if (!DEBUG) 
		   then print ((implode ss) ^ "\n")
		   else ()

fun mapunit f = 
    let fun m ([]) = ()
          | m (hd::tl) = ((f hd); m tl)
    in m
    end

(* ----------------------------------------------------------------------- *)
(*			      Data Structures				   *)
(* ----------------------------------------------------------------------- *)

datatype BreakBehavior =
	NLINDENT of int
      | EXPLICIT of string

datatype Token = 
	CHAR of int
      | SETB
      | ENDB
      | BREAK of {united:bool, do_what:BreakBehavior}
      | LONG of string

datatype RefList =
	NIL 
      | CONS of Token * (RefList ref)

exception CalledErrorCont
val error_cont : unit cont = 
    callcc (fn k => (callcc (fn ek => throw k ek); raise CalledErrorCont))

exception CoroutineBug
exception PPQueueOverflowed

type Pp = {wr:Wr.Wr, 
          q: Token array,
	  qr: int ref, 
	  inp: int ref, 
	  m1: int ref,
	  m2: int ref,
	  m3: int ref,
	  outq: Token array,
	  outp: int ref,
	  indent: int ref,
	  margin: int ref,
	  empty: unit cont ref,
	  nonempty: unit cont ref}

val qlen = 500;
val outqlen = 500;
val default_margin = 76;

fun init_pp (wr:Wr.Wr) : Pp = 
    {wr = wr,
     q = array(qlen, CHAR(33)),
     qr = ref 0,
     inp = ref 0,
     m1 = ref 0,
     m2 = ref 0,
     m3 = ref 0,
     outq = array(outqlen, CHAR(33)),
     outp = ref 0,
     indent = ref 0,
     margin = ref default_margin,
     empty = ref(error_cont), 
     nonempty = ref(error_cont)}

fun enqueue (pp:Pp) tok =
    let val {q=q, qr=qr, inp=inp, m1=m1, empty=empty, nonempty=nonempty, ...} 
    	    = pp
        val curqr = !qr
	val _ = debug ["enqueue: "," qr=",makestring (!qr),
				   " inp=",makestring (!inp),
				   " m1=",makestring (!m1)]
    in
       debug ["enqueue"];
       if ((curqr+1) mod qlen = (!inp)) orelse ((curqr+1) mod qlen = (!m1)) 
          then raise PPQueueOverflowed
          else ();
       update(q, curqr, tok);
       qr := (curqr + 1) mod qlen;
       debug ["enqueue: about to switch"];
       callcc (fn k => (empty := k; throw (!nonempty) ()));
       debug ["enqueue: returning"]
    end

fun requeue (pp:Pp) =
    let val {q=q, qr=qr, inp=inp, empty=empty, nonempty=nonempty, m1=m1, ...} 
    	    = pp
	val _ = debug ["requeue: "," qr=",makestring (!qr),
				   " inp=",makestring (!inp),
				   " m1=",makestring (!m1)]
    in 
       inp := ((!inp) - 1) mod qlen
    end

fun dequeue (pp:Pp) =
    let val {q=q, qr=qr, inp=inp, empty=empty, nonempty=nonempty, m1=m1, ...} 
    	    = pp
	val _ = debug ["dequeue: "," qr=",makestring (!qr),
				   " inp=",makestring (!inp),
				   " m1=",makestring (!m1)]
        val _ = (* Front make sure there's something to dequeue *)
      		callcc (fn k => 
			(debug ["dequeue: checking for input"];
			 if (!inp) = (!qr) 
			     then (debug ["dequeue: blocking"];
				   nonempty := k; 
				   throw (!empty) ())
			 else ()))
	val _ = debug ["dequeue: unblocked"]
	val _ = if (!inp)<0 orelse (!inp)>qlen 
		   then print ("About to crash: "^(makestring (!inp))^"\n")
		   else ()
	val c = q sub (!inp)
	val _ = inp := ((!inp) + 1) mod qlen
    in 
       debug ["dequeue: returning"];
       c
    end

(* ----------------------------------------------------------------------- *)
(*				Processing				   *)
(* ----------------------------------------------------------------------- *)

exception LineTooLong
exception HowdThatGetInHere

fun raw_printline (pp as {wr=wr, outq=outq, outp=outp, ...}:Pp) =
      let val i = ref 0
      in 
         while ((!i)<(!outp)) do
           (case outq sub (!i) of
                CHAR(c) => Wr.write_wr wr (chr c)
              | LONG(s) => Wr.write_wr wr s
              | _ => raise HowdThatGetInHere;
            i := (!i)+1)
      end 

fun write_tok (pp as {outq=outq, outp=outp, 
		      indent=indent, margin=margin, ...}:Pp) 
	      c raiseok =
    let in
	update (outq,!outp,c);
	if (!outp) < outqlen 
	   then outp := (!outp)+1
	   else ();
	case c of
	    CHAR(10) => (raw_printline pp;
			 indent := 0;
			 outp := 0)
	  | _        => (indent := (!indent)+1;
				      if (!indent) > (!margin)
				         andalso raiseok
					then raise LineTooLong 
				        else ())
    end
    
fun do_break pp indent (NLINDENT(n)) =
    let val i = ref 0
    in
      write_tok pp (CHAR(10)) true;
      while ((!i)<n+indent) do
        (write_tok pp (CHAR(32)) true;
         i := (!i)+1)
    end
  | do_break pp indent (EXPLICIT(s)) =
    mapunit (fn s => enqueue pp (CHAR(ord s))) (explode s)

fun P1 pp = 
    let fun loop() = 
    	     (debug ["P1_loop"];
              case dequeue pp of
                  (c as CHAR(_)) => (write_tok pp c true; loop())
                | (c as LONG(_)) => (write_tok pp c true; loop())
                | SETB => (P1 pp; loop())
                | BREAK(_) => loop()
       		| ENDB => ())
    in debug ["P1"];
       loop();
       debug ["P1: finished"]
    end

and P2 pp = 
    let fun loop() = 
    	     (debug ["P2_loop"];
              case dequeue pp of
                  (c as CHAR(_)) => (write_tok pp c true; loop())
                | (c as LONG(_)) => (write_tok pp c true; loop())
                | SETB => (P1 pp; loop())
                | BREAK(_) => ()
       		| ENDB => ())
    in debug ["P2"];
       loop();
       (* I think the input queue needs to be backed up by one now, so that
          P3 sees this SETB or BREAK... *)
       requeue pp;
       debug ["P2: finished"]
    end

and P3 (pp as {inp=inp, outp=outp, indent=indent, 
	       m1=m1, m2=m2, m3=m3, ...} :Pp) = 
    let val saved_indent = !indent
        fun loop() = 
    	     (debug ["P3_loop"];
              case dequeue pp of
                  (c as CHAR(_)) => (write_tok pp c false; loop())
                | (c as LONG(_)) => (write_tok pp c false; loop())
                | SETB => (PP pp; loop())
                | BREAK({united=true,do_what=do_what}) => 
                      (do_break pp saved_indent do_what;
                       m1 := ~1; (* Not in CGN's original note *)
                       loop())
                | BREAK({united=false,do_what=do_what}) => 
                      (m1 := (!inp); m2 := (!outp); m3 := (!indent);
                       ((P2 pp; 
                         m1 := ~1; (* This once seemed wrong *)
                         debug ["P3: looping back"]; 
                         loop())
                        handle LineTooLong =>
                           (debug ["P3: line too long"];
                            inp := (!m1); outp := (!m2); indent := (!m3);
                            do_break pp saved_indent do_what;
                            m1 := ~1;
			    loop())))
       		| ENDB => ())
    in debug ["P3"];
       loop()
    end

and PP (pp as {inp=inp, outp=outp, indent=indent, 
	       m1=m1, m2=m2, m3=m3, ...} :Pp) = 
    let 
    in debug ["PP"];
       m1 := (!inp); m2 := (!outp); m3 := (!indent);
       (P1 pp; m1 := ~1; debug ["PP finished"])
       handle LineTooLong 
         => (debug ["PP: line too long"];
             inp := (!m1); outp := (!m2); indent := (!m3);
             m1 := ~1;
             P3 pp)
    end

exception EndbWithNoMatchingSetb
fun top_level pp = 
    let 
    in debug ["top_level"];
       P3 pp;
       raise EndbWithNoMatchingSetb
    end

(* ----------------------------------------------------------------------- *)
(*				Interaction				   *)
(* ----------------------------------------------------------------------- *)

fun setb pp = enqueue pp SETB

fun endb pp = enqueue pp ENDB

fun break pp b i = enqueue pp (BREAK {united=b, do_what=(NLINDENT i)})

fun expbreak pp b s = enqueue pp (BREAK {united=b, do_what=(EXPLICIT s)})

fun pwrite (pp as {wr=wr, ...} : Pp) s = 
    (debug ["write: '", s, "'"];
     mapunit (fn s => case ord(s) of
     			10 => break pp true 0
     		      | i  => enqueue pp (CHAR(i)))
     	     (explode s))

exception IllegalMargin

fun set_margin (pp as {margin=margin, outp=outp , ...} : Pp) n =
    if (!outp) > n orelse n >= outqlen
       then raise IllegalMargin
       else margin := n

(* ----------------------------------------------------------------------- *)
(*				  Creation				   *)
(* ----------------------------------------------------------------------- *)

fun pp_from_wr wr =
    let val _ = debug ["new"];
        val (pp as {empty=empty, ...}:Pp) = init_pp wr
    in 
       callcc (fn k => (empty := k; top_level pp));
       pp
    end

fun wr_from_pp (pp as {wr=wr, ...} : Pp)
    = Wr.to_fn (fn s => pwrite pp s) (fn () => Wr.close wr)

end (* Functor Pp *)

functor Wr () : WR =
struct

datatype wr = 
	WR of wr_spc * unit

and wr_spc = 
        TO_STDOUT
      | TO_FILE of string * outstream
      | TO_WRS of wr list
      | TO_STRING of string list ref
      | TO_FN of (string->unit) * (unit->unit)
      | TO_NOWHERE 
	
type Wr = wr

fun new spc = WR(spc, ())

fun to_stdout () = new (TO_STDOUT)

fun to_file name = 
    let val out = open_out name
    in new (TO_FILE(name,out))
    end

fun to_wrs wrs = new (TO_WRS(wrs))

fun to_fn f cl_f = new (TO_FN(f,cl_f))

fun to_string () = new (TO_STRING(ref([]:string list)))

fun to_nowhere () = new (TO_NOWHERE)

exception Not_a_TOSTRING_Wr

fun extract_str (WR(TO_STRING(ss),_)) = implode (rev (!ss))
  | extract_str _ = raise Not_a_TOSTRING_Wr

fun mapunit f = 
    let fun m ([]) = ()
          | m (hd::tl) = ((f hd); m tl)
    in m
    end

fun close (WR(spc,gen)) = 
    case spc of
       TO_STDOUT => ()
     | TO_FILE(name,out) => close_out out
     | TO_WRS(wrs) => mapunit close wrs
     | TO_FN(_,cl_f) => cl_f()
     | TO_STRING(ss) => ()
     | TO_NOWHERE => ()

fun write_wr (WR(spc,gen)) s =
    case spc of
       TO_STDOUT => output(std_out,s)
     | TO_FILE(_,out) => output(out,s)
     | TO_WRS(wrs) => mapunit (fn wr => write_wr wr s) wrs
     | TO_FN(f,_) => f s
     | TO_STRING(ss) => ss := (s :: (!ss))
     | TO_NOWHERE => ()

end 


