/* misc.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: misc.c,v 1.15 1995/06/09 21:48:38 brianp Exp $

$Log: misc.c,v $
 * Revision 1.15  1995/06/09  21:48:38  brianp
 * changed version string to 1.2.1
 *
 * Revision 1.14  1995/05/24  13:00:15  brianp
 * updated version query functions to return 1.2
 *
 * Revision 1.13  1995/05/22  21:02:41  brianp
 * Release 1.2
 *
 * Revision 1.12  1995/05/17  13:52:37  brianp
 * implemented glIndexMask(0) and glColorMask(0,0,0,0)
 *
 * Revision 1.11  1995/05/12  16:57:22  brianp
 * replaced CC.Mode!=0 with INSIDE_BEGIN_END
 *
 * Revision 1.10  1995/05/12  16:28:41  brianp
 * added const to glGetString prototype
 *
 * Revision 1.9  1995/04/17  14:48:19  brianp
 * updated glGetString for 1.1.4 beta
 *
 * Revision 1.8  1995/03/30  21:08:25  brianp
 * glClear limited to scissor box, not viewport!
 *
 * Revision 1.7  1995/03/13  20:55:59  brianp
 * new read buffer logic
 *
 * Revision 1.6  1995/03/10  16:25:19  brianp
 * updated glGetString for blending extensions
 *
 * Revision 1.5  1995/03/08  15:15:04  brianp
 * removed garbage characters from tail of file
 *
 * Revision 1.4  1995/03/08  15:10:02  brianp
 * added support for dd_clear_index and dd_clear_color
 *
 * Revision 1.3  1995/03/07  14:21:10  brianp
 * updated for new XSetForeground/GC scheme
 *
 * Revision 1.2  1995/03/04  19:29:44  brianp
 * 1.1 beta revision
 *
 * Revision 1.1  1995/02/24  14:24:57  brianp
 * Initial revision
 *
 */


/*
 * Miscellaneous functions
 */


#include <stdlib.h>
#include "accum.h"
#include "context.h"
#include "depth.h"
#include "dd.h"
#include "list.h"
#include "macros.h"
#include "stencil.h"



void glClearIndex( GLfloat c )
{
   if (CC.CompileFlag) {
      gl_save_clearindex( c );
   }
   if (CC.ExecuteFlag) {
      if (INSIDE_BEGIN_END) {
	 gl_error( GL_INVALID_OPERATION, "glClearIndex" );
	 return;
      }
      CC.Color.ClearIndex = (GLuint) c;
      dd_clear_index( CC.Color.ClearIndex );
   }
}



void glClearColor( GLclampf red,
		   GLclampf green,
		   GLclampf blue,
		   GLclampf alpha )
{
   if (CC.CompileFlag) {
      gl_save_clearcolor( red, green, blue, alpha );
   }
   if (CC.ExecuteFlag) {
      GLfloat r, g, b, a;

      if (INSIDE_BEGIN_END) {
	 gl_error( GL_INVALID_OPERATION, "glClearColor" );
	 return;
      }

      r = CLAMP( red,   0.0F, 1.0F );
      g = CLAMP( green, 0.0F, 1.0F );
      b = CLAMP( blue,  0.0F, 1.0F );
      a = CLAMP( alpha, 0.0F, 1.0F );

      CC.Color.ClearColor[0] = r;
      CC.Color.ClearColor[1] = g;
      CC.Color.ClearColor[2] = b;
      CC.Color.ClearColor[3] = a;
      dd_clear_color( CC.Color.ClearColor );
   }
}



void glClear( GLbitfield mask )
{
   if (CC.CompileFlag) {
      gl_save_clear( mask );
   }
   if (CC.ExecuteFlag) {
      if (INSIDE_BEGIN_END) {
	 gl_error( GL_INVALID_OPERATION, "glClear" );
	 return;
      }

      if (mask & GL_DEPTH_BUFFER_BIT) {
	 gl_clear_depth_buffer();
      }

      if (mask & GL_ACCUM_BUFFER_BIT) {
	 gl_clear_accum_buffer();
      }

      if (mask & GL_COLOR_BUFFER_BIT) {
	 dd_clear( !CC.Scissor.Enabled,
		   CC.Scissor.X, CC.Scissor.Y,
		   CC.Scissor.Width, CC.Scissor.Height );
      }

      if (mask & GL_STENCIL_BUFFER_BIT) {
	 gl_clear_stencil_buffer();
      }
   }
}



void glIndexMask( GLuint mask )
{
   if (CC.CompileFlag) {
      gl_save_indexmask( mask );
   }
   if (CC.ExecuteFlag) {
      if (INSIDE_BEGIN_END) {
	 gl_error( GL_INVALID_OPERATION, "glIndexMask" );
	 return;
      }
      CC.Color.IndexMask = mask;
      dd_index_mask( mask );
   }
}



void glColorMask( GLboolean red, GLboolean green,
		  GLboolean blue, GLboolean alpha )
{
   if (CC.CompileFlag) {
      gl_save_colormask( red, green, blue, alpha );
   }
   if (CC.ExecuteFlag) {
      if (INSIDE_BEGIN_END) {
	 gl_error( GL_INVALID_OPERATION, "glColorMask" );
	 return;
      }
      CC.Color.ColorMask = (red << 3) | (green << 2) | (blue << 1) | alpha;
      dd_color_mask( red, green, blue, alpha );
   }
}



GLenum glGetError( void )
{
   GLenum e;

   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glGetError" );
      return GL_INVALID_OPERATION;
   }

   e = CC.ErrorValue;
   CC.ErrorValue = GL_NO_ERROR;
   return e;
}



const GLubyte *glGetString( GLenum name )
{
   static char *vendor = "Brian Paul";
   static char *renderer = "Mesa";
   static char *version = "1.2.1";
   static char *extensions = "GL_EXT_blend_color GL_EXT_blend_minmax GL_EXT_blend_logic_op GL_EXT_blend_subtract";

   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glGetError" );
      return (GLubyte *) 0;
   }

   switch (name) {
      case GL_VENDOR:
         return (GLubyte *) vendor;
      case GL_RENDERER:
         return (GLubyte *) renderer;
      case GL_VERSION:
         return (GLubyte *) version;
      case GL_EXTENSIONS:
         return (GLubyte *) extensions;
      default:
         gl_error( GL_INVALID_ENUM, "glGetString" );
         return (GLubyte *) 0;
   }
}



void glFinish( void )
{
   /* Don't compile into display list */
   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glFinish" );
      return;
   }
   dd_flush();
}



void glFlush( void )
{
   /* Don't compile into display list */
   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glFlush" );
      return;
   }
   dd_flush();
}



void glHint( GLenum target, GLenum mode )
{
   if (CC.CompileFlag) {
      gl_save_hint( target, mode );
   }
   if (CC.ExecuteFlag) {
      if (INSIDE_BEGIN_END) {
	 gl_error( GL_INVALID_OPERATION, "glHint" );
	 return;
      }
      if (mode!=GL_DONT_CARE && mode!=GL_FASTEST && mode!=GL_NICEST) {
	 gl_error( GL_INVALID_ENUM, "glHint(mode)" );
	 return;
      }
      switch (target) {
	 case GL_FOG_HINT:
	    CC.Hint.Fog = mode;
	    break;
	 case GL_LINE_SMOOTH_HINT:
	    CC.Hint.LineSmooth = mode;
	    break;
	 case GL_PERSPECTIVE_CORRECTION_HINT:
	    CC.Hint.PerspectiveCorrection = mode;
	    break;
	 case GL_POINT_SMOOTH_HINT:
	    CC.Hint.PointSmooth = mode;
	    break;
	 case GL_POLYGON_SMOOTH_HINT:
	    CC.Hint.PolygonSmooth = mode;
	    break;
	 default:
	    gl_error( GL_INVALID_ENUM, "glHint(target)" );
      }
   }
}



void glDrawBuffer( GLenum mode )
{
   if (CC.CompileFlag) {
      gl_save_drawbuffer( mode );
   }
   if (CC.ExecuteFlag) {
      GLenum m;
      if (INSIDE_BEGIN_END) {
	 gl_error( GL_INVALID_OPERATION, "glDrawBuffer" );
	 return;
      }
      switch (mode) {
	 case GL_NONE:
	 case GL_FRONT_LEFT:
	 case GL_FRONT_RIGHT:
	 case GL_BACK_LEFT:
	 case GL_BACK_RIGHT:
	 case GL_LEFT:
	 case GL_RIGHT:
	 case GL_FRONT:
	 case GL_BACK:
	 case GL_FRONT_AND_BACK:
	 case GL_AUX0:
	    m = dd_draw_buffer( mode );
	    if (m!=mode) {
	       gl_error( GL_INVALID_ENUM, "glDrawBuffer" );
	    }
	    else {
	       CC.Color.DrawBuffer = m;
	       dd_read_buffer( mode );  /* See NOTE below */
	    }
	    break;
	 default:
	    gl_error( GL_INVALID_ENUM, "glDrawBuffer" );
      }
   }
}



void glReadBuffer( GLenum mode )
{
   if (CC.CompileFlag) {
      gl_save_readbuffer( mode );
   }
   if (CC.ExecuteFlag) {
      GLenum m;
      if (INSIDE_BEGIN_END) {
	 gl_error( GL_INVALID_OPERATION, "glReadBuffer" );
	 return;
      }
      switch (mode) {
	 case GL_FRONT_LEFT:
	 case GL_FRONT_RIGHT:
	 case GL_BACK_LEFT:
	 case GL_BACK_RIGHT:
	 case GL_FRONT:
	 case GL_BACK:
	 case GL_LEFT:
	 case GL_RIGHT:
	 case GL_AUX0:
	    m = dd_read_buffer( mode );
	    if (m!=mode) {
	       gl_error( GL_INVALID_ENUM, "glReadBuffer" );
	    }
	    else {
	       CC.Pixel.ReadBuffer = m;
	    }
	    /* NOTE: we normally keep the "read" buffer equal to the "draw"
	     * buffer so that pixel logic and blending operations read the
	     * correct pixel values.  When we execute ReadPixels, CopyPixels,
	     * or Accum functions we then point the read buffer to what
	     * was specified with glReadBuffer().
	     */
	    dd_read_buffer( CC.Color.DrawBuffer );
	    break;
	 default:
	    gl_error( GL_INVALID_ENUM, "glReadBuffer" );
      }
   }
}



void glRectd( GLdouble x1, GLdouble y1, GLdouble x2, GLdouble y2 )
{
   /*
    * TODO:  optimize all glRect calls to issue gl_save_vertex and/or
    * gl_vertex calls, etc. depending on CC.ExecuteFlag and
    * CC.CompileFlag.
    */

   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glRectd" );
      return;
   }
   glBegin( GL_POLYGON );
   glVertex2d( x1, y1 );
   glVertex2d( x2, y1 );
   glVertex2d( x2, y2 );
   glVertex2d( x1, y2 );
   glEnd();
}


void glRectf( GLfloat x1, GLfloat y1, GLfloat x2, GLfloat y2 )
{
   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glRectf" );
      return;
   }
   glBegin( GL_POLYGON );
   glVertex2f( x1, y1 );
   glVertex2f( x2, y1 );
   glVertex2f( x2, y2 );
   glVertex2f( x1, y2 );
   glEnd();
}


void glRecti( GLint x1, GLint y1, GLint x2, GLint y2 )
{
   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glRecti" );
      return;
   }
   glBegin( GL_POLYGON );
   glVertex2i( x1, y1 );
   glVertex2i( x2, y1 );
   glVertex2i( x2, y2 );
   glVertex2i( x1, y2 );
   glEnd();
}


void glRects( GLshort x1, GLshort y1, GLshort x2, GLshort y2 )
{
   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glRects" );
      return;
   }
   glBegin( GL_POLYGON );
   glVertex2s( x1, y1 );
   glVertex2s( x2, y1 );
   glVertex2s( x2, y2 );
   glVertex2s( x1, y2 );
   glEnd();
}


void glRectdv( const GLdouble *v1, const GLdouble *v2 )
{
   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glRectdv" );
      return;
   }
   glBegin( GL_POLYGON );
   glVertex2d( v1[0], v1[1] );
   glVertex2d( v2[0], v1[1] );
   glVertex2d( v2[0], v2[1] );
   glVertex2d( v1[0], v2[1] );
   glEnd();
}


void glRectfv( const GLfloat *v1, const GLfloat *v2 )
{
   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glRectfv" );
      return;
   }
   glBegin( GL_POLYGON );
   glVertex2f( v1[0], v1[1] );
   glVertex2f( v2[0], v1[1] );
   glVertex2f( v2[0], v2[1] );
   glVertex2f( v1[0], v2[1] );
   glEnd();
}


void glRectiv( const GLint *v1, const GLint *v2 )
{
   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glRectiv" );
      return;
   }
   glBegin( GL_POLYGON );
   glVertex2i( v1[0], v1[1] );
   glVertex2i( v2[0], v1[1] );
   glVertex2i( v2[0], v2[1] );
   glVertex2i( v1[0], v2[1] );
   glEnd();
}


void glRectsv( const GLshort *v1, const GLshort *v2 )
{
   if (INSIDE_BEGIN_END) {
      gl_error( GL_INVALID_OPERATION, "glRectsv" );
      return;
   }
   glBegin( GL_POLYGON );
   glVertex2s( v1[0], v1[1] );
   glVertex2s( v2[0], v1[1] );
   glVertex2s( v2[0], v2[1] );
   glVertex2s( v1[0], v2[1] );
   glEnd();
}


