/*
 *  nwsfind.c
 *
 *  Find a NetWare server and open a route to it.
 *  This tool can safely be setuid root, if normal users should be able to
 *  access any NetWare server.
 *
 *  Copyright (C) 1996 by Volker Lendecke
 *
 */

#include "ncplib.h"
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>

static char *progname;

static void
usage(void)
{
	fprintf(stderr, "usage: %s [server]\n", progname);
}

static void
help(void)
{
	printf("\n");
	printf("usage: %s [server]\n", progname);
	printf("\n"
	       "-t             Server type, default: File server\n"
	       "-a             server is in form <net>:<node>:<socket>\n"
	       "-h             Print this help text\n"
	       "\n");
}

static void
swallow_error(const char *name, long code, const char *format, va_list arg)
{
	printf("%s ", error_message(code));
	vfprintf(stdout, format, arg);
	putchar('\n');
	return;
}

int
main(int argc, char *argv[])
{
	const char *server = NULL;
	int object_type = NCP_BINDERY_FSERVER;
	struct sockaddr_ipx addr;
	long err;
#ifdef CONFIG_NATIVE_IPX
	int is_address = 0;
#endif

	int opt;

	progname = argv[0];

	set_com_err_hook(swallow_error);

	while ((opt = getopt(argc, argv, 
#ifdef CONFIG_NATIVE_IPX
					"a"
#endif
					"t:")) != EOF)
	{
		switch (opt)
		{
		case 't':
			object_type = atoi(optarg);
			break;
#ifdef CONFIG_NATIVE_IPX
		case 'a':
		    is_address = 1;
		    break;
#endif
		case 'h':
		case '?':
			help();
			exit(1);
		default:
			usage();
			exit(1);
		}
	}

	if (optind < argc - 1)
	{
		usage();
		exit(1);
	}
#ifdef CONFIG_NATIVE_IPX
	if (is_address)
	{
		struct ncp_conn *conn;
		if ((optind > argc - 1) || 
		    ipx_sscanf_saddr(argv[optind], &addr))
		{
			usage();
			exit(1);
		}
		if ((!(conn = ncp_open_addr((struct sockaddr*)&addr, &err))) ||
		    (err = ncp_close(conn))) {;
		}
		else 
		{
			server = argv[optind];
		}
        }
        else
#endif	/* CONFIG_NATIVE_IPX */
        {
		if (optind == argc - 1)
		{
			server = argv[optind];
			if (strlen(server) >= NCP_BINDERY_NAME_LEN)
			{
				com_err(argv[0], ENAMETOOLONG, "server name too long");
				exit(1);
			}
		}

		err = ncp_find_server(&server, object_type, (struct sockaddr*)&addr, sizeof(addr));

	}

	if (err != 0)
	{
		com_err(argv[0], err, "when trying to find server");
		exit(1);
	}
#ifdef CONFIG_NATIVE_IPX
	ipx_print_saddr(&addr);
#else
	printf("X");
#endif
	printf(" %s\n", server);
	return 0;
}
