/* Library extensions for nwdir.c */

#include <nwcalls.h>
#include "ncplib_err.h"
#include "ncpcode.h"

NWCCODE NWCallsInit(void* dummy1, void* dummy2) {
	(void)dummy1;
	(void)dummy2;
	return 0;
}

NWCCODE NWRequest(NWCONN_HANDLE conn, nuint function, 
	nuint numRq, const NW_CFRAGMENT* rq, 
	nuint numRp, NW_FRAGMENT* rp) {
	long result;
	nuint i;
	size_t rest;
	unsigned char* ptr;

	ncp_init_request(conn);
	for (i=numRq; i--; rq++) {
		ncp_add_mem(conn, rq->fragAddress, rq->fragSize);
	}
	result = ncp_request(conn, function & 0xFF);
	if (result) {
		if (result == NCPL_ET_REQUEST_ERROR) {
			result = NWE_SERVER_ERROR | conn->completion;
		} else {
			result = NWE_REQUESTER_FAILURE;
		}
		ncp_unlock_conn(conn);
		return result;
	}
	rest = conn->ncp_reply_size;
	ptr = ncp_reply_data(conn, 0);
	for (i=numRp; i--; rp++) {
		size_t spc = rp->fragSize;
		if (spc > rest) {
			memcpy(rp->fragAddress, ptr, rest);
			/* Do we want to modify fragList ? */
			/* I think so because of we want to propagate */
			/* reply length back to client */
			/* maybe we could do "return conn->ncp_reply_size" */
			/* instead of "return 0". */
			rp->fragSize = rest;
			rest = 0;
		} else {
			memcpy(rp->fragAddress, ptr, spc);
			rest -= spc;
		}
	}
	ncp_unlock_conn(conn);
	return 0;
}
	
/* NWRequestSimple: Simple request, 1 request fragment, max. 1 reply fragment
 * !!! check carefully whether you call this function with rp->fragAddress ?= NULL !!!
 * conn = connection handle
 * function = NCP function (0x00 - 0xFF), subfunction 0x00xx - 0xFFxx, sfn flag 0x1xxxx
 * rq = pointer to request, if rqlen == 0 then address can be NULL
 * rqlen = request length, if rqlen == 0, rq can be NULL
 * rp = pointer to reply fragment
 *      on error: return nonzero code, unlock conn
 *      on success: return 0;
 *                  if rp == NULL: throw away reply
 *                  if rp != NULL && rp->fragAddress == NULL: fill rp with pointer
 *                                 to data and with size, do NOT unlock conn
 *                  if rp != NULL && rp->fragAddress != NULL: fill buffer pointed by
 *                                 rp->fragAddress with reply up to rp->fragSize
 *                                 rp->fragSize is reply length (can be greater than
 *                                 fragSize on input), unlock conn
 */
NWCCODE NWRequestSimple(NWCONN_HANDLE conn, nuint function,
	const void* rq, size_t rqlen, NW_FRAGMENT* rp) {
	long result;
	
	if ((rp && rp->fragSize && !rp->fragAddress) 
         || (rqlen && !rq))
		return NWE_PARAM_INVALID;
	ncp_init_request(conn);
	if (function & NCPC_SUBFUNCTION) {
		ncp_add_word_hl(conn, rqlen+1);
		ncp_add_byte(conn, NCPC_SUBFN(function));
	}
	if (rqlen) 
		ncp_add_mem(conn, rq, rqlen);
	result = ncp_request(conn, NCPC_FN(function));
	if (result) {
		NWCCODE nwresult;

		if (result == NCPL_ET_REQUEST_ERROR) {
			nwresult = NWE_SERVER_ERROR | conn->completion;
		} else {
			nwresult = NWE_REQUESTER_FAILURE;
		}
		ncp_unlock_conn(conn);
		return nwresult;
	}
	if (rp) {
		size_t rest;
		void* ptr;

		rest = conn->ncp_reply_size;
		ptr = ncp_reply_data(conn, 0);
		if (rp->fragAddress) {
			size_t maxsize = rp->fragSize;
			if (maxsize > rest)
				maxsize = rest;
			rp->fragSize = rest;
			memcpy(rp->fragAddress, ptr, maxsize);
		} else {
			rp->fragAddress = ptr;
			rp->fragSize = rest;
			return 0;
		}
	}
	ncp_unlock_conn(conn);
	return 0;
}

NWCCODE NWGetObjectName(NWCONN_HANDLE conn, nuint32 ID, char* name, nuint16* type) {
	long result;
	struct ncp_bindery_object spc;

	result = ncp_get_bindery_object_name(conn, ID, &spc);
	if (result == NCPL_ET_REQUEST_ERROR) {
		return NWE_SERVER_FAILURE;
	}
	if (result) return NWE_REQUESTER_FAILURE;
	if (name) strncpy(name, spc.object_name, NCP_BINDERY_NAME_LEN);
	if (type) *type = spc.object_type;
	return 0;
}

NWCCODE NWGetNSEntryInfo(NWCONN_HANDLE conn, nuint dirHandle, const char* path, 
	nuint srcNS, nuint dstNS, nuint16 attr, nuint32 rim, NW_ENTRY_INFO* info) {
	unsigned char buff[1024];
	int rslt;

	rslt = ncp_path_to_NW_format(path, buff, sizeof(buff));
	if (rslt < 0) return rslt;
	rslt = ncp_obtain_file_or_subdir_info2(conn, srcNS, dstNS, attr, rim, 
		dirHandle?0:0xFF, 0, dirHandle, buff, rslt, info);
	if (rslt == NCPL_ET_REQUEST_ERROR) return NWE_SERVER_FAILURE;
	if (rslt) return NWE_REQUESTER_FAILURE;
	return 0;
}

NWCCODE NWParsePath(const char* path, char* server, NWCONN_HANDLE* conn, 
	char* volume, char* volpath) {
	long result;
	NWCONN_HANDLE cn;
	char tmp[1000];
	char* ptr;

	result = ncp_open_mount(path, &cn);
	if (result) {
		if (volume)
			*volume = 0;
		if (volpath)
			strcpy(volpath, path);
		if (conn)
			*conn = NULL;
		return 0;
	}
	result = ncp_ns_get_full_name(cn, NW_NS_DOS, NW_NS_DOS, 1,
		cn->i.volume_number, cn->i.directory_id, NULL, 0, tmp, sizeof(tmp));
	if (result) {
		return NWE_SERVER_FAILURE;
	}
	ptr = strchr(tmp, ':');
	if (ptr) {
		if (volume) {
			memcpy(volume, tmp, ptr-tmp);
			volume[ptr-tmp]=0;
		}
		if (volpath)
			strcpy(volpath, ptr+1);
	} else {
		if (volume)
			*volume = 0;
		if (volpath)
			strcpy(volpath, tmp);
	}
	if (conn)
		*conn = cn;
	return 0;
}

NWCCODE NWDownFileServer(NWCONN_HANDLE conn, nuint force) {
	nuint8 flag = force?0:0xFF;

	return NWRequestSimple(conn, NCPC_SFN(0x17, 0xD3), &flag, sizeof(flag), NULL);
}

NWCCODE NWCloseBindery(NWCONN_HANDLE conn) {
	return NWRequestSimple(conn, NCPC_SFN(0x17, 0x44), NULL, 0, NULL);
}

NWCCODE NWOpenBindery(NWCONN_HANDLE conn) {
	return NWRequestSimple(conn, NCPC_SFN(0x17, 0x45), NULL, 0, NULL);
}

NWCCODE NWDisableTTS(NWCONN_HANDLE conn) {
	return NWRequestSimple(conn, NCPC_SFN(0x17, 0xCF), NULL, 0, NULL);
}

NWCCODE NWEnableTTS(NWCONN_HANDLE conn) {
	return NWRequestSimple(conn, NCPC_SFN(0x17, 0xD0), NULL, 0, NULL);
}

NWCCODE NWDisableFileServerLogin(NWCONN_HANDLE conn) {
	return NWRequestSimple(conn, NCPC_SFN(0x17, 0xCB), NULL, 0, NULL);
}

NWCCODE NWEnableFileServerLogin(NWCONN_HANDLE conn) {
	return NWRequestSimple(conn, NCPC_SFN(0x17, 0xCC), NULL, 0, NULL);
}

static NWCCODE NWSMExec(NWCONN_HANDLE conn, int scode, 
	const nuint8* rqd, const char* txt, const char* txt2, 
	nuint32* rpd) {
	NW_CFRAGMENT rq[4];
	NW_FRAGMENT rp;
	static const char here20[20] = { 0 };
	nuint8 rpl[24];
	nuint8 hdr[3];
	NWCCODE err;

	if (!rqd) rq[1].fragAddress = here20; else rq[1].fragAddress = rqd;
	rq[1].fragSize = 20;
	rq[0].fragAddress = hdr;
	rq[0].fragSize = sizeof(hdr);
	rq[2].fragAddress = txt;
	rq[2].fragSize = strlen(txt) + 1;
	rq[3].fragAddress = txt2;
	rq[3].fragSize = txt2?(strlen(txt2) + 1):0;
	WSET_HL(hdr, 0, rq[3].fragSize + rq[2].fragSize + 1 + 20);
	BSET(hdr, 2, scode);
	rp.fragAddress = rpl;
	rp.fragSize = sizeof(rpl);
	err = NWRequest(conn, 0x83, 4, rq, 1, &rp);
	if (err)
		return err;
	if (rp.fragSize < 4)
		return 0x897E;	/* buffer mismatch */
	err = DVAL_LH(rpl, 0);
	if (err)
		return err;
	if (rpd) {
		if (rp.fragSize < 24)
			return 0x897E;
		*rpd = DVAL_LH(rpl, 20);
	}
	return err;
}

NWCCODE NWSMLoadNLM(NWCONN_HANDLE conn, const char* cmd) {
	return NWSMExec(conn, 0x01, NULL, cmd, NULL, NULL);
}

NWCCODE NWSMUnloadNLM(NWCONN_HANDLE conn, const char* cmd) {
	return NWSMExec(conn, 0x02, NULL, cmd, NULL, NULL);
}

NWCCODE NWSMMountVolume(NWCONN_HANDLE conn, const char* vol, nuint32* volnum) {
	return NWSMExec(conn, 0x03, NULL, vol, NULL, volnum);
}

NWCCODE NWSMDismountVolumeByName(NWCONN_HANDLE conn, const char* vol) {
	return NWSMExec(conn, 0x04, NULL, vol, NULL, NULL);
}

NWCCODE NWSMDismountVolumeByNumber(NWCONN_HANDLE conn, nuint32 vol) {
	return 0x89FF;	/* hhh */
}

NWCCODE NWSMSetDynamicCmdIntValue(NWCONN_HANDLE conn, const char* set,
		nuint32 value) {
	unsigned char hdr[20];

	memset(hdr, 0, sizeof(hdr));
	DSET_LH(hdr, 0, 1);
	DSET_LH(hdr, 4, value);
	return NWSMExec(conn, 0x06, hdr, set, NULL, NULL);
}

NWCCODE NWSMSetDynamicCmdStrValue(NWCONN_HANDLE conn, const char* set,
		const char* value) {
	return NWSMExec(conn, 0x06, NULL, set, value, NULL);
}

NWCCODE NWSMExecuteNCFFile(NWCONN_HANDLE conn, const char* cmd) {
	return NWSMExec(conn, 0x07, NULL, cmd, NULL, NULL);
}


