/* vdiesc1.c - vdi - escapes missing from GEMLIB			*/
/* Reference: 	Atari ST Profibuch 					*/
/*		Hans-Dieter Jankowski, Julian F. Reschke, Dietmar Rabich */
/*		SYBEX-Verlag GmbH, Dsseldorf, Germany 			*/
/*		7.th edition 1989 					*/
/*		ISBN : 3-88745-563-0 					*/
/*		Pages 413 ff. 						*/

/* Thomas Koenig (UI0T@DKAUNI2.BITNET, UI0T@IBM3090.RZ.UNI-KARLSRUHE.DE) */
/* This file is put into the public domain 				*/
/* Notice that, on the ST, some of these calls require GDOS.		*/

#include <stddef.h>
#include "common.h"
#include <vdibind.h>

#ifdef __DEF_ALL__

#define L_vq_tabst
#define L_v_hardco
#define L_v_dspcur
#define L_v_rmcur
#define L_v_form_a
#define L_v_output
#define L_v_clear_
#define L_v_bit_im
#define L_vq_scan
#define L_v_alpha_
#define L_vs_palet
#define L_v_sound
#define L_vs_mute
#define L_vt_resol
#define L_vt_axis
#define L_vt_origi
#define L_vq_dimen
#define L_vt_align
#define L_vsp_film
#define L_vqp_film
#define L_vsc_expo
#define L_v_meta_e
#define L_v_write_
#define L_vm_pages
#define L_vm_coord
#define L_vm_filen
#define L_v_escape

#endif /* __DEF_ALL__ */


#ifdef L_vq_tabst

/*		vq_tabstatus 						*/
/* 		screen escape 						*/
/* INQUIRE TABLET STATUS						*/
/* Inquiere wether a grapics tablet, a mouse, a Joystick or of something */
/* like that.								*/

int vq_tabstatus(int handle)
{
	__vdi__(VDI_CONTRL_ENCODE(5, 0, 0, 16), handle);
	return _intout[0];
}
#endif /* L_vq_tabst */


/*		v_hardcopy						*/
/*		screen escape						*/
/* HARD COPY								*/
/* Make a hard copy of the screen.					*/

#ifdef L_v_hardco

void v_hardcopy(int handle)
{
	__vdi__(VDI_CONTRL_ENCODE(5, 0, 0, 17), handle);
}
#endif /* L_v_hardco */


/*		v_dspcur						*/
/*		screen escape						*/
/* PLACE GRAPHICS CURSOR AT LOCATION					*/
/* Place the grapics cursor at the specified location.			*/

#ifdef L_v_dspcur

void v_dspcur (int handle, int x, int y)
/* Parameters:								*/
/*	x: x - coordinate of new graphics cursor position		*/
/*	y: y - coordinate of new graphics cursor position		*/

{
	_ptsin[0] = x;
	_ptsin[1] = y;
	__vdi__(VDI_CONTRL_ENCODE(5, 1, 0, 18), handle);
}
#endif /* L_v_dspcur */

/*		v_rmcur							*/
/*		screen escape						*/
/* REMOVE LAST GRAPHICS CURSOR						*/
/* Removes the last grapics cursor.					*/

#ifdef L_v_rmcur

void v_rmcur(int handle)
{
	__vdi__(VDI_CONTRL_ENCODE(5, 0, 0, 19), handle);
}
#endif /* L_v_rmcur */

/*		v_form_adv						*/
/*		printer escape						*/
/* FORM ADVANCE								*/
/* form feed to printer, do not erase printer buffer			*/

#ifdef L_v_form_a

void v_form_adv(int handle)
{
	__vdi__(VDI_CONTRL_ENCODE(5, 0, 0, 20), handle);
}
#endif /* L_v_form_a */


/*		v_output_window						*/
/*		printer escape						*/
/* OUTPUT WINDOW							*/
/* output part of the screen to printer, delimited by 2 points		*/

#ifdef L_v_output

void v_output_window(int handle, int *pxyarray)
/* Parameters:								*/
/*	pxyarray[0] : x - coordinate of a corner of the ouptut window	*/
/*	pxyarray[1] : y - coordinate of ...				*/
/*	pxyarray[2] : x - coordinate of other corner of the window	*/
/*	pxyarray[3] : y - coordinate of ...				*/
{
#ifdef __MSHORT__	/* we have 16 bit ints, just change vdi params */
	_vdiparams[2] = (void *) &pxyarray[0];
#else			/* 32 bit ints - let's copy */

	_ptsin[0] = pxyarray[0];
	_ptsin[1] = pxyarray[1];
	_ptsin[2] = pxyarray[2];
	_ptsin[3] = pxyarray[3];
	
#endif

	__vdi__(VDI_CONTRL_ENCODE(5, 2, 0, 21), handle);

#ifdef __MSHORT__
	_vdiparams[2] = (void *)&_ptsin[0]; /* restore vdi parameters */
#endif
}
#endif /* L_v_output */

/*		v_clear_disp_list					*/
/*		printer escape						*/
/* CLEAR DISPLAY LIST							*/
/* clear printer buffer, no form feed					*/

#ifdef L_v_clear_

void v_clear_disp_list(int handle)
{
	__vdi__(VDI_CONTRL_ENCODE(5, 0, 0, 22), handle);
}
#endif /* L_v_clear_ */


/*		v_bit_image						*/
/*		printer escape						*/
/* OUTPUT BIT IMAGE FILE						*/
/* output a bit image file (*.IMG) to a printer				*/

#ifdef L_v_bit_im

void v_bit_image(int handle, const char *filename, int aspect, int x_scale,
			int y_scale, int h_align, int v_align,
			int *pxyarray)
/* Parameter:								*/
/*	aspect: =0 aspect ratio is ignored				*/
/*		=1 use aspect ratio (circles -> circles)		*/
/*	x_scale: scaling of the x - axis, 	=0 : fractional		*/
/*					  	=1 : whole number	*/
/*	y_scale: scaling of the y - axis, see x_axis			*/
/*	h_align: horizontal alignment, 0 = left, 1 = center, 2= right   */
/*	v_align: vertical alignment, 0 = upper edge, 1 = center,	*/
/*				2 = lower edge				*/
/*	pxyarray[0], [1] : possible coordinates of upper left corner of */
/*		the output area						*/
/*	pxyarray[2], [3] : possible coordinates of lower right corner of */
/*		the output area			 			*/

{
	register short *ptmp;
	short n;

#ifdef __MSHORT__	/* we have 16 bit ints, just change vdi params */
	_vdiparams[2] = (void *) &pxyarray[0];
#else			/* 32 bit ints - let's copy */

	_ptsin[0] = pxyarray[0];
	_ptsin[1] = pxyarray[1];
	_ptsin[2] = pxyarray[2];
	_ptsin[3] = pxyarray[3];

#endif

	_intin[0] = aspect;
	_intin[1] = x_scale;
	_intin[2] = y_scale;
	_intin[3] = h_align;
	_intin[4] = v_align;

/* copy the file name */
	ptmp = _intin + 5;
	while( *ptmp ++ = (unsigned char) *filename++)
		;

	n = (ptmp - _intin) -1;

	__vdi__(VDI_CONTRL_ENCODE(5, 2, n, 23), handle);

#ifdef __MSHORT__
	_vdiparams[2] = (void *)&_ptsin[0]; /* restore vdi parameters */
#endif
}
#endif /* L_v_bit_im */


/*		vq_scan							*/
/*		printer escape						*/
/* INQUIRE PRINTER SCAN							*/
/* inquires different printer parameters				*/

#ifdef L_vq_scan

void vq_scan(int handle, int *g_slice, int *g_page, int *a_slice,
		int *a_page, int *div_fac)
/* Parameters:								*/
/*	*g_slice: The printer driver subdivides the page into several	*/
/*		'slices', which are formatted and output one after the	*/
/*		other to save memory; this is their number.		*/
/*	*g_page: number of pixels on a slice				*/
/*	*a_slice: height of a text line in pixels			*/
/*	*a_page: number of text lines per page				*/
/*	*div_fac: the other values have to be divided by this factor	*/

{
	__vdi__(VDI_CONTRL_ENCODE(5, 0, 0, 24), handle);

	*g_slice = _intout[0];
	*g_page  = _intout[1];
	*a_slice = _intout[2];
	*a_page  = _intout[3];
	*div_fac = _intout[4];
}
#endif /* L_vq_scan */

/*		v_alpha_text						*/
/*		printer escape						*/
/* OUTPUT ALPHA TEXT							*/
/* output text to printer (not in graphics mode), with certain control	*/
/* codes.								*/

#ifdef L_v_alpha_

void v_alpha_text(int handle, const char *string)
/* Parameters: 								*/
/*	*string: pointer to string, which may contain the		*/
/*		following control sequences:				*/
/*			DC2 0:	bold type on				*/
/*			DC2 1:  bold type off				*/
/*			DC2 2:  slanted type on				*/
/*			DC2 3:  slanted type off			*/
/*			DC2 4:  underscore on				*/
/*			DC2 5:  underscore off				*/
/* 		(DC2 is ASCII Nr. 18 = \022)				*/

{
	short n;
	short *ptmp = _intin;

	while (*ptmp++ = (unsigned char) *string++)
		;

	n = (ptmp - _intin) -1;

	__vdi__(VDI_CONTRL_ENCODE(5, 0, n, 25), handle);

}
#endif /* L_v_alpha_ */

/* WARNING: All bindings below this point require hand coding,
   because the subcode is > 31 */

/*		vs_palette						*/
/*		IBM CGA escape						*/
/* SELECT PALETTE							*/
/* selects a colour palette for IBM CGA grapics card in medium resolution */

#ifdef L_vs_palet

int vs_palette( int handle, int palette)
{
	short *wptr;
	_intin[0] = palette;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 1;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 60;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();

	return _intout[0];
}
#endif /* L_vs_palet */


/*		v_sound							*/
/*		screen escape						*/
/* GENERATE SPECIFIED TONE						*/
/* generates a tone of specified length and frequency.			*/

#ifdef L_v_sound

void v_sound(int handle, int frequency, int duration)
/* Parameters:								*/
/*	frequency: Frequency of the tone in Hertz			*/
/*	duration: length of tone in timer ticks				*/
{
	short *wptr;
	_intin[0] = frequency;
	_intin[1] = duration;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 2;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 61;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();
}
#endif /* L_v_sound */


/*		vs_mute							*/
/*		screen escape						*/
/* SET/CLEAR TONE MUTING FLAG						*/
/* sets or clears the tone muting flag or reports its status		*/

#ifdef L_vs_mute

int vs_mute(int handle, int action)
/* Parameter: action : -1	inquire status				*/
/*			0	switch on sound				*/
/*			1	switch off sound			*/
/* return value: 0 : sound on						*/
/*		 1 : sound off						*/
{
	short *wptr;
	_intin[0] = action;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 1;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 62;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();
	return _intout[0];
}
#endif /* L_vs_mute */


/*		vt_resolution						*/
/*		graphic tablet escape					*/
/* SET TABLET AXIS RESOLUTION IN LINES/INCH				*/
/* set horizontal and vertical resolution of a grapichs tablet		*/

#ifdef L_vt_resol

void vt_resolution( int handle, int xres, int yres, int *xset, int *yset)
/* Parameters:								*/
/*	xres, yres: resolution in lines per inch			*/
/*	*xset, *yset: the resolution set				*/
{
	short *wptr;
	_intin[0] = xres;
	_intin[1] = yres;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 2;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 81;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();

	*xset = _intout[0];
	*yset = _intout[1];
}
#endif /* L_vt_resol */


/*		vt_axis							*/
/*		graphic tablet escape					*/
/* SET TABLET AXIS RESOLUTION IN LINES					*/
/* sets the horizontal and vertical resolution of the grapics tablet	*/

#ifdef L_vt_axis

void vt_axis(int handle, int xres, int yres, int *xset, int *yset)
/* Parameters:								*/
/*	xres, yres: resolution in lines 				*/
/*	*xset, *yset: the resolution set				*/
{
	short *wptr;
	_intin[0] = xres;
	_intin[1] = yres;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 2;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 82;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();

	*xset = _intout[0];
	*yset = _intout[1];
}
#endif /* L_vt_axis */


/*		vt_origin						*/
/*		graphics tablet escape					*/
/* SET TABLET X AND Y ORIGIN						*/
/* sets tablet origin.							*/

#ifdef L_vt_origi

void vt_origin(int handle, int xorigin, int yorigin)
/* Parameters:								*/
/*	xorigin: x - coordinate for the left upper corner		*/
/*	yorigin: y - coordinate for the left upper corner		*/
{
	short *wptr;
	_intin[0] = xorigin;
	_intin[1] = yorigin;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 2;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 83;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();
}
#endif /* L_vt_origi */


/*		vt_tdimensions						*/
/*		graphics tablet escape					*/
/* RETURN TABLET X AND Y DIMENSIONS					*/
/* returns the grapics tablet's dimensions in 1/10 inch			*/

#ifdef L_vq_dimen

void vq_dimensions(int handle, int *xdimension, int *ydimension)
/* Parameters: you can guess, can't you?				*/
{
	short *wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 0;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 84;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();

	*xdimension = _intout[0];
	*ydimension = _intout[1];
}
#endif /* L_vq_dimen */


/*		vt_alignment						*/
/*		graphics tablet escape					*/
/* SET TABLET ALIGNMENT							*/

#ifdef L_vt_align

void vt_alignment(int handle, int dx, int dy)
/* Parameters: dx, dy : offsets for X- and Y - coordinate from origin	*/

{
	short *wptr;
	_intin[0] = dx;
	_intin[1] = dy;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 2;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 85;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();
}
#endif /* L_vt_align */


/*		vsp_film						*/
/*		polaroid palette escape					*/
/* SET CAMERA FILM TYPE AND EXPOSURE TIME				*/
/* just what it says...							*/

#ifdef L_vsp_film

void vsp_film(int handle, int index, int lightness)
/* Parameters:								*/
/*	index: number of film type					*/
/*	lightness: exposure time from -3 (half) to 0 (normal) to +3 (double) */
{
	short *wptr;
	_intin[0] = index;
	_intin[1] = lightness;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 2;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 91;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();
}
#endif /* L_vsp_film */


/*		vqp_filmname						*/
/*		polaroid palette escape					*/
/* INQUIRE CAMERA FILM NAME						*/
/* returns the name of a film, given its index, or the null string if	*/
/* it does not exist, with a maximum of 24 characters.			*/

#ifdef L_vqp_film

int vqp_filmname(int handle, int index, char *name)
/* Parameters:								*/
/*	index: Index of the film type					*/
/*	*name: name of the film type					*/
/* return value: 0: wrong number of film				*/
{
	short tmp, res;
	short *ptmp;
	short *wptr;

	_intin[0] = index;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 1;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 92;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();

	ptmp = _intout;
	res = _contrl[4];
	for (tmp = 0; tmp < res; tmp++)
		*name++ = *ptmp++;

	return res;
}
#endif /* L_vqp_film */


/*		vsc_expose(handle, state)				*/
/*		polaroid palette escape					*/
/* DISABLE OR ENABLE FILM EXPOSURE FOR FRAME PREVIEW			*/
/* ...									*/

#ifdef L_vsc_expo

void vsc_expose(int handle, int state)
/* Parameters:								*/
/*	state: 	= 0: switch off exposure				*/
/*		!= 0: switch on exposure				*/
{
	short *wptr;
	_intin[0] = state;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 1;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 93;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();
}
#endif /* L_vsc_expo */


/*		v_meta_extents						*/
/*		metafile escape						*/
/* UPDATE METAFILE EXTENTS						*/
/* all the information in a metafile header is renewed.			*/

#ifdef L_v_meta_e

void v_meta_extents(int handle, int min_x, int min_y, int max_x, int max_y)
/* Parameters:								*/
/*	min_x : minimum x - value of surrounding rectangle		*/
/*	min_y : dto., with y						*/
/*	max_x : maximum y - value of surrounding rectangle		*/
/*	max_y : dto, with y						*/
{
	short *wptr;
	_ptsin[0] = min_x;
	_ptsin[1] = min_y;
	_ptsin[2] = max_x;
	_ptsin[3] = max_y;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 2;		/* 1 - # of points */
	 wptr++;		/* 2 */
	*wptr++ = 0;		/* 3 - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 98;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();
}
#endif /* L_v_meta_e */


/*		v_write_meta						*/
/*		metafile escape						*/
/*		WRITE METAFILE ITEM					*/
/* write a user - defined opcode into a metafile. Note that opcodes	*/
/* 0 to 100 are reserved.						*/

#ifdef L_v_write_

void v_write_meta(int handle, int num_intin, int *a_intin, int num_ptsin,
			int *a_ptsin)
/* parameters:								*/
/*	num_intin: number of elements in the _intin - array		*/
/*	num_ptsin: number of elements in the _ptsin - array		*/
/*	a_intin[0] : user - defined sub - opcode			*/
/*	a_intin[1] ... a_intin[num_intin-1] : user - defined data	*/
/*	a_ptsin[0] ... a_ptsin[num_ptsin-1] : user - defined data	*/
/* GEM Draw uses the following opcodes:					*/
/*	START GROUP (10)						*/
/*	END GROUP (11)							*/
/*	SET NO LINE STYLE (49)						*/
/*	SET ATTRIBUTE SHADOW ON (50)					*/
/*	SET ATTRIBUTE SHADOW OFF (51)					*/
/*	START DRAW AREA PRIMITIVE (80)					*/
/*	END DRAW AREA PRIMITIVE (81)					*/
{
	short *wptr;
#ifdef __MSHORT__	/* we have 16 bit ints, just change vdi params */
	_vdiparams[1] = (void *) &a_intin[0];
	_vdiparams[2] = (void *) &a_ptsin[0];
#else			/* 32 bit ints - let's copy */
	register int i;

	for(i = 0; i < num_intin; i++)
		_intin[i] = a_intin[i];

	for(i = 0; i < num_ptsin; i++)
		_ptsin[i] = a_ptsin[i];
#endif

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = num_ptsin;	/* 1 - # of points */
	 wptr++;		/* 2 */
	*wptr++ = num_intin;	/* 3 - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 99;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();

#ifdef __MSHORT__
	_vdiparams[1] = (void *)&_intin[0]; /* restore vdi parameters */ 
	_vdiparams[2] = (void *)&_ptsin[0]; 
#endif
}
#endif /* L_v_write_ */


/*		vm_pagesize						*/
/*		metafile escape						*/
/* PHYSICAL PAGE SIZE							*/
/* set the page size in 1/10 mm						*/

#ifdef L_vm_pages

void vm_pagesize(int handle, int pgwidth, int pgheight)
/* Parameters:								*/
/*	pgwidth: width in 1/10 mm					*/
/*	pgheight: height in 1/10 mm					*/
{
	short *wptr;
	_intin[0] = 0;
	_intin[1] = pgwidth;
	_intin[2] = pgheight;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 3;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 99;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();
}
#endif /* L_vm_pages */


/*		vm_coords						*/
/*		metafile escape						*/
/* COORDINAT WINDOW							*/
/* sets the coordinate system for the page.				*/

#ifdef L_vm_coord

void vm_coords(int handle, int llx, int lly, int urx, int ury)
/* Parameters:								*/
/*	llx: x - coordinate in the left lower corner			*/
/*	lly: y - coordinate in the left lower corner			*/
/*	urx: x - coordinate in the right upper corner			*/
/*	ury: y - coordinate in the right upper corner			*/
{
	short *wptr;
	_intin[0] = 1;
	_intin[1] = llx;
	_intin[2] = lly;
	_intin[3] = urx;
	_intin[4] = ury;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 5;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 99;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();
}
#endif /* L_vm_coord */


/*		vm_filename						*/
/*		metafile escape						*/
/* CHANGE GEM VDI FILE NAME						*/
/* changes a metafile name from GEMFILE.GEM to <something esle>.GEM.	*/
/* Has no effect unless executed immediately after OPEN WORKSTATION.	*/

#ifdef L_vm_filen

void vm_filename(int handle, const char *filename)
{
	short n;
	register short *ptmp = _intin;
	short *wptr;
	char  *s=filename;

/* copy string */
	while (*ptmp++ = (unsigned char) *s++)
 		;
 	n = (s - filename) -1;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = n;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 100;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();

}
#endif /* L_vm_filen */


/*		v_escape2000						*/
/*		escape for Atari SLM804					*/
/* ESCAPE 2000								*/
/* prints the current page several times on the SLM804			*/

#ifdef L_v_escape

void v_escape2000(int handle, int times)
/* Parameters:								*/
/*	times: number of times to print the current page		*/
{
	short *wptr;
      	_intin[0] = times;

	 wptr = (short *)_contrl;
	*wptr++ = 5;		/* 0  - opcode */
	*wptr++ = 0;		/* 1 */
	 wptr++;		/* 2 */
	*wptr++ = 1;		/* 3  - # of entries in _intin */
	 wptr++;		/* 4 */
	*wptr++ = 2000;		/* 5 - id */
	*wptr   = handle;	/* 6 - handle */
	vdi();
}
#endif /* L_v_escape */
