/* 
   This file contains routines to manage states by modifying the event
   BUFFER, and then simply drawing the states out of that buffer.  This
   has the advantage of handling overlapping states properly; it allows
   allows us to display overlapping states in a clear fashion (by varying
   the width of the state lines)

   The routines are:
   FindStateInfo - modify the event buffer to contain state info.  Mostly,
   this consists of identifying events that are state boundaries, and 
   finding the matching state.  There is special handling for states that
   do not terminate within the "page"

   DisplayStates - Draw state info using the information in the event 
   buffer.
 */

#include "upshot.h"
#define EVENT_INF (EVENT_MAX + 100)

/* #define DEBUG */

/*

  FindStateInfo - run through a buffer of events and determine the matching
                  states

  Constraints:
  Overlapping states may be present (for example, a library routine may
  be logging its own states).

  Algorithm:
  For each event:
     Identify an event as a begin or end state.  
     Set offsets to matching records:
     If beginstate, add to that processor-lines list of states.  Set
         offset to +INF (incase end is not in this page)
     If endstate,   find the matching state in the processor line.
         (a) if none, then state started in a previous frame; set offset
	     to -INF
         (b) else, set offsets (for the begin and end)
     
     For simplicity, a stack is used to hold the pending states.  If
     a state value is NOT on the top of the stack, it is removed from the
     interior of a stack.  The assumption is that states are usually 
     strictly nested, and that the top of the stack contains the matching 
     state.
 */
void FindStateInfo()
{
int i,j,k;
int event,state_number,id;
struct stateNode *state;       /*pointer to state structure being examined*/
struct eventData *ev;
int   state_p;
struct state_line_info *sl;

for (i=0; i<procTaskTotal; i++) 
    state_check[i].nactive_states = 0;

/* NOTE: "next_event" is the buffer's index of next event to be referenced, 
   it indicates where to start checking for states specially helpful when a 
   shift draw has occured */

for(i = next_event; i<= last_displayed; i++) {
    ev            = buffer + i;
    event         = ev->event;
    ev->state_off = 0;
    if (event < MAX_EVTYPES) {
	state_number = state_reference[event];
	if (state_number > 0) {
	    state = states;       /* points at the header node */
	    for(k = 1; k <= state_number; k++)
		state = state->next;
	    
	    id = ev->id;
	    sl = state_check + id;

	    if(event == state->start)  /* if starting event record it */
		{
		/* These first two lines come from the old code */
 		sl->event = event;
		sl->time  = ev->time;
		/* New code starts here */
		if (sl->nactive_states >= MAX_ACTIVE_STATES) {
		    fprintf( stderr, 
			    "Exceeded number of overlapping states " );
		    fprintf( stderr, "proc = %d event = %d\n", id, ev->event );
		    state_p = MAX_ACTIVE_STATES - 1;
		    }
		else
		    state_p   = sl->nactive_states++;
		sl->state_stack[state_p] = state_number;
		sl->state_off[state_p]   = i;
		ev->state_off = EVENT_INF;
		ev->statep    = state;
		ev->state_num = state_number;
		ev->level     = state_p;
#ifdef DEBUG
if (ev->id == 2)
printf( "Adding state for event %d [%d]%d\n", i, ev->id, ev->event );
#endif
		}
	    else {
		/* End of a state */
		/* look for matching state */
		for (j=sl->nactive_states-1; j>=0; j--) {
		    if (sl->state_stack[j] == state_number) {
			/* Found the matching state */
			ev->state_off = - sl->state_off[j];
			ev->statep    = state;
			ev->state_num = state_number;
#ifdef DEBUG
if (ev->id == 2){
printf( "Found matching state for event %d [%d]%d ", -sl->state_off[j], 
        ev->id, ev->event );
printf( "Matched state stack position %d out of %d\n", j, sl->nactive_states-1 );
}
#endif
			buffer[-ev->state_off].state_off = i;
			if (j != sl->nactive_states-1) {
			    /* bump the state down */
			    for (k=j; k<=sl->nactive_states-2;k++) {
				sl->state_off[k]   = sl->state_off[k+1];
				sl->state_stack[k] = sl->state_stack[k+1];
				}
			    }
			sl->nactive_states--;
			break;
			}
		    }
		if (j == -1) {
		    /* no matching state in this page */
#ifdef DEBUG
if (ev->id == 2)
printf( "No matching state for [%d]%d\n", ev->id, ev->event );
#endif		    
                    if (firstPg->number > 0) {
			ev->state_off = -EVENT_INF;
			ev->statep    = state;
			ev->state_num = state_number;
			}
                    else {
			fprintf( stderr, 
			       "Unmatched end-of-state proc = %d event = %d\n",
				ev->id, ev->event );
			ev->statep    = 0;
			ev->state_num = 0;
			ev->state_off = 0;
			}
		    }
		}
	    }
	}
    }

/* Extra error checking */
/* printf( "pgtotal == %d firstpg = %d\n", pgTotal, firstPg->number ); */
if (pgTotal == firstPg->number + 1) {
    /* printf( "Checking for unclosed states..." ); */
    for (id = 0; id < procTaskTotal; id++) {
	sl = state_check + id;
	if (sl->nactive_states > 0) {
	    fprintf( stderr, 
		     "%d Unmatched beginning-of-state events on proc = %d\n", 
		     sl->nactive_states, id );
	    for (j=0; j<sl->nactive_states; j++) 
		fprintf( stderr, " %d", sl->state_stack[j] );
	    fprintf( stderr, "\n" );
	    }
	}
    }
}

/* Routine to display states */
DisplayStates()
{
int              i,j,k;
int              startX,endX,y; /* x and y coordinates of state bar */
int              event,state_number,id;
struct stateNode *state;       /*pointer to state structure being examined*/
struct eventData *ev;

if(!colorScreen)
    XSetFillStyle(display,gc,FillTiled);

for(i = next_event; i<= last_displayed; i++) {
    ev    = buffer + i;
    if (ev->state_off != 0) {
	state = ev->statep;
	if (ev->state_off == EVENT_INF) {
	    /* draw to right edge */
	    startX = time_to_pixels(ev->time)+8;
	    endX   = time_to_pixels(right_time)+8;
	    y      = find_proc_line(ev->id);
	    draw_state(startX,endX,y,state->color,ev->state_num,ev->level);
	    }
	else if (ev->state_off == -EVENT_INF) {
	    /* draw to left edge */
	    startX = time_to_pixels(left_time) + 8;
	    endX   = time_to_pixels(ev->time) + 8;
	    y      = find_proc_line(ev->id);
	    draw_state(startX,endX,y,state->color,ev->state_num,0);
	    }
	else if (ev->state_off > 0) {
	    /* draw to next state */
	    startX = time_to_pixels(ev->time)+8;
	    endX   = time_to_pixels(buffer[ev->state_off].time)+8;
	    y      = find_proc_line(ev->id);
	    /* ? Should we make sure that the times are clipped to
	       within the ranges on the canvas? */
	    /* if (startX < 0) printf( "Foo!\n" ); */
	    draw_state(startX,endX,y,state->color,ev->state_num,ev->level);
	    }
	}
    }
/* This does NOT handle those states that are constant across the "page" 
   (but neither does the old code) */


if(!colorScreen)
    XSetFillStyle(display,gc,FillSolid);

ShowStateKey();
}


plot_states2()
{
FindStateInfo();
DisplayStates();
}

/*
   When the ShiftPg code is correct for other than the subsequent pages,
   we can copy the data in state_check into bndy_check (including the 
   stack of unresolved state events).  We can then use this for states
   that are constant across a page
 */
