#############################################################################
##
#A  Matrix Group and G-module library                   Derek Holt
#A                                                      Charles Leedham-Green
#A                                                      Eamonn O'Brien
#A                                                      Sarah Rees 
##
#A  @(#)$Id$
##
#Y  Copyright 1994 -- School of Mathematical Sciences, ANU   
##
#H  $Log$
##
##
############################################################################
##
#F  InfoExtraSpecial (...)  . . . . . . . . . . . for debugging assistance
##
##
if not IsBound(InfoExtraSpecial)  then InfoExtraSpecial := Ignore;  fi;
#############################################################################
##
#F  ExtraSpecialDecomp( module,S)  . .  
##  module is a module for a finite matrix group G over a field of 
##  characteristic p.
##  S is a set of invertible matrices, assumed to act absolutely irreducibly 
##  on the underlying vector space of module.
##  Extraspecial returns true if (mod scalars)
##  S is an extraspecial r group, for some prime r,
##  or a 2-group of symplectic type (that is, the central product of an 
##  extra-special 2 group with a cyclic group of order 4), normalised by G.
##  Otherwise it returns false.
##
##  It attempts to prove that S is extra-special or of symplectic type by
##  construction. That is,
##  it tries to find elements x[1],y[1],x[2],y[2],...x[k],y[k],z, which generate
##  S (mod scalars) so that the x[i]'s and y[i]'s 
##  are non-central in <S>, z is central and scalar of order r, 
##  x[i] and x[j] commute for all i,j, 
##  y[i] and y[j] commute for all i,j, 
##  x[i] and y[j] commute for distinct i,j,
##  but the commutator of x[i] and y[i] is equal to z for all i.
##  Such generators  are found by adjusting the set S. 
##  x[1] and y[1] can be chosen as any two non-commuting non-scalar elements of
##  S (if there are none then S cannot be extraspecial/of symplectic type),
##  and z is set equal to their commutator.
##  Each other generator x[i] or y[i] is equal to a different non-scalar 
##  element of S multiplied by a word in those x[j]'s and y[j]'s 
##  already selected, or a power of such. 
##  More specifically, the function Stripped, applied to a generator in S,
##  multiplies it by a word in x[1],..x[i-1],y[1],..y[i-1] until the 
##  product commutes with all of those generators. We find x[i] and y[i],
##  if possible, by finding two such elements, both non-scalar,
##  and non-commuting. x[i] is the first, and y[i] is a non-scalar power 
##  of the second such that [x[i],y[i]] = z.
##  If the construction fails at any stage, the function
##  returns false. Otherwise, the construction stops with S exhausted. 
##  Finally conjugates of each x[i] and y[i] by generators of G are computed,
##  and checked for membership of <S> using the function Stripped.
##  If some conjugate is not in S false is returned, otherwise the function
##  returns true.
##  If the function returns true, the flag .extraSpecialPart is set to S,
## and .extraSpecialPrime to r.
## 
ExtraSpecialDecomp := function ( module,S) 

  local d,F,matrices,
        r,m,primes,p,x,y,z,doneGen,ngensG,ngensS,i,j,
        id,g,h,comm,u;

  d := DimFlag(module);
  F := FieldFlag(module);
  matrices := MatricesFlag(module);

  primes := FactorsInt(d);
  r := primes[1];
  m := Length(primes);
  for p in primes do
    if p <> r then 
      InfoExtraSpecial("Dimension ",d," is not a prime power.\n");
      return false;
    fi;
  od;

  ngensG := Length(matrices);
  ngensS := Length(S);
  x := [];
  y := [];
  doneGen := [];

  for j in [1..ngensS] do 
    h := S[j];
    if IsScalarMatrix(h) then doneGen[j] := true; 
    elif IsScalarMatrix(h^r) then 
      if x=[] then Add(x,h); doneGen[j] := true; 
      else doneGen[j] := false; 
      fi;
    else
      InfoExtraSpecial(r,"-th power of matrix is not scalar.\n");
      InfoExtraSpecial("ExtraSpecialDecomp returns false.\n");
      return false;
    fi;
  od;

  if x=[] then
    Error("All matrices in S are scalar.");
  fi;

  j:= 1;
  id := IdentityMat(d,F);
  while y=[] and j<= ngensS do
    if doneGen[j] = false then
      z := Comm(x[1],S[j]);
      if z <> id then
        if z^r <> id then 
          InfoExtraSpecial(r,"-th power of commutator is not identity.\n");
          return false;
        else 
          Add(y,S[j]);
          doneGen[j] := true;
        fi;
      fi;
    fi;
    j := j+1;
  od;

  if y=[] then
    Error("Z(S) contains non-scalar, S can't act abs. irred!");
  fi;

  for i in [2..m] do
    j := 1;
    while Length(x)<i and j<=ngensS do
      if doneGen[j] = false then
        h := Stripped(S[j],x,y,z,r);
        if h=false then 
          InfoExtraSpecial("Strip failed, group isn't extraspecial.\n");
          return false;
        elif IsScalarMatrix(h)=false then
          if IsScalarMatrix(h^r) then
            Add(x,h); doneGen[j] := true;
          else 
            InfoExtraSpecial(r,"-th power of matrix is not scalar.\n");
            return false;
          fi;
        fi;
      fi;
      j := j+1;
    od;

    if Length(x) < i then
      Error("Couldn't construct x[i].");
    fi;

    j := 1;
    while Length(y)<i and j<=ngensS do
      if doneGen[j] = false then
        h := Stripped(S[j],x,y,z,r);
        if h=false then 
          InfoExtraSpecial("Strip failed, group isn't extraspecial.\n");
          return false;
        else
          comm := Comm(x[i],h);
          if comm <>id then 
            u:=0;
            while u<r and comm <> z^u do u := u+1; od;
            if u=r then
              InfoExtraSpecial("Commutator isn't a power of z.\n");
              return false;
            fi;
            if u<>0 then 
              h := h^InverseMod(u,r); 
              if IsScalarMatrix(h^r) then
                Add(y,h); doneGen[j] := true;
              else 
                InfoExtraSpecial(r,"-th power of matrix is not scalar.\n");
                return false;
              fi;
            fi;
          fi;
        fi;
      fi;
      j := j+1;
    od;

    if Length(y) < i then
      Error("Couldn't construct y[i].");
    fi;

  od;

  for j in [1..ngensS] do
    if doneGen[j]=false then
      h := Stripped(S[j],x,y,z,r);
      if h=false then 
        InfoExtraSpecial("Group isn't extraspecial.\n");
        return false;
      elif IsScalarMatrix(h)=false then
        InfoExtraSpecial("Group isn't extraspecial.\n");
        return false;
      fi;
    fi;
  od;

  for j in [1..m] do
    for g in matrices do
      h := Stripped(x[j]^g,x,y,z,r);
      if h=false then 
        InfoExtraSpecial("Strip failed, group isn't extraspecial.\n");
        return false;
      elif IsScalarMatrix(h)=false then
        InfoExtraSpecial("Group isn't normalised by G.\n");
        return false;
      fi;
      h := Stripped(y[j]^g,x,y,z,r);
      if h=false then 
        InfoExtraSpecial("Strip failed, group isn't normalised by G.\n");
        return false;
      elif IsScalarMatrix(h)=false then
        InfoExtraSpecial("Group isn't normalised by G.\n");
        return false;
      fi;
    od;
  od;

  SetExtraSpecialFlag(module,true);
  SetExtraSpecialPartFlag(module,S);
  SetExtraSpecialPrimeFlag(module,r);
  return true;

end;

 

#############################################################################
##
#F  Stripped( h,x,y,z,r)  . .  
## 
## r is a prime, h,z, are invertible non-scalar matrices and x,y lists 
## of invertible matrices all of the same dimension over some finite field.
## y has length t, and x length t or t+1.
## The function returns a matrix hh equal to a product of h with a word in
## powers of x[1],y[1],...x[t],y[t] (in that order),and with the property
## that hh commutes with each of x[1],y[1],...,x[t],y[t].
## 
Stripped := function ( h,x,y,z,r) 

  local hh, t,u,v,c1,c2,i;

  t := Length(y);
  hh := Copy(h);

  for i in [1..t] do
    c1 := Comm(x[i],hh);
    c2 := Comm(y[i],hh);
    u:=0;
    while u<r and c1 <> z^u do u := u+1; od;
    if u=r then
      InfoExtraSpecial("Commutator isn't a power of z.\n");
      return false;
    fi;
    v:=0;
    while v<r and c2 <> z^v do v := v+1; od;
    if v=r then
      InfoExtraSpecial("Commutator isn't a power of z.\n");
      return false;
    fi;
  
    hh := hh*x[i]^v*y[i]^(r-u);
  od;

  return hh;
  
       
end;
     
#############################################################################
##
#F  InverseMod( u,p)  . .  
##  p is prime, 0 < u < p
##  The function returns the positive integer x less than p such that
##  ux = 1 mod p.
## 
InverseMod := function (u,p) 
   local x;
   x := GcdRepresentation(u,p)[1];

   if x>0 then return x; else return p+x; fi;
end;
