/*----------------------------------------------------------------------------
    MatVec.c	simple matrix vector structures, Mar 5, 1992

    written by Andreas Hohmann, ZIB, Hohmann@sc.ZIB-Berlin.DE
-----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>

#include "mystd.h"
#include "message.h"
#include "matvec.h"

RealVec NewRealVec(int l, int h) {
    RealVec v;
    Int j;
    if (l>h) {
    	Error("NewRealVec(%d,%d) wrong index range", l, h); 
    }
    v = (RealVec) malloc((h-l+1) * sizeof(Real)) - l;
    for (j=l; j<=h; j++) v[j] = 0;
    return v;
}

IntVec NewIntVec(int l, int h) {
    IntVec v;
    Int j;
    if (l>h) {
    	Error("NewIntVec(%d,%d) wrong index range", l, h); 
    }
    v = (IntVec) malloc((h-l+1) * sizeof(Int)) - l;
    for (j=l; j<=h; j++) v[j] = 0;
    return v;
}

RealVec* NewRealPtrVec(int l, int h) {
    if (l>h) {
    	Error("NewRealPtrVec(%d,%d) wrong index range", l, h); 
    }
    return (RealVec*) malloc((h-l+1) * sizeof(Real*)) - l;
}

RealMat NewRealMat(int rl, int rh, int cl, int ch) {
    Int i, j, nr = rh - rl + 1, nc = ch - cl + 1;
    RealMat m;
    RealVec v;
    
    if (rl>rh || cl>ch) {
    	Error("NewRealMat(%d,%d,%d,%d) wrong index range",
    				rl, rh, cl, ch); 
    }
    m = (RealMat) malloc(nr * sizeof(RealVec)) - rl;
    v = NewRealVec(0, nr * nc - 1);
    for (i=rl; i<=rh; i++) m[i] = v + (i - rl) * nc - cl;
    for (i=rl; i<=rh; i++) for (j=cl; j<=ch; j++) m[i][j] = 0;
    return m;
}

RealMat* NewRealPtrMat(int rl, int rh, int cl, int ch) {
    Int i, nr = rh - rl + 1, nc = ch - cl + 1;
    RealMat* m;
    RealVec* v;
    
    if (rl>rh || cl>ch) {
    	Error("NewRealPtrMat(%d,%d,%d,%d) wrong index range",
    				rl, rh, cl, ch); 
    }
    m = (RealMat*) malloc(nr * sizeof(RealVec*)) - rl;
    v = NewRealPtrVec(0, nr * nc - 1);
    for (i=rl; i<=rh; i++) m[i] = v + (i - rl) * nc - cl;
    return m;
}

RealMat NewRealLowMat(int rl, int rh, int cl, int ch) {
    Int i, j, k, nr = rh - rl + 1, nc = ch - cl + 1, n;
    Int ncNew = MIN(nr, nc), chNew = ncNew + cl - 1;
    RealMat m;
    RealVec v;
    
    if (rl>rh || cl>ch) {
    	Error("NewRealLowMat(%d,%d,%d,%d) wrong index range",
    				rl, rh, cl, ch); 
    }
    m = (RealMat) malloc(nr * sizeof(RealVec)) - rl;
    n = ncNew * (ncNew + 1)/2;
    if (nr>nc) n+= nc * (nr - nc);
    v = NewRealVec(0, n-1);
    for (i=rl, j=cl; i<=rh; i++, j++) {
        m[i] = v - cl;
        v += MIN(j, chNew) - cl + 1;
    }
    for (i=rl, j=cl; i<=rh; i++) {
        for (k=cl; k<=j; k++) m[i][k] = 0;
	if (j<ch) j++;
    }
    return m;
}

RealMat NewRealUpMat(int rl, int rh, int cl, int ch) {
    Int i, j, nr = rh - rl + 1, nc = ch - cl + 1, n;
    Int nrNew = MIN(nr, nc), rhNew = nrNew + rl - 1;
    RealMat m;
    RealVec v;
    
    if (rl>rh || cl>ch) {
    	Error("NewRealUpMat(%d,%d,%d,%d) wrong index range", rl, rh, cl, ch); 
    }
    m = (RealMat) malloc(nr * sizeof(RealVec)) - rl;
    n = nrNew * (nrNew + 1)/2;
    if (nc>nr) n += nr * (nc - nr);
    v = NewRealVec(0, n-1);
    for (i=rl, j=nc; i<=rhNew; i++, j--) {
       m[i] = v-i;
       v +=j;
    }
    return m;
}

RealMat* NewRealPtrLowMat(int rl, int rh, int cl, int ch) {
    Int i, j, nr = rh - rl + 1, nc = ch - cl + 1, n;
    Int ncNew = MIN(nr, nc), chNew = ncNew + cl - 1;
    RealMat* m;
    RealVec* v;
    
    if (rl>rh || cl>ch) {
    	Error("NewRealPtrLowMat(%d,%d,%d,%d) wrong index range", rl, rh, cl, ch); 
    }
    m = (RealMat*) malloc(nr * sizeof(RealVec*)) - rl;
    n = ncNew * (ncNew + 1)/2;
    if (nr>nc) n+= nc * (nr - nc);
    v = NewRealPtrVec(0, n-1);
    for (i=rl, j=cl; i<=rh; i++, j++) {
        m[i] = v - cl;
        v += MIN(j, chNew) - cl + 1;
    }
    return m;
}

RealMat RealSubMat(RealMat mOld, Int rl, Int rh, Int cl, Int ch, Int rlNew, Int clNew) {
    Int i, j;
    RealMat m;

    m = NewRealPtrVec(rlNew, rlNew + (rh-rl));
    for ( i= rl, j=rlNew; i<= rh; i++, j++) m[j] = mOld[i] + (cl - clNew);
    return m;
}

RealTen NewRealTen(int lMin, int lMax, int mMin, int mMax,
						int nMin, int nMax) {
    Int lNo = lMax - lMin + 1, mNo = mMax - mMin + 1, nNo = nMax - nMin + 1;
    Int i, j;
    RealTen t;
    RealVec v;
    
    if (lMin>lMax || mMin>mMax || nMin>nMax) {
    	Error("NewRealTen(%d,%d,%d,%d,%d,%d) wrong index range",
    				lMin, lMax, mMin, mMax, nMin, nMax); 
    }
    v = NewRealVec(0, lNo * mNo * nNo - 1);
    t = NewRealPtrMat(lMin, lMax, mMin, mMax);
    for (i=lMin; i<=lMax; i++) {
        for (j=mMin; j<=mMax; j++) {
            t[i][j] = v - nMin;
    	    v += nNo;
        }
    }
    return t;
}

/*----------------------------------------------------------------------------
    the following procedure defines a subdiagonal matrix of vectors
-----------------------------------------------------------------------------*/

RealTen NewRealLowTen(int lMin, int lMax, int mMin, int mMax,
						int nMin, int nMax) {
    Int lNo = lMax - lMin + 1, mNo = mMax - mMin + 1, nNo = nMax - nMin + 1;
    Int mNoNew = MIN(lNo, mNo);
    Int i, j, k, n;
    RealTen t;
    RealVec v;
    
    if (lMin>lMax || mMin>mMax || nMin>nMax) {
    	Error("NewRealLowTen(%d,%d,%d,%d,%d,%d) wrong index range",
    				lMin, lMax, mMin, mMax, nMin, nMax); 
    }
    n = mNoNew * (mNoNew + 1)/2;
    if (lNo>mNo) n+= mNo * (lNo - mNo);

    v = NewRealVec(0, n * nNo - 1);
    t = NewRealPtrLowMat(lMin, lMax, mMin, mMax);
    for (i=lMin, k=mMin; i<=lMax; i++) {
        for (j=mMin; j<=k; j++) {
            t[i][j] = v - nMin;
    	    v += nNo;
        }
	if (k<mMax) k++;
    }
    return t;
}

/*----------------------------------------------------------------------------
    deallocation procedures
-----------------------------------------------------------------------------*/

void FreeRealVec(RealVec x, Int nMin, Int nMax) {
    if (x!=nil) free((char *) (x+nMin));
}

void FreeIntVec(IntVec x, Int nMin, Int nMax) {
    if (x!=nil) free((char *) (x+nMin));
}

void FreeRealPtrVec(RealVec *x, Int nMin, Int nMax) {
    if (x!=nil) free((char *) (x+nMin));
}

void FreeRealMat(RealMat m, Int mMin, Int mMax, Int nMin, Int nMax) {
    if (m!=nil) {
	free((char *) (m[mMin] + nMin));
	free((char *) (m + mMin));
    }
}

void FreeRealSubMat(RealMat m, Int rl, Int rh) {
    FreeRealPtrVec(m, rl, rh);
}

void FreeRealPtrMat(RealMat *m, Int mMin, Int mMax, Int nMin, Int nMax) {
    if (m!=nil) {
	free((char *) (m[mMin] + nMin));
	free((char *) (m + mMin));
    }
}

void FreeRealTen(RealTen t, Int lMin, Int lMax, Int mMin, Int mMax,
				Int nMin, Int nMax) {
    if (t!=nil) {
	free((char *) (t[lMin][mMin] + nMin));
	free((char *) (t[lMin] + mMin));
	free((char *) (t + lMin));
    }
}

/*----------------------------------------------------------------------------
    utility procedures
-----------------------------------------------------------------------------*/

Real Distance(RealVec x, RealVec y, Int nMin, Int nMax) {
    Int j;
    Real tmp = 0;

    for (j=nMin; j<=nMax; j++) tmp += SQR(x[j]-y[j]);
    return sqrt(tmp);
}

Real Product(RealVec x, RealVec y, Int nMin, Int nMax) {
    Int j;
    Real tmp = 0;

    for (j=nMin; j<=nMax; j++) tmp += x[j] * y[j];
    return tmp;
}

Real Cosin(RealVec v, RealVec w, Int nl, Int nh) {
    Real temp;

    temp = Product(v, w, nl, nh);
    temp /= Norm(v, nl, nh);
    temp /= Norm(w, nl, nh);
    return temp;
}


Real Norm(RealVec x, Int nMin, Int nMax) {
    Int j;
    Real tmp = 0;

    for (j=nMin; j<=nMax; j++) tmp += SQR(x[j]);
    return sqrt(tmp);
}

Real SquaredNorm(RealVec v, Int nl, Int nh) {
    Int i;
    Real temp = 0;
    for (i=nl; i<=nh; i++) {
        temp += SQR(v[i]);
    }
    return temp;
}

Real SquaredDistance(RealVec v, RealVec w, Int nl, Int nh) {
    Real temp = 0;
    Int i;
    for (i=nl; i<=nh; i++) {
        temp += SQR(v[i] - w[i]);
    }
    return temp;
}   

Real MaxNorm(RealVec v, Int nl, Int nh) {
    Real temp, max;
    Int i;
    
    max = fabs(v[nl]);
    for (i=nl+1; i<=nh; i++) {
        temp = fabs(v[i]);
        if (max < temp) max = temp;
    }   
    return max;
}

/*-------------------------------------------------------------------------
    scaled Product, Distance, Norm
--------------------------------------------------------------------------*/

Real ScaledSquaredDistance(RealVec v, RealVec w, RealVec scale, Int nl, Int nh) {
    Real temp = 0;
    Int i;
    if (scale == nil)   return SquaredDistance(v, w, nl, nh);
    for (i=nl; i<=nh; i++) {
        temp += SQR((v[i] - w[i]) / scale[i]);
    }
    return temp;
}
    
Real ScaledDistance(RealVec y, RealVec z, RealVec scale, Int l, Int h) {
    return sqrt(ScaledSquaredDistance(y, z, scale, l,h));
} 

Real ScaledSquaredNorm(RealVec v, RealVec scale, Int nl, Int nh) {
    Int i;
    Real temp = 0;
    
    if (scale == nil)   return SquaredNorm(v, nl, nh);
    for (i=nl; i<=nh; i++) {
        temp += SQR(v[i] / scale[i]);
    }
    return temp;
}

Real ScaledNorm(RealVec y, RealVec scale, Int l, Int h) {
    return sqrt(ScaledSquaredNorm(y, scale, l, h));
} 

Real ScaledProduct(RealVec v, RealVec w, RealVec scale, Int nl, Int nh) {
    Int i;
    Real temp;
    
    if (scale == nil)   return Product(v, w, nl, nh);
    temp = 0;
    for (i=nl; i<=nh; i++) {
        temp += v[i] * w[i] / SQR(scale[i]);
    }
    return temp;
}

Real ScaledCosin(RealVec v, RealVec w, RealVec scale, Int nl, Int nh) {
    Real temp;

    if (scale == nil) {
       temp = Product(v, w, nl, nh);
       temp /= Norm(v, nl, nh);
       temp /= Norm(w, nl, nh);
    } else {
       temp = ScaledProduct(v, w, scale, nl, nh);
       temp /= ScaledNorm(v, scale, nl, nh);
       temp /= ScaledNorm(w, scale, nl, nh);
    }
    return temp;
}

Real ScaledMaxNorm(RealVec v, RealVec scale, Int nl, Int nh) {
    Real temp, max;
    Int i;
    
    if (scale == nil)   return MaxNorm(v, nl, nh);
    max = fabs(v[nl] / scale[nl]);
    for (i=nl+1; i<=nh; i++) {
        temp = DABS(v[i] / scale[i]);
        if (max < temp) max = temp;
    }   
    return max;
}

/*-------------------------------------------------------------------------
    vector operations
--------------------------------------------------------------------------*/

void FillRealVec(Real x, Int l, Int h, RealVec v) {
    Int i;
    for (i=l;i<=h;i++) v[i] = x;
}

void CopyIntVec(IntVec from, Int nMin, Int nMax, RealVec to) {
    Int j;

    for (j=nMin; j<=nMax; j++) to[j] = from[j];
}

void CopyRealVec(RealVec from, Int nMin, Int nMax, RealVec to) {
    Int j;

    for (j=nMin; j<=nMax; j++) to[j] = from[j];
}


void SwapRealVecs(RealVec v, RealVec w, Int nl, Int nh) {
    Int i;
    for (i=nl; i<=nh; i++) {
        REALSWAP(v[i], w[i]);
    }
}

void TurnRealVec(RealVec v, Int nl, Int nh) {
    Int i;
    for (i=nl; i<=nh; i++) {
        v[i] = - v[i];
    }
}

void Normalize(RealVec v, Int nl, Int nh) {
    Int i;
    Real tmp;

    tmp = Norm(v, nl, nh);
    for (i=nl; i<=nh; i++) v[i] /= tmp;
}

void ScalarVecMult(RealVec x, Real lambda, Int n, RealVec y) {
    Int i;
    for (i=1;i<=n;i++) y[i] = lambda*x[i];
}

void RealVecAdd(RealVec x, RealVec y, Int n, RealVec z) {
    Int i;
    for (i=1; i<=n; i++) z[i] = x[i] + y[i];
}

void RealVecSubtract(RealVec x, RealVec y, Int n, RealVec z) {
    Int i;
    for (i=1; i<=n; i++) z[i] = x[i] - y[i];
}

/*-------------------------------------------------------------------------
    matrix operations
--------------------------------------------------------------------------*/

void SwapColumns(RealMat a, Int j, Int k, Int ml, Int mh) {
    Int i;
    for (i=ml; i<=mh; i++) REALSWAP(a[i][j], a[i][k]);
}

/*-------------------------------------------------------------------------
    matrix-vector operations
--------------------------------------------------------------------------*/

void MatVecMult(RealMat a, RealVec x, Int m, Int n, RealVec b) {
    Real temp;
    Int i, j;

    for (i=1; i<=m; i++) {
        temp = 0;
        for (j=1; j<=n; j++) {
            temp += a[i][j] * x[j];
        }
        b[i] = temp;
    }
}

void MatTransVecMult(RealMat a, RealVec x, Int m, Int n, RealVec b) {
    Real temp;
    Int i, j;

    for (i=1; i<=n; i++) {
        temp = 0;
        for (j=1; j<=m; j++) {
            temp += a[j][i] * x[j];
        }
        b[i] = temp;
    }
}

/*-----------------------------------------------------------------------

 product of a symmetric matrix ( stored as lower diagonal matrix)
 and a vector    
-----------------------------------------------------------------------*/
    
void SymMatVecMult(RealMat a, RealVec x,  Int n,RealVec b) {
    Real temp;
    Int i, j;

    for (i=1; i<=n; i++) {
        temp = a[i][i] * x[i];
        for (j=1; j<i; j++) {
            temp += a[j][i] * x[j];
        }
        for (j=i+1; j<=n; j++) {
            temp += a[i][j] * x[j];
        }
        b[i] = temp;
    }
}

/*-----------------------------------------------------------------------
              matrix/ matrix operations
-----------------------------------------------------------------------*/

void CopyRealMat(RealMat a, Int nrl, Int nrh, Int ncl, Int nch, RealMat b)
{
    Int i, j;
    for (i=nrl; i<=nrh; i++) {
        for (j=ncl; j<=nch; j++)    b[i][j] = a[i][j];
    }
}

void CopyRealUpMat(RealMat from, Int nrl, Int nrh, Int ncl, Int nch, RealMat to) {
    Int i, j;

    if (nrl>ncl) Error("CopyRealUpMat: wrong index range");
    for (i=nrl; i<=nrh; i++) {
        for (j=ncl+i-nrl; j<=nch; j++)  to[i][j] = from[i][j];
    }
}

void CopyRealLowMat(RealMat from, Int nrl, Int nrh, Int ncl, Int nch, RealMat to) {
    Int i, j;

    if (ncl>nrl) Error("CopyRealLowMat: wrong index range");
    for (i=nrl; i<=nrh; i++) {
        for (j=ncl; j<=nch+i-nrh; j++)  to[i][j] = from[i][j];
    }
}

void RealMatMatMult(RealMat a, RealMat b, Int m, Int n, Int k, RealMat c)
{
    Int i, j, l;
    Real tmp;

    for (i=1; i<=m; i++) {
        for (j=1; j<=k; j++) {
	    tmp = 0;
	    for (l=1; l<=n; l++)   tmp += a[i][l] * b[l][j];
	    c[i][j] = tmp;
	}
    }
}

void RealMatAdd(RealMat a, RealMat b, Int m, Int n, RealMat c)
{
    Int i, j;

    for (i=1; i<=m; i++) {
        for (j=1; j<=n; j++)  c[i][j] = a[i][j] + b[i][j];
    }
}

void Transpose(RealMat a, Int m, Int n, RealMat b) {
    Real tmp;
    Int i, j, k = MIN(m, n);

    for (i=1; i<=k; i++) {
	for (j=i+1; j<=k; j++) {
	    tmp = a[j][i];
	    b[j][i] = a[i][j];
	    b[i][j] = tmp;
	}
    }
    if (m>=n) {
	for (i=n+1; i<=m; i++) {
	    for (j=1; j<=n; j++) b[j][i] = a[i][j];
	}
    }
    else {
	for (i=1; i<=m; i++) {
	    for (j=m+1; j<=n; j++) b[j][i] = a[i][j];
	}
    }
}

void SolveRealUpMat(RealMat r, RealVec b, Int n, RealVec x) {
    Real temp;
    Int i, j;
    
    for (i=n; i>=1; i--) {
        temp = b[i];
        for (j=i+1; j<=n; j++)  temp -= r[i][j] * x[j];
        x[i] = temp / r[i][i];
    }
}
    
void SolveTransposedRealUpMat(RealMat r, RealVec b, Int n, RealVec x) {
    Real temp;
    Int i, j;
    
    for (i=1; i<=n; i++) {
        temp = b[i];
        for (j=1; j<=i-1; j++)  temp -= r[j][i] * x[j];
        x[i] = temp / r[i][i];
    } 
}   

void SolveRealLowMat(RealMat L, RealVec b, Int n, RealVec x) {
   Real temp;
    Int i, j;
    
    for (i=n; i>=1; i--) {
        temp = b[i];
        for (j=i+1; j<=n; j++)  temp -= L[j][i] * x[j];
        x[i] = temp / L[i][i];
    }
}

void SolveTransposedRealLowMat(RealMat L, RealVec b, Int n, RealVec x) {
    Real temp;
    Int i, j;
    
    for (i=n; i>=1; i--) {
        temp = b[i];
        for (j=i+1; j<=n; j++) temp -= L[j][i] * x[j];
        x[i] = temp / L[i][i];
    }
}

/*----------------------------------------------------------------------------
    transform a vector into a matrix
-----------------------------------------------------------------------------*/

RealMat VectorMatrix(RealVec z, Int rh, Int ch) {
    RealMat m;
    Int i;
    
    m = NewRealPtrVec(1, rh);
    for (i=1; i<=rh; i++) m[i] = z + ch * (i-1);
    return m;
}

void FreeVectorMatrix(RealMat W, Int rh, Int ch) {
    FreeRealPtrVec(W, 1, rh);
}

/*----------------------------------------------------------------------------
    the following procedures realize the orthogonal projections on the
    subspace given by the ON-basis t[1..q][1..n] and its orthogonal
    complement, respectively. If P is the projection onto the subspace
    span(t[1],...,t[q]), we get

    Project:		xOut = P xIn
    OrthoProject:	xOut = (I-P) xIn

    The vectors t[1],...,t[q] have to be orthonormal with respect to the
    scaled product given by scale[1..n].
-----------------------------------------------------------------------------*/

void Project(RealMat t, RealVec xIn, RealVec scale, Int q, Int n, RealVec xOut) {
    Real tmp;
    Int j, i;

    FillRealVec(0.0, 1, n, xOut);
    for (i=1; i<=q; i++) {
	tmp = ScaledProduct(xIn, t[i], scale, 1, n);
	for (j=1; j<=n; j++) xOut[j] += tmp * t[i][j];
    }
}

void OrthoProject(RealMat t, RealVec xIn, RealVec scale, Int q, Int n, RealVec xOut) {
    Real tmp;
    Int j, i;

    CopyRealVec(xIn, 1, n, xOut);
    for (i=1; i<=q; i++) {
	tmp = ScaledProduct(xIn, t[i], scale, 1, n);
	for (j=1; j<=n; j++) xOut[j] -= tmp * t[i][j];
    }
}

/*----------------------------------------------------------------------------
    print vector to string (like the standard sprintf() routine)
-----------------------------------------------------------------------------*/

void sprintfRealVec(char *str, char *format, RealVec x, Int nMin, Int nMax) {
    Int j;
    char s[100];

    if (x==nil) sprintf(str, "nil");
    else {
	sprintf(str, "[");
	for (j=nMin; j<=nMax; j++) {
	    sprintf(s, format, x[j]);	strcat(str, s);
	    sprintf(s, " ");		strcat(str, s);
	}
	if (nMin<=nMax) strcat(str, "\b]"); else strcat(str, "]");
    }
}

void sprintIntVec(char *str, IntVec x, Int nMin, Int nMax) {
    Int j;
    char s[100];

    if (x==nil) sprintf(str, "nil");
    else {
	sprintf(str, "[");
	for (j=nMin; j<=nMax; j++) {
	    sprintf(s, "%d ", x[j]);	strcat(str, s);
	}
	if (nMin<=nMax) strcat(str, "\b]"); else strcat(str, "]");
    }
}

/*----------------------------------------------------------------------------
    output of vectors returning the output string
-----------------------------------------------------------------------------*/

char *PrintRealVec(char *format, RealVec x, Int nMin, Int nMax) {
    static char str[1000];
    sprintfRealVec(str, format, x, nMin, nMax);
    return str;
}

char *PrintIntVec(IntVec x, Int nMin, Int nMax) {
    static char str[1000];
    sprintIntVec(str, x, nMin, nMax);
    return str;
}

/*----------------------------------------------------------------------------
    print vector to file (like the standard fprintf() routine)
-----------------------------------------------------------------------------*/

void fprintfRealVec(FILE *f, char *format, RealVec x, Int nMin, Int nMax) {
    fprintf(f, "%s", PrintRealVec(format, x, nMin, nMax));
}

void fprintIntVec(FILE *f, IntVec x, Int nMin, Int nMax) {
    fprintf(f, "%s", PrintIntVec(x, nMin, nMax));
}

/*----------------------------------------------------------------------------
    print vector to stdout (like the standard printf() routine)
-----------------------------------------------------------------------------*/

void printfRealVec(char *format, RealVec x, Int nMin, Int nMax) {
     fprintfRealVec(stdout, format, x, nMin, nMax);
}

void printIntVec(IntVec x, Int nMin, Int nMax) {
     fprintIntVec(stdout, x, nMin, nMax);
}

/*----------------------------------------------------------------------------
    print matrix to string (like the standard sprintf() routine)
-----------------------------------------------------------------------------*/

void sprintfRealMat(char *str, char *format, RealMat a, Int mMin, Int mMax, Int nMin, Int nMax) {
    Int j;
    char s[1000];

    if (a==nil) sprintf(str, "nil");
    else {
	sprintf(str,"\n");
	for ( j= mMin;j<=mMax;j++) {
	    sprintfRealVec(s, format, a[j], nMin, nMax);
	    strcat(str, s);
	    strcat(str, "\n");
	}
    }
}
