/*******************************************************************************
+
+  LEDA  2.2.0                                                 03-05-1992
+
+
+  list.h
+
+
+  Copyright (c) 1992  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 6600 Saarbruecken, FRG     
+  All rights reserved.
+ 
*******************************************************************************/



#ifndef LISTH
#define LISTH

//------------------------------------------------------------------------------
//  doubly linked lists
//------------------------------------------------------------------------------

#include <LEDA/basic.h>

//------------------------------------------------------------------------------
// some declarations
//------------------------------------------------------------------------------

class dlist; 
class dlink;

typedef dlink* list_item;


//------------------------------------------------------------------------------
// class dlink (list items)
//------------------------------------------------------------------------------

class dlink {

  friend class dlist;

  dlink* succ;
  dlink* pred;
  GenPtr e;

  dlink(GenPtr a, dlink* pre, dlink* suc)
  { 
    e = a;
    succ = suc;
    pred = pre;
  }

  LEDA_MEMORY(dlink)

//space: 3 words = 12 bytes
};


//------------------------------------------------------------------------------
// dlist: base class for all doubly linked lists
//------------------------------------------------------------------------------

class dlist {

   dlink* h;                     //head
   dlink* t;                     //tail
   dlink* iterator;              //iterator
   int count;                    //length of list

//space: 4 words + virtual =  5 words = 20 bytes

virtual int  usr_cmp(GenPtr, GenPtr)  { return 0; }
virtual int  usr_ord(GenPtr)          { return 0; }
virtual void usr_apply(GenPtr&)       { }
 
virtual int  cmp_el(GenPtr x, GenPtr y)    const { return int(x)-int(y); }
virtual void read_el(GenPtr& x,istream& in)         { in >> *(int*)&x; }
virtual void print_el(GenPtr& x,ostream& out) const { out << int(x); }
virtual void clear_el(GenPtr& x) const { x=0; }
virtual void copy_el(GenPtr& x)  const { x=x; }

void quick_sort(list_item* A,int l, int r);
void quick_sort1(list_item* A,int l, int r);


public:

// access operations

   int  length() const { return count; }
   int  size()   const { return count; }
   bool empty()  const { return (count==0) ? true : false;}

   dlink* first()               const { return h; }
   dlink* first_item()          const { return h; }
   dlink* last()                const { return t; }
   dlink* last_item()           const { return t; }
   dlink* next_item(dlink* p)   const { return p->succ; }
   dlink* succ(dlink* l)        const { return l->succ; }
   dlink* pred(dlink* l)        const { return l->pred; }
   dlink* cyclic_succ(dlink*)   const;
   dlink* cyclic_pred(dlink*)   const;
   dlink* succ(dlink* l, int i) const; 
   dlink* pred(dlink* l, int i) const;
   dlink* get_item(int = 0)     const; 

   dlink* max(CMP_PTR) const;
   dlink* min(CMP_PTR) const;
   dlink* search(GenPtr) const;

   int    rank(GenPtr) const;

   GenPtr contents(dlink* l) const { return l->e; }
   GenPtr head()             const { return h ? h->e : 0;}
   GenPtr tail()             const { return t ? t->e : 0;}


// update operations

//protected:

public:

   dlink* insert(GenPtr a, dlink* l, int dir=0);
   dlink* push(GenPtr a);
   dlink* append(GenPtr a);
   void   assign(dlink* l, GenPtr a) { copy_el(a); l->e = a; }


public:

   GenPtr del(dlink* loc);
   GenPtr pop();
   GenPtr Pop();

   void   conc(dlist&);
   void   split(list_item,dlist&,dlist&);
   void   apply();
   void   sort(int);
   void   bucket_sort(int,int);
   void   permute();
   void   clear();

// iteration

   void   set_iterator(dlink* loc) const     
   { const list_item *const p = &iterator;    // hack to keep const
     (*(dlink**)p) = loc;
    }

   void  init_iterator()        const { set_iterator(nil); }
   void  reset()                const { set_iterator(nil); }
   dlink* get_iterator()        const { return iterator; }

   dlink* move_iterator(int=0)  const;
   bool   current_element(GenPtr&) const;
   bool   next_element(GenPtr&)    const;
   bool   prev_element(GenPtr&)    const;


// operators

   GenPtr&   entry(dlink* l)            { return l->e; }
   GenPtr    inf(dlink* l)        const { return l->e; }
   GenPtr&   operator[](dlink* l)       { return l->e; }
   GenPtr    operator[](dlink* l) const { return l->e; }

   dlist& operator=(const dlist&); 
   dlist  operator+(const dlist&); 


   void print(ostream&,string, char)       const;    
   void print(ostream& out,char space=' ') const { print(out,"",space);  }
   void print(string s, char space=' ')    const { print(cout,s,space);  }
   void print(char space=' ')              const { print(cout,"",space); }   


   void read(istream&,string, char);  
   void read(istream& in,char delim=EOF) { read(in,"",delim); }
   void read(string s, char delim='\n')  { read(cin,s,delim); }   
   void read(char delim='\n')            { read(cin,"",delim);}  


// constructors & destructors

   dlist();    
   dlist(GenPtr a);
   dlist(const dlist&);

  ~dlist()  { clear(); }

   int space()  const { return sizeof(dlist) + count * sizeof(dlink); }
};


//------------------------------------------------------------------------------
// list: generic dlists
//-----------------------------------------------------------------------------


#define list_body(type)\
\
: public dlist {\
\
typedef int  (*COMPARE(type,list)) (type&, type&);\
typedef void (*APPLY(type,list))   (type&);\
typedef int  (*ORD(type,list))     (type&);\
\
type name2(type,X);\
\
COMPARE(type,list) cmp_ptr;\
APPLY(type,list) apply_ptr;\
ORD(type,list)     ord_ptr;\
\
int  usr_cmp(GenPtr x, GenPtr y) \
                          { return cmp_ptr(ACCESS(type,x),ACCESS(type,y));}\
void usr_apply(GenPtr& x) { apply_ptr(ACCESS(type,x));}\
int  usr_ord(GenPtr x)    { return ord_ptr(ACCESS(type,x));}\
\
int  cmp_el(GenPtr x, GenPtr y) const\
                             { return compare(ACCESS(type,x),ACCESS(type,y));}\
void print_el(GenPtr& x,ostream& out) const { Print(ACCESS(type,x),out);  }\
void read_el(GenPtr& x,istream& in)         { Read(name2(type,X),in);\
                                              x = Copy(name2(type,X)); }\
void clear_el(GenPtr& x)              const { Clear(ACCESS(type,x)); }\
void copy_el(GenPtr& x)               const { x = Copy(ACCESS(type,x)); }\
\
public:\
\
void   conc(_list(type)& l)    { dlist::conc((dlist&)l); }\
void   split(list_item p, _list(type)& l1, _list(type)& l2)\
                              { dlist::split(p,(dlist&)l1,(dlist&)l2);}\
list_item search(type a) const{ return dlist::search(Convert(a)); }\
int       rank(type a)   const{ return dlist::rank(Convert(a)); }\
\
list_item push(type a)   { return dlist::push(Copy(a));}\
list_item append(type a) { return dlist::append(Copy(a));}\
list_item insert(type a, list_item l, int dir=0)\
                         { return dlist::insert(Copy(a),l,dir); }\
\
type  contents(list_item l) const { return ACCESS(type,dlist::contents(l)); }\
type  inf(list_item l)      const { return contents(l); }\
\
type  head()                const { return ACCESS(type,dlist::head()); }\
type  tail()                const { return ACCESS(type,dlist::tail()); }\
\
type  pop()      { type x=head(); Clear(ACCESS(type,dlist::pop())); return x; }\
type  Pop()      { type x=tail(); Clear(ACCESS(type,dlist::Pop())); return x; }\
\
type  del_item(list_item a)     { type x=contents(a);\
                                  Clear(ACCESS(type,dlist::del(a)));return x; }\
\
type  del(list_item a)          { return del_item(a); }\
\
void  assign(list_item p,type a) { dlist::assign(p,Copy(a));}\
\
list_item max(COMPARE(type,list) f=0) const { return dlist::max(CMP_PTR(f)); }\
list_item min(COMPARE(type,list) f=0) const { return dlist::max(CMP_PTR(f)); }\
\
void  sort()                       { dlist::sort(0); }\
void  sort(COMPARE(type,list) f  ) { cmp_ptr = f; dlist::sort(1); }\
void  apply(APPLY(type,list) f)    { apply_ptr=f; dlist::apply(); }\
void  bucket_sort(int i, int j, ORD(type,list) f)\
                                   { ord_ptr=f; dlist::bucket_sort(i,j); }\
\
bool current_element(type& x) const\
                                  {GenPtr y; bool b=dlist::current_element(y);\
                                   if (b) x = ACCESS(type,y); return b; }\
bool next_element(type& x)   const{GenPtr y; bool b = dlist::next_element(y);\
                                   if (b) x = ACCESS(type,y); return b; }\
bool prev_element(type& x)   const{GenPtr y; bool b = dlist::prev_element(y);\
                                   if (b) x = ACCESS(type,y); return b; }\
\
_list(type)() {}\
_list(type)(const _list(type)& a) : dlist((dlist&)a) {}\
_list(type)(type a)              : dlist(Copy(a))    {}\
\
~_list(type)() { clear(); }\
\
_list(type)& operator=(const _list(type)& a)\
                            { dlist::operator=(a); return *this;}\
\
_list(type) operator+(const _list(type)& a)\
                            { dlist L = *(dlist*)this + *(dlist*)&a;\
                              return *(_list(type)*)&L; }\
list_item operator+=(type x)\
                            { return append(x); }\
\
list_item operator[](int i) { return get_item(i); }\
\
type&  operator[](list_item p)\
                            { return *(type*)&entry(p); }\
\
type   operator[](list_item p) const\
                            { void* x = dlist::inf(p); return ACCESS(type,x); }\
\
}; 



#ifdef __TEMPLATES__
#define _list(type) list
template<class type> class list list_body(type)

#else
#define _list(type) name2(type,list)
#define list(type) name2(type,list)
#define listdeclare(type) class list(type) list_body(type)

#endif

//------------------------------------------------------------------------------
// Iteration Macros
//------------------------------------------------------------------------------

#define forall_list_items(a,b) for( a=(b).first(); a ; a=(b).succ(a) )
#define Forall_list_items(a,b) for( a=(b).last();  a ; a=(b).pred(a) )

/* forall(a,b)  defined in basic.h */

#endif
