/* copyright 1987,1988,1989 Phil Andrews, Pittsburgh Supercomputing Center */
/* all rights reserved */
#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include "defs.h"
#include "ccdefs.h"
#define byte_size 8
#define byte_mask 255
/* module for CGM clear text output */

/* store these pointers to allow lookups into the CGM data structures */
static struct one_opt 		*popt;	/* the command line options, in only */
static struct mf_d_struct 	*pc1;	/* the class 1 elements, in only */
static struct pic_d_struct 	*pc2;	/* the class 2 elements, in only */
static struct control_struct	*pc3;	/* the class 3 elements, in only */
static struct attrib_struct	*pc5;	/* the class 5 elements, in only */
static struct info_struct 	*dptr;	/* device info, in and out */
static struct info_struct *dev1_info;	/* device info to fill out, out only */

static int state_level;		/* marker for where we are in inclusions */

/* flags for the flush_cmd function */
#define final_flush 1 		/* final flush */
static double vdc_pxl;		/* conversion from pxl to vdc */
/* macros to handle direct colours */
#define inddc(i, rval) ((int) (pc1->c_v_extent.min[i] + \
(rval) * (pc1->c_v_extent.max[i] - pc1->c_v_extent.min[i])))

#define rdc(rval) inddc(0, rval)
#define gdc(rval) inddc(1, rval)
#define bdc(rval) inddc(2, rval)





/* get an output command started */
static start_cmd(class, el)
int class, el;
{
#define cl_max 15
#define el_max 127

	if ((class > cl_max) || (class < 0)) {
	    (void) fprintf(stderr, "illegal class for output %d\n", class);
	    return(0);
	}
	if ((el > el_max) || (el < 0)) {
	    (void) fprintf(stderr, "illegal element for output %d\n", el);
	    return(0);
	}
	out_string(cc_cptr[class][el]);

	return(1);
}

/* flush out the command, note we are assuming only one partition for
   metafile default replacement commands (limits total to max_long) */
static flush_cmd(this_flush)
int this_flush;	/* what kind of flush */
{
int i;
	if (this_flush != final_flush) {
	    (void) fprintf(stderr, "illegal flush in cc %d\n", this_flush);
	    return(0);
	}
	outc(term_char);
	fb();
	return(1);
}
/* put out a CGM clear text string */
static cc_string(cptr, slen)
char *cptr;		/* string to do */
int slen;		/* string length */
{
int i, no_parts, to_do;
char byte1, byte2;
	if (slen < 0) {
	    (void) fprintf(stderr, "illegal output string length %d\n", slen);
	    return(0);
	}

	outc(' ');
	outc(quote_char);
	for (i=0; i<slen; ++i) {
	    if (cptr[i] == quote_char) outc(quote_char);
	    outc(cptr[i]);
	}
	outc(quote_char);
	return(1);
}


/* general signed integer output routine */
static cc_int(xin)
{
	intout(xin);
	return(1);
}
/* general real variable */
static cc_real(xin)
float xin;
{
char buffer[max_str];

	sprintf(buffer, " %.6f", xin);
	out_string(buffer);
	return(1);
}
/* put out an integer point */
static cc_ipoint(x, y)
int x, y;
{
char buffer[max_str];

	sprintf(buffer, "(%d,%d)", x, y);
	out_string(buffer);
	return(1);
}
/* put out a real point */
static cc_rpoint(x, y)
float x, y;
{
char buffer[max_str];

	sprintf(buffer, "(%.4f,%.4f)", x, y);
	out_string(buffer);
	return(1);
}
/* general point */
static cc_vpoint(x, y)
int x, y;
{

	switch (pc1->vdc_type) {
case vdc_int:	/* shouldn't have to rescale */
	    return(cc_ipoint(x, y));
case vdc_real:	return(cc_rpoint((double) x * vdc_pxl, (double) y * vdc_pxl));
	}
	return(0);	/* trouble */
}
/* general VDC */
static cc_vdc(x)
int x;
{
	switch (pc1->vdc_type) {
case vdc_int: /* shouldn't need to rescale */
	    return(cc_int(x));
case vdc_real:	return(cc_real((double) x * vdc_pxl));
	}
	return(0);	/* trouble */
}

/* begin the commands */
/* the delimiter functions */
cc_begin(comment, file_name, prog_name)
char *comment, *file_name, *prog_name;
{

	start_cmd(0, (int) B_Mf);
	if (*comment) cc_string(comment, strlen(comment));
	else cc_string(NULL, 0);
	flush_cmd(final_flush);
	if (pc1->vdc_type == vdc_int) {
	    dev1_info->x_size = pc2->vdc_extent.i[2] - pc2->vdc_extent.i[0];
	    dev1_info->y_size = pc2->vdc_extent.i[3] - pc2->vdc_extent.i[1];
	} else {
	    dev1_info->x_size = pc2->vdc_extent.r[2] - pc2->vdc_extent.r[0];
	    dev1_info->y_size = pc2->vdc_extent.r[3] - pc2->vdc_extent.r[1];
	}
	return(1);
}

cc_end(pages_done)
int pages_done;
{
	start_cmd(0, (int) E_Mf);
	flush_cmd(final_flush);
	fb();		/* flush out the buffer */
	return(1);
}
/* start the picture */
cc_bp(pic_name)
char *pic_name;
{
	start_cmd(0, (int) B_Pic);
	if (*pic_name) cc_string(pic_name, strlen(pic_name));
	else cc_string(NULL, 0);
	flush_cmd(final_flush);
	if (pc1->vdc_type == vdc_int) {
	    dev1_info->x_size = pc2->vdc_extent.i[2] - pc2->vdc_extent.i[0];
	    dev1_info->y_size = pc2->vdc_extent.i[3] - pc2->vdc_extent.i[1];
	} else {
	    dev1_info->x_size = pc2->vdc_extent.r[2] - pc2->vdc_extent.r[0];
	    dev1_info->y_size = pc2->vdc_extent.r[3] - pc2->vdc_extent.r[1];
	}
	return(1);
}
/* start the picture body */
cc_bpage(pic_name, xoffset, yoffset, rotation, rb, gb, bb, page_no,
    xsize, ysize)
char *pic_name;
float rotation;		/* how much to rotate the page */
int xoffset, yoffset;	/* offset in pixels */
int page_no;		/* page number */
float rb, gb, bb;	/* background colour values */
int xsize, ysize;
{
	start_cmd(0, (int) B_Pic_Body);
	flush_cmd(final_flush);

	/* try to get the right vdc_pxl */
	vdc_pxl = (pc1->vdc_type == vdc_int) ? 1.0 :
	    dev1_info->x_size / 32767.0;
	return(1);
}
/* end the picture */
cc_epage(no_copies)
int no_copies;
{
	start_cmd(0, (int) E_Pic);
	flush_cmd(final_flush);

	return(1);
}
/* the Metafile descriptor elements */
/* the Metafile version */
cc_mfversion(vers_no)
int vers_no;
{
	start_cmd(1, (int) MfVersion);
	/* only version number allowed is 1 */
	if (vers_no != 1) vers_no = 1;
	cc_int(vers_no);
	flush_cmd(final_flush);
	return(1);
}
/* Metafile description */
cc_mfdescrip(char_ptr)
char *char_ptr;
{
	start_cmd(1, MfDescrip);
	cc_string(char_ptr, strlen(char_ptr));
	flush_cmd(final_flush);
	return(1);
}
/* VDC type */
cc_vdctype(new_type)
enum vdc_enum new_type;
{
	start_cmd(1, (int) vdcType);
	switch (new_type) {
case vdc_int:	out_string(" Integer"); break;
case vdc_real:	out_string(" Real"); break;
default:	fprintf(stderr, "illegal vdc type\n"); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}
/* integer precision */
cc_intprec(new_prec)
int new_prec;
{
int minint, maxint, temp;

	start_cmd(1, (int) IntPrec);
	temp = 1 << (new_prec - 2);
	maxint = temp + (temp - 1);
	minint = -2 * temp;
	cc_int(minint);
	cc_int(maxint);
	flush_cmd(final_flush);
	return(1);
}
/* real precision */
cc_realprec(fixed, exp_part, fract)
int fixed, exp_part, fract;
{
double minreal, maxreal;
int digits;

	start_cmd(1, (int) RealPrec);
	maxreal = 1 << (exp_part - 1);
	minreal = -maxreal;
	digits = fract / 4;	/* maybe */
	cc_real(minreal);
	cc_real(maxreal);
	cc_int(digits);
	flush_cmd(final_flush);
	return(1);
}
/* index precision */
cc_indexprec(new_prec)
int new_prec;
{
int maxint, minint, temp;

	start_cmd(1, (int) IndexPrec);
	temp = 1 << (new_prec - 2);
	maxint = temp + (temp - 1);
	minint = -2 * temp;
	cc_int(minint);
	cc_int(maxint);
	flush_cmd(final_flush);
	return(1);
}

/* colour precision */
cc_colprec(new_prec)
int new_prec;
{
	start_cmd(1, (int) ColPrec);
	cc_int((1 << new_prec) -1 );
	flush_cmd(final_flush);
	return(1);
}
/* colour index precision */
cc_cindprec(new_prec)
int new_prec;
{
	start_cmd(1, (int) CIndPrec);
	cc_int((1 << new_prec) -1 );
	flush_cmd(final_flush);
	return(1);
}
/* colour value extent */
cc_cvextent(cvmin, cvmax)
int *cvmin, *cvmax;
{
int i;

	start_cmd(1, (int) CVExtent);
	for (i=0; i<3; ++i) cc_int(cvmin[i]);
	for (i=0; i<3; ++i) cc_int(cvmax[i]);
	flush_cmd(final_flush);
	return(1);
}

/* maximum colour index */
cc_maxcind(new_index)
int new_index;
{
	start_cmd(1, (int) MaxCInd);
	cc_int(new_index);
	flush_cmd(final_flush);
	return(1);
}
/* Metafile element list */
cc_mfellist(no_pairs, array_ptr)
int no_pairs, *array_ptr;
{
int i, done_def = 0;
	start_cmd(1, (int) MfElList);

	outc(' ');
	outc(quote_char);
	for (i=0; i< 2 * no_pairs; i +=2) {
	    if ((array_ptr[i] == -1)) {
	    	if (array_ptr[i+1] == 0) out_string("DrawingSet");
	    	else
	    	if (array_ptr[i+1] == 1) out_string("DrawingPlus");
	    	else {
		    fprintf(stderr, "illegal element list\n");
		    return(2);
	    	}
	    } else 
	    if ((array_ptr[i] == 2) && (array_ptr[i+1] == 13) && !done_def){
		out_string("BegMFDefaults EndMFDefaults");
		done_def = 1;
	    } else {
		out_string(cc_cptr[array_ptr[i]][array_ptr[i+1]]);
	    }
	    if (i < 2 * (no_pairs - 1)) outc(' ');
	}
	outc(quote_char);
	flush_cmd(final_flush);
	return(1);
}
/* Metafile defaults replacement list */
cc_mfdefrep(state)
int state;	/* 0 => start, 1 => end */
{

	if (state == 0) {
	    out_string("BegMFDefaults;");
	    fb();
	} else if (state == 1) {
	    out_string("EndMFDefaults;");
	    fb();
	} else {
	    (void) fprintf(stderr, "illegal state in MD def rep %d\n", state);
	    return(0);
	}
	return(1);
}
/* font list */
cc_flist(no_strings, str_ptr)
int no_strings;
char str_ptr[][max_str+1];
{
int i;
	start_cmd(1, (int) FontList);
	for (i=0; i<no_strings; ++i) {
	    cc_string(str_ptr[i], strlen(str_ptr[i]));
	}
	flush_cmd(final_flush);
	return(1);
}
/* character list */
cc_clist(no_strings, type_array, str_ptr)
int no_strings, *type_array;
char str_ptr[][max_str+1];
{
int i;
	start_cmd(1, (int) CharList);
	for (i=0; i<no_strings; ++i) {
	    switch(type_array[i]) {
case 0:		out_string(" STD94"); break;
case 1:		out_string(" STD96"); break;
case 2:		out_string(" STD94MultiByte"); break;
case 3:		out_string(" STD96MultiByte"); break;
case 4:		out_string(" CompleteCode"); break;
default:	fprintf(stderr, "illegal font type\n"); return(2);
	    }
	    cc_string(str_ptr[i], strlen(str_ptr[i]));
	}
	flush_cmd(final_flush);
	return(1);
}
/* character announcer */
cc_cannounce(announcer)
int announcer;
{
	start_cmd(1, (int) CharAnnounce);
	switch(announcer) {
case 0:	    out_string(" Basic7Bit"); break;
case 1:	    out_string(" Basic8Bit"); break;
case 2:	    out_string(" Extd7Bit"); break;
case 3:	    out_string(" Extd8Bit"); break;
default:    fprintf(stderr, "illegal announcer %d\n", announcer); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}

/* the page descriptor elements */
/* the scaling mode */
cc_scalmode(mode, my_scale)
int mode;
float my_scale;
{
	start_cmd(2, (int) ScalMode);
	switch (mode) {
case 0:	    out_string(" Abstract"); break;
case 1:	    out_string(" Metric");
	    cc_real(my_scale); break;
default:    fprintf(stderr, "illegal scaling mode %d\n", mode); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}
/* the colour selection mode */
cc_colselmode(c_s_mode)
enum cs_enum c_s_mode;
{
	start_cmd(2, (int) ColSelMode);
	switch (c_s_mode) {
case i_c_mode:	out_string(" Indexed"); break;
case d_c_mode:	out_string(" Direct"); break;
default:	fprintf(stderr, "illegal col sel mode\n"); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}
/* the line width specification mode */
cc_lwsmode(mode)
enum spec_enum mode;
{
	start_cmd(2, (int) LWidSpecMode);
	switch(mode) {
case absolute:	out_string(" Abstract"); break;
case scaled:	out_string(" Scaled"); break;
default:	fprintf(stderr, "illegal lw spec mode\n"); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}
/* the marker size specification mode */
cc_msmode(mode)
enum spec_enum mode;
{
	start_cmd(2, (int) MarkSizSpecMode);
	switch(mode) {
case absolute:	out_string(" Abstract"); break;
case scaled:	out_string(" Scaled"); break;
default:	fprintf(stderr, "illegal ms spec mode\n"); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}
/* the edge width specification mode */
cc_ewmode(mode)
enum spec_enum mode;
{
	start_cmd(2, (int) EdWidSpecMode);
	switch(mode) {
case absolute:	out_string(" Abstract"); break;
case scaled:	out_string(" Scaled"); break;
default:	fprintf(stderr, "illegal ew spec mode\n"); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}
/* the vdc extent */
cc_vdcextent(int_coords, float_coords)
int *int_coords;
float *float_coords;
{
int i;
	start_cmd(2, (int) vdcExtent);
	switch (pc1->vdc_type) {
case vdc_int:	cc_ipoint(int_coords[0], int_coords[1]);
		cc_ipoint(int_coords[2], int_coords[3]); 
		dev1_info->x_size = pc2->vdc_extent.i[2] - pc2->vdc_extent.i[0];
		dev1_info->y_size = pc2->vdc_extent.i[3] - pc2->vdc_extent.i[1];
		break;
case vdc_real:	cc_rpoint(float_coords[0], float_coords[1]);
		cc_rpoint(float_coords[2], float_coords[3]); break;
default:	(void) fprintf(stderr, "illegal vdc_type %d\n", 
		    (int) pc1->vdc_type); return(0);
	}
	flush_cmd(final_flush);
	return(1);
}

/* the background colour */
cc_backcol(r, g, b)
int r, g, b;
{
	start_cmd(2, (int) BackCol);
	cc_int(r);
	cc_int(g);
	cc_int(b);
	flush_cmd(final_flush);
	return(1);
}


/* now the control functions (class 3) */
/* set the vdc integer precision */
cc_vdcintprec(precision)
int precision;
{
int maxint, minint, temp;
	start_cmd(3, (int) vdcIntPrec);
	temp = 1 << (precision - 2);
	maxint = temp + (temp - 1);
	minint = -2 * temp;
	cc_int(minint);
	cc_int(maxint);
	flush_cmd(final_flush);

	return(1);
}
/* vdc real precision */
cc_vdcrprec(fixed, exp_part, fract)
int fixed, exp_part, fract;
{
double minreal, maxreal;
int digits;

	start_cmd(3, (int) vdcRPrec);
	maxreal = 1 << (exp_part - 1);
	minreal = -maxreal;
	digits = fract / 4;	/* maybe */
	cc_real(minreal);
	cc_real(maxreal);
	cc_int(digits);
	flush_cmd(final_flush);
	return(1);
}
/* auxiliary colour */
cc_auxcol(r, g, b, index)
float r, g, b;
int index;
{
	start_cmd(3, (int) AuxCol);
	switch (pc2->c_s_mode) {
case (int) i_c_mode:	cc_int(index); break;
case (int) d_c_mode:	cc_int(rdc(r));
		cc_int(gdc(g));
		cc_int(bdc(b));
		break;
default:	(void) fprintf(stderr, "illegal col sel mode\n");
		return(0);
	}
	flush_cmd(final_flush);
	return(1);
}
/* set the transparency */
cc_transp(transparency)
enum boolean transparency;
{
	start_cmd(3, Transp);
	switch (transparency) {
case off:	out_string(" Off"); break;
case on:	out_string(" On"); break;
	}
	flush_cmd(final_flush);
	return(1);
}
/* set the clipping rectangle */
cc_cliprect(int_coords, float_coords)
int *int_coords;
float *float_coords;
{
int i;
	start_cmd(3, (int) ClipRect);
	switch (pc1->vdc_type) {
case vdc_int:	cc_vpoint(int_coords[0], int_coords[1]);
		cc_vpoint(int_coords[2], int_coords[3]); break;
case vdc_real:	cc_rpoint(float_coords[0], float_coords[1]);
		cc_rpoint(float_coords[2], float_coords[3]); break;
default:	(void) fprintf(stderr, "illegal vdc_type %d\n", 
		    (int) pc1->vdc_type); return(0);
	}

	flush_cmd(final_flush);
	return(1);
}
/* set the clipping indicator */
cc_clipindic(clip_ind)
enum boolean clip_ind;
{

	start_cmd(3, (int) ClipIndic);
	switch (clip_ind) {
case off:	out_string(" Off"); break;
case on:	out_string(" On"); break;
	}
	flush_cmd(final_flush);

	return(1);
}


/* the graphical primitives */
/* plot a set of lines */
cc_pline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i;
	start_cmd(4, (int) PolyLine);
	for (i=0; i<no_pairs; ++i) cc_vpoint(x1_ptr[i], y1_ptr[i]);
	flush_cmd(final_flush);
	return(1);
}
/* plot a set of lines between alternate points */
cc_dpline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i;

	start_cmd(4, (int) Dis_Poly);
	for (i=0; i<no_pairs; ++i) cc_vpoint(x1_ptr[i], y1_ptr[i]);
	flush_cmd(final_flush);
	return(1);
}
/* put up a series of markers, characters are set at bottom left corner */
cc_pmarker(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i;
	start_cmd(4, (int) PolyMarker);
	for (i=0; i<no_pairs; ++i) cc_vpoint(x1_ptr[i], y1_ptr[i]);
	flush_cmd(final_flush);
	return(1);
}

/* set text */
cc_text(x, y, final, buffer)
int x, y, final;
char *buffer;
{
	start_cmd(4, (int) Text);
	cc_vpoint(x, y);
	if (final) out_string(" Final");
	else out_string(" NotFinal");
	cc_string(buffer, strlen(buffer));
	flush_cmd(final_flush);
	return(1);
}
/* set restricted text */
cc_rex_text(width, height, x, y, final, buffer)
int width, height, x, y, final;
char *buffer;
{
	start_cmd(4, (int) Rex_Text);
	cc_vdc(width);
	cc_vdc(height);
	cc_vpoint(x, y);
	if (final) out_string(" Final");
	else out_string(" NotFinal");
	cc_string(buffer, strlen(buffer));
	flush_cmd(final_flush);
	return(1);
}
/* appended text */
cc_app_text(final, buffer)
enum boolean final;
char *buffer;
{
	start_cmd(4, (int) App_Text);
	if (final) out_string(" Final");
	else out_string(" NotFinal");
	cc_string(buffer, strlen(buffer));
	flush_cmd(final_flush);
	return(1);
}
/* do a polygon */
cc_pgon(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
int i;
	start_cmd(4, (int) Polygon);
	for (i=0; i<no_pairs; ++i) cc_vpoint(x1_ptr[i], y1_ptr[i]);
	flush_cmd(final_flush);
	return(1);
}
/* do a polyset */
cc_pset(no_pairs, x1_ptr, y1_ptr, edge_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
char *edge_ptr;
{
int i;
	start_cmd(4, (int) Poly_Set);
	for (i=0; i<no_pairs; ++i) {
	    cc_vpoint(x1_ptr[i], y1_ptr[i]);
	    switch(edge_ptr[i]) {
    case 0:    	out_string(" INVIS"); break;
    case 1:    	out_string(" VIS"); break;
    case 2:    	out_string(" CLOSEINVIS"); break;
    case 3:    	out_string(" CLOSEVIS"); break;
    default:	(void) fprintf(stderr, "illegal edge ptr %d\n", edge_ptr[i]);
		break;
	    }
	}
	flush_cmd(final_flush);
	return(1);
}
/* cgm cell array */
cc_carray(cp, cq, cr, nx, ny, col_prec, dat_ptr, rep_mode, no_bytes)
int *cp;	/* first corner */
int *cq;	/* the diagonal to cp */
int *cr;	/* first row is from cp to cr */
int nx;		/* number of elements in x rows (i.e., parallel to cp-cr) */
int ny;		/* number of rows */
int col_prec;	/* the colour precision (how many possible colour values */
unsigned char *dat_ptr;	/* pointer to the body of information */
int rep_mode;	/* the representation mode; 0, run length encoded, 1 normal */
long int no_bytes;	/* numer of bytes in the cell array representation */
/* without compression there should be ny * row_size bytes after dat_ptr, 
   the first nx * c_size bits of each row should contain information */
{
int i, j, carray_prec;
unsigned int index, ir, ig, ib;
enum cs_enum c_s_mode;	/* the colour selection mode, indexed or direct */
char *new_ptr = NULL, *allocate_mem();
unsigned char *start_ptr;
int true_size, row_size;/* how many bytes in a row of data (maybe rounding) */
int c_size;	/* how many bits to specify one colour value */
static int temp;
/* grab an unsigned  cgm integer at arbitrary (legal) precision */
#define mcr_aint(ptr, precision, out, done) switch (precision){\
case 32: out = (((unsigned int)ptr[0]) << 24) + \
(((unsigned int)ptr[1]) << 16) + (ptr[2] << 8) + ptr[3];\
ptr += 4; break;\
case 24:out = (((unsigned int)ptr[0]) << 16) + \
(((unsigned int)ptr[1]) << 8) + ptr[2]; ptr += 3; break;\
case 16:out = (((unsigned int)ptr[0]) << 8) + ptr[1]; ptr += 2; break;\
case 8: out = *ptr++; break;\
case 4:if (!(temp=(((done + 1) * precision) % byte_size)))out = *ptr++ & 15;\
else out = (*ptr >> temp) & 15; break;\
case 2:if (!(temp=(((done + 1) * precision) % byte_size)))out = *ptr++ & 3;\
else out = (*ptr >> (byte_size - temp)) & 3; break;\
case 1:if (!(temp=(((done + 1) * precision) % byte_size)))out = *ptr++ & 1;\
else out = (*ptr >> (byte_size - temp)) & 1; break;}

	start_cmd(4, (int) Cell_Array);
	cc_vpoint(cp[0], cp[1]);
	cc_vpoint(cq[0], cq[1]);
	cc_vpoint(cr[0], cr[1]);
	cc_int(nx);
	cc_int(ny);
	carray_prec = (col_prec < (byte_size * sizeof(int))) ? 
	    (1 << col_prec) - 1 : (1 << (col_prec - 1)) - 1;
	if (carray_prec < 255) carray_prec = 255;	/* fix later */
	cc_int(carray_prec);
	/* now handle the data itself */
	c_s_mode = pc2->c_s_mode;	/* start with this, may change */
	switch (c_s_mode) {
case d_c_mode:	c_size = 3 * col_prec; break;
case i_c_mode:	c_size = col_prec; break;
default:	fprintf(stderr, "illegal colour selection mode ! (%d)\n",
		    (int) c_s_mode); return(2);
	}
/* now get the row size in bytes allowing for rounding */
	true_size = row_size = (nx * c_size + byte_size - 1) / byte_size;	/* check */
	row_size = (row_size % 2) ? row_size + 1 : row_size; 	/* round up */

	/* expand a compressed raster */
	if (rep_mode == 0) {
	    new_ptr = allocate_mem(ny * row_size, 1);
	    if (!new_ptr) return(2);
	    (void) c_expand(dat_ptr, nx, ny, row_size, c_size, new_ptr);
	    dat_ptr = (unsigned char *) new_ptr; /* carry on as before */
	}
	
	/* now output the data, use doint for speed, efficiency */
	start_ptr = dat_ptr;
	switch (c_s_mode) {
case d_c_mode:	/* direct colour */
	    for (i=0; i < ny; ++i){
		dat_ptr = start_ptr;
		fb(); 		/* put end of line here */
		outc('(');
		for (j=0; j < nx; ++j){
		    mcr_aint(dat_ptr, col_prec, ir, 3 * j)
		    doint(ir);
		    outc(' ');
		    mcr_aint(dat_ptr, col_prec, ig, 3 * j + 1)
		    doint(ig);
		    outc(' ');
		    mcr_aint(dat_ptr, col_prec, ib, 3 * j + 2)
		    doint(ib);
		    if (j<(nx-1)) outc(',');
		}
		outc(')');
		start_ptr += row_size;
	    }
	    break;
case i_c_mode:	/* indexed colour */
	    for (i=0; i < ny; ++i){
		dat_ptr = start_ptr;
		fb(); 		/* put end of line here */
		outc('(');
	    	for (j=0; j < nx; ++j){
		    mcr_aint(dat_ptr, col_prec, index, j);
		    doint(index);
		    if (j<(nx-1)) outc(',');
		}
		outc(')');
		start_ptr += row_size;
	    }
	}
	flush_cmd(final_flush);
	return(1);
}
/* generalised graphics drawing primitive */
cc_gdp(gpd_id, no_pts, x_ptr, y_ptr, data_record)
int gpd_id, no_pts, *x_ptr, *y_ptr;
char *data_record;
{
int i;
	start_cmd(4, (int) Gen_D_Prim);
	cc_int(gpd_id);
	cc_int(no_pts);
	for (i=0; i<no_pts; ++i) cc_vpoint(x_ptr[i], y_ptr[i]);
	cc_string(data_record, strlen(data_record));
	flush_cmd(final_flush);
	return(1);
}

/* set a rectangle */
cc_rectangle(x_1, y_1, x2, y2)
int x_1, x2, y_1, y2;
{

	start_cmd(4, (int) Rectangle);
	cc_vpoint(x_1, y_1);
	cc_vpoint(x2, y2);
	flush_cmd(final_flush);
	return(1);
}
/* set  a Circle */
cc_circle(x, y, r)
int x, y, r;
{
	start_cmd(4, (int) Cgm_Circle);
	cc_vpoint(x, y);
	cc_vdc(r);
	flush_cmd(final_flush);
	return(1);
}
/* set  an arc, get the positions of 1st pt, intermdiate pt, end pt */
cc_c3(pt_ptr)
int *pt_ptr;
{
int i;
	start_cmd(4, (int) Circ_3);
	for (i=0; i<6; i +=2) cc_vpoint(pt_ptr[i], pt_ptr[i+1]);
	flush_cmd(final_flush);
	return(1);
}
/* set  a closed arc, get the positions of 1st pt, intermdiate pt, end pt */
cc_c3_close(pt_ptr, chord)
int *pt_ptr;
enum boolean chord;
{
int i;
	start_cmd(4, (int) Circ_3_Close);
	for (i=0; i<6; i +=2) cc_vpoint(pt_ptr[i], pt_ptr[i+1]);
	if (chord) out_string(" Chord");
	else out_string(" Pie");
	flush_cmd(final_flush);
	return(1);
}
/* set an arc, ends specified by vectors */
cc_c_centre(x, y, vec_array, r)
int x, y, vec_array[], r;
{
int i;
	start_cmd(4, (int) Circ_Centre);
	cc_vpoint(x,y);
	for (i=0; i<4; i += 2) cc_vpoint(vec_array[i], vec_array[i + 1]);
	cc_vdc(r);
	flush_cmd(final_flush);
	return(1);
}
/* set an arc, ends specified by vectors, but close it */
cc_c_c_close(x, y, vec_array, r, chord)
int x, y, vec_array[], r;
enum boolean chord;
{
int i;
	start_cmd(4, (int) Circ_C_Close);
	cc_vpoint(x,y);
	for (i=0; i<4; i += 2) cc_vpoint(vec_array[i], vec_array[i + 1]);
	cc_vdc(r);
	if (chord) out_string(" Chord");
	else out_string(" Pie");
	flush_cmd(final_flush);
	return(1);
}
/* set an ellipse, centre and two endpoints */
cc_ellipse(pt_array)
int *pt_array;
{
int i;
	start_cmd(4, (int) Ellipse);
	for (i=0; i<6; i += 2) cc_vpoint(pt_array[i], pt_array[i + 1]);
	flush_cmd(final_flush);
	return(1);
}
/* set an elliptical arc, centre, two endpoints, vectors for ends */
cc_ell_arc(pt_array, vec_array)
int *pt_array, *vec_array;
{
int i;
	start_cmd(4, (int) Ellip_Arc);
	for (i=0; i<6; i += 2) cc_vpoint(pt_array[i], pt_array[i + 1]);
	for (i=0; i<4; i += 2) cc_vpoint(vec_array[i], vec_array[i + 1]);
	flush_cmd(final_flush);
	return(1);
}
/* set an elliptical arc, close it */
cc_e_a_close(pt_array, vec_array, chord)
int *pt_array, *vec_array;
enum boolean chord;
{
int i;
	start_cmd(4, (int) El_Arc_Close);
	for (i=0; i<6; i += 2) cc_vpoint(pt_array[i], pt_array[i + 1]);
	for (i=0; i<4; i += 2) cc_vpoint(vec_array[i], vec_array[i + 1]);
	if (chord) out_string(" Chord");
	else out_string(" Pie");
	flush_cmd(final_flush);
	return(1);
}

/* now the attributes (class 5) */
/* set the line bundle index */
cc_lbindex(index)
int index;
{
	start_cmd(5, (int) LBIndex);
	cc_int(index);
	flush_cmd(final_flush);
	return(1);
}
/* set the line type */
cc_ltype(line_type)
enum line_enum line_type;
{
	start_cmd(5, (int) LType);
	cc_int((int) line_type);
	flush_cmd(final_flush);
	return(1);
}
/* set the line width */
cc_lwidth(line_width, rmul)
int line_width;
float rmul;
{
	start_cmd(5, (int) LWidth);
	switch (pc2->l_w_s_mode) {
case absolute	:	cc_vdc(line_width); break;
case scaled	:	cc_real(rmul); break;
default		:	(void) fprintf(stderr, "illegal l_w_s_mode");
			return(2);
	}
	flush_cmd(final_flush);
	return(1);
}
/* set the line colour */
cc_lcolour(r, g, b, index)
float r, g, b;
int index;
{
	start_cmd(5, (int) LColour);
	switch (pc2->c_s_mode) {
case (int) i_c_mode:	cc_int(index); break;
case (int) d_c_mode:	cc_int(rdc(r));
		cc_int(gdc(g));
		cc_int(bdc(b));
		break;
default:	(void) fprintf(stderr, "illegal col sel mode\n");
		return(0);
	}
	flush_cmd(final_flush);
	return(1);
}
/* set the marker bundle index */
cc_mbindex(index)
int index;
{
	start_cmd(5, (int) MBIndex);
	cc_int(index);
	flush_cmd(final_flush);
	return(1);
}

/* set the marker type */
cc_mtype(marker)
int marker;
{
	start_cmd(5, (int) MType);
	cc_int(marker);
	flush_cmd(final_flush);
	return(1);
}
/* marker size */
cc_msize(mk_size, rmul)
int mk_size;
float rmul;
{
	start_cmd(5, (int) MSize);
	switch (pc2->m_s_s_mode) {
case (int) absolute	: cc_vdc(mk_size); break;
case (int) scaled	: cc_real(rmul); break;
default 		: (void) fprintf(stderr, "illegal m_s_s_mode\n");
			  return(0);
	}
	flush_cmd(final_flush);
	return(1);
}
/* marker colour */
cc_mcolour(r, g, b, index)
float r, g, b;
int index;
{
	start_cmd(5, (int) MColour);
	switch (pc2->c_s_mode) {
case i_c_mode:	cc_int(index); break;
case d_c_mode:	cc_int(rdc(r));
		cc_int(gdc(g));
		cc_int(bdc(b));
		break;
default:	(void) fprintf(stderr, "illegal col sel mode\n");
		return(0);
	}

	flush_cmd(final_flush);
	return(1);
}
/* set the text bundle index */
cc_tbindex(index)
int index;
{
	start_cmd(5, (int) TBIndex);
	cc_int(index);
	flush_cmd(final_flush);
	return(1);
}
/* text font index */
cc_tfindex(index)
int index;
{
	start_cmd(5, (int) TFIndex);
	cc_int(index);
	flush_cmd(final_flush);
	return(1);
}
/* text precision */
cc_tprec(precision)
enum txt_enum precision;
{
	start_cmd(5, (int) TPrec);
	switch(precision) {
case string:	out_string(" String"); break;
case character:	out_string(" Character"); break;
case stroke:	out_string(" Stroke"); break;
default:	fprintf(stderr, "illegal string precision\n"); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}
/* character expansion factor */
cc_cexpfac(factor)
float factor;
{
	start_cmd(5, (int) CExpFac);
	cc_real(factor);
	flush_cmd(final_flush);
	return(1);
}
/* character space */
cc_cspace(space)
float space;
{
	start_cmd(5, (int) CSpace);
	cc_real(space);
	flush_cmd(final_flush);
	return(1);
}
/* text colour */
cc_tcolour(r, g, b, index)
float r, g, b;
int index;
{
	start_cmd(5, (int) TColour);
	switch (pc2->c_s_mode) {
case i_c_mode:	cc_int(index); break;
case d_c_mode:	cc_int(rdc(r));
		cc_int(gdc(g));
		cc_int(bdc(b));
		break;
default:	(void) fprintf(stderr, "illegal col sel mode\n");
		return(0);
	}
	flush_cmd(final_flush);
	return(1);
}
/* character height */
cc_cheight(height)
int height;
{
	start_cmd(5, (int) CHeight);
	cc_vdc(height);
	flush_cmd(final_flush);
	return(1);
}
/* character orientation */
cc_corient(x_up, y_up, x_base, y_base)
int x_up, y_up, x_base, y_base;
{
	start_cmd(5, (int) COrient);
	cc_vdc(x_up);
	cc_vdc(y_up);
	cc_vdc(x_base);
	cc_vdc(y_base);
	flush_cmd(final_flush);
	return(1);
}
/* text path */
cc_tpath(new_path)
enum path_enum new_path;
{
	start_cmd(5, (int) TPath);
	switch (new_path) {
case right:	out_string(" Right"); break;
case left:	out_string(" Left"); break;
case up:	out_string(" Up"); break;
case down:	out_string(" Down"); break;
default:	fprintf(stderr, "illegal text path\n"); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}

/* text alignment */
cc_talign(hor, ver, cont_hor, cont_ver)
enum hor_align hor;
enum ver_align ver;
float cont_hor, cont_ver;
{
	start_cmd(5, (int) TAlign);
	switch (hor) {
case normal_h:	out_string(" NormHoriz"); break;
case left_h:	out_string(" Left"); break;
case center_h:	out_string(" Ctr"); break;
case right_h:	out_string(" Right"); break;
case cont_h:	out_string(" ContHoriz"); break;
default:	fprintf(stderr, "illegal hor txt align\n"); return(2);
	}
	switch (ver) {
case normal_v:	out_string(" NormVert"); break;
case top_v:	out_string(" Top"); break;
case cap_v:	out_string(" Cap"); break;
case half_v:	out_string(" Half"); break;
case base_v:	out_string(" Base"); break;
case bottom_v:	out_string(" Bottom"); break;
case cont_v:	out_string(" ContVert"); break;
default:	fprintf(stderr, "illegal ver txt align\n"); return(2);
	}
	cc_real(cont_hor);
	cc_real(cont_ver);
	flush_cmd(final_flush);
	return(1);
}
/* character set index */
cc_csindex(new_index)
int new_index;
{
	start_cmd(5, (int) CSetIndex);
	cc_int(new_index);
	flush_cmd(final_flush);
	return(1);
}
/* alternate character set index */
cc_acsindex(new_index)
int new_index;
{
	start_cmd(5, (int) AltCSetIndex);
	cc_int(new_index);
	flush_cmd(final_flush);
	return(1);
}
/* fill bundle index */
cc_fbindex(new_index)
int new_index;
{
	start_cmd(5, (int) FillBIndex);
	cc_int(new_index);
	flush_cmd(final_flush);
	return(1);
}
/* interior style */
cc_intstyle(style)
enum is_enum style;
{
	start_cmd(5, (int) IntStyle);
	switch (style) {
case hollow:	out_string(" Hollow"); break;
case solid_i:	out_string(" Solid"); break;
case pattern:	out_string(" Pat"); break;
case hatch:	out_string(" Hatch"); break;
case empty:	out_string(" Empty"); break;
default:	fprintf(stderr, "illegal interior style\n"); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}
/* set the fill colour */
cc_fillcolour(r, g, b, index)
float r, g, b;
int index;
{
	start_cmd(5, (int) FillColour);
	switch (pc2->c_s_mode) {
case i_c_mode:	cc_int(index); break;
case d_c_mode:	cc_int(rdc(r));
		cc_int(gdc(g));
		cc_int(bdc(b));
		break;
default:	(void) fprintf(stderr, "illegal col sel mode\n");
		return(0);
	}
	flush_cmd(final_flush);
	return(1);
}
/* hatch index */
cc_hindex(new_index)
int new_index;
{
	start_cmd(5, (int) HatchIndex);
	cc_int(new_index);
	flush_cmd(final_flush);
	return(1);
}
/* pattern index */
cc_pindex(new_index)
int new_index;
{
	start_cmd(5, (int) PatIndex);
	cc_int(new_index);
	flush_cmd(final_flush);
	return(1);
}
/* edge bundle index */
cc_ebindex(new_index)
int new_index;
{
	start_cmd(5, (int) EdBIndex);
	cc_int(new_index);
	flush_cmd(final_flush);
	return(1);
}

/* edge type */
cc_etype(etype)
enum line_enum etype;
{
	start_cmd(5, (int) EType);
	cc_int((int) etype);
	flush_cmd(final_flush);
	return(1);
}
/* edge width */
cc_edwidth(width, rmul)
int width;
float rmul;
{
	start_cmd(5, (int) EdWidth);
	switch (pc2->e_w_s_mode) {
case (int) absolute	: cc_vdc(width); break;
case (int) scaled	: cc_real(rmul); break;
default 		: (void) fprintf(stderr, "illegal e_w_s_mode\n");
			  return(0);
	}
	flush_cmd(final_flush);
	return(1);
}
/* edge colour */
cc_edcolour(r, g, b, index)
float r, g, b;
int index;
{
	start_cmd(5, (int) EdColour);
	switch (pc2->c_s_mode) {
case i_c_mode:	cc_int(index); break;
case d_c_mode:	cc_int(rdc(r));
		cc_int(gdc(g));
		cc_int(bdc(b));
		break;
default:	(void) fprintf(stderr, "illegal col sel mode\n");
		return(0);
	}
	flush_cmd(final_flush);
	return(1);
}
/* edge visibility */
cc_edvis(visible)
enum boolean visible;
{
	start_cmd(5, (int) EdVis);
	switch (visible) {
case on:	out_string(" On"); break;
case off:	out_string(" Off"); break;
default:	fprintf(stderr, "illegal visibility\n"); return(2);
	}
	flush_cmd(final_flush);
	return(1);
}
/* fill reference point */
cc_fillref(x, y)
int x, y;
{
	start_cmd(5, (int) FillRef);
	cc_vpoint(x, y);
	flush_cmd(final_flush);
	return(1);
}
/* pattern table entry */
cc_pattab(index, nx, ny, col_prec, dat_ptr, no_bytes)
int index, nx, ny, col_prec, no_bytes;
char *dat_ptr;
{
int i;
	start_cmd(5, (int) PatTab);
	cc_int(index);
	cc_int(nx);
	cc_int(ny);
	cc_int((1 << col_prec) - 1);
	/* fix later */
	flush_cmd(final_flush);
	return(1);
}

/* pattern size */
cc_patsize(size_array)
int *size_array;
{
int i;
	start_cmd(5, (int) PatSize);
	for (i=0; i<4; ++i) cc_vdc(size_array[i]);
	flush_cmd(final_flush);
	return(1);
}
/* colour table */
cc_coltab(beg_index, no_entries, ctab)
int beg_index, no_entries;
float *ctab;
{
int i, j;
	start_cmd(5, (int) ColTab);
	cc_int(beg_index);
	for (i=beg_index; i<(beg_index + no_entries); ++i) 
	    for (j=0; j<3; ++j) cc_int(inddc(j, ctab[i * 3 + j]));
	flush_cmd(final_flush);
	return(1);
}
/* the aspsflags */
cc_aspsflags(no_pairs, flag_array)
int no_pairs, *flag_array;
{
int i;
	start_cmd(5, (int) AspsFlags);
	outc(sep_char);
	for (i=0; i<no_pairs; ++i) {
	    if (flag_array[2 * i] < NO_ASPS_FLAGS) {
		out_string(asps_flags[flag_array[2 * i]]);
		if (flag_array[2 * i + 1]) out_string(" BUNDLED");
		else out_string(" INDIV");
		outc(sep_char);
	    } else (void) fprintf(stderr, "illegal asp %d\n", flag_array[i]);
	}
	flush_cmd(final_flush);
	return(1);
}
/* Class 6; the Escape element */
cc_esc(id, str_ptr)
int id;
char *str_ptr;
{
	start_cmd(6, (int) Escape);
	cc_int(id);
	cc_string(str_ptr, strlen(str_ptr));
	flush_cmd(final_flush);
	return(1);
}
/* class 7; the external functions */
/* the message element */
cc_message(action, str_ptr)
enum boolean action;
char *str_ptr;
{
	start_cmd(7, (int) Message);
	switch (action) {
case on:	out_string(" Action"); break;
case off:	out_string(" NoAction"); break;
default:	fprintf(stderr, "illegal action\n"); return(2);
	}
	cc_string(str_ptr, strlen(str_ptr));
	flush_cmd(final_flush);
	return(1);
}
/* application data */
cc_apdata(id, str_ptr)
int id;
char *str_ptr;
{
	start_cmd(7, Ap_Data);
	cc_int(id);
	cc_string(str_ptr, strlen(str_ptr));
	flush_cmd(final_flush);
	return(1);
}

/* this is the routine that sets everything up */
/* the initialising routine for the postscript module */
void cc_setup(opt, dev_info, c1, c2, c3, c5, delim, mfdesc, pdesc, mfctrl,
	gprim, attr, escfun, extfun, ctrl, pargc, argv)

struct one_opt 		*opt;	/* the command line options, in only */
struct info_struct *dev_info;	/* device info to fill out, out only */
struct mf_d_struct 	*c1;	/* the class 1 elements, in only */
struct pic_d_struct 	*c2;	/* the class 2 elements, in only */
struct control_struct	*c3;	/* the class 3 elements, in only */
struct attrib_struct	*c5;	/* the class 5 elements, in only */
int (*delim[])();  		/* delimiter functions */
int (*mfdesc[])();		/* metafile descriptor functions */
int (*pdesc[])();     		/* page descriptor functions */
int (*mfctrl[])();     		/* controller functions */
int (*gprim[])();      		/* graphical primitives */
int (*attr[])();       		/* the attribute functions */
int (*escfun[])();     		/* the escape functions */
int (*extfun[])();     		/* the external functions */
int (*ctrl[])();       		/* external controller functions */
int *pargc;			/* pointer to argc */
char **argv;			/* the main argv */	
{
	/* store the command line argument pointer */
	popt = opt;
	/* store the device info pointer */
	dptr = dev_info;
	/* and the cgm data strucures */
	pc1 = c1;
	pc2 = c2;
	pc3 = c3;
	pc5 = c5;
	dev1_info = dev_info;
	/* fill out the device info structure */
	dev_info->pxl_in = 1.0;
	dev_info->ypxl_in = 1.0;
	dev_info->x_size = pc2->vdc_extent.i[2] - pc2->vdc_extent.i[0];
	dev_info->y_size = pc2->vdc_extent.i[3] - pc2->vdc_extent.i[1];
	/* get default vdc_pxl */
	vdc_pxl = 1.0;
	dev_info->x_offset = 0.0;
	dev_info->y_offset = 0.0;
	dev_info->rec_size = 80;
	dev_info->c_height	= 1;
	dev_info->c_width	= 1;
	dev_info->d_l_width	= 1;
	dev_info->d_e_width	= 1;
	dev_info->d_m_size	= 1;
	strcpy(dev_info->out_name, ".CGMC");
	dev_info->capability	= port_land | arb_rot | arb_trans | v_center
				| h_center | brk_ok | stroke_text | char_text
				| string_text | can_clip;
	dev_info->rec_size	= 80;

	/* store the CGM data structure pointers */
	pc1 = c1;
	pc3 = c3;
	pc5 = c5;

	state_level = -1;	/* just starting */
/* now fill out the function pointers */

	/* the delimiter functions */
	delim[(int) B_Mf] 	= cc_begin;
	delim[(int) E_Mf]	= cc_end;
	delim[(int) B_Pic]	= cc_bp;
	delim[(int) B_Pic_Body]	= cc_bpage;
	delim[(int) E_Pic]	= cc_epage;

	/* the Metafile Descriptor elements */
	mfdesc[(int) MfVersion] = cc_mfversion;
	mfdesc[(int) MfDescrip] = cc_mfdescrip;
	mfdesc[(int) vdcType] 	= cc_vdctype;
	mfdesc[(int) IntPrec] 	= cc_intprec;
	mfdesc[(int) RealPrec] 	= cc_realprec;
	mfdesc[(int) IndexPrec]	= cc_indexprec;
	mfdesc[(int) ColPrec] 	= cc_colprec;
	mfdesc[(int) CIndPrec] 	= cc_cindprec;
	mfdesc[(int) CVExtent] 	= cc_cvextent;
	mfdesc[(int) MaxCInd]	= cc_maxcind;
	mfdesc[(int) MfElList]	= cc_mfellist;
	mfdesc[(int) MfDefRep]	= cc_mfdefrep;
	mfdesc[(int) FontList]	= cc_flist;
	mfdesc[(int) CharList]	= cc_clist;
	mfdesc[(int) CharAnnounce]	= cc_cannounce;

	/* the picture descriptor elements */
	pdesc[(int) ScalMode]		= cc_scalmode;
	pdesc[(int) ColSelMode]		= cc_colselmode;
	pdesc[(int) LWidSpecMode]	= cc_lwsmode;
	pdesc[(int) MarkSizSpecMode]	= cc_msmode;
	pdesc[(int) EdWidSpecMode]	= cc_ewmode;
	pdesc[(int) vdcExtent]		= cc_vdcextent;
	pdesc[(int) BackCol]		= cc_backcol;

	/* the control elements */
	mfctrl[(int) vdcIntPrec]	= cc_vdcintprec;
	mfctrl[(int) vdcRPrec]		= cc_vdcrprec;
	mfctrl[(int) AuxCol]		= cc_auxcol;
	mfctrl[(int) Transp]		= cc_transp;
	mfctrl[(int) ClipRect]		= cc_cliprect;
	mfctrl[(int) ClipIndic]		= cc_clipindic;

	/* the graphical primitives */
	gprim[(int) PolyLine]	= cc_pline;
	gprim[(int) Dis_Poly]	= cc_dpline;
	gprim[(int) PolyMarker]	= cc_pmarker;
	gprim[(int) Text]	= cc_text;
	gprim[(int) Rex_Text]	= cc_rex_text;
	gprim[(int) App_Text]	= cc_app_text;
	gprim[(int) Polygon]	= cc_pgon;
	gprim[(int) Poly_Set]	= cc_pset;
	gprim[(int) Cell_Array]	= cc_carray;
	gprim[(int) Gen_D_Prim]	= cc_gdp;
	gprim[(int) Rectangle]	= cc_rectangle;
	gprim[(int) Cgm_Circle]	= cc_circle;
	gprim[(int) Circ_3]	= cc_c3;
	gprim[(int) Circ_3_Close]	= cc_c3_close;
	gprim[(int) Circ_Centre]	= cc_c_centre;
	gprim[(int) Circ_C_Close]	= cc_c_c_close;
	gprim[(int) Ellipse]	= cc_ellipse;
	gprim[(int) Ellip_Arc]	= cc_ell_arc;
	gprim[(int) El_Arc_Close] 	= cc_e_a_close;

	/* the attributes */
	attr[(int) LBIndex]		= cc_lbindex;
	attr[(int) LType]		= cc_ltype;
	attr[(int) LWidth]		= cc_lwidth;
	attr[(int) LColour]		= cc_lcolour;
	attr[(int) MBIndex]		= cc_mbindex;
	attr[(int) MType]		= cc_mtype;
	attr[(int) MSize]		= cc_msize;
	attr[(int) MColour]		= cc_mcolour;
	attr[(int) TBIndex]		= cc_tbindex;
	attr[(int) TFIndex]		= cc_tfindex;
	attr[(int) TPrec]		= cc_tprec;
	attr[(int) CExpFac]		= cc_cexpfac;
	attr[(int) CSpace]		= cc_cspace;
	attr[(int) TColour]		= cc_tcolour;
	attr[(int) CHeight]		= cc_cheight;
	attr[(int) COrient]		= cc_corient;
	attr[(int) TPath]		= cc_tpath;
	attr[(int) TAlign]		= cc_talign;
	attr[(int) CSetIndex]		= cc_csindex;
	attr[(int) AltCSetIndex]	= cc_acsindex;
	attr[(int) FillBIndex]		= cc_fbindex;
	attr[(int) IntStyle]		= cc_intstyle;
	attr[(int) FillColour]		= cc_fillcolour;
	attr[(int) HatchIndex]		= cc_hindex;
	attr[(int) PatIndex]		= cc_pindex;
	attr[(int) EdBIndex]		= cc_ebindex;
	attr[(int) EType]		= cc_etype;
	attr[(int) EdWidth]		= cc_edwidth;
	attr[(int) EdColour]		= cc_edcolour;
	attr[(int) EdVis]		= cc_edvis;
	attr[(int) FillRef]		= cc_fillref;
	attr[(int) PatTab]		= cc_pattab;
	attr[(int) PatSize]		= cc_patsize;
	attr[(int) ColTab]		= cc_coltab;
	attr[(int) AspsFlags]		= cc_aspsflags;

	/* the escape element */
	escfun[(int) Escape] 	= cc_esc;

	/* the external functions */
	extfun[(int) Message]	= cc_message;
	extfun[(int) Ap_Data]	= cc_apdata;

	return;
}

