/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*********************************************************************
*
*    file:      filml.c
*
*    Contents:  Functions calculating energy, volume, and their
*               gradients for the LINEAR SOAPFILM model.
*/

#include "include.h"
double wee_area = 0.0;

/************************************************************************
*
*  Calculates all forces on control points due to facet and
*  accumulates them at each control point.
*/

void facet_force_l(f_id)
facet_id f_id;
{
  REAL side[FACET_EDGES][MAXCOORD];
  REAL normal[MAXCOORD];
  REAL wulff[MAXCOORD]; /* area vector covector for energy */
  REAL temp[MAXCOORD];
  int i,j;
  REAL area;
  facetedge_id fe_id;
  REAL *x[FACET_VERTS];
  REAL z[FACET_VERTS];  /* for gravitational forces */
  REAL zz;    /* average z*z              */
  REAL gdensity;  /* net density difference of bodies across facet */
  body_id b_id;
  MAP surfenmap = get_f_surfen_map(f_id);
  REAL integrand[MAXCOORD];  /* surface energy integrand */
  REAL int_deriv[FACET_VERTS][MAXCOORD][MAXCOORD];  /* integrand derivatives */
  REAL density = get_facet_density(f_id);
  vertex_id v_id[FACET_VERTS];
  edge_id   e_id[FACET_EDGES];
  REAL unwrap_x[FACET_VERTS][MAXCOORD];
  WRAPTYPE wraps[FACET_VERTS];
  REAL forces[FACET_VERTS][MAXCOORD];  /* total forces from this facet */
  REAL *forceptr[FACET_VERTS];   /* pointers to forces */

  memset((char*)forces,0,sizeof(forces));  /* set to 0 */
  int_val = get_original(f_id);  /* for eval  of file parameters */
  /* get side vectors */
  generate_facet_fe_init();
  for ( i = 0 ; i < FACET_EDGES ; i++ )
    { 
      generate_facet_fe(f_id,&fe_id);
      e_id[i] = get_fe_edge(fe_id);
/*    get_edge_side(e_id[i],side[i]);  */
      v_id[i] = get_edge_tailv(e_id[i]);
/*    x[i] = get_coord(v_id[i]);  */
      x[i] = unwrap_x[i];
      forceptr[i] = forces[i];
    }
  get_facet_verts(f_id,x,wraps);  /* verts in tail order */
  for ( i = 0 ; i < FACET_EDGES ; i++ )
    { int ii = (i+1)%FACET_EDGES;
      for ( j = 0 ; j < web.sdim ; j++ )
        side[i][j] = x[ii][j] - x[i][j];
      z[i] = x[i][2];
    } 
  if ( web.metric_flag )
    { simplex_force_metric(v_id,x,density,forceptr);
      goto cumforces;  /* assume no gravity */
    }

  if ( square_curvature_flag && !kusner_flag ) sqcurve_force(v_id,e_id,side);

  /* calculate normal */ 
  cross_prod(side[0],side[1],normal);
  area = sqrt(dot(normal,normal,web.sdim));

  /* an error check, and accommodation for possibly deliberately
     degenerate triangles on boundary */
  if ( area <= wee_area )
    { facetedge_id ffe;
        
      ffe = fe_id;
      outstring("WARNING! Zero area triangle!\n");
      outstring("Facet-edges and sides: \n");
      for ( i = 0 ; i < FACET_EDGES ; i++, ffe = get_next_edge(ffe) )
        { 
	  sprintf(msg," %8lX   %18.15f %18.15f %18.15f\n",ffe,
                                        side[i][0],side[i][1],side[i][2]);
	  outstring(msg);
	}

      prompt("Hit RETURN to continue.",msg);
    }

  /* get energy covector */
  if ( web.wulff_flag ) (*get_wulff)(normal,wulff);
  else
    for ( j = 0 ; j < web.sdim ; j++ ) 
       wulff[j] = normal[j]/area;  /* normalize */
  area /= 2;   /* half of cross product magnitude */
  set_facet_area(f_id,area);
  if ( get_fattr(f_id) & DENSITY )
    for ( i = 0 ; i < web.sdim ; i++ )
       wulff[i] *= density;
  
  /* accumulate star area around each vertex and edge */
    {
      generate_facet_fe_init();
      while ( generate_facet_fe(f_id,&fe_id) )
       {
	 edge_id e_id = get_fe_edge(fe_id);
         vertex_id v_id = get_edge_headv(e_id);
         add_vertex_star(v_id,area);
	 add_edge_star(e_id,area);
        }
    }

  /* for gravity forces */
  if ( web.gravflag )
    {
      zz = (z[0]*z[0]+z[1]*z[1]+z[2]*z[2]+z[0]*z[1]+z[1]*z[2]+z[0]*z[2])/6;
      b_id = get_facet_body(f_id);
      gdensity = 0.0;
      if ( valid_id(b_id) )
        gdensity += get_body_density(b_id);
      b_id = get_facet_body(facet_inverse(f_id));
      if ( valid_id(b_id) )
        gdensity -= get_body_density(b_id);
    }

  /* surface energy integrals */
  if ( surfenmap )
    {
      int m,n,j,k,bit;
      struct surf_energy *surfen;
      REAL pt[MAXCOORD];

       /* find integrands and derivatives at center of mass */       
       memset((char *)integrand,0,sizeof(integrand));
       memset((char *)int_deriv,0,sizeof(int_deriv));
       for ( n = 0, bit = 1 ; n < SURFENMAX ; bit <<= 1, n++ )
	{ if ( !(surfenmap&bit) ) continue;
	  surfen = get_surfen(n);
          for ( m = 0 ; m < gauss_num ; m++ )
           {
             for ( i = 0 ; i < web.sdim ; i++ )
               { pt[i] = 0.0;
                 for ( j = 0 ; j < FACET_VERTS ; j++ )
                   pt[i] += gauss2Dpt[m][j]*x[j][i];
                }

             for ( i = 0 ; i < web.sdim ; i++ )
               { REAL val,derivs[MAXCOORD];
                 eval_all(surfen->envect[i],pt,web.sdim,&val,derivs);
                 integrand[i] += gauss2Dwt[m]*val;
                 for ( k = 0 ; k < FACET_VERTS ; k++ )
                   for ( j = 0 ; j < web.sdim ; j++ )
  	             int_deriv[k][j][i] += 
                        gauss2Dpt[m][k]*gauss2Dwt[m]*derivs[j];
	       }
           }
	}
    }

  /* force on each vertex */
  for ( i = 0 ; i < FACET_VERTS ; i++ )  /* vertex loop */
   { int k;

     j = (i+1)%FACET_EDGES;  /* opposite side */
     cross_prod(side[j],wulff,temp);
     for ( k = 0 ; k < web.sdim ; k++ ) forces[i][k] += temp[k]/2;

     /* gravity forces, negative of gravity energy gradient */
     if ( web.gravflag )
      {
        forces[i][0] += web.grav_const*gdensity*side[j][1]*zz/4;
        forces[i][1] -= web.grav_const*gdensity*side[j][0]*zz/4;
        forces[i][2] -= web.grav_const*gdensity*normal[2]*(z[i]+z[0]+z[1]+z[2])/24;
      }

     /* surface energy integrand */
     if ( surfenmap )
       {
	 cross_prod(side[j],integrand,temp);
	 for ( k = 0 ; k < web.sdim ; k++ )
	   forces[i][k] -= dot(int_deriv[i][k],normal,web.sdim)/2 - temp[k]/2;
       }
   }

cumforces:
  /* add to totals, unwrapping if necessary */
  for ( i = 0 ; i < FACET_VERTS ; i++ )  /* vertex loop */
   { REAL *force; 
     REAL wforce[MAXCOORD];  /* unwrapped forces */

     force = get_force(v_id[i]);
     if ( web.symmetry_flag )
       { (*sym_form_pullback)(get_coord(v_id[i]),wforce,forces[i],wraps[i]);
	 for ( j = 0 ; j < web.sdim ; j++ )
	   force[j] += wforce[j];
       }
     else
	 for ( j = 0 ; j < web.sdim ; j++ )
	   force[j] += forces[i][j];
   }

}


/*********************************************************************
*
*  Function: facet_energy_l()
*
*  Purpose:  Calculates energy due to facet for LINEAR SOAPFILM.
*            Also does facet quantity integrands.
*/

void facet_energy_l(f_id)
facet_id f_id;
{
  int i,j;
  REAL side[FACET_EDGES][MAXCOORD];
  REAL normal[MAXCOORD];
  REAL wulff[MAXCOORD];   /* energy covector to area normal vector */
  REAL energy;
  body_id b_id;
  REAL *x[FACET_VERTS];
  REAL unwrap_x[FACET_VERTS][MAXCOORD];
  REAL z[FACET_VERTS];  /* for calculating average z*z */
  REAL zz;       /* for average z*z, for gravity */
  REAL u;     /* gravitaional energy */
  facetedge_id fe_id;
  vertex_id v_id[FACET_VERTS];

  int_val = get_original(f_id);  /* for eval  of file parameters */
  /* get side vectors */
  generate_facet_fe_init();
  for ( i = 0 ; i < FACET_EDGES ; i++ )
    { 
      generate_facet_fe(f_id,&fe_id);
/*      get_fe_side(fe_id,side[i]);  */
      v_id[i] = get_fe_tailv(fe_id);
/*      x[i] = get_coord(v_id[i]);  */
x[i] = unwrap_x[i];
    }
  get_facet_verts(f_id,x,NULL);  /* in tail order */
  for ( i = 0 ; i < FACET_EDGES ; i++ )
    { int ii = (i+1)%FACET_EDGES;
      for ( j = 0 ; j < web.sdim ; j++ )
        side[i][j] = x[ii][j] - x[i][j];
      z[i] = x[i][2];
    } 
    

  if ( web.metric_flag )
    { energy = simplex_energy_metric(v_id,x);
      set_facet_area(f_id,energy);
      goto skip_from_metric;
    }

  /* calculate normal */ 
  cross_prod(side[0],side[1],normal);
    
  /* calculate surface tension energy */
  energy = sqrt(dot(normal,normal,web.sdim))/2;
  set_facet_area(f_id,energy);
  if ( web.wulff_flag ) 
    { (*get_wulff)(normal,wulff);
      energy = dot(wulff,normal,web.sdim)/2;
    }

  /* do square curvature if wanted */
  if ( square_curvature_flag  && !kusner_flag )
    sqcurve_energy(v_id,side);


skip_from_metric:

  web.total_area   += energy;
  if ( get_fattr(f_id) & DENSITY )
       energy *= get_facet_density(f_id);
     
  /* add gravitational energy, vector potential z*z/2*k  */
  if ( web.gravflag )
   {
     zz = (z[0]*z[0]+z[1]*z[1]+z[2]*z[2]+z[0]*z[1]+z[1]*z[2]+z[0]*z[2])/6;
     u = zz*normal[2]/2/2;  /* half for area, half from potential */
     b_id = get_facet_body(f_id);
     if ( valid_id(b_id) )
        energy += u*get_body_density(b_id)*web.grav_const;
     b_id = get_facet_body(facet_inverse(f_id));
     if ( valid_id(b_id) )
        energy -= u*get_body_density(b_id)*web.grav_const;
   }

  /* add surface integrands */
  if ( get_fattr(f_id) & SURF_ENERGY )
    {
      MAP surfenmap;
      struct surf_energy *surfen;
      REAL value = 0.0;  /* total integral */
      int m,n,j;
      REAL pt[MAXCOORD];

      /* find integrands at center of mass and vertices */       
      /* weighted to make quadratic integrands exact    */
      surfenmap = get_f_surfen_map(f_id);
      for ( n = 0 ; n < web.surfen_count ; surfenmap >>= 1, n++ )
        { if ( !(surfenmap&1) ) continue;
          surfen = get_surfen(n);
          for ( m = 0 ; m < gauss_num ; m++ )
            {
              for ( i = 0 ; i < web.sdim ; i++ )
                { pt[i] = 0.0;
                  for ( j = 0 ; j < FACET_VERTS ; j++ )
                      pt[i] += gauss2Dpt[m][j]*x[j][i];
                }
              for ( i = 0 ; i < web.sdim ; i++ )
                 value += gauss2Dwt[m]*normal[i]*eval(surfen->envect[i],pt);
	    }

        }
      energy += value/2;  /* 2 for triangle factor */
    }

  web.total_energy += energy;

  /* add quantity integrands */
  if ( get_fattr(f_id) & SURF_QUANTITY )
    {
      MAP quantmap;
      struct quantity *quan;
      REAL value = 0.0;  /* total integral */
      int m,n,j;
      REAL pt[MAXCOORD];

      quantmap = get_f_quant_map(f_id);
      for ( n = 0 ; n < web.quantity_count ; quantmap >>= 1, n++ )
        { if ( !(quantmap&1) ) continue;
          quan = get_quant(n);
          for ( m = 0 ; m < gauss_num ; m++ )
            {
              for ( i = 0 ; i < web.sdim ; i++ )
                { pt[i] = 0.0;
                  for ( j = 0 ; j < FACET_VERTS ; j++ )
                      pt[i] += gauss2Dpt[m][j]*x[j][i];
                }
              for ( i = 0 ; i < web.sdim ; i++ )
                 value += gauss2Dwt[m]*normal[i]*eval(quan->quanvect[i],pt);
	    }

          quan->value += value/2;  /* 2 for triangle factor */
        }
    }

}
 

/**********************************************************************
*
*  Find triangle's contribution to volumes of neighboring bodies.
*  Volumes with respect to origin are calculated for each
*         face, and then oriented contributions added for each body. 
*/

void facet_volume_l(f_id)
facet_id f_id;
{ 
  int i;
  REAL side[FACET_EDGES][MAXCOORD];
  REAL normal[MAXCOORD];
  body_id b_id0,b_id1;
  facetedge_id fe_id;
  REAL vol;
  REAL z = 0.0;
    
  int_val = get_original(f_id);  /* for eval  of file parameters */
  b_id0 = get_facet_body(f_id);
  b_id1 = get_facet_body(facet_inverse(f_id));
  if ( !valid_id(b_id0) && !valid_id(b_id1) ) return;
    
  if ( web.symmetric_content )
    { vertex_id v1,v2,v3;
      facetedge_id next_fe;
      fe_id = get_facet_fe(f_id);
      next_fe = get_next_edge(fe_id);
      v1 = get_fe_tailv(fe_id);
      v2 = get_fe_headv(fe_id);
      v3 = get_fe_headv(next_fe);
      vol = triple_prod(get_coord(v1), get_coord(v2), get_coord(v3))/6;
    }
  else
    {
      /* get side vectors */
      generate_facet_fe_init();
      for ( i = 0 ; i < FACET_EDGES ; i++ )
        { generate_facet_fe(f_id,&fe_id);
          get_fe_side(fe_id,side[i]);
          z += get_coord(get_fe_headv(fe_id))[2];
        }
    
      /* calculate normal */ 
      cross_prod(side[0],side[1],normal);
    
      /* calculate volume, integrating z dx dy */
      vol = z*normal[2]/6;   /* factor of 2 for triangle, 3 for average z */
    }
    
  /* add to body volumes */
  if ( valid_id(b_id0) ) 
    set_body_volume(b_id0,get_body_volume(b_id0) + vol);
  if ( valid_id(b_id1) ) 
    set_body_volume(b_id1,get_body_volume(b_id1) - vol);
}

/******************************************************************
*   
*  Function: film_grad_l()
*
*  Purpose: Compute volume gradients for vertices on facets.
*           Also fixed quantity gradients.
*/

void film_grad_l()
{
  body_id bi_id;  /* identifier for body i */
  body_id bj_id;  /* identifier for body j */
  facetedge_id fe;
  vertex_id v_id;
  facet_id f_id;
  facetedge_id fe_id;
  int i,k;
  volgrad *vgptr;
  REAL z;
  REAL side[FACET_EDGES][MAXCOORD];
  REAL normal[MAXCOORD];
  MAP quantmap;
  REAL integrand[MAXCOORD];  /* quantity integrand */
  REAL int_deriv[FACET_VERTS][MAXCOORD][MAXCOORD];  /* integrand derivatives */
  REAL *x[FACET_VERTS];  /* coordinates of vertices */
  REAL unwrap_x[FACET_VERTS][MAXCOORD];

  FOR_ALL_FACETS(f_id)
   { 
     int_val = get_original(f_id);  /* for eval  of file parameters */
     bi_id = get_facet_body(f_id);
     bj_id = get_facet_body(facet_inverse(f_id));

     /* get side vectors */
     generate_facet_fe_init();
     for ( i = 0 ; i < FACET_EDGES ; i++ )
       { 
         generate_facet_fe(f_id,&fe_id);
         get_fe_side(fe_id,side[i]);
         x[i] = get_coord(get_fe_headv(fe_id));
       }

     if ( web.symmetry_flag )
	{
          for ( i = 0 ; i < web.sdim ; i++ )
	  unwrap_x[0][i] = x[0][i];
	  for ( i = 0 ; i < FACET_VERTS ; i++ ) x[i] = unwrap_x[i];
	  for ( i = 0 ; i < web.sdim ; i++ )
	     { x[1][i] = x[0][i] + side[1][i];
	       x[2][i] = x[1][i] +  side[2][i];
	     }
	}



     if ( web.symmetric_content )
       {
          /* do each of the three vertices */
          fe = get_facet_fe(f_id);
          for ( k = 0 ; k < FACET_VERTS ; k++, fe = get_next_edge(fe) )
            {
              fe_id = get_next_edge(fe);
              v_id = get_fe_headv(fe_id);
              if ( get_vattr(v_id) & FIXED ) continue;

	      cross_prod(get_coord(get_fe_tailv(fe)),
	          get_coord(get_fe_headv(fe)),normal);

              if ( valid_id(bi_id) && (get_battr(bi_id) & (PRESSURE|FIXEDVOL)) )
               { 
                  vgptr = get_bv_new_vgrad(bi_id,v_id);
		  for ( i = 0 ; i < web.sdim ; i++ )
                    vgptr->grad[i] +=  normal[i]/6.0;
                }

              if ( valid_id(bj_id) && (get_battr(bj_id) & (PRESSURE|FIXEDVOL)) )
                { 
                  vgptr = get_bv_new_vgrad(bj_id,v_id);
		  for ( i = 0 ; i < web.sdim ; i++ )
                    vgptr->grad[i] -=  normal[i]/6.0;
                }
	    }
       }
     else
       {  /* content from integrating z dx dy */
    
         /* get centroid z (divide by 3 later) */
         for ( i = 0,z = 0.0; i < FACET_EDGES ; i++ ) z += x[i][2];
              
         /* calculate normal */ 
         cross_prod(side[0],side[1],normal);

          /* now do each of the three vertices */
    
          fe = get_facet_fe(f_id);
          for ( k = 0 ; k < FACET_VERTS ; k++, fe = get_next_edge(fe) )
            {
              fe_id = get_next_edge(fe);
              v_id = get_fe_headv(fe_id);
              if ( get_vattr(v_id) & FIXED ) continue;

              if ( valid_id(bi_id) && (get_battr(bi_id) & (PRESSURE|FIXEDVOL)) )
                { 
                  vgptr = get_bv_new_vgrad(bi_id,v_id);
                  vgptr->grad[0] += -side[k][1]*z/6.0;
                  vgptr->grad[1] +=  side[k][0]*z/6.0;
                  vgptr->grad[2] +=  normal[2]/6.0;
                }

              if ( valid_id(bj_id) && (get_battr(bj_id) & (PRESSURE|FIXEDVOL)) )
                { 
                  vgptr = get_bv_new_vgrad(bj_id,v_id);
                  vgptr->grad[0] -= -side[k][1]*z/6.0;
                  vgptr->grad[1] -=  side[k][0]*z/6.0;
                  vgptr->grad[2] -=  normal[2]/6.0;
                }
            }
        }  

      /* quantity gradients */
      /* calculate normal */ 
      cross_prod(side[0],side[1],normal);
      quantmap = get_f_quant_map(f_id);
      if ( quantmap )
       {
         int m,n,j,k,bit;
         struct quantity *quan;
         REAL pt[MAXCOORD];

         memset((char *)integrand,0,sizeof(integrand));
         memset((char *)int_deriv,0,sizeof(int_deriv));
         for ( n = 0, bit = 1 ; n < web.quantity_count ; bit <<= 1, n++ )
          { if ( !(quantmap&bit) ) continue;
	    quan = get_quant(n);
            if ( !(quan->attr & QFIXED) ) continue;
            for ( m = 0 ; m < gauss_num ; m++ )
             {
               for ( i = 0 ; i < web.sdim ; i++ )
                 { pt[i] = 0.0;
                   for ( j = 0 ; j < FACET_VERTS ; j++ )
                     pt[i] += gauss2Dpt[m][j]*x[j][i];
                 }

               for ( i = 0 ; i < web.sdim ; i++ )
                 { REAL val,derivs[MAXCOORD];
                   eval_all(quan->quanvect[i],pt,web.sdim,&val,derivs);
                   integrand[i] += gauss2Dwt[m]*val;
                   for ( k = 0 ; k < FACET_VERTS ; k++ )
                     for ( j = 0 ; j < web.sdim ; j++ )
  	               int_deriv[k][j][i] += 
                          gauss2Dpt[m][k]*gauss2Dwt[m]*derivs[j];
	         }
             }

            /* now add onto appropriate vertices */
            generate_facet_fe_init();
            for ( i = 0 ; i < FACET_VERTS ; i++ )  /* vertex loop */
             { 
               vertex_id v_id;
               int k;
	       REAL temp[MAXCOORD];

               generate_facet_fe(f_id,&fe_id);
               v_id = get_fe_tailv(fe_id);
               if ( get_vattr(v_id) & FIXED ) continue;
               vgptr = get_bv_new_vgrad((body_id)n,v_id);
               j = (i+1)%FACET_EDGES;  /* opposite side */
               cross_prod(side[j],integrand,temp);
	       for ( k = 0 ; k < web.sdim ; k++ )
	        vgptr->grad[k] += 
		 dot(int_deriv[i][k],normal,web.sdim)/2 - temp[k]/2;
             } /* end vertices */
          } /* end quantity */
       } /* end quantities */
   } /* end facet loop */
}

/******************************************************
*
*  Function: film_bdry_grad()
*
*  Purpose: temporary stub.
*/

void film_bdry_grad()
{
}

/*******************************************************************
*
*  Function: film_constr_grad()
*
*  Purpose: Add cell volume gradients due to constraint integrals.
*           And quantity integrals.
*/

void film_constr_grad()
{
  edge_id e_id;
  REAL side[MAXCOORD];
  REAL tgrad[MAXCOORD];  
  REAL hgrad[MAXCOORD];  
  REAL grad;  
  int i,j,k,m,sign,bodysign;
  REAL green[MAXCOORD];
  REAL green_deriv[MAXCOORD][MAXCOORD];
  REAL midpt[MAXCOORD];
  REAL *tcoord,*hcoord;
  struct constraint *constr;
  vertex_id headv,tailv;
  struct volgrad *hvgptr,*tvgptr;

  FOR_ALL_EDGES(e_id)
    {
      struct volgrad *vgptri;
      facet_id f_id;
      ATTR attr = get_eattr(e_id);
      MAP conmap = get_e_constraint_map(e_id);

      if ( attr & FIXED ) continue;
      if ( !(attr & CONSTRAINT) ) continue;

      headv = get_edge_headv(e_id);
      tailv = get_edge_tailv(e_id);
      tcoord = get_coord(tailv);
      hcoord = get_coord(headv);
      int_val = get_original(e_id);  /* for eval  of file parameters */
      for ( j = 0 ; j < web.sdim ; j++ )
          side[j] = hcoord[j] - tcoord[j];
      if ( !(attr & BDRY_CONTENT) ) goto do_quants;
      
      if ( web.modeltype == QUADRATIC ) 
        { constr_vol_grad_q(e_id);
          continue;
        }
      f_id = get_fe_facet(get_edge_fe(e_id));
      if ( attr & NEGBOUNDARY ) sign = -1;
      else sign = 1;
      if ( inverted(e_id) ) sign = -sign;

      for ( i = 0 ; i < web.sdim ; i++ )
        tgrad[i] = hgrad[i] = 0.0;

      for ( j = 0 ; j < web.concount ; j++,conmap>>=1 )
       {
         if ( !(conmap & 1) ) continue;
         constr = get_constraint(j);
         if ( constr->compcount != web.sdim ) continue;
         for ( m = 0 ; m <  abs(web.gauss_order) ; m++ )
          {
           for ( i = 0 ; i < web.sdim ; i++ )
             midpt[i] = gauss1Dpt[m]*hcoord[i] + (1 - gauss1Dpt[m])*tcoord[i];
           for ( i = 0 ; i < web.sdim ; i++ )
             eval_all(constr->convect[i],midpt,web.sdim,&green[i],
                                                       green_deriv[i]);
           for ( i = 0 ; i < web.sdim ; i++ )
             { 
               for ( k = 0,grad = 0.0 ; k < web.sdim ; k++ )
                 grad += side[k]*green_deriv[k][i];
               tgrad[i] += sign*gauss1Dwt[m]*((1-gauss1Dpt[m])*grad - green[i]);
               hgrad[i] += sign*gauss1Dwt[m]*(gauss1Dpt[m]*grad + green[i]);
             }
          }
       }

      /* assuming constraint edge is on just one facet */
      vgptri = get_vertex_vgrad(tailv);
      for  ( ; vgptri ; vgptri = vgptri->chain )
        {
          if ( !valid_id(vgptri->b_id) ) continue; /* skip quantities */
          if ( !equal_id(get_facet_body(f_id),vgptri->b_id) ) 
            if ( !equal_id(get_facet_body(inverse_id(f_id)),vgptri->b_id) )
              { outstring("Can't find vertex body.\n");
              }
            else  bodysign = -sign;
          else bodysign = sign;
          for ( i = 0 ; i < web.sdim ; i++ )
            vgptri->grad[i] += bodysign*tgrad[i];  
        }

      vgptri = get_vertex_vgrad(headv);
      for  ( ; vgptri ; vgptri = vgptri->chain )
        {
          if ( !valid_id(vgptri->b_id) ) continue; /* skip quantities */
          if ( !equal_id(get_facet_body(f_id),vgptri->b_id) ) 
            if ( !equal_id(get_facet_body(inverse_id(f_id)),vgptri->b_id) )
              { outstring("Can't find vertex body.\n");
              }
            else  bodysign = -sign;
          else bodysign = sign;
          for ( i = 0 ; i < web.sdim ; i++ )
            vgptri->grad[i] += bodysign*hgrad[i];  
        }

do_quants:
      /* quantity integrands */
      if ( web.modeltype == QUADRATIC )
        { error("Cannot do quantities in quadratic model.\n",RECOVERABLE );
          return;
        }
      conmap = get_e_constraint_map(e_id);
      for ( j = 0 ; j < web.concount ; j++,conmap >>= 1 )
       {
	MAP quantmap;
	int n;
        if ( !(conmap & 1) ) continue;
        constr = get_constraint(j);
        if ( constr->compcount != web.sdim ) continue;
        quantmap = constr->quantity_map;
        if ( !quantmap ) continue;
        for ( m = 0 ; m < abs(web.gauss_order) ; m++ )
         {
           for ( i = 0 ; i < web.sdim ; i++ )
             midpt[i] = gauss1Dpt[m]*hcoord[i] + (1 - gauss1Dpt[m])*tcoord[i];
           quantmap = constr->quantity_map; /* refresh */	
           for ( n = 0 ; quantmap ; n++,quantmap>>=1 )
            { struct quantity *quan = get_quant(n);
	      if ( !(quan->attr & QFIXED ) ) continue;
              for ( i = 0 ; i < web.sdim ; i++ )
               eval_all(constr->quanvect[n][i],midpt,web.sdim,&green[i],
						       green_deriv[i]);
              tvgptr = get_bv_new_vgrad((body_id)n,tailv);
              hvgptr = get_bv_new_vgrad((body_id)n,headv);
              for ( i = 0 ; i < web.sdim ; i++ )
               { for ( grad = 0.0, j = 0 ; j < web.sdim ; j++ )
                   grad += side[j]*green_deriv[j][i];
                 tvgptr->grad[i]+=gauss1Dwt[m]*((1-gauss1Dpt[m])*grad-green[i]);
                 hvgptr->grad[i] += gauss1Dwt[m]*(gauss1Dpt[m]*grad+green[i]);
               }
            }
         }
       }



    }
}
