#!/bin/sh

# Generate the list of rules for the single-binary option based on all the other
# binaries found in src/local.mk.
#
# We need to duplicate the specific rules to build each program into a new
# static library target. We can't reuse the existing target since we need to
# create a .a file instead of linking the program. We can't do this at
# ./configure since the file names need to be available when automake runs
# to let it generate all the required rules in Makefile.in.  The configure
# step will select which ones will be used to build, but they need to be
# generated beforehand.
#
# Instead of maintaining a duplicated list of rules, we generate the
# single-binary required rules based on the normal configuration found on
# src/local.mk with this script.

if test "x$1" = "x"; then
  echo "Usage: $0 path/to/src/local.mk" >&2
  exit 1
fi

set -e

LOCAL_MK=$1
GEN_LISTS_OF_PROGRAMS="`dirname "$0"`/gen-lists-of-programs.sh"

ALL_PROGRAMS=$($GEN_LISTS_OF_PROGRAMS --list-progs \
    | grep -v -F -e coreutils -e libstdbuf.so \
    | tr '[' '_')

# Compute default SOURCES. automake will assume the source file for the
# src_${cmd} target to be src/${cmd}.c, but we will add rules to generate
# the lib src_libsinglebin_${cmd}_a which won't match the autogenerated source
# file. This loop will initialize the default source file and will be reset
# later if needed.
for cmd in $ALL_PROGRAMS; do
  eval "src_${cmd}_SOURCES=src/${cmd}.c"
done

# Load actual values from src/local.mk. This will read all the variables from
# the local.mk matching the src_${cmd}_... case.
while read l; do
  if echo "$l" | grep -E '^src_\w+ +\+?=' > /dev/null; then
    var=$(echo $l | cut -f 1 -d ' ')
    value=$(echo $l | cut -f 2- -d =)
    if [ "$value" != " \$(LDADD)" ]; then
      oldvalue=""
      if echo $l | grep -F '+=' >/dev/null; then
        eval "oldvalue=\${$var}"
      fi
      value=$(echo "$value" | sed "s/'/'\"'\"'/g")
      eval "$var='$oldvalue "$value"'"
    fi
  fi
done < $LOCAL_MK

me=`echo "$0" | sed 's,.*/,,'`
echo "## Automatically generated by $me.  DO NOT EDIT BY HAND!"

# Override the sources for dir and vdir. We use a smaller version of dir and
# vdir that relies on the ls main.
src_dir_SOURCES="src/coreutils-dir.c"
src_dir_LDADD="$src_dir_LDADD src/libsinglebin_ls.a"
echo src_libsinglebin_dir_a_DEPENDENCIES = src/libsinglebin_ls.a
src_vdir_SOURCES="src/coreutils-vdir.c"
src_vdir_LDADD="$src_vdir_LDADD src/libsinglebin_ls.a"
echo src_libsinglebin_vdir_a_DEPENDENCIES = src/libsinglebin_ls.a

# Override the sources for arch likewise, using the main from uname.
src_arch_SOURCES="src/coreutils-arch.c"
src_arch_LDADD="$src_arch_LDADD src/libsinglebin_uname.a"
echo src_libsinglebin_arch_a_DEPENDENCIES = src/libsinglebin_uname.a

for cmd in $ALL_PROGRAMS; do
  echo "# Command $cmd"
  echo noinst_LIBRARIES += src/libsinglebin_${cmd}.a
  base="src_libsinglebin_${cmd}_a"
  # SOURCES
  var=src_${cmd}_SOURCES
  eval "value=\$$var"
  echo "${base}_SOURCES = $value"

  # LDADD
  var=src_${cmd}_LDADD
  eval "value=\$$var"
  if [ "x$value" != "x" ]; then
    echo "${base}_ldadd = $value"
  fi

  # CFLAGS
  # Hack any other program defining a main() replacing its main by
  # _single_binary_main_$PROGRAM_NAME.
  echo "${base}_CFLAGS = \"-Dmain=_single_binary_main_${cmd}(int, char**) " \
       "ATTRIBUTE_NORETURN; int _single_binary_main_${cmd}\" " \
       "-Dusage=_usage_${cmd} \$(src_coreutils_CFLAGS)"
  var=src_${cmd}_CFLAGS
  eval "value=\$$var"
  if [ "x$value" != "x" ]; then
    echo "${base}_CFLAGS += $value"
  fi

  # CPPFLAGS
  var=src_${cmd}_CPPFLAGS
  eval "value=\$$var"
  if [ "x$value" != "x" ]; then
    echo "${base}_CPPFLAGS = $value"
  fi
done

exit 0
