/*
 *  linux/arch/ppc/kernel/apus_setup.c
 *
 *  Copyright (C) 1998  Jesper Skov
 *
 *  Basically what is needed to replace functionality found in
 *  arch/m68k allowing Amiga drivers to work under APUS.
 *  Bits of code and/or ideas from arch/m68k and arch/ppc files.
 */

#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/kd.h>
#include <linux/init.h>
#include <linux/hdreg.h>

/* Get the IDE stuff from the 68k file */
#define ide_init_hwif_ports m68k_ide_init_hwif_ports
#define ide_default_irq m68k_ide_default_irq
#define ide_default_io_base m68k_ide_default_io_base
#define ide_check_region m68k_ide_check_region
#define ide_request_region m68k_ide_request_region
#define ide_release_region m68k_ide_release_region
#define ide_fix_driveid m68k_ide_fix_driveid
#include <asm-m68k/ide.h>
#undef ide_init_hwif_ports
#define ide_default_irq
#define ide_default_io_base
#define ide_check_region
#define ide_request_region
#define ide_release_region
#define ide_fix_driveid


#include <asm/setup.h>
#include <asm/amigahw.h>
#include <asm/amigappc.h>
#include <asm/pgtable.h>
#include <asm/io.h>
#include <asm/machdep.h>
#include <asm/ide.h>

#include <asm/time.h>
#include "local_irq.h"

unsigned long apus_get_rtc_time(void);
int apus_set_rtc_time(unsigned long nowtime);

/* APUS defs */
extern int parse_bootinfo(const struct bi_record *);
extern char _end[];
#ifdef CONFIG_APUS
struct mem_info ramdisk;
unsigned long isa_io_base;
unsigned long isa_mem_base;
unsigned long pci_dram_offset;
#endif
/* END APUS defs */

unsigned long m68k_machtype;
char debug_device[6] = "";

void (*mach_sched_init) (void (*handler)(int, void *, struct pt_regs *)) __initdata;
/* machine dependent keyboard functions */
int (*mach_keyb_init) (void) __initdata;
int (*mach_kbdrate) (struct kbd_repeat *) = NULL;
void (*mach_kbd_leds) (unsigned int) = NULL;
/* machine dependent irq functions */
void (*mach_init_IRQ) (void) __initdata;
void (*(*mach_default_handler)[]) (int, void *, struct pt_regs *) = NULL;
void (*mach_get_model) (char *model) = NULL;
int (*mach_get_hardware_list) (char *buffer) = NULL;
int (*mach_get_irq_list) (char *) = NULL;
void (*mach_process_int) (int, struct pt_regs *) = NULL;
/* machine dependent timer functions */
unsigned long (*mach_gettimeoffset) (void);
void (*mach_gettod) (int*, int*, int*, int*, int*, int*);
int (*mach_hwclk) (int, struct hwclk_time*) = NULL;
int (*mach_set_clock_mmss) (unsigned long) = NULL;
void (*mach_reset)( void );
long mach_max_dma_address = 0x00ffffff; /* default set to the lower 16MB */
#if defined(CONFIG_AMIGA_FLOPPY) || defined(CONFIG_ATARI_FLOPPY)
void (*mach_floppy_setup) (char *, int *) __initdata = NULL;
void (*mach_floppy_eject) (void) = NULL;
#endif
#ifdef CONFIG_HEARTBEAT
void (*mach_heartbeat) (int) = NULL;
#endif

extern unsigned long amiga_model;
extern unsigned decrementer_count;/* count value for 1e6/HZ microseconds */
extern unsigned count_period_num; /* 1 decrementer count equals */
extern unsigned count_period_den; /* count_period_num / count_period_den us */

extern struct mem_info memory[NUM_MEMINFO];/* memory description */

extern void amiga_floppy_setup(char *, int *);
extern void config_amiga(void);

static int __60nsram = 0;

/*********************************************************** SETUP */
/* From arch/m68k/kernel/setup.c. */
__initfunc(void apus_setup_arch(unsigned long * memory_start_p,
				unsigned long * memory_end_p))
{
	extern char cmd_line[];
	int i;
	char *p, *q;

	m68k_machtype = MACH_AMIGA;

	/* Parse the command line for arch-specific options.
	 * For the m68k, this is currently only "debug=xxx" to enable printing
	 * certain kernel messages to some machine-specific device.  */
	for( p = cmd_line; p && *p; ) {
	    i = 0;
	    if (!strncmp( p, "debug=", 6 )) {
		    strncpy( debug_device, p+6, sizeof(debug_device)-1 );
		    debug_device[sizeof(debug_device)-1] = 0;
		    if ((q = strchr( debug_device, ' ' ))) *q = 0;
		    i = 1;
	    } else if (!strncmp( p, "60nsram", 7 )) {
		    APUS_WRITE (APUS_REG_WAITSTATE, 
				REGWAITSTATE_SETRESET
				|REGWAITSTATE_PPCR
				|REGWAITSTATE_PPCW);
		    __60nsram = 1;
		    i = 1;
	    }

	    if (i) {
		/* option processed, delete it */
		if ((q = strchr( p, ' ' )))
		    strcpy( p, q+1 );
		else
		    *p = 0;
	    } else {
		if ((p = strchr( p, ' ' ))) ++p;
	    }
	}

	config_amiga();
}


void get_current_tb(unsigned long long *time)
{
	__asm __volatile ("1:mftbu 4      \n\t"
			  "  mftb  5      \n\t"
			  "  mftbu 6      \n\t"
			  "  cmpw  4,6    \n\t"
			  "  bne   1b     \n\t"
			  "  stw   4,0(%0)\n\t"
			  "  stw   5,4(%0)\n\t"
			  : 
			  : "r" (time)
			  : "r4", "r5", "r6");
}


void apus_calibrate_decr(void)
{
	int freq, divisor;

	/* This algorithm for determining the bus speed was
           contributed by Ralph Schmidt. */
	unsigned long long start, stop;
	int bus_speed;

	{
		unsigned long loop = amiga_eclock / 10;

		get_current_tb (&start);
		while (loop--) {
			unsigned char tmp;

			tmp = ciaa.pra;
		}
		get_current_tb (&stop);
	}

	bus_speed = (((unsigned long)(stop-start))*10*4) / 1000000;
	if (AMI_1200 == amiga_model)
		bus_speed /= 2;

	if ((bus_speed >= 47) && (bus_speed < 53)) {
		bus_speed = 50;
		freq = 12500000;
	} else if ((bus_speed >= 57) && (bus_speed < 63)) {
		bus_speed = 60;
		freq = 15000000;
	} else if ((bus_speed >= 63) && (bus_speed < 69)) {
		bus_speed = 66;
		freq = 16500000;
	} else {
		printk ("APUS: Unable to determine bus speed (%d). "
			"Defaulting to 50MHz", bus_speed);
		bus_speed = 50;
		freq = 12500000;
	}

	/* Ease diagnostics... */
	{
		extern int __map_without_bats;

		printk ("APUS: BATs=%d, BUS=%dMHz, RAM=%dns\n",
			(__map_without_bats) ? 0 : 1,
			bus_speed,
			(__60nsram) ? 60 : 70);

		/* print a bit more if asked politely... */
		if (!(ciaa.pra & 0x40)){
			extern unsigned int bat_addrs[4][3];
			int b;
			for (b = 0; b < 4; ++b) {
				printk ("APUS: BAT%d ", b);
				printk ("%08x-%08x -> %08x\n",
					bat_addrs[b][0],
					bat_addrs[b][1],
					bat_addrs[b][2]);
			}
		}

	}

	freq *= 60;	/* try to make freq/1e6 an integer */
        divisor = 60;
        printk("time_init: decrementer frequency = %d/%d\n", freq, divisor);
        decrementer_count = freq / HZ / divisor;
        count_period_num = divisor;
        count_period_den = freq / 1000000;
}

void arch_gettod(int *year, int *mon, int *day, int *hour,
		 int *min, int *sec)
{
	if (mach_gettod)
		mach_gettod(year, mon, day, hour, min, sec);
	else
		*year = *mon = *day = *hour = *min = *sec = 0;
}

/*********************************************************** FLOPPY */
#if defined(CONFIG_AMIGA_FLOPPY) || defined(CONFIG_ATARI_FLOPPY)
__initfunc(void floppy_setup(char *str, int *ints))
{
	if (mach_floppy_setup)
		mach_floppy_setup (str, ints);
}

void floppy_eject(void)
{
	if (mach_floppy_eject)
		mach_floppy_eject();
}
#endif

/*********************************************************** MEMORY */
extern void
map_page(struct task_struct *tsk, unsigned long va,
	 unsigned long pa, int flags);

#define KMAP_MAX 8
static unsigned long kmap_chunks[KMAP_MAX*3];
static int kmap_chunk_count = 0;

/* Based on arch/ppc/mm/init.c:ioremap() which maps the address range
   to the same virtual address as the physical address - which may
   cause problems since Z3 IO space is not the same as PCI/ISA.
   This should be rewritten to something more like the m68k version. */
unsigned long kernel_map (unsigned long phys_addr, unsigned long size,
			  int cacheflag, unsigned long *memavailp)
{
	unsigned long v_ret, end;
	/* Remap to 0x90000000. Related comment in ppc/mm/init.c. */
	static unsigned long virt = 0x90000000;
	v_ret = virt;

	if (kmap_chunk_count == KMAP_MAX*3)
		panic ("kernel_map: Can only map %d chunks.\n",
		       KMAP_MAX);

	kmap_chunks[kmap_chunk_count++] = phys_addr;
	kmap_chunks[kmap_chunk_count++] = size;
	kmap_chunks[kmap_chunk_count++] = v_ret;

	for (end = phys_addr + size ; phys_addr < end; 
	     phys_addr += PAGE_SIZE, virt += PAGE_SIZE) {
		map_page(&init_task, virt, phys_addr,
			 pgprot_val(PAGE_KERNEL_CI) | _PAGE_GUARDED);
	}
	return v_ret;
}

/* From pgtable.h */
extern __inline__ pte_t *my_find_pte(struct mm_struct *mm,unsigned long va)
{
	pgd_t *dir = 0;
	pmd_t *pmd = 0;
	pte_t *pte = 0;

	va &= PAGE_MASK;
	
	dir = pgd_offset( mm, va );
	if (dir)
	{
		pmd = pmd_offset(dir, va & PAGE_MASK);
		if (pmd && pmd_present(*pmd))
		{
			pte = pte_offset(pmd, va);
		}
	}
	return pte;
}


/* Again simulating an m68k/mm/kmap.c function. */
void kernel_set_cachemode( unsigned long address, unsigned long size,
			   unsigned int cmode )
{
	int mask, flags;

	switch (cmode)
	{
	case KERNELMAP_FULL_CACHING:
		mask = ~(_PAGE_NO_CACHE | _PAGE_GUARDED);
		flags = 0;
		break;
	case KERNELMAP_NOCACHE_SER:
		mask = ~0;
		flags = (_PAGE_NO_CACHE | _PAGE_GUARDED);
		break;
	default:
		panic ("kernel_set_cachemode() doesn't support mode %d\n", 
		       cmode);
		break;
	}
	
	size /= PAGE_SIZE;
	address &= PAGE_MASK;
	while (size--)
	{
		pte_t *pte;

		pte = my_find_pte(init_task.mm, address);
		if ( !pte )
		{
			printk("pte NULL in kernel_set_cachemode()\n");
			return;
		}

                pte_val (*pte) &= mask;
                pte_val (*pte) |= flags;
                flush_tlb_page(find_vma(init_task.mm,address),address);

		address += PAGE_SIZE;
	}
}

unsigned long mm_ptov (unsigned long paddr)
{
	unsigned long ret;
	if (paddr < 16*1024*1024)
		ret = ZTWO_VADDR(paddr);
	else {
		int i;

		for (i = 0; i < kmap_chunk_count;){
			unsigned long phys = kmap_chunks[i++];
			unsigned long size = kmap_chunks[i++];
			unsigned long virt = kmap_chunks[i++];
			if (paddr >= phys
			    && paddr < (phys + size)){
				ret = virt + paddr - phys;
				goto exit;
			}
		}
		
		ret = (unsigned long) __va(paddr);
	}
exit:
#ifdef DEBUGPV
	printk ("PTOV(%lx)=%lx\n", paddr, ret);
#endif
	return ret;
}

int mm_end_of_chunk (unsigned long addr, int len)
{
	if (memory[0].addr + memory[0].size == addr + len)
		return 1;
	return 0;
}

/*********************************************************** CACHE */

#define L1_CACHE_BYTES 32
#define MAX_CACHE_SIZE 8192
void cache_push(__u32 addr, int length)
{
	addr = mm_ptov(addr);

	if (MAX_CACHE_SIZE < length)
		length = MAX_CACHE_SIZE;

	while(length > 0){
		__asm ("dcbf 0,%0\n\t"
		       : : "r" (addr));
		addr += L1_CACHE_BYTES;
		length -= L1_CACHE_BYTES;
	}
	/* Also flush trailing block */
	__asm ("dcbf 0,%0\n\t"
	       "sync \n\t"
	       : : "r" (addr));
}
void cache_clear(__u32 addr, int length)
{
	if (MAX_CACHE_SIZE < length)
		length = MAX_CACHE_SIZE;

	addr = mm_ptov(addr);

	__asm ("dcbf 0,%0\n\t"
	       "sync \n\t"
	       "icbi 0,%0 \n\t"
	       "isync \n\t"
	       : : "r" (addr));
	
	addr += L1_CACHE_BYTES;
	length -= L1_CACHE_BYTES;

	while(length > 0){
		__asm ("dcbf 0,%0\n\t"
		       "sync \n\t"
		       "icbi 0,%0 \n\t"
		       "isync \n\t"
		       : : "r" (addr));
		addr += L1_CACHE_BYTES;
		length -= L1_CACHE_BYTES;
	}

	__asm ("dcbf 0,%0\n\t"
	       "sync \n\t"
	       "icbi 0,%0 \n\t"
	       "isync \n\t"
	       : : "r" (addr));
}

void
apus_restart(char *cmd)
{
	cli();

	APUS_WRITE(APUS_REG_LOCK, 
		   REGLOCK_BLACKMAGICK1|REGLOCK_BLACKMAGICK2);
	APUS_WRITE(APUS_REG_LOCK, 
		   REGLOCK_BLACKMAGICK1|REGLOCK_BLACKMAGICK3);
	APUS_WRITE(APUS_REG_LOCK, 
		   REGLOCK_BLACKMAGICK2|REGLOCK_BLACKMAGICK3);
	APUS_WRITE(APUS_REG_SHADOW, REGSHADOW_SELFRESET);
	APUS_WRITE(APUS_REG_RESET, REGRESET_AMIGARESET);
	for(;;);
}

void
apus_power_off(void)
{
	for (;;);
}

void
apus_halt(void)
{
   apus_restart(NULL);
}

void
apus_do_IRQ(struct pt_regs *regs,
	    int            cpu,
            int            isfake)
{
        int old_level, new_level;

	/* I don't think we need SMP code here - Corey */

        old_level = ~(regs->mq) & IPLEMU_IPLMASK;
        new_level = (~(regs->mq) >> 3) & IPLEMU_IPLMASK;
        if (new_level != 0)
        {
                APUS_WRITE(APUS_IPL_EMU, IPLEMU_IPLMASK);
                APUS_WRITE(APUS_IPL_EMU, (IPLEMU_SETRESET
                                          | (~(new_level) & IPLEMU_IPLMASK)));
                APUS_WRITE(APUS_IPL_EMU, IPLEMU_DISABLEINT);

                process_int (VEC_SPUR+new_level, regs);
                APUS_WRITE(APUS_IPL_EMU, IPLEMU_SETRESET | IPLEMU_DISABLEINT);
                APUS_WRITE(APUS_IPL_EMU, IPLEMU_IPLMASK);
                APUS_WRITE(APUS_IPL_EMU, (IPLEMU_SETRESET
                                          | (~(old_level) & IPLEMU_IPLMASK)));
        }
        APUS_WRITE(APUS_IPL_EMU, IPLEMU_DISABLEINT);
}

#if defined(CONFIG_BLK_DEV_IDE) || defined(CONFIG_BLK_DEV_IDE_MODULE)
/*
 * IDE stuff.
 */
void
apus_ide_insw(ide_ioreg_t port, void *buf, int ns)
{
	ide_insw(port, buf, ns);
}

void
apus_ide_outsw(ide_ioreg_t port, void *buf, int ns)
{
	ide_outsw(port, buf, ns);
}

int
apus_ide_default_irq(ide_ioreg_t base)
{
        m68k_ide_default_irq(base);
}

ide_ioreg_t
apus_ide_default_io_base(int index)
{
        m68k_ide_default_io_base(index);
}

int
apus_ide_check_region(ide_ioreg_t from, unsigned int extent)
{
        return m68k_ide_check_region(from, extent);
}

void
apus_ide_request_region(ide_ioreg_t from,
			unsigned int extent,
			const char *name)
{
        m68k_ide_request_region(from, extent, name);
}

void
apus_ide_release_region(ide_ioreg_t from,
			unsigned int extent)
{
        m68k_ide_release_region(from, extent);
}

void
apus_ide_fix_driveid(struct hd_driveid *id)
{
        m68k_ide_fix_driveid(id);
}

__initfunc(void
apus_ide_init_hwif_ports (ide_ioreg_t *p, ide_ioreg_t base, int *irq))
{
        m68k_ide_init_hwif_ports(p, base, irq);
}
#endif

__initfunc(void
apus_local_init_IRQ(void))
{
	ppc_md.mask_irq = amiga_disable_irq;
	ppc_md.unmask_irq = amiga_enable_irq;
	apus_init_IRQ();
}

__initfunc(void
apus_init(unsigned long r3, unsigned long r4, unsigned long r5,
	  unsigned long r6, unsigned long r7))
{
	/* Parse bootinfo. The bootinfo is located right after
           the kernel bss */
	parse_bootinfo((const struct bi_record *)&_end);
#ifdef CONFIG_BLK_DEV_INITRD
	/* Take care of initrd if we have one. Use data from
	   bootinfo to avoid the need to initialize PPC
	   registers when kernel is booted via a PPC reset. */
	if ( ramdisk.addr ) {
		initrd_start = (unsigned long) __va(ramdisk.addr);
		initrd_end = (unsigned long) 
			__va(ramdisk.size + ramdisk.addr);
	}
	/* Make sure code below is not executed. */
	r4 = 0;
	r6 = 0;
#endif /* CONFIG_BLK_DEV_INITRD */

	ISA_DMA_THRESHOLD = 0x00ffffff;

	ppc_md.setup_arch     = apus_setup_arch;
	ppc_md.setup_residual = NULL;
	ppc_md.get_cpuinfo    = apus_get_cpuinfo;
	ppc_md.irq_cannonicalize = NULL;
	ppc_md.init_IRQ       = apus_init_IRQ;
	ppc_md.do_IRQ         = apus_do_IRQ;
	ppc_md.get_irq_source = NULL;
	ppc_md.init           = NULL;

	ppc_md.restart        = apus_restart;
	ppc_md.power_off      = apus_power_off;
	ppc_md.halt           = apus_halt;

	ppc_md.time_init      = NULL;
	ppc_md.set_rtc_time   = apus_set_rtc_time;
	ppc_md.get_rtc_time   = apus_get_rtc_time;
	ppc_md.calibrate_decr = apus_calibrate_decr;

	/* These should not be used for the APUS yet, since it uses
	   the M68K keyboard now. */
	ppc_md.kbd_setkeycode    = NULL;
	ppc_md.kbd_getkeycode    = NULL;
	ppc_md.kbd_translate     = NULL;
	ppc_md.kbd_unexpected_up = NULL;
	ppc_md.kbd_leds          = NULL;
	ppc_md.kbd_init_hw       = NULL;
	ppc_md.kbd_sysrq_xlate	 = NULL;

#if defined(CONFIG_BLK_DEV_IDE) || defined(CONFIG_BLK_DEV_IDE_MODULE)
        ppc_ide_md.insw = apus_ide_insw;
        ppc_ide_md.outsw = apus_ide_outsw;
        ppc_ide_md.default_irq = apus_ide_default_irq;
        ppc_ide_md.default_io_base = apus_ide_default_io_base;
        ppc_ide_md.check_region = apus_ide_check_region;
        ppc_ide_md.request_region = apus_ide_request_region;
        ppc_ide_md.release_region = apus_ide_release_region;
        ppc_ide_md.fix_driveid = apus_ide_fix_driveid;
        ppc_ide_md.ide_init_hwif = apus_ide_init_hwif_ports;

        ppc_ide_md.io_base = _IO_BASE;
#endif		
}
