/* Accurate table for expl().
   Copyright (C) 1999-2025 Free Software Foundation, Inc.
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, see
   <https://www.gnu.org/licenses/>.  */

/* __expl_table basically consists of four tables, T_EXPL_ARG{1,2} and
   T_EXPL_RES{1,2}. All tables use positive and negative indexes, the 0 points
   are marked by T_EXPL_* defines.
   For ARG1 and RES1 tables lets B be 89 and S 256.0, for ARG2 and RES2 B is 65
   and S 32768.0.
   These table have the property that, for all integers -B <= i <= B
   expl(__expl_table[T_EXPL_ARGN+2*i]+__expl_table[T_EXPL_ARGN+2*i+1]+r) ==
   __expl_table[T_EXPL_RESN+i], __expl_table[T_EXPL_RESN+i] is some exact number
   with the low 58 bits of the mantissa 0,
   __expl_table[T_EXPL_ARGN+2*i] == i/S+s
   where absl(s) <= 2^-54 and absl(r) <= 2^-212.  */


static const long double __expl_table [] = {
 -3.47656250000000000584188889839535373E-01L, /* bffd640000000000002b1b04213cf000 */
  6.90417668990715641167244540876988960E-32L, /* 3f97667c3fdb588a6ae1af8748357a17 */
 -3.43749999999999981853132895957607418E-01L, /* bffd5ffffffffffffac4ff5f4050b000 */
 -7.16021898043268093462818380603370350E-33L, /* bf94296c8219427edc1431ac2498583e */
 -3.39843750000000013418643523138766329E-01L, /* bffd5c000000000003de1f027a30e000 */
  8.16920774283317801641347327589583265E-32L, /* 3f97a82b65774bdca1b4440d749ed8d3 */
 -3.35937500000000014998092453039303051E-01L, /* bffd5800000000000452a9f4d8857000 */
 -6.55865578425428447938248396879359670E-32L, /* bf97548b7d240f3d034b395e6eecfac8 */
 -3.32031250000000000981984049529998541E-01L, /* bffd540000000000004875277cda5000 */
  6.91213046334032232108944519541512737E-32L, /* 3f9766e5f925338a19045c94443b66e1 */
 -3.28124999999999986646017645350399708E-01L, /* bffd4ffffffffffffc26a667bf44d000 */
 -6.16281060996110316602421505683742661E-32L, /* bf973ffdcdcffb6fbffc86b2b8d42f5d */
 -3.24218749999999991645717430645867963E-01L, /* bffd4bfffffffffffd97901063e48000 */
 -7.90797211087760527593856542417304137E-32L, /* bf979a9afaaca1ada6a8ed1c80584d60 */
 -3.20312499999999998918211610690789652E-01L, /* bffd47ffffffffffffb02d9856d71000 */
  8.64024799457616856987630373786503376E-32L, /* 3f97c0a098623f95579d5d9b2b67342d */
 -3.16406249999999998153974811017181883E-01L, /* bffd43ffffffffffff77c991f1076000 */
 -2.73176610180696076418536105483668404E-32L, /* bf961baeccb32f9b1fcbb8e60468e95a */
 -3.12500000000000011420976192575972779E-01L, /* bffd400000000000034ab8240483d000 */
  7.16573502812389453744433792609989420E-32L, /* 3f977410f4c2cfc4335f28446c0fb363 */
 -3.08593750000000001735496343854851414E-01L, /* bffd3c000000000000800e995c176000 */
 -1.56292999645122272621237565671593071E-32L, /* bf95449b9cbdaff6ac1246adb2c826ac */
 -3.04687499999999982592401295899221626E-01L, /* bffd37fffffffffffafb8bc1e061a000 */
  6.48993208584888904958594509625158417E-32L, /* 3f9750f9fe8366d82d77afa0031a92e1 */
 -3.00781249999999999230616898937763959E-01L, /* bffd33ffffffffffffc73ac39da54000 */
  6.57082437496961397305801409357792029E-32L, /* 3f97552d3cb598ea80135cf3feb27ec4 */
 -2.96874999999999998788769281703245722E-01L, /* bffd2fffffffffffffa6a07fa5021000 */
 -3.26588297198283968096426564544269170E-32L, /* bf9653260fc1802f46b629aee171809b */
 -2.92968750000000015318089182805941695E-01L, /* bffd2c0000000000046a468614bd6000 */
 -1.73291974845198589684358727559290718E-32L, /* bf9567e9d158f52e483c8d8dcb5961dd */
 -2.89062500000000007736778942676309681E-01L, /* bffd280000000000023adf9f4c3d3000 */
 -6.83629745986675744404029225571026236E-32L, /* bf9762f5face6281c1daf1c6aedbdb45 */
 -2.85156250000000001367091555763661937E-01L, /* bffd2400000000000064dfa11e3fb000 */
 -5.44898442619766878281110054067026237E-32L, /* bf971aed6d2db9f542986a785edae072 */
 -2.81249999999999986958718100227029406E-01L, /* bffd1ffffffffffffc3db9265ca9d000 */
  1.13007318374506125723591889451107046E-32L, /* 3f94d569fe387f456a97902907ac3856 */
 -2.77343750000000000356078829380495179E-01L, /* bffd1c0000000000001a462390083000 */
 -4.98979365468978332358409063436543102E-32L, /* bf970315bbf3e0d14b5c94c900702d4c */
 -2.73437499999999990276993957508540484E-01L, /* bffd17fffffffffffd32919bcdc94000 */
 -8.79390484115892344533724650295100871E-32L, /* bf97c89b0b89cc19c3ab2b60da9bbbc3 */
 -2.69531250000000002434203866460082225E-01L, /* bffd14000000000000b39ccf9e130000 */
  9.44060754687026590886751809927191596E-32L, /* 3f97ea2f32cfecca5c64a26137a9210f */
 -2.65624999999999997296320716986257179E-01L, /* bffd0fffffffffffff3880f13a2bc000 */
  2.07142664067265697791007875348396921E-32L, /* 3f95ae37ee685b9122fbe377bd205ee4 */
 -2.61718750000000010237478733739017956E-01L, /* bffd0c000000000002f3648179d40000 */
 -6.10552936159265665298996309192680256E-32L, /* bf973d0467d31e407515a3cca0f3b4e2 */
 -2.57812500000000011948220522778370303E-01L, /* bffd08000000000003719f81275bd000 */
  6.72477169058908902499239631466443836E-32L, /* 3f975d2b8c475d3160cf72d227d8e6f9 */
 -2.53906249999999991822993360536596860E-01L, /* bffd03fffffffffffda4a4b62f818000 */
 -2.44868296623215865054704392917190994E-32L, /* bf95fc92516c6d057d29fc2528855976 */
 -2.49999999999999986862019457428548084E-01L, /* bffcfffffffffffff86d2d20d5ff4000 */
 -3.85302898949105073614122724961613078E-32L, /* bf96901f147cb7d643af71b6129ce929 */
 -2.46093750000000000237554160737318435E-01L, /* bffcf8000000000000230e8ade26b000 */
 -1.52823675242678363494345369284988589E-32L, /* bf953d6700c5f3fc303f79d0ec8c680a */
 -2.42187500000000003023380963205457065E-01L, /* bffcf0000000000001be2c1a78bb0000 */
 -7.78402037952209709489481182714311699E-34L, /* bf9102ab1f3998e887f0ee4cf940faa5 */
 -2.38281249999999995309623303145485725E-01L, /* bffce7fffffffffffd4bd2940f43f000 */
 -3.54307216794236899443913216397197696E-32L, /* bf966fef03ab69c3f289436205b21d02 */
 -2.34374999999999998425804947623207526E-01L, /* bffcdfffffffffffff17b097a6092000 */
 -2.86038428948386602859761879407549696E-32L, /* bf96290a0eba0131efe3a05fe188f2e3 */
 -2.30468749999999993822207406785200832E-01L, /* bffcd7fffffffffffc70519834eae000 */
 -2.54339521031747516806893838749365762E-32L, /* bf96081f0ad7f9107ae6cddb32c178ab */
 -2.26562499999999997823524030344489884E-01L, /* bffccffffffffffffebecf10093df000 */
  4.31904611473158635644635628922959401E-32L, /* 3f96c083f0b1faa7c4c686193e38d67c */
 -2.22656250000000004835132405125162742E-01L, /* bffcc8000000000002c98a233f19f000 */
  2.54709791629335691650310168420597566E-33L, /* 3f92a735903f5eed07a716ab931e20d9 */
 -2.18749999999999988969454021829236626E-01L, /* bffcbffffffffffff9a42dc14ce36000 */
 -3.77236096429336082213752014054909454E-32L, /* bf9687be8e5b2fca54d3e81157eac660 */
 -2.14843750000000010613256919115758495E-01L, /* bffcb80000000000061e3d828ecac000 */
 -4.55194148712216691177097854305964738E-32L, /* bf96d8b35c776aa3e1a4768271380503 */
 -2.10937499999999993204656148110447201E-01L, /* bffcaffffffffffffc152f2aea118000 */
 -2.95044199165561453749332254271716417E-32L, /* bf96326433b00b2439094d9bef22ddd1 */
 -2.07031250000000012233944895423355677E-01L, /* bffca80000000000070d695ee0e94000 */
  1.93146788688385419095981415411012357E-32L, /* 3f959126729135a5e390d4bb802a0bde */
 -2.03125000000000008030983633336321863E-01L, /* bffca0000000000004a129fbc51af000 */
  2.37361904671826193563212931215900137E-32L, /* 3f95ecfb3c4ba1b97ea3ad45cbb1e68a */
 -1.99218750000000001763815712796132779E-01L, /* bffc98000000000001044b12d9950000 */
 -3.63171243370923753295192486732883239E-33L, /* bf932db5fb3f27c38e0fa7bbcfc64f55 */
 -1.95312500000000004883660234506677272E-01L, /* bffc90000000000002d0b3779d1f9000 */
 -3.19989507343607877747980892249711601E-33L, /* bf9309d63de96bb3ef744c865f22f1bd */
 -1.91406250000000013720152363227519348E-01L, /* bffc88000000000007e8bcb387121000 */
 -1.89295754093147174148371614722178860E-32L, /* bf958926e2e67dfe812c508290add2e7 */
 -1.87500000000000000182342082774432620E-01L, /* bffc800000000000001ae8b06a39f000 */
 -2.96812835183184815200854214892983927E-32L, /* bf96343a62d156bbe71f55d14ca4b6e5 */
 -1.83593750000000012410147185883290345E-01L, /* bffc78000000000007276a1adda8d000 */
 -2.02191931237489669058466239995304587E-32L, /* bf95a3efab92d26ec2df90df036a117f */
 -1.79687499999999997439177363346082917E-01L, /* bffc6ffffffffffffe8616db2927d000 */
 -9.92752326937775530007399526834009465E-33L, /* bf949c5f88ed17041e1a3f1829d543cd */
 -1.75781249999999995824373974504785174E-01L, /* bffc67fffffffffffd97c94f13ea3000 */
  1.44184772065335613487885714828816178E-32L, /* 3f952b75c63476e7fcc2f5841c27bcce */
 -1.71874999999999986685050259043077809E-01L, /* bffc5ffffffffffff8530f6bc531a000 */
 -3.49007014971241147689894940544402482E-32L, /* bf966a6dfaa012aea8ffe6d90b02330f */
 -1.67968749999999997316058782350439701E-01L, /* bffc57fffffffffffe73eb914f2aa000 */
  3.34025733574205019081305778794376391E-32L, /* 3f965adf4572561fd5456a6c13d8babf */
 -1.64062499999999993322730602128318480E-01L, /* bffc4ffffffffffffc269be4f68f3000 */
 -1.83345916769684984022099095506340635E-32L, /* bf957ccb69026cb2f6024c211576d5f4 */
 -1.60156249999999992419000744447607979E-01L, /* bffc47fffffffffffba13df21784a000 */
  2.73442789798110494773517431626534726E-32L, /* 3f961bf58ff22c9b30f1e2b39f26d7d5 */
 -1.56249999999999987665010524130393080E-01L, /* bffc3ffffffffffff8e3ad45e7508000 */
  2.02695576464836145806428118889332191E-32L, /* 3f95a4fb7435a4a2f71de81eb8ae75d1 */
 -1.52343749999999989905291167951491803E-01L, /* bffc37fffffffffffa2e48aecfc24000 */
 -3.61436631548815190395331054871041524E-32L, /* bf967756567ebd108075ae527cc2e7f0 */
 -1.48437500000000006686107754967759751E-01L, /* bffc30000000000003dab20261b3c000 */
 -2.15524270159131591469319477922198390E-32L, /* bf95bfa05b82ef3a708c4f0395e9fcf6 */
 -1.44531250000000005132889939177166485E-01L, /* bffc28000000000002f57b1969e7b000 */
  2.74741116529653547935086189244019604E-32L, /* 3f961d4eb77c1185d34fe1b04a3f3cf5 */
 -1.40625000000000000707469094533647325E-01L, /* bffc2000000000000068676d3d5c4000 */
  4.40607097220049957013547629906723266E-33L, /* 3f936e0ac425daf795b42913cf0ef881 */
 -1.36718749999999995713752139187543306E-01L, /* bffc17fffffffffffd87762255991000 */
 -3.73751317180116492404578048203389108E-32L, /* bf9684202491e9cbb7ceb67d9ff7e0c9 */
 -1.32812500000000007198453630478482191E-01L, /* bffc10000000000004264de3a4379000 */
 -3.97050085179660203884930593717220728E-32L, /* bf969c52048de14be3c9c1971e50869c */
 -1.28906250000000006070486371645733082E-01L, /* bffc080000000000037fd87db2cb0000 */
  3.59610068058504988294019521946586131E-32L, /* 3f967570c10687cb8e9ebd0b280abf5a */
 -1.25000000000000003700729208608337966E-01L, /* bffc00000000000002222198bbc74000 */
  3.23464851393124362331846965931995969E-33L, /* 3f930cb95da3bfc847e593716c91d57a */
 -1.21093750000000013729038501177102555E-01L, /* bffbf000000000000fd418d1f5fda000 */
  2.45242487730722066611358741283977619E-32L, /* 3f95fd5945ad86a464292e26ac192a84 */
 -1.17187499999999999765305306880205578E-01L, /* bffbdfffffffffffffbabaf869845000 */
 -1.14557520298960389903199646350205537E-32L, /* bf94dbda735322179d9bcf392e1dd06d */
 -1.13281250000000009579647893740755690E-01L, /* bffbd000000000000b0b69bae7ab9000 */
  2.37873962873837390105423621772752350E-32L, /* 3f95ee0b7e0bd5ac1f6fab1e2a71abc3 */
 -1.09375000000000008981153004560108539E-01L, /* bffbc000000000000a5ac4bc1d2c3000 */
  1.53152444860014076105003555837231015E-32L, /* 3f953e15ce931e12ef9a152522e32bdd */
 -1.05468749999999992399063850363228723E-01L, /* bffbaffffffffffff73c998091408000 */
 -8.75920903597804862471749360196688834E-33L, /* bf946bd7e310a01bae5687ebdc47fcc5 */
 -1.01562500000000007685885179918350550E-01L, /* bffba0000000000008dc7910a648c000 */
 -4.63820993797174451904075397785059501E-33L, /* bf938153d0e54001a472da180fb5e8aa */
 -9.76562499999999887262211517861331814E-02L, /* bffb8ffffffffffff300915aa6fd6000 */
 -2.63767025974952608658936466715705903E-33L, /* bf92b64215bb8d520be5404620d38088 */
 -9.37499999999999939650246024457439795E-02L, /* bffb7ffffffffffff90aca26bd0fc000 */
 -1.72047822349322956713582039121348377E-32L, /* bf9565545015c5b9b56d02cfefca2c7d */
 -8.98437500000000033088896383977486369E-02L, /* bffb70000000000003d09ca1e3cbe000 */
  3.04831994420989436248526129869697270E-33L, /* 3f92fa7d30d2ed90e7ebbd6231fd08b1 */
 -8.59374999999999947312400115121319225E-02L, /* bffb5ffffffffffff9ecefc03376e000 */
  1.50416954438393392150792422537312281E-32L, /* 3f9538675ee99bd722fad0023c09c915 */
 -8.20312500000000054182280847004695514E-02L, /* bffb500000000000063f2dbd40200000 */
  2.68399664523430004488075638997207289E-33L, /* 3f92bdf49766629882c49a3da88928ed */
 -7.81250000000000114767533968079748798E-02L, /* bffb4000000000000d3b56f81ba70000 */
  1.72318124201659121296305402819694281E-32L, /* 3f9565e407aaabfb359e8a567d760de3 */
 -7.42187500000000035531829472486812869E-02L, /* bffb3000000000000418b6e9b5388000 */
  2.09401756478514117051383998628099655E-32L, /* 3f95b2e91221fcd74be0a86d8ad658d2 */
 -7.03124999999999987474933134860732535E-02L, /* bffb1ffffffffffffe8e53453d2ac000 */
  2.28515798224350800271565551341211666E-32L, /* 3f95da9bd6adf00894f05b5cc5530125 */
 -6.64062500000000042267533361089054159E-02L, /* bffb10000000000004df8473dbcf2000 */
  1.97576478800281368377376002585430031E-32L, /* 3f959a59acbddb2f53bd3096b66370e9 */
 -6.25000000000000066329769382774201686E-02L, /* bffb00000000000007a5b5914e336000 */
 -1.46422615813786836245343723048221678E-33L, /* bf91e69295f069fc0c4a9db181ea25a3 */
 -5.85937500000000002823707957982406053E-02L, /* bffae0000000000000a6aeab10592000 */
  9.25637741701318872896718218457555829E-33L, /* 3f94807eb021f1f40a37d4015b1eb76b */
 -5.46875000000000081586888005226044448E-02L, /* bffac0000000000012d00a3171e3a000 */
 -4.87144542459404765480424673678105050E-33L, /* bf9394b42faba6b7036fe7b36269daf3 */
 -5.07812499999999927720348253140567013E-02L, /* bffa9fffffffffffef555cc8dd914000 */
 -3.01901021987395945826043649523451725E-33L, /* bf92f59e7e3025691f290f8f67277faf */
 -4.68749999999999935349476738962633103E-02L, /* bffa7ffffffffffff117b4ea2b876000 */
  1.21521638219189777347767475937119750E-32L, /* 3f94f8c7f88c5b56674b94d984ac8ecb */
 -4.29687500000000056305562847814228219E-02L, /* bffa6000000000000cfbb19be30c0000 */
 -1.18643699217679276275559592978275214E-32L, /* bf94ecd39f0833a876550e83eb012b99 */
 -3.90624999999999962692914526031373542E-02L, /* bffa3ffffffffffff765c743922f9000 */
 -4.91277156857520035712509544689973679E-33L, /* bf939823189996193872e58ac0dececb */
 -3.51562500000000108152468207687602886E-02L, /* bffa20000000000018f031e41177f000 */
  1.18599806302656253755207072755609820E-32L, /* 3f94eca4f23e787fab73ce8f6b9b8d64 */
 -3.12500000000000077376981036742289578E-02L, /* bffa00000000000011d787e0b386f000 */
  9.97730386477005171963635210799577079E-33L, /* 3f949e70e498c46a0173ac0d46c699fc */
 -2.73437500000000139436129596418623235E-02L, /* bff9c00000000000404db66e70a08000 */
  2.25755321633070123579875157841633859E-33L, /* 3f927719b1a93074bdf9f3c2cb784785 */
 -2.34375000000000088003629211828324876E-02L, /* bff98000000000002895a27d45feb000 */
  2.84374279216848803102126617873942975E-33L, /* 3f92d87f70e749d6da6c260b68dc210b */
 -1.95312500000000107408831063404855424E-02L, /* bff9400000000000318898ba69f71000 */
  2.47348089686935458989103979140011912E-33L, /* 3f929afa3de45086fe909fdddb41edce */
 -1.56250000000000081443917555362290635E-02L, /* bff9000000000000258f335e9cdd6000 */
 -2.43379314483517422161458863218426254E-33L, /* bf9294621c8a9ccacf2b020ec19cad27 */
 -1.17187500000000051490597418161403184E-02L, /* bff88000000000002f7ddfa26221f000 */
  1.83405297208145390679150568810924707E-33L, /* 3f9230bbfc5d5fe1b534fbcda0465bb9 */
 -7.81249999999999715861805208310174953E-03L, /* bff7ffffffffffffcb95f3fff157d000 */
  3.51548384878710915171654413641872451E-34L, /* 3f8fd349b76c22966f77a39fc37ed704 */
 -3.90625000000000309326013918295097128E-03L, /* bff7000000000000390f820c8e153000 */
  6.38058004651791109324060099097251911E-36L, /* 3f8a0f665d3ac25a1ac94d688273dbcd */
#define T_EXPL_ARG1 (2*89)
  0.00000000000000000000000000000000000E+00L, /* 00000000000000000000000000000000 */
  0.00000000000000000000000000000000000E+00L, /* 00000000000000000000000000000000 */
  3.90625000000000245479958859972588985E-03L, /* 3ff70000000000002d48769ac9874000 */
 -6.58439598384342854976169982902779828E-36L, /* bf8a1811b923e6c626b07ef29761482a */
  7.81250000000001311374391093664996358E-03L, /* 3ff800000000000078f3f3cd89111000 */
  2.60265650555493781464273319671555602E-33L, /* 3f92b070c3b635b87af426735a71fc87 */
  1.17187500000000269581156218247101912E-02L, /* 3ff8800000000000f8a50d02fe20d000 */
  1.00961747974945520631836275894919326E-33L, /* 3f914f80c1a4f8042044fe3b757b030b */
  1.56249999999999797878275270751825475E-02L, /* 3ff8ffffffffffff45935b69da62e000 */
  2.03174577741375590087897353146748580E-33L, /* 3f925194e863496e0f6e91cbf6b22e26 */
  1.95312499999999760319884511789111533E-02L, /* 3ff93fffffffffff917790ff9a8f4000 */
  4.62788519658803722282100289809515007E-33L, /* 3f9380783ba81295feeb3e4879d7d52d */
  2.34374999999999822953909016349145918E-02L, /* 3ff97fffffffffffae5a163bd3cd5000 */
 -3.19499956304699705390404384504876533E-33L, /* bf93096e2037ced8194cf344c692f8d6 */
  2.73437500000000137220327275871555682E-02L, /* 3ff9c000000000003f481dea5dd51000 */
 -2.25757776523031994464630107442723424E-33L, /* bf92771abcf988a02b414bf2614e3734 */
  3.12499999999999790857640618332718621E-02L, /* 3ff9ffffffffffff9f8cd40b51509000 */
 -4.22479470489989916319395454536511458E-33L, /* bf935efb7245612f371deca17cb7b30c */
  3.51562499999999840753382405747597346E-02L, /* 3ffa1fffffffffffdb47bd275f722000 */
  1.08459658374118041980976756063083500E-34L, /* 3f8e2055d18b7117c9db1c318b1e889b */
  3.90624999999999989384433621470426757E-02L, /* 3ffa3ffffffffffffd8d5e18b042e000 */
 -7.41674226146122000759491297811091830E-33L, /* bf94341454e48029e5b0205d91baffdc */
  4.29687500000000107505739500500200462E-02L, /* 3ffa60000000000018ca04cd9085c000 */
 -4.74689012756713017494437969420919847E-34L, /* bf903b7c268103c6f7fbaaa24142e287 */
  4.68749999999999978700749928325717352E-02L, /* 3ffa7ffffffffffffb16b6d5479e3000 */
 -1.06208165308448830117773486334902917E-32L, /* bf94b92be4b3b5b5a596a0a5187cc955 */
  5.07812499999999815072625435955786253E-02L, /* 3ffa9fffffffffffd55bd086d5cbc000 */
 -9.37038897148383660401929567549111394E-33L, /* bf94853b111b0175b491c80d00419416 */
  5.46874999999999809511553152189867394E-02L, /* 3ffabfffffffffffd4138bfa74a61000 */
  1.06642963074562437340498606682822123E-32L, /* 3f94bafa3fe991b39255d563dfa05d89 */
  5.85937500000000184331996330905145551E-02L, /* 3ffae000000000002a810a5f2f8bf000 */
 -1.76639977694797200820296641773791945E-34L, /* bf8ed596f07ce4408f1705c8ec16864c */
  6.25000000000000021544696744852045001E-02L, /* 3ffb000000000000027be32045e2b000 */
  1.68616371995798354366633034788947149E-32L, /* 3f955e33d7440794d8a1b25233d086ab */
  6.64062499999999965563110718495802889E-02L, /* 3ffb0ffffffffffffc079a38a3fed000 */
 -1.82463217667830160048872113565316215E-32L, /* bf957af6163bcdb97cefab44a942482a */
  7.03124999999999759989183341261898222E-02L, /* 3ffb1fffffffffffe454218acea05000 */
 -1.07843770101525495515646940862541503E-32L, /* bf94bff72aada26d94e76e71c07e0580 */
  7.42187499999999898968873730710101412E-02L, /* 3ffb2ffffffffffff45a166496dc1000 */
  1.28629441689592874462780757154138223E-32L, /* 3f950b2724597b8b93ce1e9d1cf4d035 */
  7.81249999999999957198938523510804668E-02L, /* 3ffb3ffffffffffffb10bc52adbc5000 */
  1.13297573459968118467100063135856856E-33L, /* 3f91787eea895b3c245899cf34ad0abd */
  8.20312500000000199911640621145851159E-02L, /* 3ffb500000000000170c59a661a89000 */
 -1.51161335208135146756554123073528707E-32L, /* bf9539f326c5ca84e7db5401566f3775 */
  8.59375000000000134175373433347670743E-02L, /* 3ffb6000000000000f78287547af0000 */
  1.09763629458404270323909815379924900E-32L, /* 3f94c7f0b61b6e3e27d44b9f5bbc7e9d */
  8.98437500000000036533922600308306335E-02L, /* 3ffb70000000000004364a83b7a14000 */
  3.11459653680110433194288029777718358E-33L, /* 3f9302c0248136d65cebeab69488d949 */
  9.37500000000000184977946245216914691E-02L, /* 3ffb800000000000155395d870b17000 */
 -4.66656154468277949130395786965043927E-33L, /* bf9383aec9b993b6db492b1ede786d8a */
  9.76562500000000237839723100419376084E-02L, /* 3ffb9000000000001b6bca237f6c4000 */
 -1.03028043424658760249140747856831301E-32L, /* bf94abf6352e3d2bb398e47919a343fb */
  1.01562500000000012345545575236836572E-01L, /* 3ffba000000000000e3bc30cd9a1f000 */
  2.15755372310795701322789783729456319E-32L, /* 3f95c01b3b819edd9d07548fafd61550 */
  1.05468749999999976493840484471911438E-01L, /* 3ffbafffffffffffe4e634cd77985000 */
  1.78771847038773333029677216592309083E-32L, /* 3f95734b6ae650f33dd43c49a1df9fc0 */
  1.09375000000000002267015055992785402E-01L, /* 3ffbc00000000000029d1ad08de7b000 */
  6.23263106693943817730045115112427717E-33L, /* 3f9402e4b39ce2198a45e1d045868cd6 */
  1.13281250000000022354208618429577398E-01L, /* 3ffbd0000000000019c5cc3f9d2b5000 */
  5.40514416644786448581426756221178868E-33L, /* 3f93c10ab4021472c662f69435de9269 */
  1.17187500000000013252367133076817603E-01L, /* 3ffbe000000000000f47688cc561b000 */
 -7.12412585457324989451327215568641325E-33L, /* bf9427ecb343a8d1758990565fcfbf45 */
  1.21093750000000020759863992944300792E-01L, /* 3ffbf0000000000017ef3af97bf04000 */
  6.26591408357572503875647872077266444E-33L, /* 3f940446a09a2da771b45fc075514d12 */
  1.25000000000000004739659392396765618E-01L, /* 3ffc00000000000002bb7344ecd89000 */
 -1.55611398459729463981000080101758830E-32L, /* bf95433135febefa9e6aa4db39e263d2 */
  1.28906249999999982360888081057894783E-01L, /* 3ffc07fffffffffff5d4ed3154361000 */
 -1.77531518652835570781208599686606474E-32L, /* bf9570b7f225ea076f97f418d11359c1 */
  1.32812500000000010568583998727400436E-01L, /* 3ffc1000000000000617a5d09526a000 */
  2.12104021624990594668286391598300893E-32L, /* 3f95b885d767a1048d93055927a27adc */
  1.36718749999999998434125157367005292E-01L, /* 3ffc17ffffffffffff18eaebc7970000 */
  2.50454798592543203967309921276955297E-32L, /* 3f9604164e5598528a76faff26cd1c97 */
  1.40625000000000015550032422969330356E-01L, /* 3ffc20000000000008f6c79d8928c000 */
  7.80972982879849783680252962992639832E-33L, /* 3f9444674acf2b3225c7647e0d95edf3 */
  1.44531250000000012402535562111122522E-01L, /* 3ffc28000000000007264a8bc1ff1000 */
  2.79662468716455159585514763921671876E-32L, /* 3f96226b095bd78aa650faf95a221993 */
  1.48437500000000007761020440087419948E-01L, /* 3ffc3000000000000479530ff8fe3000 */
  2.15518492972728435680556239996258527E-32L, /* 3f95bf9d49295e73a957906a029768cb */
  1.52343750000000001733189947520484032E-01L, /* 3ffc38000000000000ffc6109f71f000 */
  8.34032236093545825619420380704500188E-33L, /* 3f945a71851226a1d0ce5e656693153e */
  1.56249999999999988073295321246958484E-01L, /* 3ffc3ffffffffffff91fedd62ae0f000 */
  2.44119337150624789345260194989620908E-32L, /* 3f95fb041a57bc1c1280680ac1620bea */
  1.60156250000000002076894210913572460E-01L, /* 3ffc48000000000001327ed84a199000 */
 -7.36124501128859978061216696286151753E-33L, /* bf9431c62f01e59d2c1e00f195a0037f */
  1.64062500000000000950861276373482172E-01L, /* 3ffc500000000000008c5285fba85000 */
 -4.80566184447001164583855800470217373E-33L, /* bf938f3d1fcafd390f22f80e6c19421f */
  1.67968749999999989878071706155265999E-01L, /* 3ffc57fffffffffffa2a445c548c5000 */
 -4.42154428718618459799673088733365064E-32L, /* bf96cb28cf1c1b28006d53ffe633b22a */
  1.71874999999999999459734108403218175E-01L, /* 3ffc5fffffffffffffb04554e9dd4000 */
 -3.29736288190321377985697972236270628E-32L, /* bf96566af0ebc852e84be12859b24a31 */
  1.75781249999999997987525759778901845E-01L, /* 3ffc67fffffffffffed702df6ffff000 */
 -1.28800728638468399687523924685844352E-32L, /* bf950b8236b88ca0c1b739dc91a7e3fc */
  1.79687500000000004929565820437175783E-01L, /* 3ffc70000000000002d779bb32d2e000 */
  1.60624461317978482424582320675174225E-32L, /* 3f954d9a9cc0c963fd081f3dc922d04e */
  1.83593750000000016873727045739708856E-01L, /* 3ffc78000000000009ba1f6263c9a000 */
 -3.83390389582056606880506003118452558E-32L, /* bf968e22a5d826f77f19ee788474df22 */
  1.87500000000000013443068740761666872E-01L, /* 3ffc80000000000007bfd8c72a1bf000 */
 -2.74141662712926256150154726565203091E-32L, /* bf961caf5ac59c7f941f928e324c2cc1 */
  1.91406249999999981494101786848611970E-01L, /* 3ffc87fffffffffff55502eeae001000 */
  3.68992437075565165346469517256118001E-32L, /* 3f967f2f03f9096793372a27b92ad79d */
  1.95312499999999989069921848800501648E-01L, /* 3ffc8ffffffffffff9b3015280394000 */
  3.69712249337856518452988332367785220E-32L, /* 3f967fee5fdb5bd501ff93516999faa0 */
  1.99218750000000021148042946919300804E-01L, /* 3ffc9800000000000c30e67939095000 */
  2.50142536781142175091322844848566649E-32L, /* 3f9603c34ae58e10b300b07137ee618a */
  2.03124999999999977732559198825437141E-01L, /* 3ffc9ffffffffffff329e7df079e4000 */
 -2.41951877287895024779300892731537816E-32L, /* bf95f683aefe6965f080df8f59dd34a1 */
  2.07031249999999996744030653771913124E-01L, /* 3ffca7fffffffffffe1f80f4b73ca000 */
 -1.94346475904454000031592792989765585E-32L, /* bf9593a44f87870a3d100d498501ecc7 */
  2.10937500000000000251399259834392298E-01L, /* 3ffcb000000000000025199873310000 */
 -1.33528748788094249098998693871759411E-33L, /* bf91bbb9b25c813668d6103d08acac35 */
  2.14843749999999993936323609611875097E-01L, /* 3ffcb7fffffffffffc8128c866236000 */
  1.14839877977014974625242788556545292E-32L, /* 3f94dd06b4655c9b83a1305b240e7a42 */
  2.18750000000000015181732784749663837E-01L, /* 3ffcc0000000000008c06da5fff24000 */
  1.42689085313142539755499441881408391E-32L, /* 3f95285a87dfa7ea7dad5b3be8c669f4 */
  2.22656249999999992172647770539596569E-01L, /* 3ffcc7fffffffffffb7ce2fe531f6000 */
 -3.34421462850496887359128610229650547E-32L, /* bf965b487962b5c2d9056ca6ac0c2e5c */
  2.26562499999999989595607223847082419E-01L, /* 3ffccffffffffffffa0095277be5c000 */
 -3.08983588107248752517344356508205569E-32L, /* bf9640dded57157f8eded311213bdbcd */
  2.30468749999999979130462438434567117E-01L, /* 3ffcd7fffffffffff3f8332996560000 */
 -3.01407539802851697849105682795217019E-32L, /* bf9638ffde35dbdfe1a1ffe45185de5d */
  2.34375000000000012194252337217891971E-01L, /* 3ffce0000000000007078dd402c86000 */
 -8.46879710915628592284714319904522657E-33L, /* bf945fc7b29a2ac6c9eff9eb258a510f */
  2.38281249999999982991877076137149870E-01L, /* 3ffce7fffffffffff6320b486eece000 */
 -2.93563878880439245627127095245798544E-32L, /* bf9630daaa4f40ff05caf29ace2ea7d4 */
  2.42187499999999981447559841442773990E-01L, /* 3ffceffffffffffff54e24a09a8d5000 */
 -4.56766746558806021264215486909850481E-32L, /* bf96da556dee11f3113e5a3467b908e6 */
  2.46093749999999991067720539980207318E-01L, /* 3ffcf7fffffffffffad9d405dcb5d000 */
  2.14033004219908074003010247652128251E-32L, /* 3f95bc8776e8f9ae098884aa664cc3df */
  2.50000000000000016613825838126835953E-01L, /* 3ffd00000000000004c9e24c12bb3000 */
  2.57617532593749185996714235009382870E-32L, /* 3f960b867cc01178c0ec68226c6cb47d */
  2.53906250000000013372004437827044321E-01L, /* 3ffd04000000000003daae05b3168000 */
  7.20177123439204414298152646284640101E-32L, /* 3f9775eff59ddad7e7530b83934af87f */
  2.57812499999999995765234725413886085E-01L, /* 3ffd07fffffffffffec7878bad9d5000 */
  6.51253187532920882777046064603770602E-32L, /* 3f975226659ca241402e71c2011583b0 */
  2.61718750000000007647689994011222248E-01L, /* 3ffd0c000000000002344cc793a0f000 */
  3.02370610028725823590045201871491395E-32L, /* 3f9639ffe55fa2fa011674448b4e5b96 */
  2.65624999999999986893899042596554269E-01L, /* 3ffd0ffffffffffffc38f0c0a1e9f000 */
 -2.07683715950724761146070082510569258E-32L, /* bf95af579a92e872fef81abfdf06bae8 */
  2.69531249999999979842788204900639327E-01L, /* 3ffd13fffffffffffa30a908d67db000 */
  8.71465252506557329027658736641075706E-32L, /* 3f97c47d99e19830447a42b1c0ffac61 */
  2.73437500000000006712165837793818271E-01L, /* 3ffd18000000000001ef453a58edb000 */
 -6.62704045767568912140550474455810301E-32L, /* bf9758187a204dcb06ece46588aeeaba */
  2.77343749999999994411329302988535617E-01L, /* 3ffd1bfffffffffffe63a0fec9c9e000 */
 -4.87273466291944117406493607771338767E-32L, /* bf96fa0381b0844a0be46bac2d673f0c */
  2.81250000000000012677892447379453135E-01L, /* 3ffd20000000000003a7769e125d6000 */
 -8.55871796664700790726282049552906783E-32L, /* bf97bc64e01332cf7616b0091b8dff2c */
  2.85156249999999998558643013736363981E-01L, /* 3ffd23ffffffffffff95a5894bccf000 */
 -1.33068334720606220176455289635046875E-32L, /* bf95145f43290ecf5b7adcb24697bc73 */
  2.89062500000000008831431235621753924E-01L, /* 3ffd280000000000028ba504fac59000 */
 -9.34157398616814623985483776710704237E-32L, /* bf97e50ad1115b941fcb5f0c88a428f7 */
  2.92968750000000019840235286110877063E-01L, /* 3ffd2c000000000005b7f372d184f000 */
  4.99302093775173155906059132992249671E-33L, /* 3f939ecdcfb97bad3f8dbec5df5ec67d */
  2.96875000000000015867911730971630513E-01L, /* 3ffd3000000000000492d860c79db000 */
  7.86107787827057767235127454590866211E-33L, /* 3f944689517ee8f16cdb97d6a6938f32 */
  3.00781250000000015814100002286124758E-01L, /* 3ffd340000000000048edfe73a17d000 */
 -1.65419431293024229981937172317171504E-32L, /* bf9557900e3efca16c89646b57f68dc0 */
  3.04687499999999985213157159965287195E-01L, /* 3ffd37fffffffffffbbcec6f99b36000 */
  9.68753602893894024018934325652944198E-32L, /* 3f97f70170e5458660c33a7e8d43d049 */
  3.08593749999999989969324338045156215E-01L, /* 3ffd3bfffffffffffd1bdde4d0fb1000 */
  7.10268609610294706092252562643261106E-32L, /* 3f9770cae45cdf615010401a4b37d8d4 */
  3.12500000000000002971606591018488854E-01L, /* 3ffd40000000000000db440fbc06b000 */
  6.38924218802905979887732294952782964E-32L, /* 3f974bbf988bb5622bd8fbaa46e8b811 */
  3.16406250000000006594921047402056305E-01L, /* 3ffd44000000000001e69e8954814000 */
  3.96079878754651470094149874444850097E-32L, /* 3f969b5017b9fa7a1e86975258c73d3d */
  3.20312500000000006713799366908329147E-01L, /* 3ffd48000000000001ef64159c065000 */
 -1.86401314975634286055150437995880517E-32L, /* bf958323f0434911794e5fb8bfe136ba */
  3.24218749999999987061246567584951210E-01L, /* 3ffd4bfffffffffffc4549db9b928000 */
 -3.18643523744758601387071062700407431E-32L, /* bf964ae5fa7e26c2c3981bed12e14372 */
  3.28124999999999991782776266707412953E-01L, /* 3ffd4ffffffffffffda1ad0840ca8000 */
 -4.46964199751314296839915534813144652E-32L, /* bf96d0277729ffd74727150df6d15547 */
  3.32031250000000000393816557756032682E-01L, /* 3ffd540000000000001d0efc04fad000 */
 -9.03246333902065439930373230002688649E-33L, /* bf947731a008748cc6dee948839ef7ae */
  3.35937499999999983810482995064392173E-01L, /* 3ffd57fffffffffffb556cab8ae61000 */
  5.27742727066129518825981597650621794E-32L, /* 3f9712050a6ddbf1cabf1b971f4b5d0b */
  3.39843750000000004310441349760912471E-01L, /* 3ffd5c0000000000013e0def5ddc4000 */
 -3.85927263474732591932884416445586106E-32L, /* bf9690c51088ef3db9ca000829c450c2 */
  3.43749999999999990248130003997484364E-01L, /* 3ffd5ffffffffffffd3070624a0af000 */
  9.62005170171527308106468341512327487E-34L, /* 3f913fae595cea84432eb01430817fca */
  3.47656250000000004085726414568625697E-01L, /* 3ffd640000000000012d79309e291000 */
 -6.59664093705705297250259434519072507E-32L, /* bf97568465eafb0e662e64a5dbfaf35f */

 -1.98364257812501251077851763965418372E-03L, /* bff6040000000001cd90f658cf0b1000 */
 -3.71984513103117734260309047540278737E-34L, /* bf8fee73c54483194782aac4a6154d11 */
 -1.95312500000000378520649630233891879E-03L, /* bff60000000000008ba643bb5e2e8000 */
 -1.12194202736719050440745599339855038E-34L, /* bf8e2a436aeff7bc529873354f47a3f5 */
 -1.92260742187499397430259771221991482E-03L, /* bff5f7fffffffffe4361cb51170da000 */
 -2.30068299876822157331268484824540848E-34L, /* bf8f31d02f85cfe8c0cc02276ce0f437 */
 -1.89208984375001137424603270262074989E-03L, /* bff5f0000000000347456ed490c23000 */
 -1.15012507244426243338260435466985403E-34L, /* bf8e31c174d5677a937a34ad8d2a70b4 */
 -1.86157226562500172319250342061336738E-03L, /* bff5e800000000007f262fa3617b4000 */
 -3.12438344643346437509767736937785561E-34L, /* bf8f9f4d426a2457c273d34ef7d9bde9 */
 -1.83105468749999505256246872355430379E-03L, /* bff5dffffffffffe92f18c1c2b6fa000 */
 -5.91130415288336591179087455220308942E-35L, /* bf8d3a4c80b42dc036bae446c9807f78 */
 -1.80053710937499445182387245573120522E-03L, /* bff5d7fffffffffe669dea82b4a4c000 */
 -1.92396289352411531324908916321392100E-34L, /* bf8eff7a2123fb573ba9778550d669bd */
 -1.77001953125000387737631542516323906E-03L, /* bff5d000000000011e19915c3ddb7000 */
  7.91101758977203355387806553469731354E-36L, /* 3f8a507f5a70faaccf469e3461873dea */
 -1.73950195312500034854670281415554486E-03L, /* bff5c8000000000019b7dc6ef97bd000 */
  1.55906551582436824067407021178835755E-34L, /* 3f8e9e7880333e34955aebcde3cfb053 */
 -1.70898437499998955782591472611429852E-03L, /* bff5bffffffffffcfd80e88aa6b96000 */
  8.22951661962611381718215899498500357E-35L, /* 3f8db58e6031a779b59f6ece191de7cc */
 -1.67846679687500586652037711131708544E-03L, /* bff5b80000000001b0df6fd21c133000 */
 -8.96642618848426299713145894522897419E-35L, /* bf8ddcbcab46d531801bfae4121f2f8a */
 -1.64794921875000109499161354039904782E-03L, /* bff5b0000000000050cbce8915575000 */
 -2.88077905394253859590587789680486639E-34L, /* bf8f7eebd4dd860ef73b674d5e707959 */
 -1.61743164062501133830507079150388351E-03L, /* bff5a80000000003449e8700c3e82000 */
 -3.68271725851639066312899986829350273E-34L, /* bf8fe9845fe20a5fe74059e0cae185d6 */
 -1.58691406249999015546015764131101956E-03L, /* bff59ffffffffffd2999e668cdd28000 */
  8.48197657099957029953716507898788812E-35L, /* 3f8dc2faaebb97392e451b07b28c4b12 */
 -1.55639648437500317366570219290722587E-03L, /* bff5980000000000ea2cd9a40d256000 */
 -3.45156704719737676412949957712570373E-36L, /* bf8925a079505516c8e317ac1ff53255 */
 -1.52587890625000568759013197767046039E-03L, /* bff5900000000001a3ab8a3f6b698000 */
 -1.01902948542497496574967177677556729E-34L, /* bf8e0ee78d94d9b5ad3d63ae35c9b554 */
 -1.49536132812500945889014955936485340E-03L, /* bff5880000000002b9f1621b57743000 */
 -3.32264697086631598830366079048117140E-34L, /* bf8fb9a7d14c32289204fbb0c9eb20e0 */
 -1.46484374999999931883259902869504725E-03L, /* bff57fffffffffffcdbd1c90e1b4a000 */
 -1.76487524793892929381101031660811433E-34L, /* bf8ed52f2f724bc1ae870b18356337b4 */
 -1.43432617187498876325946983333888768E-03L, /* bff577fffffffffcc2dff8faa5570000 */
 -3.54550084538495708816233114576143814E-34L, /* bf8fd74724576915868c1e8ce9f430f1 */
 -1.40380859374999215367421282192718062E-03L, /* bff56ffffffffffdbd0b18aac65ed000 */
 -1.90585907028351204486765167064669639E-34L, /* bf8efaaa0c0e23e50c11b2120348054f */
 -1.37329101562499692341771212945644892E-03L, /* bff567ffffffffff1cfd00f1b0577000 */
 -3.59631150411372589637918252836880320E-34L, /* bf8fde08239ac74942a46298ea4fb715 */
 -1.34277343749999137467356674296739172E-03L, /* bff55ffffffffffd839030b05d53d000 */
 -1.49571076125940368185068762485268117E-35L, /* bf8b3e1a3d5c684b27a9f835b1d8d3c9 */
 -1.31225585937499247038404301859788734E-03L, /* bff557fffffffffdd469936e691e3000 */
  3.10375845385355395586146533282311300E-34L, /* 3f8f9c8f6d63b7a4145716ffd92491fb */
 -1.28173828124999024755581675764821898E-03L, /* bff54ffffffffffd306589b0ab21d000 */
 -1.98541096105909793397376077900810019E-34L, /* bf8f07e808bbb1e35106c294ffbb9687 */
 -1.25122070312500340204619591143332523E-03L, /* bff5480000000000fb06d5f16ad2c000 */
  3.62884195935761446237911443317457521E-34L, /* 3f8fe25b17d623178a386a6fa6c5afb2 */
 -1.22070312499999591578388993012071279E-03L, /* bff53ffffffffffed2a356c440074000 */
 -2.96756662615653130862526710937493307E-35L, /* bf8c3b90d8ff2a991e5bd16718fb0645 */
 -1.19018554687498821966212632349422735E-03L, /* bff537fffffffffc9ac3b585dda89000 */
  1.44659971891167323357060028901142644E-34L, /* 3f8e809279ab249edf1dad9fe13fb0bf */
 -1.15966796875000160938908064907298384E-03L, /* bff530000000000076c0800db9639000 */
  2.50088010538742402346270685365928513E-34L, /* 3f8f4c6c8a483b60201d30c1a83c3cb7 */
 -1.12915039062500267151512523291939657E-03L, /* bff5280000000000c51f7e7315137000 */
  7.56402096465615210500092443924888831E-35L, /* 3f8d922c1e485d99aea2668ed32b55a6 */
 -1.09863281249998665006360103291051571E-03L, /* bff51ffffffffffc26f2d4c9ce2ba000 */
  1.43982174467233642713619821353592061E-34L, /* 3f8e7ec530b3d92b6303bec1c81214d1 */
 -1.06811523437500522742248711752028025E-03L, /* bff518000000000181b7380f10446000 */
  5.41265133745862349181293024531133174E-35L, /* 3f8d1fc9313d018b30e790e06b6be723 */
 -1.03759765624999980942114138999770552E-03L, /* bff50ffffffffffff1f01130490e1000 */
  1.21525139612685854366189534669623436E-34L, /* 3f8e4311b96b6fcde412caf3f0d86fb9 */
 -1.00708007812499602697537601515759439E-03L, /* bff507fffffffffedad7afcce7051000 */
  1.00020246351201558505328236381833392E-34L, /* 3f8e09e640992512b1300744a7e984ed */
 -9.76562499999992592487302113340463694E-04L, /* bff4fffffffffffbbad8151f8adf6000 */
 -1.64984406575162932060422892046851002E-34L, /* bf8eb69a919986e8054b86fc34300f24 */
 -9.46044921874989085824996924138179594E-04L, /* bff4effffffffff9b55a204fd9792000 */
 -9.29539174108308550334255350011347171E-35L, /* bf8dee3a50ed896b4656fa577a1df3d7 */
 -9.15527343750013735214860599791540029E-04L, /* bff4e00000000007eaf5bf103f82d000 */
  3.07557018309280519949818825519490586E-35L, /* 3f8c470cfbef77d32c74cb8042f6ee81 */
 -8.85009765625012292294986105781516428E-04L, /* bff4d000000000071605c65403b97000 */
  4.77499983783821950338363358545463558E-35L, /* 3f8cfbc3dc18884c4c4f9e07d90d7bd3 */
 -8.54492187499986941239470706817188192E-04L, /* bff4bffffffffff878ddf9cab264a000 */
 -1.60128240346239526958630011447901568E-34L, /* bf8ea9b1a21e19e2d5bd84b0fbffcf95 */
 -8.23974609374996290174598690241743810E-04L, /* bff4affffffffffddc86c249ebe06000 */
  1.61677540391961912631535763471935882E-34L, /* 3f8eadd00841366b0dc2bc262c2c8c36 */
 -7.93457031249988696952538334288757473E-04L, /* bff49ffffffffff97bf6f0aa85a5f000 */
  1.22318577008381887076634753347515709E-34L, /* 3f8e452db5b5d250878f71040da06d14 */
 -7.62939453124996723316499040007097041E-04L, /* bff48ffffffffffe1c7265b431108000 */
 -1.03845161748762410745671891558398468E-34L, /* bf8e14115ad884c96d1a820c73647220 */
 -7.32421874999998242520117923997325794E-04L, /* bff47ffffffffffefca4498b7aa8a000 */
  5.64005211953031009549514026639438083E-35L, /* 3f8d2be06950f68f1a6d8ff829a6928e */
 -7.01904296874999772890934814265622012E-04L, /* bff46fffffffffffde7c0fe5d8041000 */
  5.90245467325173644235991233229525762E-35L, /* 3f8d39d40cc49002189243c194b1db0e */
 -6.71386718750008699269643939210658742E-04L, /* bff460000000000503c91d798b60c000 */
 -5.20515801723324452151498579012322191E-35L, /* bf8d14c0f08a6a9285b32b8bda003eb5 */
 -6.40869140625005499535275057463709988E-04L, /* bff45000000000032b969184e9751000 */
 -6.69469163285461870099846471658294534E-35L, /* bf8d63f36bab7b24d936c9380e3d3fa6 */
 -6.10351562499999293780097329596079841E-04L, /* bff43fffffffffff97c7c433e35ed000 */
 -1.16941808547394177991845382085515086E-34L, /* bf8e36e27886f10b234a7dd8fc588bf0 */
 -5.79833984375000068291972326409994795E-04L, /* bff43000000000000a13ff6dcf2bf000 */
  1.17885044988246219185041488459766001E-34L, /* 3f8e3964677e001a00412aab52790842 */
 -5.49316406249990904622170867910987793E-04L, /* bff41ffffffffffac1c25739c716b000 */
 -3.31875702128137033065075734368960972E-35L, /* bf8c60e928d8982c3c99aef4f885a121 */
 -5.18798828125011293653756992177727236E-04L, /* bff410000000000682a62cff36775000 */
 -5.69971237642088463334239430962628187E-35L, /* bf8d2f0c76f8757d61cd1abc7ea7d066 */
 -4.88281249999990512232251384917893121E-04L, /* bff3fffffffffff50fb48992320df000 */
  1.02144616714408655325510171265051108E-35L, /* 3f8ab279a3626612710b9b3ac71734ac */
 -4.57763671874997554564967307956493434E-04L, /* bff3dffffffffffd2e3c272e3cca9000 */
 -8.25484058867957231164162481843653503E-35L, /* bf8db6e71158e7bf93e2e683f07aa841 */
 -4.27246093749991203999790346349633286E-04L, /* bff3bffffffffff5dbe103cba0eb2000 */
 -3.51191203319375193921924105905691755E-35L, /* bf8c757356d0f3dd7fbefc0dd419ab50 */
 -3.96728515624986649402960638705483281E-04L, /* bff39ffffffffff09b996882706ec000 */
 -5.51925962073095883016589497244931171E-36L, /* bf89d586d49f22289cfc860bebb99056 */
 -3.66210937499999945095511981300980754E-04L, /* bff37fffffffffffefcb88bfc7df6000 */
 -2.11696465278144529364423332249588595E-35L, /* bf8bc23a84d28e5496c874ef9833be25 */
 -3.35693359374992480958458008559640163E-04L, /* bff35ffffffffff754c548a8798f2000 */
 -8.58941791799705081104736787493668352E-35L, /* bf8dc8b1192fb7c3662826d43acb7c68 */
 -3.05175781250009811036303273640122156E-04L, /* bff340000000000b4fb4f1aad1c76000 */
 -8.61173897858769926480551302277426632E-35L, /* bf8dc9e0eabb1c0b33051011b64769fa */
 -2.74658203124987298321920308390303850E-04L, /* bff31ffffffffff15b2056ac252fd000 */
  3.35152809454778381053519808988046631E-37L, /* 3f85c82fb59ff8d7c80d44e635420ab1 */
 -2.44140624999999992770514819575735516E-04L, /* bff2fffffffffffffbbb82d6a7636000 */
  3.54445837111124472730013879165516908E-35L, /* 3f8c78e955b01378be647b1c92aa9a77 */
 -2.13623046875012756463165168672749438E-04L, /* bff2c0000000001d6a1635fea6bbf000 */
  1.50050816288650121729916777279129473E-35L, /* 3f8b3f1f6f616a61129a58e131cbd31d */
 -1.83105468749991323078784464300306893E-04L, /* bff27fffffffffebfe0cbd0c82399000 */
 -9.14919506501448661140572099029756008E-37L, /* bf873754bacaa9d9513b6127e791eb47 */
 -1.52587890625013337032336300236461546E-04L, /* bff240000000001ec0cb57f2cc995000 */
  2.84906084373176180870418394956384516E-35L, /* 3f8c2ef6d03a7e6ab087c4f099e4de89 */
 -1.22070312499990746786116828458007518E-04L, /* bff1ffffffffffd553bbb49f35a34000 */
  6.71618008964968339584520728412444537E-36L, /* 3f8a1dacb99c60071fc9cd2349495bf0 */
 -9.15527343750029275602791047595142231E-05L, /* bff180000000000d8040cd6ecde28000 */
 -1.95753652091078750312541716951402172E-35L, /* bf8ba0526cfb24d8d59122f1c7a09a14 */
 -6.10351562499913258461494008080572701E-05L, /* bff0ffffffffffaffebbb92d7f6a9000 */
  5.69868489273961111703398456218119973E-36L, /* 3f89e4ca5df09ef4a4386dd5b3bf0331 */
 -3.05175781250092882818419203884960853E-05L, /* bff0000000000055ab55de88fac1d000 */
  9.03341100018476837609128961872915953E-36L, /* 3f8a803d229fa3a0e834a63abb06662b */
#define T_EXPL_ARG2 (2*T_EXPL_ARG1 + 2 + 2*65)
  0.00000000000000000000000000000000000E+00L, /* 00000000000000000000000000000000 */
  0.00000000000000000000000000000000000E+00L, /* 00000000000000000000000000000000 */
  3.05175781249814607084128277672749162E-05L, /* 3feffffffffffeaa02abb9102f499000 */
  1.00271855391179733380665816525889949E-36L, /* 3f8755351afa042ac3f58114824d4c10 */
  6.10351562500179243748093427073421439E-05L, /* 3ff1000000000052a95de07a4c26d000 */
  1.67231624299180373502350811501181670E-36L, /* 3f881c87a53691cae9d77f4e40d66616 */
  9.15527343749970728685313252158399200E-05L, /* 3ff17ffffffffff28040cc2acde28000 */
  2.43665747834893104318707597514407880E-36L, /* 3f889e9366c7c6c6a2ecb78dc9b0509e */
  1.22070312500027751961838150070880064E-04L, /* 3ff200000000003ffddde6c153b53000 */
 -1.73322146370624186623546452226755405E-35L, /* bf8b709d8d658ed5dbbe943de56ee84e */
  1.52587890624995916105682628143179430E-04L, /* 3ff23ffffffffff6954b56e285d23000 */
  1.23580432650945898349135528000443828E-35L, /* 3f8b06d396601dde16de7d7bc27346e6 */
  1.83105468750008670314358488289621794E-04L, /* 3ff2800000000013fe0cdc8c823b7000 */
  4.30446229148833293310207915930740796E-35L, /* 3f8cc9ba9bfe554a4f7f2fece291eb23 */
  2.13623046875005741337455947623248132E-04L, /* 3ff2c0000000000d3d1662de21a3f000 */
 -3.96110759869520786681660669615255057E-35L, /* bf8ca5379b04ff4a31aab0ceacc917e6 */
  2.44140624999981493573336463433440506E-04L, /* 3ff2ffffffffffd553bbdf48e0534000 */
 -1.39617373942387888957350179316792928E-35L, /* bf8b28eeedc286015802b63f96b8c5cd */
  2.74658203124984920706309918754626834E-04L, /* 3ff31fffffffffee9d60c8439ec1d000 */
 -3.16168080483901830349738314447356223E-36L, /* bf890cf74f81c77a611abc1243812444 */
  3.05175781250008648918265055410966055E-04L, /* 3ff3400000000009f8b5c9a346636000 */
  8.54421306185008998867856704677221443E-35L, /* 3f8dc649cd40922fc08adc6b6b20ead0 */
  3.35693359374988945462612499316774515E-04L, /* 3ff35ffffffffff34146c540f15b2000 */
  7.96443137431639500475160850431097078E-35L, /* 3f8da77638ed3148fc4d99d1c9e13446 */
  3.66210937500027690542093987739604535E-04L, /* 3ff380000000001fecce34bea89c4000 */
  2.14507323877752361258862577769090367E-35L, /* 3f8bc834e554d38894cf91957b0253d3 */
  3.96728515625003928083564943615052121E-04L, /* 3ff3a00000000004875d9a4acf6ab000 */
  4.88358523466632050664019922448605508E-35L, /* 3f8d03a7eaeef1a9f78c71a12c44dd28 */
  4.27246093750017799227172345607351585E-04L, /* 3ff3c00000000014856794c3ee850000 */
  6.66520494592631402182216588784828935E-35L, /* 3f8d6262118fcdb59b8f16108f5f1a6c */
  4.57763671875002108342364320152138181E-04L, /* 3ff3e000000000026e45d855410b9000 */
  7.21799615960261390920033272189522298E-35L, /* 3f8d7fc645cff8879462296af975c9fd */
  4.88281249999999768797631616370963356E-04L, /* 3ff3ffffffffffffbbc2d7cc004df000 */
 -5.30564629906905979452258114088325361E-35L, /* bf8d1a18b71929a30d67a217a27ae851 */
  5.18798828124997339054881383202487041E-04L, /* 3ff40ffffffffffe775055eea5851000 */
 -4.03682911253647925867848180522846377E-35L, /* bf8cad44f0f3e5199d8a589d9332acad */
  5.49316406249980511907933706754958501E-04L, /* 3ff41ffffffffff4c410b29bb62fb000 */
 -2.08166843948323917121806956728438051E-35L, /* bf8bbab8cf691403249fe5b699e25143 */
  5.79833984374989593561576568548497165E-04L, /* 3ff42ffffffffffa0047df328d817000 */
 -1.72745033420153042445343706432627539E-34L, /* bf8ecb3c2d7d3a9e6e960576be901fdf */
  6.10351562500008540711511259540838154E-04L, /* 3ff4400000000004ec62f54f8c271000 */
  7.41889382604319545724663095428976499E-35L, /* 3f8d8a74c002c81a47c93b8e05d15f8e */
  6.40869140625020444702875407535884986E-04L, /* 3ff450000000000bc91b09718515d000 */
 -4.47321009727305792048065440180490107E-35L, /* bf8cdbac5c8fe70822081d8993eb5cb6 */
  6.71386718750007531635964622352684074E-04L, /* 3ff460000000000457792973db05c000 */
  5.13698959677949336513874456684462092E-35L, /* 3f8d112114436949c5ef38d8049004ab */
  7.01904296875006634673332887754430334E-04L, /* 3ff4700000000003d31adf2cb8b1d000 */
 -8.25665755717729437292989870760751482E-35L, /* bf8db6ffcc8ef71f8e648e3a8b160f5a */
  7.32421874999998244664170215504673504E-04L, /* 3ff47ffffffffffefcf5498bd5c8a000 */
 -5.64005234937832153139057628112753364E-35L, /* bf8d2be06a1dfe90e7bf90fba7c12a98 */
  7.62939453125017456345986752604096408E-04L, /* 3ff490000000000a101a1b093d4a8000 */
 -1.11084094120417622468550608896588329E-34L, /* bf8e274feabd2d94f6694507a46accb1 */
  7.93457031249987558617598988993908016E-04L, /* 3ff49ffffffffff8d3f9dcab74bbf000 */
 -1.22966480225449015129079129940978828E-34L, /* bf8e46e6a65eef8fa9e42eddf3da305e */
  8.23974609374997378723747633335135819E-04L, /* 3ff4affffffffffe7d2afbaa55b26000 */
 -1.62270010016794279091906973366704963E-34L, /* bf8eaf633f057ebdb664a34566401c4e */
  8.54492187500023938282350821569920958E-04L, /* 3ff4c0000000000dccaabce399e59000 */
 -1.39076361712838158775374263169606160E-34L, /* bf8e71ba779364b3bbdba7841f2c4ca1 */
  8.85009765624987932362186815286691297E-04L, /* 3ff4cffffffffff90b218886edc2a000 */
  4.07328275060905585228261577392403980E-35L, /* 3f8cb1254dbb6ea4b8cfa5ed4cf28d24 */
  9.15527343749975579461305518559161974E-04L, /* 3ff4dffffffffff1ec2a21f25df33000 */
  1.16855112459192484947855553716334015E-35L, /* 3f8af10bf319e9f5270cf249eeffbe5c */
  9.46044921875016761584725882821122521E-04L, /* 3ff4f00000000009a992c46c16d71000 */
  9.51660680007524262741115611071680436E-35L, /* 3f8df9fd56e81f8edf133843910ee831 */
  9.76562499999974118878133088548272636E-04L, /* 3ff4fffffffffff1149edc46a6df6000 */
 -5.65271128977550656964071208289181661E-36L, /* bf89e0e12689dd721aa2314c81eb6429 */
  1.00708007812498671732140389760347830E-03L, /* 3ff507fffffffffc2be94b90ed091000 */
 -1.43355074891483635310132767255371379E-34L, /* bf8e7d1a688c247b16022daab1316d55 */
  1.03759765625002637786192745235343007E-03L, /* 3ff51000000000079a57b966bc158000 */
  2.95905815240957629366749917020106928E-34L, /* 3f8f895387fc73bb38f8a1b254c01a60 */
  1.06811523437500860568717813047520763E-03L, /* 3ff51800000000027afcd5b35f5e6000 */
 -5.98328495358586628195372356742878314E-35L, /* bf8d3e204130013bf6328f1b70ff8c76 */
  1.09863281250001439958487251556220070E-03L, /* 3ff5200000000004268077c6c66bd000 */
  2.41371837889426603334113000868144760E-34L, /* 3f8f40d6948edf864054ccf151f9815e */
  1.12915039062501298413451613770002366E-03L, /* 3ff5280000000003be0f5dd8fe81b000 */
 -1.28815268997394164973472617519705703E-34L, /* bf8e567321172ea089dce4bc8354ecb7 */
  1.15966796874997272036339054191407232E-03L, /* 3ff52ffffffffff8231e3bcfff1e8000 */
  1.02996064554316248496839462594377804E-34L, /* 3f8e11cf7d402789244f68e2d4f985b1 */
  1.19018554687502744121802585360546796E-03L, /* 3ff5380000000007e8cdf3f8f6c20000 */
 -1.43453217726255628994625761307322163E-34L, /* bf8e7d5d3370d85a374f5f4802fc517a */
  1.22070312499997743541996266398850614E-03L, /* 3ff53ffffffffff97f0722561f454000 */
 -1.41086259180534339713692694428211646E-34L, /* bf8e77125519ff76244dfec5fbd58402 */
  1.25122070312501024092560690174507039E-03L, /* 3ff5480000000002f3a59d8820691000 */
  3.84102646020099293168698506729765213E-34L, /* 3f8ffe8f5b86f9c3569c8f26e19b1f50 */
  1.28173828124997986521442660131425390E-03L, /* 3ff54ffffffffffa3250a764439d9000 */
  1.44644589735033114377952806106652650E-34L, /* 3f8e808801b80dcf38323cdbfdca2549 */
  1.31225585937501665804856968749058137E-03L, /* 3ff5580000000004cd25a414c6d62000 */
  1.67474574742200577294563576414361377E-34L, /* 3f8ebd394a151dbda4f81d5d83c0f1e9 */
  1.34277343749997290265837386401818888E-03L, /* 3ff55ffffffffff83091b042cfd59000 */
 -1.55650565030381326742591837551559103E-34L, /* bf8e9dca490d7fecfadba9625ffb91c5 */
  1.37329101562497720784949380297774268E-03L, /* 3ff567fffffffff96e3c7312f5ccf000 */
  1.65279335325630026116581677369221748E-34L, /* 3f8eb763496f5bd7404f2298b402074f */
  1.40380859374999099958354100336136647E-03L, /* 3ff56ffffffffffd67e2f09f2a381000 */
  1.89919944388961890195706641264717076E-34L, /* 3f8ef8e4d0ffdfeba982aa8829501389 */
  1.43432617187497484122173130998160625E-03L, /* 3ff577fffffffff8bf9c1d71af8a8000 */
  2.57638517142061429772064578590009568E-34L, /* 3f8f5675d82c1cc4ada70fd3a957b89a */
  1.46484374999999929342158925502052945E-03L, /* 3ff57fffffffffffcbdd1c7671b46000 */
  1.76487201934184070490166772482073801E-34L, /* 3f8ed52ef732458f6e4c5c07504f33cc */
  1.49536132812502318451070466256902933E-03L, /* 3ff5880000000006aeb7066c8ad43000 */
  2.38068367275295804321313550609246656E-34L, /* 3f8f3c7277ae6fc390ace5e06c0b025b */
  1.52587890625000448053340248672949543E-03L, /* 3ff59000000000014a9ae2104b3bc000 */
  1.01174455568392813258454590274740959E-34L, /* 3f8e0cf7c434762991bb38e12acee215 */
  1.55639648437501113499837053523090913E-03L, /* 3ff5980000000003359e2c204355e000 */
 -2.82398418808099749023517211651363693E-35L, /* bf8c2c4c2971d88caa95e15fb1ccb1a1 */
  1.58691406249999937955142588308171026E-03L, /* 3ff59fffffffffffd2380ecbc87c2000 */
 -1.27361695572422741562701199136538047E-34L, /* bf8e5295e0e206dfb0f0266c07225448 */
  1.61743164062498000531048954475329309E-03L, /* 3ff5a7fffffffffa3ca6fe61ed94c000 */
 -1.22606548862580061633942923016222044E-34L, /* bf8e45f1b17bb61039d21a351bb207b8 */
  1.64794921875001835451453858682255576E-03L, /* 3ff5b000000000054a52fa20f6565000 */
  1.39132339594152335892305491425264583E-34L, /* 3f8e71e0904c5449b414ee49b191cef2 */
  1.67846679687501263995029340691547953E-03L, /* 3ff5b80000000003a4a9e912c910b000 */
  6.67245854693585315412242764786197029E-35L, /* 3f8d62c4ccac1e7511a617d469468ccd */
  1.70898437500002646861403514115369655E-03L, /* 3ff5c00000000007a109fbaa7e015000 */
  6.87367172354719289559624829652240928E-36L, /* 3f8a245fa835eceb42bae8128d9336db */
  1.73950195312501174308226096992992128E-03L, /* 3ff5c80000000003627c8d637a005000 */
 -2.20824271875474985927385878948759352E-34L, /* bf8f25869b1cbefb25e735992f232f57 */
  1.77001953124997491747605207736194513E-03L, /* 3ff5cffffffffff8c53c84b6883b8000 */
  3.43123048533596296514343180408963705E-34L, /* 3f8fc816b91d173ddadbbf09b1287906 */
  1.80053710937497698911127570705069398E-03L, /* 3ff5d7fffffffff95e1899f4a8430000 */
  3.99231237340890073475077494556136100E-35L, /* 3f8ca889148f62fa854da5674df41279 */
  1.83105468750002267094899598630423914E-03L, /* 3ff5e0000000000688d21e62ba674000 */
 -3.22274595655810623999007524769365273E-34L, /* bf8fac605cb9ae01eb719675ced25560 */
  1.86157226562500499224728040579690330E-03L, /* 3ff5e80000000001705ce28a6d89e000 */
  3.07094985075881613489605622068441083E-34L, /* 3f8f98330225ec7e2c8f3c0d1c432b91 */
  1.89208984374998234666824993196980949E-03L, /* 3ff5effffffffffae969fdc7cd8cf000 */
 -3.06287628722973914692165056776495733E-34L, /* bf8f9720477d9cfa10e464df7f91020c */
  1.92260742187501225343755557292811682E-03L, /* 3ff5f800000000038824e428ed49a000 */
  6.30049124729794620592961282769623368E-35L, /* 3f8d4efdd7cd4336d88a6aa49e1e96bc */
  1.95312499999998514894032051116231258E-03L, /* 3ff5fffffffffffbb82f6a04f1ae0000 */
 -6.14610057507500948543216998736262902E-35L, /* bf8d46c862d39255370e7974d48daa7e */
  1.98364257812501222021119324146882732E-03L, /* 3ff6040000000001c2d8a1aa5188d000 */
  3.71942298418113774118754986159801984E-34L, /* 3f8fee6567d9940495519ffe62cbc9a4 */

  7.06341639425619532977052017486130353E-01L, /* 3ffe69a59c8245a9ac00000000000000 */
  7.09106182437398424589503065362805501E-01L, /* 3ffe6b0ff72deb89d000000000000000 */
  7.11881545564596485142772053222870454E-01L, /* 3ffe6c7bbce9a6d93000000000000000 */
  7.14667771155948150507697391731198877E-01L, /* 3ffe6de8ef213d71e000000000000000 */
  7.17464901725936049503573599395167548E-01L, /* 3ffe6f578f41e1a9e400000000000000 */
  7.20272979955439790478166628417966422E-01L, /* 3ffe70c79eba33c06c00000000000000 */
  7.23092048692387218133958981525211129E-01L, /* 3ffe72391efa434c7400000000000000 */
  7.25922150952408251622927082280511968E-01L, /* 3ffe73ac117390acd800000000000000 */
  7.28763329919491220643124052003258839E-01L, /* 3ffe752077990e79d000000000000000 */
  7.31615628946641782803794740175362676E-01L, /* 3ffe769652df22f7e000000000000000 */
  7.34479091556544505525749855223693885E-01L, /* 3ffe780da4bba98c4800000000000000 */
  7.37353761442226890432394270646909717E-01L, /* 3ffe79866ea5f432d400000000000000 */
  7.40239682467726090031590047146892175E-01L, /* 3ffe7b00b216ccf53000000000000000 */
  7.43136898668758316688354170764796436E-01L, /* 3ffe7c7c70887763c000000000000000 */
  7.46045454253390638577059235103661194E-01L, /* 3ffe7df9ab76b20fd000000000000000 */
  7.48965393602715662213498148958024103E-01L, /* 3ffe7f78645eb8076400000000000000 */
  7.51896761271528629722027403659012634E-01L, /* 3ffe80f89cbf42526400000000000000 */
  7.54839601989007347171423134568613023E-01L, /* 3ffe827a561889716000000000000000 */
  7.57793960659394638668118204805068672E-01L, /* 3ffe83fd91ec46ddc000000000000000 */
  7.60759882362683631518152083117456641E-01L, /* 3ffe858251bdb68b8c00000000000000 */
  7.63737412355305483879774897104653064E-01L, /* 3ffe87089711986c9400000000000000 */
  7.66726596070820082262642358728044201E-01L, /* 3ffe8890636e31f54400000000000000 */
  7.69727479120609181517664865168626420E-01L, /* 3ffe8a19b85b4fa2d800000000000000 */
  7.72740107294572486917871856348938309E-01L, /* 3ffe8ba4976246833800000000000000 */
  7.75764526561826289752232810315035749E-01L, /* 3ffe8d31020df5be4400000000000000 */
  7.78800783071404878477039801509818062E-01L, /* 3ffe8ebef9eac820b000000000000000 */
  7.81848923152964780936002853195532225E-01L, /* 3ffe904e8086b5a87800000000000000 */
  7.84908993317491698871180005880887620E-01L, /* 3ffe91df97714512d800000000000000 */
  7.87981040258010162480317717381694820E-01L, /* 3ffe9372403b8d6bcc00000000000000 */
  7.91065110850296016042904057030682452E-01L, /* 3ffe95067c78379f2800000000000000 */
  7.94161252153591734614934694036492147E-01L, /* 3ffe969c4dbb800b4800000000000000 */
  7.97269511411324433014513601847284008E-01L, /* 3ffe9833b59b38154400000000000000 */
  8.00389936051826789142893403550260700E-01L, /* 3ffe99ccb5aec7bec800000000000000 */
  8.03522573689060742863077280162542593E-01L, /* 3ffe9b674f8f2f3d7c00000000000000 */
  8.06667472123343942680406826184480451E-01L, /* 3ffe9d0384d70893f800000000000000 */
  8.09824679342079301047618855591281317E-01L, /* 3ffe9ea15722892c7800000000000000 */
  8.12994243520486992160556383169023320E-01L, /* 3ffea040c80f8374f000000000000000 */
  8.16176213022339780422953481320291758E-01L, /* 3ffea1e1d93d687d0000000000000000 */
  8.19370636400700819157449927843117621E-01L, /* 3ffea3848c4d49954c00000000000000 */
  8.22577562398664585696650419777142815E-01L, /* 3ffea528e2e1d9f09800000000000000 */
  8.25797039950100647542896581398963463E-01L, /* 3ffea6cede9f70467c00000000000000 */
  8.29029118180400342863478613253391813E-01L, /* 3ffea876812c0877bc00000000000000 */
  8.32273846407226292054559735333896242E-01L, /* 3ffeaa1fcc2f45343800000000000000 */
  8.35531274141265073440720811959181447E-01L, /* 3ffeabcac15271a2a400000000000000 */
  8.38801451086982535754188461396552157E-01L, /* 3ffead7762408309bc00000000000000 */
  8.42084427143382358016410194068157580E-01L, /* 3ffeaf25b0a61a7b4c00000000000000 */
  8.45380252404767357221615498019673396E-01L, /* 3ffeb0d5ae318680c400000000000000 */
  8.48688977161503960155997106085123960E-01L, /* 3ffeb2875c92c4c99400000000000000 */
  8.52010651900789478530029441571969073E-01L, /* 3ffeb43abd7b83db1c00000000000000 */
  8.55345327307422548246407245642330963E-01L, /* 3ffeb5efd29f24c26400000000000000 */
  8.58693054264576483003423845730139874E-01L, /* 3ffeb7a69db2bcc77800000000000000 */
  8.62053883854575708767242758767679334E-01L, /* 3ffeb95f206d17228000000000000000 */
  8.65427867359675251357487013592617586E-01L, /* 3ffebb195c86b6b29000000000000000 */
  8.68815056262843166123843730019871145E-01L, /* 3ffebcd553b9d7b62000000000000000 */
  8.72215502248546159513864495238522068E-01L, /* 3ffebe9307c271855000000000000000 */
  8.75629257203538208242932228131394368E-01L, /* 3ffec0527a5e384ddc00000000000000 */
  8.79056373217652342599848225290770642E-01L, /* 3ffec213ad4c9ed0d800000000000000 */
  8.82496902584595399599010079327854328E-01L, /* 3ffec3d6a24ed8221800000000000000 */
  8.85950897802745995779361010136199184E-01L, /* 3ffec59b5b27d9696800000000000000 */
  8.89418411575955636383383762222365476E-01L, /* 3ffec761d99c5ba58800000000000000 */
  8.92899496814352794382685374330321793E-01L, /* 3ffec92a1f72dd70d400000000000000 */
  8.96394206635150403439382671422208659E-01L, /* 3ffecaf42e73a4c7d800000000000000 */
  8.99902594363456265202927397695020773E-01L, /* 3ffeccc00868c0d18800000000000000 */
  9.03424713533086704009278378180169966E-01L, /* 3ffece8daf1e0ba94c00000000000000 */
  9.06960617887383580004723171441582963E-01L, /* 3ffed05d24612c2af000000000000000 */
  9.10510361380034133338412516422977205E-01L, /* 3ffed22e6a0197c02c00000000000000 */
  9.14073998175894436579724811053893063E-01L, /* 3ffed40181d094303400000000000000 */
  9.17651582651815816982221463149471674E-01L, /* 3ffed5d66da13970f400000000000000 */
  9.21243169397474526149949269893113524E-01L, /* 3ffed7ad2f48737a2000000000000000 */
  9.24848813216204823639543519675498828E-01L, /* 3ffed985c89d041a3000000000000000 */
  9.28468569125835141431224428743007593E-01L, /* 3ffedb603b7784cd1800000000000000 */
  9.32102492359527579068867453315760940E-01L, /* 3ffedd3c89b26894e000000000000000 */
  9.35750638366620729469147477175283711E-01L, /* 3ffedf1ab529fdd41c00000000000000 */
  9.39413062813475779888605643463961314E-01L, /* 3ffee0fabfbc702a3c00000000000000 */
  9.43089821584325888048638830696290825E-01L, /* 3ffee2dcab49ca51b400000000000000 */
  9.46780970782128888929563004239753354E-01L, /* 3ffee4c079b3f8000400000000000000 */
  9.50486566729423443256052905780961737E-01L, /* 3ffee6a62cdec7c7b000000000000000 */
  9.54206665969188322362626308859034907E-01L, /* 3ffee88dc6afecfbfc00000000000000 */
  9.57941325265705301283958306157728657E-01L, /* 3ffeea77490f0196b000000000000000 */
  9.61690601605425299247542625380447134E-01L, /* 3ffeec62b5e5881fb000000000000000 */
  9.65454552197837823079851204965962097E-01L, /* 3ffeee500f1eed967000000000000000 */
  9.69233234476344074348475032820715569E-01L, /* 3ffef03f56a88b5d7800000000000000 */
  9.73026706099133165128733935489435680E-01L, /* 3ffef2308e71a927a800000000000000 */
  9.76835024950062025261843245971249416E-01L, /* 3ffef423b86b7ee79000000000000000 */
  9.80658249139538557015427500118676107E-01L, /* 3ffef618d68936c09c00000000000000 */
  9.84496437005408397968864164795377292E-01L, /* 3ffef80feabfeefa4800000000000000 */
  9.88349647113845042323276857132441364E-01L, /* 3ffefa08f706bbf53800000000000000 */
  9.92217938260243514925207364285597578E-01L, /* 3ffefc03fd56aa225000000000000000 */
  9.96101369470117486981664001177705359E-01L, /* 3ffefe00ffaabffbbc00000000000000 */
#define T_EXPL_RES1 (T_EXPL_ARG2 + 2 + 2*65 + 89)
  1.00000000000000000000000000000000000E+00L, /* 3fff0000000000000000000000000000 */
  1.00391388933834757590801700644078664E+00L, /* 3fff0100802ab5577800000000000000 */
  1.00784309720644799091004983893071767E+00L, /* 3fff0202015600445c00000000000000 */
  1.01178768355933151879000320150225889E+00L, /* 3fff0304848362076c00000000000000 */
  1.01574770858668572692806719715008512E+00L, /* 3fff04080ab55de39000000000000000 */
  1.01972323271377413034244341361045372E+00L, /* 3fff050c94ef7a206c00000000000000 */
  1.02371431660235789884438872832106426E+00L, /* 3fff06122436410dd000000000000000 */
  1.02772102115162167201845022646011785E+00L, /* 3fff0718b98f42085000000000000000 */
  1.03174340749910264936062276319717057E+00L, /* 3fff08205601127ec800000000000000 */
  1.03578153702162378824169763902318664E+00L, /* 3fff0928fa934ef90800000000000000 */
  1.03983547133622999947277776300325058E+00L, /* 3fff0a32a84e9c1f5800000000000000 */
  1.04390527230112850620713516036630608E+00L, /* 3fff0b3d603ca7c32800000000000000 */
  1.04799100201663270004459604933799710E+00L, /* 3fff0c49236829e8bc00000000000000 */
  1.05209272282610977189420964350574650E+00L, /* 3fff0d55f2dce5d1e800000000000000 */
  1.05621049731693195106174698594259098E+00L, /* 3fff0e63cfa7ab09d000000000000000 */
  1.06034438832143151909548350886325352E+00L, /* 3fff0f72bad65671b800000000000000 */
  1.06449445891785943185681162503897212E+00L, /* 3fff1082b577d34ed800000000000000 */
  1.06866077243134810492719566354935523E+00L, /* 3fff1193c09c1c595c00000000000000 */
  1.07284339243487741866189821848820429E+00L, /* 3fff12a5dd543ccc4c00000000000000 */
  1.07704238275024494209120007326419000E+00L, /* 3fff13b90cb25176a400000000000000 */
  1.08125780744903959851299646288680378E+00L, /* 3fff14cd4fc989cd6400000000000000 */
  1.08548973085361949442173568058933597E+00L, /* 3fff15e2a7ae28fecc00000000000000 */
  1.08973821753809324563988525369495619E+00L, /* 3fff16f9157587069400000000000000 */
  1.09400333232930546678574046381982043E+00L, /* 3fff18109a3611c35000000000000000 */
  1.09828514030782586896606289883493446E+00L, /* 3fff192937074e0cd800000000000000 */
  1.10258370680894224324930519287590869E+00L, /* 3fff1a42ed01d8cbc800000000000000 */
  1.10689909742365749645287564817408565E+00L, /* 3fff1b5dbd3f68122400000000000000 */
  1.11123137799969046168868658241990488E+00L, /* 3fff1c79a8dacc350c00000000000000 */
  1.11558061464248076122274255794764031E+00L, /* 3fff1d96b0eff0e79400000000000000 */
  1.11994687371619722204840741142106708E+00L, /* 3fff1eb4d69bde569c00000000000000 */
  1.12433022184475073235176978414529003E+00L, /* 3fff1fd41afcba45e800000000000000 */
  1.12873072591281087273529237791080959E+00L, /* 3fff20f47f31c92e4800000000000000 */
  1.13314845306682632219974493636982515E+00L, /* 3fff2216045b6f5cd000000000000000 */
  1.13758347071604959399593326452304609E+00L, /* 3fff2338ab9b32134800000000000000 */
  1.14203584653356560174586320499656722E+00L, /* 3fff245c7613b8a9b000000000000000 */
  1.14650564845732405583333957110880874E+00L, /* 3fff258164e8cdb0d800000000000000 */
  1.15099294469117646722011727433709893E+00L, /* 3fff26a7793f60164400000000000000 */
  1.15549780370591653744227755851170514E+00L, /* 3fff27ceb43d84490400000000000000 */
  1.16002029424032515603215642840950750E+00L, /* 3fff28f7170a755fd800000000000000 */
  1.16456048530221917269855680387991015E+00L, /* 3fff2a20a2ce96406400000000000000 */
  1.16911844616950438835445424956560601E+00L, /* 3fff2b4b58b372c79400000000000000 */
  1.17369424639123270948104504896036815E+00L, /* 3fff2c7739e3c0f32c00000000000000 */
  1.17828795578866324378353169777255971E+00L, /* 3fff2da4478b620c7400000000000000 */
  1.18289964445632783673900689791480545E+00L, /* 3fff2ed282d763d42400000000000000 */
  1.18752938276310060494722620205720887E+00L, /* 3fff3001ecf601af7000000000000000 */
  1.19217724135327157730657177125976887E+00L, /* 3fff31328716a5d63c00000000000000 */
  1.19684329114762477708211463323095813E+00L, /* 3fff32645269ea829000000000000000 */
  1.20152760334452030077656559114984702E+00L, /* 3fff339750219b212c00000000000000 */
  1.20623024942098072687102217059873510E+00L, /* 3fff34cb8170b5835400000000000000 */
  1.21095130113378179892436037334846333E+00L, /* 3fff3600e78b6b11d000000000000000 */
  1.21569083052054743854242246925423387E+00L, /* 3fff373783a722012400000000000000 */
  1.22044890990084875515009343871497549E+00L, /* 3fff386f56fa7686e800000000000000 */
  1.22522561187730755216662714701669756E+00L, /* 3fff39a862bd3c106400000000000000 */
  1.23002100933670455162882717559114099E+00L, /* 3fff3ae2a8287e7a8000000000000000 */
  1.23483517545109100499445276000187732E+00L, /* 3fff3c1e2876834aa800000000000000 */
  1.23966818367890557750499169742397498E+00L, /* 3fff3d5ae4e2cae92c00000000000000 */
  1.24452010776609517384017067342938390E+00L, /* 3fff3e98deaa11dcbc00000000000000 */
  1.24939102174724003813111039562500082E+00L, /* 3fff3fd8170a52071800000000000000 */
  1.25428099994668373895478907797951251E+00L, /* 3fff41188f42c3e32000000000000000 */
  1.25919011697966698459794088194030337E+00L, /* 3fff425a4893dfc3f800000000000000 */
  1.26411844775346637881341393949696794E+00L, /* 3fff439d443f5f159000000000000000 */
  1.26906606746853711786826579555054195E+00L, /* 3fff44e183883d9e4800000000000000 */
  1.27403305161966090564007458851847332E+00L, /* 3fff462707b2bac20c00000000000000 */
  1.27901947599709753244923149395617656E+00L, /* 3fff476dd2045ac67800000000000000 */
  1.28402541668774150540599521264084615E+00L, /* 3fff48b5e3c3e8186800000000000000 */
  1.28905095007628295311619126550795045E+00L, /* 3fff49ff3e397492bc00000000000000 */
  1.29409615284637330434591717676084954E+00L, /* 3fff4b49e2ae5ac67400000000000000 */
  1.29916110198179535206719492634874769E+00L, /* 3fff4c95d26d3f440800000000000000 */
  1.30424587476763775839572190307080746E+00L, /* 3fff4de30ec211e60000000000000000 */
  1.30935054879147461104338390214252286E+00L, /* 3fff4f3198fa0f1cf800000000000000 */
  1.31447520194454914310711046709911898E+00L, /* 3fff50817263c13cd000000000000000 */
  1.31961991242296217130558488861424848E+00L, /* 3fff51d29c4f01cb3000000000000000 */
  1.32478475872886558573071624778094701E+00L, /* 3fff5325180cfacf7800000000000000 */
  1.32996981967165983640200010995613411E+00L, /* 3fff5478e6f02823d000000000000000 */
  1.33517517436919680440254865061433520E+00L, /* 3fff55ce0a4c58c7bc00000000000000 */
  1.34040090224898678084031189428060316E+00L, /* 3fff57248376b033d800000000000000 */
  1.34564708304941055283521222918352578E+00L, /* 3fff587c53c5a7af0400000000000000 */
  1.35091379682093615244298234756570309E+00L, /* 3fff59d57c910fa4e000000000000000 */
  1.35620112392734021300455538039386738E+00L, /* 3fff5b2fff3210fd9400000000000000 */
  1.36150914504693443252136830778908916E+00L, /* 3fff5c8bdd032e770800000000000000 */
  1.36683794117379636690046140756749082E+00L, /* 3fff5de9176045ff5400000000000000 */
  1.37218759361900544124779344201670028E+00L, /* 3fff5f47afa69210a800000000000000 */
  1.37755818401188367960941150158760138E+00L, /* 3fff60a7a734ab0e8800000000000000 */
  1.38294979430124120867162673675920814E+00L, /* 3fff6208ff6a88a46000000000000000 */
  1.38836250675662681297595213436579797E+00L, /* 3fff636bb9a983258400000000000000 */
  1.39379640396958309755959248832368758E+00L, /* 3fff64cfd75454ee7c00000000000000 */
  1.39925156885490681313299887733592186E+00L, /* 3fff663559cf1bc7c400000000000000 */
  1.40472808465191417726103395580139477E+00L, /* 3fff679c427f5a49f400000000000000 */
  1.41022603492571069194738697660795879E+00L, /* 3fff690492cbf9432c00000000000000 */
  1.41574550356846662335641440222389065E+00L, /* 3fff6a6e4c1d491e1800000000000000 */

  9.98018323540573404351050612604012713E-01L, /* 3ffefefc41f8d4bdb000000000000000 */
  9.98048781107475468932221929208026268E-01L, /* 3ffeff003ff556aa8800000000000000 */
  9.98079239603882895082165305211674422E-01L, /* 3ffeff043df9d4986000000000000000 */
  9.98109699029824021243584297735651489E-01L, /* 3ffeff083c064e972c00000000000000 */
  9.98140159385327269125909310787392315E-01L, /* 3ffeff0c3a1ac4b6ec00000000000000 */
  9.98170620670420977171843901487591211E-01L, /* 3ffeff10383737079400000000000000 */
  9.98201082885133511579667242585856002E-01L, /* 3ffeff14365ba5991c00000000000000 */
  9.98231546029493238547658506831794512E-01L, /* 3ffeff183488107b7c00000000000000 */
  9.98262010103528552029672482603928074E-01L, /* 3ffeff1c32bc77beb000000000000000 */
  9.98292475107267818223988342651864514E-01L, /* 3ffeff2030f8db72b000000000000000 */
  9.98322941040739375573309644096298143E-01L, /* 3ffeff242f3d3ba77000000000000000 */
  9.98353407903971645787066790944663808E-01L, /* 3ffeff282d89986cf000000000000000 */
  9.98383875696992967307963340317655820E-01L, /* 3ffeff2c2bddf1d32400000000000000 */
  9.98414344419831761845429696222709026E-01L, /* 3ffeff302a3a47ea0c00000000000000 */
  9.98444814072516340086593800151604228E-01L, /* 3ffeff34289e9ac19800000000000000 */
  9.98475284655075123740886056111776270E-01L, /* 3ffeff38270aea69c800000000000000 */
  9.98505756167536479006585636852832977E-01L, /* 3ffeff3c257f36f29400000000000000 */
  9.98536228609928799837547330753295682E-01L, /* 3ffeff4023fb806bf800000000000000 */
  9.98566701982280452432050310562772211E-01L, /* 3ffeff44227fc6e5ec00000000000000 */
  9.98597176284619802988373749030870385E-01L, /* 3ffeff48210c0a706800000000000000 */
  9.98627651516975245460372434536111541E-01L, /* 3ffeff4c1fa04b1b6800000000000000 */
  9.98658127679375173801901155457017012E-01L, /* 3ffeff501e3c88f6e800000000000000 */
  9.98688604771847954211239084543194622E-01L, /* 3ffeff541ce0c412e000000000000000 */
  9.98719082794421980642241010173165705E-01L, /* 3ffeff581b8cfc7f4c00000000000000 */
  9.98749561747125619293186105096538085E-01L, /* 3ffeff5c1a41324c2400000000000000 */
  9.98780041629987291873504773320746608E-01L, /* 3ffeff6018fd65896800000000000000 */
  9.98810522443035364581476187595399097E-01L, /* 3ffeff6417c196471000000000000000 */
  9.98841004186298203615379520670103375E-01L, /* 3ffeff68168dc4951400000000000000 */
  9.98871486859804230684645176552294288E-01L, /* 3ffeff6c1561f0837400000000000000 */
  9.98901970463581839743127943620493170E-01L, /* 3ffeff70143e1a222c00000000000000 */
  9.98932454997659369233531378995394334E-01L, /* 3ffeff74132241813000000000000000 */
  9.98962940462065268620861502313346136E-01L, /* 3ffeff78120e66b08400000000000000 */
  9.98993426856827904103397486323956400E-01L, /* 3ffeff7c110289c02000000000000000 */
  9.99023914181975669634994119405746460E-01L, /* 3ffeff800ffeaac00000000000000000 */
  9.99054402437536959169506189937237650E-01L, /* 3ffeff840f02c9c02000000000000000 */
  9.99084891623540138905212870668037795E-01L, /* 3ffeff880e0ee6d07800000000000000 */
  9.99115381740013658307120181234495249E-01L, /* 3ffeff8c0d2302010c00000000000000 */
  9.99145872786985911329082910015131347E-01L, /* 3ffeff900c3f1b61d800000000000000 */
  9.99176364764485236413804614130640402E-01L, /* 3ffeff940b633302d000000000000000 */
  9.99206857672540083026291313217370771E-01L, /* 3ffeff980a8f48f3f800000000000000 */
  9.99237351511178817364822180024930276E-01L, /* 3ffeff9c09c35d454800000000000000 */
  9.99267846280429861138827618560753763E-01L, /* 3ffeffa008ff7006c000000000000000 */
  9.99298341980321608302162417203362565E-01L, /* 3ffeffa4084381485c00000000000000 */
  9.99328838610882452808681364331278019E-01L, /* 3ffeffa8078f911a1800000000000000 */
  9.99359336172140816367814863951934967E-01L, /* 3ffeffac06e39f8bf400000000000000 */
  9.99389834664125092933417704443854745E-01L, /* 3ffeffb0063facadec00000000000000 */
  9.99420334086863676459344674185558688E-01L, /* 3ffeffb405a3b88ffc00000000000000 */
  9.99450834440384988655026177184481639E-01L, /* 3ffeffb8050fc3422400000000000000 */
  9.99481335724717395718741386190231424E-01L, /* 3ffeffbc0483ccd45c00000000000000 */
  9.99511837939889374871071936468069907E-01L, /* 3ffeffc003ffd556ac00000000000000 */
  9.99542341085929264554721385138691403E-01L, /* 3ffeffc40383dcd90800000000000000 */
  9.99572845162865514234695751838444266E-01L, /* 3ffeffc8030fe36b7400000000000000 */
  9.99603350170726517864849824945849832E-01L, /* 3ffeffcc02a3e91dec00000000000000 */
  9.99633856109540669399038392839429434E-01L, /* 3ffeffd0023fee006c00000000000000 */
  9.99664362979336418302267475155531429E-01L, /* 3ffeffd401e3f222f800000000000000 */
  9.99694870780142130772816244643763639E-01L, /* 3ffeffd8018ff5958800000000000000 */
  9.99725379511986284031266336569387931E-01L, /* 3ffeffdc0143f8682400000000000000 */
  9.99755889174897216520321308053098619E-01L, /* 3ffeffe000fffaaac000000000000000 */
  9.99786399768903377704987178731244057E-01L, /* 3ffeffe400c3fc6d6000000000000000 */
  9.99816911294033217050269968240172602E-01L, /* 3ffeffe8008ffdc00800000000000000 */
  9.99847423750315072998873233700578567E-01L, /* 3ffeffec0063feb2ac00000000000000 */
  9.99877937137777450526954226006637327E-01L, /* 3ffefff0003fff555800000000000000 */
  9.99908451456448688077216502279043198E-01L, /* 3ffefff40023ffb80000000000000000 */
  9.99938966706357262870241697783058044E-01L, /* 3ffefff8000fffeaac00000000000000 */
  9.99969482887531541104308985268289689E-01L, /* 3ffefffc0003fffd5400000000000000 */
#define T_EXPL_RES2 (T_EXPL_RES1 + 1 + 89 + 65)
  1.00000000000000000000000000000000000E+00L, /* 3fff0000000000000000000000000000 */
  1.00003051804379100575559391472779680E+00L, /* 3fff0002000200015400000000000000 */
  1.00006103701893306334724798034585547E+00L, /* 3fff00040008000aac00000000000000 */
  1.00009155692545448346209013834595680E+00L, /* 3fff0006001200240000000000000000 */
  1.00012207776338379883185325525118969E+00L, /* 3fff0008002000555800000000000000 */
  1.00015259953274932014366527255333494E+00L, /* 3fff000a003200a6ac00000000000000 */
  1.00018312223357958012925905677548144E+00L, /* 3fff000c004801200400000000000000 */
  1.00021364586590294498691378066723701E+00L, /* 3fff000e006201c95c00000000000000 */
  1.00024417042974783642605984823603649E+00L, /* 3fff0010008002aab400000000000000 */
  1.00027469592514273166727889474714175E+00L, /* 3fff001200a203cc1000000000000000 */
  1.00030522235211605242000132420798764E+00L, /* 3fff001400c805357000000000000000 */
  1.00033574971069616488250630936818197E+00L, /* 3fff001600f206eed000000000000000 */
  1.00036627800091160178652671675081365E+00L, /* 3fff0018012009003800000000000000 */
  1.00039680722279067381919048784766346E+00L, /* 3fff001a01520b71a000000000000000 */
  1.00042733737636191371223048918182030E+00L, /* 3fff001c01880e4b1000000000000000 */
  1.00045786846165368766392589350289200E+00L, /* 3fff001e01c211948400000000000000 */
  1.00048840047869447289485833607614040E+00L, /* 3fff0020020015560000000000000000 */
  1.00051893342751269111445822090900037E+00L, /* 3fff0022024219978400000000000000 */
  1.00054946730813676403215595200890675E+00L, /* 3fff002402881e611000000000000000 */
  1.00058000212059516886853316464112140E+00L, /* 3fff002602d223baa800000000000000 */
  1.00061053786491632733302026281307917E+00L, /* 3fff0028032029ac4c00000000000000 */
  1.00064107454112866113504765053221490E+00L, /* 3fff002a0372303dfc00000000000000 */
  1.00067161214926059198404573180596344E+00L, /* 3fff002c03c83777b800000000000000 */
  1.00070215068934059710059614189958666E+00L, /* 3fff002e04223f618400000000000000 */
  1.00073269016139709819412928482051939E+00L, /* 3fff0030048048036000000000000000 */
  1.00076323056545857248522679583402351E+00L, /* 3fff003204e251655000000000000000 */
  1.00079377190155338617216784768970683E+00L, /* 3fff003405485b8f5000000000000000 */
  1.00082431416971007198668530691065826E+00L, /* 3fff003605b266896800000000000000 */
  1.00085485736995705163820957750431262E+00L, /* 3fff00380620725b9800000000000000 */
  1.00088540150232269132501983222027775E+00L, /* 3fff003a06927f0ddc00000000000000 */
  1.00091594656683552377884893758164253E+00L, /* 3fff003c07088ca83c00000000000000 */
  1.00094649256352402622027852885366883E+00L, /* 3fff003e07829b32bc00000000000000 */
  1.00097703949241650933643654752813745E+00L, /* 3fff00400800aab55400000000000000 */
  1.00100758735354156137020709138596430E+00L, /* 3fff00420882bb381000000000000000 */
  1.00103813614692760403102056443458423E+00L, /* 3fff00440908ccc2f000000000000000 */
  1.00106868587260300351715613942360505E+00L, /* 3fff00460992df5df000000000000000 */
  1.00109923653059629256034668287611566E+00L, /* 3fff00480a20f3111800000000000000 */
  1.00112978812093589287002259879955091E+00L, /* 3fff004a0ab307e46800000000000000 */
  1.00116034064365022615561429120134562E+00L, /* 3fff004c0b491ddfe000000000000000 */
  1.00119089409876788066000585786241572E+00L, /* 3fff004e0be3350b8c00000000000000 */
  1.00122144848631711155917400901671499E+00L, /* 3fff00500c814d6f6000000000000000 */
  1.00125200380632656260715407370298635E+00L, /* 3fff00520d2367136c00000000000000 */
  1.00128256005882454449107399341301061E+00L, /* 3fff00540dc981ffa800000000000000 */
  1.00131311724383964545381786592770368E+00L, /* 3fff00560e739e3c2000000000000000 */
  1.00134367536140017618251363273884635E+00L, /* 3fff00580f21bbd0cc00000000000000 */
  1.00137423441153472492004539162735455E+00L, /* 3fff005a0fd3dac5b800000000000000 */
  1.00140479439427171337584354660066310E+00L, /* 3fff005c1089fb22e400000000000000 */
  1.00143535530963956325933850166620687E+00L, /* 3fff005e11441cf05000000000000000 */
  1.00146591715766680730226312334707472E+00L, /* 3fff0060120240360400000000000000 */
  1.00149647993838186721404781565070152E+00L, /* 3fff006212c464fc0000000000000000 */
  1.00152704365181316470412298258452211E+00L, /* 3fff0064138a8b4a4400000000000000 */
  1.00155760829798923250422149067162536E+00L, /* 3fff00661454b328d800000000000000 */
  1.00158817387693849232377374391944613E+00L, /* 3fff00681522dc9fbc00000000000000 */
  1.00161874038868942138336137759324629E+00L, /* 3fff006a15f507b6f400000000000000 */
  1.00164930783327055241471725821611471E+00L, /* 3fff006c16cb34768800000000000000 */
  1.00167987621071025161612055853765924E+00L, /* 3fff006e17a562e67400000000000000 */
  1.00171044552103705171930414508096874E+00L, /* 3fff00701883930ec000000000000000 */
  1.00174101576427937443369842185347807E+00L, /* 3fff00721965c4f76c00000000000000 */
  1.00177158694046569697988502412044909E+00L, /* 3fff00741a4bf8a87c00000000000000 */
  1.00180215904962455208959681840497069E+00L, /* 3fff00761b362e29f800000000000000 */
  1.00183273209178441698341543997230474E+00L, /* 3fff00781c246583e400000000000000 */
  1.00186330606697365785962006157205906E+00L, /* 3fff007a1d169ebe3c00000000000000 */
  1.00189388097522080744994354972732253E+00L, /* 3fff007c1e0cd9e10800000000000000 */
  1.00192445681655439848611877096118405E+00L, /* 3fff007e1f0716f45000000000000000 */
  1.00195503359100279716642489802325144E+00L, /* 3fff0080200556001000000000000000 */
  1.00198561129859459173374602869444061E+00L, /* 3fff00822107970c5400000000000000 */
};
