/* (c) Copyright 1990, 1991, 1992 Carnegie Mellon University */

/* Cprim.c: Primitives for sml2c.
 * Written by David Tarditi
 */
  

/* #define DEBUG */

#include <stdio.h>
#include <math.h>
#include <errno.h>
#include <setjmp.h>
#include "tags.h"
#include "request.h"
#include "cause.h"
#include "ml_state.h"
#include "ml_types.h" 
#include "prim.h"
#include "cstate.h"

/* define position of important registers in C state vector */

#define LIMIT_PTR 2
#define SIGNAL_LIMIT_PTR 3
#define STORE_PTR 4
#define DATA_PTR 5
#define EXN_PTR 6
#define PC 7
#define VAR_PTR 8
#define STANDARD_CLOSURE 9
#define STANDARD_ARG 10
#define STANDARD_CONT 11

/* 12-30 are misc. registers */

#define MSP 31


/* max/min floating point values that can be converted to integers
  are +/-2^30 respectively */

#define FLOOR_MAX 1073741824.0
#define FLOOR_MIN -1073741824.0

typedef double Creal;
#define CREAL_SIZE 8

/* some header files are missing this declaration */

extern int errno;

/* exception values constructed elsewhere in the runtime */

extern ML_val_t overflow_e0[], sqrt_e0[], ln_e0[];

/* register descriptor for functions using the standard calling
   convention */

#define STDGCMASK 7
#define CLOSURE(name,func_name) int name[2] = { MAKE_DESC(1,tag_record), \
(int) func_name};

/* declare some local static functions */

static void quicksave(),moveregs(),fetchregs();

#define RAISE(x,csp) \
{ csp[STANDARD_ARG] = (Cint) (x); \
  csp[STANDARD_CONT] = csp[EXN_PTR]; \
  return (*(Cint *)csp[EXN_PTR]); }

Cint sig_return_v_function(csp)
CState_ptr csp;
{ register MLState_ptr msp = (MLState_ptr) csp[MSP];
  msp->request = REQ_SIG_RETURN;
  quicksave(csp,msp);
}

Cint sigh_resume(csp)
CState_ptr csp;
{ register MLState_ptr msp = (MLState_ptr) csp[MSP];
  msp->request = REQ_SIG_RESUME;
  quicksave(csp,msp);
}

Cint handle_c_function(csp)
CState_ptr csp;
{ register MLState_ptr msp = (MLState_ptr) csp[MSP];
  msp->request = REQ_EXN;
  quicksave(csp,msp);
}

Cint return_c_function(csp)
CState_ptr csp;
{ register MLState_ptr msp = (MLState_ptr) csp[MSP];
  msp->request = REQ_RETURN;
  quicksave(csp,msp);
}

Cint callc_v_function(csp)
CState_ptr csp;
{ l0: if (csp[DATA_PTR] <= csp[SIGNAL_LIMIT_PTR])
         { register MLState_ptr msp = (MLState_ptr) csp[MSP];
           msp->request = REQ_CALLC;
	   quicksave(csp,msp);
         }
      else 
	{ invoke_gc(STDGCMASK,csp,callc_v_function);
          goto l0;
        }
}

static void quicksave(csp,msp)
CState_ptr csp;
MLState_ptr msp;
{ msp->inML = 0;
  msp->ml_allocptr = (int) csp[DATA_PTR];
  msp->ml_storeptr = (int) csp[STORE_PTR];
  msp->ml_roots[EXN_INDX] = (ML_val_t) csp[EXN_PTR];
  msp->ml_roots[CONT_INDX] = (ML_val_t) csp[STANDARD_CONT];
  msp->ml_roots[ARG_INDX] = (ML_val_t) csp[STANDARD_ARG];
  msp->ml_roots[VAR_INDX] = (ML_val_t) csp[VAR_PTR];
  _longjmp(msp->ReturnEnv,1);
}

static void moveregs(csp,msp)
register CState_ptr csp;
register MLState_ptr msp;
{ register Cint *s;
  register ML_val_t *roots;
  msp->ml_allocptr = (int) csp[DATA_PTR];
  msp->ml_storeptr = (int) csp[STORE_PTR];
  for (roots = msp->ml_roots, s = csp+CNUMREGS, csp += 6;
       csp < s; *roots++ = (ML_val_t) *csp++);
}

static void fetchregs(csp,msp)
register CState_ptr csp;
register MLState_ptr msp;
{ register Cint *s;
  register ML_val_t *roots;
  register Cint limit = msp->ml_limitptr; 
  csp[LIMIT_PTR] = limit;
  csp[SIGNAL_LIMIT_PTR] = limit;
  csp[DATA_PTR] = msp->ml_allocptr;
  csp[STORE_PTR] = msp->ml_storeptr;
  for (roots = msp->ml_roots, s= csp + CNUMREGS, csp += 6;
       csp < s; *csp++ = (Cint) *roots++);
}

void saveregs(msp)
register MLState_ptr msp;
{ register Cint *csp = msp->csp;
   msp->inML = 0;
  moveregs(msp->csp,msp);
  _longjmp(msp->ReturnEnv,1);
}

void savefpregs() {}
void restorefpregs() {}

void restoreregs(msp)
register MLState_ptr msp;
{ register Cint *csp = msp->csp;
  register Cint (*next)();

#ifdef DEBUG
  pchatting(msp,"about to restore regs\n");
#endif
  csp = msp->csp;

  fetchregs(csp,msp); 
  csp[MSP] = (Cint) msp;
  next = (Cint (*)()) csp[PC];

 if (msp->NumPendingSigs && !msp->maskSignals && !msp->inSigHandler) {
       msp->handlerPending = 1;
       csp[SIGNAL_LIMIT_PTR] = 0;
  }
 msp->inML = 1;
 if (_setjmp(msp->ReturnEnv) == 0)
    while(1)
#ifdef DEBUG
      { Cint (*next0)(),(*next1)(),(*next2)(),(*next3)(),(*next4)(),
             (*next5)(),(*next6)(),(*next7)();
        next0 = (Cint (*)()) ((*next)(csp));
        next1 = (Cint (*)()) ((*next0)(csp));
        next2 = (Cint (*)()) ((*next1)(csp));
        next3 = (Cint (*)()) ((*next2)(csp));
        next4 = (Cint (*)()) ((*next3)(csp));
        next5 = (Cint (*)()) ((*next4)(csp));
        next6 = (Cint (*)()) ((*next5)(csp));
        next7 = (Cint (*)()) ((*next6)(csp));
        next = (Cint (*)()) ((*next7)(csp));
     }
#else
      { next = (Cint (*)()) ((*next)(csp));
        next = (Cint (*)()) ((*next)(csp));
        next = (Cint (*)()) ((*next)(csp));
        next = (Cint (*)()) ((*next)(csp));
        next = (Cint (*)()) ((*next)(csp));
        next = (Cint (*)()) ((*next)(csp));
        next = (Cint (*)()) ((*next)(csp));
        next = (Cint (*)()) ((*next)(csp));
        next = (Cint (*)()) ((*next)(csp));
     }
#endif
 else 
   {
#ifdef DEBUG
     pchatting(msp,"return from restore regs\n");
#endif
     return;  
  }
}

int invoke_gc(mask,csp,func)
unsigned int mask;
CState_ptr csp;
{ MLState_ptr msp = (MLState_ptr) csp[MSP];
  msp->inML = 0;
  if (msp->handlerPending) {
#ifdef DEBUG
    fprintf(stderr,"handler pending for %d\n",func);
#endif
    sig_setup(msp);
    msp->mask=mask;
    csp[PC]=func;
    saveregs(msp);
  }
  moveregs(csp,msp);
  callgc0(msp, CAUSE_GC,4096, mask);
  fetchregs(csp,msp);
  msp->inML = 1;
}

int inlined_gc(mask,csp)
unsigned int mask;
CState_ptr csp;
{ MLState_ptr msp = (MLState_ptr) csp[MSP];
  msp->inML =0;
  moveregs(csp,msp);
  callgc0(msp,CAUSE_GC,4096,mask);
  fetchregs(csp,msp);
  msp->inML = 1;
}

#define UNTAG(v) ((v) >> 1)

Cint array_v_function(csp)
register CState_ptr csp;
{ register Cint *dataptr,*finish,init;
  register Cint l = UNTAG( *((Cint *) csp[STANDARD_ARG]));
  register Cint newtag = (l << width_tags) | tag_array;
l0:  dataptr = (Cint *) csp[DATA_PTR];
     if (l+dataptr < (Cint *) csp[SIGNAL_LIMIT_PTR])
       { init = *(((Cint *) csp[STANDARD_ARG]) + 1);
         *dataptr++ = newtag;
         csp[STANDARD_ARG] = (Cint) dataptr;
         for (finish = dataptr + l; dataptr < finish;
	      *dataptr++ = init);
         csp[DATA_PTR] = (Cint) dataptr;
         return( *((Cint *) csp[STANDARD_CONT]));
       }
  invoke_gc(STDGCMASK,csp,array_v_function);
  goto l0;
}

Cint create_s_v_function(csp)
register CState_ptr csp;
{ register Cint *dataptr;
  register Cint l = UNTAG(csp[STANDARD_ARG]);
  register Cint newtag = (l << width_tags) | tag_string;

  /* # of longwords needed */

  l = (l+3) >> 2;

l0:  dataptr = (Cint *) csp[DATA_PTR];
      if (l + dataptr  < (Cint *) csp[SIGNAL_LIMIT_PTR])
      { *dataptr++ = newtag;
        csp[STANDARD_ARG] = (Cint) dataptr;
        csp[DATA_PTR] = (Cint) (dataptr += l);
        return( *((Cint *) csp[STANDARD_CONT]));
      }
    invoke_gc(STDGCMASK,csp,create_s_v_function); goto l0;
}

Cint create_b_v_function(csp)
register CState_ptr csp;
{ register Cint *dataptr;
  register Cint l = UNTAG(csp[STANDARD_ARG]);
  register Cint newtag = (l << width_tags) | tag_bytearray;

  /* # of longwords needed */

  l = (l+3) >> 2;

l0:  dataptr = (Cint *) csp[DATA_PTR];
      if (l + dataptr  < (Cint *) csp[SIGNAL_LIMIT_PTR])
      { *dataptr++ = newtag;
        csp[STANDARD_ARG] = (Cint) dataptr;
        csp[DATA_PTR] = (Cint) (dataptr += l);
        return( *((Cint *) csp[STANDARD_CONT]));
      }
    invoke_gc(STDGCMASK,csp,create_b_v_function); goto l0;
}

/* create_v_v : int * 'a list -> 'a vector
 * 	creates a vector with elements taken from a list.
 * The front-end ensures that the list cannot be nil
 */

Cint create_v_v_function(csp)
register CState_ptr csp;
{ register Cint *dataptr;
  register Cint l = UNTAG(*((Cint *)csp[STANDARD_ARG]));
  register Cint newtag = (l << width_tags) | tag_record;
  register ML_val_t cell;

l0:  dataptr = (Cint *) csp[DATA_PTR];
     if (l + dataptr  < (Cint *) csp[SIGNAL_LIMIT_PTR])
      { cell = (ML_val_t) (*((Cint *)csp[STANDARD_ARG]+1));
	*dataptr++ = newtag;
        csp[STANDARD_ARG] = (Cint) dataptr;
     	while (cell != ML_nil)
	  { *dataptr++ = (Cint) (ML_hd(cell));
	    cell = ML_tl(cell);
          }
        csp[DATA_PTR] = (Cint) dataptr;
        return( *((Cint *) csp[STANDARD_CONT]));
      }
    invoke_gc(STDGCMASK,csp,create_v_v_function); goto l0;
}

/* try_lock_v_function: TAS a lock.  A lock holds the ML value false
   when it is set.  This is a trivial uni-processor version */

Cint try_lock_v_function(csp)
CState_ptr csp;
{ register ML_val_t *lock = (ML_val_t *) csp[STANDARD_ARG];
  ML_val_t old = *(lock);
  *lock = ML_false;       
  csp[STANDARD_ARG] = (Cint) old;
  return (*((Cint *) csp[STANDARD_CONT]));
}

/* unlock_v: unlock a TAS lock */

Cint unlock_v_function(csp)
CState_ptr csp;
{ register ML_val_t *lock = (ML_val_t *) csp[STANDARD_ARG];
  *lock = ML_true;              /* true = unlocked */
  csp[STANDARD_ARG] = (Cint) ML_unit;   
  return (*((Cint *) csp[STANDARD_CONT]));
}

Cint logb_v_function(csp)
CState_ptr csp;
{ RAISE(overflow_e0+1,csp); }

Cint scalb_v_function(csp)
CState_ptr csp;
{ RAISE(overflow_e0+1,csp); }

Cint floor_v_function(csp)
CState_ptr csp;
{ register Creal d = floor(*(double *) csp[STANDARD_ARG]);
  if (d< FLOOR_MIN || d>FLOOR_MAX) {
                RAISE(overflow_e0+1,csp);
    }
  csp[STANDARD_ARG] = (Cint) ((Cint) d * 2 + 1) ;
  return (*(Cint *) csp[STANDARD_CONT]);
}

#define MATH_FUNC(f,name) \
Cint name(csp) \
CState_ptr csp; \
{ register Creal d; \
  register Cint *dataptr; \
l0: dataptr = (Cint *) csp[DATA_PTR]; \
    if (3 + dataptr <= (Cint *) csp[SIGNAL_LIMIT_PTR]) { \
    d = f (* ((double *) csp[STANDARD_ARG])); \
    *dataptr++ = MAKE_DESC(CREAL_SIZE,tag_string); \
    csp[STANDARD_ARG] = (Cint) dataptr; \
    *(Creal *) dataptr = d; \
    csp[DATA_PTR] = (Cint) ((char *) dataptr+CREAL_SIZE); \
    return (*((Cint *) csp[STANDARD_CONT])); \
  } \
  invoke_gc(STDGCMASK,csp,name); \
  goto l0; \
}

/* possible portability problem here; errno isn't reset by some math functions
   and edition 2 of Kernighan and Ritchie says nothing about resetting
   errno.  So we're resetting it ourselves here.  This could be wrong for
   other OS/local environments. */

#define MATH_FUNC_WITH_ERR(f,name,err) \
Cint name(csp) \
CState_ptr csp; \
{ register Creal d; \
  register Cint *dataptr; \
l0: dataptr = (Cint *) csp[DATA_PTR]; \
    if (3 + dataptr <= (Cint *) csp[SIGNAL_LIMIT_PTR]) { \
    d = f (* ((double *) csp[STANDARD_ARG])); \
    if ((errno == EDOM) || (errno == ERANGE)) {errno = -1; RAISE(err,csp); } \
    *dataptr++ = MAKE_DESC(CREAL_SIZE,tag_string); \
    csp[STANDARD_ARG] = (Cint) dataptr; \
    *(Creal *) dataptr = d; \
    csp[DATA_PTR] = (Cint) ((char *) dataptr+CREAL_SIZE); \
    return (*((Cint *) csp[STANDARD_CONT])); \
  } \
  invoke_gc(STDGCMASK,csp,name); \
  goto l0; \
} 

MATH_FUNC(sin, sin_v_function)
MATH_FUNC(cos,  cos_v_function)
MATH_FUNC(atan, arctan_v_function)

MATH_FUNC_WITH_ERR(exp,  exp_v_function, overflow_e0+1)
MATH_FUNC_WITH_ERR(log, ln_v_function, ln_e0+1)
MATH_FUNC_WITH_ERR(sqrt, sqrt_v_function, sqrt_e0+1)

int startptr;

CLOSURE(sigh_return_c,sig_return_v_function)
CLOSURE(handle_c,handle_c_function)
CLOSURE(return_c,return_c_function)
CLOSURE(callc_v,callc_v_function)
CLOSURE(array_v,array_v_function)
CLOSURE(create_b_v,create_b_v_function)
CLOSURE(create_s_v,create_s_v_function)
CLOSURE(create_v_v,create_v_v_function)

CLOSURE(try_lock_v,try_lock_v_function)
CLOSURE(unlock_v,unlock_v_function)

CLOSURE(arctan_v,arctan_v_function)
CLOSURE(cos_v,cos_v_function)
CLOSURE(exp_v,exp_v_function)
CLOSURE(floor_v,floor_v_function)
CLOSURE(ln_v,ln_v_function)
CLOSURE(sin_v,sin_v_function)
CLOSURE(sqrt_v,sqrt_v_function)
CLOSURE(logb_v,logb_v_function)
CLOSURE(scalb_v,scalb_v_function)

/* multiplication with overflow checking.
   We break the operands into 16-bit parts, multiply them, and put them
   back together.
*/

/* overflow check for unsigned integer addition, where a and b are the
   msb of the operands:

        a b r | ov
        ----------
        0 0 0   0
        0 0 1   0
        0 1 0   1
        0 1 1   0
        1 0 0   1
        1 0 1   0
        1 1 0   1
        1 1 1   1

    Overflow = and(a,b)|((eor(a,b)&(~r)))
*/

#define NO_OVERFLOW(a,b,r) (((a&b)|((a^b)&(~r)))>=0)
#define WORD_SIZE 16
#define LONG_WORD_SIZE 32
#define POW2(x) (1<<x)

/* mult: multiply two two's complement numbers, raise exception
   if an overflow occurs. */

int mult(b,d,csp)
register unsigned int b,d;
CState_ptr csp;
{ register unsigned int a,c;
  register int sign = b^d;

/* break b and d into hi/lo words 

      -------------   ---------
      |  a  |  b  |  |c   |  d|
      -------------  ---------
*/

  if ((int)b<0) {b = -(int)b; }
  if ((int)d<0) {d = -(int)d; }
  a = b >> WORD_SIZE;
  b = b & (POW2(WORD_SIZE)-1);
  c = d >> WORD_SIZE;
  d = d & (POW2(WORD_SIZE)-1);
  if (a&c) goto overflow;
  a = a*d;
  c = c*b;
  b = b*d;
  if (a<(POW2(LONG_WORD_SIZE-WORD_SIZE)) &&
      c<(POW2(LONG_WORD_SIZE-WORD_SIZE)))
    { d = a+c;
      if (d<(POW2(LONG_WORD_SIZE-WORD_SIZE)))
          { d <<= WORD_SIZE;
            a=d+b;
           if NO_OVERFLOW(d,b,a)
	       if (sign<0)
		  if (a<=POW2(LONG_WORD_SIZE-1))
                      return (-a);
		  else goto overflow;
	       else if (a<(POW2(LONG_WORD_SIZE-1))) return (a);
	  }
    }
 overflow:
#ifdef DEBUG
      printf("overflow occurred\n");
#endif

/* buggy:
         RAISE (overflow_e0+1,csp);

   mult is a *function* that is supposed to return the result of
   the multiplication.
*/

/* correct code */

  {  register MLState_ptr msp = (MLState_ptr) csp[MSP];
     msp->request = REQ_FAULT;
     msp->fault_exn = (ML_val_t) (overflow_e0+1);
     quicksave(csp,msp);
  }
}

/* instrumenting and profiling code */

int _s2c_jc, /* count of jumps */
    _s2c_gc, /* count of gotos */
    _s2c_ic, /* instruction count */
    _s2c_wc, /* writes to spill local vars */
    _s2c_rc, /* reads to load local vars */
    _s2c_hc; /* count of heap checks */

#include<stdio.h>

void ml_sml2c_instrument(msp, arg)
MLState_ptr msp;
ML_val_t arg;
{ int fd = INT_MLtoC(arg);
  ML_val_t s;
  char buf[256];
  int overhead = _s2c_rc + _s2c_wc + _s2c_hc + _s2c_jc * 2;
  sprintf(buf,"jumps = %d\ngotos = %d\nwrites= %d\nreads=%d\nheap checks = %d\ninstructions= %d\noverhead= %d\n%% overhead = %3.1f\n",
               _s2c_jc,
	       _s2c_gc,
	       _s2c_wc,
	       _s2c_rc,
	       _s2c_hc,
	       _s2c_ic,
               overhead,
	       100.0 * (double)overhead/(double) _s2c_ic);
  s = ML_alloc_string(msp,buf);
  msp->ml_arg = s;
  return;
}
