(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created by stolfi on Wed Apr 19 01:48:27 1989               *)
(* Last modified on Fri Aug 14 23:06:43 PDT 1992 by meehan     *)
(*      modified on Wed Jun 17 12:00:14 PDT 1992 by stolfi     *)
(*      modified on Thu Jun 11 18:50:04 1992 by mhb        *)

(*      modified on Tue Feb 11 21:39:48 PST 1992 by muller     *)
(*      modified on Mon Nov 11 16:26:22 PST 1991 by steveg     *)


MODULE ColorName EXPORTS ColorName, ColorNameF;

IMPORT 
  Text, TextF, Char, RGB, RGBDist,
  HSV, TxtIntTbl, TxtRefTbl, List;

FROM ColorNameTable IMPORT Basic;

CONST
  NumFractions = 10; (* Number of meta-modifiers (Very, Slightly, etc. *)
    
TYPE FrEntry = RECORD name: TEXT; val: REAL END;   (* Entry of fraction table *)

CONST Fraction = ARRAY OF FrEntry{
  
    (* Fraction prefixes: *)
    (* Note: if A is a prefix of B, then A must come after B. *)
    (* Also the last prefix must be "" *)

    FrEntry {name := "VeryVeryVery",     val := 15.0 / 16.0},
    FrEntry {name := "VeryVerySlightly", val := 1.0 / 16.0 },
    FrEntry {name := "VeryVery",         val := 7.0 / 8.0  },
    FrEntry {name := "VerySlightly",     val := 1.0 / 8.0  },
    FrEntry {name := "Very",             val := 3.0 / 4.0  },
    FrEntry {name := "Slightly",         val := 1.0 / 4.0  },
    FrEntry {name := "Somewhat",         val := 3.0 / 8.0  },
    FrEntry {name := "Quite",            val := 5.0 / 8.0  },
    FrEntry {name := "Rather",           val := 1.0 / 2.0  },
    FrEntry {name := "",                 val := 1.0 / 3.0  }
  };

EXCEPTION FoundName;

PROCEDURE FromRGB (color: RGB.T): TEXT =
  CONST Threshold = 0.01; (* Threshold for color equality *)
  
  VAR nBest: TEXT; (* Best candidate name so far *)
      dBest: REAL; (* Distance from best candidate to target. *)

  PROCEDURE TryBasic (name: TEXT; READONLY c: RGB.T) RAISES {FoundName} = 
    (* Tries one basic color, unmodified *)
    BEGIN
      IF (c = RGB.Undefined) THEN
        (* ignore *)
      ELSE
        WITH d = RGBDist.Perceptual(c, color, 1.0) DO
          IF (d < dBest) THEN 
            dBest := d; 
            nBest := name;
            IF dBest < Threshold THEN RAISE FoundName END;
          END
        END
      END
    END TryBasic;

  PROCEDURE TryModified(name: TEXT; c: RGB.T) RAISES {FoundName} = 
    (* Tries all modifications of one basic color, returns as
       soon as it got something better than threshold *)
    VAR m: RGB.T; 
        mName: TEXT;
    BEGIN
      (* Try all possible modifiers of c: *)
      FOR k := 0 TO 7 DO
        (* Pick one modifier: *)
        CASE k OF
        | 0 => m := RGB.Black; mName := "Dark";
        | 1 => m := RGB.White; mName := "Light";
        | 2 => m := RGB.Grey (RGB.Brightness (c)); mName := "Drab";
        | 3 => m := HSV.RGBFromHue (HSV.HueFromRGB (c)); mName := "Vivid";
        | 4 => m := RGB.Red; mName := "Reddish";
        | 5 => m := RGB.Green; mName := "Greenish";
        | 6 => m := RGB.Blue; mName := "Bluish";
        | 7 => m := RGB.Yellow; mName := "Yellowish";
        END;
        TryFractions(
          base := c, baseName := name, 
          modif := m, modifName := mName
        );
        IF dBest < Threshold THEN RAISE FoundName END
      END
    END TryModified;

  PROCEDURE TryFractions (
      READONLY base: RGB.T; 
      baseName: TEXT; 
      READONLY modif: RGB.T;
      modifName: TEXT
    ) RAISES {FoundName} =
    (* Tries colors interpolating between /base/ and /modif/ *)
    VAR d, dvv, dvx, s: REAL; 
        cc: RGB.T; 
        v, x: RGB.T;
    CONST BallPark = 0.10;  (* RGBDist.Perceptual threshold to consider fine tuning *)
    BEGIN
      (* See if modifier is a significant change: *)
      IF RGBDist.Perceptual (base, modif) > 1.5 * Threshold THEN
        (* See if modifier gets towards given color: *)
        FOR j := 0 TO 2 DO
          v[j] := modif[j] - base[j];
          x[j] := color[j] - base[j];
        END;
        dvv := v[0] * v[0] + v[1] * v[1] + v[2] * v[2];
        dvx := v[0] * x[0] + v[1] * x[1] + v[2] * x[2];
        IF (dvx > 0.0) AND (dvx < dvv) THEN
          (* See if modifier can get close enough to given color: *)
          s := dvx / dvv;
          cc := RGB.Mix (modif, s, base, 1.0 - s);
          IF RGBDist.Perceptual (cc, color) < BallPark THEN
            (* Try all known fractions.  (Can't jump to best one because *)
            (*   RGBDist.Perceptual is non-linear)                       *)
            FOR f := 0 TO NumFractions - 1 DO
              s := Fraction[f].val;
              cc := RGB.Mix (modif, s, base, 1.0 - s);
              d := RGBDist.Perceptual (cc, color);
              IF d < dBest THEN
                dBest := d;
                nBest := Fraction[f].name & modifName & baseName;
                IF dBest < Threshold THEN RAISE FoundName END;
              END;
            END
          END
        END
      END
    END TryFractions;

  BEGIN
    IF color = RGB.Undefined THEN RETURN "Undefined" END;
    
    dBest := LAST(REAL);
    TRY
      (* Try basic colors: *)
      FOR i := 0 TO LAST(Basic) DO
        TryBasic(Basic[i].name, Basic[i].rgb)
      END;
      (* Try modified colors: *)
      FOR i := 0 TO LAST(Basic) DO
        TryModified(Basic[i].name, Basic[i].rgb)
      END;
    EXCEPT 
    | FoundName => (*OK*) 
    END;
    RETURN nBest
  END FromRGB;

PROCEDURE IsPrefix (a, b: TEXT; VAR (*OUT*) rest: TEXT): BOOLEAN =
  (* 
    If /a/ is a prefix of /b/ (ignoring case), return TRUE and put rest
    of /b/ in /rest/.  Else return false and leave /rest/alone.  *)
  BEGIN
    <* ASSERT a # NIL *>
    <* ASSERT b # NIL *>
    WITH 
      aa  = a^, 
      bb = b^ 
    DO
      IF NUMBER(aa) <= 1 THEN 
        rest := b;
        RETURN TRUE
      ELSIF NUMBER(bb) < NUMBER(aa) THEN
        RETURN FALSE
      ELSE
        FOR i := 0 TO LAST(aa) - 1 DO
          IF Char.Lower[aa[i]] # Char.Lower[bb[i]] THEN
            RETURN FALSE
          END
        END;
        rest := Text.FromChars(SUBARRAY(bb, NUMBER(aa) - 1, NUMBER(bb) - NUMBER(aa)));
        RETURN TRUE
      END;
    END
  END IsPrefix;

PROCEDURE NormalizeName (a: TEXT): TEXT =
  (* 
    Deletes all whitespace in /a/ and converts to lower case *)
  VAR b := NEW(REF ARRAY OF CHAR, Text.Length(a)); 
      j := 0;
  BEGIN
    IF NUMBER(b^) > 0 THEN 
      WITH aa = a^, bb = b^ DO 
        FOR i := 0 TO LAST(aa) - 1 DO
          WITH c = aa[i] DO
            IF  NOT (c IN Char.Spaces) THEN
              bb[j] := Char.Lower[c]; INC(j)
            END
          END
        END
      END
    END;
    RETURN Text.FromChars(SUBARRAY(b^, 0, j))
  END NormalizeName;

PROCEDURE ToRGB (name: TEXT): RGB.T RAISES {NotFound} =
  VAR
    value : REFANY;
    rgb   : RGB.T;
    rgbRef: REF RGB.T;
  PROCEDURE fail (<* UNUSED *> name: TEXT): RGB.T RAISES {NotFound} =
    BEGIN
      RAISE NotFound
    END fail;
  BEGIN
    LOCK nameCache DO
      IF nameCache.table.in (name, value) THEN
        RETURN NARROW (value, REF RGB.T)^
      END
    END;
    WITH normalized = NormalizeName (name) DO
      IF Text.Equal ("undefined", normalized)
           OR Text.Equal ("nil", normalized) THEN
        rgb := RGB.Undefined
      ELSE
        rgb := LowerCaseToRGB (normalized, fail)
      END
    END;
    LOCK nameCache DO
      rgbRef := NEW (REF RGB.T);
      rgbRef^ := rgb;
      EVAL nameCache.table.put (name, rgbRef);
    END;
    RETURN rgb
  END ToRGB;

PROCEDURE LowerCaseToRGB (name: TEXT; p: NotInTable): RGB.T RAISES {NotFound} =
  VAR
    f         : CARDINAL;
    index     : INTEGER;
    frac      : REAL;
    rgb       : RGB.T;
    bare, rest: TEXT;
  BEGIN
    (* Strips fraction modifier: *)
    f := 0;
    WHILE NOT IsPrefix (Fraction [f].name, name, rest) DO INC (f) END;
    frac := Fraction [f].val;
    bare := rest;
    (* Strips color modifier: *)
    IF IsPrefix ("dark", bare, rest) OR IsPrefix ("dim", bare, rest) THEN
      rgb := LowerCaseToRGB (rest, p);
      RETURN RGB.Mix (RGB.Black, frac, rgb, 1.0 - frac)

    ELSIF IsPrefix ("pale", bare, rest) OR IsPrefix ("light", bare, rest) THEN
      rgb := LowerCaseToRGB (rest, p);
      RETURN RGB.Mix (RGB.White, frac, rgb, 1.0 - frac)

    ELSIF IsPrefix ("medium", bare, rest) THEN
      (* There must be no fraction modifier: *)
      IF NOT Text.Equal (bare, name) THEN RAISE NotFound END;
      frac := 0.25;
      rgb := LowerCaseToRGB (rest, p);
      RETURN RGB.Mix (RGB.Black, frac, rgb, 1.0 - frac)

    ELSIF IsPrefix ("reddish", bare, rest) THEN
      rgb := LowerCaseToRGB (rest, p);
      RETURN RGB.Mix (RGB.Red, frac, rgb, 1.0 - frac)

    ELSIF IsPrefix ("greenish", bare, rest) THEN
      rgb := LowerCaseToRGB (rest, p);
      RETURN RGB.Mix (RGB.Green, frac, rgb, 1.0 - frac)

    ELSIF IsPrefix ("bluish", bare, rest) THEN
      rgb := LowerCaseToRGB (rest, p);
      RETURN RGB.Mix (RGB.Blue, frac, rgb, 1.0 - frac)

    ELSIF IsPrefix ("yellowish", bare, rest) THEN
      rgb := LowerCaseToRGB (rest, p);
      RETURN RGB.Mix (RGB.Yellow, frac, rgb, 1.0 - frac)

    ELSIF IsPrefix ("drab", bare, rest) OR IsPrefix ("dull", bare, rest)
            OR IsPrefix ("weak", bare, rest) THEN
      rgb := LowerCaseToRGB (rest, p);
      RETURN RGB.Mix (RGB.Grey (RGB.Brightness (rgb)), frac, rgb, 1.0 - frac)

    ELSIF IsPrefix ("strong", bare, rest) OR IsPrefix ("vivid", bare, rest)
            OR IsPrefix ("bright", bare, rest) THEN
      rgb := LowerCaseToRGB (rest, p);
      RETURN
        RGB.Mix (HSV.RGBFromHue (HSV.HueFromRGB (rgb)), frac, rgb, 1.0 - frac)

    ELSE
      (* No color modifier -- there must be no fraction modifier: *)
      IF NOT Text.Equal (bare, name) THEN RAISE NotFound END;
      IF NOT table.in (name, (*OUT*) index) THEN
        RETURN p (name)
      ELSE
        RETURN Basic [index].rgb
      END
    END
  END LowerCaseToRGB;

PROCEDURE NameList (): List.T =
  VAR list: List.T;
  BEGIN
    list := NIL;
    FOR i := LAST(Basic) TO 0 BY -1 DO
      list := List.New(Basic[i].name, list);
    END;
    RETURN List.SortD(list)
  END NameList;

PROCEDURE Init () =
  BEGIN
    nameCache := NEW (Cache, table := TxtRefTbl.New (16));
    table := TxtIntTbl.New (initialSize := NUMBER (Basic));
    FOR i := 0 TO LAST (Basic) DO
      IF table.put (NormalizeName (Basic [i].name), i) THEN
        (* ignore duplicates (case-variants) *)
      END;
    END
  END Init;

BEGIN
  Init ();
  <* ASSERT Text.Empty (Fraction [LAST (Fraction)].name) *>
END ColorName.

