
/* Copyright (C) 1988, 1989 Herve' Touati, Aquarius Project, UC Berkeley */

/* Copyright Herve' Touati, Aquarius Project, UC Berkeley */


% A new & possibly correct temporary allocation routine:

% Uses the variable list created by varlist
% and the lifetime list created by lifetime.
% Takes the overlap of registers caused by calls into account.
% The Life list does not have to contain any instantiated entries.

% Optimization - 11/16/84:
% Modified tempa so that it will identify temporaries which are not 
% arguments in the head and aren't 
% arguments of a call, by allocating them outside of the registers being
% currently used for arguments, thereby leaving them available for other
% allocation.  This allows a more efficient allocation and solves the
% 'determinate concat' optimization.

% Optimization - 12/4/84:
% Modified tempa so that if a variable first occurs between the head and the
% first clause, it will attempt to allocate into the next call's argument
% registers.  Modification done at statement (1) below.
% Similar modification can probably be done for calls after first call.

% bug fix - 3/27/86:
% cut inserted in alloc procedure so that retract will succeed only once.
% This was not a problem in 1.2, where retract only succeeded once,
% no matter how many unifiable clauses were available.
% Peter's algorithm took advantage of this bug.  
% This bug doesn't exist in 1.5.  
% In order to simulate the bug, the cut has been inserted.

% bug fix - 1/15/87:
% alloc/3 fixed.  Old version would generate a choice point for each recursive
% call, whereas correct version generates only one choice point per allocation.
% This bug sometimes led to an enormous increase in allocation time.

tempalloc([arity(HeadArity)|Vars], [_|Life]) :-
	set(3, none),
	tempa(Vars, Life, 1, HeadArity, [], head), !.


% Fail if there is a conflict:
tempa(Vars, [Live|R], N, Arity, OK, Place) :-
	conflict_interval(Place, N, Arity, Interval),
	conflict(Live, Interval, I),
	notin(I,OK),
	set(3, I),
	!, fail.

tempa([], _, _, _, _, _) :- !.

% Try to allocate to an argument:
tempa([X|Vars], [Left,Right|LifeList], N, Arity, OK, Place) :-
	var(X), in(X, Right), !,
	alloc_start_reg(Place, X, Vars, N, Arity, StartReg),
	alloc(X, Right, StartReg),
	update_params(X, N, Arity, OK, NewN, NewArity, NewOK),
	tempa(Vars, [Right|LifeList], NewN, NewArity, NewOK, Place). 
			% failure of tempa backtracks to alloc
		        % which redoes the allocation causing the conflict.
tempa([X|Vars], [_|LifeList], _, _, _, Place) :-
	nonvar(X), X=arity(Arity), !,
	tempa(Vars, LifeList, 1, Arity, [], body).
tempa([X|Vars], [Left,In,Right|LifeList], _, _, _, Place) :-
	nonvar(X), X=(_;_), In=(_;_), !,
	distempa(X, In),
	tempa(Vars, [Right|LifeList], 1, 0, [], body).
tempa([X|Vars], [_|LifeList], N, Arity, OK, Place) :-
	update_params(X, N, Arity, OK, NewN, NewArity, NewOK),
	tempa(Vars, LifeList, NewN, NewArity, NewOK, Place).

% Handle disjunctions:
distempa((A;B), (ALife;BLife)) :- !, % cut necessary for correct conflict detect
	tempa(A, ALife, 1, 0, [], body),
	distempa(B, BLife).
distempa(B, BLife) :-
	tempa(B, BLife, 1, 0, [], body).

% Calculate conflict interval.
% Depends on place in a call sequence
conflict_interval(body, 1, _, empty) :- !.
conflict_interval(body, N, _, int(1,N1)) :- !, N1 is N-1 .
conflict_interval(head, N, Arity, empty) :- N>Arity, !.
conflict_interval(head, N, Arity, int(N,Arity)) :- !.

% Update parameters of tempa.
update_params(X, N, Arity, OK, NewN, NewArity, NewOK) :-
	N=<Arity, !,
	NewN is N+1,
	NewArity=Arity,
	newok(X, N, OK, NewOK).
update_params(_, _, _, _, 1, 0, []) :- !.

	% New value of OK list
	newok(X, N, OK, [N|OK]) :-
		nonvar(X), X=x(N), !.
	newok(X, N, OK, OK) :- !.

% Calculate register to start allocation with.

% If in head, avoid using arg. reg. of next call
alloc_start_reg(head, X, Vars, N, Arity, StartReg) :-
	N>Arity, notinnextcall(X, Vars, NextArity), !,
	StartReg is NextArity+1 .
alloc_start_reg(head, _, _,    N, Arity, StartReg) :-
	N=<Arity, !,
	StartReg = N.
% Default starting value is register 1
alloc_start_reg(head, _, _,    _,     _,        1) :- !.
alloc_start_reg(body, _, _,    N,     _,        N) :- !.

% Succeeds iff there is a register conflict:
% The interval [L, L+1, ..., H] is also considered as live registers.
% It is represented as int(L,H) or as the atom `empty'.
conflict(Live, int(L,H), I) :-
	L=<H,
	range(L, I, H),
	in(x(I), Live).
conflict(Live, R, I) :-
	conflict(Live, I).

conflict([V|Live], I) :-
	nonvar(V), V=x(I),
	in(V, Live).
conflict([R|Live], I) :-
	conflict(Live, I).

% Allocate a register.
% When there is a conflict, 
% supports sophisticated backtracking to the cause.
% Don't allocate X8.

% Bug fix Jan. 15: every recursive call generated a choice point,
% whereas only one choice point per allocation mey be generated.
alloc(X, Alive, N) :- '\+'(N=8), notin(x(N),Alive), X=x(N).
alloc(X, Alive, N) :-
	access(3, none), !, % <- Bug fix: this cut is essential.
	N1 is N+1,
	alloc(X, Alive, N1).
alloc(X, Alive, N) :-
	access(3, N), set(3, none),
	N1 is N+1,
	alloc(X, Alive, N1).

% Find next call and return arity.
% Fails if no next call or if X is not an argument of it.
notinnextcall(X,Vars,NextArity) :-
	isnextcall(Vars,Call,NextArity),!,
	notin(X, Call).

isnextcall([V|RestVars],RestVars,NextArity) :-
	nonvar(V),
	V = arity(NextArity),!.
isnextcall([_|RestVars],Call,NextArity) :-
	isnextcall(RestVars,Call,NextArity).

