
/* Copyright (C) 1988, 1989 Herve' Touati, Aquarius Project, UC Berkeley */

/* Copyright Herve' Touati, Aquarius Project, UC Berkeley */


% Compile one procedure.
% Input is a list of clauses in unaltered form.
% Output is complete code for the procedure.
% The labels remain uninstantiated.
% The special compilation for lists, constants, structures
% is not needed if:
%   1. Arity=0, no first arguments.
%   2. procedure consists of just one clause.
%   3. all first arguments are variables.
% Also recognized are the cases where all first arguments are either
% variables or one other kind.

compileproc(_/Arity, Clauses, Code-Link) :-
	compileclauses(Clauses, CompC),
	var_block(CompC, VarLbl, VCode-VLink),
	cp(Arity, CompC, VarLbl, VCode, VLink, Code, Link), !.

% Easy optimizations 
cp(Arity, _, _, VCode, VLink, VCode, VLink) :- Arity=0, !.
cp(_, CompC, _, VCode, VLink, VCode, VLink) :- my_length(CompC,1), !.
cp(_, CompC, _, VCode, VLink, VCode, VLink) :- all_var(CompC), !.

% Only variables and one other kind present:
cp(_, CompC, VarLbl, VCode, VLink, Code, Link) :-
	same_or_var(CompC, Kind), !,
	filterv(CompC, VarC),
	try_block(VarC, TryLbl, VLink-TLink),
	cp_sub(Kind, CompC, TryLbl, VarLbl, TLink, Link, CLS),
	Switch=..[switch_on_term|CLS],
	Code=[Switch|VCode].

% General case: code for list, constant, and structure
cp(_, CompC, _, VCode, VLink, Code, Link) :-
	filterlcs(CompC, ListC, ConstC, StrucC),
	try_block(ListC, ListLbl, VLink-LLink),
	cs_block(ConstC, ConstLbl, LLink-CLink, _),
	cs_block(StrucC, StrucLbl, CLink-Link, _),
	Code=[switch_on_term(ConstLbl,ListLbl,StrucLbl)|VCode].

% Part of var & one other kind optimization:
cp_sub(list, _, TryLbl, VarLbl, TLink, TLink, CLS) :- !,
	CLS=[TryLbl, VarLbl, TryLbl], !.
cp_sub(Kind, CompC, TryLbl, VarLbl, TLink, Link, CLS) :-
	\+(Kind=list), !,
	cs_block(CompC, BlkLbl, BlkCode-BlkLink, Hashed),
	cp_hash(Hashed,CSLbl,TLink,VarLbl,Link,BlkLbl,BlkCode,BlkLink),
	cp_const_struc(Kind, CLS, CSLbl, TryLbl).

	cp_hash(no_hash, VarLbl, Link,    VarLbl, Link, _, _, _).
	cp_hash(yes_hash,BlkLbl, BlkCode, _,      Link, BlkLbl, BlkCode, Link).

	cp_const_struc(constant, [CSLbl,TryLbl,TryLbl], CSLbl, TryLbl).
	cp_const_struc(structure,[TryLbl,TryLbl,CSLbl], CSLbl, TryLbl).

% Succeeds if first arguments are all variable and one other kind:
same_or_var([clause(FArg,_,_)|Rest], Kind) :-
	kind(FArg, K),
	(K=variable; K=Kind),
	same_or_var(Rest, Kind).
same_or_var([], _).

% Succeeds if first arguments are all variables:
all_var(CompC) :- same_or_var(CompC, variable).


compileclauses([C|Clauses], [clause(FArg,Lbl,[label(Lbl)|Code]-Link)|Rest]) :-
	% !! getfirstarg must come before compileclause, since
	% compileclause instantiates variables in the head to registers.
	getfirstarg(C, FArg),
	gc(compileclause(C, Code, Link)), % garbage collect it.
	compileclauses(Clauses, Rest).
compileclauses([], []).

getfirstarg(Clause, FArg) :-
	(Clause=(Head:-Body); Clause=Head),
	Head=..[Name|HArgs],
	(HArgs=[Arg1|_]; true),
	gfa(Arg1, FArg).

gfa(Arg1, FArg) :- var(Arg1), !.
gfa(Arg1, Arg1) :- atomic(Arg1), !.
gfa(Arg1, Struc/Arity) :-
	Arg1=..[Struc|Args],
	my_length(Args, Arity).

%******************************************************************************

% Generate code for the four blocks:

% First block: 
% Link the clauses together with try_elses.
% Jumped to if the calling argument is a variable.
% Correctly handles cases of 1, 2, or more clauses.

var_block([clause(_,Lbl,Code-Link)], Lbl, Code-Link).
var_block(Clauses, Lbl, Code-Link) :-
	var_block(try, Clauses, Lbl, Code-Link).

var_block(_, [clause(_,_,C-L)], Lbl, [label(Lbl),trust(else,fail)|C]-L).
var_block(Type, [clause(_,_,C-L)|Clauses], PrevLbl, 
	        [label(PrevLbl),Instr|C]-Link) :-
	Instr=..[Type,else,NextLbl],
	var_block(retry, Clauses, NextLbl, L-Link).


% Filter out clauses which could match with a list, const, or struc
% as first argument.  Note that a variable as first argument matches
% with all of them.

filterlcs([], [], [], []).
filterlcs([X|Rest], [X|ListLbls], [X|ConstLbls], [X|StrucLbls]) :-
	X=clause(FArg, _, _),
	var(FArg), !,
	filterlcs(Rest, ListLbls, ConstLbls, StrucLbls).
filterlcs([X|Rest], [X|ListLbls], ConstLbls, StrucLbls) :-
	X=clause('.'/2, _, _),
	filterlcs(Rest, ListLbls, ConstLbls, StrucLbls).
filterlcs([X|Rest], ListLbls, [X|ConstLbls], StrucLbls) :-
	X=clause(FArg, _, _),
	atomic(FArg), !,
	filterlcs(Rest, ListLbls, ConstLbls, StrucLbls).
filterlcs([X|Rest], ListLbls, ConstLbls, [X|StrucLbls]) :-
	filterlcs(Rest, ListLbls, ConstLbls, StrucLbls).

% Filter out clauses with variables as first argument:

filterv([], []).
filterv([X|Rest], [X|VarLbls]) :-
	X=clause(FArg, _, _),
	var(FArg), !,
	filterv(Rest, VarLbls).
filterv([_|Rest], VarLbls) :-
	filterv(Rest, VarLbls).


% Try block: Generate a generic try-block to try
% all clauses in the given list.
% Optimizes code if only 0 or 1 clauses are given.

try_block([], fail, Link-Link).
try_block([clause(_,Lbl,_)], Lbl, Link-Link).
try_block([clause(_,Lbl,_)|Clauses], Label, 
	   [label(Label),try(Lbl)|LCode]-Link) :- 
	try_block(Clauses, LCode-Link).

try_block([clause(_,Lbl,_)], [trust(Lbl)|Link]-Link) :- !.
try_block([clause(_,Lbl,_)|Clauses], [retry(Lbl)|LCode]-Link) :-
	try_block(Clauses, LCode-Link).


% Const and Struc block: First argument is a constant or a structure.
% This routine works for both constants and structures.
% Difference with try_block: generates hash tables if needed.
% Variable Hashed indicates if hash tables were generated.
% It is either no_hash or yes_hash.

cs_block([], fail, Link-Link, no_hash).
cs_block([clause(_,Lbl,_)], Lbl, Link-Link, no_hash).
cs_block(Clauses, Lbl, [label(Lbl)|Code]-Link, Hashed) :-
	cs_gather(Clauses, [], Gather-[], Hashed),
	set_hashed(Hashed),
	cs_link(try, Gather, Code-Link).

	% Instantiate argument:
	set_hashed(no_hash) :- !.
	set_hashed(yes_hash).

% Gather contiguous arguments which are not variables together.
% The other arguments are left separate.

cs_gather([X|Rest], Collect, Gather-Link, H) :-
	X=clause(FArg, Lbl, _),
	var(FArg), !,
	dump(Collect, Gather-G, H),
	G=[X|G2],
	cs_gather(Rest, [], G2-Link, H).
cs_gather([X|Rest], Collect, Gather-Link, H) :-
	X=clause(FArg, Lbl, _),
	member(clause(FArg,_,_), Collect), !,
	dump(Collect, Gather-G, H),
	cs_gather(Rest, [X], G-Link, H).
cs_gather([X|Rest], Collect, Gather-Link, H) :-
	cs_gather(Rest, [X|Collect], Gather-Link, H).
cs_gather([], Collect, Gather-Link, H) :-
	dump(Collect, Gather-Link, H).

% Convert a collection of clause(s) to a member of Gather:
% If Collect is longer than one, it (as list) is a member.
% Else just its element clause is member.
dump([], L-L, _).
dump([X], [X|L]-L, _) :- X=clause(_, _, _).
dump(Collect, [Collect|L]-L, yes_hash).

% Link all elements of Gather together with try, retry, trust:

cs_link(Type, [Gr], Code-Link) :-
	cs_endlink(Gr, Type, Code, Link).
cs_link(Type, [Gr|Rest], Code-Link) :-
	cs_midlink(Gr, Type, Code, L),
	cs_link(retry, Rest, L-Link).

	% Middle hash table or (re)try instruction:
	cs_midlink(clause(_,Lbl,_), Type, [Instr|L], L) :- !,
		Instr=..[Type,Lbl].
	cs_midlink(Gr, Type, [Instr|Hash], L) :-
		hash(Gr, Hash-HLink),
		Instr=..[Type,else,ElseLbl],
		HLink=[label(ElseLbl)|L].

	% Last hash table or trust instruction:
	cs_endlink(clause(_,Lbl,_), _, [trust(Lbl)|Link], Link) :- !.
	cs_endlink(Gr, Type, Code, Link) :-
		hash(Gr, Hash-Link),
		cs_addtrust(Type, Code, Hash).

	% Add a trust if necessary:
	cs_addtrust(try, Hash, Hash) :- !.
	cs_addtrust(_, [trust(else,fail)|Hash], Hash).

% Generate hash table with switch instruction:
% This routine is mainly cosmetic.
hash(Gr, Code-Link) :-
	hash_table(Gr, HashTbl-Link, 0, HashLen),
	Mask is 2*HashLen-1,
	cs_kind(Gr, Kind),
	Code=[switch(Kind,Mask,Label),label(Label)|HashTbl].

% See if Gr is a bunch of constants or structures:
% No parameter needs to be passed to cs_block for this.
cs_kind([clause(FArg,_,_)|_], Kind) :- kind(FArg, Kind).

% Construct hash table.
% Dummy code here:

% put final pair on end, pad with fail instructions
hash_table([clause(FArg,Lbl,_)], [cdrpair(FArg,Lbl)|FailList]-Link,SoFar,Len) :-
	SoFar1 is SoFar + 1,
	ceil_2(SoFar1,Len),	% hash table length must be power of 2.
	PadLen is Len - SoFar1,
	failpad(FailList,PadLen,Link).
hash_table([clause(FArg,Lbl,_)|Rest], [pair(FArg,Lbl)|Hash]-Link, SoFar, Len) :-
	SoFar1 is SoFar + 1,
	hash_table(Rest, Hash-Link, SoFar1, Len).

% General utility: Returns kind of argument, can be
% 'variable', 'list', 'constant', 'structure'.
% Argument is in form struc/arity for lists and structures.
kind(Arg, variable) :- var(Arg), !.
kind(Arg, constant) :- atomic(Arg), !.
kind('.'/2, list) :- !.
kind(_, structure).

% Pad end of hash table with pairs of fails.
failpad(Link,0,Link).
failpad([cdrpair(fail,fail)|Rest],More,Link) :-
	M1 is More - 1, failpad(Rest,M1,Link).

% Find smallest power of two larger than given value.
ceil_2(In,Out) :- ceil_2(In,Out,1).
ceil_2(In,Power,Power) :- In =< Power, !.
ceil_2(In,Out,Power) :- Power2 is Power*2, ceil_2(In,Out,Power2).
