/*-
 * avail.c --
 *	Functions to check the status of the local machine to see
 *	if it can accept processes.
 *
 * Copyright (c) 1988, 1989 by the Regents of the University of California
 * Copyright (c) 1988, 1989 by Adam de Boor
 * Copyright (c) 1989 by Berkeley Softworks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any non-commercial purpose
 * and without fee is hereby granted, provided that the above copyright
 * notice appears in all copies.  The University of California,
 * Berkeley Softworks and Adam de Boor make no representations about
 * the suitability of this software for any purpose.  It is provided
 * "as is" without express or implied warranty.
 */
#ifndef lint
static char *rcsid =
"$Id: avail.c,v 1.24 1992/07/31 00:12:31 stolcke Exp $ ICSI (Berkeley)";
#endif lint

#include    <signal.h>

#include    "customsInt.h"

extern long		OS_Idle();
extern unsigned long	OS_Load();

static unsigned long   	maxLoad = DEF_MAX_LOAD * LOADSCALE;
static int  	  	minSwapFree = DEF_MIN_SWAP;
static int  	  	minIdle = DEF_MIN_IDLE;
static int  	  	maxImports = DEF_MAX_IMPORTS;
static int  	  	minProcFree = DEF_MIN_PROC;
int			cpuLimit = DEF_CPU_LIMIT;
int			niceLevel = DEF_NICE_LEVEL;
int			checkUser = DEF_CHECK_USER;
static int		evictDelay = DEF_EVICT_DELAY;

static Rpc_Event  	availEvent;	/* Event for checking availability */
static struct timeval	availInterval;	/* Interval at which checks should
					 * be made for the availability of
					 * this host. */
static int  	    	availCheck;  	/* Mask of criteria to examine */
int	    	    	avail_Bias; 	/* Bias for rating calculation */

static Rpc_Event	evictEvent;	/* Event for evicting imported jobs */
static struct timeval	evictInterval;	/* Delay till eviction */

/*-
 *-----------------------------------------------------------------------
 * Avail_Send --
 *	Send the availability of the local host.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	An availability packet is sent to the master.
 *
 *-----------------------------------------------------------------------
 */
Boolean
Avail_Send ()
{
    Avail   	  avail;
    static int	  sending = 0;	    /* XXX: A kludge to prevent endless
				     * recursion. At times, for no reason I've
				     * been able to determine, the avail event
				     * will be triggered during the call to
				     * CUSTOMS_AVAIL (hard to believe since
				     * the timeout for the avail event is
				     * twice as long as the total for the
				     * rpc, but...). Once it starts, it
				     * continues and the calls never seem to
				     * complete. To prevent this, we use a
				     * static flag and don't send anything
				     * if a call is already being sent. */

    if (sending) {
	return(FALSE);
    } else {
	sending = 1;
    }
    
    avail.addr =  	localAddr.sin_addr;
    avail.interval = 	availInterval;
    avail.avail = 	Avail_Local(AVAIL_EVERYTHING, &avail.rating);

    if (verbose) {
	xlog (XLOG_DEBUG, "Avail_Send: localhost %s available",
		avail.avail ? "not" : "is");
    }

    if (!Elect_InProgress() &&
	(Rpc_Call(udpSocket, &masterAddr,
		  (Rpc_Proc)CUSTOMS_AVAIL,
		  sizeof(avail), (Rpc_Opaque)&avail,
		  0, (Rpc_Opaque)0,
		  CUSTOMSINT_NRETRY, &retryTimeOut) != RPC_SUCCESS)) {
		      Elect_GetMaster();
    }
    sending = 0;
    return (FALSE);
}

/*-
 *-----------------------------------------------------------------------
 * AvailSet --
 *	Set the availability criteria. Returns an OR of bits if the
 *	parameters are out-of-range.
 *
 * Results:
 *	Any of AVAIL_IDLE, AVAIL_SWAP, AVAIL_LOAD, AVAIL_IMPORTS, AVAIL_PROC,
 *	AVAIL_CPU, AVAIL_NICE, AVAIL_CHECK, AVAIL_EVICT or'ed together
 *	(or 0 if things are ok).
 *
 * Side Effects:
 *	The availabilty criteria are altered.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
AvailSet (from, msg, len, adPtr, fromRemote)
    struct sockaddr_in	*from;	    /* Address of sender */
    Rpc_Message	  	msg;	    /* Message for return */
    int	    	  	len;	    /* Length of criteria */
    Avail_Data	  	*adPtr;	    /* New criteria */
    Boolean 	  	fromRemote; /* TRUE if from remote call */
{
    int	    	  result;

    if (!Local(from)) {
	Rpc_Error(msg, RPC_ACCESS);
    } else if (len != sizeof(Avail_Data)) {
	Rpc_Error(msg, RPC_BADARGS);
    } else {
	/*
	 * Criteria change only by root.
	 */
	if (adPtr->changeMask) {
	    CustomsReserved("AvailSet", from, msg);
	}
	/*
	 * Bounds-check the passed parameters, setting bits in result to
	 * correspond to bad values.
	 */
	result = 0;
	if ((adPtr->changeMask & AVAIL_IDLE) && (adPtr->idleTime > MAX_IDLE)) {
	    result |= AVAIL_IDLE;
	}
	if ((adPtr->changeMask & AVAIL_SWAP) && (adPtr->swapPct > MAX_SWAP)) {
	    result |= AVAIL_SWAP;
	}
	if ((adPtr->changeMask & AVAIL_LOAD) &&
	    (adPtr->loadAvg < MIN_LOAD) &&
	    (adPtr->loadAvg != 0))
	{
	    result |= AVAIL_LOAD;
	}
	if ((adPtr->changeMask & AVAIL_IMPORTS) &&
	    (adPtr->imports < MIN_IMPORTS) &&
	    (adPtr->imports != 0))
	{
	    result |= AVAIL_IMPORTS;
	}
	if ((adPtr->changeMask & AVAIL_PROC) && (adPtr->procs > MAX_PROC)) {
	    result |= AVAIL_PROC;
	}
	if ((adPtr->changeMask & AVAIL_CPU) &&
	    (adPtr->cpuLimit < MIN_CPU) &&
	    (adPtr->cpuLimit != 0))
	{
	    result |= AVAIL_CPU;
	}
	if ((adPtr->changeMask & AVAIL_NICE) &&
	    ((adPtr->niceLevel > MAX_NICE) ||
	     (adPtr->niceLevel < 0)))
	{
	    result |= AVAIL_NICE;
	}
	if ((adPtr->changeMask & AVAIL_CHECK) &&
	    ((adPtr->checkUser < DONT_CHECK) ||
	     (adPtr->checkUser > MAX_CHECK)))
	{
	    result |= AVAIL_CHECK;
	}
	if ((adPtr->changeMask & AVAIL_EVICT) &&
	    (adPtr->evictDelay < MIN_EVICT) &&
	    (adPtr->evictDelay != 0))
	{
	    result |= AVAIL_EVICT;
	}
	if (result == 0) {
	    /*
	     * Everything ok -- change what needs changing.
	     */
	    if (adPtr->changeMask & AVAIL_IDLE) {
		minIdle = adPtr->idleTime;
	    }
	    if (adPtr->changeMask & AVAIL_SWAP) {
		minSwapFree = adPtr->swapPct;
	    }
	    if (adPtr->changeMask & AVAIL_LOAD) {
		maxLoad = adPtr->loadAvg;
	    }
	    if (adPtr->changeMask & AVAIL_IMPORTS) {
		maxImports = adPtr->imports;
	    }
	    if (adPtr->changeMask & AVAIL_PROC) {
		minProcFree = adPtr->procs;
	    }
	    if (adPtr->changeMask & AVAIL_CPU) {
		cpuLimit = adPtr->cpuLimit;
	    }
	    if (adPtr->changeMask & AVAIL_NICE) {
		niceLevel = adPtr->niceLevel;
	    }
	    if (adPtr->changeMask & AVAIL_CHECK) {
		checkUser = adPtr->checkUser;
	    }
	    if (adPtr->changeMask & AVAIL_EVICT) {
		evictDelay = adPtr->evictDelay;
	    }
	}
	/*
	 * Set return value: changeMask gets error bits. the other fields get
	 * the current criteria.
	 */
	adPtr->changeMask = result;
	adPtr->idleTime = availCheck & AVAIL_IDLE ? minIdle : 0;
	adPtr->swapPct = availCheck & AVAIL_SWAP ? minSwapFree : 0;
	adPtr->loadAvg = availCheck & AVAIL_LOAD ? maxLoad : 0;
	adPtr->imports = availCheck & AVAIL_IMPORTS ? maxImports : 0;
	adPtr->procs = availCheck & AVAIL_PROC ? minProcFree : 0;
#ifdef SYSV
	adPtr->cpuLimit = 0;
#else
	adPtr->cpuLimit = cpuLimit;
#endif
	adPtr->niceLevel = niceLevel;
	adPtr->checkUser = checkUser;
	adPtr->evictDelay = evictDelay;

	/*
	 * Only send a reply if the call was actually remote (it's not
	 * when called from main...)
	 */
	if (fromRemote) {
	    Rpc_Return(msg, len, (Rpc_Opaque)adPtr);
	}
    }
}

/*-
 *-----------------------------------------------------------------------
 * AvailSetInterval --
 *	Alter the interval at which availability checks are made.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The interval in availInterval is changed and availEvent is altered
 *	to reflect this change.
 *
 *-----------------------------------------------------------------------
 */
static void
AvailSetInterval (from, msg, len, intervalPtr)
    struct sockaddr_in	*from;
    Rpc_Message	  	msg;
    int	    	  	len;
    struct timeval	*intervalPtr;
{
    if (!Local(from)) {
	Rpc_Error(msg, RPC_ACCESS);
    } else if (len != sizeof(struct timeval)) {
	Rpc_Error(msg, RPC_BADARGS);
    } else if (intervalPtr->tv_sec < MIN_CHECK) {
	Rpc_Error(msg, RPC_BADARGS);
    } else {
	/*
	 * Allow change only by root.
	 */
	CustomsReserved("AvailSetInterval", from, msg);

	availInterval = *intervalPtr;
	Rpc_EventReset(availEvent, &availInterval);
	Rpc_Return(msg, 0, (Rpc_Opaque)0);
    }
}

/*-
 *-----------------------------------------------------------------------
 * AvailEvict --
 *	Evict current imports from his machine.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Imports are notified of impending eviction (SIGUSR2 by default),
 *	terminated (SIGXCPU by default), or killed (SIGKILL).
 *
 *-----------------------------------------------------------------------
 */
static Boolean
AvailEvict (signo)
    int		signo;
{
    int nJobs;

    nJobs = Import_NJobs();
    Rpc_EventDelete(evictEvent);
    evictEvent = (Rpc_Event)0;

    /*
     * Depending on the severity of the current action, decide what to
     * do next.
     */
    switch (signo) {
    case EVICT_NOTIFY:
	if (verbose) {
	    xlog (XLOG_DEBUG,
		    "AvailEvict: notifying %d import(s) with signal %d",
		    nJobs, signo);
	}
	(void)Import_Kill(0, signo, (struct sockaddr_in *)0);
	/*
	 * Setup grace period for jobs that don't heed the warning.
	 * Prepare for orderly termination after that.
	 */
	evictInterval.tv_sec = evictDelay;
	evictInterval.tv_usec = 0;
	evictEvent = Rpc_EventCreate(&evictInterval, AvailEvict,
	                             (Rpc_Opaque)EVICT_SIGNAL);
	break;
    case EVICT_SIGNAL:
	if (verbose || nJobs > 0) {
	    xlog (verbose ? XLOG_DEBUG : XLOG_INFO,
		    "AvailEvict: evicting %d import(s) with signal %d",
		    nJobs, signo);
	}
	(void)Import_Kill(0, signo, (struct sockaddr_in *)0);
	/*
	 * Set up unconditional termination.
	 */
	evictInterval.tv_sec = GRACE_CPU;
	evictInterval.tv_usec = 0;
	evictEvent = Rpc_EventCreate(&evictInterval, AvailEvict,
	                             (Rpc_Opaque)SIGKILL);
	break;
    case SIGKILL:
	if (verbose || nJobs > 0) {
	    xlog (verbose ? XLOG_DEBUG : XLOG_INFO,
		    "AvailEvict: killing %d import(s) with signal %d",
		    nJobs, signo);
	}
	(void)Import_Kill(0, signo, (struct sockaddr_in *)0);
	/*
	 * It's all over.
	 */
	break;
    default:
	if (verbose)
	    xlog (XLOG_DEBUG, 
		    "AvailEvict: not sure what to do with signal %d", signo);
    }

    return (FALSE);
}

/*-
 *-----------------------------------------------------------------------
 * Avail_Init --
 *	Initialize things for here...
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	We exit if can't initialize.
 *
 *-----------------------------------------------------------------------
 */
void
Avail_Init(criteria, checkTime)
    Avail_Data	  *criteria;	    /* Initial criteria */
    int	    	  checkTime;	    /* Initial check interval */
{
    availInterval.tv_sec = checkTime ? checkTime : DEF_CHECK_TIME;
    availInterval.tv_usec = 0;

    availCheck = OS_Init();
    
    availEvent = Rpc_EventCreate(&availInterval, Avail_Send, (Rpc_Opaque)0);

    evictEvent = (Rpc_Event)0;
	
    Rpc_ServerCreate(udpSocket, CUSTOMS_AVAILINTV, AvailSetInterval,
		     Swap_Timeval, Rpc_SwapNull, (Rpc_Opaque)0);
    Rpc_ServerCreate(udpSocket, CUSTOMS_SETAVAIL, AvailSet,
		     Swap_Avail, Swap_Avail, (Rpc_Opaque)TRUE);
	
    AvailSet(&localAddr, (Rpc_Message)0, sizeof(Avail_Data), criteria,
	     FALSE);
}

/*-
 *-----------------------------------------------------------------------
 * Avail_Local --
 *	See if the local host is available for migration
 *
 * Results:
 *	0 if it is, else one of the AVAIL bits indicating which criterion
 *	wasn't satisfied.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
int
Avail_Local(what, ratingPtr)
    int	    what; 	/* Mask of things to check */
    long    *ratingPtr;	/* Place to store rating of current availabilty */
{
    static Boolean wasIdle = FALSE;	/* Idle last time we checked ? */

    /*
     * Mask out bits the OS module says it can't check.
     */
    what &= availCheck;
    /*
     * Start the rating out with the bias factor. The bias is intended for
     * situations where certains machines are noticeably faster than others
     * and are to be prefered even if the two appear to be loaded the same.
     */
#ifdef OLD_RATING
    *ratingPtr = avail_Bias;
#else
    *ratingPtr = 0;
#endif
    
    /*
     * If an minimum idle time was specified, check to make sure the
     * keyboard idle time exceeds that.
     */
    if ((what & AVAIL_IDLE) && minIdle) {
	long idleTime = OS_Idle();

	if (idleTime < minIdle) {
	    if (verbose) {
		xlog (XLOG_DEBUG,
			"Avail_Local: only %ld seconds idle (minimum %d)",
			idleTime, minIdle);
	    }
	    if (wasIdle) {
		/*
		 * Schedule evictions.
		 */
		wasIdle = FALSE;
		if (evictDelay > 0) {
		    evictInterval.tv_sec = 1;
		    evictInterval.tv_usec = 0;
		    evictEvent = Rpc_EventCreate(&evictInterval, AvailEvict,
		                                 (Rpc_Opaque)EVICT_NOTIFY);
		}
	    }
	    return AVAIL_IDLE;
	}

#ifdef OLD_RATING
	*ratingPtr += idleTime - minIdle;
#endif
    }

    /*
     * Idle again -- abort any evictions.
     */
    wasIdle = TRUE;
    if (evictEvent) {
	if (verbose)
	    xlog (XLOG_DEBUG, "Avail_Local: retracting eviction notice");
	Rpc_EventDelete(evictEvent);
	evictEvent = (Rpc_Event)0;
    }

    /*
     * Either the machine has been idle long enough or the user didn't
     * specify an idle time, so now, if the user gave a free swap space
     * percentage beyond which the daemon may not go, tally up the total
     * free blocks in the swap map and see if it's too few.
     */
    if ((what & AVAIL_SWAP) && minSwapFree) {
	int swapPct = OS_Swap();
	
	if (swapPct < minSwapFree) {
	    if (verbose) {
		xlog (XLOG_DEBUG, "Avail_Local: only %d%% free swap blocks",
			swapPct);
	    }
	    return AVAIL_SWAP;
	}
#ifdef OLD_RATING
	*ratingPtr += swapPct - minSwapFree;
#endif
    }

    /*
     * So far so good. Now if the user gave some maximum load average (note
     * that it can't be 0) which the daemon may not exceed, check all three
     * load averages to make sure that none exceeds the limit.
     */
    if ((what & AVAIL_LOAD) && maxLoad > 0) {
	unsigned long	load = OS_Load();

	if (load > maxLoad) {
	    if (verbose) {
		xlog (XLOG_DEBUG, "Avail_Local: load = %f",
			(double) load/LOADSCALE);
	    }
	    return AVAIL_LOAD;
	}
	*ratingPtr += maxLoad - load;
    }
#ifndef OLD_RATING
    else {
	/*
	 * Something non-zero so the division by the number of jobs will
	 * still be useful.
	 */
	*ratingPtr += 100;
    }
#endif

    /*
     * Reduce the rating proportional to the amount of work we've accepted if
     * we're not completely full. We weight this heavily in an attempt
     * to avoid double allocations by the master (by changing the rating
     * drastically, we hope to shift the focus to some other available machine)
     */
    if ((what & AVAIL_IMPORTS) && maxImports && (Import_NJobs() >= maxImports))
    {
	return AVAIL_IMPORTS;
    }
#ifdef OLD_RATING
    *ratingPtr -= Import_NJobs() * 200;
#else
    /*
     * A new criterion for availability is that a minimum additional number of
     * processes can still be created.
     */
    if ((what & AVAIL_PROC) && minProcFree) {
	int procs = OS_Proc();
	
	if (procs < minProcFree) {
	    if (verbose) {
		xlog (XLOG_DEBUG, "Avail_Local: only %d processes left",
			procs);
	    }
	    return AVAIL_PROC;
	}
    }
    /*
     * the rational here is that all imported jobs will get about equal
     * share of the cpu 
     */
    *ratingPtr /= (Import_NJobs() + 1);

    /* now interpret bias as a cpu speed factor (like MIPS)  */
    if ( avail_Bias == 0 ) avail_Bias = 1;	/* default bias */
    *ratingPtr *= avail_Bias;

    if (verbose) {
	xlog (XLOG_DEBUG,
		"Avail_Local: overall rating = %ld (bias factor = %d)",
		*ratingPtr, avail_Bias);
    }
#endif

    /*
     * Great! This machine is available.
     */
    return 0;
}
