/************************************************************************
 *									*
 * 	Background File Transfer Program (BFTP)				*
 *	May, 1991							*
 *									*
 *	Copyright (c) 1991 University of Southern California.		*
 *	All rights reserved.						*
 *									*
 *	Redistribution and use in source and binary forms are permitted	*
 * 	provided that the above copyright notice and this paragraph are	*
 * 	duplicated in all such forms and that any documentation,	*
 * 	advertising materials, and other materials related to such	*
 * 	distribution and use acknowledge that the software was		*
 *	developed by the University of Southern California, Information	*
 *	Sciences Institute.  The name of the University may not be used *
 *	to endorse or promote products derived from this software 	*
 * 	without specific prior written permission.			*
 *	THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR	*
 * 	IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
 * 	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 	*
 * 	PURPOSE.							*
 *									*
 ************************************************************************/

/*
 *	clbftp.c
 *
 *	The "bftp" command line interface is implemented in this module.
 *	
*/

#include <stdio.h>
#include <ctype.h>
#include <sys/param.h>

#ifndef SYSV
#include <sys/file.h>
#else
#include <sys/fcntl.h>
#endif /* SYSV */

#include "bftp.h"

extern boolean
	parse_date(),
	read_req(),
	write_req(),
	queue_req(),
	print_req();

extern char
	bftp_dir[MAXPATHLEN],
	def_user[20],
	def_mailbox[101],
	def_hostname[80],
	*expand_stru(),
	*expand_mode(),
	*get_request_file(),
	**get_save_files();

extern void 
	format_time(),
	unpack_filetype(),
	init_user(),
	init_req();

extern int
  	resolve_name();

extern struct hostinfo src, dst;
extern struct fileinfo fil;
extern struct reqinfo nullreq;
   
extern u_long start;

static struct reqinfo req;
u_long	local_addr;
int 
	verbose = FALSE,
	type = ASCII,    /* Ascii plus form, Ebcdic plus form,
			(form = Non-print, Telnet, or Carriage-control)
			Image, or Local plus Byte-size */
	bytesiz = DEFAULT_BYTESZ,
	form = NONPRINT;

char *flag;
boolean showC, transferC, verifyC, findC, exampleC, helpC;

#define p_help(param, str)\
	{ show(); if (str) printf("\n'%s %s'?  %s\n", flag, param, str);}

set_form(argc, argv, helpstr)
   int argc;
   char *argv[];
   char *helpstr;
{
   int len;
   
   if (argc < 1) {
      p_help("NONE", helpstr);
      exit(1);
      }

    len = strlen(argv[0]);
    if (!strncmp(argv[0], "nonprint", len?len:1))
       form = NONPRINT;
    else if (!strncmp(argv[0], "telnet", len?len:1))
       form = TELNET; 
    else if (!strncmp(argv[0], "carriage-control", len?len:1))
       form = CCNTRL; 
    else {
       p_help(argv[0], helpstr);
       exit(1);
       }
   sprintf(fil.filetype,"%c %s", (type == ASCII)? 'A': 'E',
	   (form==NONPRINT)?"N":(form==TELNET)?"T":"C");
} /* set_form */

int
set_host(argc, argv, helpstr, result)
   int argc;
   char *argv[];
   char *helpstr, *result;
{
   u_long inaddr;

   if (argc < 1) {
      p_help("NONE", helpstr);
      exit(1);
      }

   if (resolve_name(argv[0], &inaddr, 1)) {
      if (result) 
	 strcpy(result, argv[0]);
      }
   else {
      p_help(argv[0], helpstr);
      exit(1);
      }
} /* set_host */

set_mbox(argc, argv, helpstr)
   int argc;
   char *argv[];
   char *helpstr;
{
   char temp[80], errorstr[40], helpstring[200];

   if (argc)
      strcpy(temp, argv[0]);
   else if (strlen(def_mailbox))
      strcpy(temp, def_mailbox);
   else {
      p_help("NONE", helpstr);
      exit(1);
      }

   if (!mailbox_ok(temp, errorstr)) {
      errorstr[strlen(errorstr)-1] = '\0';
      sprintf(helpstring, "%s; %s", helpstr, errorstr);
      p_help(temp, helpstring);
      exit(1);
      }
   else
      strcpy(req.mailbox, temp);
} /* set_mbox */

set_mode(argc, argv, helpstr)
   int argc;
   char *argv[];
   char *helpstr;
{
   int len;
   
   if (argc < 1) {
	 p_help("NONE", helpstr);
	 exit(1);
         }

   len = strlen(argv[0]);
   if (!strncmp(argv[0], "stream", len?len:1))
      fil.mode = 'S';
   else if (!strncmp(argv[0], "block", len?len:1))
      fil.mode = 'B';
   else if (!strncmp(argv[0], "compressed", len?len:1))
      fil.mode = 'C';
   else {
       p_help(argv[0], helpstr);
       exit(1);
       }
} /* set_mode */

int
check_number(arg)
   char *arg;
{
   int num;
   char *cp;
	
   for (cp = arg; *cp; cp++) 
      if (! isdigit(*cp))
	 return(-1);
   num = atoi(arg);
   return(num);
} /* check_number */

set_number(argc, argv, helpstr, result)
   int argc;
   char *argv[];
   char *helpstr;
   int *result;
{
   int temp;
   
   if (argc < 1) {
	 p_help("NONE", helpstr);
	 exit(1);
      }

   if ((temp = check_number(argv[0])) > 0)
      *result = temp;
   else {
       p_help(argv[0], helpstr);
       exit(1);
       }
} /* set_number */

set_string(argc, argv, helpstr, result)
   int argc;
   char *argv[];
   char *helpstr, *result;
{
   if (argc && argv[0][0] == '?') {
      printf("Type <CR> to blank field.\n%s%s\n", 
	     (helpstr)?"?":NULLSTR, (helpstr)?helpstr:NULLSTR);
      exit(1);
      }

   if (result) strcpy(result, (argc)?argv[0]:NULLSTR);
} /* set_string */

set_stru(argc, argv, helpstr)
   int argc;
   char *argv[];
   char *helpstr;
{
   int len;
   
   if (argc < 1) {
	 p_help("NONE", helpstr);
	 exit(1);
         }

   len = strlen(argv[0]);
   if (!strncmp(argv[0], "file", len?len:1))
      fil.stru = 'F';
   else if (!strncmp(argv[0], "page", len?len:1))
      fil.stru = 'P';
   else if (!strncmp(argv[0], "record", len?len:1))
      fil.stru = 'R';
   else {
       p_help(argv[0], helpstr);
       exit(1);
       }

   if (fil.stru =='P') {
      fil.mode = 'S';
      strcpy(fil.filetype, "L 36");
      printf("Setting mode = 'stream'; type = 'local 36'.\n");
      }
} /* set_stru */

set_time(argc, argv, helpstr, def)
   int argc;
   char *argv[];
   char *helpstr;
   time_t def;
{
   int i;
   time_t date;
   char temp[80];

   if (argc < 1) {
       p_help("NONE", helpstr);
       exit(1);
       }

   start = (def > time(NULL))? def: 0;
   if (parse_date(argv[0], &date)) 
	 start = (date>time(NULL))? date: 0;
      else {
	 p_help(argv[0], helpstr);
	 exit(1);
         }
} /* set_time */

set_tmode(argc, argv, helpstr)
   int argc;
   char *argv[];
   char *helpstr;
{
   int len;
   
   if (argc < 1) {
	 p_help("NONE", helpstr);
	 exit(1);
         }

    len = strlen(argv[0]);
    if (!strncmp(argv[0], "copy", len?len:1))
       fil.reqtype = COPY;
    else if (!strncmp(argv[0], "move", len?len:1))
       fil.reqtype = MOVE; 
    else if (!strncmp(argv[0], "delete", len?len:1))
       fil.reqtype = DFILE; 
    else {
       p_help(argv[0], helpstr);
       exit(1);
       }
} /* set_tmode */

set_type(argc, argv, prompt, helpstr)
   int argc;
   char *argv[];
   char *helpstr;
{
   int len;
   
   if (argc < 1) {
	 p_help("NONE", helpstr);
	 exit(1);
         }

   len = strlen(argv[0]);
   if (!strncmp(argv[0], "text", len?len:1)) {
       type = ASCII;
       form = NONPRINT;
       strcpy(fil.filetype, "A N");
       }
   else if (!strncmp(argv[0], "ascii", len?len:1)) {
       type = ASCII;
       sprintf(fil.filetype,"A %s",
	       (form==NONPRINT)?"N":(form==TELNET)?"T":"C");
       }
   else if (!strncmp(argv[0], "ebcdic", len?len:1)) {
       type = EBCDIC;
       sprintf(fil.filetype,"E %s",
	       (form==NONPRINT)?"N":(form==TELNET)?"T":"C");
       }
   else if (!strncmp(argv[0], "local", len?len:1)) {
       type = LOCAL;
       sprintf(fil.filetype,"L %d",bytesiz);
       }
   else if (!strncmp(argv[0], "image", len?len:1) ||
	    !strncmp(argv[0], "binary", len?len:1)) {
       type = IMAGE;
       strcpy(fil.filetype,"I");
       }
   else {
       p_help(argv[0], helpstr);
       exit(1);
       }
} /* set_type */

example()
{
char *xText = 
"\nCommand-line bftp (clbftp):\n\n\
 See the online help text for the \"bftp\" program for more explanation\n\
 of how BFTP works.  Try \"clbftp -H\" for a complete list of options.\n\n\
 Simple example of how to submit a background file tranfer request\n\
 to be run immediately:\n\n\
    clbftp -sh venera.isi.edu -su anonymous -sp guest -sd pub\n\
           -sf sample-file.txt -dh hobgoblin.isi.edu -du deschon\n\
	   -dp secret<CR>\n\n\
 To submit a request that will run at 2:00AM and send the results to a\n\
 mailbox other than the mailbox of the current login:\n\n\
    clbftp -sh venera.isi.edu -su anonymous -sp guest -sd pub\n\
	   -sf sample-file.txt -dh hobgoblin.isi.edu -du deschon\n\
	   -dp secret -s '2:00AM' -m jsmith@xyz.com<CR>\n\n\
 To avoid putting passwords in the command line one can create a\n\
 request-file using the \"bftp\" program, and read it into \"clbftp\".\n\
 To read in parameters from a request-file made with \"bftp\", specify all\n\
 file names matching the wildcard specification '*.c', and run in the\n\
 foreground:\n\n\
    clbftp -g myBackup -w -sf '*.c' -T<CR>\n\n\
 To submit a job, with the keyword \"myjobs\":\n\n\
    clbftp -g myBackup -w -sf '*.c' -k myjobs<CR>\n\n\
 To find all previously submited jobs, with the keyword \"myjobs\" that have,\n\
 not yet completed:\n\n\
    clbftp -k myjobs -F<CR>\n\n\
";
   printf(xText);
}

help()
{
char *hText1 = 
"\n\
  The following is a list of options; try \"clbftp -X\" for examples.\n\n\
  Note that key words can be abbreviated, for example \"-fs record\" is the\n\
  same as \"-fs r\".  Also only one command will be performed even if more\n\
  than one is specified; e.g. if the options \"-H -T\" are specified, only\n\
  the help command will be performed.  If contradictory options other than\n\
  commands are entered, the last will take precedence.\n\n\
    -a		   Append onto the destination file rather than replacing it.\n\
    -b byte-size   Byte size used in \"local file-type\" transfers; e.g.\n\
		   \"-t local -b 8\".\n\
    -c mode	   Transfer mode; \"-c copy\", the default, in which the\n\
		   source file is copied to the destination file name,\n\
		   \"-c move\", in which the source file is deleted after\n\
		   it has been copied, or \"-c delete\", in which the source\n\
		   file is deleted.\n\
    -da	account	   Destination account string; e.g. \"-da myaccount\".\n\
    -df	filename   Destination file name, e.g. \"-df new_name.txt\"; this\n\
		   parameter does not need to be specified unless it is\n\
		   different from the source file name.\n\
    -dh	host	   Destination host name/number; e.g. \"-dh venera.isi.edu\"\n\
		   or \"-dh 128.9.0.32\".\n\
    -dn	port	   Destination port number (the default is 21 -- this\n\
		   parameter is rarely needed); e.g. \"-dn 4200\".\n\
    -dp	password   Destination password, e.g. \"-dp secret\".\n\
    -du	login	   Destination login; e.g. \"-du deschon\".\n\
    -e #tries	   Maximum number of times to attempt the transfer, e.g.\n\
		   \"-e 10\".\n\
    -ff format	   File format; \"-ft ascii -ff nonprint\" (the default),\n\
		   \"-ff telnet\", or \"-ff carriage-control\"\n\
    -fm	mode	   File mode; \"-fm stream\" (the default), \"-fm block\",\n\
		   \"-fm compress\".\n\
    -fs structure  File structure; \"-fs file\" (the default),\"-fs record\",\n\
		   or \"-fs page\".\n\
    -ft	type	   File type; e.g. \"-ft text\" (the default), \"-ft image\",\n\
		   \"-ft binary\" (same as image), \"-ft ascii -ff nonprint\"\n\
		   (same as text), \"-ft ebcdic\", or \" -ft local -b 8\".\n";
char *hText2 = "\
    -g req-file	   Get a request file previously entered using the \"bftp\"\n\
		   program; e.g. \"-g myRequestFile\".  If some parameters\n\
		   are to be over-written with additional options, the \"-g\"\n\
		   option must appear first.\n\
    -i minutes	   Starting value for interval in minutes between tries,\n\
		   e.g. \"-i 5\".\n\
    -k req-name	   Key word for this request (optional); e.g. \"-k lists\".\n\
		   This parameter can be used to narrow down the submitted\n\
		   requests that are displayed by the \"find\" command, e.g.\n\
		   \"-k lists -F\".\n\
    -m mailbox	   The mailbox to which a report on the outcome of a\n\
		   background transfer will be returned; e.g.\n\
		   \"-m deschon@isi.edu\".\n\
    -s start-time  The date/time that a background file transfer will be\n\
		   started.  Multiple words must be enclosed in single\n\
		   quotes; e.g. \"-s 'Mar 13 1991 2:00AM'\"\n\
    -sa	account	   Source account string; e.g. \"-sa myaccount\".\n\
    -sf	filename   Source file name, e.g. \"-sf sample_file.txt\", special\n\
		   characters must be enclosed in single quotes,\n\
		   e.g. \"-sf '*.c'\"\n\
    -sh	host	   Source host name/number; e.g. \"-sh hobgoblin.isi.edu\"\n\
		   or \"-sh 128.9.0.42\"\n\
    -sn	port	   SourceDestination port number (the default is 21 -- this\n\
		   parameter is rarely needed); e.g. \"-sn 4200\".\n\
    -sp	password   Source password, e.g. \"-sp guest\".\n\
    -su	login	   Source login; e.g. \"-su anonymous\".\n\
    -u		   Unique option -- The STOU (FTP) command will be used to\n\
		   create the file.\n\
    -v		   Verbose option -- in a foreground transfer FTP commands\n\
		   will be echoed; e.g. \"-v -T\".\n\
    -w		   Wildcard option -- enables wildcard matching, e.g.\n\
		   \"-w -sf '*.c'\".\n\
    -F		   Find command -- find previous request and display status.\n\
    -H		   Help command -- Prints this list of options.\n\
    -S		   Show command -- Prints the current value of all of the\n\
		   parameters without actually attempting a transfer.  This\n\
		   command is used for testing \"clbftp\".\n\
    -T		   Transfer command -- transfer the file in the foreground.\n\
    -V		   Verify command -- make connections to check parameters.\n\
    -X		   Example command -- display some examples.\n\
";
   printf("%s%s", hText1, hText2);
   /* split into two parts so that string will not be too long for
      SunOS3 compiler */
}

show()
{
    char temp[40];

    printf("\n");
    print_req(stdout, &req, &src, &dst, &fil, TRUE);
    
    if (start) {
	format_time(start, temp);
	printf("    Start after %s.  Verbose mode = %s\n", 
		temp, (verbose)? "ON": "OFF");
	}
    else
       printf("    Start immediately.  Verbose mode = %s\n", 
       		(verbose)? "ON": "OFF");
}
getport(argc, argv, source)
   int argc;
   char *argv[];
   boolean source;
{
   int port;

   set_number(argc, argv,
	    "Port number (usually 21)",
	    &port);
   if (port > 0xffff) {
      p_help(argv[0], "Illegal port number");
      exit(1);
      }
   else
      if (source)
	 src.port = port;
      else
	 dst.port = port;
}

int
finderrors()
{
   u_long addr;
   int count = 0;
	
   printf("  Checking parameters...\n");

   if (!resolve_name(src.host, &addr, 1)) {
      printf("\tIllegal source host number.\n");
      count++;
      }
   if (!strlen(src.user)) {
      printf("\tSource user name must be specified.\n");
      count++;
      }
   if (!strlen(src.file)) {
      printf("\tSource file name must be specified.\n");
      count++;
      }
      
   if (fil.reqtype != DFILE)
      if (!resolve_name(dst.host, &addr, 1)) {
	 printf("\tIllegal destination host number.\n");
	 count++;
	 }
      if (!strlen(dst.user)) {
         printf("\tDestination user name must be specified.\n");
         count++;
         }

   if (count)
      show();
   return(count);
} /* finderrors */

verify(argc, argv)
   int argc;
   char *argv[];
{
   int save_mode,
       filelen = strlen(dst.file);

   if (!filelen)
      strcpy(dst.file, src.file);
   if (!finderrors()) {
      save_mode = fil.reqtype;
      fil.reqtype = (save_mode == DFILE)? VERIFY_SRC : VERIFY;
      submit_req(TRUE);
      fil.reqtype = save_mode;
      }
   if (!filelen)
      dst.file[0] = '\0';
}

transfernow(argc, argv)
   int argc;
   char *argv[];
{
   int filelen = strlen(dst.file);

   if (!filelen)
      strcpy(dst.file, src.file);
   if (!finderrors())
      submit_req(TRUE);
   if (!filelen)
      dst.file[0] = '\0';
}

submit(argc, argv)
   int argc;
   char *argv[];
{
   char temp[80];
   int len = strlen(req.mailbox);
   int filelen = strlen(dst.file);

   if (!len || !mailbox_ok(req.mailbox, temp)) {
      printf("No valid mailbox specified: '%s'; use -M option.\n", req.mailbox);
      show();
      }
   else {
      if (!filelen)
	 strcpy(dst.file, src.file);
      if (!finderrors())
	 submit_req(FALSE);
      if (!filelen)
	 dst.file[0] = '\0';
      }
} /* submit */

submit_req(realtime)
   int realtime;
{
   FILE *xxx;
   char temp[MAXPATHLEN], path[MAXPATHLEN];
   u_long now;

   now = time(NULL);
   sprintf(path,"%s%d",REQPREFIX,now);
   
   sprintf(temp, "%s.req", path);
   if (realtime) {
      write_req(temp,&nullreq,&src,&dst,&fil,NULL);
      sprintf(temp, "%s %s%s.req\n", FXPATH, (verbose)?"-v ":NULLSTR, path);
      printf(temp);
      system(temp); 
      }
   else {
      sprintf(req.mailfile,"%s.msg",path);
      sprintf(req.cmdfile,"%s.cmd",path);
      write_req(temp,&req,&src,&dst,&fil,NULL);
      
      sprintf(temp, "%s.cmd", path);
      xxx = fdopen(open(temp,(O_WRONLY|O_CREAT),0600),"w");
      fprintf(xxx, "%s -v %s.req\n", FXPATH, path);
      fclose(xxx);
   
      sprintf(temp, "%s.msg", path);
      xxx = fdopen(open(temp,(O_WRONLY|O_CREAT),0600),"w");
      queue_req(&req, (now > start) ? now : start, temp);
      print_req(xxx, &req, &src, &dst, &fil, FALSE);
      fprintf(xxx,"\n%s\n",temp);
      fclose(xxx);
      printf("\n%s\n\n",temp);
      }
} /* submit_req */

/* Find and display a request */

summarize(req)
   struct reqinfo *req;
{
   char temp[150];

   if (strlen(req->mailfile)) {
      printf("History:\n\n");
      sprintf(temp,"egrep 'submitted|starting|completed' %s\n",req->mailfile);
      system(temp);
      printf("\n");
      }
   else
      printf("History: No mail file found!\n\n");
}

char request_id[20], request_key[MAXLEN];

find(argc, argv)
   int argc;
   char *argv[];
{
   char temp[MAXLEN], id [20];
   char filename[MAXPATHLEN], listfile[MAXPATHLEN], *filep;
   u_long now;
   boolean found;
   int tmp;
   struct hostinfo tmpsrc, tmpdst;
   struct fileinfo tmpfil;
   struct reqinfo tmpreq;

   if (strlen(request_id)) {
      sprintf(filename,"%s.req",request_id);
      if (! read_req(filename, &tmpreq, &tmpsrc, &tmpdst, &tmpfil, listfile)) {
         printf("\nRequest %s not found.\n",request_id);
	 return;
         }
      }

   if (strlen(request_id))
      filep = filename;
   else 
      filep = get_request_file(request_key);
   
   while (filep) {
      if ((read_req(filep, &tmpreq, &tmpsrc, &tmpdst, &tmpfil, listfile)) &&
          (strcmp(request_key, tmpreq.rpasswd) == 0)) {
	 get_id(id, filep);
	 printf("\nRequest: %s\n\n",id);
	 print_req(stdout, &tmpreq, &tmpsrc, &tmpdst, &tmpfil, FALSE);
	 summarize(&tmpreq);
	     
	 found = request_queued(filep);
	 printf("Your request is %scurrently queued.\n",
		(found)?NULLSTR:"NOT ");
         }
      if (strlen(request_id))
         break;
      else 
         filep = get_request_file(request_key);
      }
} /* find */

/* Request file management routines */

boolean
reqname(argc, argv, nptr)
   int argc;
   char *argv[];
   char *nptr;
{
   char temp[40], *cp;

   if (!strlen(def_user)) {
      printf(
	 "\n\tSorry, you must be logged in for temporary request storage.\n\n");
      return(FALSE);
      }
   set_string(argc, argv,
	      "Request name",
	      temp);

   if (!strlen(temp))
      return(FALSE);
   else {
      sprintf(nptr, "%s.%s", SAVEPREFIX, temp);
      return(TRUE);
      }
} /* reqname */

listsavef()
{
   char **list, **temp;
   int i, maxlen = 0, linelen = 80;
	
   list = get_save_files();
   temp = list;
   while (*temp)
      if ((i = strlen(*temp++)) > maxlen) maxlen = i;

   maxlen += 4;
   while (*list) {
      if ((linelen+maxlen) > 80) {
	   linelen = 2;
	   printf("\n  ");
	   };
      printf("%-*s", maxlen, *list++);
      linelen += maxlen;
      }
   printf("\n\n");
} /* listsavef */

getfile(argc, argv)
   int argc;
   char *argv[];
{
   char temp[MAXLEN], listfile[MAXLEN];

   if (reqname(argc, argv, temp)) {
      printf("\n Reading %s%s\n",bftp_dir, temp);
      if (! read_req(temp,&req,&src,&dst,&fil,listfile)) {
         printf("\n Request file not found; existing files are:\n");
         listsavef();
	 exit(1);
         }
      else
         unpack_filetype(fil.filetype,&type, &form, &bytesiz);
      }
}

init()
{
    resolve_name(def_hostname, &local_addr, 1);

    strcpy(src.dir, NULLSTR);
    strcpy(src.file, NULLSTR);
    strcpy(src.host, NULLSTR);
    strcpy(src.user, NULLSTR);
    strcpy(src.passwd, NULLSTR);
    strcpy(src.acct, NULLSTR);
    src.port = DEFAULT_PORT;
    strcpy(dst.dir, NULLSTR);
    strcpy(dst.file, NULLSTR);
    strcpy(dst.host, NULLSTR);
    strcpy(dst.user, NULLSTR);
    strcpy(dst.passwd, NULLSTR);
    strcpy(dst.acct, NULLSTR);
    dst.port = DEFAULT_PORT;
    
    init_req(&req);
    
    fil.reqtype = COPY;
    fil.stru = 'F';
    type = ASCII;
    bytesiz = DEFAULT_BYTESZ;
    form = NONPRINT;
    strcpy(fil.filetype,"A N");
    fil.mode = 'S';
    start = 0;
    fil.creation = STOR;
    fil.multflag = FALSE;

    /* the default command is "submit" */
    showC = transferC = verifyC = findC = exampleC = helpC = FALSE;
} /* init */

/* Command table for BFTP */
/*
struct cmd cmdtab[] = {
	{ "submit",
	    "Submit the current request for background FTP.",
	    submit },
	{ "find",
	    "Find and display a previous request (or cancel it).",
	    find },
	{ "transfer", 
	    "Perform the current request in the foreground.",
	    transfernow },
	{ "verify", 
	    "Make the connections now to check parameters.",
	    verify },
	{ 0 },
};
*/

void
get_args(argcp, argvp)
   int *argcp;
   char ***argvp;
{
   boolean source;

   while (--*argcp > 0 && *++*argvp) {
      source = FALSE;
      flag = **argvp;
      if (***argvp == '-')
	 switch (*(**argvp+1)) {
	   case 'a':
	     fil.creation = APPE;
	     break;
	   case 'b':
	     set_number(--*argcp, ++*argvp, 
			"An integer byte size",
			&bytesiz);
	     if (type == LOCAL)
	        sprintf(fil.filetype,"L %d",bytesiz);
	     break;
	   case 'c':
	     set_tmode(--*argcp, ++*argvp,
		       "Set the transfer mode.\n\
\t    copy:   Source file is copied to the destination file name.\n\
\t    move:   Source file is deleted after it has been copied.\n\
\t    delete: Source file is deleted.");
	     break;
	   case 'e':
	     set_number(--*argcp, ++*argvp, 
			"An integer maximum number of attempts.",
			&req.ntries);
	     break;
	   case 'F': 
	     findC = TRUE;
	     break;
	   case 'g':
	     getfile(--*argcp, ++*argvp);
	     break;
	   case 'H':
	     helpC = TRUE;
	     break;
	   case 'i':
	     set_number(--*argcp, ++*argvp, 
		 "An integer starting value for minutes between tries.",
		  &req.interval);
	     break;
	   case 'k':
	     /* req.rpasswd used for transfer or submit */
	     set_string(--*argcp, ++*argvp,
			"Key word for this request (optional)",
			req.rpasswd);
	     /* request_key used for find */
	     strcpy(request_key, req.rpasswd);
	     break;
	   case 'm':
	     set_mbox(--*argcp, ++*argvp,
		      "Where results will be returned");
	     break;
	   case 'T':
	     transferC = TRUE;
	     break;
	   case 'u':
	     fil.creation = STOU;
	     break;
	   case 'V':
	     verifyC = TRUE;
	     break;
	   case 'v':
	     verbose = TRUE;
	     break;
	   case 'w':
	     fil.multflag = TRUE;
	     break;
	   case 'X':
	     exampleC = TRUE;
	     break;
	   case 'S':
	     showC = TRUE;
	     break;

	   /* two-letter options f*, s* and d* */
	   case 'f':
	     switch (*(**argvp+2)) {
	       case 'f': /* -ff */
	         set_form(--*argcp, ++*argvp, 
		 "File format (nonprint/telnet/carriage-control -- usually 'nonprint')");
	         break;
	       case 'm': /* -fm */
		 set_mode(--*argcp, ++*argvp, 
		      "Mode (stream/block/compress -- usually 'stream')");
		 break;
	       case 's': /* -fs */
		 set_stru(--*argcp, ++*argvp, 
		     "File structure (file/record/page -- usually 'file')");
		 break;
	       case 't': /* -ft */
		 set_type(--*argcp, ++*argvp, 
		      "File type (text ('a n') / binary (image) / ascii / ebcdic / image / local)");
		 break;
               default:
	         printf("\nBad option: %s\n", **argvp);
		 exit(1);
	       }
	     break;
	   case 's': 
	     if ((*(**argvp+2) == 't') || (*(**argvp+2) == '\0')) {
	        /* '-st' or '-s' for start time */
	        set_time(--*argcp, ++*argvp,
		    "Enter start time in single quotes, e.g. 'May 30 10:00AM'",
		    0);
		break;
	        }
	     source = TRUE;
	     /* fall through */
	   case 'd':
	     switch (*(**argvp+2)) {
	       case 'a':
		 set_string(--*argcp, ++*argvp,
			    "Account string for this login",
			    (source)?src.acct:dst.acct);
		 break;
	       case 'd':
		 set_string(--*argcp, ++*argvp,
			    (source)? "Directory where source file is located"
			            : "Directory where destination file will be put",
			    (source)? src.dir: dst.dir);
		 break;
	       case 'f':
		 set_string(--*argcp, ++*argvp,
			    (source)? "Source file name"
			    	    : "Destination file name",
			    (source)? src.file: dst.file);
		 break;
	       case 'h':
		 set_host(--*argcp, ++*argvp,
			  "Host name / dotted-decimal host number.",
			  (source)?src.host:dst.host);
		 break;
	       case 'n':
		 getport(--*argcp, ++*argvp, source);
		 break;
	       case 'p':
		 set_string(--*argcp, ++*argvp,
			    "Password for this login",
			    (source)?src.passwd:dst.passwd);
		 break;
	       case 'u':
		 set_string(--*argcp, ++*argvp,
			    "User name / login.",
			    (source)?src.user:dst.user);
		 break;
	       default:
		 printf("\nBad option: %s\n", **argvp);
		 exit(1);
	       }
	     break;
	   default:
	     printf("\nBad option: %s\n", **argvp);
	     exit(1);
	   }
      else {
	 printf("\nBad option: %s\n", **argvp);
	 exit(1);
         }
      }
}

main(argc, argv)
    int argc;
    char *argv[];
{
   init_user();
   init();

   get_args(&argc, &argv);

   if (exampleC)
      example();
   else if (helpC)
      help();
   else if (showC)
      show();
   else if (findC)
      find();
   else if (verifyC)
      verify();
   else if (transferC)
      transfernow();
   else
      submit();
   exit(0);
} /* main */
