/************************************************************************
 *									*
 * 	Background File Transfer Program (BFTP)				*
 *	May, 1991							*
 *									*
 *	Copyright (c) 1991 University of Southern California.		*
 *	All rights reserved.						*
 *									*
 *	Redistribution and use in source and binary forms are permitted	*
 * 	provided that the above copyright notice and this paragraph are	*
 * 	duplicated in all such forms and that any documentation,	*
 * 	advertising materials, and other materials related to such	*
 * 	distribution and use acknowledge that the software was		*
 *	developed by the University of Southern California, Information	*
 *	Sciences Institute.  The name of the University may not be used *
 *	to endorse or promote products derived from this software 	*
 * 	without specific prior written permission.			*
 *	THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR	*
 * 	IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED	*
 * 	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 	*
 * 	PURPOSE.							*
 *									*
 ************************************************************************/

/*
 *	bftp_req.c
 *
 *	These routines are used in "bftp" and "bftptool", user interface
 *	modules, and also in "fts", the file transfer driver.  Included
 *	are routines which read and write a standard BFTP request file.
*/

#include <stdio.h>
#include <errno.h>
#ifndef SYSV
#include <strings.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/timeb.h>
#else /* SYSV */
#include <string.h>
#include <sys/types.h>  /* must include before file.h */
#include <sys/file.h>
#include <sys/fcntl.h>
#include <time.h>
#endif /* SYSV */
#include "time.h"
	  
#include "bftp.h"

#define endof(x) (x + strlen(x))

#ifdef SYSV
#define rindex strrchr
#endif /* SYSV */

char *expand_stru(c)
   char c;
{
   return( (c == 'F')? "file":
   	   (c == 'R')? "record":
	   (c == 'P')? "page":
	   	       "unknown");
}
char *expand_mode(c)
   char c;
{
   return( (c == 'S')? "stream":
   	   (c == 'B')? "block":
	   (c == 'C')? "compress":
	   	       "unknown");
}
char *expand_type(c)
   char c;
{
   return( (c == 'A')? "ascii":
   	   (c == 'E')? "ebcdic":
	   (c == 'I')? "image":
	   (c == 'L')? "local":
	   	       "unknown");
}
char *expand_form(c)
   char c;
{
   return( (c == 'N')? "nonprint":
   	   (c == 'T')? "telnet":
	   (c == 'C')? "carriage-control":
	   	       "unknown");
}

boolean
get_id(id, filename)
   char *id, *filename;
{
   char *temp, *dot;

   if ((temp = rindex(filename, '/')) == NULL)
      temp = filename;
   else 
      temp++;
   if (strlen(temp) == 0)
      return(FALSE);
   else {
      strcpy(id, temp);
      if ((dot = rindex(temp, '.')) != NULL) {
         dot = rindex(id, '.');
	 *dot = '\0';
	 }
      return(TRUE);
      }
} /* get_id */

void
name_dotfile(file, dotfile)
   char *file, *dotfile;
{
   int len;
   char *cp;

   if (strlen(file))
      if ((cp = rindex(file, '/')) != NULL) {
	 len = strlen(file) - strlen(cp) + 1;
	 strncpy(dotfile, file, len);
	 dotfile[len] = '\0';
	 strcat(dotfile, ".");
	 strcat(dotfile, ++cp);
         }
      else {
	 strcpy(dotfile, ".");
	 strcat(dotfile, file);
         }
   else
      dotfile[0] = '\0';

} /* name_dotfile */

int
#ifndef SYSV
find_job(reqfile)
   char *reqfile;
#else /* SYSV */
find_job(reqfile, jobname)
   char *reqfile, *jobname;
#endif /* SYSV */
{
   FILE *stream;
   char temp[80];
   int jobno = 0;
   
#ifndef SYSV
   strcpy(temp, "atq | grep ");
#else /* SYSV */
   strcpy(temp, "grep ");
#endif /* SYSV */
   if (get_id(endof(temp), reqfile))
#ifdef SYSV
      sprintf(endof(temp), " %s 2> /dev/null", JOBDATABASE);
#endif /* SYSV */
      if (stream = popen(temp,"r")) {
#ifndef SYSV
         if (fgets(temp,sizeof(temp),stream))
	    sscanf(temp, "%*s %*s %*d, %*d %*d:%*d %*s %d",&jobno);
#else /* SYSV */
	/*
	 * Make sure that we get the last job that was entered into the
	 * database with that name so that when fts wants to clear out
	 * the extra job it created, it gets the correct one.
	 */
         while (fgets(temp,sizeof(temp),stream))
	    jobno = sscanf(temp, "%*s %s", jobname);
#endif /* SYSV */
	 pclose(stream);
         }
   return(jobno);
} /* find_job */

void
cancel_job(jobno)
#ifndef SYSV
   int jobno;
#else
   char *jobno;
#endif /* SYSV */
{
   char temp[80];
#ifdef SYSV
   void mark_jobbase();
#endif /* SYSV */

   if (jobno) {
#ifndef SYSV
      sprintf(temp, "atrm %d 1> /dev/null 2>&1",jobno);
#else /* SYSV */
      sprintf(temp, "at -r %s > /dev/null 2>&1", jobno);
#endif /* SYSV */
      system(temp);
      }

#ifdef SYSV
   /*
    * Although finish_req will also try to remove this entry from
    * the job database, fts may call cancel_job on a job that
    * does not need to be finished, so get it here as well as
    * there.
    */
   mark_jobbase(jobno);
#endif /* SYSV */

}

boolean
print_req(out, req,src,dst,file, verbose)
   FILE *out;
   struct reqinfo *req;
   struct hostinfo *src, *dst;
   struct fileinfo *file;
   int verbose;
{
   char *cp;

   fprintf(out,"    Request type: %s\n",
		(file->reqtype==COPY)?"COPY":
		(file->reqtype==MOVE)?"MOVE":
		(file->reqtype==DFILE)?"DFILE":
		(file->reqtype==VERIFY)?"VERIFY":
		(file->reqtype==VERIFY_SRC)?"VERIFY_SRC":
		"unknown");
   if (verbose) {
      fprintf(out,"\n");
      fprintf(out,"    Source --\n");
      fprintf(out,"\tHost: '%s'\n", src->host);
      fprintf(out,"\tUser: '%s'\n", src->user);
      fprintf(out,"\tPass: %s\n", (strlen(src->passwd)==0) ? "NOT SET": "SET");
      fprintf(out,"\tAcct: '%s'\n", src->acct);
      fprintf(out,"\tDir: '%s'\n", src->dir);
      fprintf(out,"\tFile: '%s'\n", src->file);
      fprintf(out,"\tPort: %d\n", src->port);
    if ((file->reqtype != VERIFY_SRC) && (file->reqtype != DFILE)) {
	fprintf(out,"\n");
	fprintf(out,"    Destination --\n");
	fprintf(out,"\tHost: '%s'\n", dst->host);
	fprintf(out,"\tUser: '%s'\n", dst->user); 
	fprintf(out,"\tPass: %s\n",(strlen(dst->passwd)==0) ? "NOT SET": "SET");

	fprintf(out,"\tAcct: '%s'\n", dst->acct);
	fprintf(out,"\tDir: '%s'\n", dst->dir);
	if (! file->multflag)
	   fprintf(out,"\tFile:'%s'\n", dst->file);
	fprintf(out,"\tPort: %d\n", dst->port);
	};
    fprintf(out,"\n");
    if ((file->reqtype != DFILE) && (file->reqtype != VERIFY_SRC)) {
       if (file->creation != STOR)
          fprintf(out,"    Creation: %s\n",
    		(file->creation == APPE) ? "appending to existing file":
		"using STOU command");
       fprintf(out,"    %s%s", "Structure: ", expand_stru(file->stru));
       cp = file->filetype;
       fprintf(out,"%s%s%s%s", ", Mode: ", expand_mode(file->mode), 
	  		     ", Type: ", expand_type(*cp));
       switch (*cp) {
             case 'L':
	        cp = cp + 2;
		fprintf(out,"%s%s",", Byte size: ", cp);
		break;
	     case 'A':
	     case 'E':
	        cp = cp + 2;
		fprintf(out,"%s%s",", Format: ", expand_form(*cp));
		break;
	     }
       fprintf(out,"\n");
       };
      }
   else { /* not verbose */
      fprintf(out,"    Source: %s,%s,XXX,%s,%d,%s,%s\n",
   	   src->host, src->user, src->acct, src->port, src->dir, src->file);
      if ((file->reqtype != VERIFY_SRC) && (file->reqtype != DFILE)) {
	 fprintf(out,"    Destination: %s,%s,XXX,%s,%d,%s,%s\n",
	   dst->host, dst->user, dst->acct, dst->port, dst->dir, dst->file);
	 fprintf(out,"    Stru: %c",file->stru);
	 fprintf(out,", Mode: %c, Type: %s", file->mode, file->filetype);
	 fprintf(out,", Creation: %s\n",
			(file->creation==APPE)?"APPE":
			(file->creation==STOU)?"STOU":
			"STOR");
	 }
      }
    
   fprintf(out,"    Multiple matching: %s", 
   			(file->multflag)?"TRUE": "FALSE");
   if ((file->reqtype != VERIFY) && (file->reqtype != VERIFY_SRC)) {
      fprintf(out, ", Return mailbox: '%s'\n", req->mailbox);
     fprintf(out,"    Remaining tries: %d, Current retry interval: %d minutes",
      		req->ntries, req->interval);
      }
   fprintf(out,"\n\n");

} /* print_req */
	

void
format_time(t, tstr)
   time_t t;
   char *tstr;
{
#ifndef SYSV
   struct timeb tp;
#else /* SYSV */
   time_t tp;
#endif /* SYSV */
   struct tm *tmP;

#ifndef SYSV
   ftime(&tp);
   tmP = localtime((t)? &t: &tp.time);
	
   strcpy(tstr, asctime(tmP) + 4);
   /* Note: the previous localtime() call has magical properties that 
      make the upcoming timezone call work right! */
   sprintf(tstr+strlen(tstr)-1, " %s", timezone(tp.timezone, tmP->tm_isdst));
#else /* SYSV */
   time(&tp);
   tmP = localtime((t==0)? &tp : &t);

   strcpy(tstr, asctime(tmP) + 4);
   /* Note: the previous asctime() call has magical properties that 
      make the upcoming use of tzname work right! */
   sprintf(tstr+strlen(tstr)-1, " %s", (tmP->tm_isdst ? tzname[1] :tzname[0]));
#endif /* SYSV */
}

void
queue_req(req, timeval, out)
   struct reqinfo *req;
   long timeval;
   char *out;
{
   char temp[80],tmp1[30], tmp2[20], tmp3[20];
#ifdef SYSV
   FILE *stream;
   char pout[100], *ptr, jobname[JOBNAMELEN];
   int jobcount, activecount;
#endif /* SYSV */

#ifdef SUNOS4
   if (time(NULL) > (timeval-30))
      sprintf(temp, "at -s now + 1 minute %s 1> /dev/null 2>&1", req->cmdfile);
   else
#else
#ifdef SYSV
   if ((time(NULL) + 59) >= timeval)
	sprintf(temp, "at now + 1 minute <%s 2>&1", req->cmdfile);
   else
#endif /* SYSV */
#endif /* SUNOS4 */
      {
      strcpy(temp,ctime(&timeval));
      temp[strlen(temp)-1] = 0;
      sscanf(temp,"%*s%s%s%s",tmp1,tmp2,tmp3);
         /* Tue Apr 28 14:20:18 1987  is example of format */
      tmp3[strlen(tmp3)-3] = 0;
#ifndef SYSV
      sprintf(temp, "at -s %s %s %s %s 1> /dev/null 2>&1", 
      		tmp3, tmp1, tmp2, req->cmdfile);
#else /* SYSV */
      sprintf(temp, "at %s %s %s <%s 2>&1", 
      		tmp3, tmp1, tmp2, req->cmdfile);
#endif /* SYSV */
      }
#ifndef SYSV
   system(temp);
#else /* SYSV */
   if (stream = popen(temp,"r")) {
	while (fgets(pout, sizeof(pout), stream)) {
		/*
		 * find the job name that at writes when it accepts
		 * the job.  If none is found, assume an error occurred.
		 */
		pout[3] = '\0';
		if (strcmp("job", pout))
			continue;
		ptr = &pout[4];
		while (*ptr != ' ' && *ptr != '\t')
			ptr++;
		*ptr = '\0';
		strcpy(jobname, &pout[4]);
	}
	pclose(stream);
   }

   /*
    * Next step is to add the jobname and request file name to the job database
    * that we must use since System V at does not provide the facility
    * of keeping the jobname with the command file as do BSD systems.
    * We do not need to worry about what directory to find the proper
    * database file in because if bftp is running then bftp put us there,
    * or if fts is running, then at has put us in the proper directory.
    */
   if ((stream = fopen(JOBDATABASE, "r+")) != NULL) {
	fgets(temp, sizeof(temp), stream);
	sscanf(temp, "%09d %09d", &jobcount, &activecount);
	jobcount += 1;
	activecount += 1;
	rewind(stream);
   } else {
	if ((stream = fopen(JOBDATABASE, "w")) == NULL) {
	    printf("bftp: can't open job database file!\n");
	    goto nofile;
	}
	jobcount = activecount = 1;
   }

   fprintf(stream, "%09d %09d\n", jobcount, activecount);
   fseek(stream, 0, 2);
   fprintf(stream, "%s %s\n", req->cmdfile, jobname);
   fclose(stream);
nofile:
#endif /* SYSV */

   if (out != NULL) {
      get_id(temp, req->cmdfile);
      format_time(timeval, tmp1);
      sprintf(out, "Request %s submitted to run at %s.", temp, tmp1);
      }
}

void
finish_req(reqfile, req, subject, multlist)
   char *reqfile;
   struct reqinfo *req;
   char *subject, *multlist;
{
   char id[80], temp[200];
   
#ifdef SYSV
   /* update the job database */
   get_id(id, reqfile);
   mark_jobbase(id);
#endif /* SYSV */

   /* send the results message */ 
   if (strlen(req->mailbox)!=0 && strlen(req->mailfile)!=0) {
      if (!subject)
	 get_id(id, reqfile);
      sprintf(temp,"%s -s \"BFTP Results: %s\" %s < %s\n", 
      		   MAILPATH, (subject)?subject : id, 
		   req->mailbox,req->mailfile);
      system(temp); 
      }
	       
   /* delete the request files */      
   sprintf(temp,"rm %s %s %s %s 1> /dev/null 2>&1\n", 
   		req->mailfile, req->cmdfile, reqfile, multlist);

   system(temp);
}

boolean
read_req(filename,req,src,dst,file,multlist)
   char *filename;
   struct reqinfo *req;
   struct hostinfo *src, *dst;
   struct fileinfo *file;
   char *multlist;   
{
   FILE *rfile;
	
   if ((rfile = fopen(filename,"r")) == NULL)
      return(FALSE);
      
   bzero((char *)src, sizeof(struct hostinfo));
   bzero((char *)dst, sizeof(struct hostinfo));
   bzero((char *)req, sizeof(struct reqinfo));
   bzero((char *)file, sizeof(struct fileinfo));

   if ((5 == fscanf(rfile,"%[^\n] %[^\n] %[^\n] %[^\n] %d ",
 		req->rpasswd, req->mailbox, req->mailfile, 
		req->cmdfile, &file->reqtype)) &&
       (7 == fscanf(rfile,"%[^\n] %[^\n] %[^\n] %[^\n] %hd %[^\n] %[^\n] ",
   		src->host, src->user, src->passwd,
		src->acct, &src->port, src->dir, src->file)) &&
       (7 == fscanf(rfile,"%[^\n] %[^\n] %[^\n] %[^\n] %hd %[^\n] %[^\n] ",
   		dst->host, dst->user, dst->passwd,
		dst->acct, &dst->port, dst->dir, dst->file)) &&
       (7 == fscanf(rfile,"%[^\n] %c %c %d %d %d %d ",
   		file->filetype, &file->stru, &file->mode, &file->creation,
		&file->multflag, &req->ntries, &req->interval)))
      {
      if (file->multflag) {
         if (fscanf(rfile,"%[^\n]",multlist) != 1)
            multlist[0] = '\0';
	 }
      else
         multlist[0] = '\0';
      }
   fclose(rfile);
   return(TRUE);
}

void
put_string(fptr,sptr)
   FILE *fptr;
   char *sptr;
{
   if (strlen(sptr))
      fprintf(fptr,"%s\n", sptr);
   else
      fprintf(fptr,"%c\n", '\0');
}

boolean
write_req(filename,req,src,dst,file,multlist)
   char *filename;
   struct reqinfo *req;
   struct hostinfo *src, *dst;
   struct fileinfo *file;
   char *multlist;
{
   FILE *rfile;
   
   if (!(rfile = fdopen(open(filename,(O_WRONLY|O_CREAT),0600),"w")))
/*   if ((rfile = fopen(filename,"w")) == NULL) */
      return(FALSE);
      
   put_string(rfile,req->rpasswd);
   put_string(rfile,req->mailbox);
   put_string(rfile,req->mailfile);
   put_string(rfile,req->cmdfile);
   fprintf(rfile, "%d\n",file->reqtype);
   put_string(rfile, src->host);
   put_string(rfile, src->user);
   put_string(rfile, src->passwd);
   put_string(rfile, src->acct);
   fprintf(rfile, "%d\n",src->port);
   put_string(rfile, src->dir);
   put_string(rfile, src->file);
   put_string(rfile, dst->host);
   put_string(rfile, dst->user);
   put_string(rfile, dst->passwd);
   put_string(rfile, dst->acct);
   fprintf(rfile, "%d\n",dst->port);
   put_string(rfile, dst->dir);
   put_string(rfile, (file->multflag)? "" : dst->file);
   
   put_string(rfile, file->filetype);
   fprintf(rfile, "%c\n%c\n%d\n%d\n%d\n%d\n",
		file->stru,file->mode,file->creation,file->multflag, 
		req->ntries,req->interval);
   put_string(rfile, (multlist)? multlist: "");
   fclose(rfile);
	   
   return(TRUE);
}

void
unpack_filetype(ftype, type, form, bsize)
   char *ftype;
   int *type, *form, *bsize;
{
   char *cp = NULL;
   *type = (*ftype == 'A')? ASCII :
   	  (*ftype == 'E')? EBCDIC :
	  (*ftype == 'I')? IMAGE :
	  (*ftype == 'L')? LOCAL : NOTYPE;
	 
   switch (*type) {
      case ASCII:
      case EBCDIC:
         if (strlen(ftype)==3) {
	    cp = ftype + 2;
	    *form = (*cp == 'N')? NONPRINT :
	    	   (*cp == 'C') ? CCNTRL :
		   (*cp == 'T')? TELNET : NOFORM;
	    }
	 else
	    *form = NOFORM;
	 *bsize = DEFAULT_BYTESZ;
         break;
      case IMAGE:
      case NOTYPE:
         *form = NONPRINT;
	 *bsize = DEFAULT_BYTESZ;
         break;
      case LOCAL:
         if (strlen(ftype) > 2) {
	    cp = ftype + 2;
	    *bsize = atoi(cp);
	    }
	 else
	    *bsize = DEFAULT_BYTESZ;
	 *form = NONPRINT;
         break;
      }
} /* unpack_filetype */

#ifdef SYSV
/*
 * clean_jobbase
 * 
 * clean out all the inactive jobs from the job database.
 * return 0 on success
 * return 1 if failure before closing from
 * return -1 if failure after closing from
 */
clean_jobbase(from, activecount, lineno)
	register FILE *from;
	register int activecount, lineno;
{
	register FILE *to;
	char line[80];

	/*
	 * kill file if no more active jobs to keep track of.
	 */
	if (activecount == 0) {
		fclose(from);
		if (unlink(JOBDATABASE))
			return -1;
		if ((from = fopen(JOBDATABASE, "w")) != NULL) {
			fprintf(from, "%09d %09d\n", 0, 0);
			fclose(from);
		}
		return 0;
	}

	if ((to = fopen(JOBBASETEMP, "w")) == NULL)
		return 1;
	
	fprintf(to, "%09d %09d\n", activecount, activecount);
	lineno -= 2;
	while (fgets(line, sizeof(line), from)) {
		if (lineno != 0 && strncmp(line, "ABCxyz", 6) &&
							strlen(line) > 1) {
			if (fputs(line, to) == EOF) {
				fclose(to);
				unlink(JOBBASETEMP);
				return 1;
			}
			putc('\n', to);
		}
		if (lineno >= 0)
			lineno--;
	}

	fclose(from);
	fclose(to);
	if (rename(JOBBASETEMP, JOBDATABASE))
		return -1;

	return 0;

}  /* end of clean_jobbase */

/*
 * mark_jobbase
 *
 * Update the job database file so that the specified entry is marked
 * as no longer active.
 */
void
mark_jobbase(jobname)
char *jobname;
{
	char temp[80];
	FILE *fp;
	int lineno, jobcount, activecount;
	register int found = 0, status, clean = 0;

	/* let grep find the correct line */
	sprintf(temp, "grep -n %s %s 2> /dev/null", jobname, JOBDATABASE);
	if ((fp = popen(temp, "r")) == NULL)
		return;

	while (fgets(temp, sizeof(temp), fp)) {
		found = sscanf(temp, "%d", &lineno);
	}
	pclose(fp);

	if (found == 0)
		return;

	if ((fp = fopen(JOBDATABASE, "r+")) == NULL)
		return;

	/* get count of jobs on first line */
	if (fgets(temp, sizeof(temp), fp) == NULL) {
		fclose(fp);
		return;
	}
	sscanf(temp, "%09d %09d", &jobcount, &activecount);
	activecount--;

	/*
	 * cleanup database if it is getting full of inactive jobs.
	 */
	if (activecount == 0 ||
		((activecount < jobcount/2) && (jobcount > 2))) {
		clean = 1;
		status = clean_jobbase(fp, activecount, lineno);
	}

	/*
	 * If no clean was attempted or the clean failed, then mark the
	 * record.
	 */
	if (!clean || (clean && status)) {
		/*
		 * reposition file or reopen file if clean failed.
		 */
		if (clean && status == -1) {
			if ((fp = fopen(JOBDATABASE, "r+")) == NULL)
				return;
		} else {
			rewind(fp);
		}
		fprintf(fp, "%09d %09d\n", jobcount, activecount);
		fseek(fp, 0, 1);
		
		/* skip to line to be marked */
		for (lineno -= 2; lineno; lineno--) {
			if (fgets(temp, sizeof(temp), fp) == NULL) {
				fclose(fp);
				return;
			}
		}

		fseek(fp, 0, 1);
		fputs("ABCxyz", fp);
		fclose(fp);
	}

	return;

}  /* end of mark_jobbase */
#endif /* SYSV */
