/*
 * event.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.20 $
 * $Date: 1992/02/07 04:40:07 $
 */

#include "xk_debug.h"
#include "event.h"
#include "event_i.h"

#define E_RUNNING (1 << 0)
#define E_FINISHED (1 << 1)
#define E_DETACHED  (1 << 2)

#define THIS_IS_THE_HEADER ((EvFunc)-42)

/* BIG_N should be a function of the estimated number of scheduled events,
 * such that queues don't get too long, but with as little overhead as
 * possible. 128 is probably good for up to some 500-1000 events. [mats]
 */

#define BIG_N 128 

extern int	insque();
extern int	remque();

static struct Event 	evHead[BIG_N];
static int 		tickmod;

void
evInit()
{
    int i;

    for(i=0; i<BIG_N; i++)
      {
	  evHead[i].next = &evHead[i];
	  evHead[i].prev = &evHead[i];
	  evHead[i].func = THIS_IS_THE_HEADER;
      }
    tickmod = 0;
}


static void
e_insque(int q, Event e)
{
  Event a;

  for (a = evHead[q].next; a != &evHead[q]; a = a->next) {
    if (a->deltat < e->deltat) {
      /* E goes after a */
      e->deltat -= a->deltat;
      continue;
    } else {
      /* E goes just before a */
      a->deltat -= e->deltat;
      insque(e, a->prev);
      return;
    }
  }
  /* E goes at the end */
  insque(e, evHead[q].prev);
}


static void
e_remque(Event e)
{
  if (e->next->func != THIS_IS_THE_HEADER) {
    e->next->deltat += e->deltat;
  }
  remque(e);
}


static void
stub(Event e)
{
    e->func(e->arg);
    e->flags &= ~E_RUNNING;
    e->flags |= E_FINISHED;
    if (e->flags & E_DETACHED) {
	xFree((char *)e);
    }
}
      

Event
evSchedule( EvFunc func, void *arg, unsigned time)  /* time in usec */
{
  Event e;
  u_int delt;

  xTrace3(event, TR_GROSS_EVENTS, "evSchedule: f = %x, arg = %x, time = %d",
	  func, arg, time);
  e = (Event)xMalloc(sizeof(struct Event));
  e->func = func;
  e->arg = arg;
  /*
   * EVENT_INTERVAL, time between scans of the event list, is also in
   * usec.
   */
  delt = (time + EVENT_INTERVAL / 2) / EVENT_INTERVAL;
  if (delt == 0 && time != 0) {
      delt = 1;
  }
  xTrace1(event, TR_EVENTS, "event requires %d clock ticks", delt);
  e->deltat = delt / BIG_N + 1;
  e->flags = 0;
  if (delt == 0) {
    e->flags |= E_RUNNING;
    /* createprocess(stub, e); */
    CreateKernelProcess(stub, STD_PRIO, 1, e);
  } else {
    xTrace1(event, TR_FUNCTIONAL_TRACE, "Placing event in queue %d", (tickmod+delt)%BIG_N);
    e_insque((tickmod + delt) % BIG_N, e);
  }
  xTrace1(event, TR_MAJOR_EVENTS, "evSchedule returns event %x", e);
  return e;
}


void
evDetach(Event e)
{
  xTrace1(event, TR_GROSS_EVENTS, "evDetach: event = %x", e);
  if (e->flags & E_FINISHED) {
    xFree((char *)e);
  } else {
    e->flags |= E_DETACHED;
  }
}


/* cancel event e:
   returns -1 if it knows that the event has already executed to completion
   returns  0 if it can guarantee that either f is currently running or that
       it will run in the future
   returns  1 if it can guarantee that f has not run, and will not run in
       the future
 */
int
evCancel(Event e)
{
  int ans;

  xTrace1(event, TR_GROSS_EVENTS, "evCancel: event = %x", e);
  if (e->flags & E_RUNNING) {
    e->flags |= E_DETACHED;
    ans = 0;
  } else if (e->flags & E_FINISHED) {
    ans = -1;
    xFree((char *)e);
  } else {
    e_remque(e);
    ans = 1;
    xFree((char *)e);
  }
  return ans;
}
  

void
clock_ih()
{
    Event e;
    Event head;
    
    xTrace0(event, TR_GROSS_EVENTS, "event clock interrupt");
    head = &evHead[tickmod];
    e = head->next;
    if (e != head) {
	e->deltat--;
	xTrace2(event, TR_EVENTS, "lead event in queue %d has %d interrupts to go",
		tickmod, e->deltat);
	xIfTrace(event, TR_FUNCTIONAL_TRACE) {
	    int i;
	    for (i=0; e != head; e = e->next) {
		i++;
	    }
	    e = head->next;
	    xTrace2(event, TR_FUNCTIONAL_TRACE, "%d events in queue %d", i, tickmod);
	}
	for (; e != head && e->deltat == 0; e = e->next) {
	    /*
	     * Fire this event and remove it from the queue. No adjusting
	     * of the following deltat is required.
	     */
	    xTrace1(event, TR_GROSS_EVENTS, "Event %x expired", e->func);
	    remque(e);
	    e->flags |= E_RUNNING;
	    /* createprocess(stub, e); */
	    CreateProcess1((Pfi)stub, STD_PRIO, (int)e);
	}
    } else {
	xTrace1(event, TR_FUNCTIONAL_TRACE, "0 events in queue %d", tickmod);
    }
    tickmod = (tickmod + 1 ) % BIG_N;
    xTrace0(event, TR_GROSS_EVENTS, "end event clock interrupt");
}


