/*
 * vaddr.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.10 $
 * $Date: 1992/02/11 23:30:21 $
 */


#include "xkernel.h"
#include "vaddr_i.h"

    
/* 
 * Check for a valid participant list
 */
#define partCheck(pxx, name, max, retval)				\
	{								\
	  if ( ! (pxx) || partLen(pxx) < 1 || partLen(pxx) > (max)) { 	\
		xTrace1(vaddrp, TR_ERRORS,				\
			"VADDR %s -- bad participants",			\
			(name));  					\
		return (retval);					\
	  }								\
	}

/*
 * If SIMULATE_ARP_FAILURE is defined, vaddr will not attempt to make a
 * local connection.  This is primarily useful for testing.
 */
/* #define SIMULATE_ARP_FAILURE */

int tracevaddrp;

static	XObj	arp;


#ifdef __STDC__

static void	sessnInit( XObj );
static xkern_return_t	vaddrClose( XObj );
static XObj	vaddrCreateSessn( XObj, XObj, XObj, int );

#else

static void	sessnInit();
static xkern_return_t	vaddrClose();
static XObj	vaddrCreateSessn();

#endif


/* 
 * freePartState -- xFrees all pointers on the stacks of the participants
 */
static void
freePartState(p)
    Part *p;
{
    int		i;
    VOID	*ptr;

    for ( i=0; i < partLen(p); i++ ) {
	while ( ptr = partPop(p[i]) ) {
	    xFree((char *)ptr);
	}
    }
}


static XObj
vaddrOpen( self, hlpRcv, hlpType, p )
    XObj	self, hlpRcv, hlpType;
    Part 	*p;
{
    Pstate	*pstate = (Pstate *)self->state;
    IPhost 	*remIpHost;
    XObj	lls, s;
    int		llsIndex;
    HostBuf	hostBuf;
    ActiveKey	activeKey;
    
    xTrace0(vaddrp, TR_MAJOR_EVENTS, "VADDR open");
    partCheck(p, "vaddrOpen", 100, ERR_XOBJ);
    remIpHost = (IPhost *)partPop(*p);
    xTrace1(vaddrp, TR_MAJOR_EVENTS, "remote IP host is %s",
	    ipHostStr(remIpHost));
    bcopy((char *)remIpHost, (char *)&hostBuf, sizeof(IPhost));
    lls = ERR_XOBJ;
#ifdef SIMULATE_ARP_FAILURE
    xTrace0(vaddrp, TR_EVENTS, "vaddr simulating arp lookup failure");
#else
    if ( xControl(arp, RESOLVE, (char *)&hostBuf, sizeof(hostBuf)) > 0 ) {
	/* 
	 * Resolved the address, we should be able to open a
	 * local session  
	 */
	Part	part;
	ETHhost	remEthHost;

	bcopy((char *)&hostBuf, (char *)&remEthHost, sizeof(ETHhost));
	xTrace1(vaddrp, TR_EVENTS,
		"vaddrOpen - address is local, translation as %s",
		ethHostStr(&remEthHost));
	partInit(&part, 1);
	partPush(part, &remEthHost);
	lls = xOpen(self, hlpType, xGetDown(self, LOCAL), &part);
    } 
#endif
    if ( lls == ERR_XOBJ ) {
	xTrace0(vaddrp, TR_MAJOR_EVENTS,
		"vaddrOpen -- local open failed.  Trying remote");
	/*
	 * Try opening remote-network session
	 */
	partPush(*p, remIpHost);
	lls = xOpen(self, hlpType, xGetDown(self, REMOTE), p);
	if ( lls == ERR_XOBJ ) {
	  xTrace0(vaddrp, TR_MAJOR_EVENTS,
		  "vaddrOpen -- remote open failed.  Giving up");
	    return ERR_XOBJ;
	}
	llsIndex = REMOTE;
    } else {
	llsIndex = LOCAL;
    }
    activeKey = lls;
    if ( (s = (XObj)mapResolve(pstate->activeMap, &activeKey)) == ERR_XOBJ) {
	s = vaddrCreateSessn(self, hlpRcv, lls, llsIndex);
    }
    return s;
}


static XObj
vaddrCreateSessn( self, hlp, lls, index )
    XObj self, hlp, lls;
    int	index;
{
    XObj 	s;
    ActiveKey	key;
    Pstate	*pstate = (Pstate *)self->state;
    Sstate	*sstate;

    if ( ! xIsSession(lls) ) {
	xTrace0(vaddrp, TR_ERRORS, "lls is not valid in vaddrCreateSessn");
	return ERR_XOBJ;
    }
    if ( (s = xCreateSessn(sessnInit, hlp, self, 1, &lls)) == ERR_XOBJ ) {
	xTrace0(vaddrp, TR_ERRORS, "xCreateSessn failed in vaddrCreateSessn");
	xClose(lls);
	return ERR_XOBJ;
    }
    key = lls;
    if ( mapBind(pstate->activeMap, (char *)&key, s) == ERR_BIND ) {
	xTrace0(vaddrp, TR_ERRORS, "mapBind failed in vaddrCreateSessn");
	vaddrClose(s);
	return ERR_XOBJ;
    }
    sstate = X_NEW(Sstate);
    s->state = (VOID *)sstate;
    sstate->llsIndex = index;
    /* 
     * Cause the lls upper protocol pointer to point to this
     * *session*, not to this protocol.
     */
    lls->up = s;

    return s;
}


static xkern_return_t
vaddrClose( s )
    XObj s;
{
    XObj	lls;

    xAssert(xIsSession(s));
    lls = xGetDown(s, 0);
    /* 
     * Cause the lower session's up pointer, previously pointing to
     * this session, to point to the protocol
     */
    lls->up = s->myprotl;
    xClose(lls);
    xDestroy(s);
    return XK_SUCCESS;
}


static xkern_return_t
vaddrOpenDone( self, lls, hlpType )
    XObj self, lls, hlpType;
{
    XObj	s;
    PassiveKey	key;
    Pstate	*pstate = (Pstate *)self->state;
    int		llsIndex;
    Enable	*e;
    
    xTrace0(vaddrp, 3, "In VADDR OpenDone");
    key = hlpType;
    e = (Enable *) mapResolve(pstate->passiveMap, (char *)&key);
    if ( e == ERR_ENABLE ) {
	/* 
	 * This shouldn't happen
	 */
	xTrace0(vaddrp, TR_ERRORS,
		"vaddrDemux: Couldn't find hlp for incoming session");
	return XK_FAILURE;
    }
    /* 
     * Figure out which of my lower protocols owns the lower session
     */
    if ( xMyProtl(lls) == xGetDown(self, REMOTE) ) {
	llsIndex = REMOTE;
    } else if ( xMyProtl(lls) == xGetDown(self, LOCAL) ) {
	llsIndex = LOCAL;
    } else {
	xError("Impossible lower level session in VADDR openDone");
	return XK_FAILURE;
    }
    if ( (s = vaddrCreateSessn(self, e->hlpRcv, lls, llsIndex)) == ERR_XOBJ ) {
	return XK_FAILURE;
    }
    xTrace0(vaddrp, TR_EVENTS,
	    "vaddr Passively opened session successfully created");
    xDuplicate(lls);
    return xOpenDone(s, hlpType);
}


/* 
 * This shouldn't ever get called.  When a VADDR session is created,
 * the up pointer of the lls is set to point to a VADDR session. 
 */
static xkern_return_t
vaddrProtlDemux( self, lls, m )
    XObj	self, lls;
    Msg		*m;
{
    xTrace0(vaddrp, TR_ERRORS, "vaddrProtlDemux called");
    return XK_SUCCESS;
}


/* 
 * vaddrPop and vaddrSessnDemux must be used (i.e., they can't be
 * bypassed) for the UPI reference count mechanism to work properly. 
 */
static xkern_return_t
vaddrPop(self, lls, msg)
    XObj self;
    XObj lls;
    Msg *msg;
{
    xTrace0(vaddrp, TR_EVENTS, "vaddr Pop");
    return xDemux(self, msg);
}


static xkern_return_t
vaddrSessnDemux(self, lls, msg)
    XObj self;
    XObj lls;
    Msg *msg;
{
    xTrace0(vaddrp, TR_EVENTS, "vaddr Session Demux");
    return xPop(self, lls, msg);
}


static xmsg_handle_t
vaddrPush( self, m )
    XObj	self;
    Msg		*m;
{
    xTrace0(vaddrp, TR_EVENTS, "vaddr push");
    return xPush(xGetDown(self, 0), m);
}


static int
saveEnable( self, hlpRcv, hlpType )
    XObj self, hlpRcv, hlpType;
{
    Map		m = ((Pstate *)self->state)->passiveMap;
    Bind	b;
    Enable	*e;
    
    if ( (e = (Enable *)mapResolve(m, &hlpType)) == ERR_ENABLE ) {
	e = X_NEW(Enable);
	e->rcnt = 1;
	e->hlpRcv = hlpRcv;
	e->hlpType = hlpType;
	b = mapBind(m, &hlpType, (int)e);
	xAssert(b != ERR_BIND);
    } else {
	if ( e->hlpRcv != hlpRcv || e->hlpType != hlpType ) {
	    xTrace1(vaddrp, TR_SOFT_ERROR,
		    "vaddr saveEnable: hlpType %s is already bound",
		    hlpType->name);
	    return -1;
	}
	e->rcnt++;
    }
    return 0;
}


static int
removeEnable( self, hlpRcv, hlpType )
    XObj self, hlpRcv, hlpType;
{
    Map		m = ((Pstate *)self->state)->passiveMap;
    Enable	*e;
    
    if ( (e = (Enable *)mapResolve(m, &hlpType)) != ERR_ENABLE ) {
	if ( e->hlpRcv == hlpRcv && e->hlpType == hlpType ) {
	    if ( --e->rcnt <= 0 ) {
		return mapUnbind(m, &hlpType);
	    }
	    return 0;
	} else {
	    xTrace0(vaddrp, TR_SOFT_ERROR, "vaddr removeEnable: key mismatch");
	}
    }
    return -1;
}



static xkern_return_t
vaddrOpenEnable( self, hlpRcv, hlpType, p )
    XObj	self, hlpRcv, hlpType;
    Part	*p;
{
    Part	savedPart[2];

    xTrace0(vaddrp, TR_MAJOR_EVENTS, "VADDR open enable");
    partCheck(p, "openEnable", 2, XK_FAILURE);

    /* 
     * Save the participant to use in the second lower openenable
     */
    bcopy((char *)p, (char *)savedPart, partLen(p) * sizeof(Part));
    saveEnable(self, hlpRcv, hlpType);
    if ( xOpenEnable(self, hlpType, xGetDown(self, LOCAL), p)
							== XK_SUCCESS ) {
	/* 
	 * Restore the participant to its pristene state
	 */
	bcopy((char *)savedPart, (char *)p, partLen(p) * sizeof(Part));
	if ( xOpenEnable(self, hlpType, xGetDown(self, REMOTE), p)
	    						== XK_SUCCESS ) {
	    return XK_SUCCESS;
	} else {
	    xTrace0(vaddrp, 3, "Can't openenable remote protocol");
	    xOpenDisable(self, hlpType, xGetDown(self, LOCAL), savedPart);
	}
    } else {
	xTrace0(vaddrp, 3, "Can't openenable local protocol");
    }
    removeEnable(self, hlpRcv, hlpType);
    return XK_FAILURE;
}


static xkern_return_t
vaddrOpenDisable( self, hlpRcv, hlpType, p )
    XObj self, hlpRcv, hlpType;
    Part *p;
{
    Part		savedPart[2];
    xkern_return_t	rv1, rv2;

    xTrace0(vaddrp, TR_MAJOR_EVENTS, "VADDR open disable");

    partCheck(p, "openDisable", 2, XK_FAILURE);
    bcopy((char *)p, (char *)savedPart, partLen(p) * sizeof(Part));
    if ( removeEnable(self, hlpRcv, hlpType) ) {
	return XK_FAILURE;
    }
    rv1 = xOpenDisable(self, hlpType, xGetDown(self, LOCAL), p);
    rv2 = xOpenDisable(self, hlpType, xGetDown(self, REMOTE), savedPart);
    if ( rv1 == XK_FAILURE || rv2 == XK_FAILURE ) {
	return XK_FAILURE;
    }
    return XK_SUCCESS;
}


static int
vaddrControlSessn( self, opcode, buf, len )
    XObj self;
    int opcode, len;
    char *buf;
{
    XObj	lls = xGetDown(self, 0);

    xTrace0(vaddrp, TR_FUNCTIONAL_TRACE, "vaddrControlSessn");
    /* 
     * Since vaddr rewrites addresses, it's sessions have to undo the
     * process in this control op.  This is actually the reason why
     * vaddr sessions exist at all.
     */
    switch (opcode) {
      case GETPARTICIPANTS:
	{
	    int	i, res;
	    Part	*p;
	    Sstate	*sstate;
	    ETHhost	*localHost;
	    IPhost	*remHost;
	    HostBuf	hostBuf;
	    
	    if ( (res = xControl(lls, opcode, buf, len)) >= sizeof(Part) ) {
		sstate = (Sstate *)self->state;
		/* 
		 * Only local addresses have to be rewritten
		 */
		if ( sstate->llsIndex == LOCAL ) {
		    p = (Part *)buf;
		    /* 
		     * Reverse-resolve the hosts
		     */
		    xTrace0(vaddrp, TR_MORE_EVENTS,
			    "vaddr GETPARTICIPANTS RevResolving addrs");
		    for ( i=0; i < partLen(p); i++ ) {
			localHost = (ETHhost *)partPop(p[i]);
			bcopy((char *)localHost, (char *)&hostBuf,
			      sizeof(ETHhost));
			if ( xControl(arp, RRESOLVE, (char *)&hostBuf,
				      sizeof(hostBuf)) < 0 ) {
			    xTrace1(vaddrp, TR_ERRORS,
			     "VADDR control could not rev-resolve eth host %s",
				    ethHostStr(localHost));
			    freePartState(p);
			    return -1;
			}
			remHost = X_NEW(IPhost);
			bcopy((char *)&hostBuf, (char *)remHost,
			      sizeof(IPhost));
			xTrace2(vaddrp, TR_EVENTS,
				"VADDR control rev-resolves %s as %s",
				ethHostStr(localHost), ipHostStr(remHost));
			xFree((char *)localHost);
			partPush(p[i], remHost);
		    }
		} else {
		    xTrace0(vaddrp, TR_MORE_EVENTS,
			   "vaddrCntl GETPART -- lls is remote -- no rewrite");
		}
	    } else {
		xTrace0(vaddrp, TR_SOFT_ERRORS,
			"vaddr GETPARTICIPANTS call on lls failed");
	    }
	    return res;
        }
	
     default:
	/*
	 * All other opcodes are forwarded to the remote lower session
	 */
	return xControl(xGetDown(self, REMOTE), opcode, buf, len);
    }
}


static int
vaddrControlProtl( self, opcode, buf, len )
    XObj self;
    int opcode, len;
    char *buf;
{
    /*
     * All opcodes are forwarded to the remote-network protocol
     */
    return xControl(xGetDown(self, REMOTE), opcode, buf, len);
}


void
vaddr_init( self )
    XObj self;
{
    Pstate	*pstate;

    xTrace0(vaddrp, TR_MAJOR_EVENTS, "VADDR init");
    xAssert(xIsProtocol(self));
    if ( ! xIsProtocol(xGetDown(self, REMOTE)) ) {
	xError("VADDR can't find remote protocol -- not initializing");
	return;
    }
    if ( ! xIsProtocol(xGetDown(self, LOCAL)) ) {
	xError("VADDR can't find local protocol -- not initializing");
	return;
    }
    arp = xGetDown(self, TRANSLATE);
    if ( ! xIsProtocol(arp) ) {
	xError("VADDR can't find translation protocol -- not initializing");
	return;
    }
    self->control = vaddrControlProtl;
    self->open = vaddrOpen;
    self->openenable = vaddrOpenEnable;
    self->opendisable = vaddrOpenDisable;
    self->opendone = vaddrOpenDone;
    self->demux = vaddrProtlDemux;
    pstate = X_NEW(Pstate);
    self->state = (VOID *)pstate;
    pstate->activeMap = mapCreate(11, sizeof(ActiveKey));
    pstate->passiveMap = mapCreate(11, sizeof(PassiveKey));
}


static void
sessnInit( s )
    XObj s;
{
    s->close = vaddrClose;
    s->demux = vaddrSessnDemux;
    s->pop = vaddrPop;
    s->push = vaddrPush;
    s->control = vaddrControlSessn;
}
  

