/* 
 * port_mgr.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.12 $
 * $Date: 1992/02/05 19:47:29 $
 */

/* 
 * Management of ports
 *
 * This file is designed to be included by another source file which
 * defines these macros:
 *
 *	PORT -- the port type 
 *	MAX_PORT -- maximum allowed port
 *	FIRST_USER_PORT -- the first port which may be handed out through
 *		'getFreePort'
 *	NAME -- token to prepend to the routine names
 *	PROT_NAME -- string of protocol name (for debugging)
 *	TRACE_VAR -- trace variable to use in tracing
 * 
 * NOTE -- this code assumes a port is no larger than an int.
 * 
 */

#include "xkernel.h"

#define new(Type) (Type *)xMalloc(sizeof(Type))


int	traceportmgr;
#define DUMP    xIfTrace(portmgr, TR_DETAILED) { displayMap(); }



typedef enum {
    LOCK,
    NO_LOCK
} LockStatus;

typedef struct {
    LockStatus	lock;
    int		rcnt;
    long	port;
} PortDesc;

static	Map		portMap;
static	unsigned long	nextPort = FIRST_USER_PORT;
static	char		msgBuf[200];

void
#ifdef __STDC__
PASTE(NAME,PortMapInit)
#else
NAME/**/PortMapInit  
#endif
  ()
{
    if ( ! portMap ) {
	portMap = mapCreate(PORT_MAP_SIZE, sizeof(long));
    }
}


static int
displayElem( key, value, index )
    VOID *key;
    int value;
    VOID *index;
{
    PortDesc	*pd = (PortDesc *)value;

    xAssert(pd);
    sprintf(msgBuf, 
	    "Element %d:	  port = %d  rcnt = %d  lock = %s",
	   ++*(int *)index, pd->port, pd->rcnt,
	   pd->lock == LOCK ? "LOCK" : "NO LOCK");
    xError(msgBuf);
    return 1;
}


static void
displayMap()
{
    int	i = 0;
    sprintf(msgBuf, "dump of %s port map:", PROT_NAME);
    xError(msgBuf);
    mapForEach(portMap, displayElem, &i);
}



/* 
 * Binds 'port' into the map with the indicated reference count and
 * locking status.  Returns 0 on a successful bind, 1 if the port
 * could not be bound (indicating that it was already bound.)
 */
static int
portBind( port, rcnt, lock )
    long port;
    int rcnt;
    LockStatus lock;
{
    PortDesc	*pd;

    pd = new(PortDesc);
    pd->rcnt = rcnt;
    pd->lock = lock;
    pd->port = port;
    if ( mapBind(portMap, (char *)&port, (int)pd) == ERR_BIND ) {
	xFree((char *)pd);
	return 1;
    } 
    return 0;
}


static void
portUnbind( pd )
    PortDesc *pd;
{
    xAssert( pd && pd != (PortDesc *) -1 );
    mapUnbind(portMap, &pd->port);
    xFree((char *)pd);
}


int
#ifdef __STDC__
PASTE(NAME,GetFreePort)
#else
NAME/**/GetFreePort
#endif
  ( port )
    long *port;
{
    unsigned long	firstPort;

    xAssert(portMap);
    firstPort = nextPort;
    do {
	if ( portBind(nextPort, 1, NO_LOCK) == 0 ) {
	    /* 
	     * Found a free port
	     */
	    *port = nextPort;
	    DUMP;
	    return 0;
	} else {
	    if ( nextPort >= MAX_PORT ) {
		nextPort = FIRST_USER_PORT;
	    } else {
		nextPort++;
	    }
	}
    } while ( nextPort != firstPort );
    return 1;
}


int
#ifdef __STDC__
PASTE(NAME,LockPort)
#else
NAME/**/LockPort
#endif
  ( port )
    long port;
{
    int	res;

    xAssert(portMap);
    if ( port > MAX_PORT ) {
	res = 2;
    } else {
	res = portBind(port, 0, LOCK);
    }
    DUMP;
    return res;
}


int
#ifdef __STDC__
PASTE(NAME,DuplicatePort)
#else
NAME/**/DuplicatePort
#endif
  ( port, lockCmd )
    long port;
    LockCmd lockCmd;
{
    PortDesc	*pd;
    int		res;

    xAssert(portMap);
    if ( port > MAX_PORT ) {
	res = 2;
    } else {
	pd = (PortDesc *) mapResolve(portMap, (char *)&port);
	if ( pd == (PortDesc *) -1 ) {
	    /* 
	     * Port is not used, so we know portBind will succeed.
	     */
	    res = portBind(port, 1, NO_LOCK);
	} else {
	    if ( pd->lock == NO_LOCK || lockCmd == OVERRIDE_PORT_LOCK ) {
		pd->rcnt++;
		res = 0;
	    } else {
		res = 1;
	    }
	}
    }
    DUMP;
    return res;
}


void
#ifdef __STDC__
PASTE(NAME,ReleasePort)
#else
NAME/**/ReleasePort
#endif
  ( port )
    long port;
{
    PortDesc	*pd;

    xAssert(portMap);
    pd = (PortDesc *) mapResolve(portMap, (char *)&port);
    if ( pd != (PortDesc *) -1 ) {
	if ( pd->rcnt > 0 ) {
	    if ( --pd->rcnt == 0 && pd->lock == NO_LOCK ) {
		portUnbind(pd);
	    }
	}
    }
    DUMP;
}


void
#ifdef __STDC__
PASTE(NAME,UnlockPort)
#else
NAME/**/UnlockPort
#endif
  ( port )
    long port;
{
    PortDesc	*pd;

    xAssert(portMap);
    pd = (PortDesc *)mapResolve(portMap, (char *)&port);
    if ( pd != (PortDesc *) -1 ) {
	if ( pd->lock == LOCK ) {
	    if ( pd->rcnt == 0 ) {
		portUnbind(pd);
	    } else {
		pd->lock = NO_LOCK;
	    }
	}
    }
    DUMP;
}


