/* 
 * arp_table.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.9 $
 * $Date: 1992/02/05 18:22:34 $
 */


#include "xkernel.h"
#include "eth.h"
#include "ip.h"
#include "arp.h"
#include "arp_i.h"


/*
 * Number of msec between age ticks
 */
#define AGE_INTERVAL 60 * 1000		/* 1 minute   */

/*
 * Default number of ticks
 */
#define TTL_DEF 10			/* 10 minutes */

typedef struct arpent {
    ArpStatus	status;
    ArpWait	wait;
    ETHhost	arp_Ead;
    IPhost 	arp_Iad;
    bool	locked;
    int		ttl;
} ArpEnt;


static ETHhost	ethBcast = BCAST_ETH_AD;
static ArpEnt	arp_table[ARP_TAB];

#ifdef __STDC__

static void 	ageTimer( void * );
static int	getPos( void );			
static int	clearIndex( int );
static char *	dispEntry( int );
static void	dispTable( void );
static int	ethToIndex( ETHhost *h );
static int	ipToIndex( IPhost *h );
static void	signalAll( int );
static int	waitForAnswer( int );

#else

static void 	ageTimer();
static int	getPos();			
static int	clearIndex();
static char *	dispEntry();
static void	dispTable();
static int	ethToIndex();
static int	ipToIndex();
static void	signalAll();
static int	waitForAnswer();

#endif


/* 
 * See description in arp_table.h
 */
void
arpTableInit()
{
    ArpEnt	*e;
    int 	i;

    for (i=0; i < ARP_TAB; i++) {
	e = &arp_table[i];
	e->status = ARP_FREE;
    }
    ageTimer(0);
}


/* 
 * See description in arp_table.h
 */
int
arpLookup(ipHost, ethHost)
    IPhost *ipHost;
    ETHhost *ethHost; 
{
    int		pos, reply;
    Interface	*ifs;
    ArpEnt	*ent;
    
    xTrace1(arpp, 3, "arpLookup looking up %s", ipHostStr(ipHost));
    if ( (pos = ipToIndex(ipHost)) == -1 ) {
	ifs = arpIpToInterface( ipHost );
	if ( ifs == NULL ) {
	    xTrace1(arpp, 3, "arpLookup could not find interface for %s",
		    ipHostStr(ipHost));
	    return -1;
	}
	if ( ipIsBroadcast( ipHost ) ) {
	    xTrace0(arpp, 3, "returning eth broadcast address");
	    *ethHost = ethBcast;
	    return 0;
	}
	pos = getPos();
	ent = &arp_table[pos];
	newArpWait( &ent->wait, ipHost, ifs, &arp_table[pos].status );
	ent->status = ARP_ALLOC;
	ent->arp_Iad = *ipHost;
	arpSendRequest( &ent->wait );
	/*
	 * We'll wait for the reply in the ARP_ALLOC branch of the switch
	 */
    }
    switch ( arp_table[pos].status ) {

      case ARP_RSLVD:
	xTrace0(arpp, 4, "Had it");
	*ethHost = arp_table[pos].arp_Ead;
	xTrace1(arpp, TR_MORE_EVENTS, "eth host is %s", ethHostStr(ethHost));
	reply = 0;
	break;

      case ARP_ALLOC:
	/* someone else must have requested it */
	xTrace0(arpp, 4, "Request has been sent -- waiting for reply");
	reply = waitForAnswer(pos);
	*ethHost = arp_table[pos].arp_Ead;
	break;

      default:
	xTrace1(arpp, 3, "arp_lookup_ip returns bizarre code %d",
		arp_table[pos].status);
	xAssert(0);
	reply = 0;
    }
    return reply;
}


/* 
 * reverseLookup -- Find the IP host equivalent of the given ETH host.
 * If the value is not in the table, network RARP requests will be sent if
 * useNet is true.
 * Returns 0 if the lookup was successful and -1 if it was not.
 */
static int
reverseLookup(ipHost, ethHost, useNet)
    IPhost *ipHost;
    ETHhost *ethHost;
    bool useNet; 
{
    int		pos, reply;
    ArpEnt	*ent;
    
    xTrace1(arpp, 3, "arpRevLookup looking up %s", ethHostStr(ethHost));
    if ( (pos = ethToIndex(ethHost)) == -1 ) {
	/*
	 * Entry was not in the table
	 */
	if ( ! useNet ) {
	    return -1;
	}
	pos = getPos();
	ent = &arp_table[pos];
	newRarpWait( &ent->wait, ethHost, &arp_table[pos].status );
	ent->status = ARP_ALLOC;
	ent->arp_Ead = *ethHost;
	arpSendRequest( &ent->wait );
	/*
	 * We'll wait for the reply in the ARP_ALLOC branch of the switch
	 */
    }
    switch ( arp_table[pos].status ) {

      case ARP_RSLVD:
	xTrace0(arpp, 4, "Had it");
	*ipHost = arp_table[pos].arp_Iad;
	reply = 0;
	break;

      case ARP_ALLOC:
	/* someone else must have requested it */
	if ( useNet ) {
	    xTrace0(arpp, 4, "Request has been sent -- waiting for reply");
	    reply = waitForAnswer(pos);
	    *ipHost = arp_table[pos].arp_Iad;
	} else {
	    reply = -1;
	}
	break;

      default:
	xTrace1(arpp, 3, "arp_lookup_eth returns bizarre code %d",
		arp_table[pos].status);
	xAssert(0);
	reply = -1;
    }
    return reply;
}


/* 
 * See description in arp_table.h
 */
int
arpRevLookup(ipHost, ethHost)
    IPhost *ipHost;
    ETHhost *ethHost;
{
    return reverseLookup( ipHost, ethHost, TRUE );
}


/* 
 * See description in arp_table.h
 */
int
arpRevLookupTable(ipHost, ethHost)
    IPhost *ipHost;
    ETHhost *ethHost;
{
    return reverseLookup( ipHost, ethHost, FALSE );
}


static int
waitForAnswer(pos)
    int pos;  	/* block until answer arives */
{
    if ( arp_table[pos].status != ARP_RSLVD ) {
	arp_table[pos].wait.numBlocked++;
	semWait(&arp_table[pos].wait.s);
    }
    return arp_table[pos].status == ARP_RSLVD ? 0 : -1;
}


/* 
 * See description in arp_table.h
 */
void
arpLock(h)
    IPhost *h;
{
    int i;
    
    if ( (i = ipToIndex(h)) >= 0 ) {
	xAssert( arp_table[i].status != ARP_FREE );
	arp_table[i].locked = 1;
    }
}


/* 
 * See description in arp_table.h
 */
void
arpSaveBinding(ip, eth)
    IPhost *ip;
    ETHhost *eth;		
{
    int 	pos;
    ArpEnt	*t;
    
    if ( ip == NULL ) {
	clearIndex(ethToIndex(eth));
	return;
    }
    if ( eth == NULL ) {
	clearIndex(ipToIndex(ip));
	return;
    }
    xTrace2(arpp, 4, "arp saving %s -> %s", ethHostStr(eth), ipHostStr(ip));
    /*
     * Find a table position for this binding, clearing out old entries
     * with these host values.  Each ip and physical host can have at most
     * one table entry.
     */
    if ((pos = ipToIndex(ip)) >= 0) {
	/*
	 * This ip address is already bound to something
	 */
	if (bcmp((char *)&arp_table[pos].arp_Ead, (char *)eth,
		 sizeof(ETHhost)) == 0) {
	    xTrace0(arpp, 3, "arpSaveBinding: already had this binding");
	    return;
	}
	/*
	 * The ip address was bound to a different physical address.  This
	 * binding will be overridden.  First clear the binding of 'eth'
	 * to a different ip address if such a binding existed.
	 */
	if ( clearIndex( ethToIndex(eth) ) ) {
	    xTrace0(arpp, 3,
		    "arpSaveBinding: error clearing previous eth binding");
	    return;
	}
    } else if ( ( pos = ethToIndex(eth) ) < 0 ) {
	/*
	 * Neither 'eth' nor 'ip' were bound to anything.  
	 */
	pos = getPos();
    }
    t = &arp_table[pos];
    if ( t->locked ) {
	xTrace1(arpp, 3,
		"saveBinding can't override lock: \n%s", dispEntry(pos));
    } else {
	t->arp_Iad = *ip;
	t->arp_Ead = *eth;
	t->status = ARP_RSLVD;
	t->locked = 0;
	t->ttl = TTL_DEF;
	signalAll(pos);
    }
}


static void
ageTimer(arg)
    VOID	*arg;
{
    int 	i;
    ArpEnt	*t;

    xTrace0(arpp, 3, "ARP ageTimer runs");
    xIfTrace(arpp, 9) dispTable();
    for ( i=0; i < ARP_TAB; i++ ) {
	t = &arp_table[i];
	if ( t->locked ) continue;
	if ( t->status == ARP_RSLVD && t->ttl-- <= 0 ) {
	    xTrace1(arpp, 3, "ageTimer kicks out %s", dispEntry(i));
	    clearIndex(i);
	}
    }
    evDetach( evSchedule( ageTimer, arg, AGE_INTERVAL * 1000 ) );
    xTrace0(arpp, 3, "ARP ageTimer completes");
    xIfTrace(arpp, 9) dispTable();
}


/*
 * ipToIndex -- return the table index containing the entry for the
 * specified ip host if it exists, -1 if it does not.
 */
static int
ipToIndex(h)
    IPhost *h;
{
    int	i;

    for (i=0; i<ARP_TAB; i++) {
	if (arp_table[i].status != ARP_FREE &&
	    ! bcmp((char *)h, (char *)&arp_table[i].arp_Iad, sizeof(IPhost))) {
	    return(i);
	}
    }
    return(-1);
}
  

/*
 * ethToIndex -- return the table index containing the entry for the
 * specified eth host if it exists, -1 if it does not.
 */
static int
ethToIndex(h)
    ETHhost *h;			
{
    int	i;
    
    for (i=0; i<ARP_TAB; i++) {
	if (arp_table[i].status != ARP_FREE &&
	    ! bcmp((char *)h, (char *)&arp_table[i].arp_Ead,
		   sizeof(ETHhost))) {
	    return(i);
	}
    }
    return(-1);
}


/*
 * getPos -- return a new position for binding
 */
static int
getPos()			
{
    int		i;
    int		ouster;
    ArpEnt	*t;
    
    ouster = -1;
    for (i=0; i<ARP_TAB; i++) {
	t = &arp_table[i];
	switch( t->status ) {

	  case ARP_FREE:
	    t->status = ARP_ALLOC;
	    return i;
	    
	  case ARP_RSLVD:
	    if ( ouster == -1 ||
		( ! t->locked && arp_table[ouster].ttl > t->ttl ) ) {
		ouster = i;
	    }
	    break;
	    
	  case ARP_ALLOC:
	    break;
	    
	  default:
	    xTrace1(arpp, 0,
		    "getPos finds bizarre table status:\n%s",
		    dispEntry(i));
	    break;
	}
    }
    /*
     * Need to kick someone appropriate out
     */
    if ( ouster < 0 ) {
	dispTable();
	Kabort("arp couldn't free any table entries");
    }
    clearIndex(ouster);
    return ouster;
}


/*
 * signalAll -- release all threads waiting on this table entry and
 * free the state associated with this entry
 */
static void
signalAll(index)
    int index;
{
    int		i;
    ArpWait	*w;
    
    w = &arp_table[index].wait;
    for (i=0; i < w->numBlocked; i++) {
	semSignal(&w->s);
    }
    /* Prevent anyone else from blocking */
    w->numBlocked = -1;
    if ( w->event ) {
	evCancel(w->event);
	w->event = 0;
    }
   xIfTrace(arpp, 9) dispTable();
}


/*
 * Clear the indexed table entry.  Returns -1 only if the entry was locked.
 */
static int
clearIndex(index)
    int index; 
{
    ArpEnt	*t;

    if ( index < 0 || index >= ARP_TAB ) {
	return 0;
    }
    t = &arp_table[index];
    if ( t->status == ARP_RSLVD && t->locked ) {
	return -1;
    }
    signalAll(index);
    t->status = ARP_FREE;
    t->locked = 0;
    return 0;
}


static char *
dispEntry(i)
    int i;
{
    ArpEnt	*t;
    static char	buf[160];
    char	tmpBuf[80];
    
    t = &arp_table[i];
    sprintf(buf, "i=%d  ", i);
    switch ( t->status ) {
	
      case ARP_FREE:
	strcat(buf, "FREE    ");
	break;
	
      case ARP_ALLOC:
	{
	    strcat(buf, "ALLOC    ");
	    switch ( t->wait.type ) {
		
	      case ARP_ARP:
		sprintf(tmpBuf, "ARP     %s  %d waiters   ",
			ipHostStr(&t->wait.u.reqMsg.arp_tpa),
			t->wait.numBlocked);
		strcat(buf, tmpBuf);
		break;
		
	      case ARP_RARP:
		sprintf(tmpBuf, "RARP     %s  %d waiters   ",
			ethHostStr(&t->wait.u.remPhysHost),
			t->wait.numBlocked);
		strcat(buf, tmpBuf);
		break;
		
	      default:
		strcat(buf, "ALLOC -- type unknown");
		break;
	    }
	}
	break;
	
      case ARP_RSLVD:
	sprintf(tmpBuf, "RSLVD   %s <-> %s   ttl = %d   ",
		ipHostStr(&t->arp_Iad), ethHostStr(&t->arp_Ead),
		t->ttl);
	strcat(buf, tmpBuf);
	break;
	
      default:
	sprintf(tmpBuf, "Unknown ArpStatus %d   ", t->status);
	strcat(buf, tmpBuf);
    }
    if ( t->locked ) {
	strcat(buf, "LOCKED  ");
    }
    return buf;
}


static void
dispTable()
{
    int i;

    for( i=0; i < ARP_TAB; i++ ) {
	if ( arp_table[i].status == ARP_FREE ) continue;
	xTrace0(arpp, 0, dispEntry(i));
    }
}
