/* 
 * idmap.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.14 $
 * $Date: 1992/02/05 18:22:20 $
 */

#include "idmap.h"
#include "idmap_internal.h"
#include "upi.h"
#ifndef XKMACHKERNEL
#include "x_libc.h"
#endif XKMACHKERNEL

static struct {
  int (*resolve)();
  Bind(*bind)();
  int (*unbind)();
  int (*remove)();
} map_functions[MAX_MAP_KEY_SIZE];
  
static int  mgenericresolve();
static Bind mgenericbind();
static int  mgenericunbind();
static int  mgenericremove();


/*
 * Create and return a new map containing a table with nEntries entries 
 * mapping keys of size keySize to integers
 */

Map
mapCreate(nEntries, keySize)
    int nEntries;
    int keySize;
{
    register Map m;
    
    if (keySize > MAX_MAP_KEY_SIZE) {
	return (Map)-1;
    }
    m = (Map)xMalloc(sizeof(*m));
    m->tableSize = nEntries;
    m->keySize = keySize;
    m->cache = 0;
    m->freelist = 0;
    m->table = (MapElement **)xMalloc(nEntries * sizeof(MapElement *));
    bzero((char *)m->table, nEntries * sizeof(MapElement *));
    if (map_functions[keySize].resolve != 0) {
	m->resolve = map_functions[keySize].resolve;
	m->bind    = map_functions[keySize].bind   ;
	m->unbind  = map_functions[keySize].unbind ;
	m->remove  = map_functions[keySize].remove;
    } else {
	m->resolve = mgenericresolve;
	m->bind    = mgenericbind   ;
	m->unbind  = mgenericunbind ;
	m->remove  = mgenericremove;
    }    
    return m;
}


void
mapClose(m)
    Map m;
{
  free((char *)m->table);
  free((char *)m);
}


void
mapForEach(m, f, arg)
    Map m;
    MapForEachFun f;
    VOID *arg;
{
    char	key[MAX_MAP_KEY_SIZE];
    MapElement	*elem, *prevElem;
    int 	i;

    for ( i = 0; i < m->tableSize; i++ ) {
	prevElem = 0;
	do {
	    /* 
	     * Set elem to first item in the bucket
	     */
	    elem = m->table[i];
	    if ( prevElem != 0 ) {
		/*
		 * Set elem to the next element after the old elem in the
		 * same hash bucket.  If the previous element has been
		 * removed we will skip the rest of this bucket.  If
		 * the previous element has been removed and
		 * reinserted (with a possibly different key/value),
		 * we will skip it and everything before it in the
		 * list. 
		 */
		for ( ; elem != 0 ; elem = elem->next ) {
		    if ( prevElem == elem ) {
			elem = elem->next;
			break;
		    }
		}
	    }
	    if ( elem != 0 ) {
		bcopy(elem->externalid, key, m->keySize);
		if ( f(key, elem->internalid, arg) == 0 ) {
		    return;
		}
		prevElem = elem;
	    }
	} while ( elem != 0 );
    } 
}


static int
generichash(key, tableSize, keySize)
    char *key;
    int tableSize;
    int keySize;
{
#ifdef MACH  
    int howmanybytes = keySize, i;
    unsigned int hash = 0;
    for (i = 0; i < howmanybytes; i++) {
	hash = (hash << 2) ^ (* (unsigned char *)key);
	key++;
    }
#else
    
    int howmanylongs = keySize / 4, i;
    unsigned int hash = 0;
    
    for (i = 0; i < howmanylongs; i++) {
	hash = (hash << 2) ^ (* (unsigned long *)key);
	key += 4;
    }
    if (keySize % 4 == 2) {
	hash = (hash << 2) ^ (* (unsigned short *)key);
	key += 2;
    }
    if (keySize & 1) {
	hash = (hash << 2) ^ (* (unsigned char *)key);
    }
#endif MACH
    return hash % tableSize;
}

#define KEYSIZE generic
#define BINDNAME mgenericbind
#define REMOVENAME mgenericremove
#define RESOLVENAME mgenericresolve
#define UNBINDNAME mgenericunbind
#define HASH(K, T) generichash(K, T, table->keySize)
#define COMPBYTES(s1, s2) (!bcmp(s1, s2, table->keySize))
#define COPYBYTES(s1, s2) bcopy(s2, s1, table->keySize)

#include "idmap_templ.c"

#define KEYSIZE 2
#define BINDNAME m2bind
#define REMOVENAME m2remove
#define RESOLVENAME m2resolve
#define UNBINDNAME m2unbind
#define HASH(K, T)        (*(unsigned short *)(K) % (T))
#define COMPBYTES(s1, s2) (*(short *)(s1) == (*(short *)(s2)))
#define COPYBYTES(s1, s2) { *(short *)(s1) = (*(short *)(s2));}

#include "idmap_templ.c"

#define KEYSIZE 4
#define BINDNAME m4bind
#define REMOVENAME m4remove
#define RESOLVENAME m4resolve
#define UNBINDNAME m4unbind
#define HASH(K, T)        (*(unsigned  *)(K) % (T))
#define COMPBYTES(s1, s2) (*(int *)(s1) == (*(int *)(s2)))
#define COPYBYTES(s1, s2) { *(int *)(s1) = (*(int *)(s2));}

#include "idmap_templ.c"

#define KEYSIZE 6
#define BINDNAME m6bind
#define REMOVENAME m6remove
#define RESOLVENAME m6resolve
#define UNBINDNAME m6unbind
#define HASH(K, T) ((((*(unsigned *)(K)) ^ (*((unsigned short *)(K) + 2))) % *(unsigned *)(T)))
#define COMPBYTES(s1, s2) (*(int *)(s1) == (*(int *)(s2)) && \
	(*((short *)(s1)+2)) == (*((short *)(s2)+2)))
#define COPYBYTES(s1, s2) { *(int *)(s1) = (*(int *)(s2)); \
	*((short *)(s1)+2) = *((short *)(s2)+2); }

#include "idmap_templ.c"

#define KEYSIZE 8
#define BINDNAME m8bind
#define REMOVENAME m8remove
#define RESOLVENAME m8resolve
#define UNBINDNAME m8unbind
#define HASH(K, T) ((((*(unsigned *)(K)) ^ (*((unsigned *)(K) + 1))) % *(unsigned *)(T)))

#define COMPBYTES(s1, s2) (*(int *)(s1) == (*(int *)(s2)) && \
	(*((int *)(s1)+1)) == (*((int *)(s2)+1)))

#define COPYBYTES(s1, s2) { *(int *)(s1) = (*(int *)(s2)); \
	*((int *)(s1)+1) = *((int *)(s2)+1); }

#include "idmap_templ.c"

#define KEYSIZE 10
#define BINDNAME m10bind
#define REMOVENAME m10remove
#define RESOLVENAME m10resolve
#define UNBINDNAME m10unbind
#define HASH(K, T) ((((*(unsigned *)(K)) ^ (*((unsigned *)(K) + 1)) ^ (*((unsigned short *)(K) + 4))) % *(unsigned *)(T)))

#define COMPBYTES(s1, s2) (*(int *)(s1) == (*(int *)(s2)) && \
	(*((int *)(s1)+1)) == (*((int *)(s2)+1)) && \
	(*((short *)(s1)+4)) == (*((short *)(s2)+4)))

#define COPYBYTES(s1, s2) { *(int *)(s1) = (*(int *)(s2)); \
	*((int *)(s1)+1) = *((int *)(s2)+1); \
	*((short *)(s1)+4) = *((short *)(s2)+4); }

#include "idmap_templ.c"

#define KEYSIZE 12
#define BINDNAME m12bind
#define REMOVENAME m12remove
#define RESOLVENAME m12resolve
#define UNBINDNAME m12unbind
#define HASH(K, T) ((((*(unsigned *)(K)) ^ (*((unsigned *)(K) + 1)) ^ (*((unsigned *)(K) + 2))) % *(unsigned *)(T)))

#define COMPBYTES(s1, s2) (*(int *)(s1) == (*(int *)(s2)) && \
	(*((int *)(s1)+1)) == (*((int *)(s2)+1)) && \
	(*((int *)(s1)+2)) == (*((int *)(s2)+2)))

#define COPYBYTES(s1, s2) { *(int *)(s1) = (*(int *)(s2)); \
	*((int *)(s1)+1) = *((int *)(s2)+1); \
	*((int *)(s1)+2) = *((int *)(s2)+2); }

#include "idmap_templ.c"

#define KEYSIZE 16
#define BINDNAME m16bind
#define REMOVENAME m16remove
#define RESOLVENAME m16resolve
#define UNBINDNAME m16unbind
#define HASH(K, T) ((((*(unsigned *)(K)) ^ (*((unsigned *)(K) + 1)) ^ (*((unsigned *)(K) + 2)) ^ (*((unsigned *)(K) + 3))) % *(unsigned *)(T)))

#define COMPBYTES(s1, s2) (*(int *)(s1) == (*(int *)(s2)) && \
	(*((int *)(s1)+1)) == (*((int *)(s2)+1)) && \
	(*((int *)(s1)+2)) == (*((int *)(s2)+2)) && \
	(*((int *)(s1)+3)) == (*((int *)(s2)+3)))

#define COPYBYTES(s1, s2) { *(int *)(s1) = (*(int *)(s2)); \
	*((int *)(s1)+1) = *((int *)(s2)+1); \
	*((int *)(s1)+2) = *((int *)(s2)+2); \
	*((int *)(s1)+3) = *((int *)(s2)+3); }

#include "idmap_templ.c"

#define KEYSIZE 24
#define BINDNAME m24bind
#define REMOVENAME m24remove
#define RESOLVENAME m24resolve
#define UNBINDNAME m24unbind
#define HASH(K, T) ((((*(unsigned *)(K)) ^ (*((unsigned *)(K) + 1)) ^ (*((unsigned *)(K) + 2)) ^ (*((unsigned *)(K) + 3)) ^ (*((unsigned *)(K) + 4)) ^ (*((unsigned *)(K) + 5))) % *(unsigned *)(T)))

#define COMPBYTES(s1, s2) (*(int *)(s1) == (*(int *)(s2)) && \
	(*((int *)(s1)+1)) == (*((int *)(s2)+1)) && \
	(*((int *)(s1)+2)) == (*((int *)(s2)+2)) && \
	(*((int *)(s1)+3)) == (*((int *)(s2)+3)) && \
	(*((int *)(s1)+4)) == (*((int *)(s2)+4)) && \
	(*((int *)(s1)+5)) == (*((int *)(s2)+5)))

#define COPYBYTES(s1, s2) { *(int *)(s1) = (*(int *)(s2)); \
	*((int *)(s1)+1) = *((int *)(s2)+1); \
	*((int *)(s1)+2) = *((int *)(s2)+2); \
	*((int *)(s1)+3) = *((int *)(s2)+3); \
	*((int *)(s1)+4) = *((int *)(s2)+4); \
	*((int *)(s1)+5) = *((int *)(s2)+5); }

#include "idmap_templ.c"

#define KEYSIZE 28
#define BINDNAME m28bind
#define REMOVENAME m28remove
#define RESOLVENAME m28resolve
#define UNBINDNAME m28unbind
#define HASH(K, T) ((((*(unsigned *)(K)) ^ (*((unsigned *)(K) + 1)) \
        ^ (*((unsigned *)(K) + 2)) ^ (*((unsigned *)(K) + 3)) \
        ^ (*((unsigned *)(K) + 4)) ^ (*((unsigned *)(K) + 5)) \
        ^ (*((unsigned *)(K) + 6))) % *(unsigned *)(T)))

#define COMPBYTES(s1, s2) (*(int *)(s1) == (*(int *)(s2)) && \
	(*((int *)(s1)+1)) == (*((int *)(s2)+1)) && \
	(*((int *)(s1)+2)) == (*((int *)(s2)+2)) && \
	(*((int *)(s1)+3)) == (*((int *)(s2)+3)) && \
	(*((int *)(s1)+4)) == (*((int *)(s2)+4)) && \
	(*((int *)(s1)+5)) == (*((int *)(s2)+5)) && \
	(*((int *)(s1)+6)) == (*((int *)(s2)+6))) 

#define COPYBYTES(s1, s2) { *(int *)(s1) = (*(int *)(s2)); \
	*((int *)(s1)+1) = *((int *)(s2)+1); \
	*((int *)(s1)+2) = *((int *)(s2)+2); \
	*((int *)(s1)+3) = *((int *)(s2)+3); \
	*((int *)(s1)+4) = *((int *)(s2)+4); \
	*((int *)(s1)+5) = *((int *)(s2)+5); \
	*((int *)(s1)+6) = *((int *)(s2)+6); }

#include "idmap_templ.c"

#define KEYSIZE 32
#define BINDNAME m32bind
#define REMOVENAME m32remove
#define RESOLVENAME m32resolve
#define UNBINDNAME m32unbind
#define HASH(K, T) ((((*(unsigned *)(K)) ^ (*((unsigned *)(K) + 1)) \
        ^ (*((unsigned *)(K) + 2)) ^ (*((unsigned *)(K) + 3)) \
        ^ (*((unsigned *)(K) + 4)) ^ (*((unsigned *)(K) + 5)) \
        ^ (*((unsigned *)(K) + 6)) ^ (*((unsigned *)(K) + 7))) \
        % *(unsigned *)(T)))

#define COMPBYTES(s1, s2) (*(int *)(s1) == (*(int *)(s2)) && \
	(*((int *)(s1)+1)) == (*((int *)(s2)+1)) && \
	(*((int *)(s1)+2)) == (*((int *)(s2)+2)) && \
	(*((int *)(s1)+3)) == (*((int *)(s2)+3)) && \
	(*((int *)(s1)+4)) == (*((int *)(s2)+4)) && \
	(*((int *)(s1)+5)) == (*((int *)(s2)+5)) && \
	(*((int *)(s1)+6)) == (*((int *)(s2)+6)) && \
	(*((int *)(s1)+7)) == (*((int *)(s2)+7))) 

#define COPYBYTES(s1, s2) { *(int *)(s1) = (*(int *)(s2)); \
	*((int *)(s1)+1) = *((int *)(s2)+1); \
	*((int *)(s1)+2) = *((int *)(s2)+2); \
	*((int *)(s1)+3) = *((int *)(s2)+3); \
	*((int *)(s1)+4) = *((int *)(s2)+4); \
	*((int *)(s1)+5) = *((int *)(s2)+5); \
	*((int *)(s1)+6) = *((int *)(s2)+6); \
	*((int *)(s1)+7) = *((int *)(s2)+7); }

#include "idmap_templ.c"

void
map_init()
{
#define KEYSIZE 2
#define BINDNAME m2bind
#define REMOVENAME m2remove
#define RESOLVENAME m2resolve
#define UNBINDNAME m2unbind
#include "idmap_init.c"
#define KEYSIZE 4
#define BINDNAME m4bind
#define REMOVENAME m4remove
#define RESOLVENAME m4resolve
#define UNBINDNAME m4unbind
#include "idmap_init.c"
#define KEYSIZE 6
#define BINDNAME m6bind
#define REMOVENAME m6remove
#define RESOLVENAME m6resolve
#define UNBINDNAME m6unbind
#include "idmap_init.c"
#define KEYSIZE 8
#define BINDNAME m8bind
#define REMOVENAME m8remove
#define RESOLVENAME m8resolve
#define UNBINDNAME m8unbind
#include "idmap_init.c"
#define KEYSIZE 10
#define BINDNAME m10bind
#define REMOVENAME m10remove
#define RESOLVENAME m10resolve
#define UNBINDNAME m10unbind
#include "idmap_init.c"
#define KEYSIZE 12
#define BINDNAME m12bind
#define REMOVENAME m12remove
#define RESOLVENAME m12resolve
#define UNBINDNAME m12unbind
#include "idmap_init.c"
#define KEYSIZE 16
#define BINDNAME m16bind
#define REMOVENAME m16remove
#define RESOLVENAME m16resolve
#define UNBINDNAME m16unbind
#include "idmap_init.c"
#define KEYSIZE 24
#define BINDNAME m24bind
#define REMOVENAME m24remove
#define RESOLVENAME m24resolve
#define UNBINDNAME m24unbind
#include "idmap_init.c"
#define KEYSIZE 28
#define BINDNAME m28bind
#define REMOVENAME m28remove
#define RESOLVENAME m28resolve
#define UNBINDNAME m28unbind
#include "idmap_init.c"
#define KEYSIZE 32
#define BINDNAME m32bind
#define REMOVENAME m32remove
#define RESOLVENAME m32resolve
#define UNBINDNAME m32unbind
#include "idmap_init.c"
}
