(* Gene Rollins
   School of Computer Science
   Carnegie-Mellon University
   Pittsburgh, PA 15213
   rollins@cs.cmu.edu *)

functor PathnameFun () :PATHNAME = struct

fun findChr (ch :string) ((i,s) :int * string) :int =
  let val len = String.length s
      fun find j =
        if j=len
	  then 0
	  else if ch = chr(ordof(s,j))
	         then j+1
		 else find (j+1)
  in if (size ch) = 0 then 0 else find i end

fun findChrFromRight (ch :string) ((i,s) :int * string) :int =
  let val len = String.length s
      fun find j =
        if j = ~1
	  then len
	  else if ch = chr(ordof(s,j))
	         then j
		 else find (j-1)
  in if (size ch) = 0 then len else find (i-1) end

fun extension (name :string) =
  let val len = size name
      val dot = findChrFromRight "." (len, name)
  in if dot = len then ""
       else (substring (name, dot+1, len-dot-1))
  end

fun stripExtension (name :string) =
  let val len = size name
      val dot = findChrFromRight "." (len, name)
  in
    substring (name, 0, dot)
  end

fun splitFileExtension (name :string) =
let val len = size name
      val dot = findChrFromRight "." (len, name)
  in if dot = len then (name, "")
       else (substring (name,0, dot), substring (name, dot+1, len-dot-1))
  end

val slash = ord "/"

fun mergeDirFile dirname filename =
  if dirname = ""
    then filename
    else if (ord filename = slash) orelse
            (ordof (dirname,(size dirname)-1) = slash)
           then (dirname ^ filename)
           else (dirname ^ "/" ^ filename)

fun directoryPart name =
  let val name'len = size name
      val pos = findChrFromRight "/" (name'len, name)
      val dirname = substring (name, 0, pos)
  in
    if pos = name'len then "" else dirname
  end

fun stripDirectory name =
  let val name'len = size name
      val pos = findChrFromRight "/" (name'len, name)
      val dirname = substring (name, 0, pos)
  in
    if pos = name'len
      then dirname
      else substring (name, pos+1, name'len-(pos+1))
  end

fun splitDirFile name =
  let val name'len = size name
      val pos = findChrFromRight "/" (name'len, name)
      val dirname = substring (name, 0, pos)
  in
    if pos = name'len
      then ("", dirname)
      else (dirname, substring (name, pos+1, name'len-(pos+1)))
  end

fun splitDirFile' name =
  let val (first, second) = splitDirFile name in
    if first = "" then (second, first) else (first, second) end

fun explodePath (path:string) :string list =
  let val slash = findChr "/" (0,path)
      val len = size path
  in
    if slash = 0
      then [path]
      else ((substring (path, 0, slash-1)) ::
            (explodePath (substring (path, slash, len - slash))))
  end;

fun implodePath (pathlist :string list) :string =
  let fun merge (x,y) = if y = "" then x else (x ^ "/" ^ y) in
    fold merge pathlist ""
  end

fun clearPath' (path'list :string list) :string list =
  let fun processDots (prefix:string list) (suffix:string list) =
        case suffix of
           [] => rev prefix
         | (s0::s'tail) =>
             case prefix of
                [] => processDots [s0] s'tail
              | (p0::p'tail) =>
                  if s0 = ".."
                    then if p0 = "" (* rmDot guarantees p'tail=[] *)
                           then p0::s0::(processDots [] s'tail)
                           else processDots [] ((rev p'tail) @ s'tail)
                    else processDots (s0::prefix) s'tail
      fun rmDot (pathlist:string list) =
        let fun rm'dot ([]:string list) = []
              | rm'dot (name::tail) =
                  if (name = ".") orelse (name = " ")
                    then rm'dot tail else name::(rm'dot tail)
        in
          case pathlist of
             [] => []
           | (first::rest) =>
               if first = " " then first::(rm'dot rest)
                 else if (first = ".") andalso (rest = []) then pathlist
                   else rm'dot pathlist
        end
  in
    processDots [] (rmDot path'list)
  end;

fun clearPath (path :string) :string =
  implodePath (clearPath' (explodePath path))

end
