/*
** Haskell I/O.
*/
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <setjmp.h>

#if 0
typedef long    fd_mask;
#define NFDBITS (sizeof (fd_mask) * 8)
typedef struct fd_set {
        fd_mask fds_bits[1];
} fd_set;

#define FD_SET(n, p)    ((p)->fds_bits[(n)/NFDBITS] |= (1 << ((n) % NFDBITS)))
#define FD_CLR(n, p)    ((p)->fds_bits[(n)/NFDBITS] &= ~(1 << ((n) % NFDBITS)))
#define FD_ISSET(n, p)  ((p)->fds_bits[(n)/NFDBITS] & (1 << ((n) % NFDBITS)))
#define FD_ZERO(p)      bzero((char *)(p), sizeof (*(p)))
#endif

#define ReadFile 0
#define WriteFile 1
#define AppendFile 2
#define ReadBinFile 3
#define WriteBinFile 4
#define AppendBinFile 5
#define DeleteFile 6
#define StatusFile 7
#define ReadChan 8
#define AppendChan 9
#define ReadBinChan 10
#define AppendBinChan 11
#define StatusChan 12
#define Echo 13
#define GetArgs 14
#define GetEnv 15
#define SetEnv 16
#define ReadChannels 17
#define ReadBinChannels 18
#define CreateProcess 19
#define CreateDirectory 20
#define OpenFile 21
#define OpenBinFile 22
#define CloseFile 23
#define ReadVal 24
#define ReadBinVal 25
#define WriteVal 26
#define WriteBinVal 27
/* extra */
#define Sleep 28
#define ChangeDirectory 29

#define Success 0
#define Str 1
#define Bn 2
#define Failure 3
#define Tag 4
#define BinTag 5
#define Fil 6

#define WriteError 0
#define ReadError 1
#define SearchError 2
#define FormatError 3
#define OtherError 4

#define Stdin "stdin"
#define Stdout "stdout"
#define Stderr "stderr"
#define Stdecho "stdecho"

#define MAXNAME 5120

extern char *malloc(), *getenv();

extern int ***ep;
extern int **mknode(), **mknode1(), **evaluate(), **mkstring();
extern int *PAIR1[], *TAG[], *STRING[], *INPUT[], *AP[], *TAG0[], *PAIR0[], *PAIR2[], *PAIR3[], *PAIR4[], *INT[], *DFLOAT[], *CHAR[], *VEK[];
extern int **concargs;
#define NIL ((int**)0)

static void doreq();
extern int ***nextresp, **respfail[];
extern int *readchan[];

static jmp_buf topjmp;

/*
** This is the top loop where everything happens.
** Evaluate next request, execute it, and stick the response on the response list.
** On entry p = AP main 0
*/
void
toploop(p)
int **p;
{
    int **q;

    if (q = (int **)setjmp(topjmp)) {
	/* from child process */
	p[0] = (int *)AP;
	p[1] = (int *)q;
    } else {
	nextresp = respfail;
    }
    p[2] = (int *)nextresp;
    p = evaluate(p);		/* evaluate request list */
    *--ep = p;			/* save p */
    for(;;) {
	if (p[0] != (int *)PAIR1)
	    break;		/* end of list */
	q = evaluate((int **)p[1]); /* evaluate request */
	doreq(q);		/* execute the request */
	p = *ep;		/* restore p */
	q = evaluate((int **)p[2]); /* evaluate rest of request list */
	p = *ep;		/* restore p */
	p[0] = q[0]; p[1] = q[1]; p[2] = q[2]; /* copy this node on top of old cons to avoid space leak. */
    }
    p = *ep++;			/* pop p */
    /* Terminate responses */
    nextresp[0] = TAG0;
    nextresp[1] = 0;
}

int 
gettag(p)
int ***p;
{
    if (*p == PAIR0)
	return 0;
    else if (*p == PAIR1)
	return 1;
    else if (*p == PAIR2)
	return 2;
    else if (*p == PAIR3)
	return 3;
    else if (*p == PAIR4)
	return 4;
    else
	return (int)p[1];
}

static void
mkresponse(n)
int **n;
{
    int **f;
    int ***nr = nextresp;

    nextresp = (int ***)mknode(nr[0], nr[1], nr[2]); /* Copy failure node */
    nr[0] = PAIR1;		/* A cons */
    nr[1] = n;			/* The response */
    nr[2] = (int **)nextresp;	/* New failure */
}

static void
mkerrresp(k, msg)
int k;
char *msg;
{
    mkresponse(mknode(TAG, Failure, mknode(TAG, k, mknode(STRING, (int **)msg, NIL))));
}

extern int errno;
static void
fileerror(r)
int r;
{
    extern char *sys_errlist[];

    mkerrresp(r, sys_errlist[errno]);
}

/* Output a list of char, assumes type correctness */
/* *************** WRONG!! IF SHARING!!! Avoid space leaks by updateing the original node */
static int
outstring(f, p)
FILE *f;
int **p;
{
    int **q, **cp;

    p = evaluate(p);
    if (p[0] != (int *)PAIR1)
	return 0;		/* end of list */
    cp = (int **)p[1];
    *--ep = p;		/* save p */
    for(;;) {
	q = evaluate(cp); /* evaluate char */
	if (putc((int)q[1], f) == EOF) {		/* output char */
	    ep++;		/* pop p */
	    return -1;
	}
	p = *ep;		/* restore p */
	q = evaluate((int **)p[2]); /* evaluate rest */
	p = *ep;		/* restore p */
	if (q[0] != (int *)PAIR1)
	    break;
	cp = (int **)q[1];		/* pointer to next char */
	*ep = q;
/***** WRONG!! if sharing	p[2] = q[2]; /* copy this node on top of old cons to avoid space leak. */	
    }
    ep++;
    return 0;
}

static struct chnames {
    struct chnames *next;
    char *chname;
} *chused = 0;

static int
channelused(s)
char *s;
{
    struct chnames *p;

    for(p = chused; p; p = p->next)
	if (strcmp(s, p->chname) == 0)
	    return 1;
    p = (struct chnames *)malloc(sizeof(struct chnames));
    p->next = chused;
    chused = p;
    p->chname = malloc(strlen(s)+1);
    strcpy(p->chname, s);
    return 0;
}

static void
mksuccess()
{
    mkresponse(mknode(TAG0, Success, NIL));
}

static void
mkstrresp(s)
{
    mkresponse(mknode(TAG0, Str, s));
}

/*
** Binary value are internally coded as lists.
** This makes showBin(::*a->Bin->Bin) and readBin(::Bin->(*a,Bin) easy (just cons or split).
** Externally this list is flattened, and the tags are kept to make reading possible.
*/
#define T_INT 0x1000
#define T_DFLOAT 0x1001
#define T_CHAR 0x1002
#define T_TAG 0x1003
#define T_TAG0 0x1004
#define T_PAIR0 0x1005
#define T_PAIR1 0x1006
#define T_PAIR2 0x1007
#define T_PAIR3 0x1008
#define T_PAIR4 0x1009
#define T_VEK 0x1010

#define T(x) (b[0] == (int *)x)
static int
writebin1(f, b)
FILE *f;
int **b;
{
    int *p, n, i;

 top:
    b = evaluate(b);
    /* Don't bother with error check */
    if (T(INT)) {
	putw(T_INT, f);
	putw(b[1], f);
    } else if (T(DFLOAT)) {
	putw(T_DFLOAT, f);
	putw(b[1], f);
	putw(b[2], f);
    } else if (T(CHAR)) {
	putw(T_CHAR, f);
	putw(b[1], f);
    } else if (T(TAG)) {
	putw(T_TAG, f);
	putw(b[1], f);
	b = (int **)b[2];
	goto top;		/* avoid tail rec */
    } else if (T(TAG0)) {
	putw(T_TAG0, f);
	putw(b[1], f);
    } else if (T(PAIR0)) {
	putw(T_PAIR0, f);
	writebin1(f, b[1]);
	b = (int **)b[2];
	goto top;		/* avoid tail rec */
    } else if (T(PAIR1)) {
	putw(T_PAIR1, f);
	*--ep = b;
	writebin1(f, b[1]);
	b = *ep++;
	b = (int **)b[2];
	goto top;		/* avoid tail rec */
    } else if (T(PAIR2)) {
	putw(T_PAIR2, f);
	*--ep = b;
	writebin1(f, b[1]);
	b = *ep++;
	b = (int **)b[2];
	goto top;		/* avoid tail rec */
    } else if (T(PAIR3)) {
	putw(T_PAIR3, f);
	*--ep = b;
	writebin1(f, b[1]);
	b = *ep++;
	b = (int **)b[2];
	goto top;		/* avoid tail rec */
    } else if (T(PAIR4)) {
	putw(T_PAIR4, f);
	*--ep = b;
	writebin1(f, b[1]);
	b = *ep++;
	b = (int **)b[2];
	goto top;		/* avoid tail rec */
    } else if (T(VEK)) {
	putw(T_VEK, f);
	n = (int)b[1];
	putw(n, f);
	*--ep = b;
	for(i = 0; i < n; i++) {
	    writebin1(b[i+2], f);
	    b = *ep;
	}
	ep++;
    } else {
	fprintf(stderr, "Bad tag in writebin\n");
	exit(1);
    }
    return 0;
}
#undef T

static int *snil[2];

static void
readbin1(f, rp)
FILE *f;
int ***rp;
{
    int n, i, **p;

 top:
    switch(getw(f)) {
    case EOF:
	*rp = 0; /* EOF */
	break;
    case T_INT:
	*rp = mknode1(INT, getw(f));
	break;
    case T_DFLOAT:
	i = getw(f);
	*rp = mknode(DFLOAT, i, getw(f));
	break;
    case T_CHAR:
	*rp = mknode1(CHAR, getw(f));
	break;
    case T_TAG:
	*rp = mknode(TAG, getw(f), snil);
	rp = (int ***)&(*rp)[2];
	goto top;
    case T_TAG0:
	*rp = mknode1(TAG0, getw(f));
	break;
    case T_PAIR0:
	readbin1(f, &p);
	*rp = mknode(PAIR0, p, snil);
	rp = (int ***)&(*rp)[2];
	goto top;
    case T_PAIR1:
	readbin1(f, &p);
	*rp = mknode(PAIR1, p, snil);
	rp = (int ***)&(*rp)[2];
	goto top;
    case T_PAIR2:
	readbin1(f, &p);
	*rp = mknode(PAIR2, p, snil);
	rp = (int ***)&(*rp)[2];
	goto top;
    case T_PAIR3:
	readbin1(f, &p);
	*rp = mknode(PAIR3, p, snil);
	rp = (int ***)&(*rp)[2];
	goto top;
    case T_PAIR4:
	readbin1(f, &p);
	*rp = mknode(PAIR4, p, snil);
	rp = (int ***)&(*rp)[2];
	goto top;
    case T_VEK:
	n = getw(f);		/* number of elems */
	/* should check for GC */
	{
	    extern int **hp, **ehp;
	    int **ohp;

	    if (hp+n > ehp)
		gcstack(0);
	    ohp = hp;
	    *rp = ohp;
	    hp += n+2;
	    hp[0] = (int *)VEK;
	    hp[1] = (int *)n;
	    for(i = 0; i < n; i++)
		hp[i+2] = (int *)snil;
	    *--ep = ohp;
	    for(i = 0; i < n; i++) {
		readbin1(f, &ohp[i+2]);
		ohp = *ep;
	    }
	    ep++;
	}
	break;
    default:
	fprintf(stderr, "Bad tag in readbin\n");
	exit(1);
    }
}
	
static int **
readbin(f)
{
    int **b, **p, **r;

    snil[0] = (int *)TAG0; snil[1] = 0;
    /* GC will corrupt this!!! */
    b = mknode(TAG0, 0, NIL);
    *--ep = b;
    *--ep = 0;			/* store p here */
    for(p = b;;) {
	*ep = p;
	readbin1(f, &r);
	p = *ep;
	if (r == 0) {
	    ep++;
	    b = *ep++;
	    return b;
	}
	p[0] = (int *)PAIR1;
	p[1] = (int *)r;
	p[2] = (int *)mknode(TAG0, 0, NIL);
	p = (int **)&p[2];
    }
}

static int
writebin(f, p)
FILE *f;
int **p;
{
    int **cp, **q;			/* point to next binary to write */

    p = evaluate(p);
    if (p[0] != (int *)PAIR1)
	return 0;		/* end of list */
    cp = (int **)p[1];
    *--ep = p;		/* save p */
    for(;;) {
	if (writebin1(f, cp) < 0) {
	    ep++;
	    return -1;
	}
	p = *ep;		/* restore p */
	q = evaluate((int **)p[2]); /* evaluate rest */
	p = *ep;		/* restore p */
	if (q[0] != (int *)PAIR1)
	    break;
	cp = (int **)q[1];		/* pointer to next char */
	p[2] = q[2]; /* copy this node on top of old cons to avoid space leak. */	
    }
    ep++;
    return 0;
}    

static void
doreq(p)
int **p;
{
    char buf[MAXNAME];
    int **namep, **stringp;
    FILE *f;
    int r, t;
    char *mode, *s;
    struct stat sb;
    int **pp;

    switch(t = gettag(p)) {
    case ReadFile:
	/* Uses TAG */
	namep = (int **)p[2];
	evalstring(namep, buf, sizeof buf);
  inopen:
	f = fopen(buf, "r");
	if (f == NULL)
	    fileerror(errno == ENOENT ? SearchError : ReadError);
	else {
  doin:
	    mkresponse(mknode(TAG, (int **)Str, mknode(INPUT, (int **)f, NIL)));
	}
	break;
    case WriteFile:
	/* Uses PAIR1 */
	mode = "w";
	goto wrapp;
    case AppendFile:
	/* Uses PAIR2 */
	mode = "a";
  wrapp:
	namep = (int **)p[1];
	stringp = (int **)p[2];
	*--ep = stringp;
	evalstring(namep, buf, sizeof buf);
	stringp = *ep++;
  outopen:
	f = fopen(buf, mode);
	if (f == NULL)
	    fileerror(WriteError);
	else {
  doout:
	    r = outstring(f, stringp);
	    if (r < 0)
		fileerror(WriteError);
	    else
		mksuccess();
	    if (f != stdout && f != stderr)
		fclose(f);
	    else
		fflush(f);
	}
	break;
    case ReadChan:
	/* Uses TAG */
	namep = (int **)p[2];
	evalstring(namep, buf, sizeof buf);
	if (channelused(buf)) {
	    mkerrresp(ReadError, "Channel already used");
	} else {
	    if (strcmp(buf, Stdin) == 0) {
		f = stdin;
		goto doin;
	    } else
		goto inopen;
	}
	break;
    case AppendChan:
	/* Uses TAG PAIR0 */
	pp = (int **)p[2];
	namep = (int **)pp[1];
	stringp = (int **)pp[2];
	*--ep = stringp;
	evalstring(namep, buf, sizeof buf);
	stringp = *ep++;
	if (strcmp(buf, Stdout) == 0) {
	    f = stdout;
	    goto doout;
	} else if (strcmp(buf, Stderr) == 0) {
	    f = stderr;
	    goto doout;
	} else if (strcmp(buf, Stdecho) == 0) {
	    f = stderr;		/* The best we can do under UNIX */
	    goto doout;
	} else
	    goto outopen;
	break;
    case StatusChan:
	/* Uses TAG */
	namep = (int **)p[2];
	evalstring(namep, buf, sizeof buf);
	if (strcmp(buf, Stdout) == 0 || strcmp(buf, Stderr) == 0 || strcmp(buf, Stdecho) == 0) {
	    mkstrresp(mknode(STRING, "0 0", NIL));
	} else {
	    mkerrresp(SearchError, "No status available");
	}
	break;
    case DeleteFile:
	/* Uses TAG */
	namep = (int **)p[2];
	evalstring(namep, buf, sizeof buf);
	if (unlink(buf) < 0) {
	    fileerror(errno == ENOENT ? SearchError : WriteError);
	} else {
	    mksuccess();
	}
	break;
    case StatusFile:
	/* Uses TAG */
	namep = (int **)p[2];
	evalstring(namep, buf, sizeof buf);
	if (stat(buf, &sb) < 0) {
	    fileerror(SearchError);
	} else {
	    static char s[4] = "???";
	    s[0] = (sb.st_mode & S_IFMT) == S_IFREG ? 'f' : (sb.st_mode & S_IFMT) == S_IFDIR ? 'd' : 'u';
	    s[1] = access(buf, R_OK) == 0 ? 'r' : '-'; /* Must use access if setuid  */
	    s[2] = access(buf, W_OK) == 0 ? 'w' : '-';
	    mkstrresp(mkstring(s));
	}
	break;
    case CreateDirectory:
	/* Uses TAG PAIR0 */
	pp = (int **)p[2];
	namep = (int **)pp[1];
	stringp = (int **)pp[2];
	*--ep = stringp;
	evalstring(namep, buf, sizeof buf);
	stringp = *ep++;
	/* string ignored for now */
	if (mkdir(buf, 0777) < 0) {
	    fileerror(OtherError);
	} else {
	    mksuccess();
	}
	break;
    case Sleep:
	/* Uses TAG */
	pp = (int **)p[2];
	pp = evaluate(pp);
	sleep((int)pp[1]);
	mksuccess();
	break;
    case ChangeDirectory:
	/* Uses TAG */
	namep = (int **)p[2];
	evalstring(namep, buf, sizeof buf);
	if (chdir(buf) < 0) {
	    fileerror(OtherError);
	} else {
	    mksuccess();
	}
	break;
    case CreateProcess:
	/* Uses TAG */
	pp = (int **)p[2];
	/* pp points to a function that should be run. Fork and try to run it. */
	switch(fork()) {
	case -1:
	    fileerror(OtherError);
	    break;
	case 0:
	    /* Child */
	    freopen("/dev/null", "r", stdin); /* To avoid problems */
	    longjmp(topjmp, pp);
	    break;
	default:
	    /* parent: No wait!  Trust init to take care of that.  Could disinherit child by double fork. */
	    mksuccess();
	    break;
	}
	break;
    case Echo:
	/* Uses TAG */
	pp = (int **)p[2];
	pp = evaluate(pp);
	set_tty(!pp[1]);
	mksuccess();
	break;
    case GetEnv:
	/* Uses TAG */
	namep = (int **)p[2];
	evalstring(namep, buf, sizeof buf);
	if (s = getenv(buf)) {
	    mkstrresp(mkstring(s));
	} else {
	    mkerrresp(SearchError, "");
	}
	break;
    case SetEnv:
	/* Uses TAG PAIR0 */
	pp = (int **)p[2];
	namep = (int **)pp[1];
	stringp = (int **)pp[2];
	*--ep = stringp;
	evalstring(namep, buf, sizeof buf);
	stringp = *ep++;
	{
	    char buf1[MAXNAME];
	    evalstring(stringp, buf1, sizeof buf1);
	    xsetenv(buf, buf1);
	}
	mksuccess();
	break;
    case GetArgs:
	mkstrresp(concargs);
	break;
    case WriteBinFile:
	/* Uses PAIR4 */
	mode = "w";
	namep = (int **)p[1];
	stringp = (int **)p[2];
	goto bwrapp;
    case AppendBinFile:
	/* Uses TAG PAIR0 */
	mode = "a";
	pp = (int **)p[2];
	namep = (int **)pp[1];
	stringp = (int **)pp[2];
  bwrapp:
	*--ep = stringp;
	evalstring(namep, buf, sizeof buf);
	stringp = *ep++;
  boutopen:
	f = fopen(buf, mode);
	if (f == NULL)
	    fileerror(WriteError);
	else {
  bdoout:
	    r = writebin(f, stringp);
	    if (r < 0)
		fileerror(WriteError);
	    else
		mksuccess();
	    if (f != stdout && f != stderr)
		fclose(f);
	    else
		fflush(f);
	}
	break;
    case AppendBinChan:
	/* Uses TAG PAIR0 */
	pp = (int **)p[2];
	namep = (int **)pp[1];
	stringp = (int **)pp[2];
	*--ep = stringp;
	evalstring(namep, buf, sizeof buf);
	stringp = *ep++;
	if (strcmp(buf, Stdout) == 0) {
	    f = stdout;
	    goto bdoout;
	} else if (strcmp(buf, Stderr) == 0) {
	    f = stderr;
	    goto bdoout;
	} else if (strcmp(buf, Stdecho) == 0) {
	    f = stdout;		/* The best we can do under UNIX */
	    goto bdoout;
	} else
	    goto boutopen;
	break;
    case ReadBinFile:
	/* Uses TAG */
	namep = (int **)p[2];
	evalstring(namep, buf, sizeof buf);
  binopen:
	f = fopen(buf, "r");
	if (f == NULL)
	    fileerror(errno == ENOENT ? SearchError : ReadError);
	else {
  bdoin:
	    mkresponse(mknode(TAG, (int **)Bn, readbin(f)));
	    if (f != stdin)
		fclose(f);
	}
	break;
    case ReadBinChan:
	/* Uses TAG */
	namep = (int **)p[2];
	evalstring(namep, buf, sizeof buf);
	if (channelused(buf)) {
	    mkerrresp(ReadError, "Channel already used");
	} else {
	    if (strcmp(buf, Stdin) == 0) {
		f = stdin;
		goto bdoin;
	    } else
		goto binopen;
	}
	break;
    case ReadChannels:
	/* Uses TAG */
	{
	    int **nlist = (int **)p[2], **pp, **vec;
	    FILE *table[1000], *f;
	    int nf, i;

	    /* evaluate the list and each string in it */
	    *--ep = nlist;
	    for(pp = nlist, nf = 0;; nf++) {
		pp = evaluate(pp);
		if (pp[0] != (int *)PAIR1)
		    break;		/* end of list */
		*--ep = pp;
		evalstring(pp[1], buf, sizeof buf); /* evaluate string */
		if (channelused(buf)) {
		    mkerrresp(ReadError, "Channel already used");
		    goto rcerr;
		}
		if (strcmp(buf, Stdin) == 0) {
		    f = stdin;
		} else {
		    f = fopen(buf, "r");
		}
		if (f == NULL) {
		    fileerror(errno == ENOENT ? SearchError : ReadError);
		    goto rcerr;
		}
		table[nf] = f;
		setbuf(f, NULL);
/*printf("opened=%s %d\n", buf, fileno(f));*/
		pp = *ep++;
		pp = (int **)pp[2];
	    }
	    if (hp + nf*7 + 10 > ehp)
		gcstack(0);
	    nlist = *ep++;
	    vec = hp;
	    hp += nf+2;
	    vec[0] = (int *)VEK;
	    vec[1] = (int *)nf;
	    /* Now build a vector */
	    for(i = 0, pp = nlist; i < nf; i++) {
		vec[i+2] = (int *)mknode(PAIR0, pp[1], mknode(INPUT, table[i], 0));
		pp = (int **)pp[2];
	    }
	    mkresponse(mknode(TAG, (int **)Tag, mknode(AP, readchan, vec)));
	    break;
	rcerr:
	    for(i = 0; i < nf; i++)
		fclose(table[i]);
	}
	break;
    case ReadBinChannels:
    case OpenFile:
    case OpenBinFile:
    case CloseFile:
    case ReadVal:
    case ReadBinVal:
    case WriteVal:
    case WriteBinVal:
	fprintf(stderr, "Uninplemented I/O request %d\n", t);
	exit(1);
	break;	
    default:
	fprintf(stderr, "Unknown I/O request %d\n", t);
	exit(1);
	break;
    }
}

/* Enter here when a readchannel request is evaluated. */
int **
creadchan(vp)
int **vp;
{
    fd_set mask;
    int nf, i, n, fd, mfd, ch;
    int **p, **ip;
    FILE *f;

/*printf("creadchan ");*/
    if (vp[0] != (int *)VEK) {
	fprintf(stderr, "Bad channels arg to creadchan\n");
	exit(1);
    }
    nf = (int)vp[1];
    FD_ZERO(&mask);
    for(mfd = -1, i = 0; i < nf; i++) {
	p = (int **)vp[i+2];
	ip = (int **)p[2];
	f = (FILE *)ip[1];
	fd = fileno(f);
/*printf("%d ", fd);*/
	FD_SET(fd, &mask);
	if (fd > mfd)
	    mfd = fd;
    }
/*printf("selecting\n");*/
    n = select(mfd+1, &mask, (fd_set *)0, (fd_set *)0, (struct timeval *)0);
/*printf("select returns %d\n", n);*/
    if (n <= 0) {
	fprintf(stderr, "unexpected value from select %d, errno=%d\n", n, errno);
	exit(1);
    }
    /* figure out which file can deliver data */
    for(i = 0; i < nf; i++) {
	p = (int **)vp[i+2];
	ip = (int **)p[2];
	f = (FILE *)ip[1];
	fd = fileno(f);
	if (FD_ISSET(fd, &mask)) {
	    /* fount it! */
/*printf("found input on %d\n", fd);*/
	    ch = getc(f);
/*printf("got %02x\n", ch);*/
	    return mknode(PAIR1, mknode(PAIR0, p[1], mknode1(CHAR, ch)), mknode(AP, readchan, vp));
	}
    }
    fprintf(stderr, "No fd found!\n");
    exit(1);
}

#ifdef sequent
xsetenv(var, val)
char *val, *var;
{
    setenv(var, val, 1);
}
#endif

#if defined(sun) || defined(mips)
xsetenv(var, val)
char *var, *val;
{
    char b[MAXNAME];

    sprintf("%s=%s", b, var, val);
    putenv(b);
}
#endif

/* xsetenv should be defined for other machines as well */
