\begin{caml_primitive}
mult_digit_nat  : nat * int * int * nat * int * int * nat * int -> int
mult_nat  : nat * int * int * nat * int * int * nat * int * int -> int
shift_left_nat  : nat * int * int * nat * int * int -> unit
square_nat  : nat * int * int * nat * int * int -> int
\end{caml_primitive}
\subsubsection{Multiplication of a natural number by a digit}
\verb"mult_digit_nat (nat1, off1, len1, nat2, off2, len2, nat3, off3)" 
multiplies subnat $nat2_{off2, len2}$ by digit $nat3_{off3}$, adds the 
result to $nat1_{off1, len1}$ and returns the carry out $c\_out$.  

More formally speaking \verb"mult_digit_nat" performs the following
operation~: 
\begin{eqnarray*} 
\lefteqn{nat1_{off1, len1} + nat3_{off3} \times nat2_{off2, len2} =}\\
& & \left(\sum_{i=0}^{i < len1} nat1_{off1+i} \,B^i \right) +
nat3_{off3} \, \left(\sum_{i=0}^{i<len2} nat2_{off2+i} \, B^i \right)=\\ 
& & \left(\sum_{i=0}^{i < len1} nat'_{off1+i} \,B^i \right) + 
c\_out \, B^{len1}.  
\end{eqnarray*} 
%
% nat1         + nat3     * nat2           =
%    off1, len1      off3       off2, len2         
%                         
%    /                       \              /                       \
%    | i < len1              |              | i < len2              |  
%    | --------              |              | --------              | 
%    |  \                  i |              |  \                  i |       
%    |   )      nat1      B  | + nat3     * |   )      nat2      B  | =
%    |  /           off1+i   |       off3   |  /          off2+i    |
%    | --------              |              | --------              |   
%    |  i = 0                |              |  i = 0                |
%    \                       /              \                       /
%
%        /                        \
%        | i < len1               |
%        | --------               |
%        |  \                   i |          len1
%        |   )       nat'      B  | + c_out B     .
%        |  /           off1+i    |   
%        | --------               |
%        |  i = 0                 |
%        \                        /

\verb"mult_digit_nat" replaces each digit $nat1_{off1+i}$ with its 
equivalent digit $nat'_{off1+i}$ and returns the carry out $c\_out$, with 
value 0 or 1. This implies $len1 > len2$.  

If $len2=0$ and $c\_out=0$ and no side effect is performed.  
$nat1$ and $nat2$ can be physically the same one under the condition 
$off1 \leq off2$. 
The digit $nat3_{off3}$ may be any digit of $nat1$ or $nat2$.

\begin{beware}
\item For speed, the cases $nat3_{off3}=0$ and $nat3_{off3}=1$ are 
explicitly tested.
\end{beware}

\exple
\begin{small}
\begin{verbatim}
#let nat1 = make_nat #5
#and nat2 = create_nat #4;;
Value nat1 is #<0> : nat
Value nat2 is #<0> : nat

#set_digit_nat (nat1, #1, #2);;
() : unit

#for i = #0 to #3 do set_digit_nat (nat2, i, i) done;;
() : unit

#debug_print_nat nat1;;
|00000000|00000000|00000000|00000002|00000000|() : unit

#debug_print_nat nat2;;
|00000003|00000002|00000001|00000000|() : unit

#mult_digit_nat (nat1, #0, #5, nat2, #0, #4, nat_of_int #15, #0);;
0 : int

#debug_print_nat nat1;;
|00000000|0000002D|0000001E|00000011|00000000|() : unit

#mult_digit_nat (nat1, #0, #5, nat2, #0, #4, nat2, #2);;
0 : int

#debug_print_nat nat1;;
|00000000|00000033|00000022|00000013|00000000|() : unit
\end{verbatim}
\end{small}

\subsubsection{Multiplication on natural numbers}
\verb"mult_nat (nat1, off1, len1, nat2, off2, len2, nat3, off3, len3)" 
multiplies subnats $nat2_{off2, len2}$ and $nat3_{off3, len3}$,
adds the result to $nat1_{off1, len1}$ and returns the carry out
$c\_out$. 

More formally speaking \verb"mult_nat" performs the following operation : 
\begin{eqnarray*}
\lefteqn{nat1_{off1, len1} + nat2_{off2, len2} \times nat3_{off3, len3} =}\\ 
& & \left(\sum_{i=0}^{i < len1} nat1_{off1+i} \,B^i \right) +
\left(\sum_{i=0}^{i<len2} nat2_{off2+i} \, B^i \right) \,
\left(\sum_{i=0}^{i<len3} nat3_{off3+i} \, B^i \right)=\\ 
& & \left(\sum_{i=0}^{i < len1} nat'_{off1+i} \,B^i \right) 
+ c\_out \, B^{len1}.  
\end{eqnarray*} 
%
% nat1         + nat2           * nat3           =
%    off1, len1      off2, len2       off3, len3  
%                         
%/                       \   /                       \ /                       \
%| i < len1              |   | i < len2              | | i < len3              |  
%| --------              |   | --------              | | --------              |
%|  \                  i |   |  \                  i | |  \                  i       
%|   )      nat1      B  | + |   )      nat2      B  | |   )      nat3      B  |
%|  /           off1+i   |   |  /          off2+i    | |  /          off3+i    |
%| --------              |   | --------              | | --------              |  
%|  i = 0                |   |  i = 0                | |  i = 0                |
%\                       /   \                       / \                       /
%
%        /                        \
%        | i < len1               |
%        | --------               |
%        |  \                   i |          len1
%      = |   )       nat'      B  | + c_out B     .
%        |  /           off1+i    |   
%        | --------               |
%        |  i = 0                 |
%        \                        /

\verb"mult_nat" replaces each digit $nat1_{off1+i}$ with its equivalent 
digit $nat'_{off1+i}$ and returns the carry out $c\_out$, with value 0 or 1. 
This implies $len1 \geq len2 + len3$. If $len2=0$ and $c\_out=0$ no
side effect is performed. 

\begin{beware}
\item $nat2$ and $nat3$ have quite similar roles,  but for speed reasons
it is preferable that $len2 \geq len3$,
\item No overlap is possible between $nat1_{off1, len1}$
and $nat2_{off2, len2}$ or $nat3_{off3, len3}$.
\end{beware}

\exple
\begin{small}
\begin{verbatim}
#let nat1 = create_nat #4 
#and nat2 = create_nat #4;;
Value nat1 is #<0> : nat
Value nat2 is #<0> : nat

#for i = #0 to #3 do 
#  set_digit_nat (nat2, i, i); 
#  set_digit_nat (nat1, i, add_int (i, #4)) 
#done; 
#debug_print_nat nat2;;
|00000003|00000002|00000001|00000000|() : unit

#debug_print_nat nat1;;
|00000007|00000006|00000005|00000004|() : unit

#mult_nat (nat1, #0, #4, nat2, #0, #2, nat2, #2, #2);;
0 : int

#debug_print_nat nat1;;
|00000007|00000009|00000007|00000004|() : unit

#complement_nat (nat2, #0, #4);;
() : unit

#debug_print_nat nat2;;
|FFFFFFFC|FFFFFFFD|FFFFFFFE|FFFFFFFF|() : unit

#mult_nat (nat1, #0, #4, nat2, #0, #2, nat2, #2, #2);;
1 : int

#debug_print_nat nat1;;
|00000003|00000008|0000000D|00000007|() : unit
\end{verbatim}
\end{small}

\subsubsection{Shifting left natural numbers}
\verb"shift_left_nat (nat1, off1, len1, nat2, off2, shift)" shifts left 
any digit of subnat $nat1_{off1, len1}$. The $shift$ bits out of any digit 
replace the $shift$ free bits of the next digit.  
$shift$ zeros replace the $shift$ free bits of the first digit and the least
significant bits of digit $nat2_{off2}$ get back the $shift$ bits
out of the last digit. 
So \verb"shift_left_nat" multiplies $nat1_{off1, len1}$ by $2^{shift}$.  

More formally speaking \verb"shift_left_nat" performs the following operation~:
\begin{eqnarray*} 
\lefteqn{2^{shift}\, nat1_{off1, len1} =}\\
& & 2^{shift}\, \left(\sum_{i=0}^{i < len1} nat1_{off1+i} \,B^i \right) = \\
& & \left(\sum_{i=0}^{i < len1} nat1'_{off1+i} \,B^i \right) 
+ nat2'_{off2} \, B^{len1}.  
\end{eqnarray*}
%                                  /                        \
%                                  | i < len1               |  
%                                  | --------               |   
%    shift                   shift |  \                   i |  
%   2      nat1           = 2      |   )      nat1       B  | =            
%              off1, len1          |  /           off1+i    | 
%                                  | --------               | 
%                                  |  i = 0                 |  
%                                  \                        /

%                           /                          \
%                           |  i < len1                |
%                           |  --------                |          
%                           |   \                    i |              len1
%                         = |    )      nat1'       B  | + nat2'     B     .
%                           |   /           off1+i     |        off2
%                           |  --------                |
%                           |    i = 0                 |
%                           \                          /

\verb"shift_left_nat" replaces each digit $nat1_{off1+i}$ with its
equivalent digit $nat1'_{off1+i}$ and $nat2_{off2}$ with $nat2'_{off2}$.  
This implies $0 \leq shift < length\_of\_digit$ .

\begin{beware}
\item For speed, the case $shift=0$ is explicitly tested.  
\end{beware}
\exple
\begin{small}
\begin{verbatim}
#let nat = create_nat #4;;
Value nat is #<0> : nat

#for i = #0 to #3 do set_digit_nat (nat, i, i) done;;
() : unit

#debug_print_nat nat;;
|00000003|00000002|00000001|00000000|() : unit

#shift_left_nat (nat, #1, #3, nat, #0, #8);;
() : unit

#debug_print_nat nat;;
|00000300|00000200|00000100|00000000|() : unit

#shift_left_nat (nat, #1, #3, nat, #0, #16);;
() : unit

#debug_print_nat nat;;
|03000000|02000000|01000000|00000000|() : unit

#shift_left_nat (nat, #1, #3, nat, #0, #16);;
() : unit

#debug_print_nat nat;;
|00000200|00000100|00000000|00000300|() : unit
\end{verbatim}
\end{small}

\subsubsection{Squaring natural numbers}
\verb"square_nat (nat1, off1, len1, nat2, off2, len2)" squared subnat
$nat2_{off2, len2}$, adds the result to $2\;nat1_{off1, len1}$ and
returns the carry out \verb"c_out".  

More formally speaking \verb"square_nat" performs the following operation : 
\begin{eqnarray*}
\lefteqn{2\;nat1_{off1, len1} + \left({nat2_{off2, len2}}^2 \right) =}\\
& & \left(\sum_{i=0}^{i < len1} nat1_{off1+i} \,B^i \right) + {\left(
\sum_{i=0}^{i<len2} nat2_{off2+i} \, B^i \right)}^2 = \\ 
& & \left(\sum_{i=0}^{i < len1} nat'_{off1+i} \,B^i \right) + c\_out \,
B^{len1}.  
\end{eqnarray*} 
%                    /              \ 2
% 2 nat1          + | nat2           |   =
%       off1, len1  |     off2, len2 |  
%                    \              /
%
%    /                       \   /                       \ 2
%    | i < len1              |   | i < len2              | 
%    | --------              |   | --------              | 
%    |  \                  i |   |  \                  i | 
%  2 |   )      nat1      B  | + |   )      nat2      B  |    =
%    |  /           off1+i   |   |  /          off2+i    | 
%    | --------              |   | --------              | 
%    |  i = 0                |   |  i = 0                | 
%    \                       /   \                       / 
%
%        /                        \
%        | i < len1               |
%        | --------               |
%        |  \                   i |          len1
%      = |   )       nat'      B  | + c_out B     .
%        |  /           off1+i    |   
%        | --------               |
%        |  i = 0                 |
%        \                        /

\verb"square_nat" replaces each digit $nat1_{off1+i}$ with its equivalent 
digit $nat'_{off1+i}$ and returns the carry out $c\_out$, with value 0 
or 1. This implies $len1 \geq 2 \times \, len2$. 
If $len2$ is null, $c\_out=0$ and no side effect is performed. 
No overlap is possible between $nat1_{off1, len1}$ and $nat2_{off2, len2}$.  

\exple
\begin{small}
\begin{verbatim}
#let nat2 = create_nat #2
#and nat1 = make_nat #4;;
Value nat2 is #<0> : nat
Value nat1 is #<0> : nat

#set_digit_nat (nat2, #0, #1);
#set_digit_nat (nat2, #1, #2);
#debug_print_nat nat2;;
|00000002|00000001|() : unit

#square_nat (nat1, #0, #4, nat2, #0, #2);;
0 : int

#debug_print_nat nat1;;
|00000000|00000004|00000004|00000001|() : unit
\end{verbatim}
\end{small}
