(*************************************************************************)
(*                                                                       *)
(*                     Projet      Formel                                *)
(*                                                                       *)
(*                    CAML: users' library                               *)
(*                                                                       *)
(*************************************************************************)
(*                                                                       *)
(*                            LIENS                                      *)
(*                        45 rue d'Ulm                                   *)
(*                         75005 PARIS                                   *)
(*                            France                                     *)
(*                                                                       *)
(*************************************************************************)

(* sketches.ml                                                           *)
(*              Emmanuel Chailloux & Guy Cousineau                       *)

module sketches
using
type point = {xc:float;yc:float};
type geom_element =
  Seg of point list
| Arc of point * float * float * float   
        (* center,radius,start_angle,end_angle*)
| Curve of point * point * point * point;
        (* start,control1,control2,end *)
type frame = {xmin:float;xmax:float;ymin:float;ymax:float};
type extension;
type transformation;
value compute_geom_elem_frame : geom_element -> frame;
value compute_frame : geom_element list -> frame;
value extend_frame : extension -> float -> frame -> frame;
value frame_center : frame -> point;
value compose_frames : frame -> frame -> frame;
value transform_frame : transformation -> frame -> frame;
value frame_to_frame_transform : frame -> frame -> transformation;
value translation : float * float -> transformation;
value scaling: float * float -> transformation;
value rotation: point -> float -> transformation;
value vsymetry : float -> transformation;
value hsymetry : float -> transformation;
value CT: transformation * transformation -> transformation;;

#pragma infix "CT";;
#pragma infix "JSK";;
#pragma infix "BSK";;
#pragma infix "OSK";;

#arith float;;
#arith int;;

type path = Spath of  geom_element list  
          | Tpath of  transformation * path 
          | Cpath of path * path;;

type sketch = {path: path; frame:frame; size:int};;

let sketch_frame sk = sk.frame;;

let compute_size =
  it_list (fun n ge -> n + (match ge 
                            with (Seg ptl) -> length ptl
                            | (Arc _)  ->  3
                            | (Curve _) -> 4))
          0;;
                                 
let make_sketch gel =
    {path= Spath gel; frame= compute_frame gel; size=compute_size gel};;

let frame_sketch ({xmin=a;xmax=b;ymin=c;ymax=d} as f) =
    {path= Spath [Seg [{xc=a;yc=c};{xc=a;yc=d};
                        {xc=b;yc=d};{xc=b;yc=c};{xc=a;yc=c}]];
     frame= f;
     size = 5};;


let sketch_center (sk:sketch) = frame_center (sk.frame);;  


let join_sketches {path=p1; frame=f1; size=s1} 
                  {path=p2; frame=f2; size=s2} = 
  {path= Cpath(p1,p2) ; frame=compose_frames f1 f2; size=s1+s2};;

let sk1 JSK sk2 =  join_sketches sk1 sk2;;

let join_sketch_list = function
  [] -> failwith "join_sketch_list"
| sk::skl ->
  it_list join_sketches sk skl;;

let transform_sketch t =
  let tf = transform_frame t  in
 fun  {path=Tpath(t',p) ; frame =f; size=s}
       ->  {path=Tpath(t CT t',p) ; frame = tf f; size=s}
   | {path= p ; frame =f; size=s}
       -> {path = Tpath(t,p) ; frame = tf f; size=s};;

let fit_sketch_in_frame (sk:sketch) f =
   let t = frame_to_frame_transform (sk.frame) f
   in  transform_sketch t sk;;

let force_sketch_in_frame f sk =   
   {path=sk.path;frame=f;size=sk.size};;

let scale_sketch (hscale,vscale) (sk:sketch) =
  let a= sk.frame.xmin and c=sk.frame.ymin
  in let T1 = translation (-a,-c)
     and S = scaling (hscale,vscale)
     and T2 = translation (a,c)
     in
       transform_sketch (T2 CT S CT T1) sk;;



let vflip_sketch sk =
   let s = vsymetry (sketch_center sk).xc
   in  transform_sketch s sk;;

let hflip_sketch sk =
   let s = hsymetry (sketch_center sk).yc
   in  transform_sketch s sk;;

let rotate_sketch a sk =
     transform_sketch (rotation (sketch_center sk) a ) sk;;  
    

let besides_sketch (sk1:sketch) (sk2:sketch) =    
    let {xmin=a; xmax=b; ymin=c; ymax=d} = sk1.frame
    and  width2 = sk2.frame.xmax - sk2.frame.xmin 
in sk1 JSK (fit_sketch_in_frame  sk2 {xmin=b;xmax=b+width2;ymin=c;ymax=d})  ;;

let over_sketch (sk1:sketch) (sk2:sketch) =    
    let {xmin=a; xmax=b; ymin=c; ymax=d} = sk1.frame
    and  height2 = sk2.frame.ymax - sk2.frame.ymin 
in sk1 JSK (fit_sketch_in_frame  sk2 {xmin=a;xmax=b;ymin=c-height2;ymax=c})  ;;

let sk1 BSK sk2 = besides_sketch sk1 sk2;;
let sk1 OSK sk2 = over_sketch sk1 sk2;;

let extend_sketch_frame str k {path=p;frame=fr;size=n} =
    {path=p;frame=extend_frame str k fr;size=n};;

end module
with
type sketch
 and path;
value sketch_frame
  and make_sketch
  and frame_sketch
  and sketch_center
  and join_sketches
  and prefix JSK
  and join_sketch_list
  and transform_sketch
  and fit_sketch_in_frame
  and force_sketch_in_frame
  and scale_sketch
  and vflip_sketch
  and hflip_sketch
  and rotate_sketch
  and besides_sketch
  and over_sketch
  and prefix BSK
  and prefix OSK
  and extend_sketch_frame;;

