(* $Id: test.ml,v 8.5 91/06/20 11:36:49 ddr Exp $
 *
 * Rogloglo Toolkit: test
 *
 * $Log:	test.ml,v $
 * Revision 8.5  91/06/20  11:36:49  ddr
 * - remerge avec zinc
 * 
 * Revision 8.4  91/06/19  20:01:40  ddr
 * - merge avec zinc 1.11
 * 
 * Revision 8.3  91/06/17  10:02:01  ddr
 * - imports
 * 
 * Revision 8.2  91/06/15  16:04:20  ddr
 * - evolution
 * 
 * Revision 8.1  91/06/15  11:03:02  ddr
 * - merge avec zinc
 * 
 * Revision 7.10  91/06/07  20:14:52  ddr
 * - redistrib
 *)

module test using
  (* from rt *)
  type xdata; type widget; type widget_desc; type xargs;
  type attribute; type background; type color; type orientation;
  value rt_initialize : string -> xdata;
  value rt_create_widget : xdata * string * string * widget_desc -> widget;
  value rt_destroy_widget : widget -> unit;
  value rt_map_widget : widget -> unit;
  value rt_unmap_widget : widget -> unit;
  value rt_main_loop : xargs -> unit;
  value rt_stop_main_loop : xargs -> unit;
  value rt_end : xdata -> unit;
  value rt_select_file : xargs * num * (unit -> unit) -> unit;
  value rt_set_timeout : xargs * num -> unit;
  value rt_reset_timeout : xargs -> unit;
  value rt_current_time : xargs -> num;
  value rt_set_timeout_fun : xargs * (unit -> unit) -> unit;
  value rt_args : xdata list -> xargs;
  value NoneBg : background;
  value ColorBg : color -> background;
  value BackgroundAtt : background -> attribute;
  value FillerAtt : attribute;
  value NameAtt : string -> attribute;
  value WidthAtt : num -> attribute;
  value HeightAtt : num -> attribute;
  value BorderAtt : num -> attribute;
  value widget_named : xdata -> string -> widget;
  value widget_x : widget -> num;
  value widget_y : widget -> num;
  value widget_width : widget -> num;
  value widget_height : widget -> num;
  value widget_border : widget -> num;
  value xevent_x : xdata -> num;
  value xevent_y : xdata -> num;
  value xevent_x_root : xdata -> num;
  value xevent_y_root : xdata -> num;
  value Horizontal : orientation;
  value Vertical : orientation;

  (* from c_pack *)
  value PackA : attribute list ->
    orientation * widget_desc list -> widget_desc;
  value PackD : orientation * widget_desc list -> widget_desc;

  (* from c_button *)
  value ButtonD : string * (widget -> unit) -> widget_desc;
  value ButtonA : attribute list -> string * (widget -> unit) -> widget_desc;
  value PopupD : string * (widget -> unit) -> widget_desc;
  value PopupA : attribute list -> string * (widget -> unit) -> widget_desc;
  value CommD : string * (widget -> unit) -> widget_desc;
  value CommA : attribute list -> string * (widget -> unit) -> widget_desc;

  (* from c_scroll *)
  value ScrollD : orientation * num * num * num *
    (widget * num * num -> unit) -> widget_desc;
  value ScrollA : attribute list -> orientation * num * num * num *
    (widget * num * num -> unit) -> widget_desc;
  value scroll_set : widget * num -> unit;
  value scroll_val : widget -> num;

  (* from c_text *)
  value TextA : attribute list ->
    num * num * num * (widget * string -> unit) *
    (widget * num * num * num -> unit) -> widget_desc;
  value text_send_string : widget * string -> unit;
  value text_shift : widget * num -> unit;
  value text_shift_value : widget -> num;
  value text_set_mark : widget * num * num -> unit;
  value text_get_text : widget * num * num -> string;
  value text_goto : widget * num * num -> unit;
  value text_home : widget -> unit;
  value rt_set_cut_buffer : xdata * string -> unit;
  value rt_get_cut_buffer : xdata -> string;

  (* from c_title *)
  value TitleD : string -> widget_desc;

  (* from c_select *)
  value SelectA : attribute list -> widget_desc list -> widget_desc;
  value select_raise : widget * num -> unit;

  (* from c_raw *)
  type property;
  value ButtonPressedPr : (widget * num * num * num -> unit) -> property;
  value RawA : attribute list ->
    num * num * num * property list -> widget_desc;

  (* from color *)
  value rt_create_color : xdata * num * num * num -> color;
  value rt_white_color : xdata -> color;

  (* from misc *)
  value rt_create_subwidget : widget * num * num * widget_desc -> widget;
  value rt_move_widget : widget * num * num -> unit;
  value rt_resize_widget : widget * num * num -> unit;
  type position;
  value UserPosition : num * num -> position;
  value AutoPosition : position;
  value rt_create_located_widget :
    xdata * string * string * position * widget_desc -> widget;
  value rt_create_transient_widget : widget * string * widget_desc -> widget;
  value rt_map_transient_widget : widget * num * num -> unit;
  value rt_create_popup_widget : xdata * widget_desc -> widget;
  value rt_map_popup_widget : widget * num * num * num -> unit;

  type timeb = {time: num; millitm: num};
  value ftime : unit -> timeb;
;

#standard arith false;;
#fast arith false;;

let string_length = length_string;;
let COLOR = ref false;;

let my_scroll_set vmin vmax shift act (wid, but, val) =
  let oval = (scroll_val wid)-shift in
  let val =
    if but = 1 then oval-1
    else if but = 3 then oval+1
    else val in
  let val = max vmin (min vmax val) in
  if val <> oval then (
    scroll_set(wid, val+shift);
    act val
  )
;;

let my_text_button xd (wid, but, lin, col) =
  match but with
    1 ->
      text_set_mark(wid, lin, col)
  | -1 | 3 ->
      let txt = text_get_text(wid, lin, col) in
      rt_set_cut_buffer(xd, txt)
  | -2 ->
      let txt = rt_get_cut_buffer xd in
      text_send_string(wid, txt)
  | _ ->
      ()
;;

let action wid_named _ =
  message "le bouton \"bouton\" provoque l'appel de cette fonction";
  let main_wid = wid_named "test" in
  display_string " x="; display_num (widget_x main_wid);
  display_string " y="; display_num (widget_y main_wid);
  display_string " w="; display_num (widget_width main_wid);
  display_string " h="; display_num (widget_height main_wid);
  display_newline()

and test_popup wid_named txt _ =
  text_send_string(wid_named "T", txt ^ " ")

and autre_action xd _ =
  let ca_marche wid = message "ca marche " in
  let wid = rt_create_widget(
    xd, "glop", "glop",
    ButtonD("bouton", ca_marche)
  ) in
  rt_map_widget wid

and quit_fun wid_named _ =
  let wid = wid_named "test"
  and wid_q = wid_named "question" in
  let w = widget_width wid and h = widget_height wid
  and sw = widget_width wid_q and sh = widget_height wid_q in
  rt_move_widget(wid_q, (w-sw)quo 2, (h-sh)quo 2);
  rt_map_widget wid_q

and keyp_fun(wid, c) =
  text_send_string(wid, " \b" ^ c ^ "_\b")

and stdin_fun _ =
  let c = read std_in #1 in
  print_string "<"; print_string c; print_string ">"; print_flush()

and trace_scr val =
  print_string "scroll ";
  print_string "; val="; print_num val;
  print_newline()

and answer xa wid_named b _ =
  if b then rt_stop_main_loop xa
  else rt_unmap_widget(wid_named "question")

and popup_on_raw popup_wid (wid, but, x, y) =
  rt_map_popup_widget(popup_wid, x, y, 0)
;;

let xll xd wid = xevent_x_root xd - xevent_x xd - widget_border wid
and yll xd wid = xevent_y_root xd - xevent_y xd + widget_height wid
;;

let sub_popup xd (txt, wid, lev) =
  PackD(Horizontal, [
    PopupA [FillerAtt] (txt, fun _ -> rt_unmap_widget wid);
    PopupD (">", fun bwid ->
      let xll = xll xd bwid and yll = yll xd bwid in
      let x = xll + (widget_width bwid quo 2)
      and y = yll - (widget_height bwid quo 2) in
      rt_map_popup_widget(wid, x, y, lev)
    )
  ])

and empty_wid xd =
  rt_create_popup_widget(xd, PackD(Horizontal, []))
;;

let translate_wid xd =
  let wid_named = widget_named xd in
  let test_popup = test_popup wid_named in
  rt_create_popup_widget(xd, PackD(Vertical, [
    TitleD "Hello";
    CommD("this", test_popup "ceci");
    CommD("is", test_popup "est");
    CommD("a", test_popup "un");
    CommD("popup menu", test_popup "menu deroulant");
    CommD("", fun _ -> ());
    CommD("gag", fun _ -> select_raise(wid_named "Menu", 1))
  ]))

and gag_wid xd =
  let wid_named = widget_named xd in
  let test_popup = test_popup wid_named in
  rt_create_popup_widget(xd, PackD(Vertical, [
    PackD(Horizontal, [
      TitleD "Salut";
      CommD("Ca", test_popup "my");
      CommD("marche", test_popup "tailor");
      CommD("aussi", test_popup "is");
      CommD("horizontalement", test_popup "rich")
    ]);
    CommD("ok", fun _ -> select_raise(wid_named "Menu", 0))
  ]))

and button_wid xd =
  let wid_named = widget_named xd in
  rt_create_popup_widget(xd, CommD("bouton", action wid_named))

and machines_wid xd service_wid sun_wid workstations_wid =
  let test_popup = test_popup (widget_named xd) in
  rt_create_popup_widget(xd, PackD(Vertical, [
    sub_popup xd ("service", service_wid, 1);
    sub_popup xd ("sun", sun_wid, 1);
    sub_popup xd ("workstations", workstations_wid, 1)
  ]))

and service_wid xd =
  let test_popup = test_popup (widget_named xd) in
  rt_create_popup_widget(xd, PackD(Vertical, [
    CommD("Margaux", test_popup "Margaux");
    CommD("Pommard", test_popup "Pommard");
    CommD("Seti", test_popup "Seti");
    CommD("Nuri", test_popup "Nuri");
    CommD("Romanee", test_popup "Romanee");
    CommD("Corton", test_popup "Corton");
    CommD("Ens", test_popup "Ens")
  ]))

and sun_wid xd =
  let test_popup = test_popup (widget_named xd) in
  rt_create_popup_widget(xd, PackD(Vertical, [
    CommD("Beaune", test_popup "Beaune");
    CommD("Brouilly", test_popup "Brouilly");
    CommD("Cornas", test_popup "Cornas");
    CommD("Yquem", test_popup "Yquem")
  ]))

and workstations_wid xd algo_wid croap_wid formel_wid para_wid ens_wid =
  rt_create_popup_widget(xd, PackD(Vertical, [
    sub_popup xd ("algo", algo_wid, 2);
    sub_popup xd ("croap", croap_wid, 2);
    sub_popup xd ("formel", formel_wid, 2);
    sub_popup xd ("para", para_wid, 2);
    sub_popup xd ("ens", ens_wid, 2)
  ]))

and algo_wid xd =
  let test_popup = test_popup (widget_named xd) in
  rt_create_popup_widget(xd, PackD(Vertical, [
    CommD("Amour", test_popup "Amour");
    CommD("Bandol", test_popup "Bandol");
    CommD("Charmes", test_popup "Charmes");
    CommD("Fleurie", test_popup "Fleurie");
    CommD("Nuits", test_popup "Nuits");
    CommD("Quarts", test_popup "Quarts");
    CommD("Tokay", test_popup "Tokay")
  ]))

and croap_wid xd =
  let test_popup = test_popup (widget_named xd) in
  rt_create_popup_widget(xd, PackD(Vertical, [
    CommD("Barr", test_popup "Barr");
    CommD("Barsac", test_popup "Barsac");
    CommD("Fixin", test_popup "Fixin");
    CommD("Huaine", test_popup "Huaine");
    CommD("Marix", test_popup "Marix");
    CommD("Moulis", test_popup "Moulis");
    CommD("Muscat", test_popup "Muscat");
    CommD("Pogo", test_popup "Pogo");
    CommD("Sancerre", test_popup "Sancerre");
    CommD("Sylvaner", test_popup "Sylvaner");
    CommD("Tobago", test_popup "Tobago");
    CommD("Wallis", test_popup "Wallis");
    CommD("Zeus", test_popup "Zeus")
  ]))

and formel_wid xd =
  let wid_named = widget_named xd in
  let test_popup = test_popup wid_named in
  rt_create_popup_widget(xd, PackD(Vertical, [
    CommD("Ausone", test_popup "Ausone");
    CommD("Banyuls", test_popup "Banyuls");
    CommD("Bergerac", test_popup "Bergerac");
    CommD("Bouzy", test_popup "Bouzy");
    CommD("Jurancon", test_popup "Jurancon");
    CommD("Latour", test_popup "Latour");
    CommD("Madiran", test_popup "Madiran");
    CommD("Morgon", test_popup "Morgon");
    CommD("Musigny", test_popup "Musigny");
    CommA [
      BackgroundAtt(ColorBg(
        if !COLOR then rt_create_color(xd, 255, 0, 0) else
        rt_white_color xd
      ))
    ] ("Peray", test_popup "Peray");
    CommD("", (fun _ -> ()));
    CommD("Pernand", test_popup "Pernand");
    CommD("Petrus", test_popup "Petrus");
    CommD("Pomerol", test_popup "Pomerol");
    CommD("Pouilly", test_popup "Pouilly");
    CommD("Quincy", test_popup "Quincy");
    CommD("Reuilly", test_popup "Reuilly");
    CommD("Riesling", test_popup "Riesling");
    CommD("Rieussec", test_popup "Rieussec");
    CommD("Santenay", test_popup "Santenay");
    CommD("Vougeot", test_popup "Vougeot")
  ]))

and para_wid xd =
  let test_popup = test_popup (widget_named xd) in
  rt_create_popup_widget(xd, PackD(Vertical, [
    CommD("Bellet", test_popup "Bellet");
    CommD("Chablis", test_popup "Chablis");
    CommD("Couchey", test_popup "Couchey");
    CommD("Givry", test_popup "Givry");
    CommD("Kpriss", test_popup "Kpriss");
    CommD("Volnay", test_popup "Volnay");
    CommD("Saumur", test_popup "Saumur")
  ]))

and ens_wid xd =
  let test_popup = test_popup (widget_named xd) in
  rt_create_popup_widget(xd, PackD(Vertical, [
    CommD("Aneth", test_popup "Aneth");
    CommD("Ares", test_popup "Ares");
    CommD("Asterix", test_popup "Asterix");
    CommD("Basilic", test_popup "Basilic");
    CommD("Berlioz", test_popup "Berlioz");
    CommD("Bouleau", test_popup "Bouleau");
    CommD("Cannelle", test_popup "Cannelle");
    CommD("Cassis", test_popup "Cassis");
    CommD("Cedre", test_popup "Cedre");
    CommD("Crocus", test_popup "Crocus");
    CommD("Debussy", test_popup "Debussy");
    CommD("Diamant", test_popup "Diamant");
    CommD("Ens", test_popup "Ens");
    CommD("Fuchsia", test_popup "Fuchsia");
    CommD("Gentiane", test_popup "Gentiane");
    CommD("Hermes", test_popup "Hermes");
    CommD("Jasmin", test_popup "Jasmin");
    CommD("Magnolia", test_popup "Magnolia");
    CommD("Menthe", test_popup "Menthe");
    CommD("Merisier", test_popup "Merisier");
    CommD("Mimosa", test_popup "Mimosa");
    CommD("Muscade", test_popup "Muscade");
    CommD("Myrtille", test_popup "Myrtille");
    CommD("Obelix", test_popup "Obelix");
    CommD("Oseille", test_popup "Oseille");
    CommD("Ravel", test_popup "Ravel");
    CommD("Rubis", test_popup "Rubis");
    CommD("Safran", test_popup "Safran");
    CommD("Saphir", test_popup "Saphir");
    CommD("Sibelius", test_popup "Sibelius");
    CommD("Verdi", test_popup "Verdi")
  ]))
;;

let rec make_test_widget xa xd translate_wid gag_wid machines_wid button_wid =
  let wid_named = widget_named xd in
  let NVIS = 10 and NSAV = 30 in
  let wid = rt_create_widget(xd, "test", "test",
    PackA [NameAtt "test"] (Horizontal, [
      SelectA [NameAtt "S"] [
        PackD(Vertical, [
          CommA [FillerAtt] ("bouton", action wid_named);
          ButtonD("pustule", autre_action xd);
          PackA [FillerAtt] (Vertical, [
            ButtonD("trans & pos", fun _ -> select_raise(wid_named "S", 3));
            ButtonD("restart", fun _ ->
              rt_destroy_widget (widget_named xd "test");
              rt_destroy_widget (widget_named xd "trans");
              let wid = make_test_widget xa xd
                translate_wid gag_wid machines_wid button_wid in
              rt_map_widget wid
            );
            ButtonD("surprise", fun _ -> select_raise(wid_named "S", 1))
          ])
        ]);
        PackD(Vertical, [
          ButtonD("next", fun _ -> select_raise(wid_named "S", 2));
          PackA [FillerAtt] (Vertical, []);
          ButtonD("return", fun _ -> select_raise(wid_named "S", 0))
        ]);
        PackD(Vertical, [
          TitleD("Scrolling text");
          ButtonD("home", fun _ -> text_home (wid_named "st"));
          ButtonD("up", fun _ ->
            let wid = wid_named "st" in
            text_shift(wid, text_shift_value wid + 1);
            scroll_set(wid_named "scr", text_shift_value wid + NVIS)
          );
          PackA [FillerAtt] (Horizontal, [
            ScrollA [NameAtt "scr"] (Vertical, 0, NSAV, NVIS,
              my_scroll_set 0 (NSAV-NVIS) NVIS (fun val ->
                let wid = wid_named "st" in
                text_shift(wid, val)
              )
            );
            TextA [NameAtt "st"] (
              NVIS, 25, NSAV, text_send_string, my_text_button xd
            )
          ]);
          ButtonD("down", fun _ ->
            let wid = wid_named "st" in
            text_shift(wid, text_shift_value wid - 1);
            scroll_set(wid_named "scr", text_shift_value wid + NVIS)
          );
          ButtonD("return", fun _ -> select_raise(wid_named "S", 1))
        ]);
        PackA [BorderAtt 0] (Vertical, [
          TitleD "trans & pos";
          ButtonD("timeout", fun _ ->
            rt_set_timeout(xa, rt_current_time xa + 1000);
            rt_set_timeout_fun(xa, fun _ ->
              let t = ftime()
              and wid = wid_named "T" in
              text_send_string(wid, "\n");
              text_send_string(wid, string_of_num t.time);
              text_send_string(wid, ".");
              let t = "00" ^ string_of_num t.millitm in
              let t = sub_string t (string_length t - 3) 3 in
              text_send_string(wid, t)
            )
          );
          ButtonD("timer", fun _ ->
            rt_set_timeout(xa, rt_current_time xa + 1000);
            rt_set_timeout_fun(xa, fun _ ->
              let t = ftime()
              and wid = wid_named "T" in
              text_send_string(wid, "\n");
              text_send_string(wid, string_of_num t.time);
              text_send_string(wid, ".");
              let t = "00" ^ string_of_num t.millitm in
              let t = sub_string t (string_length t - 3) 3 in
              text_send_string(wid, t);
              rt_set_timeout(xa, rt_current_time xa + 1000)
            )
          );
          ButtonD("stop timer", fun _ -> rt_reset_timeout xa);
          PackD(Horizontal, [
            ButtonD("trans", fun bwid ->
              let xll = xll xd bwid
              and yll = yll xd bwid in
              rt_map_transient_widget(wid_named "trans", xll, yll)
            );
            ButtonD("pos", fun _ -> rt_map_widget (wid_named "pos"));
            ButtonD("auto", fun _ -> rt_map_widget (wid_named "auto"))
          ]);
          ButtonA [HeightAtt 40] ("bouton gros", fun _ -> ());
          PackD (Horizontal, [
            ButtonA [WidthAtt 125] ("a", fun _ -> ());
            ButtonA [WidthAtt 125] ("meme taille que a", fun _ -> ())
          ]);
          ButtonA [BorderAtt 6] ("bouton a bords epais", fun _ -> ());
          ButtonA [BorderAtt 0] ("bouton sans bords", fun _ -> ());
          PackA [FillerAtt; BorderAtt 0] (Vertical, []);
          ButtonD("ok", fun _ -> select_raise(wid_named "S", 0))
        ])
      ];
      ScrollD(Vertical, 100, 150, 0, my_scroll_set 100 150 50 trace_scr);
      ScrollD(Vertical, 817, 819, 0, my_scroll_set 817 819 0 trace_scr);
      ScrollD(Vertical, 0, 100, 0, fun (wid,but,val) -> scroll_set(wid,val));
      ScrollD(Vertical, 0, 100, 0, fun (wid,but,val) ->
        scroll_set(wid,val+100)
      );
      PackD(Vertical, [
        PackD(Horizontal, [
          SelectA [NameAtt "Menu"] [
            PopupD("translate", fun bwid ->
              let xll = xll xd bwid
              and yll = yll xd bwid in
              rt_map_popup_widget(translate_wid, xll, yll, 0)
            );
            PopupD("gag", fun bwid ->
              let xll = xll xd bwid
              and yll = yll xd bwid in
              rt_map_popup_widget(gag_wid, xll, yll, 0)
            )
          ];
          PopupD("machines", fun bwid ->
            let xll = xll xd bwid
            and yll = yll xd bwid in
            rt_map_popup_widget(machines_wid, xll, yll, 0)
          );
          PopupD("bouton", fun bwid ->
            let xll = xll xd bwid
            and yll = yll xd bwid in
            rt_map_popup_widget(button_wid, xll, yll, 0)
          )
        ]);
        TextA [NameAtt "T"; FillerAtt] (
          1, 15, 0, keyp_fun, fun(wid,but,x,y) -> text_goto(wid,x,y)
        )
      ]);
      PackD(Vertical, [
        ButtonD("salut", fun _ ->
          let wid = wid_named "test" in
          rt_resize_widget(
            wid, widget_width wid + 10, widget_height wid + 10
          )
        );
        ScrollD(Horizontal, -3, 10, 3, my_scroll_set 0 10 0 trace_scr);
        PackA [FillerAtt] (Vertical, []);
        ButtonD("hello", fun _ ->
          text_send_string(wid_named "T", "ca marche\n")
        )
      ]);
      ButtonD("quit", quit_fun wid_named)
    ])
  ) in
  let twid = wid_named "st" in
  text_shift(twid, NSAV-NVIS);
  text_home twid;
  scroll_set(wid_named "scr", NSAV);

  rt_create_transient_widget(wid, "trans",
    PackA [NameAtt "trans"] (Horizontal, [
      TextA [FillerAtt] (
        2, 80, 0, text_send_string, my_text_button xd
      );
      ButtonD("OK", fun _ -> rt_unmap_widget (wid_named "trans"));
      ButtonD("Abort", fun _ ->
        rt_unmap_widget (wid_named "trans")
      )
    ])
  );
  rt_create_subwidget(wid, 0, 0,
    PackA [NameAtt "question"] (Vertical, [
      TitleD(" Are you sure ? ");
      PackD(Horizontal, [
        ButtonD("yes", answer xa wid_named true);
        ButtonD("cancel", answer xa wid_named false)
      ])
    ])
  );

  wid
;;

let test dname =

  let xd = rt_initialize dname in (try
    let xa = rt_args[xd] in

    let translate_wid = translate_wid xd
    and gag_wid = gag_wid xd
    and button_wid = button_wid xd
    and machines_wid = machines_wid xd
      (service_wid xd)
      (sun_wid xd)
      (workstations_wid xd
        (algo_wid xd)
        (croap_wid xd)
        (formel_wid xd)
        (para_wid xd)
        (ens_wid xd))
    in

    rt_select_file(xa, 0, stdin_fun);

    rt_create_located_widget(xd, "pos", "pos",
      UserPosition(40, 20),
      ButtonA [NameAtt "pos"] ("pos", rt_unmap_widget)
    );
    rt_create_located_widget(xd, "auto", "auto",
      AutoPosition,
      ButtonA [NameAtt "auto"] ("auto", rt_unmap_widget)
    );

    let wid = make_test_widget xa xd
      translate_wid gag_wid machines_wid button_wid in
    rt_map_widget wid;

    rt_main_loop xa
  with reraise ->
    rt_end xd; raise reraise);
  rt_end xd
;;

end module with
  value test;
  value COLOR;
;
