(* Dynamic values and objects for the Modules (Defn. V4 p57). *)

(*
$File: $
$Date: 1993/02/24 07:59:52 $
$Revision: 1.3 $
$Locker: birkedal $
*)

(*$ModuleDynObject:
	TOPDEC_GRAMMAR PPTOPDECGRAMMAR STRID SIGID FUNID VAR EXCON
	CORE_DYNOBJECT FINMAP FLAGS PRETTYPRINT REPORT CRASH
	MODULE_DYNOBJECT
 *)

functor ModuleDynObject(structure TopdecGrammar: TOPDEC_GRAMMAR

			structure PPTopdecGrammar: PPTOPDECGRAMMAR
			  sharing PPTopdecGrammar.G = TopdecGrammar

			structure StrId: STRID
			structure SigId: SIGID
			structure FunId: FUNID
			structure Var: VAR
			structure Excon: EXCON

			structure CoreDynObject: CORE_DYNOBJECT
			  sharing CoreDynObject.Var = Var
			      and CoreDynObject.Excon = Excon
			      and type CoreDynObject.strid = StrId.strid
			      and type CoreDynObject.longstrid = StrId.longstrid

			structure FinMap: FINMAP
			structure Flags: FLAGS

			structure PP: PRETTYPRINT
			  sharing type PPTopdecGrammar.StringTree
			    	       = CoreDynObject.StringTree
				       = FinMap.StringTree
				       = PP.StringTree

			structure Report: REPORT
			  sharing type PP.Report = Report.Report

			structure Crash: CRASH
		       ): MODULE_DYNOBJECT =
  struct
    type strid = StrId.strid
    type longstrid = StrId.longstrid
    type sigid = SigId.sigid
    type funid = FunId.funid
    type strexp = TopdecGrammar.strexp
    type var = Var.var
    type excon = Excon.excon
    type Env = CoreDynObject.Env
    type StrEnv = CoreDynObject.StrEnv

    type ('a, 'b) map = ('a, 'b) FinMap.map

    infix plus
    val op plus = FinMap.plus

    infix //
    val op // = Report.//

    fun debug(title, f: unit -> PP.StringTree) =
      if Flags.DEBUG_EVALTOPDEC
      then Report.print(Report.line title // PP.reportStringTree(f()))
      else ()

    fun mustFind fname (m, x) =
      case FinMap.lookup m x
	of Some y => y
	 | None => Crash.impossible("ModuleDynObject." ^ fname)

    datatype FunctorClosure =
    	       FUNCTOR_CLOSURE of (strid * Int) * (strexp * Int Option) * Basis

	 and Int = INT of IntEnv * var EqSet.Set * excon EqSet.Set
	 and IntEnv = INTENV of (strid, Int) map
	 and SigEnv = SIGENV of (sigid, Int) map
    	 and FunEnv = FUNENV of (funid, FunctorClosure) map
	 and Basis = BASIS of FunEnv * SigEnv * Env
	 and IntBasis = INTBASIS of SigEnv * IntEnv


   (* Death by prettyprinting... *)

    type StringTree = PP.StringTree

    fun layoutIntEnv(INTENV m) =
      FinMap.layoutMap {start="(IE:", eq=" -> ", sep=", ", finish=")"}
		       (PP.layoutAtom StrId.pr_StrId)
		       layoutInt
		       m

    and layoutInt(INT(IE, vars, excons)) =
      let
	val ieT = layoutIntEnv IE
	val varsT = PP.layoutSet (PP.layoutAtom Var.pr_var) vars
	val exconsT = PP.layoutSet (PP.layoutAtom Excon.pr_excon) excons
      in
	PP.NODE{start="(INT:", finish=")", indent=3, childsep=PP.RIGHT ", ",
		children=[ieT, varsT, exconsT]
	       }
      end

    fun layoutBasis(BASIS(F, G, E)) =
      let
	fun layoutFunctorClosure(FUNCTOR_CLOSURE((strid, int),
						 (strexp, int_opt),
						 B
						)
				) =
	  let
	    val t1 = PP.NODE{start="(StrId:", finish=")", indent=3,
			     children=[PP.layoutAtom StrId.pr_StrId strid,
				       layoutInt int
				      ],
			     childsep=PP.LEFT " -> "
			    }
	  
	    val t2 = PP.NODE{start="(StrExp:", finish=")", indent=3,
			     children=[PPTopdecGrammar.layoutStrexp strexp,
			     	       case int_opt
					 of Some int => layoutInt int
				          | None => PP.LEAF "[NoConstraint]"
				      ],
			     childsep=PP.LEFT ": "
			    }
	  in
	    PP.NODE{start="(FunClos:", finish=")", indent=3,
		    children=[t1, t2, layoutBasis B],
		    childsep=PP.RIGHT ", "
		   }
	  end

	fun layoutFunEnv(FUNENV F) =
	   FinMap.layoutMap {start="(F:", eq=" -> ", sep=", ", finish=")"}
			    (PP.layoutAtom FunId.pr_FunId)
			    layoutFunctorClosure
			    F

	fun layoutSigEnv(SIGENV G) =
	   FinMap.layoutMap {start="(G:", eq=" -> ", sep=", ", finish=")"}
			    (PP.layoutAtom SigId.pr_SigId)
			    layoutInt
			    G
      in
	PP.NODE{start="Dynamic Basis:", finish="", indent=3,
		childsep=PP.RIGHT "; ",
		children=[layoutFunEnv F, layoutSigEnv G,
			  CoreDynObject.layoutEnv E
			 ]
	       }
      end


   (* Death by 1000 functions... *)

    val emptyIE = INTENV FinMap.empty

    val singleIE = INTENV o FinMap.singleton

    fun IE_plus_IE(INTENV m1, INTENV m2) = INTENV(m1 plus m2)

    fun InterE E: Int =
      let
	val SE = CoreDynObject.SE_of_E E
	val VE = CoreDynObject.VE_of_E E
	val EE = CoreDynObject.EE_of_E E

	fun iter(strid :: rest) =
	      let
		val E' = CoreDynObject.lookup_StrId(E, strid)
	      in
		IE_plus_IE(singleIE(strid, InterE E'), iter rest)
	      end

	  | iter nil = emptyIE
      in
	INT(iter(EqSet.list(CoreDynObject.domSE SE)),
	    CoreDynObject.domVE VE,
	    CoreDynObject.domEE EE
	   )
      end

    fun IE_in_Int IE = INT(IE, EqSet.empty, EqSet.empty)

    fun IE_of_Int(INT(IE, _, _)) = IE

    fun InterB(BASIS(_, G, E)) = INTBASIS(G, IE_of_Int(InterE E))

    fun strId_in_IE(strid, INTENV m) =
      (debug("Looking in: ",
	     fn () => FinMap.layoutMap
		      {start="{", eq=" -> ", sep="; ", finish="}"}
		      (PP.layoutAtom StrId.pr_StrId)
		      layoutInt
		      m
	    );
       case FinMap.lookup m strid of Some _ => true
				   | None => false
      )

    fun lookup_StrId_IE(INTENV m, strid): Int =
      mustFind "lookup_StrId_IE" (m, strid)

    fun Cut(E, INT(IE, vars, excons)): Env =
      let
	val SE = CoreDynObject.SE_of_E E
	val VE = CoreDynObject.VE_of_E E
	val EE = CoreDynObject.EE_of_E E

	val _ = debug("Cut.IE: ", fn () => layoutIntEnv IE);
	val _ = debug("Cut.SE: ", fn () => CoreDynObject.layoutStrEnv SE);

       (* Run through SE, discarding strid's not in IE. For those strid's
	  in IE (giving I), map strid to `SE(strid) Cut I'. *)

	val SE1 = CoreDynObject.trimSE(SE, fn s => strId_in_IE(s, IE))

	val _ = debug("Cut.SE1: ", fn () => CoreDynObject.layoutStrEnv SE1);

	val SE' = CoreDynObject.mapSE(
		    SE1, fn (strid, E) => Cut(E, lookup_StrId_IE(IE, strid))
		  )

	val _ = debug("Cut.SE': ", fn () => CoreDynObject.layoutStrEnv SE');

	val VE' = CoreDynObject.trimVE(VE, fn v => EqSet.member v vars)
	  
	val EE' = CoreDynObject.trimEE(EE, fn e => EqSet.member e excons)

	val _ = debug("Cut.E: ", 
		      fn () => CoreDynObject.layoutEnv 
		               (CoreDynObject.mkEnv(SE', VE', EE')));

      in
	CoreDynObject.mkEnv(SE', VE', EE')
      end

    fun Vars_in_Int v = INT(emptyIE, v, EqSet.empty)

    fun Excons_in_Int e = INT(emptyIE, EqSet.empty, e)

    val emptyInt = INT(emptyIE, EqSet.empty, EqSet.empty)

    fun Int_plus_Int(INT(INTENV m1, v1, e1), INT(INTENV m2, v2, e2)) =
      INT(INTENV(m1 plus m2), EqSet.union v1 v2, EqSet.union e1 e2)

    val emptyG = SIGENV FinMap.empty

    val singleG = SIGENV o FinMap.singleton

    fun G_plus_G(SIGENV m1, SIGENV m2) = SIGENV(m1 plus m2)

    fun G_of_IB(INTBASIS(G, _)) = G

    fun G_in_IB G = INTBASIS(G, emptyIE)

    val emptyF = FUNENV FinMap.empty

    fun G_in_B G = BASIS(emptyF, G, CoreDynObject.emptyE)

    fun IB_plus_G(INTBASIS(SIGENV m1, IE), SIGENV m2) =
      INTBASIS(SIGENV(m1 plus m2), IE)

    fun IB_plus_IE(INTBASIS(G1, INTENV m1), INTENV m2) =
      INTBASIS(G1, INTENV(m1 plus m2))

    val singleF = FUNENV o FinMap.singleton

    fun F_plus_F(FUNENV m1, FUNENV m2) = FUNENV(m1 plus m2)

    fun F_in_B F = BASIS(F, emptyG, CoreDynObject.emptyE)

    fun F_of_B(BASIS(F, _, _)) = F

    fun B_plus_F(BASIS(FUNENV m1, G, E), FUNENV m2) =
      BASIS(FUNENV(m1 plus m2), G, E)

    fun E_in_B E = BASIS(FUNENV FinMap.empty, SIGENV FinMap.empty, E)

    fun E_of_B(BASIS(_, _, E)) = E

    fun B_plus_E(BASIS(F, G, E1), E2) =
      BASIS(F, G, CoreDynObject.E_plus_E(E1, E2))

    fun B_plus_B(BASIS(FUNENV F1, SIGENV G1, E1),
		 BASIS(FUNENV F2, SIGENV G2, E2)
		) =
      BASIS(FUNENV(F1 plus F2),
	    SIGENV(G1 plus G2),
	    CoreDynObject.E_plus_E(E1, E2)
	   )

    val emptyBasis = E_in_B CoreDynObject.emptyE

    val initialBasis = E_in_B CoreDynObject.initialE

    fun lookup_SigId(SIGENV m, sigid) = mustFind "lookup_SigId" (m, sigid)

    fun lookup_LongStrId_B(BASIS(_, _, E), longstrid) =
      CoreDynObject.lookup_LongStrId(E, longstrid)

   (* lookup_LongStrId_IB is a wee bit of a hassle since we have to do the
      longid decomposition ourselves. (For the other lookups we are relying
      on CoreDynObject to do all this stuff.) *)

    fun lookup_LongStrId_IB(INTBASIS(_, INTENV m), longstrid): Int =
      let
	val (strids, strid) = StrId.explode_longstrid longstrid

	fun iter(m, this :: rest): Int =
	      let
		val Int = mustFind "lookup_LongStrId_IB/1" (m, this)
		val INTENV m' = IE_of_Int Int
	      in
		iter(m', rest)
	      end

	  | iter(m, nil) = mustFind "lookup_LongStrId_IB/2" (m, strid)
      in
	iter(m, strids)
      end

    fun lookup_FunId(FUNENV m, funid) = mustFind "lookup_FunId" (m, funid)

    fun B_plus_SE(BASIS(F, G, E), SE) =
      BASIS(F, G, CoreDynObject.E_plus_SE(E, SE))

    val mkClosure = FUNCTOR_CLOSURE

    fun unClosure(FUNCTOR_CLOSURE x) = x

    type Report = Report.Report
    fun reportBasis _ = Report.line "<ModuleDynObject.basis>"
  end;
