(*
$File: Compiler/MatchCompiler.sml $
$Date: 1992/09/17 14:17:59 $
$Revision: 1.1 $
$Locker:  $
*)

(*$MatchCompiler:
	VAR CON SCON EXCON RESIDENT LAB DEC_GRAMMAR PPDECGRAMMAR
	COMPILER_ENV GRAMMAR_INFO LVARS TYPE_INFO FINMAP
	BASIC_IO REPORT FLAGS PRETTYPRINT CRASH MATCH_COMPILER
	AndOrTree DecisionList DecisionTree PatBindings MatchWarnings
 *)

(* MatchCompiler refers to the specialised match compiler functors
   explicitly, so that they aren't visible to the rest of the
   compiler at large. I don't generally like such non-local functor
   references, but it's worth it in this case to achieve modularity.
   In any case, it looks rather impressive. *)

functor MatchCompiler(structure Var: VAR
		      structure Con: CON
		      structure SCon: SCON
		      structure Lab: LAB
		      structure Excon: EXCON

		      structure ResIdent: RESIDENT
		        sharing type ResIdent.longvar = Var.longvar
			    and type ResIdent.longcon = Con.longcon
			    and type ResIdent.longexcon = Excon.longexcon

		      structure Grammar: DEC_GRAMMAR
		        sharing type Grammar.longid = ResIdent.longid
			    and type Grammar.con = Con.con
			    and type Grammar.scon = SCon.scon
			    and type Grammar.id = Var.var
			    and type Grammar.lab = Lab.lab

		      structure PPDecGrammar: PPDECGRAMMAR
		        sharing PPDecGrammar.G = Grammar

		      structure CompilerEnv: COMPILER_ENV
			sharing type CompilerEnv.var = Var.var

		      structure GrammarInfo: GRAMMAR_INFO
			sharing type Grammar.info
				     = GrammarInfo.PostElabGrammarInfo

		      structure Lvars: LVARS
			sharing type CompilerEnv.lvar = Lvars.lvar

		      structure TypeInfo: TYPE_INFO
			sharing type GrammarInfo.TypeInfo = TypeInfo.info

		      structure FinMap: FINMAP
		      structure BasicIO: BASIC_IO
		      structure Report: REPORT
		      structure Flags: FLAGS

		      structure PP: PRETTYPRINT
			sharing type FinMap.StringTree = PPDecGrammar.StringTree
			  	     = CompilerEnv.StringTree
				     = GrammarInfo.StringTree
				     = PP.StringTree
			    and type PP.Report = Report.Report

		      structure Crash: CRASH
		     ): MATCH_COMPILER =
  struct
    structure AndOrTree =
      AndOrTree(structure Var = Var
		structure Con = Con
		structure SCon = SCon
		structure Excon = Excon
		structure Lab = Lab
		structure Grammar = Grammar
		structure R = ResIdent
		structure GrammarInfo = GrammarInfo
		structure FinMap = FinMap
		structure PP = PP
		structure Crash = Crash
	       )

    structure DecisionList =
      DecisionList(structure Lab = Lab
		   structure Con = Con
		   structure SCon = SCon
		   structure Excon = Excon
		   structure AndOrTree = AndOrTree
		   structure FinMap = FinMap
		   structure PP = PP
		   structure Crash = Crash
		  )

    structure DecisionTree =
      DecisionTree(structure Lab = Lab
		   structure Var = Var
		   structure ResIdent = ResIdent
		   structure Con = Con
		   structure SCon = SCon
		   structure Excon = Excon
		   structure Grammar = Grammar
		   structure CompilerEnv = CompilerEnv
		   structure DecisionList = DecisionList

		   structure PatBindings =
		     PatBindings(structure Lab = Lab
				 structure Var = Var
				 structure ResIdent = ResIdent
				 structure Grammar = Grammar
				 structure CompilerEnv = CompilerEnv
				 structure GrammarInfo = GrammarInfo
				 structure Lvars = Lvars
				 structure PP = PP
				 structure FinMap = FinMap
				 structure Crash = Crash
				)

		   structure Lvars = Lvars
		   structure TypeInfo = TypeInfo
		 (*val nj_sml_bug = (fn x => x)*)
		   structure FinMap = FinMap
		   structure BasicIO = BasicIO
		   structure Report = Report
		   structure Flags = Flags
		   structure PP = PP
		   structure Crash = Crash
		  )

    structure MatchWarnings =
      MatchWarnings(structure Var = Var
		    structure Con = Con
		    structure SCon = SCon
		    structure CompilerEnv = CompilerEnv
		    structure DecTreeDT = DecisionTree
		    structure FinMap = FinMap
		   )

    open DecisionTree		(* Export the DecisionTree datatype, as well as
				   StringTree and layout declarations. *)

   (* Some type abbreviations to match up to the MATCH_COMPILER signature. *)
    type pat = Grammar.pat
    type lvar = Lvars.lvar
    type longcon = Con.longcon
    type scon = SCon.scon
    type longexcon = Excon.longexcon
    type lab = Lab.lab
    type longid = ResIdent.longid
    type (''a, 'b) map = (''a, 'b) FinMap.map
    type RuleNum = int

    val pr = Report.print o PP.reportStringTree

    infix //
    val op // = Report.//

   (* reporting of unreachable/inexhaustive matches. I'm not sure if the
      depths of the compiler should have any print statements at all; they
      should probably be passed out symbolically to the outermost level. *)

    fun report(msg, pats, drawArrow) =
      let
	val layoutPat = PPDecGrammar.layoutPat

	fun pr(i, pat :: rest) =
	        (Report.line((if drawArrow i then "  =--> "
					     else "       "
			     ) ^ PP.flatten(PP.format(70, layoutPat pat))
			       ^ " => ..."
			    )
		 // pr(i+1, rest)
	        )
	  | pr(_, nil) = Report.null
      in
	Report.line msg // pr(1, pats)
      end

    fun issueWarnings(pats, tree, {warnInexhaustive, warnNoBindings}) =
      let
	open MatchWarnings

	fun downfrom' 0 = nil
	  | downfrom' n = n :: downfrom'(n-1)

	val downfrom = EqSet.fromList o downfrom'

	val numRules = List.size pats
	val allRules = downfrom numRules
	val unreachables = EqSet.difference allRules (reachable tree)
      in
	case EqSet.isEmpty unreachables
	  of true => ()			(* Nothing to report *)
	   | _ => Report.print(report("Some rules are unreachable",
				      pats, fn i => EqSet.member i unreachables
				     )
			      );

	if warnNoBindings andalso not(binds tree) then
	  Report.print(report("Binding declares no variables",
			      pats, fn _ => false
			     )
		      )
	else ();

	if warnInexhaustive andalso not(exhaustive tree) then
	  Report.print(report("Match is not exhaustive", pats, fn _ => false))
	else ()
      end
	      

    fun matchCompiler(_, nil, _) = Crash.impossible "matchCompiler(nil)"
      | matchCompiler(root: lvar, pats: pat list,
		      flags: {warnInexhaustive: bool, warnNoBindings: bool}
		     ): DecisionTree =
	  let
	    val andOrTree = AndOrTree.buildAndOrTree pats

	    val _ =
	      if Flags.DEBUG_MATCHCOMPILER then
		pr(PP.NODE{start="And/Or tree: ", finish="", indent=3,
			   children=[AndOrTree.layoutAndOrTree andOrTree],
			   childsep=PP.NONE
			  }
		)
	      else ()

	    val decisions = DecisionList.decisions andOrTree

	    val _ =
	      if Flags.DEBUG_MATCHCOMPILER then
		pr(PP.NODE{start="Decision list: ", finish="", indent=3,
			   children=map DecisionList.layoutDecision decisions,
			   childsep=PP.RIGHT ", "
			  }
		)
	      else ()

	    val decTree =
	      DecisionTree.decisionTree
	        {pats=pats, root=root, decisions=decisions}
	    
	    val _ =
	      if Flags.DEBUG_MATCHCOMPILER then
		pr(PP.NODE{start="Decision tree(" ^ Lvars.pr_lvar root ^ "): ",
			   finish="", indent=3, childsep=PP.NONE,
			   children=[DecisionTree.layoutDecisionTree decTree]
			  }
		)
	      else ()
	  in
	    issueWarnings(pats, decTree, flags);
	    decTree
	  end
  end;
