/*

Simple minded remote file system interconnect.
This is the type manager.

Written by Jim Rees, Apollo Computer, 1985.

*/

#include <fcntl.h>
#include <sys/types.h>
#include <sys/errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include "/sys/ins/base.ins.c"
#include "/sys/ins/ios.ins.c"
#include "/sys/ins/io_traits.ins.c"
#include "/sys/ins/ms.ins.c"
#include "/sys/ins/rws.ins.c"
#include "/sys/ins/type_uids.ins.c"
#include <apollo/pgm.h>

#define CACHE 6		/* Number of connections to cache */

#define	unix_errno_$prefix	0x090C0000

#include "v2.h"
#include "v2d.h"

static struct errtab {
	int errno;
	int stcode;
} errtab[] = {
	ENOENT,		ios_$name_not_found,
	EPERM,		ios_$insufficient_rights,
	EBADF,		ios_$id_oor,
	ENOMEM,		ios_$insuff_memory,
	EACCES,		ios_$insufficient_rights,
	ENOTBLK,	ios_$illegal_obj_type,
	EBUSY,		ios_$resource_lock_err,
	EEXIST,		ios_$already_exists,
	EXDEV,		ios_$illegal_name_redefine,
	ENOTDIR,	ios_$illegal_obj_type,
	EISDIR,		ios_$illegal_obj_type,
	EINVAL,		ios_$illegal_operation,
	ENFILE,		ios_$no_table_space,
	EMFILE,		ios_$no_more_streams,
	ETXTBSY,	ios_$concurrency_violation,
	ENOTEMPTY,	ios_$file_not_empty,
	66,		ios_$file_not_empty,
	-1,		ios_$illegal_operation,
};

v2_$openx(xoidp, resid, resid_lenp, ooptsp, hpp, tuidp, stp)
xoid_$t *xoidp;
char *resid;
short *resid_lenp;
ios_$open_options_t *ooptsp;
v2_handle_t **hpp;
uid_$t *tuidp;
status_$t *stp;
{
	v2_open(xoidp, resid, *resid_lenp, ios_$preserve_mode, *ooptsp, &uid_$nil, hpp, tuidp, stp);
}

v2_$create(xoidp, resid, resid_lenp, cmodep, ooptsp, iuidp, hpp, ouidp, stp)
xoid_$t *xoidp;
char *resid;
short *resid_lenp;
ios_$create_mode_t *cmodep;
ios_$open_options_t *ooptsp;
uid_$t *iuidp, *ouidp;
v2_handle_t **hpp;
status_$t *stp;
{
	if (*iuidp == directory_$uid) {
		v2_dir_create(xoidp, resid, *resid_lenp, hpp, stp);
		*ouidp = v2_$uid;
	} else
		v2_open(xoidp, resid, *resid_lenp, *cmodep, (*ooptsp | ios_$create_opt),
			iuidp, hpp, ouidp, stp);
}

v2_open(xoidp, resid, resid_len, cmode, oopts, iuidp, hpp, ouidp, stp)
xoid_$t *xoidp;
char *resid;
short resid_len;
ios_$create_mode_t cmode;
ios_$open_options_t oopts;
uid_$t *iuidp, *ouidp;
v2_handle_t **hpp;
status_$t *stp;
{
	v2_handle_t *hp;
	char *name, nbuf[256];
	short namelen;
	long flag;
	int type;
	status_$t xst;

	*hpp = NULL;

	handle_init(&hp, resid, resid_len, stp);
	if (stp->all)
		return;
	name = hp->name;
	namelen = strlen(name);
	hp->xoid = *xoidp;

	if (oopts & (ios_$write_opt | ios_$read_intend_write_opt)) {
		flag = O_RDWR;
		hp->read_only = false;
	} else {
		flag = O_RDONLY;
		hp->read_only = true;
	}

	if (oopts & ios_$position_to_eof_opt) {
		flag |= XO_APPEND;
		hp->append_mode = true;
	} else
		hp->append_mode = false;

	if (oopts & ios_$create_opt)
		flag |= XO_CREAT;

	switch (cmode) {
	case ios_$no_pre_exist_mode:
		flag |= XO_EXCL;
		break;
	case ios_$preserve_mode:
		break;
	case ios_$recreate_mode:
	case ios_$truncate_mode:
		flag |= XO_TRUNC;
		break;
	case ios_$make_backup_mode:
		flag |= XO_TRUNC;
		/* Gen up a temp file name */
		strcpy(nbuf, name);
		nbuf[namelen++] = '~';
		name = nbuf;
		hp->bak_mode = true;
		break;
	default:
		rws_$release_heap_pool(hp, rws_$stream_tm_pool, xst);
		stp->all = ios_$illegal_param_comb;
		return;
	}

	if ((hp->sock_fd = v2_sconnect(xoidp, stp)) < 0) {
		rws_$release_heap_pool(hp, rws_$stream_tm_pool, xst);
		return;
	}

	if (oopts & ios_$inquire_only_opt) {
		putlong(C_STAT);
		putlong(namelen);
		putbytes(name, namelen);
		if ((type = getlong()) < 0) {
			v2_sclose(hp);
			stp->all = v2_tr_errno(type);
			return;
		}
		if (read(hp->sock_fd, &(hp->a_cache), sizeof (struct attr)) < 0) {
			v2_sclose(hp);
			stp->all = ios_$name_not_found;
			return;
		}
		hp->inq_only = true;
		hp->a_cache_valid = true;
		hp->cant_write = true;
		hp->type = (hp->a_cache.mode & A_IFDIR) ? v2_$uid : v2f_$uid;
		*ouidp = hp->type;
		stp->all = status_$ok;
		*hpp = hp;
		return;
	}

	putlong(C_OPEN);
	putlong(namelen);
	putbytes(name, namelen);
	putlong(flag);
	if ((type = getlong()) < 0) {
		v2_sclose(hp);
		stp->all = v2_tr_errno(type);
		return;
	}
	hp->cant_write = ((type & T_RW) == T_RO);
	hp->type = ((type & T_TYPE) == T_FILE) ? v2f_$uid : v2_$uid;
	*ouidp = hp->type;
	stp->all = status_$ok;
	*hpp = hp;
}

static
v2_dir_create(xoidp, name, name_len, hpp, stp)
xoid_$t *xoidp;
char *name;
short name_len;
v2_handle_t **hpp;
status_$t *stp;
{
	v2_handle_t *hp;
	int type;
	status_$t xst;

	*hpp = NULL;

	handle_init(&hp, name, name_len, stp);
	if (stp->all)
		return;
	hp->xoid = *xoidp;
	hp->read_only = true;

	if ((hp->sock_fd = v2_sconnect(xoidp, stp)) < 0) {
		rws_$release_heap_pool(hp, rws_$stream_tm_pool, xst);
		return;
	}

	putlong(C_MKDIR);
	putlong(name_len);
	putbytes(name, name_len);
	if ((type = getlong()) < 0) {
		v2_sclose(hp);
		stp->all = v2_tr_errno(type);
		return;
	}
	hp->cant_write = true;
	hp->type = v2_$uid;
	stp->all = status_$ok;
	*hpp = hp;
}

v2_tr_errno(e)
int e;
{
	extern status_$t errno_$status;
	struct errtab *etp;

	if (e >= 0)
		return status_$ok;

	if (e == -1) {
		if (errno_$status.all != status_$ok)
			return errno_$status.all;
		if (errno != 0)
			return (unix_errno_$prefix | errno);
		return ios_$object_not_found;
	}

	e = -1 - e;

	for (etp = errtab; etp->errno > 0; etp++)
		if (e == etp->errno)
			break;
	if (etp->errno < 0)
		return (unix_errno_$prefix | e);
	return etp->stcode;
}

static
handle_init(hpp, name, name_len, stp)
v2_handle_t **hpp;
char *name;
short name_len;
status_$t *stp;
{
	v2_handle_t *hp;
	char tname[200], *cp, *comp[20];
	int i, n;
	boolean start;

	hp = (v2_handle_t *) rws_$alloc_heap_pool(rws_$stream_tm_pool, (long) sizeof (v2_handle_t));
	if (hp == NULL) {
		stp->all = ios_$insuff_memory;
		return;
	}
	bzero(hp, sizeof *hp);
	hp->sock_fd = -1;
	hp->unreg = true;
	hp->delete = false;
	hp->bak_mode = false;
	hp->a_cache_valid = false;
	hp->seek_key = 0;
	hp->seek_key_valid = true;

	/* The file name may contain ../ or ./, so we break it into the
	   constituant parts, put them on a stack, and canonicalize.
	   Note this doesn't help link text.  */

	/* Put on stack */
	strncpy(tname, name, name_len);
	strcpy(&tname[name_len], "/");
	start = true;
	n = 0;
	for (cp = tname; *cp; cp++) {
		if (start && *cp != '/') {
			start = false;
			if (n > 0 && !strncmp(cp, "../", 3))
				n--;
			else if (strncmp(cp, "./", 2))
				comp[n++] = cp;
		} else if (!start && *cp == '/') {
			start = true;
			*cp = '\0';
		}
	}

	/* Reconstruct from stack */
	hp->name[0] = '\0';
	for (i = 0; i < n; i++) {
		if (i != 0)
			strcat(hp->name, "/");
		strcat(hp->name, comp[i]);
	}

	/* Fix up null name */
	if (hp->name[0] == '\0')
		strcpy(hp->name, ".");

	*hpp = hp;
	stp->all = status_$ok;
}

v2_$change_name(hpp, name, name_lenp, stp)
v2_handle_t **hpp;
char *name;
short *name_lenp;
status_$t *stp;
{
	v2_handle_t *hp;
	int n;

	stp->all = status_$ok;
	hp = *hpp;

	putlong(C_CHN);
	n = strlen(hp->name);
	putlong(n);
	putbytes(hp->name, n);
	putlong(*name_lenp);
	putbytes(name, *name_lenp);
	if ((n = getlong()) != 0) {
		stp->all = v2_tr_errno(n);
		return;
	}
	strncpy(hp->name, name, *name_lenp);
	hp->name[*name_lenp] = '\0';
}

v2_$inq_name(hpp, typep, name, nlp, stp)
v2_handle_t **hpp;
io_xoc_$name_type_t *typep;
char *name;
short *nlp;
status_$t *stp;
{
	v2_handle_t *hp;
	char *cp;
	extern char *rindex();

	hp = *hpp;

	if (*typep == io_xoc_$leaf_name) {
		cp = rindex(hp->name, '/');
		if (cp == NULL)
			cp = hp->name;
		else
			cp++;
	} else
		cp = hp->name;

	if (!strcmp(cp, "."))
		*nlp = 0;
	else {
		*nlp = strlen(cp);
		strncpy(name, cp, *nlp);
	}
	stp->all = status_$ok;
}

boolean
v2_$close(hpp, stp)
v2_handle_t **hpp;
status_$t *stp;
{
	v2_handle_t *hp;
	int n, ok;
	status_$t stx;

	stp->all = status_$ok;
	hp = *hpp;

	if (hp == NULL)
		return false;
	if (hp->delete) {
		putlong(C_UNLINK);
		n = strlen(hp->name);
		putlong(n);
		putbytes(hp->name, n);
		if ((n = getlong()) < 0)
			stp->all = v2_tr_errno(n);
	} else if (hp->bak_mode)
		make_bak(hp, stp);

	if (hp->sock_fd >= 0)
		putlong(C_CLOSE);
	v2_sclose(hp);
	return false;
}

static
make_bak(hp, stp)
v2_handle_t *hp;
status_$t *stp;
{
	int n;
	char buf[256];

	/* Delete the old .bak file */
	strcpy(buf, hp->name);
	n = strlen(buf);
	strcpy(&buf[n], ".bak");
	putlong(C_UNLINK);
	putlong(n + 4);
	putbytes(buf, n + 4);
	getlong();

	/* Rename the old file to .bak */
	putlong(C_CHN);
	putlong(n);
	putbytes(buf, n);
	putlong(n + 4);
	putbytes(buf, n + 4);
	getlong();

	/* Rename the temp file to the old file name */
	buf[n] = '~';
	putlong(C_CHN);
	putlong(n + 1);
	putbytes(buf, n + 1);
	putlong(n);
	putbytes(buf, n);
	return getlong();
}

#ifdef CACHE
/* Connection cache */

static int cache_fd[CACHE];
static uid_$t cache_uid[CACHE];
#endif CACHE

v2_sconnect(xoidp, stp)
xoid_$t *xoidp;
status_$t *stp;
{
	struct sockaddr_in saddr;
	long len;
	char *p;
	ios_$id_t fd;
	int i;
	status_$t st;

#ifdef CACHE
	for (i = 0; i < CACHE; i++)
		if (cache_fd[i] > 0 && xoidp->uid == cache_uid[i]) {
			fd = cache_fd[i];
			cache_fd[i] = 0;
			/* Make sure it's good */
			len = sizeof saddr;
			if (getsockname(fd, &saddr, &len) >= 0)
				return fd;
		}
#endif CACHE

	p = ms_$mapl_stream(*xoidp, 0L, 1024L, ms_$nr_xor_1w, ms_$r, false, len, *stp);
	if (stp->all != status_$ok)
		return -1;
	bcopy(p, &saddr, sizeof saddr);
	ms_$unmap(p, 1024L, st);
	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		stp->all = v2_tr_errno(-1);
		return -1;
	}
	if (connect(fd, &saddr, sizeof saddr) < 0) {
		stp->all = v2_tr_errno(-1);
		close(fd);
		return -1;
	}
	fd = ios_$switch(fd, ios_$max, *stp);
	if (stp->all != status_$ok)
		return -1;
	v2_putlong(fd, mygeteuid());
	return fd;
}

v2_sclose(hp)
v2_handle_t *hp;
{
	register i;
	static int is_dm;
	short narg;
	pgm_$argv_ptr argvp;
	status_$t st;

	if (hp->sock_fd >= 0) {
#ifdef CACHE
		if (is_dm == 0) {
			pgm_$get_args(&narg, &argvp);
			is_dm = (!strncmp(argvp[0]->chars, "dm", 2) ? 1 : -1);
		}

		if (is_dm != 1) {
			/* If there's room, cache this connection.  */
			for (i = 0; i < CACHE; i++)
				if (cache_fd[i] == 0) {
					cache_fd[i] = hp->sock_fd;
					cache_uid[i] = hp->xoid.uid;
					break;
				}
			if (i >= CACHE)
				/* No room; close it */
				close(hp->sock_fd);
		} else
			close(hp->sock_fd);
#else
		close(hp->sock_fd);
#endif CACHE
		hp->sock_fd = -1;
	}
	rws_$release_heap_pool(hp, rws_$stream_tm_pool, st);
}

/* version of geteuid that doesn't require /etc/passwd, and maps to
   remote system uids.  */

struct umap {
	char *user;
	unsigned short uid;
} umap[] = {
	"rees", 395,
	"ers", 11,
	"nazgul", 72,
	"jah", 190,
	"swin", 2652,
	"lwa", 6095,
	"wyant", 127,
	"jabs", 187,
	"joelm", 6457,
	"alm", 2845,
	NULL, -2
};

static
mygeteuid()
{
	char *user;
	extern char *getenv();
	struct umap *up;

	user = getenv("LOGNAME");
	if (user == NULL)
		return -2;
	for (up = &umap[0]; up->user != NULL; up++)
		if (!strcmp(up->user, user))
			break;
	if (up->user == NULL)
		return geteuid();
	return up->uid;
}
