/* driver_generic.c

Known problems and bugs are marked with *zzzzz problem zzzzz*.
Now solved problems with prsvr/prmgr/prf are marked with yyyyy.
To add a new printer, look for xxxxx. */

/*--------------------------------------------------*/

/* Generic driver for any ASCII printer.
Supports several specific printers, or the really generic
dumb ASCII printer, which needs only understand the linefeed (LF)
character, but may optionally also use (any of) the formfeed (FF),
tab and backspace (BS) characters.

Specific printer models supported:
dumb_ascii
epson_lq800
epson_lq1000
epson_lq1500
epson_lx80
epson_fx80		(DIP switch 1-1 must be OFF)
epson_fx85		(Epson emulation mode)
epson_fx105		(Epson emulation mode)
apple_imagewriter
apple_imagewriter_II
HP_LaserJet		(No support for font cartridges;
HP_LaserJetPlus		only portrait mode for ASCII text.)
microline_84
microline_380		(Epson emulation mode)
microline_390		(Epson emulation mode)
microline_391		(Epson emulation mode)
NEC_pinwriter_p5
star_nx1000		(Epson emulation mode)
*xxxxx Add new printer here xxxxx*

Can be used through an SIO line (1,2,3),
or optionally through the SPE parallel PIO line.

Compatible with SR10.0, 10.1, 10.2 or 10.3. */

/*--------------------------------------------------*/

/* Has five modes (i.e. can handle five file types):

File type UASC_ (vanilla ascii):
Uses formatting as specified in prsvr config file and/or on the
prf command line. Handles tab, LF and FF characters in text,
any other control characters are printed as hex values.

File type OS_ (overstrike/bolded backspaced ascii):
Same as file type UASC_ but also handles BS characters in text.

File type FTN_ (fortran carriage control):
Same as file type UASC_ but will use first character in
each line for carriage control instead of printing it:
1 for new page,
0 for new line (thus making it double spaced),
+ for overprinting previous line,
anything else ignored.

File type BM1_ (single plane pixel oriented bitmap):
Graphics data. Using the filter_chain feature, file types
BMSC_ (gpr section oriented bitmap of from 1 to 8 bits/section),
BMP_ (gpr pixel oriented bitmap of from 1 to 8 bits/pixel) and
GMF_ (gmf file) will be converted to type BM1_, and thus are
printable without any user intervention.
Obviously, graphics is NOT available on the dumb_ascii printer.

File type TRANS_ (send anything):
Anything (including control characters) is sent to the printer,
no formatting attempted.
While this driver will attempt to send a printer initialization
string before and after each job, it is the user's responsibility
to ensure that the printer is not left in a non-operable mode by
sending some nasty control characters.
This mode can be disabled, for security reasons. (From SR10.2,
it is controlled by the -passwd option.) */


/*--------------------------------------------------*/


/* Define compilation parameters */


#define SR10_0    0
#define SR10_1    1
#define SR10_2    2
#define SR10_3    3
/*  Define the operating system release level: must be
    one of SR10_0, SR10_1, SR10_2 or SR10_3.
    While the running program could find out the version it is using,
    the include files are different in an incompatible way, so
    that we must know this at compile time. Besides, we really
    want the version of prsvr and prmgr, which might be different
    from the rest of the operating system.
    In particular, if you install patch m0147 from the M68K_9006 tape
    over SR10.1 or SR10.2, then you probably need to define SR_level as
    SR10_3. */
#define SR_level SR10_2

/*  DO NOT change the following checking lines,
    even if you change the value above. */
#if SR_level != SR10_0 && SR_level != SR10_1 && SR_level != SR10_2 && SR_level != SR10_3
    ERROR: Cannot compile: SR_level must be SR10_0, SR10_1, SR10_2 or SR10_3 .
#endif


/*  Define x_SPE_installed yes if the SPE parallel board and software are
    installed. If you define it yes without the software being available,
    you will not be able to compile this program. */
#define x_SPE_installed yes


/*  Define x_debug yes if you want all sorts of informative messages.
    When x_debug is defined yes, if you want any more you can also define:
    x_debug_db   yes for databases;
    x_debug_ctrl yes for general printer control;
    x_debug_file yes for reading/sending of each line of the file;
    x_debug_feed yes for sending of linefeed/formfeeds;
    x_debug_head yes for page header and footer;
    x_debug_line yes for setting of (sio or pio) line characteristics. */
#define x_debug      no
#define x_debug_db   no
#define x_debug_ctrl no
#define x_debug_file no
#define x_debug_feed no
#define x_debug_head no
#define x_debug_line no


/*  Define the buffer sizes bufsize, bigbufsize and graphsize, and
    also pinsize.
    Buffer sizes should be big enough, but the exact sizes are not
    important; could survive on 256, 256 and 64 respectively.
    For ASCII (and OS and FTN) files, the page size will be limited
    to bufsize lines long and bufsize characters each in the left
    margin and as printed characters in the line.
    If WRAP is on, spurious linefeeds will be inserted for lines
    longer than bigbufsize.
    Linelength for TRANS files is unlimited.
    Graphics images will be limited to graphsize pixels horizontally;
    vertical size is unlimited (up to the paper length).
    pinsize is the highest number of (vertical) pins in the printhead
    of any of these printers. */
#define bufsize 512
#define bigbufsize 12288
#define graphsize 4096
#define pinsize 24

/*  DO NOT change the following checking lines,
    even if you change the values above. */
#if bufsize < 256
    ERROR: Cannot compile: BUFSIZE must be at least 256.
#endif
#if bigbufsize < bufsize
    ERROR: Cannot compile: BIGBUFSIZE smaller than BUFSIZE.
#endif
#if graphsize < 64
    ERROR: Cannot compile: GRAPHSIZE smaller than 64.
#endif
#if bigbufsize < ((pinsize + 7) / 8) * graphsize
    ERROR: Cannot compile: BIGBUFSIZE smaller than 3 * GRAPHSIZE.
#endif
#if pinsize != 24
    ERROR: Cannot compile: PINSIZE not equal to 24.
#endif


/*--------------------------------------------------*/


/* No more features in need of change by user below this line */


/*--------------------------------------------------*/


/* How to use:


Compile and bind the driver.

Total run times
big:  DN10000, 32MB RAM (1 CPU)
c6123: DN4000, 24MB RAM
c717:  DN3500,  8MB RAM (diskless)
+-------+-------+-------+-------+---------+
|       |         Times         | Size of |
| Level |   big | c6123 |  c717 |  output |
+-------+-------+-------+-------+---------+
| opt 0 |  0:30 |  2:05 |  2:00 |   74970 |
| opt 1 |  0:35 |  2:25 |  2:35 |   74430 |
| opt 2 |  1:00 |  3:40 |       |   74378 |
| opt 3 |  3:00 |  9:50 |       |   71242 |
| opt 4 |  5:30 | 32:30 |       |  106474 |
+-------+-------+-------+-------+---------+

Execute the following script
(you may need root privileges to do so):

[Begin file]
#!/com/sh
# Build a printer driver for prsvr.

args "Create object type"
crty generic -l
args "Create binary object"
crtyobj generic
args "Delete object type"
dlty generic

args " "
args "Compiling driver_generic.c ..."
cc driver_generic.c -pic -cpu m68k # -info 4

args " "
args "Bind the lot"
# prsvr_driver_trait can be found in
#  /domain_examples/hardcopy/driver.
args "Ignore error messages about Undefined Globals"
# (all  First referenced in driver_generic.bin)
#   add_db_range_c
#   add_db_string_c
#   add_db_value_c
#   alloc_db
#   free_db_c
#   get_db_string_c
#   get_db_value_c
#   inq_entry_c
#   print_entries_c
#   set_db_value_c
#   spe_$pio_set_mode
bind -b generic - <<!
-allmark
driver_generic.bin
generic_uid.bin
prsvr_driver_trait
-entry generic_$initialize
-end
!

args "All done"
[End file]

----------

Copy the file produced to /sys/hardcopy/drivers:
/com/cpf generic /sys/hardcopy/drivers -r -l

----------

Create a config file for prmgr.
The recommended name is /etc/daemons/prmgr
(/etc/daemons is a link to `node_data/etc/daemons):

[Begin file]
spool_node = //c730d
prmgr_name = prmgr_test
[End file]

----------

*yyyyy Do not use tab character in config file! Use blanks only. yyyyy*
*yyyyy At SR10.0 the lines left_margin and right_margin may not be accepted. yyyyy*

Create a config file for prsvr.
The recommended name is /etc/daemons/prsvr:

[Begin file]
# Config file for prsvr, using generic driver

# Manager to use
prmgr_site        prmgr_test

# Line to appear with prf -list_printers (up to 32 characters)
description       Test of printing software
# Line printed on banner pages (up to 30 characters)
logo              Print server on test

# Name of printer (p or something else)
printer_name      test
# First printer of this name
printer_ext       0

# Interface
io_device_name    /dev/sio1
speed             1200

# Driver
device_driver     generic
model_number      dumb_ascii
# Further config options
ddf_name          /etc/daemons/generic_config

# Paper size
paper_length      11
paper_width       13.6

# Margins
top_margin        0
bottom_margin     1
left_margin       0.5
right_margin      0.5

# Character size
point             12
lpi               6

# Formatting
form_feeds        1

# Banners
file_banners      first

# Headers
#if SR_level == SR10_0 || SR_level == SR10_1 || SR_level == SR10_2
# The following head_ and foot_string would work fine.
# But we cannot have them off by default; the user would
# have to type '-headers off' just about every time.
# head_string     !/Last*modified*&/Page#
# foot_string     !/Printed*%/Page#
#endif
#if SR_level == SR10_3
# If the user wants these headers, use '-headers [on]';
# or have different ones by -head_string and -foot_string.
head_string       !/Last*modified*&/Page#
foot_string       !/Printed*%/Page#
#endif
[End file]

for use on a SIO line. For the SPE parallel PIO line, replace the
two lines after "#Interface" with

[Begin file insert]
io_device_name    /dev/pio
interface         spe_parallel
[End file insert]

*xxxxx Add new printer here xxxxx*
Obviously, change the line model_number to specify the printer
you have. Possible choices are:
dumb_ascii
epson_lq800
epson_lq1000
epson_lq1500
epson_lx80
epson_fx80
epson_fx85
epson_fx105
apple_imagewriter
apple_imagewriter_II
HP_LaserJet
HP_LaserJetPlus
microline_84
microline_380
microline_390
microline_391
NEC_pinwriter_p5
star_nx1000

----------

Create the config file /etc/daemons/generic_config:

[Begin file]
# This is the config file for the generic driver.
# In a separate (from the prsvr config) file, since the prsvr
# does not allow user-defined options.

passwd_file		/etc/daemons/generic_passwd

reinit_line		on

sio_stop_bits		1.0
sio_parity		none
sio_bits_per_char	8
sio_nlc_delay		0
sio_bp_enable		off
sio_cts_enable		off
sio_dcd_enable		off
sio_drain_out		on
sio_flush_in		off
sio_flush_out		off
sio_host_sync		on
sio_hup_close		off
sio_input_sync		on
sio_no_NL		off
sio_no_echo		on
sio_quit_enable		off
sio_raw			off
sio_raw_nl		off
sio_rts_enable		off
sio_susp_enable		off

end_of_line		LF
use_CR_without_LF	off

big_job_chars		30
big_job_lines		600
big_job_pages		20
reject_big_jobs		off

tab_length		8
word_wrap		on
eightbit		~
ctrl_char		caret

graph_border		on
reverse_bw		light
[End file]

----------

#if SR_level == SR10_2 || SR_level == SR10_3

Create the password file /etc/daemons/generic_passwd:

[Begin file]
pretty
The first line of this file is the password for the driver.
The entire first line is used: avoid (embedded or trailing) blanks.
So that it is not easy to guess the length of the password,
do not make the file contain a single line.
This file must be readable to the prsvr (prfd.prfd.none) only.
[End file]

#endif

--------

Now start the servers. Initially you may wish to do this
in a window so you can see all sorts of error messages.
In any case, you may need root privileges to start prsvr
for the first time (to crty/dlty print_generic), or
in fact any time you wish to use a new printer driver.

Ensure LLBD is running: use
/etc/server /etc/ncs/llbd &
if necessary.

Enter the line
/sys/hardcopy/prmgr -cfg /etc/daemons/prmgr
in a window. When prmgr is running, enter the line
/sys/hardcopy/prsvr /etc/daemons/prsvr
in another window.

Test the working of the printer (from still another window).

If everything seems OK then kill the servers (prsvr first, then prmgr).

You will want to have the servers starting automatically at node startup.
For this, ensure llbd will start (/com/crf /etc/daemons/llbd), and include
the following lines into /etc/rc.user (in fact `node_data/etc/rc.user):

[Begin file insert]
# start print manager
 if [ -f /sys/hardcopy/prmgr -a -f /etc/daemons/prmgr ]; then
	(echo " prmgr\c" >/dev/console)
	/sys/hardcopy/prmgr -cfg /etc/daemons/prmgr 1>/dev/null 2>>\`node_data/system_logs/prmgr.log &
 fi
#
# start print server
 if [ -f /sys/hardcopy/prsvr -a -f /etc/daemons/prsvr ]; then
	(echo " prsvr\c" >/dev/console)
	/sys/hardcopy/prsvr /etc/daemons/prsvr 1>/dev/null 2>>\`node_data/system_logs/prsvr.log &
 fi
#
[End file insert]

----------

Now the print servers should start automatically. If you find
any problems, just look in the log files (after the server died)
`node_data/system_logs/prmgr.log and prsvr.log.

#if SR_level == SR10_0 || SR_level == SR10_1
Prsvr will always emit to the message
?(prsvr) Unable to delete current manager for type print_generic - insufficient rights (OS/naming server)
#endif
#if SR_level == SR10_0
and will sometimes die after this. To get it going, do
dlty print_generic
and then start prsvr in a window. Kill it and reboot; usually
prsvr will come up alive after this first reboot.
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
You need to make sure that the prsvr process has rwx access
to the /sys/mgrs directory.
#endif

----------

The log files should also come handy for detecting who has printed
inadmissibly large jobs. Look for the strings "error", "warning" and
"big job". */


/*--------------------------------------------------*/


/*zzzzz Known bugs, limitations and problems:

Comments as at SR10.3:
(listing new or fixed problems, as compared to SR10.2, 10.1 and 10.0 below)

The -postscript option seems to get ignored (from SR10.2, but only
noticed lately, maybe due to patch m0147?): cannot convince a
PostScript printer that a file is really PostScript, even if it does
not begin with '%!'. (Could -transparent be used for this?) Cannot
print the PostScript source on a non-PostScript printer. (Could
-filter_chain be used for this somehow?)
At least, impress files do not crash the prsvr any more.

Manager type (mngr) is 0 for all file types.

-point and -cpi are used the same as at 10.2.

-trans and -ftn files: extra elements trans and ftn are present
in database, with INTEGER values 0 or 1. But we do not need
these as f_type seems OK.

Resolution specified in prsvr config file is ignored.

If there is no head and foot_string specified on the prsvr config
file, then head and foot strings are not present in database
(except for flag page, when they are the empty string).
Seems that a new element headers_on_off with INTEGER value 0 or 1
is sometimes present in the database. It is not present if you do not
use any one of -headers, -head_string or -foot_string. It is 0 if you
use -headers off, in this case head and foot strings are empty strings.
Otherwise headers_on_off is 1. If you use only one of -head or -foot,
you get the prsvr config file default for the other.

New element margins_on_off is always present in database. It has
INTEGER value 1, unless you use -margins off; in this case, all margins
have value 0.

A couple of other new things are present in the database.
What are these for?
printer_name: string test
del_spool:    INTEGER value 1
ftn:          INTEGER value 0
data_file:    string //c730d/sys/print/spooler/szabo_p.ftn
priority:     INTEGER value 127
sig_user:     INTEGER value 1

----------

Comments as at SR10.2:
(listing new or fixed problems, as compared to SR10.1 and SR10.0 below)

We now use (and expect this to work at SR10.0 or SR10.1 as well)
the ddf_name option to pass the name of the config file to the
driver. This has to be in a separate file, since user-defined
options are not accepted in the prsvr config file.

The following new options were added (to prf).
For more information look in the manual file ('man prf').
  -passwd password
  -wwrap [off|char|word|on]
  -tab_length n
  -eightbit [c|ignore|blank|value|convert]
  -ctrl_char [ignore|blank|value|ascii|control|caret]
  -graph_border [on|off]
  -reverse_bw [on|off|light|dark]
  -italics [on|off]
  -double_height [on|off]

The value of resolution in the config file is somehow ignored,
the default used is 300. Make sure you always specify -resolution
explicitly on the prf command line.

form_feed is set to 0 for banner pages.

Occasionally the printer died misteriously. It did not print, just left messages
?(ERROR in routine generic_$iosput)
    Unable to write to output (printer) line
    (stcode = 0127004C) - tty has changed sessions (stream manager/IOS)
in log file. To cure, kill prsvr, then restart it.

If option -cpi is used, then entry Cpi_ will be entered in obj_db,
without translating it to Point_. We can of course look for the Cpi_ entry
as well. However, at pre-SR10.2, if you specified both -cpi and -point on the
command line, -point was used and -cpi ignored. We cannot mimic this very well,
since Point_ is always in the database, and we do not really know what is the
default value, nor if that value is simply the default or was explicitly entered.
(Option -pitch is the same as -cpi.)

The error about setting drain_out was always there, but at SR10.2 it seems
to mess up the prsvr (causing it to hang). Fortunately (at least at SR10.2)
we can simply ignore this error without further problems.

New bug about filters/filter chains. We can no longer in the init routine
declare a filter_chain of "gmf$bmsc|bmsc$bm1", or we get the error
?(prsvr) Unable to find object /sys/hardcopy/filters/gmf$bmsc|bmsc$bm1 - name not found (OS/naming server)
If we just ask for gmf$bmsc and bmsc$bm1 separately, then the GMF file will be
translated to bmsc, but not to bm1. Maybe no big deal. But then, probably they
messed up the gmf$bmsc filter, so that GMF files created on a color node
will print garbage.
Note that Apollo drivers (at least the PostScript one for LaserWriters)
also print garbage from GMF files created on color nodes, even if they
do quite a lot of filtering (for all GMF files, created on color or BW nodes):
Filters used to process this job = gmf$bmsc -> bmsc$bm1 -> bm1$bmsc -> bmsc$postscpt ->

Hopefully, lint would now make more sense, since generic pointer
type 'void *genp' was used only due to bug in sio.h.

----------

Comments as at SR10.1:
(listing new or fixed problems, as compared to SR10.0 below)

Now the '-cancel job_id' and '-sig_printer abort' options work.
Only that anybody can kill any other person's jobs.

Now headers and footers sort of work.
Need to use inq_entry instead of get_db call to get hold of
the strings. No documentation on what needs to be replaced by page number.
The '!' for file name does not work properly (you may try it though).
And often you get garbage in the strings: '-head_string Date*%//Page#'
does not work, but '-head_string Date*%/Other*date*&/Page#' does.

The prsvr config file also accepts left_margin and right_margin.

The actual file is sent without lpi in the obj_db (if none is
specified explicitly on the prf command line). Likewise for
any error pages.

Under SR10.0, get_db calls did not crash the print server if there was
no such entry in the database, simply returned status print_$no_db_entry.
However under SR10.1 the prsvr will crash if we try to get_db_value
on the missing lpi. Need to use inq_entry instead.

The actual file will have values in obj_db for Lq_ and Print__Weight,
but not the banner page. Options are now working.

Value for Copies_, but not for Page__Layout, for the banner page has
been fixed.

----------

Comments as at SR10.0:

For the LaserJet printers, watch out for unprintable regions
at top and bottom margins: make sure both are over 0.375 inch.
Memory may be exceeded when printing graphics: restricted to
75 dpi on LaserJet for this reason; may be exceeded at 300 dpi
for LaserJetPlus.

There is an obscure bug in this program: the value of variable
mngr_type occasionally gets changed from 0 to 2048.

The filter bmsc$bm1 forgets to re-set file type from BMSC_ to BM1_.

When printing graphics files (which at the driver end are all type BM1_),
the manager type Mngr__Type is set to IOS, instead of GPR, type.

Not really a bug: when printing graphics files, the gpr_$init call
always fails with "Primitives already initialized".

When printing proper GPR files, the gpr_$set_bitmap call fails
with "Bitmap is read only". Thus cannot use grp_$read_pixels on it;
but can BLT from it into a temporary bitmap. In fact this temporary
bitmap buffer may be advantageous: makes taking care of bw_reverse
quite simple.

The values of Lq_ and Print__Weight are not reflected in the obj_db,
even if they are specified explicitly on the prf command line, thus
these options are not working.

No support as yet for proportional spacing. It is possible, but too
tedious, to keep track of the width of each character. However the
main requisite for taking advantage of proportional spacing, word
wrapping, is already implemented.

No support for right-justification or centering
(some of these printers in fact do full justification as well).

The only linefeed character supported is the LF.
There is no provision for CR or CR/LF combinations. However these
may be mimicked by SIO line settings.

The print server will insist in printing the banner page at 10 point
(12 chars per inch), irrespective of what is specified as default in
the prsvr config file or on the prf command line. May create problems
with line length. Can be fixed by accepting a single point size only
instead of a range.

The file itself is sent with lines_per_inch set to the first available
value (more accurately: usually the allowed lpi's are a range in init,
and got the file sent with the lower end of the range), regardless of
the default in the prsvr config file. Any lpi value specified on
the prf command line is however used correctly.

Also the status page at prsvr startup is printed regardless of default
page size etc. Can again be fixed by allowing a single size instead of
a range.

The banner page is sent to render with Copies_ set to 0.
Also Page__Layout is set to 0.

Stuff put in render_db by the init routine is NOT reflected in the
object_db passed to the render routine. We use user_db instead.

Headers and footers don't work. We invariably get 'ref to illegal addr'
errors.
Besides: Default headers are not used.
There is no documentation on how to get hold of the three parts of
the header strings for when the user explicitly specifies them with
prf -headers on -head str1/str2/str3 file
nor on the (control) characters used for page number etc.

Check the default (left and right) margins as well, don't seem to work.

Paper lengths are set on most printers with the ESC C null command.
This will accept only integer number of inches; not very good for
metric sized paper (microline84 accepts multiples of 1/2 inch, and
imagewriterII in units of 1/144 inch). For this reason, on the
epson_fx80 the paper length is set as number of lines. (This method
is inherently inferior to directly setting the length, as it cannot
give the correct length for all line spacings.)

The prsvr software will print to stderr (stream 2), instead of
stdout (stream 1). To make sure messages are intermixed properly,
this driver also uses fprintf(stderr,...) instead of printf.

In OS_ files when the BS character follows another control character,
the last bracket of <ctrl-something> is overprinted. (What else could
we do? No way of printing the control character itself.)
When the BS character is just past the end of the physical line, all
characters past the end, including the BS, are either ignored or bumped
to the next line, instead of overprinting that last character. (Maybe
the printer would bump it down itself, messing up the line count.)


You cannot use the lint utility to check this program out fully:
lint does not understand the generic pointer type 'void *genp'
declaration. The SYSV version is better, since it only issues
warnings for this, while the BSD version gives up.
Using the SYSV version of lint, watch out for the following:
You must separate the driver_init part off, generic pointers
are used in pfm.h too profusely for it.
Lint does not know about #options and #attribute (in pfm.h).
Lint does not accept prototypes in typedef declarations:
gives error on 'typedef void function(something)', wants
just empty parenthesis '()'; however the compiler accepts
(and does also use?) the form with prototypes.

zzzzz*/


/*--------------------------------------------------*/

/* general boolean */
#define no	0
#define yes	1
#define off	0
#define on	1

/* reverse_bw */
#define light	2
#define dark	3

/* parity */
#define none	0
#define odd	1
#define even	2

/* end_of_line */
#define eol_cr		0
#define eol_lf		1
#define eol_cr_lf	2
#define eol_lf_cr	3

/* use_bs */
#define one_at_a_time	2
#define use_cr_instead	3

/* ctrl_char */
#define ctrl_ignore	0
#define ctrl_blank	1
#define ctrl_value	2
#define ctrl_ascii	3
#define ctrl_control	4
#define ctrl_caret	5

/* eightbit */
#define eightbit_ignore		0x00
#define eightbit_blank		0x01
#define eightbit_value		0x02
#define eightbit_convert	0x03

/* Limits on big_job */
#define huge_job_chars 10000
#define huge_job_lines 10000
#define huge_job_pages 200

/*--------------------------------------------------*/


/* Include files */

/*zzzzz There are bugs in #include <math.h>: missing prototypes.
Try this instead.
[Begin insert file] zzzzz*/
/*	math.h	4.6	9/11/85	*/

/*
 *	Apollo Changes:
 *
 *      15 May 90  psz      Added prototypes for cabs,copysign,drem,logb,scalb,finite
                            Renamed second argument of ldexp
                            Moved declaration of struct exception
 *	08/11/89 rps		matherr decls from system V
 *      01/07/89 mills      Add asinh,acosh,atanh,copysign,drem,logb,scalb, and cabs
 *                          note no prototype checking for cabs()
 *      03/23/88 lwa        Fix incorrect fabs() decl.
 *	12/04/87 james_w	Provide function prototypes.
 */

/*  Enable function prototypes for ANSI C and C++  */
#if defined(__STDC__) || defined(c_plusplus) || defined(__cplusplus)
#    define _PROTOTYPES
#endif

/*  Required for C++ V2.0  */
#ifdef  __cplusplus
    extern "C" {
#endif

#ifndef _PROTOTYPES
extern double asinh(), acosh(), atanh();
extern double erf(), erfc();
extern double exp(), expm1(), log(), log10(), log1p(), pow();
extern double fabs(), floor(), ceil(), rint();
extern double lgamma();
extern double hypot(), cabs();
extern int matherr();
extern double copysign(), drem(), logb(), scalb();
extern int finite();
#ifdef vax
extern double infnan();
#endif
extern double j0(), j1(), jn(), y0(), y1(), yn();
extern double sin(), cos(), tan(), asin(), acos(), atan(), atan2();
extern double sinh(), cosh(), tanh();
extern double cbrt(), sqrt();
extern double modf(), ldexp(), frexp(), atof();
#else
extern double asinh(double x), acosh(double x), atanh(double x);
extern double j0(double x), j1(double x), jn(int n, double x),
              y0(double x), y1(double x), yn(int n, double x);
extern double erf(double x), erfc(double x);
extern double exp(double x), log(double x), log10(double x),
              pow(double x, double y), sqrt(double x),
              expm1(double x), log1p(double x);
extern double floor(double x), ceil(double x), rint(double x),
              fabs(double x);
extern double lgamma(double x);
extern double hypot(double x, double y), cabs(struct {double x,y;} z);
struct exception {
	int type;
	char *name;
	double arg1;
	double arg2;
	double retval; };
extern int    matherr(struct exception *x);
extern double sinh(double x), cosh(double x), tanh(double x);
extern double sin(double x), cos(double x), tan(double x),
              asin(double x), acos(double x), atan(double x),
              atan2(double y, double x);
extern double atof(char *nptr);
extern double ldexp(double value, int expp),
              frexp(double value, int *eptr),
              modf(double value, double *iptr);
extern double copysign(double x, double y), drem(double x, double y), logb(double x), scalb(double x, int n);
extern int    finite(double x);
#endif

#undef _PROTOTYPES

#ifdef  __cplusplus
    }
#endif

#define HUGE		((float) 0.3402823466e+39)
#define MAXDOUBLE	1.79769313486231570e+308

#define	DOMAIN		1
#define	SING		2
#define	OVERFLOW	3
#define	UNDERFLOW	4
#define	TLOSS		5
#define	PLOSS		6
/*zzzzz [End insert file] zzzzz*/


#include <stdio.h>      /*zzzzz There are two bugs in stdio.h, up to SR10.1 zzzzz*/
#include <string.h>
#include <apollo/base.h>
#include <apollo/cal.h>
#include <apollo/error.h>
#include <apollo/ios.h>
#include <apollo/type_uids.h>


/*zzzzz There are bugs in #include <apollo/sio.h> up to SR10.1:
type sio_$value_t never used.
Try this instead: this is the SR10.3 version.
[Begin insert file] zzzzz*/
  /* *** sio.h, /usr/include/apollo
     declarations for sio-line access on an open stream
   Changes:
         01/30/90  garyf      add inclusion protection
         02/24/89  johnston   Bug: 'void &value' is not legal in a function prototype
         02/02/89  everett_m  add sio_$break_detect to sio_$err_possibilities_t
         12/21/88  gilbert    Modifications for C++.
         10/13/87  swin (tas) added ANSI-std function prototypes
   */

       /* status codes */

#ifndef apollo_sio_h
#define apollo_sio_h

#define sio_$stream_not_sio   0x04060001
#define sio_$bad_option       0x04060002
#define sio_$illegal_strid    0x04060003
#define sio_$stream_not_open  0x04060004
#define sio_$incompatible_speed   0x04060005

#define sio_$max_line  3

#define sio_$50     0         /* baud rate constants */
#define sio_$75     1         /* use in sio_$control with speed option */
#define sio_$110    2
#define sio_$134    3
#define sio_$150    4
#define sio_$300    5
#define sio_$600    6
#define sio_$1200   7
#define sio_$2000   9
#define sio_$2400   10
#define sio_$3600   11
#define sio_$4800   12
#define sio_$7200   13
#define sio_$9600   14
#define sio_$19200  15

#define sio_$stop_1  1       /* coding of number of stop bits */
#define sio_$stop_1_point_5  2
#define sio_$stop_2  3

#define sio_$no_parity  0    /* coding of parity control bits */
#define sio_$odd_parity  1
#define sio_$even_parity  3

#define sio_$5bpc  0         /* coding of bits per character control */
#define sio_$6bpc  2
#define sio_$7bpc  1
#define sio_$8bpc  3


typedef short sio_$line_t;

/* enables for some HW "error" conditions */

#define sio_$check_parity     1
#define sio_$check_framing    2
#define sio_$check_dcd_change 4
#define sio_$check_cts_change 8
#define sio_$break_detect    16

typedef short sio_$err_enables_t;

typedef union {
        char c;
        pinteger i;
        char b;        /* boolean */
        sio_$err_enables_t es;
} sio_$value_t;

/* sio options */

typedef short enum {
    sio_$erase,         /* set erase character */
    sio_$kill,          /* set kill character  */
    sio_$eofchr,        /* set EOF character   */
    sio_$raw,           /* transparent input and output */
    sio_$no_echo,       /* don't echo input    */
    sio_$no_NL,         /* don't special case newlines */
    sio_$speed,         /* set bitrate */
    sio_$host_sync,     /* use xoff/xon to sync with host */
    sio_$nlc_delay,     /* constant delay for newlines */
    sio_$quit_enable,   /* enable quits from this line to calling process */
    sio_$input_sync,    /* respond xoff/xon on receive side*/
    sio_$line,          /* return line number - inquire only */
    sio_$rts,           /* set/clear RTS bit */
    sio_$dtr,           /* set/clear DTR bit */
    sio_$dcd,           /* read DTR bit (inquire only) */
    sio_$dcd_enable,    /* enable fault on dcd loss */
    sio_$cts,           /* read CTS bit (inquire only) */
    sio_$cts_enable,    /* enable CTS gating of output */
    sio_$parity,        /* control parity setting/processing */
    sio_$bits_per_char, /* number of bits per chararacter */
    sio_$stop_bits,     /* number of stop bits */
    sio_$err_enable,    /* enable error reporting */
    sio_$send_break,    /* establish break for value millisec. */
    sio_$quitchr,       /* set quit character */
    sio_$bp_enable,     /* enable bitpad processing on line */
    sio_$int_enable,    /* enable interrupts in this process */
    sio_$intchr,        /* set interrupt character */
    sio_$susp_enable,   /* enable process suspension character */
    sio_$suspchr,       /* set process suspension character */
    sio_$raw_nl,        /* display nl/cr on nl output in raw mode */
    sio_$unused,
    sio_$hup_close,     /* set hangup-on-close */
    sio_$rts_enable,    /* enable rts flow control */
    sio_$speed_force,   /* set bitrate, even if disturbs partner channel */
    sio_$pgroup,        /* process group that gets faults */
    sio_$unused2,
    sio_$unused3,
    sio_$flush_in,      /* flush input buffer */
    sio_$flush_out,     /* flush output buffer */
    sio_$drain_out,     /* wait for output buffer to drain */
} sio_$opt_t;

/*  Required for C++ V2.0  */
#ifdef  __cplusplus
    extern "C" {
#endif

extern void sio_$control(
        stream_$id_t          &stream_id,
        sio_$opt_t            &option,
        sio_$value_t          &value,
        status_$t             *status
        );

extern void sio_$inquire(
        stream_$id_t          &stream_id,
        sio_$opt_t            &option,
        sio_$value_t          *value,
        status_$t             *status
        );

#ifdef  __cplusplus
    }
#endif

#endif /* apollo_sio_h */
/*zzzzz [End insert file] zzzzz*/


#include <apollo/gpr.h>
#if SR_level == SR10_0 || SR_level == SR10_1 || SR_level == SR10_2
/*zzzzz There are bugs in gpr.h.
Mostly these are not relevant to us, except the
following which ought to be defined there: zzzzz*/
#endif
#if SR_level == SR10_3
/*zzzzz There are bugs in gpr.h.
The following must be unsigned, as used e.g. in gpr_$raster_op_prim_set,
but it is defined signed (and the definition never again used). zzzzz*/
#endif
/* typedef unsigned short gpr_$rop_prim_set_t; */
/* So define my own */
typedef unsigned short my_gpr_$rop_prim_set_t;


/*zzzzz New style of #include "/sys/ins/prsvr.ins.c" not yet available.
Try this instead.
[Begin insert file] zzzzz*/
/*
    PRSVR.INS.C

    This file contains definitions used to write filters and drivers
    for the print server. For a complete explanation of the procedure
    refer to "Printing in the Domain Environment"

Changes:
 11 Oct 90  psz  compatibility with SR10.3 added
  2 May 89  psz  added function prototypes
		 fixed type status_$t of returned status
		 fixed declaration of close_flags_t
*/

#ifndef prsvr_ins_c
#define prsvr_inc_c

#define print_$success               0
#define print_$bad_link              0X4020001
#define print_$dir_not_found         0X4020002
#define print_$not_records_file      0X4020003
#define print_$bad_data_file         0X4020004
#define print_$prf_option_unknown    0X4020005
#define print_$bad_prf_option_value  0X4020006
#define print_$multiple_files_found  0X4020007
#define print_$non_printable_file    0X4020008
#define print_$postscript_error      0X4020009
#define print_$file_device_mismatch  0X402000A
#define print_$bad_instance          0X402000B
#define print_$max_instance          0X402000C
#define print_$target_invalid        0X402000D
#define print_$no_prtr               0X402000E
#define print_$no_job                0X402000F
#define print_$non_null_queue        0X4020010
#define print_$max_queued            0X4020011
#define print_$job_active            0X4020012
#define print_$dup_admin_val         0X4020013
#define print_$bad_signal_type       0X4020014
#define print_$ncs_error             0X4020015
#define print_$no_db_entry           0X4020016
#define print_$wrong_db_entry_type   0X4020017
#define print_$rdr_read_error        0X4020018
#define print_$rdr_write_error       0X4020019
#define print_$rdr_dev_open_error    0X402001A
#define print_$rdr_bad_db_data       0X402001B
#define print_$rdr_dev_close_error   0X402001C
#define print_$rdr_format_error      0X402001D
#define print_$fltr_read_error       0X402001E
#define print_$fltr_write_error      0X402001F
#define print_$fltr_open_error       0X4020020
#define print_$fltr_format_error     0X4020021
#define print_$fltr_bad_type         0X4020022
#define print_$memory_error          0X4020023
#define print_$dup_printer_name      0X4020024
#define print_$user_aborted_job      0X4020025
#define print_$job_suspended         0X4020026

/* Printer status to set in the print manager */
#define prsvr_$ok 0
#define prsvr_$Offline 2
#define prsvr_$Fault 3
#define prsvr_$Out_of_Paper 6

typedef char* db_obj;

/*
Database value fields
    Value       = 1 float value
    Range r0,r1 = 2 float values
    String      = null terminated character string
*/
#if SR_level == SR10_0 || SR_level == SR10_1
typedef enum { Value_Field, Range_Field, String_Field } db_field_t;
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
typedef enum { Value_Field, Range_Field, String_Field, Int_Field, Unb_Val_Field, Unb_Str_Field, Max_Field } db_field_t;
#endif

typedef union
{
    float   *rval;
    char    *str;
#if SR_level == SR10_2 || SR_level == SR10_3
    int     *ival;
#endif
} val_t;

/* User structure to retrieve and load database information */
typedef struct
{
    int              num_entries;   /* Number of database value elements */
    db_field_t       *type; /* pointer to type of first value */
    val_t            v;  /* pointer to first value */
} db_data_t;



/* Allocate and free a database object that the following functions operate on */
extern db_obj    alloc_db (void);
extern void      free_db_c (db_obj db);

/* Direct call into the database */
extern status_$t inq_entry_c     (db_obj db ,char *key, db_data_t *data);

/* Convenient interfaces to the database, derived from inq_entry */
#if SR_level == SR10_0 || SR_level == SR10_1
extern status_$t set_db_string_c (db_obj db, char *key, char *str);
extern status_$t set_db_value_c  (db_obj db, char *key, float *value);

extern char*     get_db_string_c (db_obj db, char *key, status_$t *status);
extern float*    get_db_value_c  (db_obj db, char *key, status_$t *status);

extern status_$t add_db_value_c  (db_obj db, char *key, float *value);
extern status_$t add_db_range_c  (db_obj db, char *key, float *r0, float *r1);
extern status_$t add_db_string_c (db_obj db, char *key, char *str);
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
/*
  Set Functions:

  1) Set a float value.
  2) Set an integer value.
  3) Set a unbounded value.
  4) Set a value range.
  5) Set a string.
  6) Set an unbounded string.
*/
extern status_$t set_db_value_c  (db_obj db, char *key, float *value);
extern status_$t set_db_int_c    (db_obj db, char *key, int *value);
extern status_$t set_db_ubv_c    (db_obj db, char *key);
extern status_$t set_db_range_c  (db_obj db, char *key, float *r0, float *r1);
extern status_$t set_db_string_c (db_obj db, char *key, char *str);
extern status_$t set_db_ubs_c    (db_obj db, char *key);

/*
  Retrieval Functions:

  1) Get a float value.
  2) Get an integer value.
  3) Get a unbounded value.
     => Return a successful status if key has an unbounded value.
  4) Get a value range. <== No such routine !!!
  5) Get a string.
  6) Get an unbounded string.
     => Return a successful status if key has an unbounded string.
*/
extern float*    get_db_value_c  (db_obj db, char *key, status_$t *status);
extern int*      get_db_int_c    (db_obj db, char *key, status_$t *status);
extern void      get_db_ubv_c    (db_obj db, char *key, status_$t *status);
extern char*     get_db_string_c (db_obj db, char *key, status_$t *status);
extern void      get_db_ubs_c    (db_obj db, char *key, status_$t *status);

/*
  Addition Functions:

  1) Add a float value.
  2) Add an integer value.
  3) Add a unbounded value.
  4) Add a value range.
  5) Add a string.
  6) Add an unbounded string.
*/
extern status_$t add_db_value_c  (db_obj db, char *key, float *value);
extern status_$t add_db_int_c    (db_obj db, char *key, int *value);
extern status_$t add_db_ubv_c    (db_obj db, char *key);
extern status_$t add_db_range_c  (db_obj db, char *key, float *r0, float *r1);
extern status_$t add_db_string_c (db_obj db, char *key, char *str);
extern status_$t add_db_ubs_c    (db_obj db, char *key);
#endif
/* Flush entire database */
extern void      flush_entries_c (db_obj db);

/* Clean inq_entryed memory allocations */
extern void      garbage_collect_c (db_obj db);

/* Debug only */
extern void      print_entries_c (db_obj db);

extern ios_$id_t msg_strid;  /* stream to the error log /sys/node_data/system_logs/prsvr_error_log */

/* built in databases */
extern db_obj    control_db;  /* print server config file parameters */
extern db_obj    interface_db; /* I/O information */
extern db_obj    render_db;  /* what the printer/renderer can do */
extern db_obj    default_db; /* default job properties */
extern db_obj    filter_list_db;  /* list of available filters */


#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
/* This call updates the printer status in the print manager.
Use it (HOW?) in the monitor routine to signal printer fault conditions.
Arguments:
printer_status = prsvr_$ok || prsvr_$Offline || prsvr_$Fault || prsvr_$Out_of_paper
retry = -1 => repeat until this operation succeeds
retry =  0 => return if the update fails */

extern status_$t prsvr_$update_status (int printer_status, int retry);
#endif

/*
-------------------------
 definitions for drivers
-------------------------
*/

typedef long close_flags_t; /* for future use */

typedef db_obj    (*init_func)    (/* db_obj db, status_$t* status */);
typedef status_$t (*render_func)  (/* int* stream, db_obj db */);
typedef void      (*eoj_func)     (/* db_obj db, status_$t *status */);
typedef void      (*close_func)   (/* close_flags_t flags, status_$t *status */);
typedef void      (*monitor_func) (/* db_obj int_db, status_$t *status */);

typedef struct prsvr_driver_$epv_t
{
    init_func    init;
    monitor_func monitor;
    render_func  render;
    eoj_func     end_job;
    close_func   close;
};

typedef struct prsvr_driver_$epv_t *prsvr_driver_$epv_ptr_t;

extern short prsvr_driver_$trait;

#if SR_level == SR10_0 || SR_level == SR10_1
/*
For debugging drivers. Create a module driver_debug.c
with the procedure prsvr_$driver_debug that allocates
a prsvr_driver_$epv_ptr_t and initializes it as follows:
  prsvr_driver_epv_ptr->init = driver_name_$init;
  prsvr_driver_epv_ptr->monitor = driver_name_$monitor;
  prsvr_driver_epv_ptr->render = driver_name_$render;
  prsvr_driver_epv_ptr->end_job = driver_name_$end_job;
  prsvr_driver_epv_ptr->close = driver_name_$close;

Bind this module and the actual driver code with prsvr and add
the device parameter 'debug_driver' in the config file BEFORE the
device declaration. This will allow breakpoints to be set in the driver.
*/

extern prsvr_driver_$epv_ptr_t prsvr_$driver_debug(
			      char name[80],
			      int *namelen,
			      status_$t *status);
#endif

/*
-------------------------
 definitions for filters
-------------------------
*/

typedef char*     (*inquire_func) (/* char *type, status_$t *status */);
typedef int*      (*process_func) (/* int* stream, db_obj job_db, status_$t *status */);

typedef struct prsvr_filter_$epv_t
{
    process_func   proc;
    inquire_func   inq;
};

typedef struct prsvr_filter_$epv_t *prsvr_filter_$epv_ptr_t;

extern short prsvr_filter_$trait;

#if SR_level == SR10_0 || SR_level == SR10_1
/*
for debugging filters. Create a module filter_debug.c
with the procedure prsvr_$filter_debug that allocates
a prsvr_filter_$epv_ptr_t and initializes it as follows :
    prsvr_filter_epv_ptr->process := NAME_$process;
    prsvr_filter_epv_ptr->inquire := NAME_$inquire;

Bind this module and the filter code with prsvr and add
the device parameter 'debug_filter' in the config file.
*/

extern prsvr_filter_$epv_ptr_t prsvr_$filter_debug(
			      char name[80],
			      int *namelen,
			      status_$t *status);
#endif

/* ------------------------------------------------------- */

/* c/ccxx database values header file */

#ifndef db_value_data_hxx
#define db_value_data_hxx

/* */
#define Io__Mode   "io_mode"

/* only needed if output port is an sio line */
#define Device__Speed   "dev_speed"

/* name of the output device, can also be a pad or file */
#define Device__Name   "dev_name"

/* if driver supports more than one model */
#define Printer__Model   "prtr_model"

/* stream connected to the output device */
#define Output__Stream   "o_stream"

/* used when the printer sends status info to a separate sio channel */
#define Mon__Name   "mon_name"
#define Mon__Speed   "mon_speed"
#define Mon__Stream   "m_stream"

/* only needed if output port is a parallel port without a streams mng */
#define Ddf__Name   "ddf_nm"

#if SR_level == SR10_3
/* name of print server config file (for reference only) */
#define ConfigFileName   "cfg_nm"
#endif

/* entry point vectors into the driver */
#define Driver__Epv   "d_epv"

/* unit number for PBU devices */
#define Pbu__Unit   "pbu_unit"

/* character point size (32 bit real) */
#define Point_   "point"

/* inter line spacing (32 bit real) */
#define Lpi_   "lpi"

#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
/* inter character spacing - generally for dot matrix printers (32 bit real) */
#define Cpi_   "cpi"
#endif

/* width of output page (32 bit real) */
#define Paper__Width   "p_width"

/* length of output page (32 bit real) */
#define Paper__Length   "p_length"

/* type of paper (32 bit real) */
#define Paper__Type   "p_type"

/* Margins */
#define Left__Margin   "l_mar"
#define Right__Margin   "r_mar"
#define Top__Margin   "t_mar"
#define Bottom__Margin   "b_mar"

#if SR_level == SR10_0 || SR_level == SR10_1 || SR_level == SR10_2
/* */
#define Fonts_   "fonts"
#endif

#if SR_level == SR10_3
/* Print text with this typeface (string) */
#define Typeface_   "typf"

/* (Obsolete, use Typeface_) */
/*yyyyy But why is Fonts_ in the obj_db, not Typeface_? And why did they drop the plural from fonts? yyyyy*/
#define Fonts_   "font"
#endif

/* */
#define Resolution_   "res"

/* */
#define Color_   "col"

/* */
#define Magnification_   "mag"

/* */
#define BW__reverse   "bw_rev"

#if SR_level == SR10_0 || SR_level == SR10_1 || SR_level == SR10_2
/* */
#define Char__Set   "c_set"
#endif

#if SR_level == SR10_3
/* Use this symbol set to map char codes to glyphs (string) */
/*zzzzz Is this a misprint: Symbol__Set ? zzzzz*/
#define Symbol_Set   "sy_set"

/* (Obsolete, use Symbol_Set) */
#define Char__Set   "c_set"
#endif

/* */
#define Page__Layout   "p_out"

/* */
#define F__Type   "f_type"

/* */
#define Header_   "head"

/* */
#define Footer_   "foot"

/* (32 bit real) */
#define Print__Weight   "p_wt"

/* (32 bit real) */
#define Lq_   "lq"

/* Wrap at and of line (on or off; otherwise truncate) */
#define Wrap_   "wrap"

/* */
#define Copies_   "cp"

/* */
#define PageRev_   "page_rev"

/* */
#define ColCop_   "col_cop"

/* */
#define Bitmap__X   "bit_x"

/* */
#define Bitmap__Y   "bit_y"

/* */
#define Bitmap__Planes   "bit_pl"

/* */
#define Form__Feeds   "f_feed"

/* */
#define Columns_   "colm"

/* */
#define Filter__Chain   "f_chn"

#if SR_level == SR10_0
/* */
#define Rotation_   "rota$"
#endif

/* internal prsvr control functions */
#define logo_str_   "logo_str"
#define site_   "site"
#define dis_time_   "dis_time"
#define err_msg_   "err_msg"
#define err_log_   "err_log"
#define pr_ext_   "pr_ext"
#define pr_name_   "pr_name"
#define descr_   "descr"
#define pre10_nm_   "pre10_nm"
#define file_banners_   "file_banners"
#define driv_nm_   "driv_nm"

/* bitmap descriptor header */
#define Bm__Hdr   "bm_desc"

#if SR_level == SR10_0 || SR_level == SR10_1
/* pointer to renderer database */
#define Rend__Db   "red_db"
#endif

#if SR_level == SR10_2 || SR_level == SR10_3
/* Pointer to default database */
#define Def__Db   "def_db"

/* Pointer to interface database */
#define Intf__Db   "interface_db"

/* Pointer to driver database */
#define Driver__Db   "ren_db"
#endif

#if SR_level == SR10_3
/* Pointer to the user database - attributes specifed by user */
#define User__Db   "user_db"
#endif

#if SR_level == SR10_1
/* pointer to extensible option database */
#define Ext__Db   "ext_db"

/* extensible option list */
#define Ext__OptList   "ext_olst"

/* extensible option help */
#define Ext__OptHelp   "ext_help"

/* extensible option comments */
#define Ext__OptCmts   "ext_ocmt"
#endif

#if SR_level == SR10_2 || SR_level == SR10_3
/* pointer to extensible option database */
#define Ext__Db   "ext_db"

/* database key list */
#define DbKeys__List   "db_kLst"

/* extensible option help */
#define Ext__OptHelp   "ext_help"
#endif

/* manager used to process data object */
#define Mngr__Type   "mngr"

/* source filename */
#define Src_Name_   "src"

#if SR_level == SR10_3
/*source filename leaf */
#define Src_Leaf_   "src_leaf"
#endif


/* known file types */

/* vanilla uasc */
#define UASC_   "uasc"

#if SR_level == SR10_2 || SR_level == SR10_3
/* formatted uasc text - includes headers, footers and cc chars */
#define FASC_   "fasc"
#endif

/* overstrike/bolded backspaced uasc */
#define OS_   "os"

/* fortran carriage control */
#define FTN_   "ftn"

/* tml model */
#define TML_   "tml"

/* gpr section oriented bitmap of from 1 to 8 bits/section */
#define BMSC_   "bmsc"

/* gpr pixel oriented bitmap of from 1 to 8 bits/pixel */
#define BMP_   "bmp"

/* gpr pixel oriented bitmap of 1,2,4 or 8 bits/pixel */
#define BMPX_   "bmpx"

#if SR_level == SR10_0
/* single plane pixel oriented bitmap */
#define BM1_   "1px"
#endif
#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
/* single plane bitmap */
#define BM1_   "bm1"
#endif

/* 3 section gpr bitmap of r,g,b components */
#define BMRGB_   "bmrgb"

/* 4 section gpr bitmap of r,g,b and alpha components */
#define BMRGBA_   "bmrgba"

/* gmf file */
#define GMF_   "gmf"

#if SR_level == SR10_0 || SR_level == SR10_1
/* formatted file for tektronix 4692, 4693 */
#define TEK46XX_   "tek46xx"
#endif

#if SR_level == SR10_2 || SR_level == SR10_3
/* formatted file for tektronix 4693D */
#define TEK46XX_   "tek46xx"

/* formatted file for tektronix 4692 */
#define T4692_   "t4692"
#endif

#if SR_level == SR10_3
/* HP Printer Control Language */
#define PCL_   "pcl"
#endif

/* PostScript code */
#define POST_   "postscpt"

/* Impress code */
#define IMP_   "impress"

/* send anything */
#define TRANS_   "trans"


/* convert file types - in-built filters */

#if SR_level == SR10_0 || SR_level == SR10_1
/* convert ascii to tml with default properties */
#define UASCTML_   "uasc$tml"
#endif

#if SR_level == SR10_2 || SR_level == SR10_3
/* convert ascii file to tml w/ job properties */
#define UASCTML_   "uasc$tml"

/* format ascii file w/ job properties */
#define UASCFASC_   "uasc$fasc"

/* format ascii overstrike file w/ job properties */
#define OSFASC_   "os$fasc"
#endif

/* convert ascii overstrike to tml with default properties */
#define OSTML_   "os$tml"

/* convert fortran to ascii overstrike */
#define FTNOS_   "ftn$os"

/* convert gmf to section bitmap */
#define GMFBMSC_   "gmf$bmsc"

/* convert section bitmap to postscript */
#define BMSCPOST_   "bmsc$postscpt"

/* convert pixel bitmap to tek commands */
#define BMTEK_   "bmsc$tek"

/* convert section bitmap to impress */
#define BM1IMP_   "bm1$impress"

/* convert tml to PostScript */
#define TMLPOST_   "tml$postscpt"

/* convert section bitmap to pixel bitmap */
#define BMSCBMP_   "bmsc$bmp"

/* convert pixel bitmap to section bitmap */
#define BMPBMSC_   "bmp$bmsc"

/* convert section bitmap to single plane bitmap */
#define BMSCBM1_   "bmsc$bm1"

#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
/* convert single plane bitmap to section bitmap (it is only a type change) */
#define BM1BMSC_   "bm1$bmsc"

/* pack pixel bitmap into a 1,2,4,8 bits/pixel bitmap */
#define BMPBMPX_   "bmp$bmpx"
#endif

#if SR_level == SR10_2 || SR_level == SR10_3
/* convert pixel bitmap to a tek 4692 byte stream */
#define BMPT4692_   "bmp$t4692"
#endif

/* check properties of tml against render_db */
#define TMLTML_   "tml$tml"

/* convert tml into impress */
#define TMLIMP_   "tml$impress"


/* values Mngr__Type */
#define mgr__ios    0
#define mgr__gpr    1
#define mgr__tml    2

#define prsvr__mng_ios    0
#define prsvr__mng_gpr    1
#define prsvr__mng_tml    2

/* generic boolean values */
#define On_     1
#define Off_    0

/* values for Io__Mode */
/*
#define Serial__Mode    1
#define Parallel__Mode  2
#define IKON__Mode      3
#define dsp80__Mode     4
#define GPIO__Mode      5
*/

#define prsvr__Io_Serial    1	/* sio port, pad or file */
#define prsvr__Io_Parallel  2	/* SPE parallel port */
#define prsvr__Io_IKON      3	/* IKON controller board */
#define prsvr__Io_dsp80     4	/* DSP80A, DSP90, dn4xx or dn 6xx npa port in apollo mode */
#define prsvr__Io_GPIO      5	/* User written GPIO driver. Be sure to specify ddf name */
#if SR_level == SR10_2 || SR_level == SR10_3
#define prsvr__Io_DPC       6	/* DSP80A, DSP90, dn4xx or dn 6xx npa port in dataproducts mode */
#define prsvr__Io_FILE      7	/* FILE or a PAD */
#endif

/* values for Page__Layout */
#define Portrait_    1
#define Landscape_    2

/* values for Paper__Type */
#define Cut_    1
#define FanFold_    2
#define Roll_    3
#define Transpar_    4

/* values for Print__Weight - for dot matrix printers */
#define Light_weight    1
#define Medium_weight    2
#define Bold_weight    3

/* Values for Lq_ - for dot matrix printers */
#define Draft_lq    1
#define NLQ_lq    2

#endif /* db_value_data_hxx */
#endif
/*zzzzz [End insert file] zzzzz*/


#if x_SPE_installed
/*zzzzz New style of #include "/sys/ins/spe.ins.c" not yet available.
Try this instead.
[Begin insert file] zzzzz*/
/*
    SPE.INS.C
    Insert file for users of Serial Parallel Expansion board

    26 Apr 89   psz added prototypes, got rid of trailing semicolon on error code definitions
    10/23/86    jjm original version
    12/11/86    jjm fixed syntax errors in error codes
*/

#define  spe_$no_controller         0x5190001 /* controller not present */
#define  spe_$not_init              0x5190002 /* controller not initialized */
#define  spe_$busy                  0x5190003 /* controller is busy */
#define  spe_$offline               0x5190004 /* unit not ready */
#define  spe_$paper_empty           0x5190005 /* device out of paper */
#define  spe_$fault                 0x5190006 /* error reported by external device */
#define  spe_$invalid_sio_line      0x5190007 /* sio line not available */
#define  spe_$already_init          0x5190008 /* controller already initialized */
#define  spe_$interrupt_error       0x5190009 /* error in interrupt handler */

typedef short enum {spe_$no_handshake,spe_$hand_busy,spe_$hand_ack,
			spe_$hand_ack_busy} pio_$handshake_t;

#define spe_$no_status_check ((pio_$status_check_t) 0x00)
#define spe_$chk_busy        ((pio_$status_check_t) 0x01)
#define spe_$chk_pe          ((pio_$status_check_t) 0x02)
#define spe_$chk_sel         ((pio_$status_check_t) 0x04)
#define spe_$chk_fault       ((pio_$status_check_t) 0x08)
#define spe_$wait_status     ((pio_$status_check_t) 0x10)

typedef short pio_$status_check_t;

void spe_$pio_set_mode (
    pio_$handshake_t    &handshake,
    pio_$status_check_t &status_check,
    status_$t           *status);

void spe_$pio_inq_mode (
    pio_$handshake_t    *handshake,
    pio_$status_check_t *status_check,
    status_$t           *status);

void spe_$pio_status (
    status_$t           *status);
/*zzzzz [End insert file] zzzzz*/
#endif


/*--------------------------------------------------*/

/* Routine declarations */

/* Standard driver routine: initialize printer */
extern db_obj generic_$init (
    db_obj iface_db,
    status_$t *status);

/* Standard driver routine: monitor printer (not used) */
extern void generic_$monitor (
    db_obj iface_db,
    status_$t *status);

/* Standard driver routine: do the job */
extern status_$t generic_$render (
    int* obj_id,
    db_obj obj_db);

/* Standard driver routine: finish job */
extern void generic_$end_job (
    db_obj obj_db,
    status_$t *status);

/* Standard driver routine: close printer connection */
extern void generic_$close (
    close_flags_t close_flags,
    status_$t *status);

/* Read config file */
extern status_$t generic_$read_config (
    char *file_namep,
    db_obj ren_db,
    db_obj usr_db,
    long i_o_mode,
    long printer_model);

/* */
extern status_$t generic_$check_config (
    long word_count,
    char *words[3],
    char *type,
    long low_val,
    long high_val,
    float *r1,
    float *r2,
    long *j1,
    long *j2);

/* */
extern status_$t generic_$check_config_sio (
    long word_count,
    char *words[3],
    char *type,
    long low_val,
    long high_val,
    long i_o_mode,
    float *r0,
    long *j0);

/* */
extern status_$t generic_$check_config_dumb (
    long word_count,
    char *words[3],
    char *type,
    long low_val,
    long high_val,
    long printer_model,
    float *r0,
    long *j0);

/* */
extern status_$t generic_$check_config_dumb_add (
    long word_count,
    char *words[3],
    char *type,
    long low_val,
    long high_val,
    long printer_model,
    db_obj ren_db,
    char *element,
    boolean *done);

/* */
extern void generic_$read_config_not_parallel (
    char *option);

/* */
extern void generic_$read_config_obsolete (
    char *option);

/* */
extern void generic_$read_config_not_dumb (
    char *option);

/* Add float value to database */
extern status_$t generic_$add_value (
    db_obj database,
    char *db_name,
    char *element,
    float value,
    long minval,
    long maxval,
    long printout);

/* Add float range to database */
extern status_$t generic_$add_range (
    db_obj database,
    char *db_name,
    char *element,
    float value_low,
    float value_high,
    long minval,
    long maxval,
    long printout);

/* Add string value to database */
extern status_$t generic_$add_string (
    db_obj database,
    char *db_name,
    char *element,
    char *value,
    long printout);

/* Convert float to long, avoiding overflow */
extern long generic_$float_to_long (
    float value);

/* Replace unprintables by blanks */
extern void generic_$cleanup_string (
    char *char_arr);

/* Get float value and integer equivalent from database */
extern status_$t generic_$get_value (
    db_obj database,
    char *db_name,
    char *element,
    float *valuep,
    long *convertp,
    boolean optional,
    long special,
    long minval,
    long maxval,
    long defval,
    long printout);

#if SR_level == SR10_3
/* Get integer value from database */
extern status_$t generic_$get_integer_value (
    db_obj database,
    char *db_name,
    char *element,
    long *valuep,
    boolean optional,
    long special,
    long minval,
    long maxval,
    long defval,
    long printout);
#endif

/* Get string value from database */
extern status_$t generic_$get_string (
    db_obj database,
    char *db_name,
    char *element,
    char **stringp,
    boolean optional,
    long printout);

/* */
extern status_$t generic_$check_string (
    db_obj database,
    char *db_name,
    char *element,
    char **stringp,
    boolean optional,
    char *type,
    long *j0,
    long printout);

/* Initialize printer line */
extern status_$t generic_$init_line (void);

/* Set boolean SIO line characteristic */
extern status_$t generic_$sio_control (
    sio_$opt_t option,
    char *name,
    long j);

/* Reset printer */
extern status_$t generic_$reset_printer (void);

/* Send number of linefeeds */
extern status_$t generic_$send_lf (
    long feeds);

/* Send formfeed */
extern status_$t generic_$send_ff (
    boolean footer_valid,
    char *footerp);

/* Check room on page for another line */
extern status_$t generic_$render_check_room (
    boolean header_valid,
    boolean footer_valid,
    char *headerp,
    char *footerp);

/* Again, for when just counting up chars/lines/pages */
extern void generic_$test_send_lf (
    long feeds);
extern void generic_$test_send_ff (
    boolean footer_valid);
extern void generic_$test_check_room (
    boolean header_valid,
    boolean footer_valid);

/* Print error message (for log file) and maybe also send to printer */
extern void generic_$print_error (
    status_$t status,
    char *routine,
    char *message1,
    char *message2,
    long printout);

/* Make up header (or footer) string in buffer */
extern long generic_$make_header (
    char *headerp,
    char *buffer);

/* */
extern void generic_$print_find_error (
    status_$t status,
    char *routine,
    char *element,
    char *database,
    char *def_val,
    long printout);

/* */
extern void generic_$print_sio_error (
    status_$t status,
    char *routine,
    char *property);

/* */
extern void generic_$print_crazy (
    char *routine,
    char *name,
    float value,
    char *def_val,
    long printout);

#if SR_level == SR10_2 || SR_level == SR10_3
/* */
extern void generic_$print_wrong_value (
    char *option,
    char *value);

/* */
extern void generic_$print_need_passwd (
    char *msg);
#endif

/* Easy use of ios_$put */
extern status_$t generic_$iosput (
    char *buffer,
    long bytes);

/* Output a chunk of graphics (non-blank parts) */
extern status_$t generic_$raster_chunk (
    long start1, long start0,
    long mult, long mult1, long i_fac,
    long imode);

#if SR_level == SR10_2 || SR_level == SR10_3
/* Check password */
extern status_$t generic_$check_passwd (
    char *passwdp,
    boolean *passwd_ok);
#endif

/* Compare strings, true if equal (about opposite of strcmp) */
extern boolean generic_$strcmp (
    char *str1,
    char *str2,
    boolean case_ignore);

/* Length of string (same as strlen) */
/* Currently not used */
/*
extern long generic_$strlen (
    char *str);
*/

/* Length of string, ignoring trailing blanks (and tabs and backspaces) */
extern long generic_$str_nonblank_len (
    char *buffer,
    long max_len);

/* Set up date-time string */
extern void generic_$date_time (
    char date[32]);

/*--------------------------------------------------*/

/* Printer models supported */
/*xxxxx Add new printer here xxxxx*/
/*  Make sure dumb_ascii is zero, all others positive,
    no gaps left, last_model is last value used.
    These are always referred to symbolically, and
    are only tested for equality never for a range,
    so ordering does not matter. */
#define dumb_ascii      0
#define epsonlq800      1
#define epsonlq1000     2
#define epsonlq1500     3
#define epsonlx80       4
#define epsonfx80       5
#define epsonfx85       6
#define epsonfx105      7
#define imagewriter     8
#define imagewriterII   9
#define HPlaserjet      10
#define HPlaserjetplus  11
#define microline84     12
#define microline380    13
#define microline390    14
#define microline391    15
#define pinwriterp5     16
#define starnx1000      17
#define last_model      17


/* Non-printable ASCII characters */

#define chr_nul    0x00
#define chr_soh    0x01
#define chr_stx    0x02
#define chr_etx    0x03
#define chr_eot    0x04
#define chr_enq    0x05
#define chr_ack    0x06
#define chr_bel    0x07
#define chr_bs     0x08
#define chr_tab    0x09
#define chr_lf     0x0A
#define chr_vt     0x0B
#define chr_ff     0x0C
#define chr_cr     0x0D
#define chr_so     0x0E
#define chr_si     0x0F
#define chr_dle    0x10
#define chr_dc1    0x11
#define chr_dc2    0x12
#define chr_dc3    0x13
#define chr_dc4    0x14
#define chr_nak    0x15
#define chr_syn    0x16
#define chr_etb    0x17
#define chr_can    0x18
#define chr_em     0x19
#define chr_sub    0x1A
#define chr_esc    0x1B
#define chr_fs     0x1C
#define chr_gs     0x1D
#define chr_rs     0x1E
#define chr_us     0x1F
#define chr_sp     0x20

#define chr_del    0x7F

/* Alternate names */
#define chr_null   0x00

#define chr_ctrl_a 0x01
#define chr_ctrl_b 0x02
#define chr_ctrl_c 0x03
#define chr_ctrl_d 0x04
#define chr_ctrl_e 0x05
#define chr_ctrl_f 0x06
#define chr_ctrl_g 0x07
#define chr_ctrl_h 0x08
#define chr_ctrl_i 0x09
#define chr_ctrl_j 0x0A
#define chr_ctrl_k 0x0B
#define chr_ctrl_l 0x0C
#define chr_ctrl_m 0x0D
#define chr_ctrl_n 0x0E
#define chr_ctrl_o 0x0F
#define chr_ctrl_p 0x10
#define chr_ctrl_q 0x11
#define chr_ctrl_r 0x12
#define chr_ctrl_s 0x13
#define chr_ctrl_t 0x14
#define chr_ctrl_u 0x15
#define chr_ctrl_v 0x16
#define chr_ctrl_w 0x17
#define chr_ctrl_x 0x18
#define chr_ctrl_y 0x19
#define chr_ctrl_z 0x1A

#define chr_blank  0x20

/*--------------------------------------------------*/

static ios_$id_t out_id;

static db_obj usr_db;
static long line_number;
static long page_number;
static boolean need_lf;
static long ff_bufflen,lf_bufflen;
static long total_chars_read;
static long total_chars_job;
static long total_lines_job;
static long total_pages_job;
static long total_jobs_uasc;
static long total_jobs_os;
static long total_jobs_ftn;
static long total_jobs_trans;
static long total_jobs_bm1;
static long total_jobs_other;
static long total_rejected_ascii;
static long total_rejected_trans;
static long total_rejected_bm1;
static long total_chars_ascii;
static long total_lines_ascii;
static long total_pages_ascii;
static long total_chars_trans;
static long total_chars_bm1;
static long total_lines_bm1;
static long total_pages_bm1;
static long printer_model;
static long chars_line;
static long chars_marg;
static long lines_paper;
static long lines_high;
static long lines_marg;
static long lines_page;
static long raster_x_cur,raster_y_cur; /* Current position */
static long raster_x_want,raster_y_want; /* Desired position */ /*zzzzz raster_x_want is never used zzzzz*/

/* General buffer. Must be at least 256 long */
char genbuff[bufsize];
/* Output line buffer. We keep writing output lines while bytes < bufsize, but may need 9 characters.
   Must also be long enough to hold printer setup strings, which seem to be at most 200 bytes. */
char lilbuff[bufsize + 10];
char bigbuff[bigbufsize];
char sp_buff[bufsize];
char lf_buff[2*bufsize];

char date[32]; /* For date-time started and shut down */

gpr_$pixel_value_t pixel_a[pinsize * graphsize];
/* Instead of 'gpr_$pixel_array_t pixel_a;' since that makes it 131073 big */
/* And instead of declaring it in the routine, since that makes the array to be stored in stack */
/* We need it 24 * 4096 = 98304 big. What is so special about 131073? */

/* =========================================================================== */

db_obj generic_$init (
  db_obj iface_db,
  status_$t *status)

{
    float r1,r2;
    long i,j,i_o_mode;
    char *dev_namp,*printer_modelp,*config_namep;
    long dev_num;
    static db_obj ren_db;
#if SR_level == SR10_2 || SR_level == SR10_3
    db_obj ext_db;
#endif
#if x_SPE_installed
    pio_$status_check_t stchk;
#endif

    (*status).all = status_$ok;
    line_number = 0;
    need_lf = false;
    page_number = 0;
    total_chars_read = 0;
    total_chars_job = 0;
    total_lines_job = 0;
    total_pages_job = 0;
    total_jobs_uasc = 0;
    total_jobs_os = 0;
    total_jobs_ftn = 0;
    total_jobs_trans = 0;
    total_jobs_bm1 = 0;
    total_jobs_other = 0;
    total_rejected_ascii = 0;
    total_rejected_trans = 0;
    total_rejected_bm1 = 0;
    total_chars_ascii = 0;
    total_lines_ascii = 0;
    total_pages_ascii = 0;
    total_chars_trans = 0;
    total_chars_bm1 = 0;
    total_lines_bm1 = 0;
    total_pages_bm1 = 0;


    /* Initialize string pointers (only to avoid compiler warning messages) */
    dev_namp = "";
    printer_modelp = "";
    config_namep = "";


#if x_debug
#if x_debug_db
    printf ("\n\nContents of iface_db:\n");
    print_entries_c (iface_db);

    printf ("\n\nContents of default_db:\n");
    print_entries_c (default_db);

    printf ("\n\nContents of control_db:\n");
    print_entries_c (control_db);
#endif
#endif


    /* create the render_db */
    ren_db = alloc_db ();
    if (! ren_db)
	{
	generic_$print_error (*status,"init","Unable to allocate render_db","",(long)Off_);
	return (0);
	}


    /* create the user_db */
    usr_db = alloc_db ();
    if (! usr_db)
	{
	generic_$print_error (*status,"init","Unable to allocate user_db","",(long)Off_);
	return (0);
	}


    /* I/O device name */
    *status = generic_$get_string (iface_db,"iface_db",Device__Name,&dev_namp,false,(long)Off_);
    if ((*status).all) return (0);

    i = prsvr__Io_Serial; /* default Io__Mode */
    if (generic_$strcmp(dev_namp,"/dev/sio1",false)) dev_num = 1;
    else if (generic_$strcmp(dev_namp,"/dev/sio2",false)) dev_num = 2;
    else if (generic_$strcmp(dev_namp,"/dev/sio3",false)) dev_num = 3;
    else if (generic_$strcmp(dev_namp,"/dev/pio",false)) { dev_num = -1; i = prsvr__Io_Parallel; }
    else dev_num = 0;
    r1 = dev_num;
    *status = generic_$add_value (usr_db,"usr_db","Device__Number",r1,(long)-1,(long)3,(long)Off_);
    if ((*status).all) return (0);


    /* I/O mode */
#if SR_level == SR10_0 || SR_level == SR10_1
    *status = generic_$get_value (iface_db,"iface_db",Io__Mode,&r1,&i_o_mode,false,(long)0,(long)prsvr__Io_Serial,(long)prsvr__Io_GPIO,i,(long)Off_);
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
    *status = generic_$get_value (iface_db,"iface_db",Io__Mode,&r1,&i_o_mode,true,(long)0,(long)prsvr__Io_Serial,(long)prsvr__Io_FILE,i,(long)Off_);
#endif
    if ((*status).all) return (0);
    r1 = (float)i_o_mode;
    *status = generic_$add_value (usr_db,"usr_db","Io__Mode",r1,(long)-10,(long)10,(long)Off_);
    if ((*status).all) return (0);



    if (dev_num >= 1 && dev_num <= 3 && i_o_mode == prsvr__Io_Serial)
	{
#if x_debug
	fprintf (stderr,"Using serial line %s\n",dev_namp);
#endif


	*status = generic_$get_value (iface_db,"iface_db",Device__Speed,&r1,&i,false,(long)0,(long)10,(long)30000,(long)1200,(long)Off_);
	if ((*status).all) return (0);
	*status = generic_$add_value (usr_db,"usr_db","Device__Speed",r1,(long)10,(long)30000,(long)Off_);
	if ((*status).all) return (0);
	}


#if x_SPE_installed
    else if (dev_num == -1 && i_o_mode == prsvr__Io_Parallel)
	{
#if x_debug
	fprintf (stderr,"Using parallel line %s\n",dev_namp);
#endif
	}
#endif


    else
	{
	(*status).all = status_$ok;
#if x_SPE_installed
	generic_$print_error (*status,"init",
	    "This printer driver generic can only be used on a SIO or SPE line","",(long)Off_);
#else
	generic_$print_error (*status,"init",
	    "This printer driver generic can only be used on a SIO line","",(long)Off_);
#endif
	fprintf (stderr,"    Not on device name = %s\n",dev_namp);
	fprintf (stderr,"    With io_mode = %d\n",i_o_mode);
	return (0);
	}


    /* figure out printer model */
    *status = generic_$get_string (iface_db,"iface_db",Printer__Model,&printer_modelp,false,(long)Off_);
    if ((*status).all) return (0);
/*xxxxx Add new printer here xxxxx*/
    if (generic_$strcmp(printer_modelp,"dumb_ascii",true))			printer_model = dumb_ascii;
    else if (generic_$strcmp(printer_modelp,"epson_lq800",true))		printer_model = epsonlq800;
    else if (generic_$strcmp(printer_modelp,"epson_lq1000",true))		printer_model = epsonlq1000;
    else if (generic_$strcmp(printer_modelp,"epson_lq1500",true))		printer_model = epsonlq1500;
    else if (generic_$strcmp(printer_modelp,"epson_lx80",true))			printer_model = epsonlx80;
    else if (generic_$strcmp(printer_modelp,"epson_fx80",true))			printer_model = epsonfx80;
    else if (generic_$strcmp(printer_modelp,"epson_fx85",true))			printer_model = epsonfx85;
    else if (generic_$strcmp(printer_modelp,"epson_fx105",true))		printer_model = epsonfx105;
    else if (generic_$strcmp(printer_modelp,"apple_imagewriter",true))		printer_model = imagewriter;
    else if (generic_$strcmp(printer_modelp,"apple_imagewriter_II",true))	printer_model = imagewriterII;
    else if (generic_$strcmp(printer_modelp,"HP_LaserJet",true))		printer_model = HPlaserjet;
    else if (generic_$strcmp(printer_modelp,"HP_LaserJetPlus",true))		printer_model = HPlaserjetplus;
    else if (generic_$strcmp(printer_modelp,"microline_84",true))		printer_model = microline84;
    else if (generic_$strcmp(printer_modelp,"microline_380",true))		printer_model = microline380;
    else if (generic_$strcmp(printer_modelp,"microline_390",true))		printer_model = microline390;
    else if (generic_$strcmp(printer_modelp,"microline_391",true))		printer_model = microline391;
    else if (generic_$strcmp(printer_modelp,"NEC_pinwriter_p5",true))		printer_model = pinwriterp5;
    else if (generic_$strcmp(printer_modelp,"star_nx1000",true))		printer_model = starnx1000;
    else printer_model = -1;

    r1 = printer_model;
    *status = generic_$add_value (usr_db,"usr_db","Printer__Model",r1,(long)0,(long)last_model,(long)Off_);
    if ((*status).all) return (0);


    /* config file name */
    *status = generic_$get_string (iface_db,"iface_db",Ddf__Name,&config_namep,false,(long)Off_);
    if ((*status).all) return (0);

    *status = generic_$read_config (config_namep,ren_db,usr_db,i_o_mode,printer_model);
    if ((*status).all) return (0);


    i = strlen(dev_namp);
    out_id = ios_$open (dev_namp,(short)i,ios_$write_opt,status);
    if ((*status).all)
	{
	sprintf (genbuff,"Unable to open output device %.40s",dev_namp);
	generic_$cleanup_string (genbuff);
	generic_$print_error (*status,"init",genbuff,"",(long)Off_);
	return (0);
	}


    *status = generic_$init_line ();
    if ((*status).all) return (0);


#if SR_level == SR10_2 || SR_level == SR10_3

    /* define new options
    Extensible option definitions (prflib syntax): 'prf -new_option [val]'.
    You should load these either as strings or as values: any values typed
    on the prf command line are not checked against the allowed values
    specified here; you might as well set the null string and the value 0.
    You will not get anything at all (in obj_db) if you load a range.
    If you do not specify a value on the prf command line, you will get the
    null string (for strings) or 0.0 (for values).
    (What about unbound values and strings, and integers?) */

    /* allocate a database to store extensible values in */
    ext_db = alloc_db ();
    if (! ext_db)
	{
	generic_$print_error (*status,"init","Unable to allocate ext_db","",(long)Off_);
	return (0);
	}

    /* load the key-values */
    r1 = 0.0;

    *status = generic_$add_string (ext_db,"ext_db","passwd","",(long)On_);
    if ((*status).all) return (0);

    *status = generic_$add_value (ext_db,"ext_db","tab_length",r1,(long)1,(long)0,(long)On_);
    if ((*status).all) return (0);

    *status = generic_$add_string (ext_db,"ext_db","wwrap","",(long)On_);
    if ((*status).all) return (0);

    *status = generic_$add_string (ext_db,"ext_db","eightbit","",(long)On_);
    if ((*status).all) return (0);

    *status = generic_$add_string (ext_db,"ext_db","ctrl_char","",(long)On_);
    if ((*status).all) return (0);

    *status = generic_$add_string (ext_db,"ext_db","graph_border","",(long)On_);
    if ((*status).all) return (0);

    *status = generic_$add_string (ext_db,"ext_db","reverse_bw","",(long)On_);
    if ((*status).all) return (0);

    *status = generic_$add_string (ext_db,"ext_db","italics","",(long)On_);
    if ((*status).all) return (0);

    *status = generic_$add_string (ext_db,"ext_db","double_height","",(long)On_);
    if ((*status).all) return (0);

    /* store the extensible option database in the driver database */
    *status = generic_$add_value (ren_db,"ren_db",Ext__Db,*((float *)&ext_db),(long)1,(long)0,(long)On_);
    if ((*status).all) return (0);

#endif

    /* remember date-time started */
    generic_$date_time (date);

    fprintf (stderr,"\nPrint server started (i.e. generic_$init completed) on %s\n\n",date);

    return (ren_db);
}

/* =========================================================================== */

status_$t generic_$render (
  int* obj_id,
  db_obj obj_db)

{
    long i,j,k,n,bytes,buff_last,buff_pos,copy;
    long start0,start1; /* Position where a group of zeros and non-zeros starts */
    long n1,n2,n3,n4,n5,n6,n7,n8;
    long char_pos,tab_length,ctrl_char;
    long big_job_chars,big_job_lines,big_job_pages;
    boolean reject_big_jobs;
    long copies,mngr_type,resolution,form_feed;
    long buff_poss[bufsize],char_poss[bufsize];
    long imode,x_dpi,y_dpi,x_pin,y_pin,i_fac,mult,mult1;
    long left_dot,right_dot,top_dot,bottom_dot;
    long dots_wide,dots_long,lines_xtra;
    long pixel_a_last;
    long reverse_bw,reverse_bw_default;
    long pixels_total,pixels_dark;
    long buff_pos_save,line_number_save;
    long funny_characters;
    long char_pos_max,char_pos_save,bufflen;
    boolean graph_border;
    long gaps;
    float paper_width,paper_length,point,cpi,lpi;
    float left_marg,right_marg,top_marg,bottom_marg;
    float r0;
    boolean wrap,word_wrap,use_cr_without_lf,use_tab,bw_reverse,landscape;
    long use_bs;
    boolean line_has_bs;
    char c,eightbit;
    char *file_typep;
    boolean ftype_uasc,ftype_os,ftype_ftn,ftype_trans,ftype_bm1;
    boolean header_valid,footer_valid;
    char *headerp,*footerp;
    ios_$id_t in_strid;
    uid_$t type_uid;
    status_$t status,status2;
    gpr_$rgb_plane_t hi_plane,my_hi_plane;
    gpr_$bitmap_desc_t bitmap_desc,my_bitmap_desc;
    gpr_$offset_t bitmap_size,my_bitmap_size;
    gpr_$window_t src_window,my_window;
    gpr_$attribute_desc_t attribute_desc;
    gpr_$raster_op_t raster_op;
    my_gpr_$rop_prim_set_t prim_set;
#if SR_level == SR10_0 || SR_level == SR10_1
    boolean allow_trans,allow_bold,allow_lq;
#endif
    boolean boldface,letter_quality;
#if SR_level == SR10_2 || SR_level == SR10_3
    boolean passwd_ok;
    boolean italics,double_height;
#endif

    char *strp;
    db_data_t db_data;
    db_field_t db_field;
    char dummyhead[10],dummyfoot[10];

/*zzzzz Currently unused features
    long columns,color,page_reverse,paper_type;
    float magnification;
    char *fontp,*src_namep;
    boolean collate_copies;
zzzzz*/


    /* Initialize string pointers (only to avoid compiler warning messages) */
    file_typep = "";
    strp = "";
/*zzzzz Currently unused features
    fontp = "";
    src_namep = "";
zzzzz*/


#if x_debug
#if x_debug_db
    printf ("\n\nContents of obj_db:\n");
    print_entries_c (obj_db);

    printf ("\n\nContents of usr_db:\n");
    print_entries_c (usr_db);
#endif
#endif


    line_number = 0;
    need_lf = false;
    page_number = 0;
    chars_line = 0;
    chars_marg = 0;
    lines_paper = 0;
    lines_high = 0;
    lines_marg = 0;
    lines_page = 0;
    total_chars_read = 0;
    total_chars_job = 0;
    total_lines_job = 0;
    total_pages_job = 0;


    status = generic_$get_value (usr_db,"usr_db","Reinit__Line",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
    if (j == On_)
	{
	status = generic_$init_line ();
	if (status.all) goto close_input_stream;
	}


    status = generic_$reset_printer ();
    if (status.all) goto close_input_stream;


    for (i = 0; i < bufsize; i++) sp_buff[i] = chr_sp;

    my_bitmap_desc = gpr_$nil_bitmap_desc;
    attribute_desc = gpr_$nil_attribute_desc;

    status = generic_$get_value (usr_db,"usr_db","Use__Formfeed",&r0,&j,false,(long)0,(long)0,(long)1,(long)1,(long)On_);
    if (j == On_)
	{
	status = generic_$get_value (usr_db,"usr_db","Use__LF_before_FF",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)On_);
	if (j == On_) ff_bufflen = 2;
	else ff_bufflen = 1;
	}
    else ff_bufflen = 0;

    status = generic_$get_value (usr_db,"usr_db","End_of_Line",&r0,&j,false,(long)-1,(long)eol_cr,(long)eol_lf_cr,(long)eol_lf,(long)On_);
	 if (j == eol_cr)	{ lf_bufflen = 1; for (i = 0; i < bufsize; i++) lf_buff[i] = chr_cr; }
    else if (j == eol_lf)	{ lf_bufflen = 1; for (i = 0; i < bufsize; i++) lf_buff[i] = chr_lf; }
    else if (j == eol_cr_lf)	{ lf_bufflen = 2; for (i = 0; i < 2 * bufsize; i++) lf_buff[i] = chr_cr; i++; lf_buff[i] = chr_lf; }
    else if (j == eol_lf_cr)	{ lf_bufflen = 2; for (i = 0; i < 2 * bufsize; i++) lf_buff[i] = chr_lf; i++; lf_buff[i] = chr_cr; }


    status = generic_$get_value (obj_db,"obj_db",Mngr__Type,&r0,&mngr_type,false,(long)0,(long)mgr__ios,(long)mgr__tml,(long)mgr__ios,(long)On_);


    /*yyyyy Filter bmsc$bm1 forgets to re-set file type. Or maybe they are really the same ??? yyyyy*/
    sprintf (lilbuff,"#%s#%s#%s#%s#%s#%s#",UASC_,OS_,FTN_,TRANS_,BM1_,BMSC_);
    status = generic_$check_string (obj_db,"obj_db",F__Type,&file_typep,false,lilbuff,&j,(long)On_);
    if (j == -100) goto close_input_stream;
    ftype_uasc =  false;
    ftype_os =    false;
    ftype_ftn =   false;
    ftype_trans = false;
    ftype_bm1 =   false;
	 if (j == 0)			{ftype_uasc  = true; total_jobs_uasc++;}
    else if (j == 1)			{ftype_os    = true; total_jobs_os++;}
    else if (j == 2)			{ftype_ftn   = true; total_jobs_ftn++;}
    else if (j == 3)			{ftype_trans = true; total_jobs_trans++;}
    else if ((j == 4) || (j == 5))	{ftype_bm1   = true; total_jobs_bm1++;}
    else
	{
	sprintf (genbuff,"Wrong file type %.40s",file_typep);
	generic_$cleanup_string (genbuff);
	status.all = status_$ok;
	generic_$print_error (status,"render",genbuff,"Cannot print this type of file",(long)On_);
	total_jobs_other++;
	goto close_input_stream;
	}


#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$get_string (obj_db,"obj_db","passwd",&strp,true,(long)On_);
    status = generic_$check_passwd (strp,&passwd_ok);
#endif


    status = generic_$get_value (obj_db,"obj_db",Copies_,&r0,&copies,false,(long)0,(long)1,(long)10,(long)1,(long)On_);
#if SR_level == SR10_0
    /*yyyyy Banner page is sent with Copies_ set to 0 yyyyy*/
#endif

    status = generic_$get_value (obj_db,"obj_db",Page__Layout,&r0,&j,false,(long)0,(long)Portrait_,(long)Landscape_,(long)Portrait_,(long)On_);
    /*yyyyy Banner page is sent with Page__Layout set to 0 yyyyy*/
    landscape = (j == Landscape_);


#if SR_level == SR10_0 || SR_level == SR10_1
    status = generic_$get_value (obj_db,"obj_db",Form__Feeds,&r0,&form_feed,false,(long)-1,(long)1,(long)5,(long)1,(long)On_);
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$get_value (obj_db,"obj_db",Form__Feeds,&r0,&form_feed,false,(long)0,(long)1,(long)5,(long)1,(long)On_);
    /*yyyyy form_feed set to 0 for banner pages yyyyy*/
#endif


    status = generic_$get_value (obj_db,"obj_db",Paper__Length,&paper_length,&j,false,(long)0,(long)1,(long)20,(long)11,(long)On_);


#if SR_level == SR10_0
    status = generic_$get_value (obj_db,"obj_db",Lpi_,&lpi,&j,false,(long)1,(long)1,(long)20,(long)6,(long)On_);
    /*yyyyy lpi is set to lowest (or first?) available (1.0), unless you specify it on prf command yyyyy*/
#endif
#if SR_level == SR10_1
    status = generic_$get_value (obj_db,"obj_db",Lpi_,&lpi,&j,true,(long)0,(long)1,(long)20,(long)6,(long)On_);
    /*yyyyy lpi is not in the database, unless you specify it on prf command yyyyy*/
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$get_value (obj_db,"obj_db",Lpi_,&lpi,&j,false,(long)0,(long)1,(long)20,(long)6,(long)On_);
#endif


    status = generic_$get_value (obj_db,"obj_db",Top__Margin,&top_marg,&j,false,(long)0,(long)0,(long)20,(long)0,(long)On_);


    status = generic_$get_value (obj_db,"obj_db",Bottom__Margin,&bottom_marg,&j,false,(long)0,(long)0,(long)20,(long)0,(long)On_);


    status = generic_$get_value (obj_db,"obj_db",Paper__Width,&paper_width,&j,false,(long)0,(long)1,(long)20,(long)8,(long)On_);


    status = generic_$get_value (obj_db,"obj_db",Point_,&point,&j,false,(long)0,(long)4,(long)30,(long)12,(long)On_);

    /* Fixed width font. Character width = 0.6 points, or chars_per_inch = 120/point_size */
#if SR_level == SR10_0 || SR_level == SR10_1
    cpi = 120.0 / point;
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$get_value (obj_db,"obj_db",Cpi_,&cpi,&j,true,(long)0,(long)4,(long)30,(long)-1,(long)On_);
    /*yyyyy cpi is not in the database, unless you specify it on prf command, and point should be used if not found yyyyy*/

    if ( cpi < 4.0 ) cpi = 120.0 / point;
#endif


    status = generic_$get_value (obj_db,"obj_db",Left__Margin,&left_marg,&j,false,(long)0,(long)0,(long)20,(long)0,(long)On_);


    status = generic_$get_value (obj_db,"obj_db",Right__Margin,&right_marg,&j,false,(long)0,(long)0,(long)20,(long)0,(long)On_);


    status = generic_$get_value (obj_db,"obj_db",Wrap_,&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)On_);
    wrap = (j == On_);

    status = generic_$get_value (usr_db,"usr_db","Word__Wrap",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)On_);
    word_wrap = (j == On_);


#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$check_string (obj_db,"obj_db","wwrap",&strp,true,"#off#on#char#character#word#",&j,(long)On_);
    if (j == 0) wrap = false;
    else if (j >= -1)
	{
	wrap = true;
	if (j == 2 || j == 3) word_wrap = false;
	else word_wrap = true;
	}
#endif


#if SR_level == SR10_0
    status = generic_$get_value (obj_db,"obj_db",Print__Weight,&r0,&j,true,(long)0,(long)Light_weight,(long)Bold_weight,(long)Medium_weight,(long)On_);
    /*yyyyy Print__Weight is never in the database yyyyy*/
#endif
#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$get_value (obj_db,"obj_db",Print__Weight,&r0,&j,true,(long)0,(long)Light_weight,(long)Bold_weight,(long)Medium_weight,(long)On_);
    /*yyyyy Print__Weight is not in the database for banner pages yyyyy*/
#endif
    boldface = (j == Bold_weight);


#if SR_level == SR10_0
    status = generic_$get_value (obj_db,"obj_db",Lq_,&r0,&j,true,(long)0,(long)Draft_lq,(long)NLQ_lq,(long)Draft_lq,(long)On_);
    /*yyyyy Lq_ is never in the database yyyyy*/
#endif
#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$get_value (obj_db,"obj_db",Lq_,&r0,&j,true,(long)0,(long)Draft_lq,(long)NLQ_lq,(long)Draft_lq,(long)On_);
    /*yyyyy Lq_ is not in the database for banner pages yyyyy*/
#endif
    letter_quality = (j == NLQ_lq);


    status = generic_$get_value (obj_db,"obj_db",Resolution_,&r0,&resolution,false,(long)0,(long)1,(long)1000,(long)60,(long)On_);


    status = generic_$get_value (usr_db,"usr_db","Use__CR_without_LF",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)On_);
    use_cr_without_lf = (j == On_);


    status = generic_$get_value (usr_db,"usr_db","Use__Backspace",&r0,&use_bs,false,(long)0,(long)0,(long)one_at_a_time,(long)0,(long)On_);
    if (use_bs == Off_ && use_cr_without_lf) use_bs = use_cr_instead;


    status = generic_$get_value (usr_db,"usr_db","Use__Tab",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)On_);
    use_tab = (j == On_);


    status = generic_$get_value (usr_db,"usr_db","Tab__Length",&r0,&tab_length,false,(long)0,(long)1,(long)20,(long)8,(long)On_);


#if SR_level == SR10_2 || SR_level == SR10_3
    j = tab_length;
    status = generic_$get_value (obj_db,"obj_db","tab_length",&r0,&tab_length,true,(long)0,(long)1,(long)20,j,(long)On_);
#endif


    status = generic_$get_value (usr_db,"usr_db","Eightbit_",&r0,&j,false,(long)-1,(long)0,(long)chr_del - 1,(long)*"~",(long)On_);
    eightbit = (char)j;


#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$get_string (obj_db,"obj_db","eightbit",&strp,true,(long)On_);
    if (!generic_$strcmp(strp,"not_present",false))
	{
	     if (generic_$strcmp(strp,"ignore",true))	eightbit = eightbit_ignore;
	else if (generic_$strcmp(strp,"",true))		eightbit = eightbit_ignore;
	else if (generic_$strcmp(strp,"blank",true))	eightbit = eightbit_blank;
	else if (generic_$strcmp(strp,"value",true))	eightbit = eightbit_value;
	else if (generic_$strcmp(strp,"convert",true))	eightbit = eightbit_convert;
	else if (strlen(strp) == 1 && *strp > chr_sp && *strp < chr_del) eightbit = *strp;
	else generic_$print_wrong_value ("eightbit",strp);
	}
#endif


    status = generic_$get_value (usr_db,"usr_db","Ctrl__Char",&r0,&ctrl_char,false,(long)0,(long)ctrl_ignore,(long)ctrl_caret,(long)0,(long)Off_);


#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$check_string (obj_db,"obj_db","ctrl_char",&strp,true,"#ignore#blank#value#ascii#control#caret#",&j,(long)On_);
    if (j >= 0) ctrl_char = j;
    else if (j == -1) ctrl_char = ctrl_ignore;
#endif


    status = generic_$get_value (usr_db,"usr_db","Big__Job_chars",&r0,&big_job_chars,false,(long)0,(long)2,(long)huge_job_chars,(long)30,(long)Off_);
    big_job_chars = 1024 * big_job_chars;


    status = generic_$get_value (usr_db,"usr_db","Big__Job_lines",&r0,&big_job_lines,false,(long)0,(long)50,(long)huge_job_lines,(long)600,(long)Off_);


    status = generic_$get_value (usr_db,"usr_db","Big__Job_pages",&r0,&big_job_pages,false,(long)0,(long)1,(long)huge_job_pages,(long)20,(long)Off_);


    status = generic_$get_value (usr_db,"usr_db","Reject__Big_jobs",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
    reject_big_jobs = (j == On_);


#if SR_level == SR10_0 || SR_level == SR10_1
    status = generic_$get_value (usr_db,"usr_db","Allow__TRANS",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
    allow_trans = (j == On_);


    status = generic_$get_value (usr_db,"usr_db","Allow__Bold",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
    allow_bold = (j == On_);


    status = generic_$get_value (usr_db,"usr_db","Allow__Lq",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
    allow_lq = (j == On_);
#endif


    status = generic_$get_value (usr_db,"usr_db","Graph__Border",&r0,&j,false,(long)0,(long)Off_,(long)On_,(long)Off_,(long)On_);
    graph_border = (j == On_);


#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$check_string (obj_db,"obj_db","graph_border",&strp,true,"#off#on#",&j,(long)On_);
	 if (j == off) graph_border = false;
    else if (j == on)  graph_border = true;
    else if (j == -1)  graph_border = (! graph_border);
#endif


    status = generic_$get_value (usr_db,"usr_db","Reverse__BW",&r0,&reverse_bw,false,(long)0,(long)Off_,(long)dark,(long)Off_,(long)On_);
    reverse_bw_default = reverse_bw;


    status = generic_$get_value (obj_db,"obj_db",BW__reverse,&r0,&j,false,(long)0,(long)Off_,(long)On_,(long)Off_,(long)On_);
    bw_reverse = (j == On_);


#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$check_string (obj_db,"obj_db","reverse_bw",&strp,true,"#off#on#light#dark#",&j,(long)On_);
	 if (j == off)	{ reverse_bw = Off_; bw_reverse = false; }
    else if (j == on)	{ reverse_bw = Off_; bw_reverse = true; }
    else if (j == light)  reverse_bw = light;
    else if (j == dark)	  reverse_bw = dark;
    else if (j == -1)	{ if (reverse_bw == light) reverse_bw = dark;
		     else if (reverse_bw == dark)  reverse_bw = light;
		     else { reverse_bw = Off_; bw_reverse = (! bw_reverse); } }


    status = generic_$check_string (obj_db,"obj_db","italics",&strp,true,"#off#on#",&j,(long)On_);
    italics = (j == 1 || j == -1);


    status = generic_$check_string (obj_db,"obj_db","double_height",&strp,true,"#off#on#",&j,(long)On_);
    double_height = (j == 1 || j == -1);
#endif


    header_valid = false;
    footer_valid = false;
    headerp = "";
    footerp = "";

/*yyyyy
    Headers and footers don't work properly.

    You cannot use get_db_string_c for them: either entries are not found
    (for status page) or get blank strings (for banner pages, or if
    -headers on is not specified explicitly), or get error
    (stcode = 04020017) - wrong type specified for key  (US/print utility)
    since entries are of three strings.

    status = generic_$get_string (obj_db,"obj_db",Header_,&headerp,true,(long)On_);
#if x_debug
#if x_debug_head
    fprintf (stderr,"Header string is %s\n",headerp);
#endif
#endif
yyyyy*/


#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
    /*yyyyy
    However you can use inq_entry_c for them. The string db_data.v.str will then
    have not a single null byte at the end, but three null bytes at the ends of
    each substring.

    It appears that the filename specifier character '!' works only occasionally,
    and never completely right.

    The page number specifier character '#' gets across as the formfeed character.
    (Used to be the two-character string '/f' under SR10.0 .)

    Luckily the dates (and characters escaped with @) come across already expanded.

    Whenever headers or footers are in use, entries psc_head and psc_foot also seem
    to be present in the obj_db. Are these reliable? These contain the strings as typed
    by the user (so we would have a hard time figuring out the dates etc).

    Under SR10.0, could not even use inq_entry: invariably we ended up with error
    reference to illegal address (OS/MST manager)
    This error was generated OUTSIDE of the render routine.
    yyyyy*/


#if SR_level == SR10_3
    /*yyyyy There is no #define'd name for headers_on_off yyyyy*/
    status = generic_$get_integer_value (obj_db,"obj_db","headers_on_off",&j,true,(long)0,(long)Off_,(long)On_,(long)Off_,(long)On_);
    if (j == On_)
	{
#endif


	for (i = 0; i < 10; i++) dummyhead[i] = chr_nul;
	db_data.num_entries = 3;
	db_field = String_Field;
	db_data.type = &db_field;
	db_data.v.str = dummyhead;
	status = inq_entry_c (obj_db,Header_,&db_data);
	if (status.all)
	    {
	    if (status.all != print_$no_db_entry) generic_$print_find_error (status,"render","Header_","obj_db","none",(long)On_);
	    }
	else if (db_data.num_entries == 3 && *db_data.type == String_Field)
	    {
	    headerp = db_data.v.str;
	    strp = headerp;
	    for (i = 0, j = 0; i < bufsize && j < 3; i++, strp++)
		{
		if (*strp == chr_nul) j++;
		}
	    if (i > 3 && j == 3) header_valid = true;
	    }
#if x_debug
#if x_debug_head
	if (header_valid)
	    {
	    fprintf (stderr,"Header string left is  <%s>\n",headerp);
	    i = strlen(headerp) + 1;
	    fprintf (stderr,"Header string center is <%s>\n",headerp + i);
	    i = i + strlen(headerp + i) + 1;
	    fprintf (stderr,"Header string right is <%s>\n",headerp + i);
	    }
	else
	    {
	    fprintf (stderr,"Header string is not defined\n");
	    }
#endif
#endif


	for (i = 0; i < 10; i++) dummyfoot[i] = chr_nul;
	db_data.num_entries = 3;
	db_field = String_Field;
	db_data.type = &db_field;
	db_data.v.str = dummyfoot;
	status = inq_entry_c (obj_db,Footer_,&db_data);
	if (status.all)
	    {
	    if (status.all != print_$no_db_entry) generic_$print_find_error (status,"render","Footer_","obj_db","none",(long)On_);
	    }
	else if (db_data.num_entries == 3 && *db_data.type == String_Field)
	    {
	    footerp = db_data.v.str;
	    strp = footerp;
	    for (i = 0, j = 0; i < bufsize && j < 3; i++, strp++)
		{
		if (*strp == chr_nul) j++;
		}
	    if (i > 3 && j == 3) footer_valid = true;
	    }
#if x_debug
#if x_debug_head
	if (footer_valid)
	    {
	    fprintf (stderr,"Footer string left is  <%s>\n",footerp);
	    i = strlen(footerp) + 1;
	    fprintf (stderr,"Footer string center is <%s>\n",footerp + i);
	    i = i + strlen(footerp + i) + 1;
	    fprintf (stderr,"Footer string right is <%s>\n",footerp + i);
	    }
	else
	    {
	    fprintf (stderr,"Footer string is not defined\n");
	    }
#endif
#endif


#if SR_level == SR10_3
	}
#if x_debug
#if x_debug_head
    else
	{
	fprintf (stderr,"headers_on_off is not On_, no headers or footers\n");
	}
#endif
#endif
#endif

#endif


/*zzzzz Currently unused features
    status = generic_$get_value (obj_db,"obj_db",Columns_,&r0,&columns,false,(long)0,(long)1,(long)1,(long)1,(long)On_);


    status = generic_$get_value (obj_db,"obj_db",PageRev_,&r0,&page_reverse,false,(long)0,(long)Off_,(long)Off_,(long)Off_,(long)On_);


    status = generic_$get_value (obj_db,"obj_db",Paper__Type,&r0,&paper_type,false,(long)0,(long)Cut_,(long)Transpar_,(long)FanFold_,(long)On_);


    status = generic_$get_value (obj_db,"obj_db",Color_,&r0,&color,false,(long)0,(long)Off_,(long)Off_,(long)Off_,(long)On_);


    Magnification for gpr files is 1; for gmf files it gives
    same sized image as on the screen. (Assumes screen is 100dpi.
    Thus if you ask for resolution 300, you get magnification 3,
    but at resolution 60 you get magnification 0.6.)
    status = generic_$get_value (obj_db,"obj_db",Magnification_,&magnification,&j,false,(long)0,(long)-1,(long)20,(long)1,(long)On_);


    status = generic_$get_string (obj_db,"obj_db",Fonts_,&fontp,false,(long)On_);


    (name of source file; not present in database for banner pages)
    status = generic_$get_string (obj_db,"obj_db",Src_Name_,&src_namep,true,(long)On_);


    status = generic_$get_value (obj_db,"obj_db",ColCop_,&r0,&j,false,(long)0,(long)0,(long)10,(long)0,(long)On_);
    collate_copies = (j == On_);
zzzzz*/


    if (ftype_uasc || ftype_os || ftype_ftn)
	    {
#if x_debug
	    fprintf (stderr,"Printing %s file\n",file_typep);
#endif

	    if (mngr_type != mgr__ios)
		{
		sprintf (genbuff,"Wrong mngr_type = %d, must be %d for IOS",mngr_type,mgr__ios);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"",(long)On_);
		status.all = print_$file_device_mismatch;
		goto close_input_stream;
		}

	    in_strid = (ios_$id_t)*obj_id;


#if SR_level == SR10_2 || SR_level == SR10_3
	    if (! passwd_ok)
		{
		strp = "";
		     if (boldface)		strp = "boldface";
		else if (letter_quality)	strp = "letter quality";
		else if (italics)		strp = "italics";
		else if (double_height)		strp = "double height";
		else if (cpi < 9.9)		strp = "-cpi less than 10 (or -point greater than 12)";
		else if (lpi < 5.9)		strp = "-lpi less than 6";
		else if (lpi > 8.1)		strp = "-lpi greater than 8";

		if (!generic_$strcmp(strp,"",true))
		    {
		    generic_$print_need_passwd (strp);
		    status.all = status_$ok;
		    total_rejected_ascii++;
		    goto close_input_stream;
		    }
		}
#endif


	    /*yyyyy Check that file type is uasc or unstruct. Should have been done by prf command yyyyy*/

	    ios_$inq_type_uid (in_strid,&type_uid,&status);
	    if (status.all)
		generic_$print_error (status,"render","Unable to find type_uid of file","",(long)Off_);
	    else
		if (! (type_uid == uasc_$uid || type_uid == unstruct_$uid))
		    {
		    strp = "unknown";
			 if (type_uid == case_hm_$uid)		strp = "case_hm";
		    else if (type_uid == cmpexe_$uid)		strp = "cmpexe";
		    else if (type_uid == coff_$uid)		strp = "coff";
		    else if (type_uid == d3m_area_$uid)		strp = "d3m_area";
		    else if (type_uid == d3m_sch_$uid)		strp = "d3m_sch";
		    else if (type_uid == directory_$uid)	strp = "directory";
		    else if (type_uid == dm_edit_$uid)		strp = "dm_edit";
		    else if (type_uid == hdr_undef_$uid)	strp = "hdr_undef";
		    else if (type_uid == input_pad_$uid)	strp = "input_pad";
		    else if (type_uid == mbx_$uid)		strp = "mbx";
		    else if (type_uid == mt_$uid)		strp = "mt";
		    else if (type_uid == nulldev_$uid)		strp = "nulldev";
		    else if (type_uid == object_file_$uid)	strp = "object_file";
		    else if (type_uid == pad_$uid)		strp = "pad";
		    else if (type_uid == pty_$slave_uid)	strp = "pty_slave";
		    else if (type_uid == pty_$uid)		strp = "pty";
		    else if (type_uid == records_$uid)		strp = "records";
		    else if (type_uid == sio_$uid)		strp = "sio";
		    else if (type_uid == tcp_$uid)		strp = "tcp";
		    sprintf (genbuff,"Files of type <%s> cannot be printed",strp);
		    generic_$print_error (status,"render",genbuff,"This print file rejected",(long)On_);
		    total_rejected_ascii++;
		    goto close_input_stream;
		    }


	    if (top_marg + bottom_marg + 2 > paper_length)
		{
		sprintf (genbuff,
		    "Crazy margins:\n      top_margin    = %f\n      bottom_margin = %f\n      paper_length  = %f",
		    top_marg,bottom_marg,paper_length);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"Using default 0 margins",(long)On_);
		line_number += 3;
		top_marg = 0;
		bottom_marg = 0;
		}

	    buff_pos = 0;

/*xxxxx Add new printer here xxxxx*/
	    /* Set lines_per_inch */
	    switch (printer_model)
	      {
	      case epsonlq800:
	      case epsonlq1000:
	      case epsonlq1500:
	      case microline380: /* No command to set n/360 inch line spacing? */
	      case pinwriterp5:
		lilbuff[buff_pos++] = chr_esc;
		if (lpi < 2)
		    {
		    i = generic_$float_to_long (60.0 / lpi);
		    lpi = 60.0 / (float)i;
		    lilbuff[buff_pos++] = *"A";
		    lilbuff[buff_pos++] = i;
		    }
		else
		    {
		    i = generic_$float_to_long (180.0 / lpi);
		    lpi = 180.0 / (float)i;
		    lilbuff[buff_pos++] = *"3";
		    lilbuff[buff_pos++] = i;
		    }
		break;
	      case epsonlx80:
	      case epsonfx80:
	      case epsonfx85:
	      case epsonfx105:
	      case starnx1000:
		lilbuff[buff_pos++] = chr_esc;
		if (lpi < 2)
		    {
		    i = generic_$float_to_long (72.0 / lpi);
		    lpi = 72.0 / (float)i;
		    lilbuff[buff_pos++] = *"A";
		    lilbuff[buff_pos++] = i;
		    }
		else
		    {
		    i = generic_$float_to_long (216.0 / lpi);
		    lpi = 216.0 / (float)i;
		    lilbuff[buff_pos++] = *"3";
		    lilbuff[buff_pos++] = i;
		    }
		break;
	      case imagewriter:
	      case imagewriterII:
		i = generic_$float_to_long (144.0 / lpi);
		if (i > 99) i = 99;
		lpi = 144.0 / (float)i;
		sprintf (lilbuff,"\033T%02.2d",i);
		buff_pos = 4;
		break;
	      case HPlaserjet:
	      case HPlaserjetplus:
		i = generic_$float_to_long (48.0 / lpi);
		lpi = 48.0 / (float)i;
		sprintf (lilbuff,"\033&l%dC",i);
		buff_pos = strlen(lilbuff);
		break;
	      case microline84:
		i = generic_$float_to_long (144.0 / lpi);
		if (i > 127) i = 127;
		lpi = 144.0 / (float)i;
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"%";
		lilbuff[buff_pos++] = *"9";
		lilbuff[buff_pos++] = i;
		break;
	      case microline390:
	      case microline391:
		lilbuff[buff_pos++] = chr_esc;
		if (lpi < 2)
		    {
		    i = generic_$float_to_long (60.0 / lpi);
		    lpi = 60.0 / (float)i;
		    lilbuff[buff_pos++] = *"A";
		    lilbuff[buff_pos++] = i;
		    }
		else
		    {
		    i = generic_$float_to_long (360.0 / lpi);
		    lpi = 360.0 / (float)i;
		    lilbuff[buff_pos++] = *"[";
		    lilbuff[buff_pos++] = i;
		    }
		break;
	      }

#if x_debug
#if x_debug_ctrl
	    fprintf (stderr,"Paper length is %f\n",paper_length);
	    fprintf (stderr,"Lines per inch is %f\n",lpi);
	    fprintf (stderr,"Top_margin is %f\n",top_marg);
	    fprintf (stderr,"Bottom_margin is %f\n",bottom_marg);
#endif
#endif

	    lines_paper = paper_length * lpi;
	    lines_high = (paper_length - bottom_marg) * lpi;
	    lines_marg = top_marg * lpi;
	    if (lines_paper > bufsize) lines_paper = bufsize;
	    if (lines_high > lines_paper) lines_high = lines_paper;
	    if (lines_marg >= lines_high) lines_marg = 0;
	    lines_page = lines_high - lines_marg;
#if x_debug
	    fprintf (stderr,"Print lines in page = %d, lines in margin = %d, total lines in paper = %d\n",lines_page,lines_marg,lines_paper);
#endif

/*xxxxx Add new printer here xxxxx*/
	    /* Set paper length */
	    switch (printer_model)
	      {
	      case epsonlq800:
	      case epsonlq1000:
	      case epsonlq1500:
	      case microline390:
	      case microline391:
	      case pinwriterp5:
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"C";
		lilbuff[buff_pos++] = chr_nul;
		lilbuff[buff_pos++] = generic_$float_to_long (paper_length);
		break;
	      case epsonlx80:
	      case epsonfx80:
	      case epsonfx85:
	      case epsonfx105:
	      case microline380:
	      case starnx1000:
		/*zzzzz Usually we use metric paper in fx-80, thus form length
		cannot be set as number of inches. Use number of lines instead.
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"C";
		lilbuff[buff_pos++] = chr_nul;
		lilbuff[buff_pos++] = generic_$float_to_long (paper_length); zzzzz*/
		if (lines_paper <= 127)
		    {
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"C";
		    lilbuff[buff_pos++] = lines_paper;
		    }
		break;
	      case imagewriterII:
		/* Old imagewriter cannot do */
		i = generic_$float_to_long (paper_length * 144);
		sprintf (&lilbuff[buff_pos],"\033H%04.4d",i);
		buff_pos += 6;
		break;
	      case HPlaserjet:
	      case HPlaserjetplus:
		sprintf (&lilbuff[buff_pos],"\033&l%dp0e%dF",lines_paper,lines_paper); /* Zero top or bottom margins */
		buff_pos += strlen(&lilbuff[buff_pos]);
		break;
	      case microline84:
		i = generic_$float_to_long (paper_length * 2);
		sprintf (&lilbuff[buff_pos],"\033G%02.2d",i);
		buff_pos += 4;
		break;
	      }


	    if (left_marg + right_marg + 2 > paper_width)
		{
		sprintf (genbuff,
		    "Crazy margins:\n      left_margin  = %f\n      right_margin = %f\n      paper_width  = %f",
		    left_marg,right_marg,paper_width);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"Using default 0 margins",(long)On_);
		line_number += 3;
		left_marg = 0;
		right_marg = 0;
		}


/*xxxxx Add new printer here xxxxx*/
	    /* Set character_per_inch */
	    switch (printer_model)
	      {
	      case epsonlq800:
	      case epsonlq1000:
		lilbuff[buff_pos++] = chr_esc;
		if (cpi < 5.5)
		    {
		    cpi = 5;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 6.75)
		    {
		    cpi = 6;
		    lilbuff[buff_pos++] = *"M";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 8.75)
		    {
		    cpi = 7.5;
		    lilbuff[buff_pos++] = *"g";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 11)
		    {
		    cpi = 10;
		    lilbuff[buff_pos++] = *"P";
		    }
		else if (cpi < 13.5)
		    {
		    cpi = 12;
		    lilbuff[buff_pos++] = *"M";
		    }
		else if (cpi < 16.05)
		    {
		    cpi = 15;
		    lilbuff[buff_pos++] = *"g";
		    }
		else if (cpi < 18.55)
		    {
		    cpi = 17.1;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_si;
		    }
		else
		    {
		    cpi = 20;
		    lilbuff[buff_pos++] = *"M";
		    lilbuff[buff_pos++] = chr_si;
		    }
		break;
	      case epsonlq1500:
	      case epsonlx80:
	      case pinwriterp5: /*zzzzz Not sure if could use 'chr_fs *"E" n' sequence: 1 double, 2 triple horizontal enlargement zzzzz*/
	      case starnx1000:
		lilbuff[buff_pos++] = chr_esc;
		if (cpi < 5.5)
		    {
		    cpi = 5;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 7.25)
		    {
		    cpi = 6;
		    lilbuff[buff_pos++] = *"M";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 9.25)
		    {
		    cpi = 8.5;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_si;
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 11)
		    {
		    cpi = 10;
		    lilbuff[buff_pos++] = *"P";
		    }
		else if (cpi < 14.55)
		    {
		    cpi = 12;
		    lilbuff[buff_pos++] = *"M";
		    }
		else if (cpi < 18.55)
		    {
		    cpi = 17.1;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_si;
		    }
		else
		    {
		    cpi = 20;
		    lilbuff[buff_pos++] = *"M";
		    lilbuff[buff_pos++] = chr_si;
		    }
		break;
	      case epsonfx80:
		lilbuff[buff_pos++] = chr_esc;
		if (cpi < 5.5)
		    {
		    cpi = 5;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 7.25)
		    {
		    cpi = 6;
		    lilbuff[buff_pos++] = *"M";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 9.25)
		    {
		    cpi = 8.5;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_si;
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 11)
		    {
		    cpi = 10;
		    lilbuff[buff_pos++] = *"P";
		    }
		else if (cpi < 14.55)
		    {
		    cpi = 12;
		    lilbuff[buff_pos++] = *"M";
		    }
		else
		    {
		    cpi = 17.1;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_si;
		    }
		break;
	      case epsonfx85:
	      case epsonfx105:
		lilbuff[buff_pos++] = chr_esc;
		if (cpi < 5.5)
		    {
		    cpi = 5;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 8)
		    {
		    cpi = 6;
		    lilbuff[buff_pos++] = *"M";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 11)
		    {
		    cpi = 10;
		    lilbuff[buff_pos++] = *"P";
		    }
		else if (cpi < 14.55)
		    {
		    cpi = 12;
		    lilbuff[buff_pos++] = *"M";
		    }
		else if (cpi < 18.55)
		    {
		    cpi = 17.1;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_si;
		    }
		else
		    {
		    cpi = 20;
		    lilbuff[buff_pos++] = *"M";
		    lilbuff[buff_pos++] = chr_si;
		    }
		break;
	      case imagewriter:
	      case imagewriterII:
		lilbuff[buff_pos++] = chr_esc;
		if (cpi < 4.75)
		    {
		    cpi = 4.5;
		    lilbuff[buff_pos++] = *"n";
		    lilbuff[buff_pos++] = chr_ctrl_n;
		    }
		else if (cpi < 5.5)
		    {
		    cpi = 5;
		    lilbuff[buff_pos++] = *"N";
		    lilbuff[buff_pos++] = chr_ctrl_n;
		    }
		else if (cpi < 6.35)
		    {
		    cpi = 6;
		    lilbuff[buff_pos++] = *"E";
		    lilbuff[buff_pos++] = chr_ctrl_n;
		    }
		else if (cpi < 7.1)
		    {
		    cpi = 6.7;
		    lilbuff[buff_pos++] = *"e";
		    lilbuff[buff_pos++] = chr_ctrl_n;
		    }
		else if (cpi < 8)
		    {
		    cpi = 7.5;
		    lilbuff[buff_pos++] = *"q";
		    lilbuff[buff_pos++] = chr_ctrl_n;
		    }
		else if (cpi < 8.75)
		    {
		    cpi = 8.5;
		    lilbuff[buff_pos++] = *"Q";
		    lilbuff[buff_pos++] = chr_ctrl_n;
		    }
		else if (cpi < 9.5)
		    {
		    cpi = 9;
		    lilbuff[buff_pos++] = *"n";
		    }
		else if (cpi < 11)
		    {
		    cpi = 10;
		    lilbuff[buff_pos++] = *"N";
		    }
		else if (cpi < 12.7)
		    {
		    cpi = 12;
		    lilbuff[buff_pos++] = *"E";
		    }
		else if (cpi < 14.2)
		    {
		    cpi = 13.4;
		    lilbuff[buff_pos++] = *"e";
		    }
		else if (cpi < 16)
		    {
		    cpi = 15;
		    lilbuff[buff_pos++] = *"q";
		    }
		else
		    {
		    cpi = 17;
		    lilbuff[buff_pos++] = *"Q";
		    }
		break;
	      case HPlaserjet:
		cpi = 10;
		sprintf(&lilbuff[buff_pos],"\033(8U\033(s0p10h12v0s0b3T");
		buff_pos += strlen(&lilbuff[buff_pos]);
		break;
	      case HPlaserjetplus:
		if (cpi < 13)
		    {
		    cpi = 10;
		    sprintf(&lilbuff[buff_pos],"\033(8U\033(s0p10h12v0s0b3T");
		    }
		else
		    {
		    cpi = 16.66;
		    sprintf(&lilbuff[buff_pos],"\033(8U\033(s0p16.66h8.5v0s0b0T");
		    }
		buff_pos += strlen(&lilbuff[buff_pos]);
		break;
	      case microline84:
		if (cpi < 5.5)
		    {
		    cpi = 5;
		    lilbuff[buff_pos++] = chr_rs;
		    lilbuff[buff_pos++] = chr_us;
		    }
		else if (cpi < 7.25)
		    {
		    cpi = 6;
		    lilbuff[buff_pos++] = chr_gs;
		    lilbuff[buff_pos++] = chr_us;
		    }
		else if (cpi < 9.25)
		    {
		    cpi = 8.5;
		    lilbuff[buff_pos++] = chr_fs;
		    lilbuff[buff_pos++] = chr_us;
		    }
		else if (cpi < 11)
		    {
		    cpi = 10;
		    lilbuff[buff_pos++] = chr_rs;
		    }
		else if (cpi < 14.55)
		    {
		    cpi = 12;
		    lilbuff[buff_pos++] = chr_gs;
		    }
		else
		    {
		    cpi = 17;
		    lilbuff[buff_pos++] = chr_fs;
		    }
		break;
	      case microline380:
	      case microline390:
	      case microline391:
		lilbuff[buff_pos++] = chr_esc;
		if (cpi < 5.5)
		    {
		    cpi = 5;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 6.75)
		    {
		    cpi = 6;
		    lilbuff[buff_pos++] = *"M";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 8)
		    {
		    cpi = 7.5;
		    lilbuff[buff_pos++] = *"g";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 9.25)
		    {
		    cpi = 8.5;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_si;
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"W";
		    lilbuff[buff_pos++] = 1;
		    }
		else if (cpi < 11)
		    {
		    cpi = 10;
		    lilbuff[buff_pos++] = *"P";
		    }
		else if (cpi < 13.5)
		    {
		    cpi = 12;
		    lilbuff[buff_pos++] = *"M";
		    }
		else if (cpi < 16.05)
		    {
		    cpi = 15;
		    lilbuff[buff_pos++] = *"g";
		    }
		else if (cpi < 18.55)
		    {
		    cpi = 17.1;
		    lilbuff[buff_pos++] = *"P";
		    lilbuff[buff_pos++] = chr_si;
		    }
		else
		    {
		    cpi = 20;
		    lilbuff[buff_pos++] = *"M";
		    lilbuff[buff_pos++] = chr_si;
		    }
		break;
	      }

#if x_debug
#if x_debug_ctrl
	    fprintf (stderr,"Paper width is %f\n",paper_width);
	    point = 120.0 / cpi;
	    fprintf (stderr,"Point size is %f (characters per inch is %f)\n",point,cpi);
	    fprintf (stderr,"Left_margin is %f\n",left_marg);
	    fprintf (stderr,"Right_margin is %f\n",right_marg);
#endif
#endif

	    chars_line = (paper_width - (left_marg + right_marg)) * cpi;
	    chars_marg = left_marg * cpi;
	    if (chars_line > bufsize) chars_line = bufsize;
	    if (chars_marg > bufsize) chars_marg = bufsize;

#if x_debug
	    fprintf (stderr,"Print chars in line = %d, chars in margin = %d\n",chars_line,chars_marg);
#endif

/*xxxxx Add new printer here xxxxx*/
	    /* Set left margin */
	    switch (printer_model)
	      {
	      case epsonlq800:
	      case epsonlq1000:
	      case epsonlq1500:
	      case epsonlx80:
	      case epsonfx80:
	      case epsonfx85:
	      case epsonfx105:
	      case microline380:
	      case microline390:
	      case microline391:
	      case pinwriterp5:
	      case starnx1000:
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"l";
		lilbuff[buff_pos++] = chars_marg;
		chars_marg = 0;
		break;
	      case imagewriter:
	      case imagewriterII:
		sprintf (&lilbuff[buff_pos],"\033L%03.3d",chars_marg);
		buff_pos += 5;
		chars_marg = 0;
		break;
	      case HPlaserjet:
	      case HPlaserjetplus:
		sprintf (&lilbuff[buff_pos],"\033&a%dL",chars_marg);
		buff_pos += strlen(&lilbuff[buff_pos]);
		chars_marg = 0;
		break;
	      }


	    if (header_valid)
		{
		if (lines_page < 6)
		    {
		    header_valid = false;
		    status.all = status_$ok;
		    generic_$print_error (status,"render","Header cannot be printed: not enough room","Need at least 4 printable lines per page",(long)On_);
		    }
		else if (chars_line < 10)
		    {
		    header_valid = false;
		    status.all = status_$ok;
		    generic_$print_error (status,"render","Header cannot be printed: not enough room","Need at least 10 characters per line",(long)On_);
		    }
		else
		    {
		    lines_page -= 2;
		    }
		}


	    if (footer_valid)
		{
		if (lines_page < 6)
		    {
		    footer_valid = false;
		    status.all = status_$ok;
		    generic_$print_error (status,"render","Footer cannot be printed: not enough room","Need at least 4 printable lines per page",(long)On_);
		    }
		else if (chars_line < 10)
		    {
		    footer_valid = false;
		    status.all = status_$ok;
		    generic_$print_error (status,"render","Footer cannot be printed: not enough room","Need at least 10 characters per line",(long)On_);
		    }
		else
		    {
		    lines_high -= 2;
		    lines_page -= 2;
		    }
		}


	    /* Whenever there is backspacing, we always expand tabs.
	    If the printer can do BS (use_bs is On_) then we do not want to worry what happens if the BS follows the tab.
	    If use_bs is one_at_a_time or use_cr_instead then we need it to later aid counting positions.
	    If we cannot do BS at all and a BS follows a TAB, should we move back one character width, or forget the tab? */
	    if (ftype_os) use_tab = false;


/*xxxxx Add new printer here xxxxx*/
	    /* Set tab stops */
	    switch (printer_model)
	      {
	      case epsonlq800:
	      case epsonlq1000:
	      case epsonlq1500:
	      case epsonlx80:
	      case epsonfx80:
	      case epsonfx85:
	      case epsonfx105:
	      case microline380:
	      case microline390:
	      case microline391:
	      case pinwriterp5:
	      case starnx1000:
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"D";
		for (j = 1; j <= 32; j++)
		    {
		    k = j * tab_length;
		    if (k > chars_line || k > 255) break;
		    lilbuff[buff_pos++] = k;
		    }
		lilbuff[buff_pos++] = chr_nul;
		break;
	      case imagewriter:
	      case imagewriterII:
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"(";
		for (j = 1; j <= 32; j++)
		    {
		    k = j * tab_length;
		    if (k > chars_line) break;
		    sprintf (&lilbuff[buff_pos],"%03.3d,",k);
		    buff_pos += 4;
		    }
		lilbuff[buff_pos - 1] = *".";
		break;
	      case microline84:
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = chr_tab;
		for (j = 1; j <= 16; j++)
		    {
		    k = j * tab_length;
		    if (k > chars_line) break;
		    k += chars_marg;
		    sprintf (&lilbuff[buff_pos],"%03.3d,",k);
		    buff_pos += 4;
		    }
		lilbuff[buff_pos - 1] = chr_cr;
		break;
	      }


/*xxxxx Add new printer here xxxxx*/
	    /* boldface */
	    /* Not available on HPlaserjets. */
#if SR_level == SR10_0 || SR_level == SR10_1
	    if (boldface && allow_bold)
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
	    if (boldface)
#endif
		{
		switch (printer_model)
		  {
		  case epsonlq800:
		  case epsonlq1000:
		  case epsonlq1500:
		  case epsonlx80:
		  case epsonfx80:
		  case epsonfx85:
		  case epsonfx105:
		  case microline380:
		  case microline390:
		  case microline391:
		  case pinwriterp5:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"E";
	    /* Horizontal offset. For vertical offset use *"G", but not on epsonfx85 or epsonfx105 in letter quality. */
		    break;
		  case imagewriter:
		  case imagewriterII:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"!";
		    break;
		  case microline84:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"T";
	    /* Horizontal offset. For vertical offset use *"H". */
		    break;
		  case starnx1000:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"E";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"G";
	    /* Horizontal and vertical offsets combined as recommended */
		    break;
		  }
		}


/*xxxxx Add new printer here xxxxx*/
	    /* letter quality */
	    /* Not available on epsonfx80, imagewriter or HPlaserjets. */
	    /* Only at 5 or 10 cpi on epsonlx80. */
#if SR_level == SR10_0 || SR_level == SR10_1
	    if (letter_quality && allow_lq)
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
	    if (letter_quality)
#endif
		{
		switch (printer_model)
		  {
		  case epsonlq800:
		  case epsonlq1000:
		  case epsonlq1500:
		  case epsonfx85:
		  case epsonfx105:
		  case microline390:
		  case microline391:
		  case pinwriterp5:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"x";
		    lilbuff[buff_pos++] = 1;
		    break;
		  case epsonlx80:
		    if (cpi == 5 || cpi == 10)
		      {
		      lilbuff[buff_pos++] = chr_esc;
		      lilbuff[buff_pos++] = *"x";
		      lilbuff[buff_pos++] = 1;
		      }
		    break;
		  case imagewriterII:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"a";
		    lilbuff[buff_pos++] = *"2";
		    break;
		  case microline84:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"1";
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"N";
		    lilbuff[buff_pos++] = chr_nul;
		    break;
		  case microline380:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"x";
		    lilbuff[buff_pos++] = 1;
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"k";
		    lilbuff[buff_pos++] = 126;
	    /* 1 Helvette, 2 Courier, 7 Orator, 126 (default) Courier */
		    break;
		  case starnx1000:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"x";
		    lilbuff[buff_pos++] = 1;
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"k";
		    lilbuff[buff_pos++] = 0;
	    /* 0 Courier (default), 1 Sanserif, 2 Orator with small capitals, 3 Orator with lower case */
		    break;
		  }
		}


/*xxxxx Add new printer here xxxxx*/
#if SR_level == SR10_2 || SR_level == SR10_3
	    /* italics */
	    /* Only available in draft mode on epsonfx85 an epsonfx105, not available on microline84, imagewriters or HPlaserjets. */
	    if (italics)
		{
		switch (printer_model)
		  {
		  case epsonlq800:
		  case epsonlq1000:
		  case epsonlq1500:
		  case epsonlx80:
		  case epsonfx80:
		  case epsonfx85:
		  case epsonfx105:
		  case microline380:
		  case microline390:
		  case microline391:
		  case pinwriterp5:
		  case starnx1000:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"4";
		    break;
		  }
		}
#endif


/*xxxxx Add new printer here xxxxx*/
#if SR_level == SR10_2 || SR_level == SR10_3
	    /* Only available on microline380/390/391, starnx1000, pinwriterp5 */
	    if (double_height)
		{
		switch (printer_model)
		  {
		  case microline380:
		  case microline390:
		  case microline391:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"w";
		    lilbuff[buff_pos++] = 1;
		    break;
		  case pinwriterp5:
		    lilbuff[buff_pos++] = chr_fs;
		    lilbuff[buff_pos++] = *"V";
		    lilbuff[buff_pos++] = 1;
		    break;
		  case starnx1000:
		    if (cpi < 17)
		      {
	    /* starnx1000 could also use 'chr_esc *"h" n', with 1 double, 2 quadruple size (both width and height);
	    this would also increase the line spacing to double or quadruple.
	    Note that double height is not compatible with condensed printing (17 and 20 cpi) on the starnx1000. */
		      lilbuff[buff_pos++] = chr_esc;
		      lilbuff[buff_pos++] = *"w";
		      lilbuff[buff_pos++] = 1;
		      }
		    break;
		  }
		}
#endif


/*xxxzzz Unused printer features
microline380:	outline typeface	chr_esc *"q" *"1"
		shadow typeface		chr_esc *"q" *"2"
		outline and shadow	chr_esc *"q" *"3"
		no outline or shadow	chr_esc *"q" *"0"

Justification	chr_esc *"a" n
		0 left (default), 1 center, 2 right, 3 full (left and right)

Proportional spacing
		chr_esc *"p" 1

Inter-character spacing
		chr_esc *" " n
		n in dots, 0 to 127

National character sets, code page options (microline380, microline390/391)
zzzxxx*/


	    if ((!wrap) || chars_line <= 20) word_wrap = false;


	    /* We send one formfeed at end (with possible footer) */
	    r0 = form_feed - 1;
	    status = set_db_value_c (obj_db,Form__Feeds,&r0);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to reset Form__Feeds","",(long)On_);
		}


	    /*yyyyy Check length of file. Should have been done by prf command yyyyy*/

	    /* Preserve the count of lines in line_number;
	       the printer setup string is in lilbuff, make sure not to disturb. */
	    buff_pos_save = buff_pos;
	    line_number_save = line_number;


	    i = ios_$inq_byte_pos(in_strid,ios_$eof,&status);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to find out length of input file stream","This print file rejected",(long)On_);
		total_rejected_ascii++;
		goto close_input_stream;
		}
	    else
		{
		if (i > 1024 * huge_job_chars)
		    {
		    sprintf (genbuff,"This file is ridiculously long (%d bytes), cannot be printed",i);
		    generic_$print_error (status,"render",genbuff,"This print file rejected",(long)On_);
		    total_rejected_ascii++;
		    goto close_input_stream;
		    }
#if SR_level == SR10_0 || SR_level == SR10_1
		else if (reject_big_jobs && i > big_job_chars)
		    {
		    sprintf (genbuff,"This file is too long (%d bytes), cannot be printed",i);
		    generic_$print_error (status,"render",genbuff,"This print file rejected",(long)On_);
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
		else if (reject_big_jobs && i > big_job_chars && (! passwd_ok))
		    {
		    sprintf (lilbuff,"long files (%d bytes)",i);
		    generic_$print_need_passwd (lilbuff);
#endif
		    total_rejected_ascii++;
		    goto close_input_stream;
		    }
		}


	    /* Run through the file counting characters, lines, pages */

	    line_number = 0;
	    need_lf = false;
	    page_number = 0;

	    buff_pos = 0;
	    buff_last = 0;
	    char_pos = 0;

	    funny_characters = 0;


	    ios_$seek_to_bof (in_strid,&status);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to find beginning of input stream","",(long)On_);
		status2 = generic_$send_ff (false,"");
		goto close_input_stream;
		}


	    do
		{
next_test_line:
		if (buff_last <= buff_pos)
		    /* Read input line */
		    {
		    buff_pos = 0;
		    buff_last = ios_$get (in_strid,ios_$no_put_get_opts,bigbuff,(long)bigbufsize,&status);
		    total_chars_read += buff_last;

		    if (! status.all)
			buff_last--; /* ignore the final LF we get in each line */
		    else
			{
			if (status.all == ios_$buffer_too_small)
			    {
			    if (wrap) status.all = status_$ok;
			    else { do { i = ios_$get (in_strid,ios_$no_put_get_opts,genbuff,(long)bufsize,&status); total_chars_read += i; } while (status.all == ios_$buffer_too_small); }
			    }

			if (status.all)
			    {
			    if (status.all == ios_$end_of_file) { status.all = status_$ok; break; }
			    else { generic_$print_error (status,"render","Unable to read input file stream","",(long)On_); break; }
			    }
			}

		    if (buff_last > 0) { j = buff_last; buff_last = 0; for (i = j - 1; i >= 0; i--) { if (bigbuff[i] != chr_sp && bigbuff[i] != chr_tab) { buff_last = i + 1; break; } } }
		    }


		/* Check carriage control character for FTN_ files */
		if (ftype_ftn)
		    {
		    if (buff_pos == 0 && buff_last > 0)
			{
			if (bigbuff[0] == *"0") { generic_$test_check_room (header_valid,footer_valid); generic_$test_send_lf ((long)1); }
			else if (bigbuff[0] == *"1") { if (line_number != 0 || need_lf) generic_$test_send_ff (footer_valid); }
			else if (bigbuff[0] == *"+")
			    {
			    if (need_lf)
				{
				if (use_cr_without_lf) { total_chars_job++; need_lf = false; }
				else if (use_bs == On_) { total_chars_job += char_pos; total_chars_job += chars_marg; need_lf = false; }
				else generic_$test_send_lf ((long)0);
				}
			    }
			buff_pos = 1;
			goto next_test_line;
			}
		    }


		/* check leading FF */
		if (buff_last > buff_pos && bigbuff[buff_pos] == chr_ff)
		    {
		    generic_$test_send_ff (footer_valid);
		    buff_pos++;
		    goto next_test_line;
		    }


		/* check room on this page */
		generic_$test_check_room (header_valid,footer_valid);


		char_pos = 0;
		bytes = 0;


		if (buff_last > buff_pos)
		    {
next_test_char:
		    c = bigbuff[buff_pos];

		    if (c >= chr_sp && c < chr_del) { bytes++; char_pos++; goto done_test_char; }
		    else if (c == chr_tab)
			{
			i = (char_pos / tab_length + 1) * tab_length;
			if (use_tab) { bytes++; char_pos = i; }
			else { if (i > chars_line) i = chars_line; bytes += i - char_pos; char_pos = i; }
			goto done_test_char;
			}
		    else if (c == chr_lf) { buff_pos++; goto end_test_line; }
		    else if (c == chr_bs && ftype_os)
			{
			if (bytes > 0 && char_pos > 0)
			    {
			    if (use_bs) { bytes++; char_pos--; }
			    else { bytes--; char_pos--; }
			    }
			goto done_test_char;
			}
		    else if (c == chr_ff) goto end_test_line;
		    else if (c < chr_nul)
			{
			if (eightbit == eightbit_ignore) goto done_test_char;
			else if (eightbit == eightbit_blank) { funny_characters++; bytes++; char_pos++; goto done_test_char; }
			else if (eightbit == eightbit_convert) { bigbuff[buff_pos] = c & 0x7f; goto next_test_char; }
			else if (eightbit > chr_sp && eightbit < chr_del)
			    {
			    if (char_pos + 6 > chars_line && char_pos > 0) goto end_test_line;
			    if (ctrl_char == ctrl_control && char_pos + 9 > chars_line && char_pos > 0) goto end_test_line;
			    funny_characters++;
			    bytes++;
			    char_pos++;
			    bigbuff[buff_pos] = c & 0x7f;
			    goto next_test_char;
			    }
			}
		    else if ((c >= chr_nul && c < chr_sp) || c == chr_del)
			{
			if (ctrl_char == ctrl_ignore) goto done_test_char;
			else if (ctrl_char == ctrl_blank) { funny_characters++; bytes++; char_pos++; goto done_test_char; }
			else if (ctrl_char == ctrl_ascii)
			    {
			    if (char_pos + 5 > chars_line && char_pos > 0) goto end_test_line;
			    funny_characters++;
			    bytes += 5;
			    char_pos += 5;
			    goto done_test_char;
			    }
			else if (ctrl_char == ctrl_control)
			    {
			    if (char_pos + 8 > chars_line && char_pos > 0) goto end_test_line;
			    funny_characters++;
			    bytes += 8;
			    char_pos += 8;
			    goto done_test_char;
			    }
			else if (ctrl_char == ctrl_caret)
			    {
			    if (char_pos + 3 > chars_line && char_pos > 0) goto end_test_line;
			    funny_characters++;
			    bytes += 2;
			    char_pos += 2;
			    goto done_test_char;
			    }
			}

		    /* Last chance: print ASCII value */
		    if (char_pos + 5 > chars_line && char_pos > 0) goto end_test_line;
		    funny_characters++;
		    bytes += 5;
		    char_pos += 5;

done_test_char:
		    buff_pos++;
		    if (buff_last <= buff_pos) goto end_test_line;
		    if (bytes >= bufsize) goto end_test_line;
		    if (char_pos >= chars_line) goto end_test_line;
		    goto next_test_char;

end_test_line:
		    if (bytes > 0)
			{
			/* send blanks for margin */
			total_chars_job += chars_marg;

			/* send line */
			total_chars_job += bytes;
			}

		    if (! wrap) { buff_last = 0; }
		    }

		total_lines_job++;
		need_lf = true;

		}
		while (! status.all);

	    if (status.all) goto close_input_stream;

	    total_pages_job = page_number + 1; /* Instead of end-of-copy FF */


#if x_debug
	    fprintf (stderr,"Testing: %d characters processed; %d characters to print in %d lines, %d pages\n",
		total_chars_read,total_chars_job,total_lines_job,total_pages_job);
#endif


	    /* Ready to check length of file */

	    genbuff[0] = chr_nul; /* So it has 0 length */

		 if (total_chars_read > 1024 * huge_job_chars) sprintf (genbuff,"This file is ridiculously long (%d bytes), cannot be printed",total_chars_read);
	    else if (total_chars_job >  1024 * huge_job_chars) sprintf (genbuff,"This file is ridiculously long (%d bytes), cannot be printed",total_chars_job);
	    else if (total_lines_job > huge_job_lines)         sprintf (genbuff,"This file is ridiculously long (%d lines), cannot be printed",total_lines_job);
	    else if (total_pages_job > huge_job_pages)         sprintf (genbuff,"This file is ridiculously long (%d pages), cannot be printed",total_pages_job);
#if SR_level == SR10_0 || SR_level == SR10_1
	    else if (reject_big_jobs)
		{
		     if (total_chars_read > big_job_chars) sprintf (genbuff,"This file is too long (%d bytes), cannot be printed",total_chars_read);
		else if (total_chars_job > big_job_chars)  sprintf (genbuff,"This file is too long (%d bytes), cannot be printed",total_chars_job);
		else if (total_lines_job > big_job_lines)  sprintf (genbuff,"This file is too long (%d lines), cannot be printed",total_lines_job);
		else if (total_pages_job > big_job_pages)  sprintf (genbuff,"This file is too long (%d pages), cannot be printed",total_pages_job);
		}
#endif
	    if (strlen(genbuff) > 0)
		{
		generic_$print_error (status,"render",genbuff,"This print file rejected",(long)On_);
		total_rejected_ascii++;
		goto close_input_stream;
		}

#if SR_level == SR10_2 || SR_level == SR10_3
	    if (reject_big_jobs && (! passwd_ok))
		{
		     if (total_chars_read > big_job_chars) sprintf (genbuff,"long files (%d bytes)",total_chars_read);
		else if (total_chars_job > big_job_chars)  sprintf (genbuff,"long files (%d bytes)",total_chars_job);
		else if (total_lines_job > big_job_lines)  sprintf (genbuff,"long files (%d lines)",total_lines_job);
		else if (total_pages_job > big_job_pages)  sprintf (genbuff,"long files (%d pages)",total_pages_job);

		if (strlen(genbuff) > 0)
		    {
		    sprintf (lilbuff,"%s",genbuff);
		    generic_$print_need_passwd (lilbuff);
		    total_rejected_ascii++;
		    goto close_input_stream;
		    }
		}
#endif


	    /*yyyyy Make sure file is not binary gibberish. Should have been done by prf command yyyyy*/

#if SR_level == SR10_0 || SR_level == SR10_1
	    if (funny_characters > 100 && funny_characters > total_chars_read / 20)
		{
		generic_$print_error (status,"render","Binary files cannot be printed","This print file rejected",(long)On_);
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
	    if (funny_characters > 100 && funny_characters > total_chars_read / 20 && (! passwd_ok))
		{
		generic_$print_need_passwd ("binary files, or use '-eightbit ignore -ctrl_char ignore'");
#endif
		total_rejected_ascii++;
		goto close_input_stream;
		}


	    /*yyyyy Check length of file with number of copies. Should have been done by prf command yyyyy*/

	    if (copies > 1)
		{
		total_chars_read = copies * total_chars_read;
		total_chars_job  = copies * total_chars_job;
		total_lines_job  = copies * total_lines_job;
		total_pages_job  = copies * total_pages_job;

#if SR_level == SR10_2 || SR_level == SR10_3
		if (reject_big_jobs && (! passwd_ok))
		    {
		         if (total_chars_read > big_job_chars) sprintf (genbuff,"multiple copies of long files (%d bytes total)",total_chars_read);
		    else if (total_chars_job > big_job_chars)  sprintf (genbuff,"multiple copies of long files (%d bytes total)",total_chars_job);
		    else if (total_lines_job > big_job_lines)  sprintf (genbuff,"multiple copies of long files (%d lines total)",total_lines_job);
		    else if (total_pages_job > big_job_pages)  sprintf (genbuff,"multiple copies of long files (%d pages total)",total_pages_job);

		    if (strlen(genbuff) > 0)
		        {
		        sprintf (lilbuff,"%s",genbuff);
		        generic_$print_need_passwd (lilbuff);
		        total_rejected_ascii++;
		        goto close_input_stream;
		        }
		    }
#endif

		     if (total_chars_read > 1024 * huge_job_chars) sprintf (genbuff,"This file becomes ridiculously long (%d bytes total) with %d copies",total_chars_read);
		else if (total_chars_job >  1024 * huge_job_chars) sprintf (genbuff,"This file becomes ridiculously long (%d bytes total) with %d copies",total_chars_job);
		else if (total_lines_job > huge_job_lines)         sprintf (genbuff,"This file becomes ridiculously long (%d lines total) with %d copies",total_lines_job);
		else if (total_pages_job > huge_job_pages)         sprintf (genbuff,"This file becomes ridiculously long (%d pages total) with %d copies",total_pages_job);

		if (strlen(genbuff) > 0)
		    {
		    generic_$print_error (status,"render",genbuff,"Only one copy allowed",(long)On_);
		    copies = 1;
		    }
#if SR_level == SR10_0 || SR_level == SR10_1
		else if (reject_big_jobs)
		    {
		         if (total_chars_read > big_job_chars) sprintf (genbuff,"This file becomes too long (%d bytes total) with %d copies",total_chars_read);
		    else if (total_chars_job > big_job_chars)  sprintf (genbuff,"This file becomes too long (%d bytes total) with %d copies",total_chars_job);
		    else if (total_lines_job > big_job_lines)  sprintf (genbuff,"This file becomes too long (%d lines total) with %d copies",total_lines_job);
		    else if (total_pages_job > big_job_pages)  sprintf (genbuff,"This file becomes too long (%d pages total) with %d copies",total_pages_job);
		    if (strlen(genbuff) > 0)
			{
			generic_$print_error (status,"render",genbuff,"Only one copy allowed",(long)On_);
			copies = 1;
			}
		    }
#endif
		}


	    /* Restore a couple of things */
	    total_chars_read = 0;
	    total_chars_job  = 0;
	    total_lines_job  = 0;
	    total_pages_job  = 0;

	    buff_pos = buff_pos_save;
	    line_number = line_number_save;


/*xxxxx Add new printer here xxxxx*/
	    /* Set number of copies: each page sent just once, but collating messed up */
	    if (copies > 1)
		{
		switch (printer_model)
		  {
		  case HPlaserjet:
		  case HPlaserjetplus:
		    sprintf (&lilbuff[buff_pos],"\033&l%dX",copies);
		    buff_pos += strlen(&lilbuff[buff_pos]);
		    copies = 1;
		    break;
		  }
		}


	    if (line_number != 0)
		status2 = generic_$send_ff (false,"");


	    if (buff_pos > 0)
		{
		status = generic_$iosput (lilbuff,buff_pos);
		if (status.all)
		    {
		    generic_$print_error (status,"render","Unable to set up printer for text","",(long)Off_);
		    goto close_input_stream;
		    }
		}


	    line_number = 0;
	    need_lf = false;
	    page_number = 0;

	    buff_pos = 0;
	    buff_last = 0;
	    char_pos = 0;

	    line_has_bs = false;


	    for (copy = 0; copy < copies; copy++) /* Start copies loop */
	    {


	    ios_$seek_to_bof (in_strid,&status);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to find beginning of input stream","",(long)On_);
		status2 = generic_$send_ff (false,"");
		break;
		}


	    do
		{
next_line:
		if (buff_last <= buff_pos)
		    /* Read input line */
		    {
		    buff_pos = 0;
		    buff_last = ios_$get (in_strid,ios_$no_put_get_opts,bigbuff,(long)bigbufsize,&status);
#if x_debug
#if x_debug_file
		    fprintf (stderr,"Line read from input stream, length = %d, status = %x\n",buff_last,status.all);
#endif
#endif
		    total_chars_read += buff_last;

		    if (! status.all)
			buff_last--; /* ignore the final LF we get in each line */
		    else
			{
			if (status.all == ios_$buffer_too_small)
			    {
			    if (wrap)
				status.all = status_$ok; /* expected */
			    else
				{
				do  /* remove rest of line from input */
				    {
				    i = ios_$get (in_strid,ios_$no_put_get_opts,genbuff,(long)bufsize,&status);
				    /* i = ios_$get (in_strid,(short)0,genbuff,(long)bufsize,&status); */
				    total_chars_read += i;
				    }
				while (status.all == ios_$buffer_too_small);
				}
			    }

			if (status.all)
			    {
			    if (status.all == ios_$end_of_file)
				{
				status.all = status_$ok; /* expected */
				break;
				}
			    else
				{
				generic_$print_error (status,"render","Unable to read input file stream","",(long)On_);
				break;
				}
			    }
			}

		    if (buff_last > 0)
			{
			/* ignore trailing blanks */
			j = generic_$str_nonblank_len (bigbuff,buff_last);
			buff_last = j;
#if x_debug
#if x_debug_file
			if (buff_last < bigbufsize) bigbuff[buff_last] = chr_nul;
			fprintf (stderr,"Ignoring trailing blanks, length = %d, and line is:\n<%.40s>\n",buff_last,bigbuff);
#endif
#endif
			}
		    }


		/* Check carriage control character for FTN_ files */
		if (ftype_ftn)
		    {
		    if (buff_pos == 0 && buff_last > 0)
			{
			if (bigbuff[0] == *"0")
			    {
			    status = generic_$render_check_room (header_valid,footer_valid,headerp,footerp);
			    if (status.all) break;
			    status = generic_$send_lf ((long)1);
			    if (status.all) break;
			    }
			else if (bigbuff[0] == *"1")
			    {
			    /* Protection against bogus formfeeds.
			    This prevents multiple formfeeds (i.e. empty pages).
			    Maybe we should check (page_number != 0 || line_number != 0 || need_lf) instead,
			    to suppress just an initial formfeed?
			    Beware of files containing a column of numbers, printed with -ftn ! */
			    if (line_number != 0 || need_lf)
				{
				status = generic_$send_ff (footer_valid,footerp);
				if (status.all) break;
				}
			    }
			else if (bigbuff[0] == *"+")
			    {
			    if (need_lf)
				{
				if (use_cr_without_lf)
				    {
#if x_debug
#if x_debug_file
				    fprintf (stderr,"Sending a CR to overprint next line");
#endif
#endif
				    lilbuff[0] = chr_cr;
				    status = generic_$iosput (lilbuff,(long)1);
				    if (status.all) break;
				    need_lf = false;
				    }
				else if (use_bs == On_)
				    {
#if x_debug
#if x_debug_file
				    fprintf (stderr,"Sending many BS's to overprint next line");
#endif
#endif
				    if (char_pos > 0)
					{
					for (i = 0; i < char_pos; i++) lilbuff[i] = chr_bs;
					status = generic_$iosput (lilbuff,char_pos);
					if (status.all) break;
					}
				    if (chars_marg > 0)
					{
					for (i = 0; i < chars_marg; i++) lilbuff[i] = chr_bs;
					status = generic_$iosput (lilbuff,chars_marg);
					if (status.all) break;
					}
				    need_lf = false;
				    }
				else
				    {
				    status = generic_$send_lf ((long)0);
				    if (status.all) break;
				    }
				}
			    }
			buff_pos = 1;
			goto next_line;
			}
		    }


		/* check leading FF */
		if (buff_last > buff_pos && bigbuff[buff_pos] == chr_ff)
		    {
		    status = generic_$send_ff (footer_valid,footerp);
		    if (status.all) break;
		    buff_pos++;
		    goto next_line;
		    }


		/* check room on this page */
		status = generic_$render_check_room (header_valid,footer_valid,headerp,footerp);
		if (status.all) break;


		char_pos = 0;
		bytes = 0;


		if (buff_last > buff_pos)
		    {
next_char:
		    c = bigbuff[buff_pos];

		    if (c >= chr_sp && c < chr_del)
			{
			/* Printable character, copy */
			lilbuff[bytes] = c;
			char_poss[bytes] = char_pos;
			buff_poss[bytes] = buff_pos;
			bytes++;
			char_pos++;
			goto done_char;
			}
		    else if (c == chr_tab)
			{
			/* Tab character */
			i = (char_pos / tab_length + 1) * tab_length;
			if (use_tab)
			    {
			    lilbuff[bytes] = c;
			    char_poss[bytes] = char_pos;
			    buff_poss[bytes] = buff_pos;
			    bytes++;
			    char_pos = i;
			    }
			else
			    {
			    if (i > chars_line) i = chars_line;
			    for (; char_pos < i; char_pos++)
				{
				lilbuff[bytes] = chr_sp;
				char_poss[bytes] = char_pos;
				buff_poss[bytes] = buff_pos;
				bytes++;
				}
			    }
			goto done_char;
			}
		    else if (c == chr_lf)
			{
			/* Linefeed character */
			buff_pos++; /* Remove from buffer */
			goto end_line;
			}
		    else if (c == chr_bs && ftype_os)
			{
			/* Backspace character */
			if (bytes > 0 && char_pos > 0) /* Otherwise ignore */
			    {
			    if (use_bs)
				{
				/* Printer can execute, copy */
				line_has_bs = true;
				lilbuff[bytes] = c;
				char_poss[bytes] = char_pos;
				buff_poss[bytes] = buff_pos;
				bytes++;
				char_pos--; /* Assume that printer goes back one character wide. No tabs to worry about. */
				}
			    else
				{
				/* Printer cannot execute, remove one character from buffer */
				bytes--;
				char_pos = char_poss[bytes];
				}
			    }
			goto done_char;
			}
		    else if (c == chr_ff)
			{
			/* Formfeed character */
			/* DANGER: not removed from buffer, could cause infinite loop */
			goto end_line;
			}
		    else if (c < chr_nul)
			{
			/* Bit eight set */
			if (eightbit == eightbit_ignore)
			    {
			    /* Silently ignore */
			    buff_poss[bytes] = buff_pos;
			    goto done_char;
			    }
			else if (eightbit == eightbit_blank)
			    {
			    /* Print a blank */
			    lilbuff[bytes] = *" ";
			    char_poss[bytes] = char_pos;
			    buff_poss[bytes] = buff_pos;
			    bytes++;
			    char_pos++;
			    goto done_char;
			    }
			else if (eightbit == eightbit_convert)
			    {
			    /* Convert to 7 bits (ignore bit eight) */
			    bigbuff[buff_pos] = c & 0x7f;
			    goto next_char;
			    }
			else if (eightbit > chr_sp && eightbit < chr_del)
			    {
			    /* Use prefix */
			    if (char_pos + 6 > chars_line && char_pos > 0) goto end_line; /* Not enough room to print description */
			    if (ctrl_char == ctrl_control && char_pos + 9 > chars_line && char_pos > 0) goto end_line;
			    lilbuff[bytes] = eightbit;
			    char_poss[bytes] = char_pos;
			    buff_poss[bytes] = buff_pos;
			    bytes++;
			    char_pos++;
			    bigbuff[buff_pos] = c & 0x7f;
			    goto next_char;
			    }
			}
		    else if ((c >= chr_nul && c < chr_sp) || c == chr_del)
			{
			if (ctrl_char == ctrl_ignore)
			    {
			    /* Silently ignore */
			    buff_poss[bytes] = buff_pos;
			    goto done_char;
			    }
			else if (ctrl_char == ctrl_blank)
			    {
			    /* Print a blank */
			    lilbuff[bytes] = *" ";
			    char_poss[bytes] = char_pos;
			    buff_poss[bytes] = buff_pos;
			    bytes++;
			    char_pos++;
			    goto done_char;
			    }
			else if (ctrl_char == ctrl_ascii)
			    {
			    /* Print ASCII name */
			    if (char_pos + 5 > chars_line && char_pos > 0) goto end_line; /* Not enough room to print description */
				 if (c == chr_nul) {sprintf (&lilbuff[bytes],"<NUL>"); j = 5;}
			    else if (c == chr_soh) {sprintf (&lilbuff[bytes],"<SOH>"); j = 5;}
			    else if (c == chr_stx) {sprintf (&lilbuff[bytes],"<STX>"); j = 5;}
			    else if (c == chr_etx) {sprintf (&lilbuff[bytes],"<ETX>"); j = 5;}
			    else if (c == chr_eot) {sprintf (&lilbuff[bytes],"<EOT>"); j = 5;}
			    else if (c == chr_enq) {sprintf (&lilbuff[bytes],"<ENQ>"); j = 5;}
			    else if (c == chr_ack) {sprintf (&lilbuff[bytes],"<ACK>"); j = 5;}
			    else if (c == chr_bel) {sprintf (&lilbuff[bytes],"<BEL>"); j = 5;}
			    else if (c == chr_bs ) {sprintf (&lilbuff[bytes],"<BS>"); j = 4;}
			    else if (c == chr_tab) {sprintf (&lilbuff[bytes],"<TAB>"); j = 5;}
			    else if (c == chr_lf ) {sprintf (&lilbuff[bytes],"<LF>"); j = 4;}
			    else if (c == chr_vt ) {sprintf (&lilbuff[bytes],"<VT>"); j = 4;}
			    else if (c == chr_ff ) {sprintf (&lilbuff[bytes],"<FF>"); j = 4;}
			    else if (c == chr_cr ) {sprintf (&lilbuff[bytes],"<CR>"); j = 4;}
			    else if (c == chr_so ) {sprintf (&lilbuff[bytes],"<SO>"); j = 4;}
			    else if (c == chr_si ) {sprintf (&lilbuff[bytes],"<SI>"); j = 4;}
			    else if (c == chr_dle) {sprintf (&lilbuff[bytes],"<DLE>"); j = 5;}
			    else if (c == chr_dc1) {sprintf (&lilbuff[bytes],"<DC1>"); j = 5;}
			    else if (c == chr_dc2) {sprintf (&lilbuff[bytes],"<DC2>"); j = 5;}
			    else if (c == chr_dc3) {sprintf (&lilbuff[bytes],"<DC3>"); j = 5;}
			    else if (c == chr_dc4) {sprintf (&lilbuff[bytes],"<DC4>"); j = 5;}
			    else if (c == chr_nak) {sprintf (&lilbuff[bytes],"<NAK>"); j = 5;}
			    else if (c == chr_syn) {sprintf (&lilbuff[bytes],"<SYN>"); j = 5;}
			    else if (c == chr_etb) {sprintf (&lilbuff[bytes],"<ETB>"); j = 5;}
			    else if (c == chr_can) {sprintf (&lilbuff[bytes],"<CAN>"); j = 5;}
			    else if (c == chr_em ) {sprintf (&lilbuff[bytes],"<EM>"); j = 4;}
			    else if (c == chr_sub) {sprintf (&lilbuff[bytes],"<SUB>"); j = 5;}
			    else if (c == chr_esc) {sprintf (&lilbuff[bytes],"<ESC>"); j = 5;}
			    else if (c == chr_fs ) {sprintf (&lilbuff[bytes],"<FS>"); j = 4;}
			    else if (c == chr_gs ) {sprintf (&lilbuff[bytes],"<GS>"); j = 4;}
			    else if (c == chr_rs ) {sprintf (&lilbuff[bytes],"<RS>"); j = 4;}
			    else if (c == chr_us ) {sprintf (&lilbuff[bytes],"<US>"); j = 4;}
			    else if (c == chr_del) {sprintf (&lilbuff[bytes],"<DEL>"); j = 5;}
			    else		   {sprintf (&lilbuff[bytes],"<??>"); j = 4;}
			    for (i = 0; i < j; i++)
				{
				char_poss[bytes] = char_pos;
				buff_poss[bytes] = buff_pos;
				bytes++;
				char_pos++;
				}
			    goto done_char;
			    }
			else if (ctrl_char == ctrl_control)
			    {
			    /* Print control name */
			    if (char_pos + 8 > chars_line && char_pos > 0) goto end_line; /* Not enough room to print description */
				 if (c == chr_nul) {sprintf (&lilbuff[bytes],"<null>"); j = 6;}
			    else if (c >= chr_ctrl_a && c <= chr_ctrl_z)
						   {sprintf (&lilbuff[bytes],"<ctrl- >"); lilbuff[bytes + 6] = c | 0x40; j = 8;}
			    else if (c == chr_esc) {sprintf (&lilbuff[bytes],"<escape>"); j = 8;}
			    else if (c == chr_fs ) {sprintf (&lilbuff[bytes],"<frmsep>"); j = 8;}
			    else if (c == chr_gs ) {sprintf (&lilbuff[bytes],"<grpsep>"); j = 8;}
			    else if (c == chr_rs ) {sprintf (&lilbuff[bytes],"<recsep>"); j = 8;}
			    else if (c == chr_us ) {sprintf (&lilbuff[bytes],"<usrsep>"); j = 8;}
			    else if (c == chr_del) {sprintf (&lilbuff[bytes],"<delete>"); j = 8;}
			    else		   {sprintf (&lilbuff[bytes],"<noname>"); j = 8;}
			    for (i = 0; i < j; i++)
				{
				char_poss[bytes] = char_pos;
				buff_poss[bytes] = buff_pos;
				bytes++;
				char_pos++;
				}
			    goto done_char;
			    }
			else if (ctrl_char == ctrl_caret)
			    {
			    /* Print control name */
			    if (char_pos + 3 > chars_line && char_pos > 0) goto end_line; /* Not enough room to print description */
			    if (c >= chr_nul && c <= chr_us) {lilbuff[bytes] = *"^"; lilbuff[bytes + 1] = c | 0x40; j = 2;}
			    else if (c == chr_del) {sprintf (&lilbuff[bytes],"del"); j = 3;}
			    else		   {sprintf (&lilbuff[bytes],"???"); j = 3;}
			    for (i = 0; i < j; i++)
				{
				char_poss[bytes] = char_pos;
				buff_poss[bytes] = buff_pos;
				bytes++;
				char_pos++;
				}
			    goto done_char;
			    }
			}

		    /* Last chance: print ASCII value */
		    if (char_pos + 5 > chars_line && char_pos > 0) goto end_line; /* Not enough room to print description */
		    j = c & 0xff;
		    sprintf (&lilbuff[bytes],"<X%02X>",j);
		    for (i = 0; i < 5; i++)
			{
			char_poss[bytes] = char_pos;
			buff_poss[bytes] = buff_pos;
			bytes++;
			char_pos++;
			}

done_char:
		    buff_pos++;
		    if (buff_last <= buff_pos) goto end_line;
		    if (bytes >= bufsize) goto end_line;
		    if (char_pos >= chars_line)
			{
			if (word_wrap)
			    {
			    /* Search for blank, through 1/3 of the line length (at most 30 characters), leaving at least 10 characters. */
			    k = chars_line / 3;
			    if (k > 30) k = 30;
			    k = chars_line - k;
			    if (k < 10) k = 10;
			    lilbuff[bytes] = bigbuff[buff_pos];
			    char_poss[bytes] = char_pos;
			    buff_poss[bytes] = buff_pos;
			    for (i = bytes; char_poss[i] > k; i--)
				{
				if (lilbuff[i] == chr_sp || lilbuff[i] == chr_tab)
				    {
				    bytes = i;
				    buff_pos = buff_poss[bytes] + 1;
				    char_pos = char_poss[bytes];
				    goto end_line;
				    }
				}
			    }
			goto end_line;
			}
		    goto next_char;

end_line:
		    if (bytes > 0)
			{
			/* ignore trailing blanks */
			char_poss[bytes] = char_pos;
			j = generic_$str_nonblank_len (lilbuff,bytes);
			bytes = j;
			char_pos = char_poss[bytes];
			}

		    if (bytes > 0)
			{
			/* send blanks for margin */
			if (chars_marg > 0)
			    {
			    status = generic_$iosput (sp_buff,chars_marg);
			    if (status.all) break;
			    }

			/* send line */
			if (line_has_bs && use_bs != On_)
			    {
			    /* The string in lilbuff[0...bytes-1] has BS characters;
			    we know it has no other control characters, not even tabs. */

			    /* Save char_pos: we never really use it, only for -ftn files where BS is not allowed. But maybe in future... */
			    char_pos_save = char_pos;

next_partial_line:
			    /* Get all characters printed first */
			    char_pos = 0;
			    for (i = 0; i < bytes; i++)
				{
				genbuff[i] = chr_sp;
				c = lilbuff[i];
				if (c == chr_bs) { char_pos--; if (char_pos < 0) char_pos = 0; }
				else
				    {
				    if (c != chr_sp)
					{
					if (genbuff[char_pos] == chr_sp) { genbuff[char_pos] = c; lilbuff[i] = chr_sp; }
					}
				    char_pos++;
				    }
				}
			    bufflen = generic_$str_nonblank_len (genbuff,bytes);
			    j = generic_$str_nonblank_len (lilbuff,bytes);
			    bytes = j;

			    if (use_bs == use_cr_instead)
				{
				/* Print using CR_without_LF: print this line, next line will be overprinted */
#if x_debug
#if x_debug_file
				if (bufflen < bufsize) genbuff[bufflen] = chr_nul;
				fprintf (stderr,"Sending line of length = %d, when line is:\n<%.40s>\n",bufflen,genbuff);
#endif
#endif
				status = generic_$iosput (genbuff,bufflen);
				if (status.all) break;
				if (bytes > 0)
				    {
#if x_debug
#if x_debug_file
				    fprintf (stderr,"Sending a CR to overprint next line");
#endif
#endif
				    genbuff[0] = chr_cr;
				    status = generic_$iosput (genbuff,(long)1);
				    if (status.all) break;
				    goto next_partial_line;
				    }
				}
			    else
				{
				/* Print with use_bs set to one_at_a_time: insert all (BS c) pairs */
				for (i = 0; i < bufflen; i++) char_poss[i] = i;
				char_pos_max = bufflen;
				char_pos = 0;
				for (i = 0; i < bytes; i++)
				    {
				    c = lilbuff[i];
				    if (c == chr_sp) char_pos++;
				    else if (c == chr_bs) { char_pos--; if (char_pos < 0) char_pos = 0; }
				    else
					{
					k = char_poss[char_pos];
					for (j = bufflen - 1; j > k; j--) genbuff[j+2] = genbuff[j];
					for (j = char_pos; j < char_pos_max; j++) char_poss[j] = char_poss[j] + 2;
					bufflen = bufflen + 2;
					genbuff[k+1] = chr_bs;
					genbuff[k+2] = c;
					char_pos++;
					}
				    }
#if x_debug
#if x_debug_file
				if (bufflen < bufsize) genbuff[bufflen] = chr_nul;
				fprintf (stderr,"Sending line of length = %d, when line is:\n<%.40s>\n",bufflen,genbuff);
#endif
#endif
				status = generic_$iosput (genbuff,bufflen);
				if (status.all) break;
				}

			    char_pos = char_pos_save;
			    line_has_bs = false;
			    }
			else
			    {
#if x_debug
#if x_debug_file
			    if (bytes < bufsize) lilbuff[bytes] = chr_nul;
			    fprintf (stderr,"Sending line of length = %d, when line is:\n<%.40s>\n",bytes,lilbuff);
#endif
#endif
			    status = generic_$iosput (lilbuff,bytes);
			    if (status.all) break;
			    }
			}

		    if (! wrap)
			{
			/* Ignore rest of line */
			buff_last = 0;
			}
		    }

		total_lines_job++;
		/* flag this as needing a linefeed */
		need_lf = true;

		}
		while (! status.all);


	    status2 = generic_$send_ff (footer_valid,footerp);
	    if (status2.all)
		{
		generic_$print_error (status2,"render","Unable to send end_of_copy FF","",(long)Off_);
		if (! status.all) status = status2;
		}

	    if (status.all) break;

	    } /* end copies loop */


	    total_pages_job = page_number;

	    if (copies == 1)
		{
		fprintf (stderr,"Total %d characters processed; %d characters printed in %d lines, %d pages\n",
		    total_chars_read,total_chars_job,total_lines_job,total_pages_job);
		}
	    else
		{
		fprintf (stderr,"Total %d characters processed; %d characters printed in %d lines, %d pages (in %d copies)\n",
		    total_chars_read,total_chars_job,total_lines_job,total_pages_job,copies);
		}
	    if (total_chars_read > big_job_chars ||
		total_chars_job  > big_job_chars ||
		total_lines_job  > big_job_lines ||
		total_pages_job  > big_job_pages ) fprintf (stderr,"WARNING: this was a big job.\n");

	    total_chars_ascii += total_chars_job;
	    total_lines_ascii += total_lines_job;
	    total_pages_ascii += total_pages_job;


	    } /* end UASC_ file */


    else if (ftype_trans)
	    {
#if SR_level == SR10_0 || SR_level == SR10_1
	    if (! allow_trans)
		{
		status.all = status_$ok;
		generic_$print_error (status,"render","Printing of TRANS_ files has not been enabled","Cannot print this type of file",(long)On_);
		status.all = print_$non_printable_file;
		total_rejected_trans++;
		goto close_input_stream;
		}
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
	    if (! passwd_ok)
		{
		status.all = status_$ok;
		generic_$print_need_passwd ("trans files");
		total_rejected_trans++;
		goto close_input_stream;
		}
#endif


	    /* This is a genuine warning, not a debug statement */
	    fprintf (stderr,"WARNING: TRANS file being printed, may leave printer inoperable\n");

	    if (mngr_type != mgr__ios)
		{
		sprintf (genbuff,"Wrong mngr_type = %d, must be %d for IOS",mngr_type,mgr__ios);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"",(long)On_);
		status.all = print_$file_device_mismatch;
		goto close_input_stream;
		}

	    in_strid = (ios_$id_t)*obj_id;


	    /*yyyyy Check that file type is uasc or unstruct. Should have been done by prf command yyyyy*/
	    ios_$inq_type_uid (in_strid,&type_uid,&status);
	    if (status.all)
		generic_$print_error (status,"render","Unable to find type_uid of file","",(long)Off_);
	    else
		if (! (type_uid == uasc_$uid || type_uid == unstruct_$uid))
		    {
		    strp = "unknown";
			 if (type_uid == case_hm_$uid)		strp = "case_hm";
		    else if (type_uid == cmpexe_$uid)		strp = "cmpexe";
		    else if (type_uid == coff_$uid)		strp = "coff";
		    else if (type_uid == d3m_area_$uid)		strp = "d3m_area";
		    else if (type_uid == d3m_sch_$uid)		strp = "d3m_sch";
		    else if (type_uid == directory_$uid)	strp = "directory";
		    else if (type_uid == dm_edit_$uid)		strp = "dm_edit";
		    else if (type_uid == hdr_undef_$uid)	strp = "hdr_undef";
		    else if (type_uid == input_pad_$uid)	strp = "input_pad";
		    else if (type_uid == mbx_$uid)		strp = "mbx";
		    else if (type_uid == mt_$uid)		strp = "mt";
		    else if (type_uid == nulldev_$uid)		strp = "nulldev";
		    else if (type_uid == object_file_$uid)	strp = "object_file";
		    else if (type_uid == pad_$uid)		strp = "pad";
		    else if (type_uid == pty_$slave_uid)	strp = "pty_slave";
		    else if (type_uid == pty_$uid)		strp = "pty";
		    else if (type_uid == records_$uid)		strp = "records";
		    else if (type_uid == sio_$uid)		strp = "sio";
		    else if (type_uid == tcp_$uid)		strp = "tcp";
		    sprintf (genbuff,"Files of type <%s> cannot be printed",strp);
		    generic_$print_error (status,"render",genbuff,"This print file rejected",(long)On_);
		    total_rejected_trans++;
		    goto close_input_stream;
		    }


	    /*yyyyy Check length of file. Should have been done by prf command yyyyy*/
	    i = ios_$inq_byte_pos(in_strid,ios_$eof,&status);
	    if (status.all) generic_$print_error (status,"render","Unable to find out length of input file stream","",(long)Off_);
	    else
		{
		if (i > 1024 * huge_job_chars)
		    {
		    sprintf (genbuff,"This file is ridiculously long (%d bytes), cannot be printed",i);
		    generic_$print_error (status,"render",genbuff,"This print file rejected",(long)On_);
		    total_rejected_trans++;
		    goto close_input_stream;
		    }
#if SR_level == SR10_0 || SR_level == SR10_1
		else if (reject_big_jobs && i > big_job_chars)
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
		else if (reject_big_jobs && i > big_job_chars && (! passwd_ok))
#endif
		    {
		    sprintf (genbuff,"This file is too long (%d bytes), cannot be printed",i);
		    generic_$print_error (status,"render",genbuff,"This print file rejected",(long)On_);
		    total_rejected_trans++;
		    goto close_input_stream;
		    }
		}


	    ios_$seek_to_bof (in_strid,&status);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to find beginning of input stream","",(long)On_);
		goto close_input_stream;
		}


	    do
		{
		buff_last = ios_$get (in_strid,ios_$no_put_get_opts,bigbuff,(long)bigbufsize,&status);
#if x_debug
#if x_debug_file
		fprintf (stderr,"Line read from input stream, length = %d, status = %x\n",buff_last,status.all);
#endif
#endif
		if (status.all)
		    {
		    if (status.all == ios_$buffer_too_small)
			status.all = status_$ok; /* expected */
		    else if (status.all == ios_$end_of_file)
			{
			status.all = status_$ok; /* expected */
			break;
			}
		    else
			{
			generic_$print_error (status,"render","Unable to read input file stream","",(long)On_);
			break;
			}
		    }
		status = generic_$iosput (bigbuff,buff_last);

		}
		while (! status.all);


	    fprintf (stderr,"Total %d characters processed\n",total_chars_job);
	    if (total_chars_job > big_job_chars) fprintf (stderr,"WARNING: this was a big job.\n");

	    total_chars_trans += total_chars_job;

	    } /* end TRANS_ file */


    else if (ftype_bm1)
	    {
#if x_debug
	    fprintf (stderr,"Printing 1-pixel bitmap file %s\n",file_typep);
#endif

	    /* GPR package already initialized, no need for gpr_$init.
	    Then probably we should not use gpr_$terminate either. */


	    if (mngr_type == mgr__ios) mngr_type = mgr__gpr;
	    /*yyyyy Manager type is set to IOS, not GPR yyyyy*/


	    if (mngr_type != mgr__gpr)
		{
		sprintf (genbuff,"Wrong mngr_type = %d, must be %d for GPR",mngr_type,mgr__gpr);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"",(long)On_);
		status.all = print_$file_device_mismatch;
		goto close_input_stream;
		}

	    bitmap_desc = (gpr_$bitmap_desc_t)*obj_id;
	    /*zzzzz What is the use of element Bm__Hdr (bm_desc), present in obj_db when printing a proper GPR file? zzzzz*/


#if SR_level == SR10_2 || SR_level == SR10_3
	    if (! passwd_ok)
		{
		strp = "";
		if (reverse_bw != reverse_bw_default && (reverse_bw_default == light || reverse_bw_default == dark))	strp = "reverse_bw";

		if (!generic_$strcmp(strp,"",true))
		    {
		    generic_$print_need_passwd (strp);
		    status.all = status_$ok;
		    total_rejected_bm1++;
		    goto close_input_stream;
		    }
		}
#endif


	    buff_pos = 0;


/* Need to set the variables: x_dpi, y_dpi, x_pin, y_pin, mult, mult1, i_fac, imode and gaps. */

/* x_dpi, y_dpi: dots per inch in the x (printhead) and y (paper) directions */
/* x_pin, y_pin: pins on the printhead in each direction */
/* mult: number of bytes in each column as printed (i.e. smallest number of bytes that needs to be printed; equal to y_pin/8 or x_pin/8) */
/* i_fac: if positive, number of move dots to correspond to one printed column width;
	  if negative, number of printed column widths that correspond to one move dot */
/* mult1: number of blank bytes than can be moved over (equal mult or mult*(-i_fac)) */
/* imode: peculiar to Epson graphics, or dotsize for LaserJetPlus, or -1 for imagewriters */
/* gaps: use move command if more than this many blank bytes (thus gaps/mult blank columns) */


/*
The following table should be used for the printers using ESC/P sequences
(including microline380/390/391, epsonlq800/1000/1500, epsonlx80, epsonfx80/85/105, starnx1000, pinwriterp5):

 imode   y_pin   x_dpi   y_dpi   i_fac  UTL/LQ    comments
    0       8      60      60       2     UTL
    1       8     120      60       1     UTL     low speed
    2       8     120      60       1     UTL     high speed, no consecutive dots except microline380/390/391 or pinwriterp5
    3       8     240      60      -2     UTL     no consecutive dots except microline380/390/391 or pinwriterp5
    4       8      80      60       0       0
    5       8      72      72       0       0     only epsonlx80, epsonfx80/85/105, starnx1000
    6       8      90      60       2      LQ
    7       8     144      72       0       0     only epsonfx85/105
   32      24      60     180       2     UTL
   33      24     120     180       1     UTL
   38      24      90     180       2      LQ
   39      24     180     180       1      LQ
   40      24     360     180      -2      LQ     not on epsonlq1500 or pinwriterp5, no consecutive dots except microline380/390/391

24 pin modes available only on microline380/390/391, epsonlq800/1000/1500, pinwriterp5.
Only the 8 pin modes are available on epsonfx80/85/105, starnx1000.

Vertical density of epsonlx80, epsonfx80/85/105, starnx1000 is 72 dpi, not 60.

epsonfx80 has no horizontal move commands (other than printing a blank).

starnx1000 can move horizontally only in steps of 1/120 inch (or by printing blanks).
*/


/*xxxxx Add new printer here xxxxx*/
	    /* Resolution etc. */
	    switch (printer_model)
	      {
	      case epsonlq800:
	      case epsonlq1000:
	      case epsonlq1500:
	      case microline380:
	      case microline390:
	      case microline391:
	      case pinwriterp5:
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"P";		/* 10 cpi */
		cpi = 10;

		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"A";
		lilbuff[buff_pos++] = 8;		/* 8/60 inch line spacing */
		lpi = 60.0/8.0;

		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"x";		/* 0 = UTL (120 dpi), or 1 = LQ (180 dpi) */

		if (resolution < 75)
		    {imode =  0; x_dpi =  60; y_pin =  8; i_fac =  2; lilbuff[buff_pos++] = 0;}
		else if (resolution < 105)
		    {imode =  6; x_dpi =  90; y_pin =  8; i_fac =  2; lilbuff[buff_pos++] = 1;}
		else if (resolution < 150)
		    {imode = 33; x_dpi = 120; y_pin = 24; i_fac =  1; lilbuff[buff_pos++] = 0;}
		else if (resolution < 270 || (printer_model != microline380 && printer_model != microline390 && printer_model != microline391))
		    {imode = 39; x_dpi = 180; y_pin = 24; i_fac =  1; lilbuff[buff_pos++] = 1;}
		else
		    {imode = 40; x_dpi = 360; y_pin = 24; i_fac = -2; lilbuff[buff_pos++] = 1;}

		mult = y_pin / 8;
		y_dpi = 60 * mult;
		x_pin = 1;
		mult1 = mult; if (i_fac < 0) mult1 = - mult * i_fac;
		gaps = 30 * mult; if (imode == 4) gaps = 0; /* Optimization cannot be done for 80 dpi */
		break;
	      case epsonlx80:
	      case epsonfx80:
	      case epsonfx85:
	      case epsonfx105:
	      case starnx1000:
		lilbuff[buff_pos++] = chr_esc;

		if (resolution < 66)
		    {imode =  0; x_dpi =  60; i_fac =  -5; cpi = 12; lilbuff[buff_pos++] = *"M";}
		else if (resolution < 76)
		    {imode =  5; x_dpi =  72; i_fac =  -6; cpi = 12; lilbuff[buff_pos++] = *"M";}
		else if (resolution < 85)
		    {imode =  4; x_dpi =  80; i_fac =  -8; cpi = 10; lilbuff[buff_pos++] = *"P";}
		else if (resolution < 105)
		    {imode =  6; x_dpi =  90; i_fac =  -9; cpi = 10; lilbuff[buff_pos++] = *"P";}
		else if (resolution < 132 || (printer_model != epsonfx85 && printer_model != epsonfx105))
		    {imode =  1; x_dpi = 120; i_fac = -10; cpi = 12; lilbuff[buff_pos++] = *"M";}
		else
		    {imode =  7; x_dpi = 144; i_fac = -12; cpi = 12; lilbuff[buff_pos++] = *"M";}

		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"A";
		lilbuff[buff_pos++] = 8;		/* 8/72 inch line spacing */
		lpi = 72.0/8.0;

		y_pin = 8;
		mult = 1;
		y_dpi = 72;
		x_pin = 1;
		mult1 = -i_fac;
		gaps = 30;
		break;
	      case imagewriter:
	      case imagewriterII:
		if (resolution < 76)
		    {x_dpi =  72; cpi =  9;   sprintf (lilbuff,"\033n\033T16");}
		else if (resolution < 88)
		    {x_dpi =  80; cpi = 10;   sprintf (lilbuff,"\033N\033T16");}
		else if (resolution < 102)
		    {x_dpi =  96; cpi = 12;   sprintf (lilbuff,"\033E\033T16");}
		else if (resolution < 114)
		    {x_dpi = 107; cpi = 13.4; sprintf (lilbuff,"\033e\033T16");}
		else if (resolution < 128)
		    {x_dpi = 120; cpi = 15;   sprintf (lilbuff,"\033q\033T16");}
		else
		    {x_dpi = 136; cpi = 17;   sprintf (lilbuff,"\033Q\033T16");}
		buff_pos = strlen(lilbuff);

		lpi = 144.0/16.0; /* 16/144 inch line spacing */

		y_pin = 8;
		imode = -1; /* To distinguish from other 8-pin modes: here top dot is least significant bit, bottom dot is most significant bit */
		mult = 1;
		y_dpi = 72;
		x_pin = 1;
		i_fac = 0;
		mult1 = 1;
		gaps = 30;
		break;
	      case HPlaserjet:
		imode = 0; /* unused */
		x_dpi = 75;
		y_dpi = x_dpi;
		x_pin = 8;
		y_pin = 1;
		mult = 1;
		i_fac = 0;
		mult1 = 1;
		gaps = 0; /* To make sure lines are sent fully, since we cannot move cursor horizontally */

		cpi = 10;
		lpi = 48.0/8.0;
		sprintf (lilbuff,"\033(8U\033(s0p10h12v0s0b3T\033&l8C");
		buff_pos = strlen(lilbuff);
		break;
	      case HPlaserjetplus:
		/* imode is used for dot size: 1 to 4 */
		if (resolution < 88)
		    {x_dpi = 75; imode = 4;}
		else if (resolution < 125)
		    {x_dpi = 100; imode = 3;}
		else if (resolution < 225)
		    {x_dpi = 150; imode = 2;}
		else
		    {x_dpi = 300; imode = 1;}

		y_dpi = x_dpi;
		x_pin = 8;
		y_pin = 1;
		mult = 1;
		i_fac = 0;
		mult1 = 1;
		gaps = 32;

		cpi = 10;
		lpi = 48.0/8.0;
		sprintf (lilbuff,"\033(8U\033(s0p10h12v0s0b3T\033&l8C");
		buff_pos = strlen(lilbuff);
		break;
	      case microline84:
		if (resolution < 66)
		    {x_dpi =  60; cpi = 10; lilbuff[buff_pos++] = chr_rs;}
		else /* 102 dpi is not offered, as line length is limited */
		    {x_dpi =  72; cpi = 12; lilbuff[buff_pos++] = chr_gs;}

		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"%";
		lilbuff[buff_pos++] = *"9";
		lilbuff[buff_pos++] = 16;		/* 16/144 inch line spacing */
		lpi = 144.0/16.0;

		imode = 0; /* unused */
		i_fac = -6;
		y_pin = 8;
		mult = 1;
		y_dpi = 72;
		x_pin = 1;
		mult1 = 6;
		gaps = 30;
		break;
	      default:
		status.all = status_$ok;
		generic_$print_error (status,"render","Graphics cannot be printed on this printer","",(long)On_);
		goto close_input_stream;
		break;
	      }


#if x_debug
#if x_debug_ctrl
	    fprintf (stderr,"Horizontal dots per inch is %d\n",x_dpi);
	    fprintf (stderr,"Vertical dots per inch is   %d\n",y_dpi);
	    fprintf (stderr,"Horizontal number of pins is %d\n",x_pin);
	    fprintf (stderr,"Vertical number of pins is   %d\n",y_pin);
	    fprintf (stderr,"Variable mult is             %d\n",mult);
	    fprintf (stderr,"Variable mult1 is            %d\n",mult1);
	    fprintf (stderr,"Variable i_fac is            %d\n",i_fac);
	    fprintf (stderr,"Variable imode is            %d\n",imode);
	    fprintf (stderr,"Number of blanks to skip is %d\n",gaps);
	    fprintf (stderr,"Lines per inch is      %f\n",lpi);
	    fprintf (stderr,"Characters per inch is %f\n",cpi);
#endif
#endif


	    /* None of the following checks should ever fail
	    if (! ((y_pin == 1 && x_pin == 8) || (y_pin == 8 && x_pin == 1) || (y_pin == 24 && x_pin == 1)))
		{
		sprintf (genbuff,"Wrong pin numbers x_pin = %d and y_pin = %d",x_pin,y_pin);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"Graphics cannot be printed",(long)On_);
		goto close_input_stream;
		}

	    if (x_dpi < 40 || x_dpi > 1000)
		{
		sprintf (genbuff,"Wrong horizontal dots_per_inch = %d",x_dpi);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"Graphics cannot be printed",(long)On_);
		goto close_input_stream;
		}

	    if (y_dpi < 40 || y_dpi > 1000)
		{
		sprintf (genbuff,"Wrong vertical dots_per_inch = %d",y_dpi);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"Graphics cannot be printed",(long)On_);
		goto close_input_stream;
		}
	    if ((y_pin > 1 && mult != y_pin / 8) || (y_pin == 1 && mult != 1))
		{
		sprintf (genbuff,"Wrong value for variable mult = %d (when y_pin = %d)",mult,y_pin);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"Graphics cannot be printed",(long)On_);
		goto close_input_stream;
		}
	    if (i_fac < -10 || i_fac > 10)
		{
		sprintf (genbuff,"Wrong value for variable i_fac = %d",i_fac);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"Graphics cannot be printed",(long)On_);
		goto close_input_stream;
		}
	    if (mult1 != mult && mult1 != mult * (i_fac))
		{
		sprintf (genbuff,"Wrong value for variable mult1 = %d (when mult = %d, i_fac = %d)",mult1,mult,i_fac);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"Graphics cannot be printed",(long)On_);
		goto close_input_stream;
		}
	    */


	    gpr_$inq_bitmap_dimensions (bitmap_desc,&bitmap_size,&hi_plane,&status);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to inquire bitmap dimensions","",(long)On_);
		goto close_input_stream;
		}

#if x_debug
#if x_debug_file
	    fprintf(stderr,"Bitmap width  = %d\n",bitmap_size.x_size);
	    fprintf(stderr,"Bitmap height = %d\n",bitmap_size.y_size);
	    fprintf(stderr,"Bitmap hi_plane = %d\n",hi_plane);
#endif
#endif

	    hi_plane = 0;


	    top_dot = 0;
	    if (landscape)
		bottom_dot = bitmap_size.x_size;
	    else
		bottom_dot = bitmap_size.y_size;
	    dots_long = bottom_dot - top_dot;

	    i = (((long)(paper_length * y_dpi)) / y_pin) * y_pin;
	    if (i < dots_long)
		{
		status.all = status_$ok;
		generic_$print_error (status,"render","Graphics too high for paper size, truncated","",(long)On_);
		i = (dots_long - i + 1) / 2;
		top_dot += i;
		bottom_dot -= i;
		dots_long = bottom_dot - top_dot;
		top_marg = 0;
		bottom_marg = 0;
		}
	    else
		{
		i = (((long)((paper_length - top_marg - bottom_marg) * y_dpi)) / y_pin) * y_pin;
		if (i < dots_long)
		    {
		    sprintf (genbuff,
			"Crazy margins:\n      top_margin    = %f\n      bottom_margin = %f\n      paper_length  = %f",
			top_marg,bottom_marg,paper_length);
		    status.all = status_$ok;
		    generic_$print_error (status,"render",genbuff,"Using default 0 margins",(long)On_);
		    line_number += 3;
		    top_marg = 0;
		    bottom_marg = 0;
		    }
		}

	    lines_paper = paper_length * lpi;
	    lines_high = (paper_length - bottom_marg) * lpi;
	    lines_marg = top_marg * lpi;
	    if (lines_paper > bufsize) lines_paper = bufsize;
	    if (lines_high > lines_paper) lines_high = lines_paper;
	    if (lines_marg >= lines_high) lines_marg = 0;
	    lines_page = lines_high - lines_marg;
	    lines_xtra = ((float)lines_page / lpi - (float)dots_long / (float)y_dpi) * lpi;
	    if (lines_xtra < 0) lines_xtra = 0;


#if x_debug
#if x_debug_ctrl
	    fprintf (stderr,"Paper length is %f\n",paper_length);
	    fprintf (stderr,"Top_margin is %f\n",top_marg);
	    fprintf (stderr,"Bottom_margin is %f\n",bottom_marg);
	    fprintf (stderr,"Print lines in page = %d, lines in margin = %d, total lines in paper = %d\n",lines_page,lines_marg,lines_paper);
#endif
#endif

/*xxxxx Add new printer here xxxxx*/
	    /* Set paper length */
	    switch (printer_model)
	      {
	      case epsonlq800:
	      case epsonlq1000:
	      case epsonlq1500:
	      case microline390:
	      case microline391:
	      case pinwriterp5:
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"C";
		lilbuff[buff_pos++] = chr_nul;
		lilbuff[buff_pos++] = generic_$float_to_long (paper_length);
		break;
	      case epsonlx80:
	      case epsonfx80:
	      case epsonfx85:
	      case epsonfx105:
	      case microline380:
	      case starnx1000:
		/*zzzzz Usually we use metric paper in fx-80, thus form length
		cannot be set as number of inches. Use number of lines instead.
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"C";
		lilbuff[buff_pos++] = chr_nul;
		lilbuff[buff_pos++] = generic_$float_to_long (paper_length); zzzzz*/
		if (lines_paper <= 127)
		    {
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"C";
		    lilbuff[buff_pos++] = lines_paper;
		    }
		break;
	      case imagewriterII:
		/* Old imagewriter cannot do */
		i = generic_$float_to_long (paper_length * 144);
		sprintf (&lilbuff[buff_pos],"\033H%04.4d",i);
		buff_pos += 6;
		break;
	      case HPlaserjet:
	      case HPlaserjetplus:
		lilbuff[buff_pos++] = chr_esc;
		sprintf (&lilbuff[buff_pos],"&l%dp0e%dF",lines_paper,lines_paper); /* Zero top or bottom margins */
		buff_pos += strlen(&lilbuff[buff_pos]);
		break;
	      case microline84:
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"G";
		i = generic_$float_to_long (paper_length * 2);
		sprintf (&lilbuff[buff_pos],"%02.2d",i);
		buff_pos += 2;
		break;
	      }


	    left_dot = 0;
	    if (landscape)
		right_dot = bitmap_size.y_size;
	    else
		right_dot = bitmap_size.x_size;
	    dots_wide = right_dot - left_dot;
	    if (graphsize < dots_wide)
		{
		sprintf (genbuff,"Graphics wider than %d pixels, truncated",graphsize);
		status.all = status_$ok;
		generic_$print_error (status,"render",genbuff,"",(long)On_);
		i = (dots_wide - graphsize + 1) / 2;
		left_dot += i;
		right_dot -= i;
		dots_wide = right_dot - left_dot;
		}

	    i = (((long)(paper_width * x_dpi)) / x_pin) * x_pin;
	    if (i < dots_wide)
		{
		status.all = status_$ok;
		generic_$print_error (status,"render","Graphics too wide for paper size, truncated","",(long)On_);
		i = (dots_wide - i + 1) / 2;
		left_dot += i;
		right_dot -= i;
		dots_wide = right_dot - left_dot;
		left_marg = 0;
		right_marg = 0;
		}
	    else
		{
		i = (((long)((paper_width - left_marg - right_marg) * x_dpi)) / x_pin) * x_pin;
		if (i < dots_wide)
		    {
		    sprintf (genbuff,
			"Crazy margins:\n      left_margin  = %f\n      right_margin = %f\n      paper_width  = %f",
			left_marg,right_marg,paper_width);
		    status.all = status_$ok;
		    generic_$print_error (status,"render",genbuff,"Using default 0 margins",(long)On_);
		    line_number += 3;
		    left_marg = 0;
		    right_marg = 0;
		    }
		}

	    chars_line = (paper_width - (left_marg + right_marg)) * cpi;
	    chars_marg = left_marg * cpi;
	    if (chars_line > bufsize) chars_line = bufsize;
	    if (chars_marg > bufsize) chars_marg = bufsize;


#if x_debug
#if x_debug_ctrl
	    fprintf (stderr,"Paper width is %f\n",paper_width);
	    fprintf (stderr,"Left_margin is %f\n",left_marg);
	    fprintf (stderr,"Right_margin is %f\n",right_marg);
	    fprintf (stderr,"Print chars in line = %d, chars in margin = %d\n",chars_line,chars_marg);
#endif
#endif


	    if (chars_marg > 0)
		{
/*xxxxx Add new printer here xxxxx*/
		/* Set left margin */
		switch (printer_model)
		  {
		  case epsonlq800:
		  case epsonlq1000:
		  case epsonlq1500:
		  case epsonlx80:
		  case epsonfx80:
		  case epsonfx85:
		  case epsonfx105:
		  case microline380:
		  case microline390:
		  case microline391:
		  case pinwriterp5:
		  case starnx1000:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"l";		/* left margin */
		    lilbuff[buff_pos++] = (char)chars_marg;
		    chars_marg = 0;
		    break;
		  case imagewriter:
		  case imagewriterII:
		    sprintf (&lilbuff[buff_pos],"\033L%03.3d",chars_marg);
		    buff_pos += 5;
		    chars_marg = 0;
		    break;
		  case HPlaserjet:
		  case HPlaserjetplus:
		    sprintf (&lilbuff[buff_pos],"\033&a%dL",chars_marg);
		    buff_pos += strlen(&lilbuff[buff_pos]);
		    chars_marg = 0;
		    break;
		  }
		}


	    if (header_valid)
		{
		if (lines_xtra < 2)
		    {
		    header_valid = false;
		    status.all = status_$ok;
		    generic_$print_error (status,"render","Header cannot be printed: not enough room","",(long)On_);
		    }
		else if (chars_line < 10)
		    {
		    header_valid = false;
		    status.all = status_$ok;
		    generic_$print_error (status,"render","Header cannot be printed: not enough room","Need at least 10 characters per line",(long)On_);
		    }
		else
		    {
		    lines_page -= 2;
		    lines_xtra -= 2;
		    }
		}


	    if (footer_valid)
		{
		if (lines_xtra < 2)
		    {
		    footer_valid = false;
		    status.all = status_$ok;
		    generic_$print_error (status,"render","Footer cannot be printed: not enough room","",(long)On_);
		    }
		else if (chars_line < 10)
		    {
		    footer_valid = false;
		    status.all = status_$ok;
		    generic_$print_error (status,"render","Footer cannot be printed: not enough room","Need at least 10 characters per line",(long)On_);
		    }
		else
		    {
		    lines_high -= 2;
		    lines_page -= 2;
		    lines_xtra -= 2;
		    }
		}


	    /* Allocate a main memory bitmap, to use as a buffer */

	    gpr_$allocate_attribute_block (&attribute_desc,&status);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to allocate attribute block","",(long)On_);
		goto close_input_stream;
		}


	    if (landscape)
		{
		my_bitmap_size.x_size = y_pin;
		my_bitmap_size.y_size = dots_wide;
		}
	    else
		{
		my_bitmap_size.x_size = dots_wide;
		my_bitmap_size.y_size = y_pin;
		}
	    my_hi_plane = 0;

	    gpr_$allocate_bitmap (my_bitmap_size,my_hi_plane,attribute_desc,&my_bitmap_desc,&status);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to allocate bitmap","",(long)On_);
		goto close_input_stream;
		}


	    gpr_$set_bitmap (my_bitmap_desc,&status);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to set bitmap as current","",(long)On_);
		goto close_input_stream;
		}


	    prim_set = (1 << (short)gpr_$rop_blt);
	    gpr_$raster_op_prim_set (prim_set,&status);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to select BLT as raster op primitive","",(long)On_);
		goto close_input_stream;
		}


	    if (reverse_bw == light || reverse_bw == dark)
		{
		bw_reverse = false;
		raster_op = gpr_$rop_src;
		gpr_$set_raster_op (my_hi_plane,raster_op,&status);
		if (status.all)
		    {
		    generic_$print_error (status,"render","Unable to set raster operation","",(long)On_);
		    goto close_input_stream;
		    }


		dots_long = y_pin;

		if (landscape)
		    {
		    src_window.window_size.x_size = dots_long;
		    src_window.window_size.y_size = dots_wide;
		    src_window.window_base.x_coord = top_dot;
		    src_window.window_base.y_coord = left_dot;

		    my_window.window_size.x_size = dots_long;
		    my_window.window_size.y_size = dots_wide;
		    my_window.window_base.x_coord = 0;
		    my_window.window_base.y_coord = 0;
		    }
		else
		    {
		    src_window.window_size.x_size = dots_wide;
		    src_window.window_size.y_size = dots_long;
		    src_window.window_base.x_coord = left_dot;
		    src_window.window_base.y_coord = top_dot;

		    my_window.window_size.x_size = dots_wide;
		    my_window.window_size.y_size = dots_long;
		    my_window.window_base.x_coord = 0;
		    my_window.window_base.y_coord = 0;
		    }

		pixel_a_last = dots_wide * dots_long;


		pixels_total = 0;
		pixels_dark = 0;


		for (k = top_dot; k < bottom_dot; k += y_pin)
		    {
		    if (landscape)
			src_window.window_base.x_coord = k;
		    else
			src_window.window_base.y_coord = k;

		    if (bottom_dot - k < y_pin)
			{
			dots_long = bottom_dot - k;
			if (landscape)
			    {
			    src_window.window_size.x_size = dots_long;
			    }
			else
			    {
			    src_window.window_size.y_size = dots_long;
			    my_window.window_size.y_size = dots_long;
			    pixel_a_last = dots_wide * dots_long;
			    }
			}


		    gpr_$bit_blt (bitmap_desc,src_window,hi_plane,my_window.window_base,my_hi_plane,&status);
		    if (status.all)
			{
			generic_$print_error (status,"render","Cannot BIT_BLT from input to temp bitmap","",(long)On_);
			goto cleanup;
			}


		    gpr_$read_pixels(my_window,pixel_a,&status);
		    if (status.all)
			{
			generic_$print_error (status,"render","Cannot read pixel values","",(long)On_);
			goto cleanup;
			}

		    if (dots_long < y_pin)
			{
			if (landscape)
			    {
			    for (i = dots_long; i < y_pin; i++) for (j = i; j < pixel_a_last; j += y_pin) pixel_a[j] = 0;
			    }
			}


		    pixels_total += pixel_a_last;

		    for (i = 0; i < pixel_a_last; i++) pixels_dark += pixel_a[i];
		    /* Could be as follows, but not needed: the temporary bitmap is single plane */
		    /* for (i = 0; i < pixel_a_last; i++) pixels_dark += (pixel_a[i] != 0); */
		    }


		if (reverse_bw == light)
		    {
		    if (pixels_dark > pixels_total / 2) bw_reverse = true;
		    }
		else
		    {
		    if (pixels_dark < pixels_total / 2) bw_reverse = true;
		    }
	    }


	    if (bw_reverse)
		raster_op = gpr_$rop_not_src;
	    else
		raster_op = gpr_$rop_src;
	    gpr_$set_raster_op (my_hi_plane,raster_op,&status);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to set raster operation","",(long)On_);
		goto close_input_stream;
		}


	    /* We send one formfeed at end (with possible footer) */
	    r0 = form_feed - 1;
	    status = set_db_value_c (obj_db,Form__Feeds,&r0);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to reset Form__Feeds","",(long)On_);
		}


/*xxxxx Add new printer here xxxxx*/
	    /* Set number of copies: each page sent just once, but collating messed up */
	    if (copies > 1)
		{
		switch (printer_model)
		  {
		  case HPlaserjet:
		  case HPlaserjetplus:
		    sprintf (&lilbuff[buff_pos],"\033&l%dX",copies);
		    buff_pos += strlen(&lilbuff[buff_pos]);
		    copies = 1;
		    break;
		  }
		}


	    if (line_number != 0)
		status2 = generic_$send_ff (false,"");


	    if (buff_pos > 0)
		{
		/* Get things under way */
		status = generic_$iosput (lilbuff,buff_pos);
		if (status.all)
		    {
		    generic_$print_error (status,"render","Unable to set up printer for graphics","",(long)Off_);
		    goto close_input_stream;
		    }
		}


	    line_number = 0;
	    need_lf = false;
	    page_number = 0;


	    for (copy = 0; copy < copies; copy++) /* Start copies loop */
	    {

	    status = generic_$render_check_room (header_valid,footer_valid,headerp,footerp);
	    if (status.all)
		{
		generic_$print_error (status,"render","Unable to send linefeeds for top margin and header string","",(long)Off_);
		goto cleanup;
		}


	    /* Margins to center graphics */
	    chars_marg = (((paper_width - left_marg - right_marg) * x_dpi - dots_wide) / (2 * x_dpi) + left_marg) * cpi;
	    if (chars_marg > bufsize) chars_marg = bufsize;

	    buff_pos = 0;

	    if (chars_marg > 0)
		{
/*xxxxx Add new printer here xxxxx*/
		/* Set left margin */
		switch (printer_model)
		  {
		  case epsonlq800:
		  case epsonlq1000:
		  case epsonlq1500:
		  case epsonlx80:
		  case epsonfx80:
		  case epsonfx85:
		  case epsonfx105:
		  case microline380:
		  case microline390:
		  case microline391:
		  case pinwriterp5:
		  case starnx1000:
		    lilbuff[buff_pos++] = chr_esc;
		    lilbuff[buff_pos++] = *"l";		/* left margin */
		    lilbuff[buff_pos++] = (char)chars_marg;
		    chars_marg = 0;
		    break;
		  case imagewriter:
		  case imagewriterII:
		    sprintf (&lilbuff[buff_pos],"\033L%03.3d",chars_marg);
		    buff_pos += 5;
		    chars_marg = 0;
		    break;
		  case HPlaserjet:
		  case HPlaserjetplus:
		    sprintf (&lilbuff[buff_pos],"\033&a%dL",chars_marg);
		    buff_pos += strlen(&lilbuff[buff_pos]);
		    chars_marg = 0;
		    break;
		  }
		}

	    if (buff_pos > 0)
		{
		status = generic_$iosput (lilbuff,buff_pos);
		if (status.all)
		    {
		    generic_$print_error (status,"render","Unable to set left margin for graphics","",(long)Off_);
		    goto cleanup;
		    }
		}


	    if (lines_xtra >= 2)
		{
		status = generic_$send_lf (lines_xtra / 2);
		if (status.all)
		    {
		    generic_$print_error (status,"render","Unable to send linefeeds to center graphics","",(long)Off_);
		    goto cleanup;
		    }
		}


	    buff_pos = 0;

/*xxxxx Add new printer here xxxxx*/
	    /* Initial setup for each graphics page */
	    switch (printer_model)
	      {
	      case HPlaserjet:
	      case HPlaserjetplus:
		sprintf (lilbuff,"\033*t%dR\033*r1A",x_dpi);
		buff_pos = strlen(lilbuff);
		break;
	      }

	    if (buff_pos > 0)
		{
		status = generic_$iosput (lilbuff,buff_pos);
		if (status.all)
		    {
		    generic_$print_error (status,"render","Unable to set up page for graphics","",(long)Off_);
		    goto cleanup;
		    }
		}


	    dots_long = y_pin;

	    if (landscape)
		{
		src_window.window_size.x_size = dots_long;
		src_window.window_size.y_size = dots_wide;
		src_window.window_base.x_coord = top_dot;
		src_window.window_base.y_coord = left_dot;

		my_window.window_size.x_size = dots_long;
		my_window.window_size.y_size = dots_wide;
		my_window.window_base.x_coord = 0;
		my_window.window_base.y_coord = 0;
		}
	    else
		{
		src_window.window_size.x_size = dots_wide;
		src_window.window_size.y_size = dots_long;
		src_window.window_base.x_coord = left_dot;
		src_window.window_base.y_coord = top_dot;

		my_window.window_size.x_size = dots_wide;
		my_window.window_size.y_size = dots_long;
		my_window.window_base.x_coord = 0;
		my_window.window_base.y_coord = 0;
		}

	    pixel_a_last = dots_wide * dots_long;

	    if (y_pin == 1)
		{
		buff_last = pixel_a_last / x_pin;
		if (buff_last * x_pin < pixel_a_last)
		    {
		    buff_last++;
		    for (i = (buff_last * x_pin) - 1; i >= pixel_a_last; i--) pixel_a[i] = 0;
		    }
		}
	    else if (y_pin == 8)
		buff_last = dots_wide;
	    else if (y_pin == 24)
		buff_last = dots_wide * 3;


	    raster_x_cur = 0;
	    raster_y_cur = 0;

	    raster_y_want = 0;


	    for (k = top_dot; k < bottom_dot; k += y_pin)
		{
		if (landscape)
		    src_window.window_base.x_coord = k;
		else
		    src_window.window_base.y_coord = k;

		if (bottom_dot - k < y_pin)
		    {
		    dots_long = bottom_dot - k;
		    if (landscape)
			{
			src_window.window_size.x_size = dots_long;
			}
		    else
			{
			src_window.window_size.y_size = dots_long;
			my_window.window_size.y_size = dots_long;
			pixel_a_last = dots_wide * dots_long;
			}
		    }


		gpr_$bit_blt (bitmap_desc,src_window,hi_plane,my_window.window_base,my_hi_plane,&status);
		if (status.all)
		    {
		    generic_$print_error (status,"render","Cannot BIT_BLT from input to temp bitmap","",(long)On_);
		    goto cleanup;
		    }


		gpr_$read_pixels(my_window,pixel_a,&status);
		if (status.all)
		    {
		    generic_$print_error (status,"render","Cannot read pixel values","",(long)On_);
		    goto cleanup;
		    }

		total_chars_read += pixel_a_last;

		if (dots_long < y_pin)
		    {
		    if (landscape)
			{
			for (i = dots_long; i < y_pin; i++) for (j = i; j < pixel_a_last; j += y_pin) pixel_a[j] = 0;
			}
		    else
			{
			for (i = (y_pin * dots_wide) - 1; i >= pixel_a_last; i--) pixel_a[i] = 0;
			}
		    }


		/* The following check is not needed: the temporary bitmap is single plane */
		/* for (i = 0; i < pixel_a_last; i++) pixel_a[i] = (pixel_a[i] != 0); */


		if (graph_border)
		    {
		    if (landscape)
			{
			for (i = 0; i < dots_long; i++) pixel_a[i] = 1;
			j = pixel_a_last - y_pin;
			for (i = j + dots_long - 1; i >= j; i--) pixel_a[i] = 1;
			if (k <= top_dot)		  for (i = 0; i < pixel_a_last; i += y_pin) pixel_a[i] = 1;
			else if (bottom_dot - k <= y_pin) for (i = dots_long - 1; i < pixel_a_last; i += y_pin) pixel_a[i] = 1;
			}
		    else
			{
			for (i = 0; i < pixel_a_last; i += dots_wide) pixel_a[i] = 1;
			for (i = dots_wide - 1; i < pixel_a_last; i += dots_wide) pixel_a[i] = 1;
			if (k <= top_dot)		  for (i = 0; i < dots_wide; i++) pixel_a[i] = 1;
			else if (bottom_dot - k <= y_pin) for (i = pixel_a_last - dots_wide; i < pixel_a_last; i++) pixel_a[i] = 1;
			}
		    }


		if (landscape)
		    {
		    if (y_pin == 1)
			{
			for (i = buff_last - 1, n = 0; i >= 0; i--)
			    {
			    n1 = n++;
			    n2 = n++;
			    n3 = n++;
			    n4 = n++;
			    n5 = n++;
			    n6 = n++;
			    n7 = n++;
			    n8 = n++;
			    bigbuff[i] = (char)((pixel_a[n1])
				+ (pixel_a[n2] << 1)
				+ (pixel_a[n3] << 2)
				+ (pixel_a[n4] << 3)
				+ (pixel_a[n5] << 4)
				+ (pixel_a[n6] << 5)
				+ (pixel_a[n7] << 6)
				+ (pixel_a[n8] << 7));
			    }
			}
		    else if (y_pin == 8)
			{
			if (imode == -1) /* imagewriters: top dot is least significant bit, bottom dot is most significant bit */
			    {
			    for (i = dots_wide - 1, n = 0; i >= 0; i--)
				{
				n1 = n++;
				n2 = n++;
				n3 = n++;
				n4 = n++;
				n5 = n++;
				n6 = n++;
				n7 = n++;
				n8 = n++;
				bigbuff[i] = (char)((pixel_a[n1])
				    + (pixel_a[n2] << 1)
				    + (pixel_a[n3] << 2)
				    + (pixel_a[n4] << 3)
				    + (pixel_a[n5] << 4)
				    + (pixel_a[n6] << 5)
				    + (pixel_a[n7] << 6)
				    + (pixel_a[n8] << 7));
				}
			    }
			else
			    {
			    for (i = dots_wide - 1, n = 0; i >= 0; i--)
				{
				n1 = n++;
				n2 = n++;
				n3 = n++;
				n4 = n++;
				n5 = n++;
				n6 = n++;
				n7 = n++;
				n8 = n++;
				bigbuff[i] = (char)((pixel_a[n1] << 7)
				    + (pixel_a[n2] << 6)
				    + (pixel_a[n3] << 5)
				    + (pixel_a[n4] << 4)
				    + (pixel_a[n5] << 3)
				    + (pixel_a[n6] << 2)
				    + (pixel_a[n7] << 1)
				    + (pixel_a[n8]));
				}
			    }
			}
		    else if (y_pin == 24)
			{
			for (j = buff_last - 3, n = 0; j >= 0; j -= 3)
			    {
			    n1 = n++;
			    n2 = n++;
			    n3 = n++;
			    n4 = n++;
			    n5 = n++;
			    n6 = n++;
			    n7 = n++;
			    n8 = n++;
			    bigbuff[j] = (char)((pixel_a[n1] << 7)
				+ (pixel_a[n2] << 6)
				+ (pixel_a[n3] << 5)
				+ (pixel_a[n4] << 4)
				+ (pixel_a[n5] << 3)
				+ (pixel_a[n6] << 2)
				+ (pixel_a[n7] << 1)
				+ (pixel_a[n8]));
			    n1 = n++;
			    n2 = n++;
			    n3 = n++;
			    n4 = n++;
			    n5 = n++;
			    n6 = n++;
			    n7 = n++;
			    n8 = n++;
			    bigbuff[j + 1] = (char)((pixel_a[n1] << 7)
				+ (pixel_a[n2] << 6)
				+ (pixel_a[n3] << 5)
				+ (pixel_a[n4] << 4)
				+ (pixel_a[n5] << 3)
				+ (pixel_a[n6] << 2)
				+ (pixel_a[n7] << 1)
				+ (pixel_a[n8]));
			    n1 = n++;
			    n2 = n++;
			    n3 = n++;
			    n4 = n++;
			    n5 = n++;
			    n6 = n++;
			    n7 = n++;
			    n8 = n++;
			    bigbuff[j + 2] = (char)((pixel_a[n1] << 7)
				+ (pixel_a[n2] << 6)
				+ (pixel_a[n3] << 5)
				+ (pixel_a[n4] << 4)
				+ (pixel_a[n5] << 3)
				+ (pixel_a[n6] << 2)
				+ (pixel_a[n7] << 1)
				+ (pixel_a[n8]));
			    }
			}
		    }
		else
		    {
		    if (y_pin == 1)
			{
			for (i = 0, n = 0; i < buff_last; i++)
			    {
			    n1 = n++;
			    n2 = n++;
			    n3 = n++;
			    n4 = n++;
			    n5 = n++;
			    n6 = n++;
			    n7 = n++;
			    n8 = n++;
			    bigbuff[i] = (char)((pixel_a[n1] << 7)
				+ (pixel_a[n2] << 6)
				+ (pixel_a[n3] << 5)
				+ (pixel_a[n4] << 4)
				+ (pixel_a[n5] << 3)
				+ (pixel_a[n6] << 2)
				+ (pixel_a[n7] << 1)
				+ (pixel_a[n8]));
			    }
			}
		    else if (y_pin == 8)
			{
			if (imode == -1) /* imagewriters: top dot is least significant bit, bottom dot is most significant bit */
			    {
			    for (i = 0; i < dots_wide; i++)
				{
				n1 = i;
				n2 = n1 + dots_wide;
				n3 = n2 + dots_wide;
				n4 = n3 + dots_wide;
				n5 = n4 + dots_wide;
				n6 = n5 + dots_wide;
				n7 = n6 + dots_wide;
				n8 = n7 + dots_wide;
				bigbuff[i] = (char)((pixel_a[n1])
				    + (pixel_a[n2] << 1)
				    + (pixel_a[n3] << 2)
				    + (pixel_a[n4] << 3)
				    + (pixel_a[n5] << 4)
				    + (pixel_a[n6] << 5)
				    + (pixel_a[n7] << 6)
				    + (pixel_a[n8] << 7));
				}
			    }
			else
			    {
			    for (i = 0; i < dots_wide; i++)
				{
				n1 = i;
				n2 = n1 + dots_wide;
				n3 = n2 + dots_wide;
				n4 = n3 + dots_wide;
				n5 = n4 + dots_wide;
				n6 = n5 + dots_wide;
				n7 = n6 + dots_wide;
				n8 = n7 + dots_wide;
				bigbuff[i] = (char)((pixel_a[n1] << 7)
				    + (pixel_a[n2] << 6)
				    + (pixel_a[n3] << 5)
				    + (pixel_a[n4] << 4)
				    + (pixel_a[n5] << 3)
				    + (pixel_a[n6] << 2)
				    + (pixel_a[n7] << 1)
				    + (pixel_a[n8]));
				}
			    }
			}
		    else if (y_pin == 24)
			{
			for (i = 0, j = 0; i < dots_wide; i++)
			    {
			    n1 = i;
			    n2 = n1 + dots_wide;
			    n3 = n2 + dots_wide;
			    n4 = n3 + dots_wide;
			    n5 = n4 + dots_wide;
			    n6 = n5 + dots_wide;
			    n7 = n6 + dots_wide;
			    n8 = n7 + dots_wide;
			    bigbuff[j++] = (char)((pixel_a[n1] << 7)
				+ (pixel_a[n2] << 6)
				+ (pixel_a[n3] << 5)
				+ (pixel_a[n4] << 4)
				+ (pixel_a[n5] << 3)
				+ (pixel_a[n6] << 2)
				+ (pixel_a[n7] << 1)
				+ (pixel_a[n8]));
			    n1 = n8 + dots_wide;
			    n2 = n1 + dots_wide;
			    n3 = n2 + dots_wide;
			    n4 = n3 + dots_wide;
			    n5 = n4 + dots_wide;
			    n6 = n5 + dots_wide;
			    n7 = n6 + dots_wide;
			    n8 = n7 + dots_wide;
			    bigbuff[j++] = (char)((pixel_a[n1] << 7)
				+ (pixel_a[n2] << 6)
				+ (pixel_a[n3] << 5)
				+ (pixel_a[n4] << 4)
				+ (pixel_a[n5] << 3)
				+ (pixel_a[n6] << 2)
				+ (pixel_a[n7] << 1)
				+ (pixel_a[n8]));
			    n1 = n8 + dots_wide;
			    n2 = n1 + dots_wide;
			    n3 = n2 + dots_wide;
			    n4 = n3 + dots_wide;
			    n5 = n4 + dots_wide;
			    n6 = n5 + dots_wide;
			    n7 = n6 + dots_wide;
			    n8 = n7 + dots_wide;
			    bigbuff[j++] = (char)((pixel_a[n1] << 7)
				+ (pixel_a[n2] << 6)
				+ (pixel_a[n3] << 5)
				+ (pixel_a[n4] << 4)
				+ (pixel_a[n5] << 3)
				+ (pixel_a[n6] << 2)
				+ (pixel_a[n7] << 1)
				+ (pixel_a[n8]));
			    }
			}
		    }


		if (gaps > 5)
		    {
		    buff_pos = 0;

		    /* See if first byte is zero or not, to initialize properly. */
		    if (bigbuff[buff_pos] == 0)
			{
			start0 = buff_pos;
			start1 = buff_last;
			goto zeros;
			}
		    else
			{
			start0 = buff_last;
			start1 = buff_pos;
			goto nonzeros;
			}

		    /* Last byte was zero. Skip through to the next non-zero. */
zeros:		    for (buff_pos++; buff_pos < buff_last; buff_pos++)
			{
			if (bigbuff[buff_pos] != 0)
			    {
			    if (start1 == buff_last)
				/* This is the very first non-zero in the line */
				start1 = buff_pos;
			    else if (buff_pos - start0 >= gaps)
				/* There have been enough zeros already (after a group of non-zeroes) */
				{
				status = generic_$raster_chunk (start1,start0,
					mult,mult1,i_fac,imode);
				if (status.all) goto cleanup;
				start1 = buff_pos;
				}
			    /* We are sure of finding the next zero there */
			    start0 = buff_last;
			    goto nonzeros;
			    }
			}
		    goto endgrline;

		    /* Last byte was non-zero. Skip through to the next zero. */
nonzeros:	    for (buff_pos++; buff_pos < buff_last; buff_pos++)
			{
			if (bigbuff[buff_pos] == 0)
			    {
			    start0 = buff_pos;
			    goto zeros;
			    }
			}
		    goto endgrline;

		    /* Finished this line. Any data left? */
endgrline:	    if (start1 < buff_last)
			{
			status = generic_$raster_chunk (start1,start0,
				mult,mult1,i_fac,imode);
			if (status.all) goto cleanup;
			}
		    }
		else
		    {
		    status = generic_$raster_chunk (0,buff_last,
			    mult,mult1,i_fac,imode);
		    if (status.all) goto cleanup;
		    }


		raster_y_want++;
		total_lines_job++;
		}


cleanup:    buff_pos = 0;

/*xxxxx Add new printer here xxxxx*/
	    /* Clean up after graphics */
	    switch (printer_model)
	      {
	      case HPlaserjet:
	      case HPlaserjetplus:
		sprintf (bigbuff,"\033*rB");
		buff_pos = 4;
		/* Update line_number now, was not updated for each vertical move in generic_$raster_chunk */
		line_number += (long)((float)raster_y_cur / (float)y_dpi * lpi);
		break;
	      }

	    if (buff_pos > 0)
		{
		status2 = generic_$iosput (bigbuff,buff_pos);
		if (status2.all)
		    {
		    generic_$print_error (status2,"render","Unable to clean-up after graphics","",(long)Off_);
		    if (! status.all) status = status2;
		    }
		}


	    /* Set margin back for text (footer) */
	    chars_marg = left_marg * cpi;
	    if (chars_marg > bufsize) chars_marg = bufsize;

	    buff_pos = 0;

/*xxxxx Add new printer here xxxxx*/
	    /* Set left margin */
	    switch (printer_model)
	      {
	      case epsonlq800:
	      case epsonlq1000:
	      case epsonlq1500:
	      case epsonlx80:
	      case epsonfx80:
	      case epsonfx85:
	      case epsonfx105:
	      case microline380:
	      case microline390:
	      case microline391:
	      case pinwriterp5:
	      case starnx1000:
		lilbuff[buff_pos++] = chr_esc;
		lilbuff[buff_pos++] = *"l";	     /* left margin */
		lilbuff[buff_pos++] = (char)chars_marg;
		chars_marg = 0;
		break;
	      case imagewriter:
	      case imagewriterII:
		sprintf (&lilbuff[buff_pos],"\033L%03.3d",chars_marg);
		buff_pos += 5;
		chars_marg = 0;
		break;
	      case HPlaserjet:
	      case HPlaserjetplus:
		sprintf (&lilbuff[buff_pos],"\033&a%dL",chars_marg);
		buff_pos += strlen(&lilbuff[buff_pos]);
		chars_marg = 0;
		break;
	      }


	    if (buff_pos > 0)
		{
		status2 = generic_$iosput (lilbuff,buff_pos);
		if (status2.all)
		    {
		    generic_$print_error (status2,"render","Unable to set left margin for text","",(long)Off_);
		    if (! status.all) status = status2;
		    }
		}


	    status2 = generic_$send_ff (footer_valid,footerp);
	    if (status2.all)
		{
		generic_$print_error (status2,"render","Unable to send end_of_copy FF","",(long)Off_);
		if (! status.all) status = status2;
		}

	    if (status.all) break;

	    } /* end copies loop */


	    total_pages_job = page_number;

	    if (copies == 1)
		{
		fprintf (stderr,"Total %d pixels processed; %d characters printed in %d lines, %d pages\n",
		    total_chars_read,total_chars_job,total_lines_job,total_pages_job);
		}
	    else
		{
		fprintf (stderr,"Total %d pixels processed; %d characters printed in %d lines, %d pages (in %d copies)\n",
		    total_chars_read,total_chars_job,total_lines_job,total_pages_job,copies);
		}

	    if (total_chars_job  > big_job_chars ||
		total_lines_job  > big_job_lines ||
		total_pages_job  > big_job_pages ) fprintf (stderr,"WARNING: this was a big job.\n");

	    total_chars_bm1 += total_chars_job;
	    total_lines_bm1 += total_lines_job;
	    total_pages_bm1 += total_pages_job;


	    } /* end BM1_ file */


close_input_stream:
    /* Attempt to close input stream */
    if (mngr_type == mgr__ios || mngr_type == 2048 || mngr_type == 56640 || mngr_type == 992347032 || mngr_type == 992412568)
    /*zzzzz BUG: How could it be 2048 etc.? May have been related to the drain_out error. zzzzz*/
	{
	if (mngr_type != mgr__ios)
	    {
	    sprintf (genbuff,"Wrong mngr_type = %d, must be IOS (%d) or GPR (%d)",mngr_type,mgr__ios,mgr__gpr);
	    status2.all = status_$ok;
	    generic_$print_error (status2,"render",genbuff,"This error ignored, mngr_type set to 0",(long)Off_);
	    }

	in_strid = (ios_$id_t)*obj_id;
	ios_$close (in_strid,&status2);
	if (status2.all)
	    {
	    generic_$print_error (status2,"render","Unable to close input IOS stream","",(long)On_);
	    }
	if (! status.all) status = status2;
	}
    else if (mngr_type == mgr__gpr)
	{
	if (my_bitmap_desc != gpr_$nil_bitmap_desc)
	    {
	    gpr_$deallocate_bitmap (my_bitmap_desc,&status2);
	    if (status2.all)
		{
		generic_$print_error (status2,"render","Unable to deallocate temp bitmap","",(long)On_);
		}
	    if (! status.all) status = status2;
	    }


	if (attribute_desc != gpr_$nil_attribute_desc)
	    {
	    gpr_$deallocate_attribute_block (attribute_desc,&status2);
	    if (status2.all)
		{
		generic_$print_error (status2,"render","Unable to deallocate attribute block","",(long)On_);
		}
	    if (! status.all) status = status2;
	    }


	bitmap_desc = (gpr_$bitmap_desc_t)*obj_id;
	gpr_$deallocate_bitmap (bitmap_desc,&status2);
	if (status2.all)
	    {
	    generic_$print_error (status2,"render","Unable to close input GPR stream","",(long)On_);
	    }
	if (! status.all) status = status2;
	}
    else
	{
	sprintf (genbuff,"Wrong mngr_type = %d, must be IOS (%d) or GPR (%d)",mngr_type,mgr__ios,mgr__gpr);
	status2.all = status_$ok;
	generic_$print_error (status2,"render","Cannot close this input stream type",genbuff,(long)On_);
	}


    return (status);
}

/* =========================================================================== */

void generic_$date_time(
  char date[32])

{
    cal_$timedate_rec_t ldt;
    char *mm;

    cal_$decode_local_time (&ldt);

    if (ldt.month == 1)
      mm = "Jan";
    else if (ldt.month == 2)
      mm = "Feb";
    else if (ldt.month == 3)
      mm = "Mar";
    else if (ldt.month == 4)
      mm = "Apr";
    else if (ldt.month == 5)
      mm = "May";
    else if (ldt.month == 6)
      mm = "Jun";
    else if (ldt.month == 7)
      mm = "Jul";
    else if (ldt.month == 8)
      mm = "Aug";
    else if (ldt.month == 9)
      mm = "Sep";
    else if (ldt.month == 10)
      mm = "Oct";
    else if (ldt.month == 11)
      mm = "Nov";
    else if (ldt.month == 12)
      mm = "Dec";
    else
      mm = "???";

    sprintf (date,"%2.d %s %4.d, %2.d:%02.d:%02.d",ldt.day,mm,ldt.year,ldt.hour,ldt.minute,ldt.second);
}

/* =========================================================================== */

#if SR_level == SR10_2 || SR_level == SR10_3
void generic_$print_wrong_value (
  char *option,
  char *value)

{
    status_$t status;

    sprintf (genbuff,"Wrong value <%.40s> for option -%s",value,option);
    generic_$cleanup_string (genbuff);
    status.all = status_$ok;
    generic_$print_error (status,"render",genbuff,"Option ignored",(long)On_);
}
#endif

/* =========================================================================== */

#if SR_level == SR10_2 || SR_level == SR10_3
void generic_$print_need_passwd (
  char *msg)

{
    status_$t status;

    sprintf (genbuff,"You need a correct '-passwd password' option with %s",msg);
    status.all = status_$ok;
    generic_$print_error (status,"render",genbuff,"This print file rejected",(long)On_);
}
#endif

/* =========================================================================== */

status_$t generic_$read_config (
  char *file_namep,
  db_obj ren_db,
  db_obj usr_db,
  long i_o_mode,
  long printer_model)

{
    status_$t status,status2;
    char *namep,*strp;
    char *words[3];
    ios_$id_t config_ios_id;
    long i,len,word_count,error_count;
    boolean next_word;
    float r1,r2;
    long j1,j2;


#if SR_level == SR10_2 || SR_level == SR10_3
    char passwd_file[bufsize];
    char *passwd_filep;
#endif

    long reinit_line;

    float sio_stop_bits;
    long sio_parity;
    long sio_bits_per_char;
    long sio_nlc_delay;
    long sio_bp_enable;
    long sio_cts_enable;
    long sio_dcd_enable;
    long sio_drain_out;
    long sio_flush_in;
    long sio_flush_out;
    long sio_host_sync;
    long sio_hup_close;
    long sio_input_sync;
    long sio_no_NL;
    long sio_no_echo;
    long sio_quit_enable;
    long sio_raw;
    long sio_raw_nl;
    long sio_rts_enable;
    long sio_susp_enable;

#if x_SPE_installed
    long pio_handshake;
    pio_$status_check_t pio_status_chk;
#endif

    long end_of_line;
    long use_cr_without_lf;

    long big_job_chars;
    long big_job_lines;
    long big_job_pages;
    long reject_big_jobs;

    long tab_length;
    long word_wrap;
    char eightbit;
    long ctrl_char;

#if SR_level == SR10_0 || SR_level == SR10_1
    long allow_TRANS;
    long allow_bold;
    long allow_lq;
#endif

    long graph_border;
    long reverse_bw;

/*  The following are only for the dumb_ascii printer. */
    long dumb_use_FF;
    long dumb_use_LF_before_FF;
    long dumb_use_BS;
    long dumb_use_tab;

    float dumb_point;
    float dumb_lpi;
    float dumb_bottom_margin_low;
    float dumb_bottom_margin_high;
    float dumb_top_margin_low;
    float dumb_top_margin_high;
    float dumb_left_margin_low;
    float dumb_left_margin_high;
    float dumb_right_margin_low;
    float dumb_right_margin_high;
    float dumb_paper_width;
    float dumb_paper_length;
    long dumb_paper_type;

    boolean dumb_set_point;
    boolean dumb_set_lpi;
    boolean dumb_set_bottom_margin;
    boolean dumb_set_top_margin;
    boolean dumb_set_left_margin;
    boolean dumb_set_right_margin;
    boolean dumb_set_paper_width;
    boolean dumb_set_paper_length;
    boolean dumb_set_paper_type;


#if SR_level == SR10_2 || SR_level == SR10_3
    passwd_filep = "/etc/daemons/generic_passwd";
#endif

    reinit_line		= on;

    sio_stop_bits	= 1.0;
    sio_parity		= none;
    sio_bits_per_char	= 8;
    sio_nlc_delay	= 0;
    sio_bp_enable	= off;
    sio_cts_enable	= off;
    sio_dcd_enable	= off;
    sio_drain_out	= on;
    sio_flush_in	= off;
    sio_flush_out	= off;
    sio_host_sync	= on;
    sio_hup_close	= off;
    sio_input_sync	= on;
    sio_no_NL		= off;
    sio_no_echo		= on;
    sio_quit_enable	= off;
    sio_raw		= off;
    sio_raw_nl		= off;
    sio_rts_enable	= off;
    sio_susp_enable	= off;

#if x_SPE_installed
    pio_handshake  = none;
    pio_status_chk = spe_$no_status_check;
#endif

    end_of_line		= eol_lf;
    use_cr_without_lf	= off;

    big_job_chars	= 30;
    big_job_lines	= 600;
    big_job_pages	= 20;
    reject_big_jobs	= off;

    tab_length		= 8;
    word_wrap		= on;
    eightbit		= *"~";
    ctrl_char		= ctrl_caret;

#if SR_level == SR10_0 || SR_level == SR10_1
    allow_TRANS	= off;
    allow_bold	= off;
    allow_lq	= off;
#endif

    graph_border	= on;
    reverse_bw		= light;

    dumb_use_FF			= on;
    dumb_use_LF_before_FF	= off;
    dumb_use_BS			= on;
    dumb_use_tab		= off;

    dumb_point			= 12;
    dumb_lpi			= 6;
    dumb_bottom_margin_low	= 0;
    dumb_bottom_margin_high	= 4;
    dumb_top_margin_low		= 0;
    dumb_top_margin_high	= 4;
    dumb_left_margin_low	= 0;
    dumb_left_margin_high	= 4;
    dumb_right_margin_low	= 0;
    dumb_right_margin_high	= 4;
    dumb_paper_width		= 13.6;
    dumb_paper_length		= 11;
    dumb_paper_type		= FanFold_;

    dumb_set_point		= false;
    dumb_set_lpi		= false;
    dumb_set_bottom_margin	= false;
    dumb_set_top_margin		= false;
    dumb_set_left_margin	= false;
    dumb_set_right_margin	= false;
    dumb_set_paper_width	= false;
    dumb_set_paper_length	= false;
    dumb_set_paper_type		= false;


    namep = file_namep;
    len = strlen(namep);
    if (len <= 0)
	len = 0;
    else if (len < 4 || len > bufsize)
	{
	status.all = status_$ok;
	sprintf (genbuff,"Wrong name <%.80s> of config file",namep);
	generic_$print_error (status,"read_config",genbuff,"",(long)Off_);
	status.all = print_$wrong_db_entry_type;
	return (status);
	}
    else
	{
	sprintf (genbuff,"%.5s",namep);
	if (generic_$strcmp(genbuff,"/dev/",false)) len = 0;
	}

    if (len == 0)
	{
	status.all = status_$ok;
	sprintf (genbuff,"Wrong name <%.80s> of config file",namep);
	generic_$print_error (status,"read_config",genbuff,"Using default /etc/daemons/generic_config instead",(long)Off_);
	namep = "/etc/daemons/generic_config";
	len = strlen(namep);
	}


    config_ios_id = ios_$open (namep,(short)len,ios_$no_open_options,&status);
    if (status.all)
	{
	sprintf (genbuff,"Unable to open config file <%s>",namep);
	generic_$print_error (status,"read_config",genbuff,"",(long)Off_);
	return (status);
	}


    error_count = 0;

    while (true)
	{
	if (error_count >= 10)
	    {
	    status2.all = status_$ok;
	    sprintf (genbuff,"Reading of config file <%s> abandoned",namep);
	    generic_$print_error (status2,"read_config","Error limit exceeded",genbuff,(long)Off_);
	    break;
	    }

	len = ios_$get (config_ios_id,ios_$no_put_get_opts,lilbuff,(long)bufsize,&status2);
	if (status2.all)
	    {
	    if (status2.all == ios_$end_of_file) /* expected */ break;
	    sprintf (genbuff,"Unable to read config file <%s>",namep);
	    generic_$print_error (status2,"read_config",genbuff,"",(long)Off_);
	    status = status2;
	    break;
	    }

	len--; /* ignore the final LF we get in each line */
	     if (len <= 0) continue;
	else if (len >= bufsize ) len = bufsize - 1;
	lilbuff[len] = chr_nul;
#if x_debug
	fprintf (stderr,"Line <%s> read from file <%s>\n",lilbuff,namep);
#endif

	words[0] = "";
	words[1] = "";
	words[2] = "";
	for (strp = lilbuff, word_count = 0, next_word = true; *strp != chr_nul; strp++)
	    {
	    if (*strp <= chr_sp || *strp >= chr_del) { *strp = chr_nul; next_word = true; }
	    else if (*strp == *"#") { *strp = chr_nul; break; }
	    else if (next_word)
		{
		if (word_count >= 3) { word_count++; break; }
		words[word_count++] = strp;
		next_word = false;
		}
	    }

#if x_debug
	fprintf (stderr,"%d words in line: <%s>, <%s>, <%s>\n",word_count,words[0],words[1],words[2]);
#endif

	if (word_count <= 0) continue;

	strp = words[0];
	word_count--;

	if (generic_$strcmp(strp,"passwd_file",true))
	    {
	    status2 = generic_$check_config (word_count,words,"any_string",(long)0,(long)0,&r1,&r2,&j1,&j2);
#if SR_level == SR10_0 || SR_level == SR10_1
	    generic_$print_error (status2,"read_config","Option passwd_file in config file is only for SR10.2 (and above)","This option ignored",(long)Off_);
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
	    sprintf (passwd_file,"%s",words[1]);
	    passwd_filep = passwd_file;
#endif
	    }
	else if (generic_$strcmp(strp,"reinit_line",true))
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&reinit_line,&j2);
	else if (generic_$strcmp(strp,"sio_stop_bits",true))
	    status2 = generic_$check_config_sio (word_count,words,"float",(long)1,(long)2,i_o_mode,&sio_stop_bits,&j1);
	else if (generic_$strcmp(strp,"sio_parity",true))
	    status2 = generic_$check_config_sio (word_count,words,"#none#odd#even#",(long)0,(long)0,i_o_mode,&r1,&sio_parity);
	else if (generic_$strcmp(strp,"sio_bits_per_char",true))
	    status2 = generic_$check_config_sio (word_count,words,"integer",(long)5,(long)8,i_o_mode,&r1,&sio_bits_per_char);
	else if (generic_$strcmp(strp,"sio_nlc_delay",true))
	    status2 = generic_$check_config_sio (word_count,words,"integer",(long)0,(long)10000,i_o_mode,&r1,&sio_nlc_delay);
	else if (generic_$strcmp(strp,"sio_bp_enable",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_bp_enable);
	else if (generic_$strcmp(strp,"sio_cts_enable",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_cts_enable);
	else if (generic_$strcmp(strp,"sio_dcd_enable",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_dcd_enable);
	else if (generic_$strcmp(strp,"sio_drain_out",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_drain_out);
	else if (generic_$strcmp(strp,"sio_flush_in",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_flush_in);
	else if (generic_$strcmp(strp,"sio_flush_out",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_flush_out);
	else if (generic_$strcmp(strp,"sio_host_sync",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_host_sync);
	else if (generic_$strcmp(strp,"sio_hup_close",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_hup_close);
	else if (generic_$strcmp(strp,"sio_input_sync",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_input_sync);
	else if (generic_$strcmp(strp,"sio_no_NL",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_no_NL);
	else if (generic_$strcmp(strp,"sio_no_echo",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_no_echo);
	else if (generic_$strcmp(strp,"sio_quit_enable",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_quit_enable);
	else if (generic_$strcmp(strp,"sio_raw",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_raw);
	else if (generic_$strcmp(strp,"sio_raw_nl",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_raw_nl);
	else if (generic_$strcmp(strp,"sio_rts_enable",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_rts_enable);
	else if (generic_$strcmp(strp,"sio_susp_enable",true))
	    status2 = generic_$check_config_sio (word_count,words,"#off#on#",(long)0,(long)0,i_o_mode,&r1,&sio_susp_enable);
	else if (generic_$strcmp(strp,"pio_handshake",true))
	    {
	    status2 = generic_$check_config (word_count,words,"#none#busy#ack#ack_busy#",(long)0,(long)0,&r1,&r2,&j1,&j2);
#if x_SPE_installed
	    if (i_o_mode == prsvr__Io_Parallel) pio_handshake = j1;
	    else
#endif
		generic_$read_config_not_parallel (strp);
	    }
	else if (generic_$strcmp(strp,"pio_status_chk",true))
	    {
	    status2 = generic_$check_config (word_count,words,"#none#busy#pe#sel#fault#wait_status#",(long)0,(long)0,&r1,&r2,&j1,&j2);
#if x_SPE_installed
	    if (i_o_mode == prsvr__Io_Parallel)
		{
		     if (j1 == 1) pio_status_chk |= spe_$chk_busy;
		else if (j1 == 2) pio_status_chk |= spe_$chk_pe;
		else if (j1 == 3) pio_status_chk |= spe_$chk_sel;
		else if (j1 == 4) pio_status_chk |= spe_$chk_fault;
		else if (j1 == 5) pio_status_chk |= spe_$wait_status;
		else pio_status_chk = spe_$no_status_check;
		}
	    else
#endif
		generic_$read_config_not_parallel (strp);
	    }
	else if (generic_$strcmp(strp,"end_of_line",true))
	    status2 = generic_$check_config (word_count,words,"#cr#lf#cr/lf#lf/cr#",(long)0,(long)0,&r1,&r2,&end_of_line,&j2);
	else if (generic_$strcmp(strp,"use_cr_without_lf",true))
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&use_cr_without_lf,&j2);
	else if (generic_$strcmp(strp,"big_job_chars",true))
	    status2 = generic_$check_config (word_count,words,"integer",(long)2,(long)huge_job_chars,&r1,&r2,&big_job_chars,&j2);
	else if (generic_$strcmp(strp,"big_job_lines",true))
	    status2 = generic_$check_config (word_count,words,"integer",(long)50,(long)huge_job_lines,&r1,&r2,&big_job_lines,&j2);
	else if (generic_$strcmp(strp,"big_job_pages",true))
	    status2 = generic_$check_config (word_count,words,"integer",(long)1,(long)huge_job_pages,&r1,&r2,&big_job_pages,&j2);
	else if (generic_$strcmp(strp,"reject_big_jobs",true))
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&reject_big_jobs,&j2);
	else if (generic_$strcmp(strp,"tab_length",true))
	    status2 = generic_$check_config (word_count,words,"integer",(long)1,(long)20,&r1,&r2,&tab_length,&j2);
	else if (generic_$strcmp(strp,"word_wrap",true))
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&word_wrap,&j2);
	else if (generic_$strcmp(strp,"eightbit",true))
	    {
	    status2 = generic_$check_config (word_count,words,"any_string",(long)0,(long)0,&r1,&r2,&j1,&j2);
		 if (generic_$strcmp(words[1],"ignore",true))	eightbit = eightbit_ignore;
	    else if (generic_$strcmp(words[1],"blank",true))	eightbit = eightbit_blank;
	    else if (generic_$strcmp(words[1],"value",true))	eightbit = eightbit_value;
	    else if (generic_$strcmp(words[1],"convert",true))	eightbit = eightbit_convert;
	    else if (strlen(words[1]) == 1 && *words[1] > chr_sp && *words[1] < chr_del) eightbit = *words[1];
	    }
	else if (generic_$strcmp(strp,"ctrl_char",true))
	    status2 = generic_$check_config (word_count,words,"#ignore#blank#value#ascii#control#caret#",(long)0,(long)0,&r1,&r2,&ctrl_char,&j2);
	else if (generic_$strcmp(strp,"allow_TRANS",true))
#if SR_level == SR10_0 || SR_level == SR10_1
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&allow_TRANS,&j2);
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
	    {
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&j1,&j2);
	    generic_$read_config_obsolete (strp);
	    }
#endif
	else if (generic_$strcmp(strp,"allow_bold",true))
#if SR_level == SR10_0 || SR_level == SR10_1
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&allow_bold,&j2);
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
	    {
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&j1,&j2);
	    generic_$read_config_obsolete (strp);
	    }
#endif
	else if (generic_$strcmp(strp,"allow_lq",true))
#if SR_level == SR10_0 || SR_level == SR10_1
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&allow_lq,&j2);
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
	    {
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&j1,&j2);
	    generic_$read_config_obsolete (strp);
	    }
#endif
	else if (generic_$strcmp(strp,"graph_border",true))
	    status2 = generic_$check_config (word_count,words,"#off#on#",(long)0,(long)0,&r1,&r2,&graph_border,&j2);
	else if (generic_$strcmp(strp,"reverse_bw",true))
	    {
	    status2 = generic_$check_config (word_count,words,"#off#light#dark#",(long)0,(long)0,&r1,&r2,&reverse_bw,&j2);
	    if (reverse_bw > 0) reverse_bw++;
	    }
	else if (generic_$strcmp(strp,"dumb_use_FF",true))
	    status2 = generic_$check_config_dumb (word_count,words,"#off#on#",(long)0,(long)0,printer_model,&r1,&dumb_use_FF);
	else if (generic_$strcmp(strp,"dumb_use_LF_before_FF",true))
	    status2 = generic_$check_config_dumb (word_count,words,"#off#on#",(long)0,(long)0,printer_model,&r1,&dumb_use_LF_before_FF);
	else if (generic_$strcmp(strp,"dumb_use_BS",true))
	    status2 = generic_$check_config_dumb (word_count,words,"#off#on#one_at_a_time#",(long)0,(long)0,printer_model,&r1,&dumb_use_BS);
	else if (generic_$strcmp(strp,"dumb_use_tab",true))
	    status2 = generic_$check_config_dumb (word_count,words,"#off#on#",(long)0,(long)0,printer_model,&r1,&dumb_use_tab);
	else if (generic_$strcmp(strp,"dumb_point",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float",(long)4,(long)30,printer_model,ren_db,Point_,&dumb_set_point);
	else if (generic_$strcmp(strp,"dumb_point_range",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float_range",(long)4,(long)30,printer_model,ren_db,Point_,&dumb_set_point);
	else if (generic_$strcmp(strp,"dumb_lpi",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float",(long)1,(long)20,printer_model,ren_db,Lpi_,&dumb_set_lpi);
	else if (generic_$strcmp(strp,"dumb_lpi_range",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float_range",(long)1,(long)20,printer_model,ren_db,Lpi_,&dumb_set_lpi);
	else if (generic_$strcmp(strp,"dumb_bottom_margin",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float",(long)0,(long)20,printer_model,ren_db,Bottom__Margin,&dumb_set_bottom_margin);
	else if (generic_$strcmp(strp,"dumb_bottom_margin_range",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float_range",(long)0,(long)20,printer_model,ren_db,Bottom__Margin,&dumb_set_bottom_margin);
	else if (generic_$strcmp(strp,"dumb_top_margin",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float",(long)0,(long)20,printer_model,ren_db,Top__Margin,&dumb_set_top_margin);
	else if (generic_$strcmp(strp,"dumb_top_margin_range",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float_range",(long)0,(long)20,printer_model,ren_db,Top__Margin,&dumb_set_top_margin);
	else if (generic_$strcmp(strp,"dumb_left_margin",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float",(long)0,(long)20,printer_model,ren_db,Left__Margin,&dumb_set_left_margin);
	else if (generic_$strcmp(strp,"dumb_left_margin_range",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float_range",(long)0,(long)20,printer_model,ren_db,Left__Margin,&dumb_set_left_margin);
	else if (generic_$strcmp(strp,"dumb_right_margin",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float",(long)0,(long)20,printer_model,ren_db,Right__Margin,&dumb_set_right_margin);
	else if (generic_$strcmp(strp,"dumb_right_margin_range",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float_range",(long)0,(long)20,printer_model,ren_db,Right__Margin,&dumb_set_right_margin);
	else if (generic_$strcmp(strp,"dumb_paper_width",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float",(long)1,(long)20,printer_model,ren_db,Paper__Width,&dumb_set_paper_width);
	else if (generic_$strcmp(strp,"dumb_paper_width_range",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float_range",(long)1,(long)20,printer_model,ren_db,Paper__Width,&dumb_set_paper_width);
	else if (generic_$strcmp(strp,"dumb_paper_length",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float",(long)1,(long)20,printer_model,ren_db,Paper__Length,&dumb_set_paper_length);
	else if (generic_$strcmp(strp,"dumb_paper_length_range",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"float_range",(long)1,(long)20,printer_model,ren_db,Paper__Length,&dumb_set_paper_length);
	else if (generic_$strcmp(strp,"dumb_paper_type",true))
	    status2 = generic_$check_config_dumb_add (word_count,words,"#cut#fanfold#roll#transpar#",(long)0,(long)0,printer_model,ren_db,Paper__Type,&dumb_set_paper_type);
	else
	    {
	    status2.all = print_$prf_option_unknown;
	    sprintf (genbuff,"Unrecognized option %s in config file",strp);
	    generic_$print_error (status2,"read_config",genbuff,"",(long)Off_);
	    }


	if (status2.all)
	    {
	    if (! status.all) status = status2;
	    error_count++;
	    }
	}


    ios_$close (config_ios_id,&status2);
    if (status2.all)
	{
	sprintf (genbuff,"Unable to close config file <%s>",namep);
	generic_$print_error (status2,"read_config",genbuff,"",(long)Off_);
	if (! status.all) status = status2;
	}


    if (status.all) return (status);


#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$add_string (usr_db,"usr_db","Passwd__File",passwd_filep,(long)Off_);
    if (status.all) return (status);

    /* Check that the password file exists (or has been set to none) */
    status = generic_$check_passwd ("password",&next_word);
    if (status.all) return (status);
#endif


    r1 = (float)reinit_line;
    status = generic_$add_value (usr_db,"usr_db","Reinit__Line",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);


    if (i_o_mode == prsvr__Io_Serial)
	{
	r1 = sio_stop_bits;
	status = generic_$add_value (usr_db,"usr_db","Device__Stop_bits",r1,(long)1,(long)2,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_parity;
	status = generic_$add_value (usr_db,"usr_db","Device__Parity",r1,(long)none,(long)even,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_bits_per_char;
	status = generic_$add_value (usr_db,"usr_db","Device__Bits_per_char",r1,(long)5,(long)8,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_nlc_delay;
	status = generic_$add_value (usr_db,"usr_db","Device__Nlc_delay",r1,(long)0,(long)10000,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_bp_enable;
	status = generic_$add_value (usr_db,"usr_db","Device__Bp_enable",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_cts_enable;
	status = generic_$add_value (usr_db,"usr_db","Device__Cts_enable",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_dcd_enable;
	status = generic_$add_value (usr_db,"usr_db","Device__Dcd_enable",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_drain_out;
	status = generic_$add_value (usr_db,"usr_db","Device__Drain_out",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_flush_in;
	status = generic_$add_value (usr_db,"usr_db","Device__Flush_in",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_flush_out;
	status = generic_$add_value (usr_db,"usr_db","Device__Flush_out",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_host_sync;
	status = generic_$add_value (usr_db,"usr_db","Device__Host_sync",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_hup_close;
	status = generic_$add_value (usr_db,"usr_db","Device__Hup_close",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_input_sync;
	status = generic_$add_value (usr_db,"usr_db","Device__Input_sync",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_no_NL;
	status = generic_$add_value (usr_db,"usr_db","Device__No_NL",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_no_echo;
	status = generic_$add_value (usr_db,"usr_db","Device__No_echo",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_quit_enable;
	status = generic_$add_value (usr_db,"usr_db","Device__Quit_enable",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_raw;
	status = generic_$add_value (usr_db,"usr_db","Device__Raw",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_raw_nl;
	status = generic_$add_value (usr_db,"usr_db","Device__Raw_nl",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_rts_enable;
	status = generic_$add_value (usr_db,"usr_db","Device__Rts_enable",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);


	r1 = (float)sio_susp_enable;
	status = generic_$add_value (usr_db,"usr_db","Device__Susp_enable",r1,(long)0,(long)1,(long)Off_);
	if (status.all) return (status);
	}


#if x_SPE_installed
    else if (i_o_mode == prsvr__Io_Parallel)
	{
	r1 = (float)pio_handshake;
	status = generic_$add_value (usr_db,"usr_db","Device__Handshake",r1,(long)0,(long)3,(long)Off_);
	if (status.all) return (status);


	r1 = (float)pio_status_chk;
	status = generic_$add_value (usr_db,"usr_db","Device__Status_check",r1,(long)0,(long)32,(long)Off_);
	if (status.all) return (status);
	}
#endif


/*xxxxx Add new printer here xxxxx*/
    /* use formfeed */
    if (printer_model == imagewriter) r1 = Off_; /* Cannot set page length in a sensible way */
    else if (printer_model != dumb_ascii) r1 = On_;
    else r1 = (float)dumb_use_FF;
    status = generic_$add_value (usr_db,"usr_db","Use__Formfeed",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);


/*xxxxx Add new printer here xxxxx*/
    /* send linefeed before formfeed */
    if (printer_model != dumb_ascii) r1 = Off_;
    else r1 = (float)dumb_use_LF_before_FF;
    status = generic_$add_value (usr_db,"usr_db","Use__LF_before_FF",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);


    /* End of line character(s) */
    r1 = (float)end_of_line;
    status = generic_$add_value (usr_db,"usr_db","End_of_Line",r1,(long)eol_cr,(long)eol_lf_cr,(long)Off_);
    if (status.all) return (status);


    /* Use CR without LF (to return to left margin) */
    r1 = (float)use_cr_without_lf;
    status = generic_$add_value (usr_db,"usr_db","Use__CR_without_LF",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);


/*xxxxx Add new printer here xxxxx*/
    /* use backspace */
    if (printer_model == microline84) r1 = Off_;
    else if (printer_model == imagewriter
     || printer_model == imagewriterII) r1 = one_at_a_time;
    else if (printer_model != dumb_ascii) r1 = On_;
    else r1 = (float)dumb_use_BS;
    status = generic_$add_value (usr_db,"usr_db","Use__Backspace",r1,(long)0,(long)one_at_a_time,(long)Off_);
    if (status.all) return (status);


/*xxxxx Add new printer here xxxxx*/
    /* use tab */
    if (printer_model == HPlaserjet
     || printer_model == HPlaserjetplus) r1 = Off_; /*zzzzz Really cannot use tabs? zzzzz*/
    else if (printer_model != dumb_ascii) r1 = On_;
    else r1 = (float)dumb_use_tab;
    status = generic_$add_value (usr_db,"usr_db","Use__Tab",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);


    r1 = (float)tab_length;
    status = generic_$add_value (usr_db,"usr_db","Tab__Length",r1,(long)1,(long)20,(long)Off_);
    if (status.all) return (status);


    /* word wrapping */
    r1 = (float)word_wrap;
    status = generic_$add_value (usr_db,"usr_db","Word__Wrap",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);


    r1 = (float)eightbit;
    status = generic_$add_value (usr_db,"usr_db","Eightbit_",r1,(long)0,(long)chr_del - 1,(long)Off_);
    if (status.all) return (status);


    r1 = (float)ctrl_char;
    status = generic_$add_value (usr_db,"usr_db","Ctrl__Char",r1,(long)ctrl_ignore,(long)ctrl_caret,(long)Off_);
    if (status.all) return (status);


    r1 = (float)big_job_chars;
    status = generic_$add_value (usr_db,"usr_db","Big__Job_chars",r1,(long)2,(long)huge_job_chars,(long)Off_);
    if (status.all) return (status);


    r1 = (float)big_job_lines;
    status = generic_$add_value (usr_db,"usr_db","Big__Job_lines",r1,(long)50,(long)huge_job_lines,(long)Off_);
    if (status.all) return (status);


    r1 = (float)big_job_pages;
    status = generic_$add_value (usr_db,"usr_db","Big__Job_pages",r1,(long)1,(long)huge_job_pages,(long)Off_);
    if (status.all) return (status);


    r1 = (float)reject_big_jobs;
    status = generic_$add_value (usr_db,"usr_db","Reject__Big_jobs",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);


#if SR_level == SR10_0 || SR_level == SR10_1
    /* allow TRANS_ files */
    r1 = (float)allow_TRANS;
    status = generic_$add_value (usr_db,"usr_db","Allow__TRANS",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);


    /* allow boldface */
    r1 = (float)allow_bold;
    status = generic_$add_value (usr_db,"usr_db","Allow__Bold",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);


    /* allow letter quality */
    r1 = (float)allow_lq;
    status = generic_$add_value (usr_db,"usr_db","Allow__Lq",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);
#endif


    /* Rectangle border around graphics plot */
    r1 = (float)graph_border;
    status = generic_$add_value (usr_db,"usr_db","Graph__Border",r1,(long)0,(long)1,(long)Off_);
    if (status.all) return (status);


    /* Automatic setting of bw_reverse for graphics plot */
    r1 = (float)reverse_bw;
    status = generic_$add_value (usr_db,"usr_db","Reverse__BW",r1,(long)0,(long)dark,(long)Off_);
    if (status.all) return (status);


    /*zzzzz It would be elegant to spell out here all the point sizes available for each printer,
    together with the escape sequences to set them. But this would be rather messy; besides,
    some of the point sizes may be available only in draft mode, some only in letter quality
    mode, and there is no way of telling the prsvr database about this. In any case, prf
    will give us the nearest available point size if we type something crazy; we can handle
    that ourselves.
    If we had the escape sequences set up here, then we should also set up the sequence to set
    tab stops. But that is just about impossible! zzzzz*/

/*xxxxx Add new printer here xxxxx*/
    /* point sizes: chars_per_inch = 120/point_size */
    if (printer_model != dumb_ascii)
	{
	r1 = 120.0 / 30;
	r2 = 120.0 / 4;
	status = generic_$add_range (ren_db,"ren_db",Point_,r1,r2,(long)4,(long)30,(long)Off_);
	if (status.all) return (status);
	}
    else if (! dumb_set_point)
	{
	r1 = dumb_point;
	status = generic_$add_value (ren_db,"ren_db",Point_,r1,(long)4,(long)30,(long)Off_);
	if (status.all) return (status);
	}


/*xxxxx Add new printer here xxxxx*/
    /* lines_per_inch */
    if (printer_model != dumb_ascii)
	{
	r1 = 1;
	r2 = 16;
	status = generic_$add_range (ren_db,"ren_db",Lpi_,r1,r2,(long)1,(long)20,(long)Off_);
	if (status.all) return (status);
	}
    else if (! dumb_set_lpi)
	{
	r1 = dumb_lpi;
	status = generic_$add_value (ren_db,"ren_db",Lpi_,r1,(long)1,(long)20,(long)Off_);
	if (status.all) return (status);
	}


/*xxxxx Add new printer here xxxxx*/
    /* graphics resolution in dots per inch (horizontal) */
    if (printer_model != dumb_ascii)
	{
	r1 =    1;
	r2 = 1000;
	status = generic_$add_range (ren_db,"ren_db",Resolution_,r1,r2,(long)1,(long)1000,(long)Off_);
	if (status.all) return (status);
	}
    else
	{
	r1 = 300;
	status = generic_$add_value (ren_db,"ren_db",Resolution_,r1,(long)1,(long)1000,(long)Off_);
	if (status.all) return (status);
	}


/*xxxxx Add new printer here xxxxx*/
    /* margins */
    if (printer_model == HPlaserjet
     || printer_model == HPlaserjetplus)
	{
	r1 = 0.375; /*zzzzz Crude attempt at taking care of unprintable region (how big is it anyway?) zzzzz*/
	r2 = 5;
	status = generic_$add_range (ren_db,"ren_db",Bottom__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	if (status.all) return (status);

	r1 = 0.375;
	r2 = 5;
	status = generic_$add_range (ren_db,"ren_db",Top__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	if (status.all) return (status);

	r1 = 0;
	r2 = 5;
	status = generic_$add_range (ren_db,"ren_db",Left__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	if (status.all) return (status);

	r1 = 0;
	r2 = 5;
	status = generic_$add_range (ren_db,"ren_db",Right__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	if (status.all) return (status);
	}
    else if (printer_model != dumb_ascii)
	{
	r1 = 0;
	r2 = 5;
	status = generic_$add_range (ren_db,"ren_db",Bottom__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	if (status.all) return (status);

	r1 = 0;
	r2 = 5;
	status = generic_$add_range (ren_db,"ren_db",Top__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	if (status.all) return (status);

	r1 = 0;
	r2 = 5;
	status = generic_$add_range (ren_db,"ren_db",Left__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	if (status.all) return (status);

	r1 = 0;
	r2 = 5;
	status = generic_$add_range (ren_db,"ren_db",Right__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	if (status.all) return (status);
	}
    else
	{
	if (! dumb_set_bottom_margin)
	    {
	    r1 = dumb_bottom_margin_low;
	    r2 = dumb_bottom_margin_high;
	    status = generic_$add_range (ren_db,"ren_db",Bottom__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	    if (status.all) return (status);
	    }

	if (! dumb_set_top_margin)
	    {
	    r1 = dumb_top_margin_low;
	    r2 = dumb_top_margin_high;
	    status = generic_$add_range (ren_db,"ren_db",Top__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	    if (status.all) return (status);
	    }

	if (! dumb_set_left_margin)
	    {
	    r1 = dumb_left_margin_low;
	    r2 = dumb_left_margin_high;
	    status = generic_$add_range (ren_db,"ren_db",Left__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	    if (status.all) return (status);
	    }

	if (! dumb_set_right_margin)
	    {
	    r1 = dumb_right_margin_low;
	    r2 = dumb_right_margin_high;
	    status = generic_$add_range (ren_db,"ren_db",Right__Margin,r1,r2,(long)0,(long)20,(long)Off_);
	    if (status.all) return (status);
	    }
	}


/*xxxxx Add new printer here xxxxx*/
    /* paper sizes */
    if (printer_model == epsonlq800
     || printer_model == epsonlx80
     || printer_model == epsonfx80
     || printer_model == epsonfx85
     || printer_model == imagewriter
     || printer_model == imagewriterII
     || printer_model == microline380
     || printer_model == microline390
     || printer_model == starnx1000)
	{
	r1 = 4;
	r2 = 8;
	status = generic_$add_range (ren_db,"ren_db",Paper__Width,r1,r2,(long)1,(long)20,(long)Off_);
	if (status.all) return (status);

	r1 = 2;
	r2 = 15;
	status = generic_$add_range (ren_db,"ren_db",Paper__Length,r1,r2,(long)1,(long)20,(long)Off_);
	if (status.all) return (status);
	}
    else if (printer_model == epsonlq1000
     || printer_model == epsonlq1500
     || printer_model == epsonfx105
     || printer_model == microline84
     || printer_model == microline391
     || printer_model == pinwriterp5)
	{
	r1 = 5;
	r2 = 13.6;
	status = generic_$add_range (ren_db,"ren_db",Paper__Width,r1,r2,(long)1,(long)20,(long)Off_);
	if (status.all) return (status);

	r1 = 2;
	r2 = 15;
	status = generic_$add_range (ren_db,"ren_db",Paper__Length,r1,r2,(long)1,(long)20,(long)Off_);
	if (status.all) return (status);
	}
    else if (printer_model == HPlaserjet
     || printer_model == HPlaserjetplus)
	{
	r1 = 6.6;
	r2 = 8;
	status = generic_$add_range (ren_db,"ren_db",Paper__Width,r1,r2,(long)1,(long)20,(long)Off_);
	if (status.all) return (status);

	r1 = 9.7;
	r2 = 13.6;
	status = generic_$add_range (ren_db,"ren_db",Paper__Length,r1,r2,(long)1,(long)20,(long)Off_);
	if (status.all) return (status);
	}
    else
	{
	if (! dumb_set_paper_width)
	    {
	    r1 = dumb_paper_width;
	    status = generic_$add_value (ren_db,"ren_db",Paper__Width,r1,(long)1,(long)20,(long)Off_);
	    if (status.all) return (status);
	    }

	if (! dumb_set_paper_length)
	    {
	    r1 = dumb_paper_length;
	    status = generic_$add_value (ren_db,"ren_db",Paper__Length,r1,(long)1,(long)20,(long)Off_);
	    if (status.all) return (status);
	    }
	}


/*xxxxx Add new printer here xxxxx*/
    /* paper types */
    if (printer_model == epsonlq800
     || printer_model == epsonlq1000
     || printer_model == epsonlq1500
     || printer_model == imagewriter
     || printer_model == imagewriterII
     || printer_model == microline84
     || printer_model == microline380
     || printer_model == microline390
     || printer_model == microline391
     || printer_model == pinwriterp5
     || printer_model == starnx1000)
	{
	r1 = Cut_;
	status = generic_$add_value (ren_db,"ren_db",Paper__Type,r1,(long)Cut_,(long)Transpar_,(long)Off_);
	if (status.all) return (status);

	r1 = FanFold_;
	status = generic_$add_value (ren_db,"ren_db",Paper__Type,r1,(long)Cut_,(long)Transpar_,(long)Off_);
	if (status.all) return (status);
	}
    else if (printer_model == epsonlx80
     || printer_model == epsonfx80
     || printer_model == epsonfx85
     || printer_model == epsonfx105)
	{
	r1 = FanFold_;
	status = generic_$add_value (ren_db,"ren_db",Paper__Type,r1,(long)Cut_,(long)Transpar_,(long)Off_);
	if (status.all) return (status);
	}
    else if (printer_model == HPlaserjet
     || printer_model == HPlaserjetplus)
	{
	r1 = Cut_;
	status = generic_$add_value (ren_db,"ren_db",Paper__Type,r1,(long)Cut_,(long)Transpar_,(long)Off_);
	if (status.all) return (status);
	}
    else if (! dumb_set_paper_type)
	{
	r1 = dumb_paper_type;
	status = generic_$add_value (ren_db,"ren_db",Paper__Type,r1,(long)Cut_,(long)Transpar_,(long)Off_);
	if (status.all) return (status);
	}


    /* file types. These need to be known by the driver, so are not set in the config file */
    status = generic_$add_string (ren_db,"ren_db",F__Type,UASC_,(long)Off_);
    if (status.all) return (status);
    status = generic_$add_string (ren_db,"ren_db",F__Type,OS_,(long)Off_);
    if (status.all) return (status);
    status = generic_$add_string (ren_db,"ren_db",F__Type,FTN_,(long)Off_);
    if (status.all) return (status);
    status = generic_$add_string (ren_db,"ren_db",F__Type,TRANS_,(long)Off_);
    if (status.all) return (status);
    status = generic_$add_string (ren_db,"ren_db",F__Type,BM1_,(long)Off_);
    if (status.all) return (status);


    /* filter chains. Could these be set in the config file? */
    /* filter chains, to convert any graphics to 1-pixel bitmap */
    status = generic_$add_string (ren_db,"ren_db",Filter__Chain,"bmsc$bm1",(long)Off_);
    if (status.all) return (status);
#if SR_level == SR10_0 || SR_level == SR10_1
    status = generic_$add_string (ren_db,"ren_db",Filter__Chain,"bmp$bmsc|bmsc$bm1",(long)Off_);
    if (status.all) return (status);
    status = generic_$add_string (ren_db,"ren_db",Filter__Chain,"gmf$bmsc|bmsc$bm1",(long)Off_);
    if (status.all) return (status);
#endif
#if SR_level == SR10_2 || SR_level == SR10_3
    status = generic_$add_string (ren_db,"ren_db",Filter__Chain,"bmp$bmsc",(long)Off_);
    if (status.all) return (status);
    status = generic_$add_string (ren_db,"ren_db",Filter__Chain,"gmf$bmsc",(long)Off_);
    if (status.all) return (status);
#endif


    return (status);
}

/* =========================================================================== */

status_$t generic_$check_config_sio (
  long word_count,
  char *words[3],
  char *type,
  long low_val,
  long high_val,
  long i_o_mode,
  float *r0,
  long *j0)

{
    status_$t status;
    float r1;
    float r2;
    long j1;
    long j2;

    status = generic_$check_config (word_count,words,type,low_val,high_val,&r1,&r2,&j1,&j2);
    if (status.all) return (status);

    if (i_o_mode == prsvr__Io_Serial) { *r0 = r1; *j0 = j1; }
    else
	{
	sprintf (genbuff,"Option %s in config file is only for SIO line",words[0]);
	generic_$print_error (status,"read_config",genbuff,"This option ignored",(long)Off_);
	}

    return (status);
}

/* =========================================================================== */

status_$t generic_$check_config_dumb (
  long word_count,
  char *words[3],
  char *type,
  long low_val,
  long high_val,
  long printer_model,
  float *r0,
  long *j0)

{
    status_$t status;
    float r1;
    float r2;
    long j1;
    long j2;

    status = generic_$check_config (word_count,words,type,low_val,high_val,&r1,&r2,&j1,&j2);
    if (status.all) return (status);

    if (printer_model == dumb_ascii) { *r0 = r1; *j0 = j1; }
    else generic_$read_config_not_dumb (words[0]);

    return (status);
}

/* =========================================================================== */

status_$t generic_$check_config_dumb_add (
  long word_count,
  char *words[3],
  char *type,
  long low_val,
  long high_val,
  long printer_model,
  db_obj ren_db,
  char *element,
  boolean *done)

{
    status_$t status;
    float r1;
    float r2;
    long j1;
    long j2;

    status = generic_$check_config (word_count,words,type,low_val,high_val,&r1,&r2,&j1,&j2);
    if (status.all) return (status);

    if (printer_model == dumb_ascii)
	{
	if (generic_$strcmp(type,"float",true))
	    status = generic_$add_value (ren_db,"ren_db",element,r1,low_val,high_val,(long)Off_);
	else if (generic_$strcmp(type,"float_range",true))
	    status = generic_$add_range (ren_db,"ren_db",element,r1,r2,low_val,high_val,(long)Off_);
	else if (generic_$strcmp(type,"integer",true))
	    status = generic_$add_value (ren_db,"ren_db",element,(float)j1,low_val,high_val,(long)Off_);
	else if (generic_$strcmp(type,"integer_range",true))
	    status = generic_$add_range (ren_db,"ren_db",element,(float)j1,(float)j2,low_val,high_val,(long)Off_);
	else if (!generic_$strcmp(type,"any_string",true)) /* only #s1$s2#, not any_string */
	    status = generic_$add_value (ren_db,"ren_db",element,(float)j1,low_val,high_val,(long)Off_);
	if (status.all) return (status);
	*done = true;
	}
    else generic_$read_config_not_dumb (words[0]);

    return (status);
}

/* =========================================================================== */

void generic_$read_config_not_parallel (
  char *option)

{
    status_$t status;

    status.all = status_$ok;

    sprintf (genbuff,"Option %s in config file is only for SPE parallel line",option);
    generic_$print_error (status,"read_config",genbuff,"This option ignored",(long)Off_);
}

/* =========================================================================== */

void generic_$read_config_obsolete (
  char *option)

{
    status_$t status;

    status.all = status_$ok;

    sprintf (genbuff,"Option %s in config file is obsolete",option);
    generic_$print_error (status,"read_config",genbuff,"This option ignored (controlled by -passwd)",(long)Off_);
}

/* =========================================================================== */

void generic_$read_config_not_dumb (
  char *option)

{
    status_$t status;

    status.all = status_$ok;

    sprintf (genbuff,"Option %s in config file is only for dumb_ascii printer",option);
    generic_$print_error (status,"read_config",genbuff,"This option ignored (value known for this printer)",(long)Off_);
}

/* =========================================================================== */

status_$t generic_$check_config (
  long word_count,
  char *words[3],
  char *type,
  long low_val,
  long high_val,
  float *r1,
  float *r2,
  long *j1,
  long *j2)

/*
Check the words in a line of config file.
word_count is the number of values, without including the option word.
type may be one of the following:
  "integer" or "float": in either case one value is expected,
    which is read as a float number, returned in r1 and after conversion in j1.
  "integer_range" or "float_range": two values are read as float numbers,
    returned in r1, r2 and j1, j2.
  The value(s) for integer or float, single or range, are to be in the range low_val to high_val.
  "#s0#s1#...#sn#": one value is expected, to match one of the strings,
    the number of which is returned on j1 (these values are 0,1,... so that
    if type is "#off#on#" then j1 would be 0 or 1).
  "any_string": any one value is accepted.
 */

{
    status_$t status;
    long count;
    char *strp,*wordp;
    boolean next_word;

    status.all = status_$ok;

    if ((generic_$strcmp(type,"integer",true)) || (generic_$strcmp(type,"float",true)))
	{
	if (word_count != 1)
	    {
	    status.all = print_$bad_prf_option_value;
	    sprintf (genbuff,"Wrong number (%d) of values for option %s in config file",word_count,words[0]);
	    generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
	    return (status);
	    }
	count = sscanf(words[1],"%f%s",r1,genbuff);
	if (count != 1)
	    {
	    status.all = print_$bad_prf_option_value;
	    sprintf (genbuff,"Wrong value <%s> for option %s in config file",words[1],words[0]);
	    generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
	    return (status);
	    }
	*j1 = generic_$float_to_long (*r1);
	if (generic_$strcmp(type,"integer",true))
	    {
	    if (*j1 < low_val || *j1 > high_val)
		{
		status.all = print_$bad_prf_option_value;
		sprintf (genbuff,"Wrong value %d for option %s in config file",*j1,words[0]);
		generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
		return (status);
		}
	    }
	else
	    {
	    if (*r1 < (float)low_val || *r1 > (float)high_val)
		{
		status.all = print_$bad_prf_option_value;
		sprintf (genbuff,"Wrong value %f for option %s in config file",*r1,words[0]);
		generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
		return (status);
		}
	    }
	}
    else if ((generic_$strcmp(type,"integer_range",true)) || (generic_$strcmp(type,"float_range",true)))
	{
	if (word_count != 2)
	    {
	    status.all = print_$bad_prf_option_value;
	    sprintf (genbuff,"Wrong number (%d) of values for option %s in config file",word_count,words[0]);
	    generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
	    return (status);
	    }
	count = sscanf(words[1],"%f%s",r1,genbuff);
	if (count != 1)
	    {
	    status.all = print_$bad_prf_option_value;
	    sprintf (genbuff,"Wrong value <%s> for option %s in config file",words[1],words[0]);
	    generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
	    return (status);
	    }
	count = sscanf(words[2],"%f%s",r2,genbuff);
	if (count != 1)
	    {
	    status.all = print_$bad_prf_option_value;
	    sprintf (genbuff,"Wrong value <%s> for option %s in config file",words[2],words[0]);
	    generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
	    return (status);
	    }
	*j1 = generic_$float_to_long (*r1);
	*j2 = generic_$float_to_long (*r2);
	if (generic_$strcmp(type,"integer_range",true))
	    {
	    if (*j1 < low_val || *j1 >= *j2 || *j2 > high_val)
		{
		status.all = print_$bad_prf_option_value;
		sprintf (genbuff,"Wrong values %d, %d for option %s in config file",*j1,*j2,words[0]);
		generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
		return (status);
		}
	    }
	else
	    {
	    if (*r1 < (float)low_val || *r1 >= *r2 || *r2 > (float)high_val)
		{
		status.all = print_$bad_prf_option_value;
		sprintf (genbuff,"Wrong values %f, %f for option %s in config file",*r1,*r2,words[0]);
		generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
		return (status);
		}
	    }
	}
    else if (generic_$strcmp(type,"any_string",true))
	{
	if (word_count != 1)
	    {
	    status.all = print_$bad_prf_option_value;
	    sprintf (genbuff,"Wrong number (%d) of values for option %s in config file",word_count,words[0]);
	    generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
	    return (status);
	    }
	}
    else if (*type == *"#")
	{
	if (word_count != 1)
	    {
	    status.all = print_$bad_prf_option_value;
	    sprintf (genbuff,"Wrong number (%d) of values for option %s in config file",word_count,words[0]);
	    generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
	    return (status);
	    }
	sprintf (genbuff,"%s",type);
	status.all = print_$bad_prf_option_value;
	for (strp = genbuff, count = 0, next_word = true; *strp != chr_nul; strp++)
	    {
	    if (*strp <= chr_sp || *strp >= chr_del || *strp == *"#")
		{
		*strp = chr_nul;
		if (! next_word)
		    {
		    if (generic_$strcmp(wordp,words[1],true)) { status.all = status_$ok; break; }
		    next_word = true;
		    count++;
		    }
		}
	    else if (next_word)
		{
		wordp = strp;
		next_word = false;
		}
	    }
	if (status.all)
	    {
	    sprintf (genbuff,"Wrong value <%s> for option %s in config file",words[1],words[0]);
	    generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
	    return (status);
	    }
	*j1 = count;
	}
    else
	{
	status.all = print_$bad_prf_option_value;
	sprintf (genbuff,"Wrong type <%s> for option %s in config file",type,words[0]);
	generic_$print_error (status,"check_config",genbuff,"",(long)Off_);
	return (status);
	}

    return (status);
}

/* =========================================================================== */

status_$t generic_$add_value (
  db_obj database,
  char *db_name,
  char *element,
  float value,
  long minval,
  long maxval,
  long printout)

{
    status_$t status;

    if (minval <= maxval)
	{
	if (value < minval || value > maxval)
	    {
	    generic_$print_crazy ("add_value",element,value,"",printout);
	    status.all = print_$rdr_bad_db_data;
	    return (status);
	    }
	}

    status = add_db_value_c (database,element,&value);
    if (status.all)
	{
	sprintf (genbuff,"Unable to add element %.40s (value %f) to database %.40s",element,value,db_name);
	generic_$print_error (status,"add_value",genbuff,"",printout);
	}
#if x_debug
    else
	fprintf (stderr,"%s set to %f (in database %s)\n",element,value,db_name);
#endif
    return (status);
}

/* =========================================================================== */

status_$t generic_$add_range (
  db_obj database,
  char *db_name,
  char *element,
  float value_low,
  float value_high,
  long minval,
  long maxval,
  long printout)

{
    status_$t status;

    if (value_high <= value_low)
	{
	status = generic_$add_value (database,db_name,element,value_low,minval,maxval,printout);
	return (status);
	}

    if (value_low < minval || value_high > maxval)
	{
	generic_$print_crazy ("add_range",element,value_low,"",printout);
	generic_$print_crazy ("add_range",element,value_high,"",printout);
	status.all = print_$rdr_bad_db_data;
	return (status);
	}

    status = add_db_range_c (database,element,&value_low,&value_high);
    if (status.all)
	{
	sprintf (genbuff,"Unable to add element %.40s (range %f to %f) to database %.40s",element,value_low,value_high,db_name);
	generic_$print_error (status,"add_range",genbuff,"",printout);
	}
#if x_debug
    else
	fprintf (stderr,"%s set to range %f to %f (in database %s)\n",element,value_low,value_high,db_name);
#endif
    return (status);
}

/* =========================================================================== */

status_$t generic_$add_string (
  db_obj database,
  char *db_name,
  char *element,
  char *value,
  long printout)

{
    status_$t status;

    status = add_db_string_c (database,element,value);
    if (status.all)
	{
	sprintf (genbuff,"Unable to add element %.40s (value <%s>) to database %.40s",element,value,db_name);
	generic_$print_error (status,"add_string",genbuff,"",printout);
	}
#if x_debug
    else
	fprintf (stderr,"%s set to <%s> (in database %s)\n",element,value,db_name);
#endif
    return (status);
}

/* =========================================================================== */

status_$t generic_$get_value (
  db_obj database,
  char *db_name,
  char *element,
  float *valuep,
  long *convertp,
  boolean optional,
  long special,
  long minval,
  long maxval,
  long defval,
  long printout)

{
    status_$t status;

#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
/*yyyyy Use inq_entry instead of get_db_value, so that prsvr will not crash if element not present in database yyyyy*/
    db_data_t db_data;
    db_field_t db_field;
    float r1;

    db_data.num_entries = 1;
    db_field = Value_Field;
    db_data.type = &db_field;
    r1 = 0;
    db_data.v.rval = &r1;

    status = inq_entry_c (database,element,&db_data);
    if (! status.all)
	{
	if (db_data.num_entries == 1 && *db_data.type == Value_Field)
	    {
	    *valuep = *db_data.v.rval;
	    }
	else
	    {
	    status.all = print_$wrong_db_entry_type;
	    }
	}
#else
    *valuep = *get_db_value_c (database,element,&status);
#endif

    if (status.all)
	{
	if (optional && status.all == print_$no_db_entry)
	    {
	    status.all = status_$ok;
	    }
	else
	    {
	    sprintf (genbuff,"%d",defval);
	    generic_$print_find_error (status,"get_value",element,db_name,genbuff,printout);
	    }
	*valuep = (float)defval;
	}

#if x_debug
#if x_debug_db
    fprintf (stderr,"%s is %f (from database %s)\n",element,*valuep,db_name);
#endif
#endif

    /*zzzzz For some reason, simply testing (*valuep == (float)special) does not work zzzzz*/
	 if (*valuep < (float)special + 0.00001 && *valuep > (float)special - 0.00001) *valuep = (float)defval;
    else if (*valuep < (float)defval + 0.00001  && *valuep > (float)defval - 0.00001)  *valuep = (float)defval;
    else if (*valuep < (float)minval || *valuep > (float)maxval)
	{
	sprintf (genbuff,"%d",defval);
	generic_$print_crazy ("get_value",element,*valuep,genbuff,printout);
	*valuep = (float)defval;
	}
    *convertp = generic_$float_to_long (*valuep);

    return (status);
}

/* =========================================================================== */

#if SR_level == SR10_3
status_$t generic_$get_integer_value (
  db_obj database,
  char *db_name,
  char *element,
  long *valuep,
  boolean optional,
  long special,
  long minval,
  long maxval,
  long defval,
  long printout)

{
    status_$t status;

#if SR_level == SR10_3
/*yyyyy Use inq_entry instead of get_db_int, so that prsvr will not crash if element not present in database yyyyy*/
    db_data_t db_data;
    db_field_t db_field;
    long i1;

    db_data.num_entries = 1;
    db_field = Int_Field;
    db_data.type = &db_field;
    i1 = 0;
    db_data.v.ival = &i1;

    status = inq_entry_c (database,element,&db_data);
    if (! status.all)
	{
	if (db_data.num_entries == 1 && *db_data.type == Int_Field)
	    {
	    *valuep = (long)*db_data.v.ival;
	    }
	else
	    {
	    status.all = print_$wrong_db_entry_type;
	    }
	}
#else
    *valuep = (long)*get_db_int_c (database,element,&status);
#endif

    if (status.all)
	{
	if (optional && status.all == print_$no_db_entry)
	    {
	    status.all = status_$ok;
	    }
	else
	    {
	    sprintf (genbuff,"%d",defval);
	    generic_$print_find_error (status,"get_value",element,db_name,genbuff,printout);
	    }
	*valuep = defval;
	}

#if x_debug
#if x_debug_db
    fprintf (stderr,"%s is %d (from database %s)\n",element,*valuep,db_name);
#endif
#endif

	 if (*valuep == special) *valuep = defval;
    else if (*valuep == defval) *valuep = defval;
    else if (*valuep < minval || *valuep > maxval)
	{
	sprintf (genbuff,"%d",defval);
	generic_$print_crazy ("get_value",element,(float)*valuep,genbuff,printout);
	*valuep = defval;
	}

    return (status);
}
#endif

/* =========================================================================== */

status_$t generic_$check_string (
  db_obj database,
  char *db_name,
  char *element,
  char **stringp,
  boolean optional,
  char *type,
  long *j0,
  long printout)

/*  Get value for element in database.
    type is allowed values (separated by #).
    Set j0 to -100 in case of error from database,
	   to -99 in case of not allowed value,
	   to -2 if not_present,
	   to -1  if empty string,
	   or to the sequence number of allowed values (numbered 0,1,2,...)
*/

{
    status_$t status;
    char *strp,*wordp;
    long count;
    boolean next_word;

    status = generic_$get_string (database,db_name,element,stringp,optional,printout);
    if (status.all)
	{
	*j0 = -100;
	return (status);
	}

    if (generic_$strcmp("not_present",*stringp,false))
	{
	*j0 = -2;
	return (status);
	}
    else if (generic_$strcmp("",*stringp,false))
	{
	*j0 = -1;
	return (status);
	}

    sprintf (genbuff,"%s",type);
    status.all = print_$bad_prf_option_value;
    for (strp = genbuff, count = 0, next_word = true; *strp != chr_nul; strp++)
	{
	if (*strp <= chr_sp || *strp >= chr_del || *strp == *"#")
	    {
	    *strp = chr_nul;
	    if (! next_word)
		{
		if (generic_$strcmp(wordp,*stringp,true)) { status.all = status_$ok; break; }
		next_word = true;
		count++;
		}
	    }
	else if (next_word)
	    {
	    wordp = strp;
	    next_word = false;
	    }
	}
    if (status.all)
	{
	generic_$print_wrong_value (element,*stringp);
	*j0 = -99;
	}
    else *j0 = count;

    return (status);
}

/* =========================================================================== */

status_$t generic_$get_string (
  db_obj database,
  char *db_name,
  char *element,
  char **stringp,
  boolean optional,
  long printout)

{
    status_$t status;

#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
/*yyyyy Use inq_entry instead of get_db_string, so that prsvr will not crash if element not present in database yyyyy*/
    db_data_t db_data;
    db_field_t db_field;
    char s1;

    db_data.num_entries = 1;
    db_field = String_Field;
    db_data.type = &db_field;
    s1 = *" ";
    db_data.v.str = &s1;

    status = inq_entry_c (database,element,&db_data);
    if (! status.all)
	{
	if (db_data.num_entries == 1 && *db_data.type == String_Field)
	    {
	    *stringp = db_data.v.str;
	    }
	else
	    {
	    status.all = print_$wrong_db_entry_type;
	    }
	}
#else
    *stringp = get_db_string_c (database,element,&status);
#endif

    if (status.all)
	{
	if (optional && status.all == print_$no_db_entry)
	    {
	    status.all = status_$ok;
	    *stringp = "not_present";
	    }
	else
	    {
	    generic_$print_find_error (status,"get_string",element,db_name,"<error>",printout);
	    *stringp = "error";
	    }
	}

#if x_debug
#if x_debug_db
    fprintf (stderr,"%s is <%s> (from database %s)\n",element,*stringp,db_name);
#endif
#endif

    return (status);
}

/* =========================================================================== */

long generic_$float_to_long (
  float value)

{
    long i;

    if (value < -32000.0)
	i = -32000;
    else if (value > 32000.0)
	i = 32000;
    else
	i = floor(value + 0.5);

    return (i);
}

/* =========================================================================== */

void generic_$cleanup_string (
  char *char_arr)

{
    long i;
    char *strp;

    strp = char_arr;

    for (i = 0; i < bufsize; i++, strp++)
	{
	if (*strp == chr_nul) break;
	if (*strp < chr_sp || *strp >= chr_del) *strp = chr_sp; /* Make sure it is printable */
	}
}

/* =========================================================================== */

status_$t generic_$send_lf (
  long feeds)

{
    long i,locfeeds;
    status_$t status;

    status.all = status_$ok;

    locfeeds = feeds;
    if (need_lf) locfeeds++;

    if (locfeeds > 0 && locfeeds <= bufsize)
	{
#if x_debug
#if x_debug_feed
	fprintf (stderr,"Sending out %d linefeeds\n",locfeeds);
#endif
#endif
	if (lf_bufflen == 2) i = 2 * locfeeds;
	else i = locfeeds;
	status = generic_$iosput (lf_buff,i);
	if (status.all)
	    {
	    sprintf (genbuff,"Unable to send %d linefeeds",locfeeds);
	    generic_$print_error (status,"send_lf",genbuff,"",(long)Off_);
	    }
	line_number += locfeeds;
	need_lf = false;
	}
#if x_debug
#if x_debug_feed
    else
	fprintf (stderr,"Called to send %d linefeeds, but not done\n",locfeeds);
#endif
#endif

    return (status);
}

/* =========================================================================== */

status_$t generic_$send_ff (
  boolean footer_valid,
  char *footerp)

{
    long i;
    char c;
    status_$t status;


    if (footer_valid)
	{
	i = lines_high - line_number;
	if (need_lf) i--;
	if (lines_high + 1 < lines_paper && i >= 0 && chars_line >= 10)
	    {
#if x_debug
#if x_debug_head
	    fprintf (stderr,"Making up and sending page footer\n");
#endif
#endif
	    status = generic_$send_lf (i + 1);
	    i = generic_$make_header (footerp,genbuff);
	    if (i > 0)
		{
		if (chars_marg > 0)
		    {
		    status = generic_$iosput (sp_buff,chars_marg);
		    if (status.all)
			generic_$print_error (status,"send_ff","Unable to send blanks for left margin","",(long)Off_);
		    }
		status = generic_$iosput (genbuff,i);
		if (status.all)
		    generic_$print_error (status,"send_ff","Unable to send footer string","",(long)Off_);
		need_lf = true;
		}
	    }
	}

    if (ff_bufflen == 1 || ff_bufflen == 2)
	{
	if (ff_bufflen == 2 && need_lf) status = generic_$send_lf ((long)0);

#if x_debug
#if x_debug_feed
	fprintf (stderr,"Sending out a formfeed\n");
#endif
#endif
	c = chr_ff;
	status = generic_$iosput (&c,(long)1);
	if (status.all) generic_$print_error (status,"send_ff","Unable to send formfeed","",(long)Off_);
	need_lf = false;
	}
    else
	{
	if (lines_paper <= 0) lines_paper = 66;
	i = lines_paper - line_number;
	if (need_lf) i--;
	if (i < 0) i += lines_paper;
	status = generic_$send_lf (i);
	}

    line_number = 0;
    page_number++;
    return (status);
}

/* =========================================================================== */

status_$t generic_$render_check_room (
  boolean header_valid,
  boolean footer_valid,
  char *headerp,
  char *footerp)

{
    status_$t status;
    long i;

    status.all = status_$ok;

    /* more room on this page */
    i = line_number;
    if (need_lf) i++;
    if (i >= lines_high) status = generic_$send_ff (footer_valid,footerp);
    if (status.all) return (status);

    if (need_lf)
	{
	status = generic_$send_lf ((long)0);
	if (status.all) return (status);
	}

    if (line_number == 0)
	{
	/* skip margin lines */
	if (lines_marg > 0 && lines_marg + 2 < lines_high) status = generic_$send_lf (lines_marg);
	/* Write header */
	if (header_valid && chars_line >= 10)
	    {
#if x_debug
#if x_debug_head
	    fprintf (stderr,"Making up and sending page header\n");
#endif
#endif
	    i = generic_$make_header (headerp,genbuff);
	    if (i > 0)
		{
		if (chars_marg > 0)
		    {
		    status = generic_$iosput (sp_buff,chars_marg);
		    if (status.all)
			generic_$print_error (status,"check_room","Unable to send blanks for left margin","",(long)Off_);
		    }
		status = generic_$iosput (genbuff,i);
		if (status.all)
		    generic_$print_error (status,"check_room","Unable to send header string","",(long)Off_);
		}
	    status = generic_$send_lf ((long)2);
	    }
	}

    return (status);
}

/* =========================================================================== */

void generic_$test_send_lf (
  long feeds)

/* Aid in counting up lines in file */

{
    long locfeeds;

    locfeeds = feeds;
    if (need_lf) locfeeds++;

    if (locfeeds > 0)
	{
        total_chars_job += locfeeds;
	line_number += locfeeds;
	need_lf = false;
	}
}

/* =========================================================================== */

void generic_$test_send_ff (
  boolean footer_valid)

/* Aid in counting up pages in file */

{
    long i;

    if (footer_valid)
	{
	i = lines_high - line_number;
	if (need_lf) i--;
	if (lines_high + 1 < lines_paper && i >= 0 && chars_line >= 10)
	    {
	    generic_$test_send_lf (i + 1);
	    total_chars_job += chars_line;
	    need_lf = true;
	    }
	}

    if (ff_bufflen == 1 || ff_bufflen == 2)
	{
	if (ff_bufflen == 2 && need_lf) generic_$test_send_lf ((long)0);
	total_chars_job++;
	need_lf = false;
	}
    else
	{
	if (lines_paper <= 0) lines_paper = 66;
	i = lines_paper - line_number;
	if (need_lf) i--;
	if (i < 0) i += lines_paper;
	generic_$test_send_lf (i);
	}

    line_number = 0;
    page_number++;
}

/* =========================================================================== */

void generic_$test_check_room (
  boolean header_valid,
  boolean footer_valid)

/* Aid in counting up lines/pages in file */

{
    long i;

    /* more room on this page */
    i = line_number;
    if (need_lf) i++;
    if (i >= lines_high) generic_$test_send_ff (footer_valid);

    if (need_lf) generic_$test_send_lf ((long)0);

    if (line_number == 0)
	{
	/* skip margin lines */
	if (lines_marg > 0 && lines_marg + 2 < lines_high) generic_$test_send_lf (lines_marg);
	/* Write header */
	if (header_valid && chars_line >= 10)
	    {
	    total_chars_job += chars_line;
	    generic_$test_send_lf ((long)2);
	    }
	}
}

/* =========================================================================== */

void generic_$print_error (
  status_$t status,	/* Error to complain about, may be error_$ok for just messages */
  char *routine,	/* Up to 64 characters long */
  char *message1,	/* Up to 250 characters long */
  char *message2,	/* Up to 250 characters long */
  long printout)	/* Also send to printer if On_ */

{
    char errbuff[bufsize]; /* Must be at least 256 long */
    error_$string_t sub_n;	/* subsystem name */
    error_$string_t mod_n;	/* module name */
    error_$string_t err_t;	/* error text */
    short sub_nl;		/* subsystem name length */
    short mod_nl;		/* module name length */
    short err_tl;		/* error text length */
    long bytes;
    status_$t status2;

    sprintf (errbuff,"?(ERROR in routine generic_$%.64s)\n",routine);
    fprintf (stderr,"%s",errbuff);
    if (printout == On_) {bytes = strlen(errbuff); status2 = generic_$iosput (errbuff,bytes); line_number++;}

    if (strlen(message1) > 0)
	sprintf (errbuff,"    %.250s\n",message1);
    else
	sprintf (errbuff,"    (Unknown message)\n");
    fprintf (stderr,"%s",errbuff);
    if (printout == On_) {bytes = strlen(errbuff); status2 = generic_$iosput (errbuff,bytes); line_number++;}

    if (status.all != status_$ok)
	{
	error_$get_text (status,
	    sub_n,&sub_nl,
	    mod_n,&mod_nl,
	    err_t,&err_tl);

	/* Check lengths, put in null terminators */
	if (sub_nl >= sizeof(sub_n))
	    {sub_n[sizeof(sub_n) - 1] = chr_nul;}
	else if (sub_nl <= 0)
	    {sprintf (sub_n,"(Unknown subsystem)");}
	else
	    {sub_n[sub_nl] = chr_nul;}

	if (mod_nl >= sizeof(mod_n))
	    {mod_n[sizeof(mod_n) - 1] = chr_nul;}
	else if (mod_nl <= 0)
	    {sprintf (mod_n,"(Unknown module)");}
	else
	    {mod_n[mod_nl] = chr_nul;}

	if (err_tl >= sizeof(err_t))
	    {err_t[sizeof(err_t) - 1] = chr_nul;}
	else if (err_tl <= 0)
	    {sprintf (err_t,"(Unknown error text)");}
	else
	    {err_t[err_tl] = chr_nul;}

	sprintf (errbuff,"    (stcode = %08X) - %.70s (%.70s/%.70s)\n",
		status.all,err_t,sub_n,mod_n);
	fprintf (stderr,"%s",errbuff);
	if (printout == On_) {bytes = strlen(errbuff); status2 = generic_$iosput (errbuff,bytes); line_number++;}
	}

    if (strlen(message2) > 0)
	{
	sprintf (errbuff,"    %.250s\n",message2);
	fprintf (stderr,"%s",errbuff);
	if (printout == On_) {bytes = strlen(errbuff); status2 = generic_$iosput (errbuff,bytes); line_number++;}
	}
}

/* =========================================================================== */

long generic_$make_header (
  char *headerp,
  char buffer[bufsize])

/*  Make up header or footer string in buffer.
    These header strings have three parts: left, center and right,
    separated by null bytes.
    Special characters to look out for:
#if SR_level == SR10_0
    the string '/f', to be replaced by page number (surrounded by single blanks).
#endif
#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
    chr_ff, to be replaced by page number (surrounded by single blanks).
#endif
*/

{
    long i,j,lengh,start,finish;
    char *chrp;
    char headbuff[bufsize];


#if x_debug
#if x_debug_head
    fprintf (stderr,"Bytes in header or footer string:\n");
    lengh = 0;
    chrp = headerp;
    for (finish = 0; finish < 3 && lengh < bufsize; chrp++)
	{
	i = *chrp & 0xff;
	if (i >= chr_sp && i < chr_del)
	    {
	    headbuff[lengh] = *chrp;
	    lengh++;
	    }
	else
	  {
	  if (i == 0) finish++;
	  if (lengh > 0)
	      {
	      headbuff[lengh] = 0;
	      fprintf (stderr,"%s\n",headbuff);
	      lengh = 0;
	      }
	  fprintf (stderr,"Byte %d (hex %x)\n",i,i);
	  }
      }
#endif
#endif

    if (chars_line > bufsize || chars_line < 10)
	{
	buffer[0] = chr_nul;
	return (0);
	}

    chrp = headerp;
    finish = 0;

    for (i = 0; i < 3; i++)
	{
	for (lengh = 0; (lengh < chars_line) && (*chrp != chr_nul); chrp++, lengh++)
	    {
#if SR_level == SR10_0
	    if (*chrp == *"/" && *(chrp + 1) == *"f" && lengh + 14 < bufsize)
		/* Got a '/f' string, replace by page number */
		{
		sprintf (&headbuff[lengh]," %d ",page_number + 1);
		lengh = strlen(headbuff) - 1;
		chrp++;
		}
#endif
#if SR_level == SR10_1 || SR_level == SR10_2 || SR_level == SR10_3
	    if (*chrp == chr_ff && lengh + 14 < bufsize)
		/* Got a formfeed, replace by page number */
		{
		sprintf (&headbuff[lengh]," %d ",page_number + 1);
		lengh = strlen(headbuff) - 1;
		}
#endif
	    else if (*chrp < chr_sp || *chrp >= chr_del) headbuff[lengh] = chr_sp; /* Make sure it is printable */
	    else headbuff[lengh] = *chrp;
	    }

	chrp = chrp + strlen(chrp) + 1;

	if (lengh <= 0) continue;
	if (lengh > chars_line) lengh = chars_line;

	if (i == 0)
	    start = 0;
	else if (i == 1)
	    start = (chars_line - lengh) / 2;
	else
	    start = chars_line - lengh;

	for (j = finish; j < start; j++) buffer[j] = chr_sp;
	for (j = 0; j < lengh; j++) buffer[start + j] = headbuff[j];

	finish = start + lengh;
	}

    if (finish > chars_line) finish = chars_line;
    if (finish < bufsize) buffer[finish] = chr_nul;

#if x_debug
#if x_debug_head
    fprintf (stderr,"Header or footer string is <%s>\n",buffer);
#endif
#endif

    return (finish);
}

/* =========================================================================== */

void generic_$print_find_error (
  status_$t status,
  char *routine,
  char *element,
  char *database,
  char *def_val,
  long printout)

{
    char buffer1[120];
    char buffer2[120];

    sprintf (buffer1,"Unable to find element %.40s in database %.40s",element,database);

    if (strlen(def_val) > 0)
	sprintf (buffer2,"Using default value %.40s",def_val);
    else
	buffer2[0] = chr_nul;

    generic_$print_error (status,routine,buffer1,buffer2,printout);
}

/* =========================================================================== */

void generic_$print_sio_error (
  status_$t status,
  char *routine,
  char *property)

{
    sprintf (genbuff,"Unable to set SIO line %.40s",property);
    generic_$print_error (status,routine,genbuff,"",(long)Off_);
}

/* =========================================================================== */

void generic_$print_crazy (
  char *routine,
  char *name,
  float value,
  char *def_val,
  long printout)

{
    char buffer1[80];
    char buffer2[80];
    status_$t status;

    sprintf (buffer1,"Crazy value for %.20s = %f",name,value);
    sprintf (buffer2,"Using default value %.20s",def_val);
    status.all = status_$ok;
    generic_$print_error (status,routine,buffer1,buffer2,printout);
}

/* =========================================================================== */

void generic_$end_job (
  db_obj end_job_db,
  status_$t *status)

{
    long i,bytes;
    status_$t status2;
    float r0,form_feed;


    *status = generic_$get_value (end_job_db,"end_job_db",Form__Feeds,&form_feed,&i,true,(long)-1,(long)0,(long)5,(long)0,(long)On_);
    /*yyyyy If errors occur (like prf -ncopy -delete, or prf impress_file), there may be nothing in database. yyyyy*/

    if (form_feed > 0)
	{
	for (i = 0; i < form_feed; i++)
	    {
	    status2 = generic_$send_ff (false,"");
	    if (status2.all)
		{
		generic_$print_error (status2,"end_job","Unable to send end_of_job formfeed","",(long)Off_);
		break;
		}
	    }

	if (! (*status).all) *status = status2;
	}


    status2 = generic_$reset_printer ();
    if (! (*status).all) *status = status2;
}

/* =========================================================================== */

status_$t generic_$reset_printer ()

{
    long bytes;
    status_$t status,status2;
    float r0;


    status = generic_$get_value (usr_db,"usr_db","Printer__Model",&r0,&printer_model,false,(long)0,(long)0,(long)last_model,(long)0,(long)On_);

    bytes = 0;
/*xxxxx Add new printer here xxxxx*/
    /* Reset printer */
    switch (printer_model)
      {
      case epsonlq800:
      case epsonlq1000:
      case epsonlq1500:
      case epsonlx80:
      case epsonfx85:
      case epsonfx105:
      case microline380:
      case microline390:
      case microline391:
      case pinwriterp5:
      case starnx1000:
	genbuff[bytes++] = chr_esc;
	genbuff[bytes++] = *"@";
	/* turn letter quality off, in case it was set with DIP switches */
	genbuff[bytes++] = chr_esc;
	genbuff[bytes++] = *"x";
	genbuff[bytes++] = 0;
	break;
      case epsonfx80:
	genbuff[bytes++] = chr_esc;
	genbuff[bytes++] = *"@";
	break;
      case imagewriter:
      case imagewriterII:
	genbuff[bytes++] = chr_esc;
	genbuff[bytes++] = *"c";
	break;
      case HPlaserjet:
      case HPlaserjetplus:
	genbuff[bytes++] = chr_esc;
	genbuff[bytes++] = *"E";
	genbuff[bytes++] = chr_esc; /* So that both LF and FF first do a CR (cannot have just FF but not LF do it) */
	genbuff[bytes++] = *"&";
	genbuff[bytes++] = *"k";
	genbuff[bytes++] = *"2";
	genbuff[bytes++] = *"G";
	break;
      case microline84:
	genbuff[bytes++] = chr_can;    /* clear buffer */
	genbuff[bytes++] = chr_dc1;    /* release print suppress mode */
	genbuff[bytes++] = chr_esc;
	genbuff[bytes++] = *"0";       /* normal (draft) mode */
	genbuff[bytes++] = chr_esc;
	genbuff[bytes++] = *"I";       /* stop emphasized */
	break;
      }
   if (bytes > 0)
	{
	status2 = generic_$iosput (genbuff,bytes);
	if (status2.all)
	    {
	    generic_$print_error (status2,"reset_printer","Unable to send printer reset string","",(long)Off_);
	    if (! status.all) status = status2;
	    }
	}

    return (status);
}

/* =========================================================================== */

void generic_$close (
  close_flags_t close_flags,
  status_$t *status)

{
    long bytes;
    status_$t status2;


    generic_$date_time (genbuff);

    sprintf (bigbuff,"\
Summary\n\n\
ASCII jobs:\n\
  UASC jobs %10d\n\
  OS jobs   %10d\n\
  FTN jobs  %10d\n\
  rejected  %10d\n\
  chars   %12d\n\
  lines   %12d\n\
  pages   %12d\n\
TRANS jobs  %10d\n\
  rejected  %10d\n\
  chars   %12d\n\
BITMAP jobs %10d\n\
  rejected  %10d\n\
  chars   %12d\n\
  lines   %12d\n\
  pages   %12d\n\
Other jobs  %10d\n\n\n\
Print server started up since %s\n\
Print server shutting down on %s\n",
	total_jobs_uasc,total_jobs_os,total_jobs_ftn,total_rejected_ascii,
	total_chars_ascii,total_lines_ascii,total_pages_ascii,
	total_jobs_trans,total_rejected_trans,total_chars_trans,
	total_jobs_bm1,total_rejected_bm1,
	total_chars_bm1,total_lines_bm1,total_pages_bm1,
	total_jobs_other,
	date,genbuff);
    bytes = strlen(bigbuff);


    fprintf (stderr,"%s",bigbuff);

    *status = generic_$iosput (bigbuff,bytes);
    if ((*status).all)
	generic_$print_error (*status,"close","Unable to send shutdown message","",(long)Off_);

    line_number = 23; /* We had that many newlines */
    need_lf = false;
    page_number = 0;
    lines_paper = 0;

    status2 = generic_$send_ff (false,"");
    if (status2.all)
	generic_$print_error (status2,"close","Unable to send closing formfeed","",(long)Off_);
    if (! (*status).all) *status = status2;


    ios_$close (out_id,&status2);
    if (status2.all)
	generic_$print_error (status2,"close","Unable to close output device","",(long)Off_);
    if (! (*status).all) *status = status2;


    free_db_c (usr_db);
}

/* =========================================================================== */

status_$t generic_$init_line ()

/*  (Re-)initialize printer line.
    All the relevant settings are in usr_db. */

{
    status_$t status;
    long j,i_o_mode;
    long dev_num;
    float r0;
    short k;
    sio_$value_t sio_val;
#if x_SPE_installed
    pio_$handshake_t handshk;
    pio_$status_check_t stchk;
#endif

	/* I/O mode */
	status = generic_$get_value (usr_db,"usr_db","Io__Mode",&r0,&i_o_mode,false,(long)0,(long)0,(long)10,(long)prsvr__Io_Serial,(long)Off_);
	if (status.all) return (status);


	/* I/O device name */
	status = generic_$get_value (usr_db,"usr_db","Device__Number",&r0,&dev_num,false,(long)0,(long)-1,(long)3,(long)0,(long)Off_);
	if (status.all) return (status);


	if (dev_num >= 1 && dev_num <= 3 && i_o_mode == prsvr__Io_Serial)
	    {

	    status = generic_$get_value (usr_db,"usr_db","Device__Speed",&r0,&j,false,(long)0,(long)10,(long)30000,(long)1200,(long)Off_);
	    if (status.all) return (status);
	    switch (j)
		{
		case 50:
		    {
		    sio_val.i = sio_$50;
		    break;
		    }
		case 75:
		    {
		    sio_val.i = sio_$75;
		    break;
		    }
		case 110:
		    {
		    sio_val.i = sio_$110;
		    break;
		    }
		case 134:
		    {
		    sio_val.i = sio_$134;
		    break;
		    }
		case 150:
		    {
		    sio_val.i = sio_$150;
		    break;
		    }
		case 300:
		    {
		    sio_val.i = sio_$300;
		    break;
		    }
		case 600:
		    {
		    sio_val.i = sio_$600;
		    break;
		    }
		case 1200:
		    {
		    sio_val.i = sio_$1200;
		    break;
		    }
		case 2000:
		    {
		    sio_val.i = sio_$2000;
		    break;
		    }
		case 2400:
		    {
		    sio_val.i = sio_$2400;
		    break;
		    }
		case 3600:
		    {
		    sio_val.i = sio_$3600;
		    break;
		    }
		case 4800:
		    {
		    sio_val.i = sio_$4800;
		    break;
		    }
		case 7200:
		    {
		    sio_val.i = sio_$7200;
		    break;
		    }
		case 9600:
		    {
		    sio_val.i = sio_$9600;
		    break;
		    }
		case 19200:
		    {
		    sio_val.i = sio_$19200;
		    break;
		    }
		default:
		    {
		    sprintf (genbuff,"Wrong device speed specified = %d",j);
		    status.all = status_$ok;
		    generic_$print_error (status,"init_line",genbuff,"",(long)Off_);
		    status.all = print_$rdr_bad_db_data;
		    return (status);
		    }
		}
	    sio_$control (out_id,(short)sio_$speed,sio_val,&status);
	    if (status.all)
		{
		generic_$print_sio_error (status,"init_line","speed");
		return (status);
		}
#if x_debug
#if x_debug_line
	    fprintf (stderr,"Device speed set to %d\n",j);
#endif
#endif


	    status = generic_$get_value (usr_db,"usr_db","Device__Stop_bits",&r0,&j,false,(long)0,(long)1,(long)2,(long)1,(long)Off_);
	    if (status.all) return (status);
	    j = generic_$float_to_long (r0 * 2);
	    switch (j)
		{
		case 2:
		    {
		    sio_val.i = sio_$stop_1;
		    break;
		    }
		case 3:
		    {
		    sio_val.i = sio_$stop_1_point_5;
		    break;
		    }
		case 4:
		    {
		    sio_val.i = sio_$stop_2;
		    break;
		    }
		default:
		    {
		    sprintf (genbuff,"Wrong stop bits specified = %f",r0);
		    status.all = status_$ok;
		    generic_$print_error (status,"init_line",genbuff,"",(long)Off_);
		    status.all = print_$rdr_bad_db_data;
		    return (status);
		    }
		}
	    sio_$control (out_id,(short)sio_$stop_bits,sio_val,&status);
	    if (status.all)
		{
		generic_$print_sio_error (status,"init_line","stop_bits");
		return (status);
		}
#if x_debug
#if x_debug_line
	    fprintf (stderr,"Stop bits set to %f\n",r0);
#endif
#endif


	    status = generic_$get_value (usr_db,"usr_db","Device__Parity",&r0,&j,false,(long)0,(long)none,(long)even,(long)0,(long)Off_);
	    if (status.all) return (status);
	    switch (j)
		{
		case none:
		    {
		    sio_val.i = sio_$no_parity;
		    break;
		    }
		case odd:
		    {
		    sio_val.i = sio_$odd_parity;
		    break;
		    }
		case even:
		    {
		    sio_val.i = sio_$even_parity;
		    break;
		    }
		default:
		    {
		    sprintf (genbuff,"Wrong parity specified = %d",j);
		    status.all = status_$ok;
		    generic_$print_error (status,"init_line",genbuff,"",(long)Off_);
		    status.all = print_$rdr_bad_db_data;
		    return (status);
		    }
		}
	    sio_$control (out_id,(short)sio_$parity,sio_val,&status);
	    if (status.all)
		{
		generic_$print_sio_error (status,"init_line","parity");
		return (status);
		}
#if x_debug
#if x_debug_line
	    fprintf (stderr,"Parity set to %d\n",j);
#endif
#endif


	    status = generic_$get_value (usr_db,"usr_db","Device__Bits_per_char",&r0,&j,false,(long)0,(long)5,(long)8,(long)7,(long)Off_);
	    if (status.all) return (status);
	    switch (j)
		{
		case 5:
		    {
		    sio_val.i = sio_$5bpc;
		    break;
		    }
		case 6:
		    {
		    sio_val.i = sio_$6bpc;
		    break;
		    }
		case 7:
		    {
		    sio_val.i = sio_$7bpc;
		    break;
		    }
		case 8:
		    {
		    sio_val.i = sio_$8bpc;
		    break;
		    }
		default:
		    {
		    sprintf (genbuff,"Wrong bits_per_character specified = %d",j);
		    status.all = status_$ok;
		    generic_$print_error (status,"init_line",genbuff,"",(long)Off_);
		    status.all = print_$rdr_bad_db_data;
		    return (status);
		    }
		}
	    sio_$control (out_id,(short)sio_$bits_per_char,sio_val,&status);
	    if (status.all)
		{
		generic_$print_sio_error (status,"init_line","bits_per_char");
		return (status);
		}
#if x_debug
#if x_debug_line
	    fprintf (stderr,"Bits_per_char set to %d\n",j);
#endif
#endif


	    status = generic_$get_value (usr_db,"usr_db","Device__Nlc_delay",&r0,&j,false,(long)0,(long)0,(long)10000,(long)0,(long)Off_);
	    if (status.all) return (status);
	    sio_val.i = j;
	    sio_$control (out_id,(short)sio_$nlc_delay,sio_val,&status);
	    if (status.all)
		{
		generic_$print_sio_error (status,"init_line","nlc_delay");
		return (status);
		}
#if x_debug
#if x_debug_line
	    fprintf (stderr,"Nlc_delay set to %d\n",j);
#endif
#endif


	    status = generic_$get_value (usr_db,"usr_db","Device__Bp_enable",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$bp_enable,"bp_enable",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Cts_enable",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$cts_enable,"cts_enable",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Dcd_enable",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$dcd_enable,"dcd_enable",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Drain_out",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$drain_out,"drain_out",j);
/*zzzzz We often get error code 00350007: quit while waiting for input (OS/terminal handler) zzzzz*/
	    if (status.all)
		{
		if (status.all != 0x00350007)
		    {
		    return (status);
		    }
		else
		    {
		    status.all = status_$ok;
		    generic_$print_error (status,"init_line","NOTE: Previous error (00350007) ignored","",(long)Off_);
		    }
		}


	    status = generic_$get_value (usr_db,"usr_db","Device__Flush_in",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$flush_in,"flush_in",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Flush_out",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$flush_out,"flush_out",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Host_sync",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$host_sync,"host_sync",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Hup_close",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$hup_close,"hup_close",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Input_sync",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$input_sync,"input_sync",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__No_NL",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$no_NL,"no_NL",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__No_echo",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$no_echo,"no_echo",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Quit_enable",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$quit_enable,"quit_enable",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Raw",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$raw,"raw",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Raw_nl",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$raw_nl,"raw_nl",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Rts_enable",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$rts_enable,"rts_enable",j);
	    if (status.all) return (status);


	    status = generic_$get_value (usr_db,"usr_db","Device__Susp_enable",&r0,&j,false,(long)0,(long)0,(long)1,(long)0,(long)Off_);
	    if (status.all) return (status);
	    status = generic_$sio_control (sio_$susp_enable,"susp_enable",j);
	    if (status.all) return (status);
	    }


#if x_SPE_installed
	else if (dev_num == -1 && i_o_mode == prsvr__Io_Parallel)
	    {
	    status = generic_$get_value (usr_db,"usr_db","Device__Handshake",&r0,&j,false,(long)0,(long)0,(long)3,(long)0,(long)Off_);
	    if (status.all) return (status);
	    switch (j)
		{
		case 0:
		    {
		    handshk = spe_$no_handshake;
		    break;
		    }
		case 1:
		    {
		    handshk = spe_$hand_busy;
		    break;
		    }
		case 2:
		    {
		    handshk = spe_$hand_ack;
		    break;
		    }
		case 3:
		    {
		    handshk = spe_$hand_ack_busy;
		    break;
		    }
		default:
		    {
		    sprintf (genbuff,"Wrong handshake specified = %d",j);
		    status.all = status_$ok;
		    generic_$print_error (status,"init_line",genbuff,"",(long)Off_);
		    status.all = print_$rdr_bad_db_data;
		    return (status);
		    }
		}
	    status = generic_$get_value (usr_db,"usr_db","Device__Status_check",&r0,&j,false,(long)0,(long)0,(long)32,(long)0,(long)Off_);
	    if (status.all) return (status);
	    stchk = (short)j;

	    spe_$pio_set_mode (handshk,stchk,&status);
	    if (status.all)
		{
		generic_$print_error (status,"init_line","Unable to set PIO line mode","",(long)Off_);
		return (status);
		}
#if x_debug
#if x_debug_line
	    fprintf (stderr,"Handshake set to %d, status check set to %d\n",j,stchk);
#endif
#endif
	    }
#endif


	else
	    {
	    status.all = status_$ok;
#if x_SPE_installed
	    generic_$print_error (status,"init_line",
		"This printer driver generic can only be used on a SIO or SPE line","",(long)Off_);
#else
	    generic_$print_error (status,"init_line",
		"This printer driver generic can only be used on a SIO line","",(long)Off_);
#endif
	    fprintf (stderr,"   Not on device number = %d\n",dev_num);
	    fprintf (stderr,"   With io_mode = %d\n",i_o_mode);
	    status.all = print_$file_device_mismatch;
	    return (status);
	    }

    return (status);
}

/* =========================================================================== */

status_$t generic_$sio_control (
  sio_$opt_t option,
  char *name,
  long j)

{
    boolean bool;
    sio_$value_t sio_val;
    status_$t status;

    ((j == On_) ? (bool = true) : (bool = false));
    sio_val.b = bool;
    sio_$control (out_id,option,sio_val,&status);
    if (status.all)
	generic_$print_sio_error (status,"sio_control",name);
#if x_debug
#if x_debug_line
    else
	{
	if (bool)
	    fprintf (stderr,"%s set to true\n",name);
	else
	    fprintf (stderr,"%s set to false\n",name);
	}
#endif
#endif
    return (status);
}

/* =========================================================================== */

status_$t generic_$iosput (
  char *buffer,
  long bytes)

{
    status_$t status;

    ios_$put (out_id,ios_$no_put_get_opts,buffer,bytes,&status);
    if (status.all) generic_$print_error (status,"iosput","Unable to write to output (printer) line","",(long)Off_);
    total_chars_job += bytes;
    return (status);
}

/* =========================================================================== */

status_$t generic_$raster_chunk (
  long start1,	/* Start of non-zero bytes to send */
  long start0,	/* Start of zero bytes to end */
  long mult,	/* all ... */
  long mult1,	/* ... sorts of ... */
  long i_fac,	/* ... things */
  long imode)	/* Graphics mode; dotsize */
  /* Raster data to send is in bigbuff */

{
    long st1,st0,xwant,ywant,xshift,yshift,bytes;
    long i,lil_pos;
    status_$t status;

    typedef union
	{
	short val;
	char bit[2];
	} byt_num_t;
    byt_num_t byt_num;


    if (mult1 <= 1)
	{
	st1 = start1;
	st0 = start0;
	}
    else
	{
	st1 = (start1 / mult1) * mult1;
	st0 = (start0 / mult1) * mult1; if (st0 < start0) st0 += mult1;
	}

    bytes = st0 - st1;

    xwant = st1; if (mult > 1) xwant = st1 / mult;
    ywant = raster_y_want;

    lil_pos = 0;

/*xxxxx Add new printer here xxxxx*/
    /* Move to correct position for graphics */
    switch (printer_model)
      {
      case epsonlq800:
      case epsonlq1000:
      case epsonlq1500:
      case microline380:
      case microline390:
      case microline391:
      case pinwriterp5:
	yshift = ywant - raster_y_cur;
	if (yshift > 0)
	    {
	    status = generic_$send_lf (yshift);
	    if (status.all)
		{
		generic_$print_error (status,"raster_chunk","Unable to send linefeed","",(long)Off_);
		return(status);
		}
	    raster_x_cur = 0;
	    raster_y_cur = ywant;
	    }

	xshift = xwant - raster_x_cur;
	if (xshift > 0)
	    {
	    if (i_fac > 0)
		xshift = xshift * i_fac;
	    else if (i_fac < 0)
		xshift = xshift / ( -i_fac);

	    byt_num.val = xshift;

	    lilbuff[lil_pos++] = chr_esc;
	    lilbuff[lil_pos++] = *"\\";
	    lilbuff[lil_pos++] = byt_num.bit[1];
	    lilbuff[lil_pos++] = byt_num.bit[0];
	    }

	byt_num.val = bytes; if (mult > 1) byt_num.val = bytes / mult;

	lilbuff[lil_pos++] = chr_esc;
	lilbuff[lil_pos++] = *"*";		/* high resolution graphics */
	lilbuff[lil_pos++] = (char)imode;	/* correct mode */
	lilbuff[lil_pos++] = byt_num.bit[1];	/* number of bytes */
	lilbuff[lil_pos++] = byt_num.bit[0];

	raster_x_cur = xwant + byt_num.val;
	break;
      case epsonlx80:
      case epsonfx80:
      case epsonfx85:
      case epsonfx105:
      case starnx1000:
	yshift = ywant - raster_y_cur;

	if (yshift > 0)
	    {
	    status = generic_$send_lf (yshift);
	    if (status.all)
		{
		generic_$print_error (status,"raster_chunk","Unable to send linefeed","",(long)Off_);
		return(status);
		}
	    raster_x_cur = 0;
	    raster_y_cur = ywant;
	    }

	xshift = xwant - raster_x_cur;
	if (xshift > 0)
	    {
	    xshift /= ( -i_fac);
	    if (xshift > bufsize) xshift = bufsize;
	    status = generic_$iosput (sp_buff,xshift);
	    if (status.all)
		{
		generic_$print_error (status,"raster_chunk","Unable to send blanks for spacing","",(long)Off_);
		return(status);
		}
	    }

	byt_num.val = bytes;

	lilbuff[lil_pos++] = chr_esc;
	lilbuff[lil_pos++] = *"*";		/* high resolution graphics */
	lilbuff[lil_pos++] = (char)imode;	/* correct mode */
	lilbuff[lil_pos++] = byt_num.bit[1];	/* number of bytes */
	lilbuff[lil_pos++] = byt_num.bit[0];

	raster_x_cur = xwant + bytes;
	break;
      case imagewriter:
      case imagewriterII:
	yshift = ywant - raster_y_cur;
	if (yshift > 0)
	    {
	    status = generic_$send_lf (yshift);
	    if (status.all)
		{
		generic_$print_error (status,"raster_chunk","Unable to send linefeed","",(long)Off_);
		return(status);
		}
	    raster_y_cur = ywant;
	    }

	sprintf (lilbuff,"\033F%04.4d\033G%04.4d",xwant,bytes);
	lil_pos = 12;
	break;
      case HPlaserjet:
	/* Cannot move the cursor properly: could only use columns/rows of characters
	or decipoints, and neither give multiples of 1/75 dots in an easy way.
	So we make sure that raster lines are sent fully: no need ever for a
	horizontal move, and we can simulate vertical moves by empty raster lines. */

	/* line_number taken care of at end of graphics in generic_$render */

	yshift = ywant - raster_y_cur;
	raster_y_cur = ywant + 1;

	if (yshift > 0)
	    {
	    sprintf (lilbuff,"\033*b1W\000");
	    /* This is not repeated in lilbuff, but sent out many times, to make sure we don't overflow lilbuff */
	    for (i = 0; i < yshift; i++)
		{
		status = generic_$iosput (lilbuff,(long)6);
		if (status.all)
		    {
		    generic_$print_error (status,"raster_chunk","Unable to send empty raster line","",(long)Off_);
		    return(status);
		    }
		}
	    }

	sprintf (lilbuff,"\033*b%dW",bytes);
	lil_pos = strlen(lilbuff);
	break;
      case HPlaserjetplus:
	xshift = xwant - raster_x_cur;
	yshift = ywant - raster_y_cur;

	raster_x_cur = xwant;
	raster_y_cur = ywant + 1;

	if (xshift != 0 || yshift != 0)
	    {
	    sprintf (lilbuff,"\033*rB\033*p");
	    lil_pos = 7;

	    if (xshift != 0)
		{
		/* Since XSHIFT is in units of 8 dots as printed (by bytes), but move command in 1/300 inch. */
		xshift *= (8 * imode);
		if (yshift != 0)
		    sprintf (&lilbuff[lil_pos],"%+dx",xshift);
		else
		    sprintf (&lilbuff[lil_pos],"%+dX",xshift);
		lil_pos += strlen(&lilbuff[lil_pos]);
		}

	    if (yshift != 0)
		{
		/* Since YSHIFT is in dots as printed, but move command in 1/300 inch. */
		yshift *= imode;
		sprintf (&lilbuff[lil_pos],"%+dY",yshift);
		lil_pos += strlen(&lilbuff[lil_pos]);
		}

	    sprintf (&lilbuff[lil_pos],"\033*r1A\033*b%dW",bytes);
	    lil_pos += strlen(&lilbuff[lil_pos]);
	    }
	else
	    {
	    sprintf (&lilbuff[lil_pos],"\033*b%dW",bytes);
	    lil_pos += strlen(&lilbuff[lil_pos]);
	    }
	break;
      case microline84:
	yshift = ywant - raster_y_cur;

	if (yshift > 0)
	    {
	    status = generic_$send_lf (yshift);
	    if (status.all)
		{
		generic_$print_error (status,"raster_chunk","Unable to send linefeed","",(long)Off_);
		return(status);
		}
	    raster_x_cur = 0;
	    raster_y_cur = ywant;
	    }

	if (raster_x_cur == 0)
	    /* send blanks for margin */
	    if (chars_marg > 0)
		{
		status = generic_$iosput (sp_buff,chars_marg);
		if (status.all)
		    {
		    generic_$print_error (status,"raster_chunk","Unable to send blanks for left margin","",(long)Off_);
		    return(status);
		    }
		}

	xshift = xwant - raster_x_cur;
	if (xshift > 0)
	    {
	    xshift /= ( -i_fac);
	    if (xshift > bufsize) xshift = bufsize;
	    status = generic_$iosput (sp_buff,xshift);
	    if (status.all)
		{
		generic_$print_error (status,"raster_chunk","Unable to send blanks for spacing","",(long)Off_);
		return(status);
		}
	    }

	lilbuff[lil_pos++] = chr_esc;
	lilbuff[lil_pos++] = *"%";
	lilbuff[lil_pos++] = *"2";		/* graphics without half dots */

	byt_num.val = bytes;
	lilbuff[lil_pos + 1] = byt_num.bit[1] & 0x7f;	/* number of bytes */
	byt_num.val <<= 1;
	lilbuff[lil_pos] = byt_num.bit[0];
	lil_pos += 2;

	raster_x_cur = xwant + bytes;
	break;
      }


    if (lil_pos > 0)
	{
	status = generic_$iosput (lilbuff,lil_pos);
	if (status.all)
	    {
	    generic_$print_error (status,"raster_chunk","Unable to send move commands","",(long)Off_);
	    return(status);
	    }
	}

    status = generic_$iosput (&bigbuff[st1],bytes);
    if (status.all)
	{
	generic_$print_error (status,"raster_chunk","Unable to send raster data","",(long)Off_);
	return(status);
	}

    status.all = status_$ok;
    return(status);
}

/* =========================================================================== */

#if SR_level == SR10_2 || SR_level == SR10_3
status_$t generic_$check_passwd (
  char *passwdp,
  boolean *passwd_ok)

/* Checks password, sets passwd_ok. Reads password from (entire)
   first line of file /etc/daemons/generic_passwd .
   Make sure there are no blanks (embedded or trailing) in this line. */

{
    status_$t status,status2;
    ios_$id_t passwd_ios_id;
#define pwdsize 20
    char pwdbuff[pwdsize];
    char *namep;
    long namelen,pwdlen,goodlen;


    *passwd_ok = false;


    status = generic_$get_string (usr_db,"usr_db","Passwd__File",&namep,false,(long)Off_);
    if (status.all)
	{
	generic_$print_error (status,"check_passwd","Unable to find name of password file","",(long)Off_);
	return (status);
	}

    namelen = strlen(namep);
    if (namelen < 4 || namelen > bufsize)
	{
	sprintf (genbuff,"Wrong name <%.80s> of password file",namep);
	generic_$print_error (status,"check_passwd",genbuff,"",(long)Off_);
	status.all = print_$wrong_db_entry_type;
	return (status);
	}

    if (namelen == 4)
	{
	if (generic_$strcmp(namep,"none",true))
	    {
#if x_debug
	    fprintf (stderr,"Name of password file is <none>, any password accepted OK\n",passwdp);
#endif
	    *passwd_ok = true;
	    return (status);
	    }
	}


    pwdlen = strlen(passwdp);
    if (pwdlen < 4 || pwdlen > pwdsize)
	{
#if x_debug
	fprintf (stderr,"Password <%s> length not between 4 and 20\n",passwdp);
#endif
	return (status);
	}

    if (generic_$strcmp(passwdp,"not_present",false))
	{
#if x_debug
	fprintf (stderr,"Password was not_present, checked wrong\n");
#endif
	return (status);
	}


    passwd_ios_id = ios_$open (namep,(short)namelen,ios_$no_open_options,&status);
    if (status.all)
	{
	sprintf (genbuff,"Unable to open password file <%s>",namep);
	generic_$print_error (status,"check_passwd",genbuff,"",(long)Off_);
	return (status);
	}

    goodlen = ios_$get (passwd_ios_id,ios_$no_put_get_opts,pwdbuff,(long)pwdsize,&status);
    if (status.all)
	{
	sprintf (genbuff,"Unable to read password file <%s>",namep);
	generic_$print_error (status,"check_passwd",genbuff,"",(long)Off_);
	ios_$close (passwd_ios_id,&status2);
	return (status);
	}

    ios_$close (passwd_ios_id,&status);
    if (status.all)
	{
	sprintf (genbuff,"Unable to close password file <%s>",namep);
	generic_$print_error (status,"check_passwd",genbuff,"",(long)Off_);
	return (status);
	}


    goodlen--; /* ignore the final LF we get in each line */
	if (goodlen < 0) goodlen = 0;
    else if (goodlen >= pwdsize ) goodlen = pwdsize - 1;
    pwdbuff[goodlen] = chr_nul;
#if x_debug
	fprintf (stderr,"Password <%s> read from file %s\n",pwdbuff,namep);
#endif


    if (pwdlen == goodlen)
	{
	if (generic_$strcmp(passwdp,pwdbuff,false))
	    {
#if x_debug
	    fprintf (stderr,"Password <%s> checked OK\n",passwdp);
#endif
	    *passwd_ok = true;
	    }
#if x_debug
	else fprintf (stderr,"Password <%s> checked wrong\n",passwdp);
#endif
	}
#if x_debug
    else fprintf (stderr,"Password <%s> is not same length as <%s>\n",passwdp,pwdbuff);
#endif


    return (status);
}
#endif

/* =========================================================================== */

boolean generic_$strcmp (
  char *str1,
  char *str2,
  boolean case_ignore)

{
    char c1,c2;
    char *cp1,*cp2;

    for (cp1 = str1, cp2 = str2; ; cp1++, cp2++)
      {
      if (*cp1 != *cp2)
	{
	if (case_ignore)
	  {
	  c1 = *cp1; c2 = *cp2;
	  if (c1 >= *"A" && c1 <= *"Z") { c1 = c1 + (*"a" - *"A"); if (c1 == c2) continue; }
	  else if (c2 >= *"A" && c2 <= *"Z") { c2 = c2 + (*"a" - *"A"); if (c1 == c2) continue; }
	  }
	return (false);
	}
      /* Enough to test one string: they are same */
      if (*cp1 == chr_nul) return (true);
      }
}

/* =========================================================================== */

/* Length of string (same as strlen) */
/* Currently not used */

/*
long generic_$strlen (
  char *str)

{
    long count;
    char *cp;

    for (count = 0, cp = str; *cp != chr_nul; count++, cp++) ;

    return (count);
}
*/

/* =========================================================================== */

long generic_$str_nonblank_len (
  char *buffer,
  long max_len)

/* Length of string, ignoring trailing blanks (and tabs and backspaces) */

{
    char *strp;
    long i;

    i = max_len;
    strp = buffer + i - 1;
    for (; i > 0; i--, strp--)
	{
	if (*strp != chr_sp)
	    {
	    if (*strp != chr_tab)
		{
		if (*strp != chr_bs) return (i);
		}
	    }
	}
    return (0);
}

/* =========================================================================== */

/*  The following is really driver_init_generic.c */

/*zzzzz There is a bug in #include <apollo/trait.h>.
Try this instead.
[Begin insert file] zzzzz*/
/* T R A I T - trait system calls and defns -- phl -- 09/18/84 */
/*
 *
 * --------------------------------------------------------------------------
 * |   THE FOLLOWING PROGRAMS ARE THE SOLE PROPERTY OF APOLLO COMPUTER INC.   |
 * |         AND CONTAIN ITS PROPRIETARY AND CONFIDENTIAL INFORMATION.        |
 * --------------------------------------------------------------------------
 *
 */

/* *** This include file for use with Streams managers *** */

/*
 * CHANGES:
 *   2 May 89 psz    Fixed bug in declaration of trait_$operation_proc_t
 *   09/11/87 dbl    Cleaned up hard tabs and added member constants.
 *   04/30/87 rand   Converted to C from /sys/ins/trait.ins.pas
 */

#ifndef apollo_trait_h
#define apollo_trait_h

	/* status codes */

#define trait_$subsys            (0x13)
#define trait_$modc              (0x01)
#define trait_$module_code       (0x13010000)

#define trait_$too_many_dcls     (trait_$module_code + 0x0001)
#define trait_$no_rws            (trait_$module_code + 0x0002)

typedef void (*trait_$operation_proc_t)();  /*zzzzz Used to be 'typedef void (*trait_$operation_proc_t)(...);' zzzzz*/

typedef struct {
	    short           op_cnt;
	    uid_$t          trait_uid;
	    struct {
		uid_$t                  uid;
		trait_$operation_proc_t unimp;
	    } ops[1];
	} trait_$t;

typedef short enum {
	trait_$kind_local,        /* use this EPV for local objects */
	trait_$kind_near_remote,  /* use this EPV for remote objects in MY internet net */
	trait_$kind_far_remote,   /* use this EPV for remote objects outside my internet net */
	trait_$kind_4,
	trait_$kind_5,
	trait_$kind_6,
	trait_$kind_7,
	trait_$kind_8,
	trait_$kind_9,
	trait_$kind_10,
	trait_$kind_11,
	trait_$kind_12,
	trait_$kind_13,
	trait_$kind_14,
	trait_$kind_15,
	trait_$kind_16,
	trait_$kind_17,
	trait_$kind_18,
	trait_$kind_19,
	trait_$kind_20,
	trait_$kind_21,
	trait_$kind_22,
	trait_$kind_23,
	trait_$kind_24,
	trait_$kind_25,
	trait_$kind_26,
	trait_$kind_27,
	trait_$kind_28,
	trait_$kind_29,
	trait_$kind_30,
	trait_$kind_31,
	trait_$kind_32
} trait_$kind_val_t;

#define trait_$member_kind_local       1   /* use this EPV for local objects */
#define trait_$member_kind_near_remote 2   /* use this EPV for remote objects in MY internet net */
#define trait_$member_kind_far_remote  4   /* use this EPV for remote objects outside my internet net */


typedef unsigned long trait_$kind_t;  /* Really a set of trait_$kind_val_t */
/* Add together the ones wanted to form a set: (trait_$member_kind_local + trait_$member_kind_near_remote) */

typedef void *trait_$epv; /* array of procs, or a struct with each field being a proc */

/*
 * T R A I T _ $ M G R _ D C L -- associate an EPV with an object type UID
 * and the circumstances under which that EPV should be used.
 */

void trait_$mgr_dcl(
	uid_$t          &typuid,        /* type for which I am manager */
	void            *trait,         /* (trait_$t) trait I support */
	trait_$kind_t   &kind_set,      /* criteria for use of this EPV */
	trait_$epv      &epv,           /* addr of array of op ptrs (e.g. &struct with procs or just array of procs) */
	status_$t       *st);

#endif
/*zzzzz [End insert file] zzzzz*/

#include <apollo/pfm.h> /*zzzzz There are bugs in pfm.h: definition of pfm_$fh_func_val_t zzzzz*/


/*zzzzz BUG: compiler needs keyword struct, but (sys5.3) lint complains about it zzzzz*/
struct prsvr_driver_$epv_t prsvr_driver_epv =
	{
	generic_$init,
	NULL,
	generic_$render,
	generic_$end_job,
	generic_$close
	};

extern uid_$t generic_$uid;

extern void generic_$initialize (void);
extern prsvr_driver_$epv_ptr_t prsvr_$driver_debug (
    char name[80],
    long *namelen,
    status_$t *status);

/* =========================================================================== */

void generic_$initialize (void) /*zzzzz Why do we need (void) here? (sys5.3) lint complains if missing. zzzzz*/

/*  This is the procedure that loads the driver dynamically. Bind this module with
    the main driver code and the object generic_uid.bin generated by crtyobj. */

{
    uid_$t              typuid;
    void                *trait;
    trait_$kind_t       kind_set;
    trait_$epv          epv;
    status_$t           status;

    typuid = generic_$uid;
    trait = &prsvr_driver_$trait;
    kind_set = trait_$member_kind_local + trait_$member_kind_near_remote;
    epv = &prsvr_driver_epv;

#if x_debug
    fprintf (stderr,"Calling trait_$mgr_dcl in generic_$initialize\n");
#endif

    trait_$mgr_dcl (    typuid,         /* type for which I am manager */
			trait,          /* trait being declared */
			kind_set,       /* criteria for use of this EPV */
			epv,            /* address of array of op ptrs */
			&status);

#if x_debug
    fprintf (stderr,"trait_$mgr_dcl returned status %x\n",status.all);
#endif

    if (status.all) pfm_$error_trap (status);
}

/* =========================================================================== */

prsvr_driver_$epv_ptr_t prsvr_$driver_debug (
  char name[80],
  long *namelen,
  status_$t *status)

/*  This is the procedure to use for debugging a driver. Bind this module with
    the main driver code and set 'debug_driver' in the prsvr config file.
    Note: the print server is built with obj compilers. Therefore you must
    compile and bind this module with the obj compilers and binder. */

{
    static struct prsvr_driver_$epv_t prsvr_driver_epv;

    /* The name, namelen parameters are a check to be sure
    we are looking at the correct driver.
    In most cases they could be ignored. */
    name[*namelen] = 0;
    if (generic_$strcmp("generic",name,false))
	{
	prsvr_driver_epv.init           = generic_$init;
	prsvr_driver_epv.monitor        = NULL;
	prsvr_driver_epv.render         = generic_$render;
	prsvr_driver_epv.end_job        = generic_$end_job;
	prsvr_driver_epv.close          = generic_$close;

	(*status).all = status_$ok;
	return (&prsvr_driver_epv);
	}
    else
	{
	(*status).all = print_$rdr_dev_open_error;
	return (0);
	}
}

/* =========================================================================== */

