/* ile.c -- Freedman's Input Line Editor.  Lets you edit all input to */
/* UNIX using emacs-like commands.  Please feel free to improve upon  */
/* this code -- god knows it could use it!  It is, however, very stable */
/* and has been in use since 1986 on Suns, Vaxes, and Apollos.  If you */
/* feel like extending it, how about adding redefinable keymaps,       */
/* searching backwards for commands, proper multi-line command editing */
/* (a la Apollo's display manager), proper tab-handling, cut & paste to */
/* X windows, ... the list goes on.  If you do improve ile, please mail */
/* me your improvements so I can include them in the standard distribution */

/* Copyright 1986 Dan Freedman, Calgary, Alberta, Canada (403) 251-2729. */
/* email (for the time being) to dan@cpsc.ucalgary.ca                    */




#include <stdio.h>
#include <strings.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/wait.h>
#include <ctype.h>
#include "ile.h"

#define OK                0
#define FINISHED_EDITING -1
#define FINISHED_BUT_DONT_ADD_CTRL_M -2
#define UNIVERSAL_ARGUMENT_MAXIMUM 256
#define EXECUTIVE_PATH "/users/sys_proj/dan/bin/executive"

/* Global Variables */

int pty_master, pty_slave;

struct sgttyb old_stdin_sgttyb, new_stdin_sgttyb, pty_master_sgttyb;
struct sgttyb pty_slave_sgttyb;

char pty_master_filename[50], pty_slave_filename[50];

char tc_ent[1024], tc_seq_buf[1024], *clear_to_eol; 
char *cursor_left, *cursor_right;
char *enter_insert_mode, *exit_insert_mode; 
char *pre_insert_char, *post_insert_char;
char *enter_delete_mode, *exit_delete_mode, *delete_a_char;
char *clear_screen;

char lisp_mode = 0;

int (*keymap[256])();

struct ed_buffs {
    char string[1024];
    char *dot;
    struct ed_buffs *next_ptr;
    struct ed_buffs *prev_ptr;
    };

typedef struct _ed_struct {
    char current_input_char;
    int  universal_argument;
    char *current_buffer;
    char *dot;
    char kill_buffer[1024];
    struct ed_buffs *current_ed_buff_ptr;
    } ED_STRUCT;


ED_STRUCT editor_data;

char output_string[1024], temp_str[1024];
int  output_string_length;
int  dummy_int;


main(ac, av) int ac; char **av; {
    
    get_termcap_stuff();
    set_up_pty_and_tty();
    set_up_ioctl_stuff();
    set_up_buffers();
    set_up_keymap();
    set_up_signal_handlers();
    spawn_new_shell();
    ile();
    prepare_for_shutdown();
    shutdown_pty_and_tty();
    restore_ioctl_stuff();
    shutdown();
    exit();
    }

append_to_output_string (c) char c; {
    output_string[output_string_length++] = c;
    }

int forward_word (e) ED_STRUCT *e; {
    
    char *ch;
    int how_many, num_to_advance;

    how_many       = e->universal_argument;
    num_to_advance = 0;
    ch             = e->dot;

    while (how_many--) {
	if ((!isalnum (*ch)) && (*ch != '-') && (*ch != '_'))
	    while (*ch && (!isalnum(*ch)) && (*ch != '-') && (*ch != '_'))
	        num_to_advance++, ch++;

	while (*ch && (isalnum (*ch) || (*ch == '-') || (*ch == '_')))
	    num_to_advance++, ch++;

	if (*ch == '\0') how_many = 0;
	}

    e->universal_argument = num_to_advance;
    return forward_char (e);
    }


int backward_word (e) ED_STRUCT *e; {
    
    char *ch, *cb;
    int num_to_go, how_many;

    cb        = e->current_buffer;
    ch        = e->dot;
    num_to_go = 0;
    how_many  = e->universal_argument;

    if (ch-- == cb) return OK;

    while (how_many--) {
	while ((ch >= cb) && (!isalnum(*ch)) && (*ch != '-') && (*ch != '_'))
	    num_to_go++, ch--;

	while ((ch >= cb) && 
               (isalnum (*ch) || 
	        (*ch == '-') || 
	        (*ch == '_')))
	    num_to_go++, ch--;

	if (ch < e->current_buffer) how_many = 0;
	}

    e->universal_argument = num_to_go;
    return backward_char (e);
    }


int delete_word (e) ED_STRUCT *e; {
    
    char *ch;
    int how_many, num_to_advance;

    how_many       = e->universal_argument;
    num_to_advance = 0;
    ch             = e->dot;

    while (how_many--) {
	if ((!isalnum (*ch)) && (*ch != '-') && (*ch != '_'))
	    while (*ch && (!isalnum(*ch)) && (*ch != '-') && (*ch != '_'))
	        num_to_advance++, ch++;

	while (*ch && (isalnum (*ch) || (*ch == '-') || (*ch == '_')))
	    num_to_advance++, ch++;

	if (*ch == '\0') how_many = 0;
	}

    e->universal_argument = num_to_advance;
    return delete_char (e);
    }


int backspace_word (e) ED_STRUCT *e; {
    
    char *ch, *cb;
    int num_to_go, how_many;

    cb        = e->current_buffer;
    ch        = e->dot;
    num_to_go = 0;
    how_many  = e->universal_argument;

    if (ch-- == cb) return OK;

    while (how_many--) {
	while ((ch >= cb) && (!isalnum(*ch)) && (*ch != '-') && (*ch != '_'))
	    num_to_go++, ch--;

	while ((ch >= cb) && 
               (isalnum (*ch) || 
	        (*ch == '-') || 
	        (*ch == '_')))
	    num_to_go++, ch--;

	if (ch < e->current_buffer) how_many = 0;
	}

    e->universal_argument = num_to_go;
    return backspace_char (e);
    }



int forward_paren (e) ED_STRUCT *e; {
    
    char *ch = e->dot - 1;	/* pretend to go back one */
    int paren_count = 1;
    int num_to_advance = 0;
  
    while (*++ch && paren_count) {
	num_to_advance++;
	if (*ch == ')') paren_count--;	/* a ')' with no intervening ()'s */
	else if (*ch == '(')		/* search for ) matching this ( */
	    while (*++ch && paren_count) {
		num_to_advance++;
		if (*ch == ')') paren_count--;
		else if (*ch == '(') paren_count++;
		}
	}

    e->universal_argument = num_to_advance;
    return forward_char (e);
    }


int backward_paren (e) ED_STRUCT *e; {
    
    char *ch, *cb;
    int paren_count = 1;
    int num_to_go = 0;

    ch        = e->dot;
    cb        = e->current_buffer;

    if (ch != cb)		/* not already at the beginning */
      while ((--ch >= cb) && paren_count) {
	num_to_go++;
	if (*ch == '(') paren_count--;	/* a '(' with no intervening ()'s */
	else if (*ch == ')')		/* search for ( matching this ) */
	    while ((--ch >= cb) && paren_count) {
		num_to_go++;
		if (*ch == '(') paren_count--;
		else if (*ch == ')') paren_count++;
		}
	}

    e->universal_argument = num_to_go;
    return backward_char (e);
    }

int toggle_lisp_mode (e) ED_STRUCT *e; {
    lisp_mode = ~lisp_mode;
    return OK;
    }


int close_paren (e) ED_STRUCT *e; {
    
    char *old_dot;

    if (!lisp_mode) return self_insert(e);

    old_dot = e->dot;
    if (*old_dot != ')')		/* if not on a ')', insert one */
	self_insert(e);

    forward_char(e);			/* skip past ')' so bkd-prn works */
    backward_paren(e);			/* flash back to matching '(' */
    sleep(1);

    e->universal_argument = 1 + old_dot - e->dot;
    return forward_char (e);		/* skip ahead to old dot + 1 */
    }


int open_paren (e) ED_STRUCT *e; {
    
    char *old_dot;

    self_insert(e);			/* stick in an open_paren */
    if (!lisp_mode) return OK;

    old_dot = e->dot;
    if ((*old_dot == ')') || (*old_dot == 0)) {
	e->current_input_char = ')';	/* if on a ')' or at the end */
	self_insert(e);			/* then insert matching ')' */
	backward_char(e);		/* and go back inside () */
	}

    return OK;
    }


int increment_universal_argument (e) ED_STRUCT *e; {
    
    if (e->universal_argument > UNIVERSAL_ARGUMENT_MAXIMUM)
	write (1, "", 1);
    else
	e->universal_argument *= 4;
    
    return OK;
    }


int clear_display (e) ED_STRUCT *e; {
    
    int dummy;

    e->universal_argument = 1;
    output_string_length = 0;
    if (clear_screen) {
	tputs (clear_screen, &dummy, append_to_output_string);
	write (1, output_string, output_string_length);
	}
    else
	write(1, "\n\n", 2);
    draw_current_edit_line (e);
    
    return OK;
    }


int previous_line (e) ED_STRUCT *e; {
    
    int how_many;

    how_many = e->universal_argument;
    e->current_ed_buff_ptr->dot = e->dot;
    erase_current_edit_line (e);
    for (; how_many; --how_many)
	e->current_ed_buff_ptr = e->current_ed_buff_ptr->prev_ptr;
    e->current_buffer = e->current_ed_buff_ptr->string;
    e->dot = e->current_ed_buff_ptr->dot;
    draw_current_edit_line (e);
    e->universal_argument = 1;
    return OK;
    }


int next_line (e) ED_STRUCT *e; {
    
    int how_many;

    how_many = e->universal_argument;
    e->current_ed_buff_ptr->dot = e->dot;
    erase_current_edit_line (e);
    for (; how_many; --how_many)
	e->current_ed_buff_ptr = e->current_ed_buff_ptr->next_ptr;
    e->current_buffer = e->current_ed_buff_ptr->string;
    e->dot = e->current_ed_buff_ptr->dot;
    draw_current_edit_line (e);
    e->universal_argument = 1;
    return OK;
    }


int discard_current_edit_line (e) ED_STRUCT *e; {
    
    strcpy (e->kill_buffer, e->current_buffer);
    erase_current_edit_line (e);
    e->dot = e->current_buffer;
    *e->dot = '\0';
    
    return OK;
    }


int discard_rest_of_line (e) ED_STRUCT *e; {
    
    strcpy (e->kill_buffer, e->dot);
    e->universal_argument = strlen (e->dot);

    if (e->universal_argument == 0) {
	e->universal_argument = 1;
	return OK;
	}
    else
	return delete_char (e);
    }


int yank_from_kill_buffer (e) ED_STRUCT *e; {
    
    int count, dummy;

    output_string_length = 0;
    strcpy (temp_str, e->dot);
    strcpy (e->dot, e->kill_buffer);
    e->dot = e->dot + strlen (e->kill_buffer);
    strcpy (e->dot, temp_str);
    display_string_into_output_string (e->kill_buffer);
    count = display_string_into_output_string (temp_str);
    for (; count; --count)
	tputs (cursor_left, &dummy, append_to_output_string);
        write (1, output_string, output_string_length);

    return OK;
    }

int insert_interrupt_char (e) ED_STRUCT *e; {
    
    struct tchars pty_tchars;
    
    ioctl (pty_master, TIOCGETC, &pty_tchars);
    
    erase_current_edit_line(e);
    e->dot = e->current_buffer;
    *(e->dot++) = pty_tchars.t_intrc;
    *(e->dot) = '\0';
    return FINISHED_BUT_DONT_ADD_CTRL_M;
    }

int insert_quit_char (e) ED_STRUCT *e; {
    
    struct tchars pty_tchars;
    
    ioctl (pty_master, TIOCGETC, &pty_tchars);
    
    erase_current_edit_line(e);
    e->dot = e->current_buffer;
    *(e->dot++) = pty_tchars.t_quitc;
    *(e->dot) = '\0';
    return FINISHED_BUT_DONT_ADD_CTRL_M;
    }

int insert_suspend_char (e) ED_STRUCT *e; {
    
    struct ltchars pty_ltchars;
    
    ioctl (pty_master, TIOCGLTC, &pty_ltchars);
    
    erase_current_edit_line(e);
    e->dot = e->current_buffer;
    *(e->dot++) = pty_ltchars.t_suspc;
    *(e->dot) = '\0';
    return FINISHED_BUT_DONT_ADD_CTRL_M;
    }

    

int twiddle_chars (e) ED_STRUCT *e; {
    
    char temp_char;
    int num_to_back_up, count, dummy;

    output_string_length = 0;
    
    if (strlen (e->current_buffer) == 0)  {
	*e->dot++ = '';
	*e->dot = '\0';
	return FINISHED_BUT_DONT_ADD_CTRL_M;
	}
    else
    if ((e->dot - e->current_buffer) < 2)
	write (1, "", 1);
    else {
	temp_char = *(e->dot - 1);
	*(e->dot - 1) = *(e->dot - 2);
	*(e->dot - 2) = temp_char;
	num_to_back_up =  get_char_display_length (*(e->dot - 2));
	num_to_back_up += get_char_display_length (*(e->dot - 1));
	for (count = num_to_back_up; count; --count)
	    tputs (cursor_left, &dummy, append_to_output_string);
	display_char_into_output_string (*(e->dot - 2));
	display_char_into_output_string (*(e->dot - 1));
	write (1, output_string, output_string_length);
	}
    
    e->universal_argument = 1;

    return OK;
    }

int beginning_of_line (e) ED_STRUCT *e;  {
    
    int *dummy;

    e->universal_argument = 1;
    output_string[0] = '\0';
    output_string_length = 0;
    while (e->dot != e->current_buffer)  {
	e->dot--;
	if ((*(e->dot) < 32) || (*(e->dot) == 127)) {
	    tputs (cursor_left, &dummy, append_to_output_string);
	    tputs (cursor_left, &dummy, append_to_output_string);
	    }
	else
	    tputs (cursor_left, &dummy, append_to_output_string);
	}

    write (1, output_string, output_string_length);
    
    return OK;
    }

int end_of_line (e) ED_STRUCT *e;  {
    
    int *dummy;
    char *end;

    e->universal_argument = 1;
    output_string[0] = '\0';
    output_string_length = 0;
    display_string_into_output_string (e->dot);
    e->dot = &(e->current_buffer[strlen (e->current_buffer)]);

    write (1, output_string, output_string_length);
    
    return OK;
    }

int quote_char (e) ED_STRUCT *e; {

    char quoted_char;

    read (1,&quoted_char, 1);
    quoted_char &= 127;
    if (quoted_char == '\0') {
	write (1, "", 1);
	return OK;
	}
    else {
        e->current_input_char = quoted_char;
	return self_insert (e);
	}
    }

int meta_prefix (e) ED_STRUCT *e; {

    char metad_char;

    read (1,&metad_char, 1);
    metad_char &= 127;
    e->current_input_char = metad_char;
    if (keymap[(int)metad_char + 128])
	return keymap[(int)metad_char + 128](e);
    else  {
	e->universal_argument = 1;
	write (1, "", 1);
	return OK;
	}
    }


int backward_char (e) ED_STRUCT *e; {
    
    int num_to_go, count, dummy, needs_beep;

    output_string[0] = '\0';
    output_string_length = 0;

    if ((e->dot - e->universal_argument) < e->current_buffer)
	num_to_go = e->dot - e->current_buffer, needs_beep = 1;
    else
	num_to_go = e->universal_argument, needs_beep = 0;
    
    count = num_to_go;
    while (count-- > 0) {
	e->dot--;
        tputs (cursor_left, &dummy, append_to_output_string);
        if ((*(e->dot) < 32) || (*(e->dot) == 127))
	    tputs (cursor_left, &dummy, append_to_output_string);
	}
    
    if (needs_beep)
	output_string[output_string_length++] = '';
    
    write (1, output_string, output_string_length);
    
    e->universal_argument = 1;

    return OK;
    }


int forward_char (e) ED_STRUCT *e; {
    
    int num_to_go, count, dummy, needs_beep;

    output_string[0] = '\0';
    output_string_length = 0;

    if (e->universal_argument > strlen (e->dot))
	num_to_go = strlen (e->dot), needs_beep = 1;
    else
	num_to_go = e->universal_argument, needs_beep = 0;
    
    count = num_to_go;
    while (count-- > 0)
	display_char_into_output_string (*(e->dot++));
    
    if (needs_beep)
	output_string[output_string_length++] = '';
    
    write (1, output_string, output_string_length);
    
    e->universal_argument = 1;

    return OK;
    }


int backspace_char (e) ED_STRUCT *e; {

    int *dummy, rest_of_line_display_length, count;
    int num_to_delete;
    char *temp_str_ptr, *c, *new_dot;

    output_string[0] = '\0';
    output_string_length = 0;

    if ((e->dot - e->current_buffer) < e->universal_argument)
	new_dot = e->current_buffer;
    else
	new_dot = e->dot - e->universal_argument;

    num_to_delete = 0;
    for (c = e->dot -1; c >= new_dot; --c)
	if ((*c < 32) || (*c == 127))
	    num_to_delete += 2;
	else
	    num_to_delete++;
    
    if (delete_a_char) {
	for (count = num_to_delete; count; --count)
	    tputs (cursor_left, &dummy, append_to_output_string);

	if (enter_delete_mode)
	    tputs (enter_delete_mode, &dummy, append_to_output_string);

	for (count = num_to_delete; count; --count)
	    tputs (delete_a_char, &dummy, append_to_output_string);

	if (enter_delete_mode)
	    tputs (exit_delete_mode, &dummy, append_to_output_string);
	}
    else {
	for (count = num_to_delete; count; --count)
	    tputs (cursor_left, &dummy, append_to_output_string);

	rest_of_line_display_length = 
	    display_string_into_output_string (e->dot);

	for (count = num_to_delete; count; --count)
	    output_string[output_string_length++] = ' ';
	
	for (count=num_to_delete +rest_of_line_display_length; count; --count)
	    tputs (cursor_left, &dummy, append_to_output_string);
	}

    if (e->dot - e->universal_argument < e->current_buffer)
	output_string[output_string_length++] = '';

    strcpy (temp_str, e->dot);
    strcpy (new_dot, temp_str);
    e->dot = new_dot;
    e->universal_argument = 1;
    write (1, output_string, output_string_length);

    return OK;
    }

int delete_char (e) ED_STRUCT *e; {

    int *dummy, rest_of_line_display_length, count;
    int num_to_delete, num_to_back_up;
    char *temp_str_ptr, *c, *new_dot;
    
    if (*(e->dot) == '\0') {
	struct tchars pty_tchars;
	char *old_dot;
	ioctl (pty_master, TIOCGETC, &pty_tchars);
	old_dot = e->dot;
	erase_current_edit_line (e);
	*(e->dot++) = pty_tchars.t_eofc;
	*(e->dot) = '\0';
	return FINISHED_BUT_DONT_ADD_CTRL_M;
	}
    

    output_string[0] = '\0';
    output_string_length = 0;

    if (e->universal_argument > strlen (e->dot))
	new_dot = e->dot + strlen (e->dot);
    else
	new_dot = e->dot + e->universal_argument;

    num_to_delete = 0;
    for (c = e->dot; c < new_dot; ++c)
	if ((*c < 32) || (*c == 127))
	    num_to_delete += 2;
	else
	    num_to_delete++;
    
    if (delete_a_char) {
	if (enter_delete_mode)
	    tputs (enter_delete_mode, &dummy, append_to_output_string);

	for (count = num_to_delete; count; --count)
	    tputs (delete_a_char, &dummy, append_to_output_string);

	if (enter_delete_mode)
	    tputs (exit_delete_mode, &dummy, append_to_output_string);
	}
    else {
        num_to_back_up = display_string_into_output_string (new_dot);
	for (count = num_to_delete; count; --count)
	    output_string[output_string_length++] = ' ';
	for (count = num_to_back_up + num_to_delete; count; --count)
	    tputs (cursor_left, &dummy, append_to_output_string);
	}

    if (e->universal_argument > strlen (e->dot))
	output_string[output_string_length++] = '';

    strcpy (temp_str, new_dot);
    strcpy (e->dot, temp_str);
    e->universal_argument = 1;
    write (1, output_string, output_string_length);

    return OK;
    }


    

int finish_editing_line (e) ED_STRUCT *e; {

    char *old_dot;

    e->universal_argument = 1;
    old_dot = e->dot;
    beginning_of_line(e);
    e->dot = old_dot;            /* necessary to preserve cursor position
	                            accross newlines                      */
    return FINISHED_EDITING;
    }


int self_insert (e) ED_STRUCT *e;  {

    char *output_string_end;
    int dummy, num_to_back_up, count;

    strcpy (output_string, "");
    output_string_end = output_string;
    output_string_length = 0;

    if (*(e->dot) == '\0') {
	for (count = e->universal_argument; count; --count)
	    display_char_into_output_string (e->current_input_char);
	}
    else {
        if (enter_insert_mode || pre_insert_char) {
	    if (enter_insert_mode)
		tputs (enter_insert_mode, &dummy, append_to_output_string);

	    for (count = e->universal_argument; count; --count) {
		if (pre_insert_char)
		    tputs (pre_insert_char, &dummy, append_to_output_string);
		if ((e->current_input_char < 32)) {
		    output_string[output_string_length++] = '^';
		    if (post_insert_char)
			tputs (post_insert_char, &dummy, 
			    append_to_output_string);
		    if (pre_insert_char)
			tputs (pre_insert_char, &dummy, 
			    append_to_output_string);
		    output_string[output_string_length++] = 
			e->current_input_char + '@';
		    }
		else
		if (e->current_input_char == 127) {
		    output_string[output_string_length++] = '^';
		    if (post_insert_char)
			tputs (post_insert_char, &dummy, 
			    append_to_output_string);
		    if (pre_insert_char)
			tputs (pre_insert_char, &dummy, 
			    append_to_output_string);
		    output_string[output_string_length++] = '?';
		    }
		else
		    output_string[output_string_length++] = 
		        e->current_input_char;
		
		if (post_insert_char)
		    tputs (post_insert_char, &dummy, append_to_output_string);
		}

	    if (enter_insert_mode)
		tputs (exit_insert_mode, &dummy, append_to_output_string);
	    }
	else  {
	    for (count = e->universal_argument; count; --count)
		display_char_into_output_string (e->current_input_char);
	    num_to_back_up = 
		display_string_into_output_string (e->dot);
	    for (count = num_to_back_up; count; --count)
		tputs (cursor_left, &dummy, append_to_output_string);
	    }
	}
    
    strcpy (temp_str, e->dot);
    for (count = e->universal_argument; count; --count)
	*(e->dot++) = e->current_input_char;
    strcpy (e->dot, temp_str);

    write (1, output_string, output_string_length);
    
    e->universal_argument = 1;
    return OK;
    }


run_program_connected_to_standard_tty (cmd) char *cmd; {
    
    int status;

    ioctl (0, TIOCGETP, &new_stdin_sgttyb);
    ioctl (0, TIOCSETP, &old_stdin_sgttyb);
    printf ("Now running \"%s\"\n", cmd);
    status = system (cmd);
    printf ("Finished running \"%s\"\n", cmd);
    ioctl (0, TIOCSETP, &new_stdin_sgttyb);
    write (1, "Continue: ", 10);
    return status;
    }


quietly_run_program_connected_to_standard_tty (cmd) char *cmd; {
    
    int status;

    ioctl (0, TIOCGETP, &new_stdin_sgttyb);
    ioctl (0, TIOCSETP, &old_stdin_sgttyb);
    status = system (cmd);
    ioctl (0, TIOCSETP, &new_stdin_sgttyb);
    return status;
    }


int uppercase_word (e) ED_STRUCT *e; {
    
    char *ch, *cb;

    cb        = e->current_buffer;
    ch        = e->dot;

    if (ch != cb)		/* not at beginning of buffer */
	while ((ch >= cb) && (!isalnum(*ch)) && (*ch != '-') && (*ch != '_'))
	    ch--;		/* search backwards until in a word */

    while ((ch >= cb) && (isalnum(*ch) || (*ch == '-') || (*ch == '_')))
	  ch--;			/* find first letter in word */

    while (isalnum(*++ch) || (*ch == '-') || (*ch == '_'))
	if ((*ch >= 'a') && (*ch <= 'z'))
	    *ch -= 32;		/* uppercase all lowers */

    erase_current_edit_line(e);
    draw_current_edit_line(e);
    return OK;
    }

int lowercase_word (e) ED_STRUCT *e; {
    
    char *ch, *cb;

    cb        = e->current_buffer;
    ch        = e->dot;

    if (ch != cb)		/* not at beginning of buffer */
	while ((ch >= cb) && (!isalnum(*ch)) && (*ch != '-') && (*ch != '_'))
	    ch--;		/* search backwards until in a word */

    while ((ch >= cb) && (isalnum(*ch) || (*ch == '-') || (*ch == '_')))
	ch--;			/* find first letter in word */

    while (isalnum(*++ch) || (*ch == '-') || (*ch == '_'))
	if ((*ch >= 'A') && (*ch <= 'Z'))
	    *ch += 32;		/* lowercase all uppers */

    erase_current_edit_line(e);
    draw_current_edit_line(e);
    return OK;
    }


int capitalize_word (e) ED_STRUCT *e; {
    
    char *ch, *cb;
    int subword_flag = 0;

    cb        = e->current_buffer;
    ch        = e->dot;

    if (ch != cb)		/* not at beginning of buffer */
	while ((ch >= cb) && (!isalnum(*ch)) && (*ch != '-') && (*ch != '_'))
	    ch--;		/* search backwards until in a word */

    while ((ch >= cb) && (isalnum(*ch) || (*ch == '-') || (*ch == '_')))
	ch--;			/* find first letter in word */

    if ((*++ch >= 'a') && (*ch <= 'z'))
	*ch -= 32;		/* uppercase first letter */

    while (isalnum(*++ch) || (*ch == '-') || (*ch == '_'))
	if (!isalnum(*ch))
	    subword_flag = 1;	/* next char starts subword*/
	else if (subword_flag) {
	    subword_flag = 0;
	    if ((*ch >= 'a') && (*ch <= 'z'))
		*ch -= 32;	/* uppercase first letter */
	    }
	else if ((*ch >= 'A') && (*ch <= 'Z'))
	    *ch += 32;		/* lowercase subsequent letters */

    erase_current_edit_line(e);
    draw_current_edit_line(e);
    return OK;
    }

int ul_to_dash_word (e) ED_STRUCT *e; {
    
    char *ch, *cb;

    cb        = e->current_buffer;
    ch        = e->dot;

    if (ch != cb)		/* not at beginning of buffer */
	while ((ch >= cb) && (!isalnum(*ch)) && (*ch != '-') && (*ch != '_'))
	    ch--;		/* search backwards until in a word */

    while ((ch >= cb) && (isalnum(*ch) || (*ch == '-') || (*ch == '_')))
	ch--;			/* find first letter in word */

    while (isalnum(*++ch) || (*ch == '-') || (*ch == '_'))
	if (*ch == '_')		/* replace underscores with hyphens */
	    *ch = '-';

    erase_current_edit_line(e);
    draw_current_edit_line(e);
    return OK;
    }

int dash_to_ul_word (e) ED_STRUCT *e; {
    
    char *ch, *cb;

    cb        = e->current_buffer;
    ch        = e->dot;

    if (ch != cb)		/* not at beginning of buffer */
	while ((ch >= cb) && (!isalnum(*ch)) && (*ch != '-') && (*ch != '_'))
	    ch--;		/* search backwards until in a word */

    while ((ch >= cb) && (isalnum(*ch) || (*ch == '-') || (*ch == '_')))
	    ch--;		/* find first letter in word */

    while (isalnum(*++ch) || (*ch == '-') || (*ch == '_'))
	if (*ch == '-')		/* replace hyphens with underscores */
	    *ch = '_';

    erase_current_edit_line(e);
    draw_current_edit_line(e);
    return OK;
    }



int describe_bindings (e) ED_STRUCT *e; {
    
    char cmd[256];

    erase_current_edit_line (e);
    write (1, "One moment. . .", 15);
    strcpy (cmd, "/usr/ucb/page ");
    strcat (cmd, BINDING_DESCRIPTION);
    quietly_run_program_connected_to_standard_tty (cmd);
    write (1, "Continue: ", 10);
    draw_current_edit_line (e);
    return OK;
    }

run_tty_program (e) ED_STRUCT *e; {
    
    char cmd[512];

    erase_current_edit_line(e);
    e->dot = e->current_buffer;
    *(e->dot) = '\0';

    ioctl (0, TIOCGETP, &new_stdin_sgttyb);
    ioctl (0, TIOCSETP, &old_stdin_sgttyb);
    strcpy (cmd, "");
    printf ("Command to run ([RETURN] to cancel): ");
    fflush (stdout);
    gets (cmd);
    if (cmd[0] == '\0') {
	printf ("Command cancelled.\n");
	ioctl (0, TIOCSETP, &new_stdin_sgttyb);
	}
    else {
        ioctl (0, TIOCSETP, &new_stdin_sgttyb);
	run_program_connected_to_standard_tty (cmd);
	}

    return OK;
    }

run_talk (e) ED_STRUCT *e; {
    
    char cmd[256], *who_to;

    erase_current_edit_line(e);
    e->dot = e->current_buffer;
    *(e->dot) = '\0';

    ioctl (0, TIOCGETP, &new_stdin_sgttyb);
    ioctl (0, TIOCSETP, &old_stdin_sgttyb);
    strcpy (cmd, "talk ");
    who_to = &cmd[5];
    printf ("Talk to whom ([RETURN] to cancel): ");
    fflush (stdout);
    gets (who_to);
    if (who_to[0] == '\0') {
	printf ("Talk cancelled.\n");
	ioctl (0, TIOCSETP, &new_stdin_sgttyb);
	}
    else {
        ioctl (0, TIOCSETP, &new_stdin_sgttyb);
	run_program_connected_to_standard_tty (cmd);
	}

    return OK;
    }

run_write (e) ED_STRUCT *e; {
    
    char cmd[256], *who_to;

    erase_current_edit_line(e);
    e->dot = e->current_buffer;
    *(e->dot) = '\0';

    ioctl (0, TIOCGETP, &new_stdin_sgttyb);
    ioctl (0, TIOCSETP, &old_stdin_sgttyb);
    strcpy (cmd, "write ");
    who_to = &cmd[6];
    printf ("write to whom ([RETURN] to cancel): ");
    fflush (stdout);
    gets (who_to);
    if (who_to[0] == '\0') {
	printf ("Write cancelled.\n");
	ioctl (0, TIOCSETP, &new_stdin_sgttyb);
	}
    else {
        ioctl (0, TIOCSETP, &new_stdin_sgttyb);
	run_program_connected_to_standard_tty (cmd);
	}

    return OK;
    }

run_mesg (e) ED_STRUCT *e; {
    
    char cmd[256], *on_or_off;

    erase_current_edit_line(e);
    e->dot = e->current_buffer;
    *(e->dot) = '\0';

    ioctl (0, TIOCGETP, &new_stdin_sgttyb);
    ioctl (0, TIOCSETP, &old_stdin_sgttyb);
    strcpy (cmd, "mesg ");
    on_or_off = &cmd[5];
    printf ("Do you want \"mesg y\" or \"mesg n\" "); 
    printf ("(type y or n and press [RETURN]): ");
    fflush (stdout);
    gets (on_or_off);
    if (!strncmp (on_or_off, "on", 2))
	strcpy (on_or_off, "y");
    else 
    if (!strncmp (on_or_off, "off", 3))
	strcpy (on_or_off, "n");
    else
    if (*on_or_off == 'y' || *on_or_off == 'n')
        on_or_off[1] = '\0';
    else
    if (on_or_off[0] == '\0') {
	printf ("Mesg cancelled.\n");
	ioctl (0, TIOCSETP, &new_stdin_sgttyb);
	return OK;
	}
    else {
	printf ("Mesg cancelled because you didn't respond with y or n.\n");
	ioctl (0, TIOCSETP, &new_stdin_sgttyb);
	return OK;
	}

    ioctl (0, TIOCSETP, &new_stdin_sgttyb);
    run_program_connected_to_standard_tty (cmd);
    return OK;
    }

get_termcap_stuff() {

char *termname, *tbuf, *getenv(), *tgetstr();
    
    termname = getenv ("TERM");
    if (!termname)  {
	enter_insert_mode = exit_insert_mode = pre_insert_char = 
	    post_insert_char = 0;
	enter_delete_mode = exit_delete_mode = delete_a_char = 0;
	cursor_right = 0;
	cursor_left = "";
	clear_screen = 0;
	return;
	}
    
    tbuf = tc_seq_buf;
	if (tgetent (tc_ent, termname) != 1) {
	    enter_insert_mode = exit_insert_mode = pre_insert_char = 0;
	    post_insert_char = enter_delete_mode = exit_delete_mode = 0;
	    delete_a_char = cursor_right = clear_screen = 0;
	    cursor_left = "";
	    return;
	    }
    if (tgetflag ("bs"))
	cursor_left = "";
    else
	cursor_left = tgetstr ("bc", &tbuf);
    
    clear_to_eol      = tgetstr ("ce", &tbuf);
    cursor_right      = tgetstr ("nd", &tbuf);
    enter_insert_mode = tgetstr ("im", &tbuf);
    exit_insert_mode  = tgetstr ("ei", &tbuf);
    pre_insert_char   = tgetstr ("ic", &tbuf);
    post_insert_char  = tgetstr ("ip", &tbuf);
    enter_delete_mode = tgetstr ("dm", &tbuf);
    exit_delete_mode  = tgetstr ("ed", &tbuf);
    delete_a_char     = tgetstr ("dc", &tbuf);
    clear_screen      = tgetstr ("cl", &tbuf);
    
    if (!enter_delete_mode || !exit_delete_mode)
	enter_delete_mode = exit_delete_mode = 0;

    if (!enter_insert_mode || !exit_insert_mode)
	enter_insert_mode = exit_insert_mode = 0;
}

set_up_pty_and_tty() {

   int finished;
   char prefix, number;
   char pty_num[3], *pty_prefix_ptr, *pty_num_ptr;
   char filename[256];
   
   strcpy (filename, "/dev/pty");
   pty_prefix_ptr = &filename[8];
   pty_num_ptr    = &filename[9];
   filename[10]   = '\0';
   
   
   finished = 0;
   for (prefix = 'p'; (prefix <= 's') && !finished; ++prefix) {
       *pty_prefix_ptr = prefix;
       for (number = '1'; (number <= 'f') && !finished; 
                                   number = (number == '9') ? 'a' : ++number){
           *pty_num_ptr = number;
	   pty_master = open (filename, O_RDWR, 0700);
	   if (pty_master >= 0) {
	       strcpy (pty_master_filename, filename);
	       strcpy (pty_slave_filename, filename);
	       pty_slave_filename[5] = 't';
	       finished = 1;
	       }
	   }
       }
   if (!finished) {
       printf ("Ile: Sorry, could not open pseudo-tty channel.\n");
       printf ("      Aborting.\n\n");
       exit (-1);
       }
   }

set_up_ioctl_stuff() {

    struct sgttyb pty_sgttyb;

    ioctl (0,                   TIOCGETP, &old_stdin_sgttyb);
    ioctl (0,                   TIOCGETP, &new_stdin_sgttyb);

    new_stdin_sgttyb.sg_flags |= CBREAK;
    new_stdin_sgttyb.sg_flags &= ~ECHO;

    ioctl (0,  TIOCSETP, &new_stdin_sgttyb);

    ioctl (pty_master, TIOCGETP, &pty_sgttyb);
    pty_sgttyb.sg_flags &= ~(RAW|CBREAK|LCASE);
    pty_sgttyb.sg_flags |= ECHO|CRMOD|FF0;
    pty_sgttyb.sg_ospeed = old_stdin_sgttyb.sg_ospeed;
    pty_sgttyb.sg_ispeed = old_stdin_sgttyb.sg_ispeed;
    ioctl (pty_master, TIOCSETP, &pty_sgttyb);
    }

set_up_buffers() {

    int num_buffs;
    struct ed_buffs *prev_ptr;
    

    editor_data.current_ed_buff_ptr = 
	(struct ed_buffs *)malloc (sizeof (struct ed_buffs));
    
    prev_ptr = editor_data.current_ed_buff_ptr;
    prev_ptr->dot = prev_ptr->string;
    
    for (num_buffs = NUM_BUFFERS - 1; num_buffs; --num_buffs) {
	prev_ptr->next_ptr = 
	    (struct ed_buffs *)malloc (sizeof (struct ed_buffs));
	prev_ptr->next_ptr->prev_ptr = prev_ptr;
	prev_ptr = prev_ptr ->next_ptr;
	prev_ptr->dot = prev_ptr->string;
	*(prev_ptr->dot) = '\0';
	}
    
    prev_ptr ->next_ptr = editor_data.current_ed_buff_ptr;
    editor_data.current_ed_buff_ptr->prev_ptr = prev_ptr;
    



    editor_data.current_input_char = '\0';
    editor_data.universal_argument = 1;
    editor_data.current_buffer     = editor_data.current_ed_buff_ptr->string;
    editor_data.dot                = editor_data.current_buffer;
    editor_data.current_buffer[0]  = '\0';
    output_string[0]               = '\0';
    output_string_length           = 0;
    temp_str[0]                    = '\0';
}

set_up_keymap() {

    keymap[0]   = NULL;
    keymap[1]   = beginning_of_line;              /* CTRL-A */
    keymap[2]   = backward_char;                  /* CTRL-B */
    keymap[3]   = insert_interrupt_char;          /* CTRL-C */
    keymap[4]   = delete_char;                    /* CTRL-D */
    keymap[5]   = end_of_line;                    /* CTRL-E */
    keymap[6]   = forward_char;                   /* CTRL-F */
    keymap[7]   = NULL;
    keymap[8]   = backspace_char;                 /* CTRL-H & backspace key */
    keymap[9]   = self_insert;                    /* CTRL-I & TAB key       */
    keymap[10]  = finish_editing_line;            /* CTRL-J */
    keymap[11]  = discard_rest_of_line;           /* CTRL-K */
    keymap[12]  = clear_display;                  /* CTRL-L */
    keymap[13]  = finish_editing_line;            /* CTRL-M */
    keymap[14]  = next_line;                      /* CTRL-N */
    keymap[15]  = NULL;
    keymap[16]  = previous_line;                  /* CTRL-P */
    keymap[17]  = quote_char;                     /* CTRL-Q */
    keymap[18]  = NULL;
    keymap[19]  = NULL;
    keymap[20]  = twiddle_chars;                  /* CTRL-T */
    keymap[21]  = increment_universal_argument;   /* CTRL-U */
    keymap[22]  = NULL;
    keymap[23]  = NULL;
    keymap[24]  = NULL;
    keymap[25]  = yank_from_kill_buffer;          /* CTRL-Y */
    keymap[26]  = insert_suspend_char;            /* CTRL-Z */
    keymap[27]  = meta_prefix;                    /* CTRL-[ & ESC key */
    keymap[28]  = insert_quit_char;               /* CTRL-\ */
    keymap[29]  = NULL;
    keymap[30]  = NULL;
    keymap[31]  = NULL;
    keymap[32]  = self_insert;
    keymap[33]  = self_insert;
    keymap[34]  = self_insert;
    keymap[35]  = self_insert;
    keymap[36]  = self_insert;
    keymap[37]  = self_insert;
    keymap[38]  = self_insert;
    keymap[39]  = self_insert;
    keymap[40]  = open_paren;			  /* electric-( */
    keymap[41]  = close_paren;			  /* electric-) */
    keymap[42]  = self_insert;
    keymap[43]  = self_insert;
    keymap[44]  = self_insert;
    keymap[45]  = self_insert;
    keymap[46]  = self_insert;
    keymap[47]  = self_insert;
    keymap[48]  = self_insert;
    keymap[49]  = self_insert;
    keymap[50]  = self_insert;
    keymap[51]  = self_insert;
    keymap[52]  = self_insert;
    keymap[53]  = self_insert;
    keymap[54]  = self_insert;
    keymap[55]  = self_insert;
    keymap[56]  = self_insert;
    keymap[57]  = self_insert;
    keymap[58]  = self_insert;
    keymap[59]  = self_insert;
    keymap[60]  = self_insert;
    keymap[61]  = self_insert;
    keymap[62]  = self_insert;
    keymap[63]  = self_insert;
    keymap[64]  = self_insert;
    keymap[65]  = self_insert;
    keymap[66]  = self_insert;
    keymap[67]  = self_insert;
    keymap[68]  = self_insert;
    keymap[69]  = self_insert;
    keymap[70]  = self_insert;
    keymap[71]  = self_insert;
    keymap[72]  = self_insert;
    keymap[73]  = self_insert;
    keymap[74]  = self_insert;
    keymap[75]  = self_insert;
    keymap[76]  = self_insert;
    keymap[77]  = self_insert;
    keymap[78]  = self_insert;
    keymap[79]  = self_insert;
    keymap[80]  = self_insert;
    keymap[81]  = self_insert;
    keymap[82]  = self_insert;
    keymap[83]  = self_insert;
    keymap[84]  = self_insert;
    keymap[85]  = self_insert;
    keymap[86]  = self_insert;
    keymap[87]  = self_insert;
    keymap[88]  = self_insert;
    keymap[89]  = self_insert;
    keymap[90]  = self_insert;
    keymap[91]  = self_insert;
    keymap[92]  = self_insert;
    keymap[93]  = self_insert;
    keymap[94]  = self_insert;
    keymap[95]  = self_insert;
    keymap[96]  = self_insert;
    keymap[97]  = self_insert;
    keymap[98]  = self_insert;
    keymap[99]  = self_insert;
    keymap[100] = self_insert;
    keymap[101] = self_insert;
    keymap[102] = self_insert;
    keymap[103] = self_insert;
    keymap[104] = self_insert;
    keymap[105] = self_insert;
    keymap[106] = self_insert;
    keymap[107] = self_insert;
    keymap[108] = self_insert;
    keymap[109] = self_insert;
    keymap[110] = self_insert;
    keymap[111] = self_insert;
    keymap[112] = self_insert;
    keymap[113] = self_insert;
    keymap[114] = self_insert;
    keymap[115] = self_insert;
    keymap[116] = self_insert;
    keymap[117] = self_insert;
    keymap[118] = self_insert;
    keymap[119] = self_insert;
    keymap[120] = self_insert;
    keymap[121] = self_insert;
    keymap[122] = self_insert;
    keymap[123] = self_insert;
    keymap[124] = self_insert;
    keymap[125] = self_insert;
    keymap[126] = self_insert;
    keymap[127] = backspace_char;                 /* DEL key */
    keymap[128] = NULL;
    keymap[129] = NULL;
    keymap[130] = NULL;
    keymap[131] = NULL;
    keymap[132] = NULL;
    keymap[133] = NULL;
    keymap[134] = NULL;
    keymap[135] = NULL;
    keymap[136] = NULL;
    keymap[137] = NULL;
    keymap[138] = NULL;
    keymap[139] = NULL;
    keymap[140] = NULL;
    keymap[141] = NULL;
    keymap[142] = NULL;
    keymap[143] = NULL;
    keymap[144] = NULL;
    keymap[145] = NULL;
    keymap[146] = NULL;
    keymap[147] = NULL;
    keymap[148] = NULL;
    keymap[149] = NULL;
    keymap[150] = NULL;
    keymap[151] = NULL;
    keymap[152] = NULL;
    keymap[153] = NULL;
    keymap[154] = NULL;
    keymap[155] = discard_current_edit_line;      /* META-ESC */
    keymap[156] = NULL;
    keymap[157] = NULL;
    keymap[158] = NULL;
    keymap[159] = NULL;
    keymap[160] = NULL;
    keymap[161] = NULL;
    keymap[162] = NULL;
    keymap[163] = NULL;
    keymap[164] = NULL;
    keymap[165] = NULL;
    keymap[166] = NULL;
    keymap[167] = NULL;
    keymap[168] = backward_paren;		  /* ESC-( */
    keymap[169] = forward_paren;		  /* ESC-) */
    keymap[170] = NULL;
    keymap[171] = NULL;
    keymap[172] = NULL;
    keymap[173] = ul_to_dash_word;		  /* ESC--  */
    keymap[174] = NULL;
    keymap[175] = NULL;
    keymap[176] = NULL;
    keymap[177] = NULL;
    keymap[178] = NULL;
    keymap[179] = NULL;
    keymap[180] = NULL;
    keymap[181] = NULL;
    keymap[182] = NULL;
    keymap[183] = NULL;
    keymap[184] = NULL;
    keymap[185] = NULL;
    keymap[186] = NULL;
    keymap[187] = NULL;
    keymap[188] = NULL;
    keymap[189] = previous_line;                  /* ESC-=  */
    keymap[190] = NULL;
    keymap[191] = describe_bindings;		  /* ESC-?  */
    keymap[192] = NULL;
    keymap[193] = NULL;
    keymap[194] = NULL;
    keymap[195] = NULL;
    keymap[196] = NULL;
    keymap[197] = NULL;
    keymap[198] = NULL;
    keymap[199] = NULL;
    keymap[200] = NULL;
    keymap[201] = NULL;
    keymap[202] = NULL;
    keymap[203] = NULL;
    keymap[204] = toggle_lisp_mode;		/* ESC-L  */
    keymap[205] = NULL;
    keymap[206] = NULL;
    keymap[207] = NULL;
    keymap[208] = NULL;
    keymap[209] = NULL;
    keymap[210] = NULL;
    keymap[211] = NULL;
    keymap[212] = NULL;
    keymap[213] = NULL;
    keymap[214] = NULL;
    keymap[215] = NULL;
    keymap[216] = NULL;
    keymap[217] = NULL;
    keymap[218] = NULL;
    keymap[219] = NULL;
    keymap[220] = NULL;
    keymap[221] = NULL;
    keymap[222] = NULL;
    keymap[223] = dash_to_ul_word;		  /* ESC-_  */
    keymap[224] = NULL;
    keymap[225] = NULL;
    keymap[226] = backward_word;                  /* ESC-B  */
    keymap[227] = capitalize_word;		  /* ESC-C  */
    keymap[228] = delete_word;                    /* ESC-D  */
    keymap[229] = NULL;
    keymap[230] = forward_word;                   /* ESC-F  */
    keymap[231] = NULL;
    keymap[232] = backspace_word;                 /* ESC-H  */
    keymap[233] = NULL;
    keymap[234] = NULL;
    keymap[235] = NULL;
    keymap[236] = lowercase_word;		  /* ESC-L  */
    keymap[237] = run_mesg;                       /* ESC-M  */
    keymap[238] = NULL;
    keymap[239] = NULL;
    keymap[240] = NULL;
    keymap[241] = NULL;
    keymap[242] = run_tty_program;                /* ESC-R  */
    keymap[243] = NULL;
    keymap[244] = run_talk;                       /* ESC-T  */
    keymap[245] = uppercase_word;		  /* ESC-U  */
    keymap[246] = NULL;
    keymap[247] = run_write;                      /* ESC-W  */
    keymap[248] = NULL;
    keymap[249] = NULL;
    keymap[250] = NULL;
    keymap[251] = NULL;
    keymap[252] = NULL;
    keymap[253] = NULL;
    keymap[254] = NULL;
    keymap[255] = NULL;
}

set_up_signal_handlers() {
}

spawn_new_shell() {

    int tty;
    struct sgttyb pty_sgttyb;
    char *shellname;

    ioctl (pty_master, TIOCGETP, &pty_sgttyb);

    if (!fork()) { /* if child */
	tty = open ("/dev/tty", O_RDWR, 0700);
    dummy_int = 0;
	ioctl (tty, TIOCNOTTY, &dummy_int);
	close (tty);

    dummy_int = getpid();
	ioctl (pty_master, TIOCSPGRP, &dummy_int);

	close (0);
	close (1);
	close (2);

	pty_slave = open (pty_slave_filename, O_RDWR, 700);
	dup (0);
	dup (0);
	ioctl (pty_slave, TIOCSETP, &pty_sgttyb);
	shellname = (char *)getenv ("SHELL");
	if (!shellname)
	    execl ("/bin/csh", "-", 0);	
	else
	    execl (shellname, "-", 0);
	}
    }

ile () {

    char c[1024];
    int nfound, readfds, finished, pty_master_mask, num;
    struct sgttyb pty_sgttyb, tty_sgttyb;
    int status;

    write (1, "Welcome to Freedman's Input Line Editor (ILE).  Version: ", 
	58);
    write (1, VERSION, strlen (VERSION));
    write (1, 
	"\nPlease wait until your shell prompt appears (ESC-? for help).\n",
	63);

    ioctl (0, TIOCGETP, &tty_sgttyb);

    ioctl (pty_master, TIOCGETP, &pty_sgttyb);
    pty_sgttyb.sg_flags &= ~(RAW|CBREAK|LCASE);
    pty_sgttyb.sg_flags |= ECHO|CRMOD|FF0;
    ioctl (pty_master, TIOCSETP, &pty_sgttyb);

    tty_sgttyb.sg_flags &= ~(ECHO);
    tty_sgttyb.sg_flags |= CBREAK|RAW;
    ioctl (0, TIOCSETP, &tty_sgttyb);

    pty_master_mask = 1<<pty_master;
    finished = 0;
    while (!finished) {
	readfds = 1|pty_master_mask;
	if (wait3 (&status,  WNOHANG, 0)) finished = 1;
	else {
	    nfound = select (25, &readfds, 0, 0, 0);
	    if (readfds & 1) {
		ioctl (pty_master, TIOCGETP, &pty_sgttyb);
		if ((pty_sgttyb.sg_flags & ECHO) &&
		    !(pty_sgttyb.sg_flags & (RAW|CBREAK)))
		    num = edit_line (c);
		else
		    num = read (0, c, 1024);
		write (pty_master, c, num);
		}
	    else
		if (readfds & pty_master_mask) {
		num = read (pty_master, c, 1024);
		write (1, c, num);
		}
	    if (wait3 (&status, WNOHANG, 0)) finished = 1;
	    }
	}
    }


prepare_for_shutdown() {
}

shutdown_pty_and_tty() {

    close (pty_master);
    }

restore_ioctl_stuff() {

    ioctl (0, TIOCSETP, &old_stdin_sgttyb);
    }

shutdown() {
}

int edit_line (c) char *c; {

    char ch;
    int  status, pty_master_mask, readfds, num;
    struct ed_buffs *buf_to_use, *cur_line;

    status = OK;
    
    pty_master_mask = 1<<pty_master;

    buf_to_use = editor_data.current_ed_buff_ptr;

    while (status == OK) {
	readfds = 1|pty_master_mask;
	select (25, &readfds, 0, 0, 0);
	if (readfds & 1) {
	    read (0, &ch, 1);
	    ch &= 127; /* Must do this due to parity bit over dial-up lines */
		       /* Possibly needed for other lines too.              */
	    editor_data.current_input_char = ch;
	    if (keymap[(int)ch])
		status = keymap[(int)ch](&editor_data);
	    else {
		editor_data.universal_argument = 1;
		write (1, "", 1);
		}
	    }
	else {
	    erase_current_edit_line (&editor_data);
	    num = read (pty_master, temp_str, 1024);
	    write (1, temp_str, num);
	    draw_current_edit_line (&editor_data);
	    }
	}

    switch (status) {
        case FINISHED_EDITING:
	    strcpy (c, editor_data.current_buffer);

	    cur_line = editor_data.current_ed_buff_ptr;
	    cur_line->dot = editor_data.dot;
	    if (cur_line != buf_to_use)  {
		strcpy (buf_to_use->string, cur_line->string);
		buf_to_use->dot = 
		    &buf_to_use->string[cur_line->dot - cur_line->string];
		}
	    if (*(buf_to_use->string))  {
		editor_data.current_ed_buff_ptr = buf_to_use->next_ptr;
		editor_data.current_ed_buff_ptr->dot = 
		    editor_data.current_ed_buff_ptr->string;
		editor_data.dot = editor_data.current_ed_buff_ptr->dot;
		*editor_data.dot = '\0';
		editor_data.current_buffer = editor_data.dot;
		}
	    else {
	        editor_data.current_ed_buff_ptr = buf_to_use;
		editor_data.current_buffer      = buf_to_use->string;
		editor_data.dot                 = buf_to_use->dot;
		}
	    
	    strcat (c, "
");
	    return strlen (c);
	    break;

	case FINISHED_BUT_DONT_ADD_CTRL_M:
	    strcpy (c, editor_data.current_buffer);
	    editor_data.dot = editor_data.current_buffer;
	    *editor_data.dot = '\0';
	    return (strlen (c));
	    break;
	}
    strcpy (c, "");
    return 0;
    }


int get_display_length (s) char *s; {

    char *cptr;
    int  len;
    
    cptr = s;
    len = 0;
    while (*cptr) {
	if ((*cptr < 32) || (*cptr == 127))
	    len += 2;
	else
	    len ++;
	++cptr;
	}
    return len;
    }

int get_char_display_length (c) char c; {

    int  len;
    
    len = 0;
    if ((c < 32) || (c == 127))
	len = 2;
    else
	len++;
    return len;
    }

int display_string_into_output_string (s) char *s; {
    
    int len;
    char *cptr;
    
    cptr = s;
    len = 0;
    while (*cptr) {
	if (*cptr < 32) {
	    len += 2;
	    output_string[output_string_length++] = '^';
	    output_string[output_string_length++] = 
		*cptr++ + '@';
	    }
	else if (*cptr == 127) {
	    len += 2;
	    output_string[output_string_length++] = '^';
	    output_string[output_string_length++] = '?';
	    ++cptr;
	    }
	else {
	    len++;
	    output_string[output_string_length++] = *cptr++;
	    }
	}
    
    return len;
    }

int display_char_into_output_string (c) char c; {
    
    int len;
    
    len = 0;
    if (c < 32) {
	len += 2;
	output_string[output_string_length++] = '^';
	output_string[output_string_length++] = c + '@';
	}
    else if (c == 127) {
        len += 2;
	output_string[output_string_length++] = '^';
	output_string[output_string_length++] = '?';
	}
    else {
        len++;
	output_string[output_string_length++] = c;
	}
    
    return len;
    }

int erase_current_edit_line (e) ED_STRUCT *e; {

    int num_to_erase, dummy, count;
    char *old_dot;
    
    old_dot = e->dot;
    beginning_of_line (e);
    e->dot = old_dot;
    output_string_length = 0;

    num_to_erase = get_display_length (e->current_buffer);
    for (count = num_to_erase; count; --count)
	output_string[output_string_length++] = ' ';
    for (count = num_to_erase; count; --count)
	tputs (cursor_left, &dummy, append_to_output_string);
    write (1, output_string, output_string_length);
    
    return OK;
    }

int draw_current_edit_line (e) ED_STRUCT *e; {
    
    int count, dummy;
    
    output_string_length = 0;
    display_string_into_output_string (e->current_buffer);
    count = get_display_length (e->dot);
    for (; count; --count)
	tputs (cursor_left, &dummy, append_to_output_string);

    write (1, output_string, output_string_length);

    return OK;
    }
