/*   tiff2gpr.c  - displays a TIFF file on the apollos
 *
 *                 01/04/1991
 *
 * You need Sam Leffler's TIFF library to build this program.
 *
 *  This file contains system-specific graphics routines.
 *  These routines use Apollo Graphics Primitive Resource (GPR) calls.
 *
 *  Authors:   Roque D. Oliveira      oliveria@caen.engin.umich.edu
 *             Amadi Nwankpa             amadi@caen.engin.umich.edu
 * 
 *  If you make any improvements/bug fixes to this program, please send
 *  them to the authors.
 *
 * The bulk of this code (reading data from a TIFF file) was taken
 * right out of Sam Leffler's TIFF library (tools/tiffgt.c). 
 * You need Sam Leffler's TIFF library to build this program.
 *
 */

/*
 * Copyright (C) 1990 by the author(s).
 * All rights reserved.
 *
 * Permission is granted to copy, modify and distribute this program
 * for any purpose and without charge, provided this copyright notice
 * is included in the copy.
 * This program is distributed on an as-is basis. There will be
 * ABSOLUTELY NO WARRANTY for any part of this software to work
 * correctly. In no case will the author(s) be liable to you for any
 * lost revenue or profits or other special, indirect and consequential
 * damages caused by the usage of this program.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <apollo/base.h>
#include <apollo/gpr.h>
#include <apollo/pad.h>
#include <apollo/error.h>
#include "tiffio.h"

#define	streq(a,b)	(strcmp((a),(b)) == 0)

char *malloc();
int strncmp(); 
                            
#define FALSE 0
#define TRUE 1
char *progname;				/* the name of this program */
#define VERSION    "1"
#define PATCHLEVEL "02"
                    
status_$t            status;
gpr_$color_vector_t  color_map; 
gpr_$rgb_plane_t     hi_plane;
gpr_$disp_char_t     display_characteristics;

static short int     disp_len = sizeof(gpr_$disp_char_t);
       short int     disp_len_returned;

gpr_$offset_t        init_size, eSIZE;
gpr_$position_t      eORIGIN;
gpr_$bitmap_desc_t   display_bitmap_desc;
gpr_$display_mode_t  mode=gpr_$direct;
gpr_$pixel_value_t   *dest_pixels;
gpr_$pixel_value_t   *foo;             /* Scan line buffer */                 
stream_$id_t         unit;
pad_$window_desc_t   pad_window ;

char        wait_string[]   = {"Resizing Image.  Please wait..."};
short int   wait_string_len = sizeof(wait_string);

unsigned int save_to_bitmap = 0;
char *bitmap_file_name = NULL;

void redraw(); 
gpr_$rwin_pr_t  rwin = redraw;        /* entry point for refreshing window */
gpr_$rhdm_pr_t  rhdm = NULL;          /* entry point for refreshing hidden display memory */

#define COLMAP_OFFSET 16
                         
int dm16[16][16];
int   modN[256],   divN[256];
int  mod51[256],  div51[256];
int mod255[256], div255[256];
int gammamap[256];
double gam          = 2.0;     /* Default gamma */
int    inverse_flag = false;   /* If true, swap bits on B&W displays */
int    bw_flag      = false; 
int    four_flag    = false;   /* If true, workstation has only 4 planes */
int    gray_flag    = false;   /* True if making BW image on color screen */
int    borrow_mode  = false;   /* DN560/660 imaging mode (borrow screen)) */ 
int    bm3          = false;
int    linear_flag  = true;    /* Color map flags */
int    gam_flag     = false;       
int    verbose      = false;   /* produce verbose output */
int    tiff_info    = false;   /* display information about TIFF file */
int    display_info = false;   /* display information about the display */
int    full_screen  = false;   /* Don't expand image to max size by default */
short  map[3][256];    
    
/* Basic magic square for dithering */
int magic[4][4] = {
                    0, 14,  3, 13,
                   11,  5,  8,  6,
                   12,  2, 15,  1,
                    7,  9,  4, 10
                   };

u_long	*raster;                /* displayable image */
u_short  WIDTH,  HEIGHT;        /* image width & height */
u_short eWIDTH, eHEIGHT;        /* Expanded image width & height (when user resizes it) */

int    *ix_inc_table;           /* Resize x table buffer */

u_short	bitspersample;
u_short	samplesperpixel;
u_short	photometric;
u_short	orientation;
u_short	*redcmap, *greencmap, *bluecmap;  /* colormap for pallete images */

#define rgbi(r,g,b) ( b|(g<<8)|(r<<16) )

typedef u_long  RGBvalue ;

void usage()
{
  (void) fprintf(stderr,"usage: %s [-i] [-d] [-s] [-f] [-o bitmap] [-g gamma] [-b] tiff_file_name \n",progname);
  (void) fprintf(stderr,"\noptions                   description \n\n");
  (void) fprintf(stderr,"  -i           Display information about the TIFF file. \n");
  (void) fprintf(stderr,"  -d           Display information about the display. No TIFF file is required. \n");
  (void) fprintf(stderr,"  -s           Swap bits on B&W display. \n");
  (void) fprintf(stderr,"  -f           Display TIFF file in full screen mode, preserving aspect ratio. \n");
  (void) fprintf(stderr,"  -o filename  Choose output filename to save image as a gpr bitmap. \n");
  (void) fprintf(stderr,"  -b           Choose gray scale dithering. \n");
  (void) fprintf(stderr,"  -g gamma     Select a gamma factor for dithering the image. Default is 2. \n");
  exit(-1);
}

void fatal(s)
char *s;
{
 fprintf(stderr,"%s: %s\n",progname,s);
 exit(-1);
}

void check(messagex)
char *messagex;
{
 if (status.all)
   {
    error_$print (status);
    fprintf(stderr,"Error occurred while %s.\n", messagex);
   }    
}

main(argc, argv)
int  argc;
char     **argv;
{
	TIFF        *tif;
	int		     c;
	extern int	 optind;    /* index of which argument is next */
	extern char	*optarg;    /* pointer to argument of this option */ 

    progname = strrchr(argv[0], '/');
    if (progname)
	   progname++;
    else
	   progname = argv[0];
                           
    while ((c=getopt(argc,argv,"disfvbo:g:")) != -1) 
	switch (c)
         {
          case 'o':
                  save_to_bitmap = 1;
                  bitmap_file_name = optarg;
                  fprintf(stdout,"output gpr bitmap file name = %s \n",bitmap_file_name);
                  break;
          case 'd':
                  display_info = 1;
                  break;
          case 'i':
                  tiff_info = 1;
                  break;
          case 's':
                 inverse_flag = 1;
                 break;
          case 'f':
                  full_screen = 1;
                  break;
          case 'v':
                  verbose = 1;
                  break;  
          case 'b':
                 gray_flag = true;
                 break;
          case 'g':
                 gam = atof(optarg);
                 gam_flag = true;              
             if (gam == 0)
               {
                fprintf(stdout,"bad value for gamma, using gamma = 2.0 \n" );  
                gam = 2.0;
               }
                break;
          case '?':
          default :
                  usage();
         }

    if ((argc - optind < 1) && !display_info)
      {
       usage();
      }
    else
      {
       gpr_$inq_disp_characteristics(mode,(short)1,disp_len,&display_characteristics,&disp_len_returned,&status);
       check("in main, after calling gpr_$inq_display_characteristics");
       if (display_info)
         {
          print_display_info();
          if (argc - optind < 1) exit(0);
         }

       tif = TIFFOpen(argv[optind], "r");
       if (tif == NULL)
         {
          exit(-1);     
         }
       (void) fprintf(stdout,"\nTIFF file is   %s \n",argv[optind]);
       optind++;
      }

    if ( optind != argc ) usage();
                  
    if (tiff_info) TIFFPrintDirectory(tif, stdout,0,0,0);

	if (!TIFFGetField(tif, TIFFTAG_BITSPERSAMPLE, &bitspersample))
		bitspersample = 1;
                     
    if (verbose) fprintf(stdout,"in main,  Bits/Sample: %d \n",bitspersample);

	switch (bitspersample)
    {
	case 1: case 2: case 4:
	case 8: case 16:
		break;
	default:
		fprintf(stderr, "Sorry, can't handle %d-bit pictures\n",bitspersample);
		exit(-1);
	}

	if (!TIFFGetField(tif, TIFFTAG_SAMPLESPERPIXEL, &samplesperpixel))
		samplesperpixel = 1;

    if (verbose) fprintf(stdout,"in main,  Samples/Pixel: %d \n",samplesperpixel);

	switch (samplesperpixel)
    {
	case 1: case 3: case 4:
		break;
	default:
        fprintf(stderr, "Sorry, can't handle %d-channel images\n",samplesperpixel);
		exit(-1);
	}                           

	TIFFGetField(tif, TIFFTAG_IMAGEWIDTH,  &WIDTH);
	TIFFGetField(tif, TIFFTAG_IMAGELENGTH, &HEIGHT);

    if(!tiff_info) fprintf(stdout,"Image Width: %d Image Length: %d \n",WIDTH,HEIGHT);

	raster = (u_long *) malloc(WIDTH * HEIGHT * sizeof (long));
	if (raster == 0)
      {
       fprintf(stderr, "Not enough space for raster buffer\n");
       exit(-1);
      }

    Initialize();

    if (verbose)
      {
       (void) fprintf(stdout,"\nOptions being used are:\n");
       (void) fprintf(stdout,"  display_info  = %d \n", display_info );
       (void) fprintf(stdout,"  tiff_info     = %d \n", tiff_info );
       (void) fprintf(stdout,"  gray_flag     = %d \n", gray_flag );
       (void) fprintf(stdout,"  inverse_flag  = %d \n", inverse_flag );
       (void) fprintf(stdout,"  full_screen   = %d \n", full_screen );
       (void) fprintf(stdout,"  gamma         = %f \n", gam );
      }

    window_color_map();
                               
	gt(tif, WIDTH,HEIGHT, raster);
	TIFFClose(tif);

    redraw(); 

    gpr_$set_cursor_active(true,&status);

/* Establish the refresh function. Once established, if the window needs to be 
   refreshed as the result of a pop or grow, GPR will automatically call the redraw function. */
    gpr_$set_refresh_entry (rwin,rhdm, &status);
    check("in putimage after calling gpr_$set_refresh_entry");          

    KbEnable();

    if(save_to_bitmap) 
      { 
       SaveImage(display_bitmap_desc,bitmap_file_name);
      }

    Close();
}

/***********************************/

#define	howmany(x, y)	(((x)+((y)-1))/(y))
RGBvalue **BWmap;

static gt(tif, w, h, raster)
TIFF     *tif;
int           w , h;
u_long              *raster;
{
	u_short minsamplevalue, maxsamplevalue, planarconfig;
	RGBvalue *Map;
	int e;

	if (!TIFFGetField(tif, TIFFTAG_PHOTOMETRIC, &photometric)) 
    {
		switch (samplesperpixel)
        {
		case 1:
			photometric = PHOTOMETRIC_MINISBLACK;
			break;
		case 3: case 4:
			photometric = PHOTOMETRIC_RGB;
			break;
		default:
			fprintf(stderr, "Missing needed \"%s\" tag.\n","PhotometricInterpretation");
			return (0);
		}
		printf("No \"PhotometricInterpretation\" tag, assuming %s.\n",
		    photometric == PHOTOMETRIC_RGB ? "RGB" : "min-is-black");
	}

	if (!TIFFGetField(tif, TIFFTAG_MINSAMPLEVALUE, &minsamplevalue))
		minsamplevalue = 0;
	if (!TIFFGetField(tif, TIFFTAG_MAXSAMPLEVALUE, &maxsamplevalue))
		maxsamplevalue = (1<<bitspersample)-1;
	Map = NULL;

    if (verbose) fprintf(stdout,"in gt,  Photometric Interpretation: %d \n",photometric);

	switch (photometric)
    {
	case PHOTOMETRIC_RGB:
		if (minsamplevalue == 0 && maxsamplevalue == 255)
			break;
		/* fall thru... */
	case PHOTOMETRIC_MINISBLACK:
	case PHOTOMETRIC_MINISWHITE: 
    {
		register int x, range;

		range = maxsamplevalue - minsamplevalue;
		Map = (RGBvalue *)malloc((range + 1) * sizeof (RGBvalue));
		if (Map == NULL)
          {
			fprintf(stderr, "%s.\n","gt: no space for photometric conversion table");
			return (0);
		  }
		if (photometric == PHOTOMETRIC_MINISWHITE) 
          {
			for (x = 0; x <= range; x++)
				Map[x] = ((range - x) * 255) / range;
		  }
        else
          {
			for (x = 0; x <= range; x++)
				Map[x] = (x * 255) / range;
		  }
		if (bitspersample != 8 && photometric != PHOTOMETRIC_RGB)
        {
			if (!makebwmap(Map)) return (0);
			/* no longer need Map, free it */
			free((char *)Map);
			Map = NULL;
        }
		break;
	}
	case PHOTOMETRIC_PALETTE:
		if (!TIFFGetField(tif, TIFFTAG_COLORMAP,&redcmap, &greencmap, &bluecmap))
		   {
			fprintf(stderr, "gt: missing required \"Colormap\" tag.\n");
			return (0);
           }
		/*
		 * Convert 16-bit colormap to 8-bit.
		 */
		{
          int i;
#define	CVT(x)		(((x) * 256) / ((1L<<16)-1))
    if (verbose) fprintf(stdout,"in gt, converting 16-bit colormap to 8-bit \n");
		  for (i = (1<<bitspersample)-1; i > 0; i--)
          {
			redcmap[i]   = CVT(redcmap[i]);
			greencmap[i] = CVT(greencmap[i]);
			bluecmap[i]  = CVT(bluecmap[i]);
#undef CVT 
		  }
		}
		break;

	}  /* end switch (photometric)  */


	TIFFGetField(tif, TIFFTAG_PLANARCONFIG, &planarconfig);

    if (verbose) fprintf(stdout,"in gt,  Planar Configuration: %d \n",planarconfig);

	if (planarconfig == PLANARCONFIG_SEPARATE)
      {
        if (verbose) fprintf(stdout,"in gt, before calling gtseparate \n");
		e = gtseparate(tif, raster, Map, h, w);
        if (verbose) fprintf(stdout,"in gt, after calling gtseparate \n");
      }
	else
      {
        if (verbose) fprintf(stdout,"in gt, before calling gtcontig \n");
		e = gtcontig(tif, raster, Map, h, w);
        if (verbose) fprintf(stdout,"in gt, after calling gtcontig \n");
      }
	if (Map) free((char *)Map);
	return (e);
}

/***********************************/

setorientation(tif, h)
TIFF          *tif;
int                 h;
{
	int y;

	if (!TIFFGetField(tif, TIFFTAG_ORIENTATION, &orientation))
		orientation = ORIENTATION_TOPLEFT;

    if (verbose) fprintf(stdout,"in setorientation,  orientation: %d \n",orientation);

	switch (orientation) 
    {
	case ORIENTATION_BOTRIGHT:
	case ORIENTATION_RIGHTBOT:	/* XXX */
	case ORIENTATION_LEFTBOT:	/* XXX */
		printf("Warning, using bottom-left orientation.\n");
		orientation = ORIENTATION_BOTLEFT;
		/* fall thru... */
	case ORIENTATION_BOTLEFT:
		y = 0;
		break;
	case ORIENTATION_TOPRIGHT:
	case ORIENTATION_RIGHTTOP:	/* XXX */
	case ORIENTATION_LEFTTOP:	/* XXX */
		printf("Warning, using top-left orientation.\n");
		orientation = ORIENTATION_TOPLEFT;
		/* fall thru... */
	case ORIENTATION_TOPLEFT:
		y = h-1;
		break;

	} /* end switch (orientation) */

	return (y);
}

/***********************************/

gtcontig(tif, raster, Map, h, w)
TIFF    *tif;
u_long       *raster;
register RGBvalue    *Map;
int                        h, w;
{
	register u_char *pp;
	u_char          *buf;
	register u_long *cp;
	register int     x;
	int              scanline, row, y;

    if (verbose) fprintf(stdout,"entering gtcontig \n");
    if (verbose) fprintf(stdout,"in gtcontig,  Photometric Interpretation: %d \n",photometric);

	buf = (u_char *)malloc(TIFFScanlineSize(tif));
	if (buf == 0) 
       {
		fprintf(stderr, "gtcontig: no space for scanline buffer\n");
		return (0);
       }

	y = setorientation(tif, h);

	for (row = 0; row < h; row++)
    {
		if (TIFFReadScanline(tif, buf, row, 0) < 0)
			break;
		pp = buf;
		cp = raster + y*w;
		switch (photometric) 
        {
		case PHOTOMETRIC_RGB:
			switch (bitspersample)
            {
			case 8:
				if (Map)
                  {
					for (x = w; x-- > 0;)
                       {
						pp[0] = Map[pp[0]];
						pp[1] = Map[pp[1]];
						pp[2] = Map[pp[2]];
						pp += samplesperpixel;
                       }
					pp = buf;
                  }
				for (x = w; x-- > 0;)
                  {
					*cp++ = rgbi(pp[0], pp[1], pp[2]);
					pp += samplesperpixel;
                  }
				break;
			case 16: 
                {
				register u_short *wp;

				if (Map)
                   {
					wp = (u_short *)pp;
					for (x = w; x-- > 0;)
                       {
						wp[0] = Map[wp[0]];
						wp[1] = Map[wp[1]];
						wp[2] = Map[wp[2]];
						wp += samplesperpixel;
                       }
                   }
				wp = (u_short *)pp;
				for (x = w; x-- > 0;)
                   {
					*cp++ = rgbi(wp[0], wp[1], wp[2]);
					wp += samplesperpixel;
                   }
				break;
                }
			} /* end switch (bitspersample) */
			break;
		case PHOTOMETRIC_PALETTE:
			for (x = w; x-- > 0;)
               {
				RGBvalue c = *pp++;
				*cp++ = rgbi(redcmap[c], greencmap[c], bluecmap[c]);
               }
			break;
		case PHOTOMETRIC_MINISWHITE:
		case PHOTOMETRIC_MINISBLACK:
			if (bitspersample == 8)
            {
				register RGBvalue c;

				for (x = w; x-- > 0;)
                   {
					c = Map[*pp++];
					*cp++ = rgbi(c, c, c);
                   }
			} else
				gtbw(bitspersample, w, cp, pp);
			break;

		} /* end switch (photometric) */

		y += (orientation == ORIENTATION_TOPLEFT ? -1 : 1);

    } /* end for */

    if (verbose) fprintf(stdout,"leaving gtcontig \n");
	return (1);
}

/***********************************/

gtseparate(tif, raster, Map, h, w)
TIFF      *tif;
u_long          *raster;
register RGBvalue      *Map;
int                          h, w;
{
	register u_long *cp;
	register int x;
	u_char *red;
	int scanline, row, y;

    if (verbose) fprintf(stdout,"entering gtseparate \n");
    if (verbose) fprintf(stdout,"in gtseparate,  Photometric Interpretation: %d \n",photometric);

	scanline = TIFFScanlineSize(tif);
	switch (samplesperpixel) 
    {
	case 1:
		red = (u_char *)malloc(scanline);
		break;
	case 3: case 4:
		red = (u_char *)malloc(3*scanline);
		break;
	}

	y = setorientation(tif, h);
	for (row = 0; row < h; row++)
    {
		cp = raster + y*w;
		if (TIFFReadScanline(tif, red, row, 0) < 0)
			break;
		switch (photometric) 
        {
		  case PHOTOMETRIC_RGB:
          {
			register u_char *r, *g, *b;

			r = red;
			if (TIFFReadScanline(tif, g = r + scanline, row, 1) < 0)
				break;
			if (TIFFReadScanline(tif, b = g + scanline, row, 2) < 0)
				break;
			switch (bitspersample)
            {
			case 8:
				for (x = w; x-- > 0;)
					*cp++ = rgbi(*r++, *g++, *b++);
				break;
			case 16:
#define	wp(x)	((u_short *)(x))
				for (x = 0; x < w; x++)
					*cp++ = rgbi( Map[wp(r)[x]], Map[wp(g)[x]], Map[wp(b)[x]]);
				break;
#undef	wp
			} /* end switch (bitspersample) */
			break;
		  } 

		  case PHOTOMETRIC_PALETTE:
          {
			register u_char *pp = red;
			for (x = w; x-- > 0;)
               {
				RGBvalue c = *pp++;
				*cp++ = rgbi(redcmap[c],greencmap[c], bluecmap[c]);
               }
			break;
		  }

		  case PHOTOMETRIC_MINISWHITE:
		  case PHOTOMETRIC_MINISBLACK:
			if (bitspersample == 8) 
            {
				register u_short *pp = (u_short *)red;
				register RGBvalue c;

				for (x = w; x-- > 0;)
                   {
					c = Map[*pp++];
					*cp++ = rgbi(c, c, c);
                   }
			} else
				gtbw(bitspersample, w, Map, cp, red);
			break;

		} /* end switch (photometric)  */

		y += (orientation == ORIENTATION_TOPLEFT ? -1 : 1);

	} /* end for */     

	if (red)  free(red);
    if (verbose) fprintf(stdout,"leaving gtseparate \n");
	return (1);
}

/***********************************/

/*
 * Greyscale images with less than 8 bits/sample are handled
 * with a table to avoid lots of shits and masks.  The table
 * is setup so that gtbw (below) can retrieve 8/bitspersample
 * pixel values simply by indexing into the table with one
 * number.
 */
makebwmap(Map)
RGBvalue *Map;
{
	register int       i;
	int                nsamples = 8 / bitspersample;
	register RGBvalue *p;

    if (verbose) fprintf(stdout,"entering makebwmap \n");

	BWmap = (RGBvalue **)malloc(256*sizeof (RGBvalue *)+(256*nsamples*sizeof(RGBvalue)));
	if (BWmap == NULL) 
      {
       fprintf(stderr, "makebwmap: no space for B&W mapping table.\n");
       return (0);
      }

	p = (RGBvalue *)(BWmap + 256);

	for (i = 0; i < 256; i++)
     {
      BWmap[i] = p;
      switch (bitspersample) 
        {
		case 1:
			*p++ = Map[i>>7];
			*p++ = Map[(i>>6)&1];
			*p++ = Map[(i>>5)&1];
			*p++ = Map[(i>>4)&1];
			*p++ = Map[(i>>3)&1];
			*p++ = Map[(i>>2)&1];
			*p++ = Map[(i>>1)&1];
			*p++ = Map[i&1];
			break;
		case 2:
			*p++ = Map[i>>6];
			*p++ = Map[(i>>4)&3];
			*p++ = Map[(i>>2)&3];
			*p++ = Map[i&3];
			break;
		case 4:
			*p++ = Map[i>>4];
			*p++ = Map[i&0xf];
			break;

		}  /* end switch (bitspersample)  */

	 }  /* end for */

    if (verbose) fprintf(stdout,"leaving makebwmap \n");
	return (1);
}

/***********************************/

#define	REPEAT8(op)	REPEAT4(op); REPEAT4(op)
#define	REPEAT4(op)	REPEAT2(op); REPEAT2(op)
#define	REPEAT2(op)	op; op

gtbw(bitspersample, w, cp, pp)
int  bitspersample, w;
register u_long       *cp;
register u_char           *pp;
{
	register RGBvalue c, *bw;
	register int x;
                  
    if (verbose) fprintf(stdout,"entering gtbw \n");

	switch (bitspersample) 
    {
	case 1:
		for (x = w; x >= 8; x -= 8) 
           {
			bw = BWmap[*pp++];
			REPEAT8(c = *bw++; *cp++ = rgbi(c, c, c));
           }
		break;
	case 2:
		for (x = w; x >= 4; x -= 4) 
           {
			bw = BWmap[*pp++];
			REPEAT4(c = *bw++; *cp++ = rgbi(c, c, c));
           }
		break;
	case 4:
		for (x = w; x >= 2; x -= 2) 
           {
			bw = BWmap[*pp++];
			REPEAT2(c = *bw++; *cp++ = rgbi(c, c, c));
           }
		break;

	}  /* end switch (bitspersample)  */

	bw = BWmap[*pp++];
	for (; x > 0; x--)
       {
		c = *bw++;
		*cp++ = rgbi(c, c, c);
       }
    if (verbose) fprintf(stdout,"leaving gtbw \n");
}

/***********************************/

write_1_row(y,d1,dest_box)
int                  y;
gpr_$pixel_value_t  *d1;
gpr_$window_t        dest_box;
{
   if (gray_flag)
     {
      if (verbose) fprintf(stdout,"in write_1_row, before calling put_linebw \n");
      put_linebw(y,d1,dest_box);
     }
   else
     {
	  if (four_flag)
        {
         if (verbose) fprintf(stdout,"in write_1_row, before calling put_line4 \n");
         put_line4(y,d1,dest_box);
        }
       else        
        {
         if (verbose) fprintf(stdout,"in write_1_row, before calling put_line8 \n");
         put_line8(y,d1,dest_box);
        }
     }                                  
}                      

/***********************************/

/*
 * Map a 24 bit scanline to 4 bits through the dither matrix.
 */
#define DMAP4(v,x,y)	(mod255[v]>dm16[x][y] ? div255[v] + 1 : div255[v])
put_line4(y,d1,dest_box)
int                  y;
gpr_$pixel_value_t  *d1;
gpr_$window_t        dest_box;
{
     register unsigned char *r, *g, *b;
     register int            i, col, row;

    r = (unsigned char *) d1 + 1;
    g = r+1;
    b = r+2;

    for ( row = y % 16, col = 0, i = 0; i < eWIDTH;
          i++, r+=4, g+=4, b+=4, col = ((col + 1) & 15) )
      dest_pixels[i] = DMAP4(map[0][*r], col, row)     +
                       DMAP4(map[1][*g], col, row) * 2 +
                       DMAP4(map[2][*b], col, row) * 4 ;

    gpr_$acquire_display( &status ) ;
    gpr_$write_pixels( dest_pixels, dest_box, &status ); 
    gpr_$release_display( &status );
}

/***********************************/

/*
 * Map a 24 bit scanline to 8 bits through the dither matrix.
 */
#define DMAP8(v,x,y)	(mod51[v]>dm16[x][y] ? div51[v] + 1 : div51[v])
put_line8(y,d1,dest_box)
int                  y;
gpr_$pixel_value_t  *d1;
gpr_$window_t        dest_box;
{
    register unsigned char *r, *g, *b;
    register int            i, col, row;

    r = (unsigned char *) d1 + 1;
    g = r+1;
    b = r+2;        

    for ( row = y % 16, col = 0, i = 0; i < eWIDTH;
          i++, r+=4, g+=4, b+=4, col = ((col + 1) & 15) )
      dest_pixels[i] = DMAP8(map[0][*r], col, row)      +
                       DMAP8(map[1][*g], col, row) * 6  +
                       DMAP8(map[2][*b], col, row) * 36 + COLMAP_OFFSET;

    gpr_$acquire_display( &status ) ;
    gpr_$write_pixels( dest_pixels, dest_box, &status ); 
    gpr_$release_display( &status );
}

/***********************************/
 
/*
 * Map a 24 bit scanline through to get dithered black and white
 */
#define DMAP(v,x,y)	(modN[v]>dm16[x][y] ? divN[v] + 1 : divN[v])
put_linebw(y,d1,dest_box)
int                   y;
gpr_$pixel_value_t   *d1;
gpr_$window_t         dest_box;
{
    register unsigned char *r, *g, *b;
    register int            i, col, row;
    int                     pixel;
    gpr_$pixel_value_t      B = 0, W = 1;

    r = (unsigned char *) d1 + 1;
    g = r+1;
    b = r+2;         

    if (bw_flag)
     {
      if (inverse_flag)
       {
        B = 1; W = 0;  /* Swap meaning of Black and White */
       }

     for ( row = y % 16, col = 0, i = 0; i < eWIDTH;
	       i++, r+=4, g+=4, b+=4, col = ((col + 1) & 15) )
       {
	    /* Convert to BW (uses YIQ/percentage xformation) */
	    pixel = (35*map[0][*r] + 55*map[1][*g] + 10*map[2][*b]) / 100;
	    if (pixel < 0) pixel += 256;
	    dest_pixels[i] = ((DMAP(pixel, col, row) > 0) ? W : B);
	   }
     }
    else
     {
      for ( row = y % 16, col = 0, i = 0; i < eWIDTH;
            i++, r+=4, g+=4, b+=4, col = ((col + 1) & 15) )
        {
	     /* Convert to BW (uses YIQ/percentage xformation) */
	     pixel = (35*map[0][*r] + 55*map[1][*g] + 10*map[2][*b]) / 100;
	     if (pixel < 0) pixel += 256;
	     if (inverse_flag)
          {
           pixel = ((~pixel) & 0xFF);		/* Swap meaning of Black and White */
          }
         if (four_flag)
          {
           dest_pixels[i] = DMAP(pixel, col, row);
          }
         else
          {
           dest_pixels[i] = DMAP(pixel, col, row) + COLMAP_OFFSET;
          }
        }
     }                          

    gpr_$acquire_display( &status ) ;
    gpr_$write_pixels( dest_pixels, dest_box, &status ); 
    gpr_$release_display( &status );
}

/***********************************/

/*
 * Initialize the 8 bit color map.  Use gamma corrected map.
 */
window_color_map()
{
    int                        i, j, k, l, planes, map_offset;
    unsigned int               value;
    unsigned int               intgam;
    float                      magicfact;
    static gpr_$color_vector_t colmap;
  
    if (verbose) fprintf(stdout,"entering window_color_map \n");
	if (gam_flag)
     {
	  for (i = 0; i < 256; i++)
	     {
          intgam = (int)(0.5 + 255 * pow( i / 255.0, 1.0/gam ));
          map[0][i] = intgam;
          map[1][i] = intgam;
          map[2][i] = intgam;
         }
     }
	else
     {
      /* Default is linear map */
	  for (i=0; i < 256; i++)
	     {
          map[0][i] = i;
          map[1][i] = i;
          map[2][i] = i;
         } 
     }

    if (four_flag && !bw_flag && !gray_flag)    
     {
      /* Set up the 4-bit color map entries. */
      for (i = 0; i < 8; i++) 
        {
         colmap[i]  = ((i%2)     * 255) << 16;
         colmap[i] |= (((i/2)%2) * 255) <<  8;
         colmap[i] |= (((i/4)%2) * 255)      ;
        }
 
      /* Save DM window colors */
      gpr_$acquire_display( &status );
      gpr_$inq_color_map( (int)0, 8, color_map,&status);
      gpr_$set_color_map( (int)0, 8, colmap,   &status);
      gpr_$release_display( &status );
 
      /* Compute tables */
      for ( i = 0; i < 256; i++ )
        {
         mod255[i] = i % 255;
         div255[i] = i / 255;
        }
 
      /*
       * Expand 4x4 dither pattern to 16x16.  4x4 leaves obvious patterning,
       * and doesn't give us full intensity range (only 17 sublevels,
       * we want at least 51).  Note that 8x8 would be sufficient, but
       * 16x16 is easier to build.
       * 
       * magicfact is 253/16 so that we get numbers in the matrix from 0 to
       * 253: mod 255 gives numbers in 0 to 254, we want some chance that the
       * number is greater than a matrix entry, so the highest matrix
       * entry must be 253.
       */
      magicfact = 253. / 16.;
      for ( i = 0; i < 4; i++ )
         for ( j = 0; j < 4; j++ )
            for ( k = 0; k < 4; k++ )
               for ( l = 0; l < 4; l++ )
                  dm16[4*k+i][4*l+j] = (int)(0.5 +  magic[i][j] * magicfact
                                                 + (magic[k][l] / 16.) * magicfact);
     }
    else if (four_flag && !bw_flag && gray_flag)
     {
         bwdithermap( 8, gam, colmap, divN, modN, dm16);

         for (i = 0; i < 8; i++) 
           {
            j = colmap[i];
    	    colmap[i]  = j << 16;
    	    colmap[i] |= j << 8;
    	    colmap[i] |= j;
           }
    
    	/* Save DM window colors */
    	gpr_$acquire_display( &status );
    	gpr_$inq_color_map((int)0, 8, color_map,&status);
    	gpr_$set_color_map((int)0, 8, colmap,   &status );
    	gpr_$release_display( &status );
     }
    else if (!bw_flag && !gray_flag)
     { 
      /* 
       * Set up the 8 bit color map entries.  We don't yet know the location
       * in the map at which each will reside, so init it to 0.
       */
      for (i = 0; i < 216; i++) 
        {
         colmap[i]  = ((i%6)      * 51) << 16;
         colmap[i] |= (((i/6)%6)  * 51) <<  8;
         colmap[i] |= (((i/36)%6) * 51)      ;
        }

      /* Offset from DM window colors */
      gpr_$acquire_display( &status );
      gpr_$set_color_map( (int) COLMAP_OFFSET, 256 - COLMAP_OFFSET, colmap, &status );
      gpr_$release_display( &status );

      /* Compute tables */
      for ( i = 0; i < 256; i++ )
       {
	    mod51[i] = i % 51;
	    div51[i] = i / 51;
       } 

       /*
        * Expand 4x4 dither pattern to 16x16.  4x4 leaves obvious patterning,
        * and doesn't give us full intensity range (only 17 sublevels,
        * we want at least 51).  Note that 8x8 would be sufficient, but
        * 16x16 is easier to build.
        * 
        * magicfact is 49/16 so that we get numbers in the matrix from 0 to
        * 49: mod 51 gives numbers in 0 to 50, we want some chance that the
        * number is greater than a matrix entry, so the highest matrix
        * entry must be 49.
        */
       magicfact = 49. / 16.;
       for ( i = 0; i < 4; i++ )
          for ( j = 0; j < 4; j++ )
             for ( k = 0; k < 4; k++ )
                for ( l = 0; l < 4; l++ )
		           dm16[4*k+i][4*l+j] = (int)(0.5 +  magic[i][j] * magicfact
                                                  + (magic[k][l] / 16.) * magicfact);
     }
    else if (!bw_flag)
     {
        bwdithermap( 240, gam, colmap, divN, modN, dm16);

    	for (i = 0; i < 240; i++)
         {
            j = colmap[i];
    	    colmap[i]  = j << 16;
    	    colmap[i] |= j << 8;
    	    colmap[i] |= j;
    	 }
    
    	/* Offset from DM window colors */
    	gpr_$acquire_display( &status );
    	gpr_$set_color_map( (int) COLMAP_OFFSET, 256 - COLMAP_OFFSET, colmap, &status );
    	gpr_$release_display( &status );
     }
    else
     {
        bwdithermap( 2, gam, colmap, divN, modN, dm16);
     }
    if (verbose) fprintf(stdout,"leaving window_color_map \n");
}

/***********************************/
Initialize()
{
  float expand, expand_X, expand_Y; 

  if (verbose) fprintf(stdout,"entering Initialize \n");

  hi_plane  = display_characteristics.n_planes - 1;
  if ( hi_plane > 0 ) 
    {
     fprintf(stdout,"\nThis apollo has %d planes (color display). \n",display_characteristics.n_planes);
    }
  else
    {
     fprintf(stdout,"\nThis apollo has %d plane (black and white display).  \n",display_characteristics.n_planes);
    }        

  eWIDTH  = WIDTH;
  eHEIGHT = HEIGHT;

  if ((full_screen) | 
      (display_characteristics.x_visible_size < WIDTH) |
      (display_characteristics.y_visible_size < HEIGHT) )   
    {
     expand_X = display_characteristics.x_visible_size / (float)WIDTH;
     expand_Y = display_characteristics.y_visible_size / (float)HEIGHT;
     /* expand gets the smaller value for a even X,Y expansion. */  
     expand = (expand_X > expand_Y) ? expand_Y : expand_X;       
     eWIDTH  = WIDTH  * expand;
     eHEIGHT = HEIGHT * expand;
     full_screen = true;
    }

  foo = (gpr_$pixel_value_t *) malloc(display_characteristics.x_visible_size * 4 *
                                      sizeof(gpr_$pixel_value_t));
  if (!foo) fatal("in Initialize: not enough memory for scan line buffer");

  dest_pixels = (gpr_$pixel_value_t *) malloc(display_characteristics.x_visible_size * 4 *
                                       sizeof(gpr_$pixel_value_t));
  if (!dest_pixels) fatal("in Initialize: not enough memory for destination line buffer");

  ix_inc_table = (int *) malloc(display_characteristics.x_visible_size * 4 *
                                sizeof(int)); 
  if (!ix_inc_table) fatal("in Initialize: not enough memory for scan line table");

  pad_window.top    = 0;
  pad_window.left   = 0;
  pad_window.width  = eWIDTH;
  pad_window.height = eHEIGHT; 

  pad_$create_window((char *)0,(short)0,pad_$transcript,(short)1,pad_window,&unit,&status);
  check("in Initialiaze after calling pad_$create_window");

  pad_$set_full_window(unit,(short) 1,&pad_window,&status);
  pad_$set_auto_close(unit, (short) 1, true, &status );
  pad_$set_border (unit,(short) 1, false, &status);
  pad_$set_scale (unit,(short) 1,(short) 1, &status);

  if(display_characteristics.n_planes == 4) four_flag = true;
  if(display_characteristics.n_planes == 1) bw_flag   = true; 
  if(bw_flag)                               gray_flag = true;      

  /*init_size.x_size = display_characteristics.x_visible_size;   /* x dimension of visible screen area in pixels */ 
  /*init_size.y_size = display_characteristics.y_visible_size;   /* y dimension of visible screen area in pixels */ 
  init_size.x_size = 4096;
  init_size.y_size = 4096;
  gpr_$init(mode,unit, init_size, hi_plane, &display_bitmap_desc, &status);
  check("in Initialize after calling gpr_$init");
  if (verbose) fprintf(stdout,"leaving Initialize \n");
}

/***********************************/
Close() 
{
  int i=0;
 
  gpr_$terminate (false, &status);
  while(status.fail && i<10) 
   {
    i++;
    gpr_$terminate (false, &status);
    printf("attempting to terminate i= %d\n",i);
   }
  free(ix_inc_table);
  free(dest_pixels);
  free(foo);
}
/***********************************/
KbEnable() 
{
  gpr_$keyset_t    keys;
  gpr_$event_t     ev_type;
  gpr_$position_t  ev_pos;
  unsigned char    ev_char;
  short int        KBD_$CR=0x96;

  lib_$init_set(keys, (short)256);
  lib_$add_to_set(keys, (short)256, ' ');
  lib_$add_to_set(keys, (short)256, 'q');
  lib_$add_to_set(keys, (short)256, 'Q');
  lib_$add_to_set(keys, (short)256, KBD_$CR);
  gpr_$enable_input (gpr_$keystroke, keys, &status);
  check("in KbEnable after calling gpr_$enable_input");
  gpr_$event_wait (&ev_type, &ev_char, &ev_pos, &status);
}

/***********************************/
void inq_window_size()
{
   gpr_$inq_bitmap_position  (display_bitmap_desc, &eORIGIN , &status);
   check("in inq_window_size after calling gpr_$inq_bitmap_position");
   gpr_$inq_bitmap_dimensions(display_bitmap_desc, &eSIZE   , &hi_plane , &status);
   check("in inq_window_size after calling gpr_$inq_bitmap_dimensions");
   eWIDTH  = eSIZE.x_size ;
   eHEIGHT = eSIZE.y_size ;
}

/***********************************/
Resize()
{
    gpr_$offset_t    textsize ;
    short int        font_id;
    int              ix,iy_WIDTH,ex,ey;
    gpr_$window_t    dest_box;

    if (verbose) fprintf(stdout,"entering Resize \n");

    /* if it's a big image, this'll take a while.  mention it */
    if (eWIDTH*eHEIGHT>(300*300)) 
        {
         gpr_$acquire_display(&status);
         gpr_$load_font_file("f7x13",(short) 5, &font_id, &status);
         gpr_$set_text_font(font_id, &status);    
         gpr_$inq_text_extent (wait_string, wait_string_len, &textsize, &status);
         gpr_$move( (gpr_$coordinate_t) (eWIDTH-textsize.x_size)/2, (gpr_$coordinate_t) (eHEIGHT/2), &status); 
         gpr_$text(wait_string,wait_string_len,&status);
         gpr_$release_display(&status);
        }

    dest_box.window_base.x_coord = 0 ;
    dest_box.window_size.x_size  = eWIDTH;
    dest_box.window_size.y_size  = 1;

    for (ex = 0; ex < eWIDTH ; ex++) 
       {
        ix_inc_table[ex] = (WIDTH * ex) / eWIDTH; 
       }

    for (ey=0;  ey < eHEIGHT;  ey++)
       {
        iy_WIDTH  = ((HEIGHT * ey) / eHEIGHT) * WIDTH ; 
        for (ex=0;  ex < eWIDTH;  ex++) 
           {
            foo[ex]=  raster[ix_inc_table[ex] + iy_WIDTH];
           }
        dest_box.window_base.y_coord = eHEIGHT - ey;
        write_1_row(ey,foo,dest_box);
       }
    if (verbose) fprintf(stdout,"leaving Resize \n");
}

/***********************************/
void redraw() 
{       
  if (verbose) fprintf(stdout,"entering redraw \n");
  inq_window_size();    
  Resize();
  if (verbose) fprintf(stdout,"leaving redraw \n");
}
/***********************************/
print_display_info()
{
(void) fprintf(stdout,"\nDisplay Characteristics \n");
(void) fprintf(stdout,"  controller_type         =%d  /* type of graphics controller */ \n",display_characteristics.controller_type  );
(void) fprintf(stdout,"  accelerator_type        =%d  /* type of graphics accelerator */  \n",display_characteristics.accelerator_type  );
(void) fprintf(stdout,"  x_window_origin         =%d  /* x origin of window screen area in pixels */  \n",display_characteristics.x_window_origin );
(void) fprintf(stdout,"  y_window_origin         =%d  /* y origin of window screen area in pixels */  \n",display_characteristics.y_window_origin );
(void) fprintf(stdout,"  x_window_size           =%d  /* x dimension of window screen area in pixels */  \n",display_characteristics.x_window_size  );
(void) fprintf(stdout,"  y_window_size           =%d  /* y dimension of window screen area in pixels */  \n",display_characteristics.y_window_size  );
(void) fprintf(stdout,"  x_visible_size          =%d  /* x dimension of visible screen area in pixels */  \n",display_characteristics.x_visible_size );
(void) fprintf(stdout,"  y_visible_size          =%d  /* y dimension of visible screen area in pixels */  \n",display_characteristics.y_visible_size );
(void) fprintf(stdout,"  x_extension_size        =%d  /* x dimension of maximum extended bitmap size in pixels */  \n",display_characteristics.x_extension_size  );
(void) fprintf(stdout,"  y_extension_size        =%d  /* y dimension of maximum extended bitmap size in pixels */  \n",display_characteristics.y_extension_size  );
(void) fprintf(stdout,"  x_total_size            =%d  /* x dimension of total buffer area in pixels */  \n",display_characteristics.x_total_size );
(void) fprintf(stdout,"  y_total_size            =%d  /* y dimension of total buffer area in pixels */  \n",display_characteristics.y_total_size );
(void) fprintf(stdout,"  x_pixels_per_cm         =%d  /* number of pixels in x dimension per centimeter */  \n",display_characteristics.x_pixels_per_cm );
(void) fprintf(stdout,"  y_pixels_per_cm         =%d  /* number of pixels in y dimension per centimeter */  \n",display_characteristics.y_pixels_per_cm );
(void) fprintf(stdout,"  n_planes                =%d  /* number of planes available */  \n",display_characteristics.n_planes  );
(void) fprintf(stdout,"  n_buffers               =%d  /* number of display buffers available */  \n",display_characteristics.n_buffers );
(void) fprintf(stdout,"  delta_x_per_buffer      =%d  /* relative displacement of buffers in x */  \n",display_characteristics.delta_x_per_buffer  );
(void) fprintf(stdout,"  delta_y_per_buffer      =%d  /* relative displacement of buffers in y */  \n",display_characteristics.delta_y_per_buffer  );
(void) fprintf(stdout,"  delta_planes_per_buffer =%d  /* relative displacement of buffers in depth */  \n",display_characteristics.delta_planes_per_buffer );
(void) fprintf(stdout,"  mem_overlaps            =%d  /* set of overlaps among classes of buffer memory */  \n",display_characteristics.mem_overlaps );
(void) fprintf(stdout,"  x_zoom_max              =%d  /* maximum pixel-replication zoom factor for x */  \n",display_characteristics.x_zoom_max  );
(void) fprintf(stdout,"  y_zoom_max              =%d  /* maximum pixel-replication zoom factor for y */  \n",display_characteristics.y_zoom_max  );
(void) fprintf(stdout,"  video_refresh_rate      =%d  /* refresh rate in hz */  \n",display_characteristics.video_refresh_rate  );
(void) fprintf(stdout,"  n_primaries             =%d  /* number of primary colors (1 -> monochrome; 3 -> color */  \n",display_characteristics.n_primaries );
(void) fprintf(stdout,"  lut_width_per_primary   =%d  /* number of bits in possible shortensity values per primary */  \n",display_characteristics.lut_width_per_primary  );
(void) fprintf(stdout,"  avail_formats           =%d  /* set of available shorteractive/imaging formats */  \n",display_characteristics.avail_formats  );
(void) fprintf(stdout,"  avail_access            =%d  /* set of available pixel sizes for direct access */  \n",display_characteristics.avail_access  );
(void) fprintf(stdout,"  access_address_space    =%d  /* number of 1kb pages of address space available for direct access */  \n",display_characteristics.access_address_space  );
(void) fprintf(stdout,"  invert                  =%d  /* INVert implemention */ \n",display_characteristics.invert );
(void) fprintf(stdout,"  num_lookup_tables       =%d  /* Number of color lookup tables */ \n",display_characteristics.num_lookup_tables  );
(void) fprintf(stdout,"  rgb_color               =%d  /* Modes for separate values for RGB */ \n\n",display_characteristics.rgb_color );
}
/***********************************/
SaveImage(source_bitmap_desc,file_name)
gpr_$bitmap_desc_t  source_bitmap_desc;
char                *file_name;
{
    status_$t                       status;
    gpr_$bmf_group_header_array_t   header; 
    gpr_$color_vector_t             color_map;
    gpr_$version_t                  version;
    short int                       groups;
    gpr_$attribute_desc_t           attribs;  
    gpr_$window_t                   source_bitmap;     
    gpr_$bitmap_desc_t              disk_bitmap_desc , curs_pat_desc;
    gpr_$offset_t                   source_bitmap_size;    
    gpr_$rgb_plane_t                hi_plane;
    gpr_$position_t                 disk_bitmap_origin , source_bitmap_origin , curs_position, curs_origin;
    gpr_$raster_op_array_t          curs_raster_op;
    boolean                         disk_bmf_created , curs_active;

    gpr_$acquire_display (&status);

    gpr_$inq_cursor (&curs_pat_desc, curs_raster_op, &curs_active, &curs_position, &curs_origin,&status);
    if(curs_active) gpr_$set_cursor_active(false,&status); /* disable the cursor */

/* set the current bitmap the source bitmap */
    gpr_$set_bitmap(source_bitmap_desc,&status);
    check("in SaveImage");

/* get the size and the highest plane id of the source bitmap */
    gpr_$inq_bitmap_dimensions(source_bitmap_desc,&source_bitmap_size,&hi_plane,&status);  
    check("in SaveImage");

    gpr_$inq_color_map ((gpr_$pixel_value_t) 0,(short) 256,color_map,&status);   
    check("in SaveImage");

    header[0].n_sects = hi_plane +1;  /* # of sections in a group */
    header[0].pixel_size = 1;         /* # of bits per pixel in each section of a group */
    header[0].allocated_size  = 0;
    header[0].bytes_per_line  = 0;    /* # of bytes in one row of a bitmap. If =0, GPR takes care of it automatically. */
    header[0].bytes_per_sect  = 0;
    header[0].storage_offset  = 0;
    groups = (short) 1;
    version.gpr_$major =(short) 1;
    version.gpr_$minor =(short) 1;
/*
    source_bitmap.window_base.x_coord = source_bitmap_origin.x_coord;
    source_bitmap.window_base.y_coord = source_bitmap_origin.y_coord;
*/
    source_bitmap.window_base.x_coord = 0;
    source_bitmap.window_base.y_coord = 0;
    source_bitmap.window_size.x_size  = source_bitmap_size.x_size;
    source_bitmap.window_size.y_size  = source_bitmap_size.y_size; 
     
    gpr_$allocate_attribute_block(&attribs, &status);  

/* save the whole thing to disk */
    gpr_$open_bitmap_file(gpr_$create, file_name, (short)strlen(file_name),
        &version, &source_bitmap_size, &groups,header, attribs, &disk_bitmap_desc, &disk_bmf_created, &status);  
    check("in SaveImage");

/* set the current bitmap the disk bitmap */
    gpr_$set_bitmap(disk_bitmap_desc,&status); 
    check("in SaveImage");
  
/* set the color map in the disk bitmap file to the color map the application used. You can change the color map here */
    gpr_$set_bitmap_file_color_map (disk_bitmap_desc,(short) 0,(short) 256,color_map,&status);
    check("in SaveImage");

    disk_bitmap_origin.x_coord = 0;
    disk_bitmap_origin.y_coord = 0;

/* now just move the pixels from the source bitmap to the disk bitmap file */
    gpr_$pixel_blt(source_bitmap_desc,source_bitmap,disk_bitmap_origin,&status);
    check("in SaveImage");

/* set current bitmap back to the source bitmap */
    gpr_$set_bitmap(source_bitmap_desc, &status);    
    check("in SaveImage");

/* the disk bitmap file can be unlocked and available for print */
    gpr_$deallocate_bitmap(disk_bitmap_desc,&status);
    check("in SaveImage");

    gpr_$deallocate_attribute_block(attribs, &status);  
    check("in SaveImage");

    if ( hi_plane > 0 ) 
     {
      fprintf(stdout,"The bitmap file %s has been created. The source bitmap has %d planes. \n",file_name,(hi_plane + 1));
     }
    else
     {
      fprintf(stdout,"The bitmap file %s has been created. The source bitmap has %d plane.  \n",file_name,(hi_plane + 1));
     }

    if(curs_active) gpr_$set_cursor_active(true,&status);

    gpr_$release_display(&status) ;
}
