/* gpr2tiff.c - convert an apollo GPR bitmap to a TIFF file
 *
 *                 Version 1.0   07/09/1990
 *
 *  This file contains system-specific graphics routines.
 *  These routines use Apollo Graphics Primitive Resource (GPR) calls.
 *
 *  Authors:   Roque D. Oliveira      oliveria@caen.engin.umich.edu
 *             Amadi Nwankpa             amadi@caen.engin.umich.edu
 *
 *  If you make any improvements/bug fixes to this program, please send
 *  them to the authors.
 */

/*
 * Copyright (C) 1990 by the author(s).
 * All rights reserved.
 *
 * Permission is granted to copy, modify and distribute this program
 * for any purpose and without charge, provided this copyright notice
 * is included in the copy.
 * This program is distributed on an as-is basis. There will be
 * ABSOLUTELY NO WARRANTY for any part of this software to work
 * correctly. In no case will the author(s) be liable to you for any
 * lost revenue or profits or other special, indirect and consequential
 * damages caused by the usage of this program.
 */


#include <stdio.h>
#include <string.h>
#include <apollo/base.h>
#include <apollo/gpr.h>
#include <apollo/pfm.h>
#include <apollo/error.h>
#include "tiffio.h"
#define MAX_XRES    2048
#define MAX_YRES    2048
#define MAX_PLANES     8
gpr_$color_vector_t           color_vect;
gpr_$bitmap_desc_t            bitmap_desc, disk_bitmap;
gpr_$offset_t                 disk_bm_size;
gpr_$bmf_group_header_array_t header; 
status_$t                     status;
short                         num_colors;

int   xres, yres, num_planes;
char  *progname;				/* the name of this program */

int GetPixel();

typedef	struct { unsigned char	r, g, b, a;} RGBAPixelType;
RGBAPixelType *picio_buffer;

#define	streq(a,b)	(strcmp(a,b) == 0)

main(argc, argv)
int  argc;
char      *argv[];
{                        
	int row, linebytes;
	register TIFF *out ;
	short bitspersample = 8, samplesperpixel = 3;
	int rowsperstrip = -1;
    int compression = COMPRESSION_LZW;
    int config = PLANARCONFIG_CONTIG;
	register u_char *cp;
	u_char *buf;   
    int x, y, pix, lcolormap;
    extern unsigned char get_pixel(); 

    progname = strrchr(argv[0], '/');
    if (progname)
	   progname++;
    else
	   progname = argv[0];

	argc--, argv++;
	if (argc < 2)
		usage();
	for (; argc > 2 && argv[0][0] == '-'; argc--, argv++) {
		if (streq(argv[0], "-picio")) {
			compression = COMPRESSION_PICIO;
			continue;
		}
		if (streq(argv[0], "-none")) {
			compression = COMPRESSION_NONE;
			continue;
		}
		if (streq(argv[0], "-packbits")) {
			compression = COMPRESSION_PACKBITS;
			continue;
		}
		if (streq(argv[0], "-lzw")) {           /* default */
			compression = COMPRESSION_LZW;
			continue;
		}
		if (streq(argv[0], "-thunderscan")) {
			compression = COMPRESSION_THUNDERSCAN;
			continue;
		}
		if (streq(argv[0], "-rowsperstrip")) {
			argc--, argv++;
			rowsperstrip = atoi(argv[0]);
			continue;
		}
		if (streq(argv[0], "-8bit")) {
			bitspersample = 8;
			continue;
		}
		usage();
	}

    if (init_display() != 0 )
     {
      fprintf(stderr,"Error in initializing display.");
      exit(1);
     }

    if (open_bitmap(argv[0], &num_planes, &xres, &yres) != 0)
     {
      fprintf(stderr,"Error in opening bitmap file.");
      exit(1);
     }

    gpr_$set_bitmap(disk_bitmap,&status);  

    fprintf(stderr,"(%d,%d) bitmap read with %d planes\n",xres,yres,num_planes);    
    gpr_$inq_color_map((gpr_$pixel_value_t) 0, num_colors, color_vect, &status);   
    if(status.all == gpr_$no_color_map_in_file) 
        lcolormap=0;
    else
        lcolormap=1;  

    if (lcolormap == 1) fprintf(stderr,"A %d-color colormap read \n",num_colors);    
      
	out = TIFFOpen(argv[1], "w");
	if (out == NULL) exit(-2);

	TIFFSetField(out, TIFFTAG_IMAGEWIDTH , xres);
	TIFFSetField(out, TIFFTAG_IMAGELENGTH, yres);
	TIFFSetField(out, TIFFTAG_BITSPERSAMPLE , bitspersample);
	TIFFSetField(out, TIFFTAG_SAMPLESPERPIXEL, samplesperpixel);
	TIFFSetField(out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);
	TIFFSetField(out, TIFFTAG_PLANARCONFIG, config); 
	TIFFSetField(out, TIFFTAG_COMPRESSION, compression);
	TIFFSetField(out, TIFFTAG_MINSAMPLEVALUE, (short) 0);
	TIFFSetField(out, TIFFTAG_MAXSAMPLEVALUE, (short) 255);
	TIFFSetField(out, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
	TIFFSetField(out, TIFFTAG_SOFTWARE, "gpr2tiff");
	if (rowsperstrip <= 0) rowsperstrip = (8*1024)/TIFFScanlineSize(out);
	TIFFSetField(out, TIFFTAG_ROWSPERSTRIP, rowsperstrip == 0 ? 1 : rowsperstrip);

	buf = (u_char *) malloc(TIFFScanlineSize(out));

	if (buf == NULL) 
      {
		fprintf(stderr, "No space for scanline buffer\n");
		exit(-1);
      }
	/* XXX what about response curves? */
	/*TIFFSetField(out, TIFFTAG_MATTEING, true);  */

     for(y=0 ; y<yres ; y++)
     {  
      cp = buf;
      for(x=0 ; x<xres ; x++)
        {  
         pix   = (int) get_pixel(x,y); 
         *cp++ = (color_vect[pix] / 65536) & 0xFF;
         *cp++ = (color_vect[pix] / 256)   & 0xFF;
         *cp++ = (color_vect[pix] )        & 0xFF;
        }  
   	 if (TIFFWriteScanline(out, buf,y, 0) < 0) break;
	}

	free((char *) picio_buffer);
	(void) TIFFClose(out);
}

usage()
{
	fprintf(stderr, "usage: %s [options] input output\n",progname);
	fprintf(stderr, "where options are:\n");
	fprintf(stderr,
	    " -picio\t\tcompress output with picio encoding\n");
	fprintf(stderr,
	    " -lzw\t\tcompress output with LZW encoding (default)\n");
	fprintf(stderr,
	    " -packbits\tcompress output with packbits encoding\n");
	fprintf(stderr,
	    " -thunderscan\tcompress output with ThunderScan encoding\n");
	fprintf(stderr,
	    " -none\t\tuse no compression algorithm on output\n");
	fprintf(stderr, "\n");
	fprintf(stderr,
	    " -rowsperstrip #\tmake each strip have no more than # rows\n");
	exit(-1);
}

/****************************************************************************
 *    check the status variable returned from GPR calls
 ****************************************************************************/

int bad_status()

    {
    if (status.all == status_$ok )
        return (0);
    else
       {
       fprintf(stderr,"Got stcode=%x\n",status);
       fprintf(stderr,"Error - terminating graphics...");
       error_$print(status);
       return (1);
       }
    }

/****************************************************************************
 *    read one pixel of bitmap
 ****************************************************************************/

unsigned char get_pixel(x,y)
int x, y;
    {
    gpr_$window_t   src;
    gpr_$pixel_value_t        dest[1];

    src.window_base.x_coord = x;
    src.window_base.y_coord = y;
    src.window_size.x_size  = 1;
    src.window_size.y_size  = 1;

    gpr_$read_pixels(src, dest, &status);
    bad_status();
    return(dest[0]);
    }

/****************************************************************************
 *    read row of one bit-map plane
 ****************************************************************************/

void get_row(row, plane, x0, dx, array)

int             dx, row, plane, x0;
unsigned char   array[];

    {
    int             x;
    unsigned char   pixel, mask;
    unsigned short  bit_count, byte_count, count;

    for (count=0 ; count<dx ; count++)
        {
        x = x0 + count;
        pixel = get_pixel(x, row);

        byte_count = count >> 3;
        bit_count  = count % 8;
        mask = 128 >> bit_count;

        if (pixel & (1<<plane))   /* pixel is set on this plane   */
           array[byte_count] |= mask;
        else
           array[byte_count] &= ~mask;
        }

    return;
    }

/****************************************************************************
 *    open bitmap file
 ****************************************************************************/

int open_bitmap(file_name, num_plane, xres, yres)

char    *file_name;
int     *num_plane, *xres, *yres;

    {
    boolean             created;
    gpr_$color_vector_t color_vec;
    gpr_$version_t      version;
    short               groups;

    gpr_$attribute_desc_t   attribs;

    gpr_$allocate_attribute_block(&attribs, &status);
    if (bad_status()) return(1);

    disk_bm_size.x_size = MAX_XRES;
    disk_bm_size.y_size = MAX_YRES;
    groups = 1;
    


    gpr_$open_bitmap_file(gpr_$readonly, file_name, (short)strlen(file_name), &version,
       &disk_bm_size, &groups, header, attribs, &disk_bitmap, &created, &status);
    if (bad_status()) return(1);

    *xres = disk_bm_size.x_size;
    *yres = disk_bm_size.y_size;

    *num_plane = header[0].n_sects;
    num_colors = 1 << *num_plane;

    return(bad_status());
    }


/****************************************************************************
 * initialize display
 ****************************************************************************/

int init_display()
    {
    gpr_$display_mode_t disp_mode = gpr_$no_display;
    gpr_$offset_t       size;
    gpr_$plane_t        hi_plane;

/*
 *  initialize gpr in no-display mode
 */

    hi_plane    = MAX_PLANES - 1;
    size.x_size = MAX_XRES;
    size.y_size = MAX_YRES;

    gpr_$init(disp_mode, (short) 1, size, hi_plane, &bitmap_desc, &status);

    return(bad_status());
    }

