/*	$NetBSD: bus.h,v 1.12 1997/10/01 08:25:15 fvdl Exp $	*/

/*-
 * SPDX-License-Identifier: (BSD-2-Clause AND BSD-4-Clause)
 *
 * Copyright (c) 1996, 1997 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Jason R. Thorpe of the Numerical Aerospace Simulation Facility,
 * NASA Ames Research Center.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*-
 * Copyright (c) 1996 Charles M. Hannum.  All rights reserved.
 * Copyright (c) 1996 Christopher G. Demetriou.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Christopher G. Demetriou
 *	for the NetBSD Project.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _BUS_DMA_H_
#define _BUS_DMA_H_

#ifdef _KERNEL
#include <sys/_bus_dma.h>
#endif

/*
 * Machine independent interface for mapping physical addresses to peripheral
 * bus 'physical' addresses, and assisting with DMA operations.
 *
 * XXX This file is always included from <machine/bus_dma.h> and should not
 *     (yet) be included directly.
 */

/*
 * Flags used in various bus DMA methods.
 */
#define	BUS_DMA_WAITOK		0x00	/* safe to sleep (pseudo-flag) */
#define	BUS_DMA_NOWAIT		0x01	/* not safe to sleep */
#define	BUS_DMA_ALLOCNOW	0x02	/* perform resource allocation now */
#define	BUS_DMA_COHERENT	0x04	/* hint: map memory in a coherent way */
#define	BUS_DMA_ZERO		0x08	/* allocate zero'ed memory */
#define	BUS_DMA_BUS1		0x10	/* placeholders for bus functions... */
#define	BUS_DMA_BUS2		0x20
#define	BUS_DMA_BUS3		0x40
#define	BUS_DMA_BUS4		0x80

/*
 * The following two flags are non-standard or specific to only certain
 * architectures
 */
#define	BUS_DMA_NOWRITE		0x100
#define	BUS_DMA_NOCACHE		0x200

/*
 * The following flag is a DMA tag hint that the page offset of the
 * loaded kernel virtual address must be preserved in the first
 * physical segment address, when the KVA is loaded into DMA.
 */
#define	BUS_DMA_KEEP_PG_OFFSET	0x400

#define	BUS_DMA_LOAD_MBUF	0x800

/* Forwards needed by prototypes below. */
union ccb;
struct bio;
struct crypto_buffer;
struct cryptop;
struct mbuf;
struct memdesc;
struct pmap;
struct uio;

/*
 * Operations performed by bus_dmamap_sync().
 */
#define	BUS_DMASYNC_PREREAD	1
#define	BUS_DMASYNC_POSTREAD	2
#define	BUS_DMASYNC_PREWRITE	4
#define	BUS_DMASYNC_POSTWRITE	8

/*
 *	bus_dma_segment_t
 *
 *	Describes a single contiguous DMA transaction.  Values
 *	are suitable for programming into DMA registers.
 */
typedef struct bus_dma_segment {
	bus_addr_t	ds_addr;	/* DMA address */
	bus_size_t	ds_len;		/* length of transfer */
} bus_dma_segment_t;

#ifdef _KERNEL
/*
 * A function that returns 1 if the address cannot be accessed by
 * a device and 0 if it can be.
 */
typedef int bus_dma_filter_t(void *, bus_addr_t);

/*
 * Generic helper function for manipulating mutexes.
 */
void busdma_lock_mutex(void *arg, bus_dma_lock_op_t op);

/*
 * Internal helper function used by tags that do not defer loads.
 */
void _busdma_dflt_lock(void *arg, bus_dma_lock_op_t op);

/*
 * Allocate a device specific dma_tag encapsulating the constraints of
 * the parent tag in addition to other restrictions specified:
 *
 *	alignment:	Alignment for segments.
 *	boundary:	Boundary that segments cannot cross.
 *	lowaddr:	Low restricted address that cannot appear in a mapping.
 *	highaddr:	High restricted address that cannot appear in a mapping.
 *	filtfunc:	An optional function to further test if an address
 *			within the range of lowaddr and highaddr cannot appear
 *			in a mapping.
 *	filtfuncarg:	An argument that will be passed to filtfunc in addition
 *			to the address to test.
 *	maxsize:	Maximum mapping size supported by this tag.
 *	nsegments:	Number of discontinuities allowed in maps.
 *	maxsegsz:	Maximum size of a segment in the map.
 *	flags:		Bus DMA flags.
 *	lockfunc:	An optional function to handle driver-defined lock
 *			operations.
 *	lockfuncarg:	An argument that will be passed to lockfunc in addition
 *			to the lock operation.
 *	dmat:		A pointer to set to a valid dma tag should the return
 *			value of this function indicate success.
 */
/* XXX Should probably allow specification of alignment */
int bus_dma_tag_create(bus_dma_tag_t parent, bus_size_t alignment,
		       bus_addr_t boundary, bus_addr_t lowaddr,
		       bus_addr_t highaddr, bus_dma_filter_t *filtfunc,
		       void *filtfuncarg, bus_size_t maxsize, int nsegments,
		       bus_size_t maxsegsz, int flags, bus_dma_lock_t *lockfunc,
		       void *lockfuncarg, bus_dma_tag_t *dmat);

/*
 * Functions for creating and cloning tags via a template,
 *
 * bus_dma_template_t is made avaialble publicly so it can be allocated
 * from the caller stack.  Its contents should be considered private, and
 * should only be accessed via the documented APIs and macros
 */
typedef struct {
	bus_dma_tag_t		parent;
	bus_size_t		alignment;
	bus_addr_t		boundary;
	bus_addr_t		lowaddr;
	bus_addr_t		highaddr;
	bus_size_t		maxsize;
	int			nsegments;
	bus_size_t		maxsegsize;
	int			flags;
	bus_dma_lock_t		*lockfunc;
	void			*lockfuncarg;
	const char		*name;
} bus_dma_template_t;

/*
 * These enum values should not be re-ordered.  BD_PARAM_INVALID is an
 * invalid key and will trigger a panic.
 */
typedef enum {
	BD_PARAM_INVALID	= 0,
	BD_PARAM_PARENT		= 1,
	BD_PARAM_ALIGNMENT	= 2,
	BD_PARAM_BOUNDARY	= 3,
	BD_PARAM_LOWADDR	= 4,
	BD_PARAM_HIGHADDR	= 5,
	BD_PARAM_MAXSIZE	= 6,
	BD_PARAM_NSEGMENTS	= 7,
	BD_PARAM_MAXSEGSIZE	= 8,
	BD_PARAM_FLAGS		= 9,
	BD_PARAM_LOCKFUNC	= 10,
	BD_PARAM_LOCKFUNCARG	= 11,
	BD_PARAM_NAME		= 12
} bus_dma_param_key_t;

/* These contents should also be considered private */
typedef struct {
	bus_dma_param_key_t	key;
	union {
		void *ptr;
		vm_paddr_t pa;
		uintmax_t num;
	};
} bus_dma_param_t;

#define BD_PARENT(val)		{ BD_PARAM_PARENT, .ptr = val }
#define BD_ALIGNMENT(val)	{ BD_PARAM_ALIGNMENT, .num = val }
#define BD_BOUNDARY(val)	{ BD_PARAM_BOUNDARY, .num = val }
#define BD_LOWADDR(val)		{ BD_PARAM_LOWADDR, .pa = val }
#define BD_HIGHADDR(val)	{ BD_PARAM_HIGHADDR, .pa = val }
#define BD_MAXSIZE(val)		{ BD_PARAM_MAXSIZE, .num = val }
#define BD_NSEGMENTS(val)	{ BD_PARAM_NSEGMENTS, .num = val }
#define BD_MAXSEGSIZE(val)	{ BD_PARAM_MAXSEGSIZE, .num = val }
#define BD_FLAGS(val)		{ BD_PARAM_FLAGS, .num = val }
#define BD_LOCKFUNC(val)	{ BD_PARAM_LOCKFUNC, .ptr = val }
#define BD_LOCKFUNCARG(val)	{ BD_PARAM_LOCKFUNCARG, .ptr = val }
#define BD_NAME(val)		{ BD_PARAM_NAME, .ptr = val }

#define BUS_DMA_TEMPLATE_FILL(t, kv...) \
do {					\
	bus_dma_param_t pm[] = { kv };	\
	bus_dma_template_fill(t, pm, howmany(sizeof(pm), sizeof(pm[0]))); \
} while (0)

void bus_dma_template_init(bus_dma_template_t *t, bus_dma_tag_t parent);
int bus_dma_template_tag(bus_dma_template_t *t, bus_dma_tag_t *dmat);
void bus_dma_template_clone(bus_dma_template_t *t, bus_dma_tag_t dmat);
void bus_dma_template_fill(bus_dma_template_t *t, bus_dma_param_t *kv,
    u_int count);

/*
 * Set the memory domain to be used for allocations.
 *
 * Automatic for PCI devices.  Must be set prior to creating maps or
 * allocating memory.
 */
int bus_dma_tag_set_domain(bus_dma_tag_t dmat, int domain);

int bus_dma_tag_destroy(bus_dma_tag_t dmat);

/*
 * A function that processes a successfully loaded dma map or an error
 * from a delayed load map.
 */
typedef void bus_dmamap_callback_t(void *, bus_dma_segment_t *, int, int);

/*
 * Like bus_dmamap_callback but includes map size in bytes.  This is
 * defined as a separate interface to maintain compatibility for users
 * of bus_dmamap_callback_t--at some point these interfaces should be merged.
 */
typedef void bus_dmamap_callback2_t(void *, bus_dma_segment_t *, int, bus_size_t, int);

/*
 * Map the buffer buf into bus space using the dmamap map.
 */
int bus_dmamap_load(bus_dma_tag_t dmat, bus_dmamap_t map, void *buf,
		    bus_size_t buflen, bus_dmamap_callback_t *callback,
		    void *callback_arg, int flags);

/*
 * Like bus_dmamap_load but for mbufs.  Note the use of the
 * bus_dmamap_callback2_t interface.
 */
int bus_dmamap_load_mbuf(bus_dma_tag_t dmat, bus_dmamap_t map,
			 struct mbuf *mbuf,
			 bus_dmamap_callback2_t *callback, void *callback_arg,
			 int flags);

int bus_dmamap_load_mbuf_sg(bus_dma_tag_t dmat, bus_dmamap_t map,
			    struct mbuf *mbuf, bus_dma_segment_t *segs,
			    int *nsegs, int flags);

/*
 * Like bus_dmamap_load but for uios.  Note the use of the
 * bus_dmamap_callback2_t interface.
 */
int bus_dmamap_load_uio(bus_dma_tag_t dmat, bus_dmamap_t map,
			struct uio *ui,
			bus_dmamap_callback2_t *callback, void *callback_arg,
			int flags);

/*
 * Like bus_dmamap_load but for cam control blocks.
 */
int bus_dmamap_load_ccb(bus_dma_tag_t dmat, bus_dmamap_t map, union ccb *ccb,
			bus_dmamap_callback_t *callback, void *callback_arg,
			int flags);

/*
 * Like bus_dmamap_load but for bios.
 */
int bus_dmamap_load_bio(bus_dma_tag_t dmat, bus_dmamap_t map, struct bio *bio,
			bus_dmamap_callback_t *callback, void *callback_arg,
			int flags);

/*
 * Like bus_dmamap_load but for crypto ops.
 */
int bus_dmamap_load_crp(bus_dma_tag_t dmat, bus_dmamap_t map,
			struct cryptop *crp, bus_dmamap_callback_t *callback,
			void *callback_arg, int flags);
int bus_dmamap_load_crp_buffer(bus_dma_tag_t dmat, bus_dmamap_t map,
			       struct crypto_buffer *cb,
			       bus_dmamap_callback_t *callback,
			       void *callback_arg, int flags);

/*
 * Loads any memory descriptor.
 */
int bus_dmamap_load_mem(bus_dma_tag_t dmat, bus_dmamap_t map,
			struct memdesc *mem, bus_dmamap_callback_t *callback,
			void *callback_arg, int flags);

/*
 * Placeholder for use by busdma implementations which do not benefit
 * from optimized procedure to load an array of vm_page_t.  Falls back
 * to do _bus_dmamap_load_phys() in loop.
 */
int bus_dmamap_load_ma_triv(bus_dma_tag_t dmat, bus_dmamap_t map,
    struct vm_page **ma, bus_size_t tlen, int ma_offs, int flags,
    bus_dma_segment_t *segs, int *segp);

#ifdef WANT_INLINE_DMAMAP
#define BUS_DMAMAP_OP static inline
#else
#define BUS_DMAMAP_OP
#endif

/*
 * Allocate a handle for mapping from kva/uva/physical
 * address space into bus device space.
 */
BUS_DMAMAP_OP int bus_dmamap_create(bus_dma_tag_t dmat, int flags, bus_dmamap_t *mapp);

/*
 * Destroy a handle for mapping from kva/uva/physical
 * address space into bus device space.
 */
BUS_DMAMAP_OP int bus_dmamap_destroy(bus_dma_tag_t dmat, bus_dmamap_t map);

/*
 * Allocate a piece of memory that can be efficiently mapped into
 * bus device space based on the constraints listed in the dma tag.
 * A dmamap to for use with dmamap_load is also allocated.
 */
BUS_DMAMAP_OP int bus_dmamem_alloc(bus_dma_tag_t dmat, void** vaddr, int flags,
		     bus_dmamap_t *mapp);

/*
 * Free a piece of memory and its allocated dmamap, that was allocated
 * via bus_dmamem_alloc.
 */
BUS_DMAMAP_OP void bus_dmamem_free(bus_dma_tag_t dmat, void *vaddr, bus_dmamap_t map);

/*
 * Perform a synchronization operation on the given map. If the map
 * is NULL we have a fully IO-coherent system.
 */
BUS_DMAMAP_OP void bus_dmamap_sync(bus_dma_tag_t dmat, bus_dmamap_t dmamap, bus_dmasync_op_t op);

/*
 * Release the mapping held by map.
 */
BUS_DMAMAP_OP void bus_dmamap_unload(bus_dma_tag_t dmat, bus_dmamap_t dmamap);

#undef BUS_DMAMAP_OP
#endif /* _KERNEL */

#endif /* _BUS_DMA_H_ */
