// A simple color editor using glyphs -- Dave Sternlicht, mit X consortium
//

#include <IV-look/button.h>
#include <IV-look/dialogs.h>
#include <IV-look/kit.h>
#include <IV-look/menu.h>
#include <IV-look/field.h>
#include <IV-look/telltale.h>
#include <InterViews/adjust.h>
#include <InterViews/background.h>
#include <InterViews/box.h>
#include <InterViews/color.h>
#include <InterViews/display.h>
#include <InterViews/glue.h>
#include <InterViews/label.h>
#include <InterViews/layout.h>
#include <InterViews/patch.h>
#include <InterViews/session.h>
#include <InterViews/style.h>
#include <InterViews/window.h>
#include <OS/string.h>
#include <stdio.h>

//
// InputGroup class
//

class InputGroup : public InputHandler {
public:
    InputGroup(Glyph*, Style*);
    virtual ~InputGroup();
};

InputGroup::InputGroup(Glyph* g, Style* s) : InputHandler(g, s) { }
InputGroup::~InputGroup() { }

//
// CEBoundedValue class
//

class CEBoundedValue : public Adjustable {
protected:
    CEBoundedValue();
public:
    CEBoundedValue(Coord lower, Coord upper);
    virtual ~CEBoundedValue();

    virtual void lower_bound(Coord);
    virtual void upper_bound(Coord);
    virtual void current_value(Coord);
    virtual void scroll_incr(Coord);
    virtual void page_incr(Coord);

    virtual Coord lower(DimensionName) const;
    virtual Coord upper(DimensionName) const;
    virtual Coord length(DimensionName) const;
    virtual Coord cur_lower(DimensionName) const;
    virtual Coord cur_upper(DimensionName) const;
    virtual Coord cur_length(DimensionName) const;

    virtual void scroll_to(DimensionName, Coord position);
    virtual void scroll_forward(DimensionName);
    virtual void scroll_backward(DimensionName);
    virtual void page_forward(DimensionName);
    virtual void page_backward(DimensionName);
private:
    Coord curvalue_;
    Coord lower_;
    Coord span_;
    Coord scroll_incr_;
    Coord page_incr_;
};


CEBoundedValue::CEBoundedValue() {
    scroll_incr_ = 0.0;
    page_incr_ = 0.0;
}

CEBoundedValue::CEBoundedValue(Coord lower, Coord upper) {
    lower_ = lower;
    span_ = upper - lower;
    scroll_incr_ = span_ * 0.04;
    page_incr_ = span_ * 0.4;
    curvalue_ = (lower + upper) * 0.5;
}

CEBoundedValue::~CEBoundedValue() { }

void CEBoundedValue::lower_bound(Coord c) { lower_ = c; }
void CEBoundedValue::upper_bound(Coord c) { span_ = c - lower_; }

void CEBoundedValue::current_value(Coord value) {
    curvalue_ = value;
    constrain(Dimension_X, curvalue_);
    notify(Dimension_X);
    notify(Dimension_Y);
}

void CEBoundedValue::scroll_incr(Coord c) { scroll_incr_ = c; }
void CEBoundedValue::page_incr(Coord c) { page_incr_ = c; }

Coord CEBoundedValue::lower(DimensionName) const { return lower_; }
Coord CEBoundedValue::upper(DimensionName) const { return lower_ + span_; }
Coord CEBoundedValue::length(DimensionName) const { return span_; }
Coord CEBoundedValue::cur_lower(DimensionName) const { return curvalue_; }
Coord CEBoundedValue::cur_upper(DimensionName) const { return curvalue_; }
Coord CEBoundedValue::cur_length(DimensionName) const { return 0; }

void CEBoundedValue::scroll_to(DimensionName d, Coord position) {
    Coord p = position;
    constrain(d, p);
    if (p != curvalue_) {
       curvalue_ = p;
       notify(Dimension_X);
       notify(Dimension_Y);
    }
}

void CEBoundedValue::scroll_forward(DimensionName d) {
    scroll_to(d, curvalue_ + scroll_incr_);
}

void CEBoundedValue::scroll_backward(DimensionName d) {
    scroll_to(d, curvalue_ - scroll_incr_);
}

void CEBoundedValue::page_forward(DimensionName d) {
    scroll_to(d, curvalue_ + page_incr_);
}

void CEBoundedValue::page_backward(DimensionName d) {
    scroll_to(d, curvalue_ - page_incr_);
}

//
// Valuator
//

class Valuator : public MonoGlyph, public Observer {
public:
    Valuator(CEBoundedValue*, Style*);
    virtual ~Valuator();

    virtual InputHandler* focusable() const;

    virtual void update(Observable*);
    virtual void disconnect(Observable*);
private:
    CEBoundedValue* bvalue_;
    FieldEditor* editor_;

#ifdef OS2
    void accept_editor(FieldEditor*);
    void cancel_editor(FieldEditor*);
#else
    void accept_editor(FieldEditor*,boolean);
    void cancel_editor(FieldEditor*, boolean);
#endif
};


declareFieldEditorCallback(Valuator)
implementFieldEditorCallback(Valuator)



Valuator::Valuator(CEBoundedValue* bv, Style* style) : MonoGlyph(nil)
{
   Valuator*   v = this;
    Style* s = new Style(style);
    s->alias("Valuator");
    bvalue_ = bv;
    bv->attach(Dimension_X, this);
    editor_ = DialogKit::instance()->field_editor(
       "100.00", s,
       new FieldEditorCallback(Valuator)(
            this, &Valuator::accept_editor,
            &Valuator::cancel_editor
        )
    );
    body(editor_);
    update(bv->observable(Dimension_X));
}

Valuator::~Valuator() {
    if (bvalue_ != nil) {
       bvalue_->detach(Dimension_X, this);
    }
}


InputHandler* Valuator::focusable() const {
    return editor_;
}

void Valuator::update(Observable*) {
    Coord v = bvalue_->cur_lower(Dimension_X);
    char buf[20];
    sprintf(buf, "%.2f", v);
    editor_->field(buf);
}

void Valuator::disconnect(Observable*) {
    bvalue_ = nil;
}

void Valuator::accept_editor(FieldEditor*)
{
    Coord v;
    const String& value = *editor_->text();
    if (value.convert(v)) {
       bvalue_->current_value(v);
    }
}


void Valuator::cancel_editor(FieldEditor*)
{
    update(bvalue_->observable(Dimension_X));
}

//
// ColorViewer
//

class ColorViewer : public Patch, public Observer {
public:
    ColorViewer(CEBoundedValue*, CEBoundedValue*, CEBoundedValue*, Style*);
    virtual ~ColorViewer();

    virtual void allocate(Canvas*, const Allocation&, Extension&);
    virtual void draw(Canvas*, const Allocation&) const;
    virtual void update(Observable*);
    virtual void disconnect(CEBoundedValue*);
private:
    Adjustable* red_;
    Adjustable* green_;
    Adjustable* blue_;
    Color* color_;
};

ColorViewer::ColorViewer(
    CEBoundedValue* r, CEBoundedValue* g, CEBoundedValue* b, Style*
) : Patch(nil) {
    red_ = r;
    red_->attach(Dimension_X, this);
    green_ = g;
    green_->attach(Dimension_X, this);
    blue_ = b;
    blue_->attach(Dimension_X, this);
    color_ = nil;
}

ColorViewer::~ColorViewer() {
    if (red_ != nil) {
       red_->detach(Dimension_X, this);
    }
    if (green_ != nil) {
       green_->detach(Dimension_X, this);
    }
    if (blue_ != nil) {
       blue_->detach(Dimension_X, this);
    }
}

void ColorViewer::allocate(Canvas* c, const Allocation& a, Extension& ext) {
    ext.set(c, a);
    Patch::allocate(c, a, ext);
}

void ColorViewer::draw(Canvas* c, const Allocation& a) const {
    c->fill_rect(a.left(), a.bottom(), a.right(), a.top(), color_);
}

void ColorViewer::update(Observable*) {
    Resource::unref(color_);
    color_ = new Color(
       float(red_->cur_lower(Dimension_X)),
       float(green_->cur_lower(Dimension_X)),
       float(blue_->cur_lower(Dimension_X)),
       1.0
    );
    Resource::ref(color_);
    redraw();
}

void ColorViewer::disconnect(CEBoundedValue* a) {
    if (a == red_) {
       red_ = nil;
    } else if (a == green_) {
       green_ = nil;
    } else if (a == blue_) {
       blue_ = nil;
    }
}

//
// main
//

int main(int argc, char** argv) {
    Session* session = new Session(
       "ColorViewer", argc, argv);
    Style* style = session->style();
    WidgetKit& kit = *WidgetKit::instance();
    LayoutKit& layout = *LayoutKit::instance();
    CEBoundedValue* bv1 = new CEBoundedValue(0.0, 1.0);
    CEBoundedValue* bv2 = new CEBoundedValue(0.0, 1.0);
    CEBoundedValue* bv3 = new CEBoundedValue(0.0, 1.0);
    Window* w = new ApplicationWindow(
       new Background(
           layout.hbox(
               layout.vbox(
                   layout.vcenter(
                       layout.hbox(
                            layout.margin(
                                kit.push_button("Quit", kit.quit()),
                                2.0
                            ),
                           new Valuator(bv1, style),
                           new Valuator(bv2, style),
                           new Valuator(bv3, style)
                       ),
                       1.0
                   ),
                   layout.v_fixed_span(
                       new ColorViewer(bv1, bv2, bv3, style),
                       144.0
                   )
               ),
               kit.vscroll_bar(bv1),
               kit.vscroll_bar(bv2),
               kit.vscroll_bar(bv3)
           ),
           kit.background()
       )
    );
    bv1->current_value(0.5);
    bv2->current_value(0.5);
    bv3->current_value(0.5);
    return session->run_window(w);
}

#if 0

class Valuator__FieldEditorCallback(T) : public FieldEditorAction { \
public: \
    FieldEditorCallback(T)(T*, FieldEditorMemberFunction(T)); \
    virtual ~FieldEditorCallback(T)(); \
\
    virtual void execute(FieldEditor*, boolean accept); \
private: \
    T* obj_; \
    FieldEditorMemberFunction(T) func_; \
};

#define implementFieldEditorCallback(T) \
FieldEditorCallback(T)::FieldEditorCallback(T)( \
    T* obj, FieldEditorMemberFunction(T) func \
) { \
    obj_ = obj; \
    func_ = func; \
} \
\
FieldEditorCallback(T)::~FieldEditorCallback(T)() { } \
\
void FieldEditorCallback(T)::execute(FieldEditor* f, boolean accept) { \
    (obj_->*func_)(f, accept); \
}
#endif



